////////////////////////////////////////////////////////////////////////////
//
//	COMM.C		- Written by Mike Sax for Dr. Dobb's Journal
//
//	This file implements a few higher-level communicatons functions under
//	Microsoft Windows.
//
//	This file contains eight public functions:
//
//	int OpenComPort(int nPort);
//	BOOL CloseComPort(int nPortID);
//	BOOL SetComPortParameters(int nPortID, int nSpeed, char chParity,
//			int nDataBits, int nStopBits, BOOL bXOnXOff, BOOL bHardware);
//	int CharsWaitingToBeRead(int nPortID);
//	int ComReadChar(int nPortID);
//	int ComReadChars(int nPortID, char *pchBuffer, int cbBuffer);
//	BOOL ComWriteChar(int nPortID, int nChar);
//
////////////////////////////////////////////////////////////////////////////

#define USECOMM 1				// for 3.1 windows.h
#include <windows.h>
#include "comm.h"

// Opens the com. port (1 = COM1) and returns its ID value.
// If an error occurs, the return value is negative
int OpenComPort(int nPort)
	{
	char szPort[10];

	wsprintf(szPort, "COM%d", nPort);
	// Open the port with a 4K input queue and a 2K output queue
	return OpenComm(szPort, 4096, 2048);
	}

// Closes the comm. port specified by nPortID
// returns TRUE if success, FALSE if failure
BOOL CloseComPort(int nPortID)
	{
	if (nPortID < 0)
		return FALSE;
	FlushComm(nPortID,0);       // Flush transmit queue
	FlushComm(nPortID,1);       // Flush receive queue
	return !CloseComm(nPortID);
	}

// Sets the communications parameters of the port specified by nPortID.
// returns TRUE if success, FALSE if failure
BOOL SetComPortParameters(int nPortID, int nSpeed, char chParity, int nDataBits,
						  int nStopBits, BOOL bXOnXOff, BOOL bHardware)
	{
	DCB dcb;

	if (nPortID < 0)
		return FALSE;
	dcb.Id = nPortID;
	dcb.BaudRate = nSpeed;
	dcb.ByteSize = (BYTE)nDataBits;
	// Convert chParity to uppercase:
	AnsiUpperBuff(&chParity, 1);
	dcb.Parity =  (chParity == 'N') ? NOPARITY :
				  (chParity == 'O') ? ODDPARITY :
				  (chParity == 'E') ? EVENPARITY :
				  (chParity == 'M') ? MARKPARITY : SPACEPARITY;
	dcb.StopBits = (BYTE)((nStopBits == 1) ? ONESTOPBIT :
						  (nStopBits == 2) ? TWOSTOPBITS : ONE5STOPBITS);
	dcb.RlsTimeout= 0;
	dcb.CtsTimeout = bHardware ? 30 : 0;
	dcb.DsrTimeout = 0;
	dcb.fBinary = TRUE;
	dcb.fRtsDisable = FALSE;
	dcb.fParity = FALSE;
	dcb.fOutxCtsFlow = (BYTE)bHardware;
	dcb.fOutxDsrFlow = FALSE;
	dcb.fDummy = 0;
	dcb.fDtrDisable = FALSE;
	dcb.fOutX = (BYTE)bXOnXOff;
	dcb.fInX = (BYTE)bXOnXOff;
	dcb.fPeChar = FALSE;
	dcb.fNull = FALSE;
	dcb.fChEvt = FALSE;
	dcb.fDtrflow = (BYTE)FALSE;
	dcb.fRtsflow = (BYTE)bHardware;
	dcb.fDummy2 = 0;
	dcb.XonChar = 17;
	dcb.XoffChar = 19;
	dcb.XonLim = 4096 / 4;              // Receive buffer size / 4
	dcb.XoffLim = dcb.XonLim;
	dcb.EofChar = 26;
	dcb.EvtChar = 0;
	dcb.TxDelay = 0;
	return !SetCommState(&dcb);
	}

// Returns the number of characters waiting in the input queue
int CharsWaitingToBeRead(int nPortID)
	{
	COMSTAT ComStat;

	if (nPortID < 0)
		return 0;
	GetCommError(nPortID, &ComStat);
	return ComStat.cbInQue;
	}

// Read a character from the port specified by nPortID
// returns -1 if no character available
int ComReadChar(int nPortID)
	{
	int iResult = 0;

	if (nPortID < 0)
		return -1;
	if (ReadComm(nPortID, (LPSTR)&iResult, 1) != 1)
		{
		iResult = -1;
		GetCommError(nPortID, NULL);
		}
	return iResult;
	}

// Read a character from the port specified by nPortID
// returns the number of characters read or -1 if an error occurs.
int ComReadChars(int nPortID, char *pchBuffer, int cbBuffer)
	{
	int iResult = 0;

	if (nPortID < 0)
		return -1;
	iResult = ReadComm(nPortID, pchBuffer, cbBuffer);
	if (iResult < 0)
		{
		iResult = -1;
		GetCommError(nPortID, NULL);
		}
	return iResult;
	}

// Write a character to the port specified by nPortID
// returns TRUE if success, FALSE if failure
BOOL ComWriteChar(int nPortID, int nChar)
	{
	if (nPortID < 0)
		return FALSE;
	if (1 != WriteComm(nPortID, (LPSTR)&nChar, 1))
		{
		GetCommError(nPortID, NULL);
		return FALSE;
		}
	return TRUE;
	}
