/****************************************************************************

    PROGRAM: Datest.c
	     by Mark Jones, (214) 291-0509
             Compuserve 70511,706

    PURPOSE: Tests the DATE.DLL routines

    FUNCTIONS:

	WinMain() - calls initialization function, processes message loop
	InitApplication() - initializes window data and registers window
	InitInstance() - saves instance handle and creates main window
	MainWndProc() - processes messages
	About() - processes messages for "About" dialog box

****************************************************************************/
/*****************************************************************************

                           MODIFICATION LOG

Date     Version  Who      Change
----     -------  ---      --------------------------------------------------
3 OCT 90 1.00     Mark J.  Originated
22 OCT   1.01     Mark J.  Add _WINDOWS and recompiled with warning level 3
                           (W3), fixing certain conversion warnings
2/10/91                    Changed "<win.h>" reference to <windows.h>


*****************************************************************************/

#define _WINDOWS

#include <windows.h>                // needed for all windows app's
#include <stdlib.h>                 /* needed for strtol() */
#include "Datest.h"                 /* specific to this program */
#include "Date.h"

HANDLE hInst;                       /* current instance */
HWND hParent;                       /* main parent window */
HWND hTestDialog;                   /* handle of DateTest dialog box */
HWND hDateObjDialog;                /* handle of DateObjDlg dialog box */
HWND hDateObjLB;                    /* handle to DateObjDlg list box */
HANDLE hDateObj;                    /* handle to DateObj */

/****************************************************************************

    FUNCTION: WinMain(HANDLE, HANDLE, LPSTR, int)

    PURPOSE: calls initialization function, processes message loop

****************************************************************************/

int PASCAL WinMain(hInstance, hPrevInstance, lpCmdLine, nCmdShow)
HANDLE hInstance;                            /* current instance             */
HANDLE hPrevInstance;                        /* previous instance            */
LPSTR lpCmdLine;                             /* command line                 */
int nCmdShow;                                /* show-window type (open/icon) */
{
    MSG msg;                                 /* message                      */

   if (!hPrevInstance)                  /* Other instances of app running? */
	if (!InitApplication(hInstance)) /* Initialize shared things */
	    return (FALSE);              /* Exits if unable to initialize     */

    /* Perform initializations that apply to a specific instance */

    if (!InitInstance(hInstance, nCmdShow))
      return (FALSE);

    /* Acquire and dispatch messages until a WM_QUIT message is received. */

    while (GetMessage(&msg,        /* message structure                      */
	         NULL,                  /* handle of window receiving the message */
	         NULL,                  /* lowest message to examine              */
	         NULL))                 /* highest message to examine             */
      {
      if (!IsDialogMessage(hDateObjDialog, &msg) &&
          !IsDialogMessage(hTestDialog, &msg)
         )
	      {
	      TranslateMessage(&msg);        /* Translates virtual key codes   */
	      DispatchMessage(&msg);         /* Dispatches message to window   */
	      }
      }
    return (msg.wParam);           /* Returns the value from PostQuitMessage */
}


/****************************************************************************

    FUNCTION: InitApplication(HANDLE)

    PURPOSE: Initializes window data and registers window class

    COMMENTS:

	This function is called at initialization time only if no other 
	instances of the application are running.  This function performs 
	initialization tasks that can be done once for any number of running 
	instances.  

	In this case, we initialize a window class by filling out a data 
	structure of type WNDCLASS and calling the Windows RegisterClass() 
	function.  Since all instances of this application use the same window 
	class, we only need to do this when the first instance is initialized.  


****************************************************************************/

BOOL InitApplication(hInstance)
HANDLE hInstance;                              /* current instance           */
{
    WNDCLASS  wc;

    /* Fill in window class structure with parameters that describe the       */
    /* main window.                                                           */

    wc.style = NULL;                    /* Class style(s).                    */
    wc.lpfnWndProc = MainWndProc;       /* Function to retrieve messages for  */
                                        /* windows of this class.             */
    wc.cbClsExtra = 0;                  /* No per-class extra data.           */
    wc.cbWndExtra = 0;                  /* No per-window extra data.          */
    wc.hInstance = hInstance;           /* Application that owns the class.   */
    wc.hIcon = LoadIcon(hInstance, "DatestIcon");
    wc.hCursor = LoadCursor(NULL, IDC_ARROW);
    wc.hbrBackground = COLOR_BACKGROUND;
    wc.lpszMenuName =  "DatestMenu";   /* Name of menu resource in .RC file. */
    wc.lpszClassName = "DatestWClass"; /* Name used in call to CreateWindow. */

    /* Register the window class and return success/failure code. */

    return (RegisterClass(&wc));

}


/****************************************************************************

    FUNCTION:  InitInstance(HANDLE, int)

    PURPOSE:  Saves instance handle and creates main window

    COMMENTS:

	This function is called at initialization time for every instance of 
	this application.  This function performs initialization tasks that 
	cannot be shared by multiple instances.  

	In this case, we save the instance handle in a static variable and 
	create and display the main program window.  
	
****************************************************************************/

BOOL InitInstance(hInstance, nCmdShow)
   HANDLE          hInstance;          /* Current instance identifier.       */
   int             nCmdShow;           /* Param for first ShowWindow() call. */
{
   HWND            hWnd;               /* Main window handle.                */
   long lResult;                       /* LONG return value */

   /* Save the instance handle in static variable, which will be used in  */
   /* many subsequence calls from this application to Windows.            */

   hInst = hInstance;
   hParent = 0;

   /* Create a main window for this application instance.  */

   hWnd = CreateWindow(
	  "DatestWClass",                /* See RegisterClass() call.          */
	  "Datest - Test DATE.DLL functions", /* Text for window title bar.            */
	  WS_OVERLAPPEDWINDOW,            /* Window style.                      */
	  CW_USEDEFAULT,                  /* Default horizontal position.       */
	  CW_USEDEFAULT,                  /* Default vertical position.         */
	  CW_USEDEFAULT,                  /* Default width.                     */
	  CW_USEDEFAULT,                  /* Default height.                    */
	  NULL,                           /* Overlapped windows have no parent. */
	  NULL,                           /* Use the window class menu.         */
	  hInstance,                      /* This instance owns this window.    */
	  NULL                            /* Pointer not needed.                */
   );

   /* If window could not be created, return "failure" */
   if (!hWnd)
     return (FALSE);
   hParent = hWnd;

   // Create my modeless DateTest dialog box
   hTestDialog = CreateDialog(hInst, "DATETEST", hParent,
	      MakeProcInstance(DateTest, hInst));
   if (!hTestDialog)
     {
     MessageBox(hParent, "Cannot create Date-test dialog box",
	         "FATAL ERROR", MB_ICONSTOP | MB_OK);
     return (FALSE);
     }

   // Create my modeless DateObjDlg dialog box
   hDateObjDialog = CreateDialog(hInst, "DATEOBJ", hParent,
	      MakeProcInstance(DateObjDlg, hInst));
   if (!hDateObjDialog)
      {
      MessageBox(hParent, "Cannot create DateObjDlg dialog box",
	            "FATAL ERROR", MB_ICONSTOP | MB_OK);
      return (FALSE);
      }
   else
      {
                                          // Set up Normal as calculation mode
      CheckDlgButton(hDateObjDialog, IDD_NORMALMODE, 1);
      hDateObj = NULL;                    // Allocate a DateObj instance
      lResult = DateObj(0, CREATE, DATE, 0);
      if (HIWORD(lResult) == DATE_ERROR)
         {
         MessageBox(hParent, "Cannot create DateObj instance",
	               "FATAL ERROR", MB_ICONSTOP | MB_OK);
         return (FALSE);
         }
      else
         hDateObj = LOWORD(lResult);      // Store handle to DateObj instance
                                          // Store handle to list box
         hDateObjLB = GetDlgItem(hDateObjDialog, IDD_DATEOBJCONTENTS);
      }

   /* Make the window visible; update its client area; and return "success" */
   ShowWindow(hWnd, nCmdShow); /* Show the window                         */
   UpdateWindow(hWnd);         /* Sends WM_PAINT message                  */
   SetFocus(hWnd);             /* Moves focus to the main window */

   return (TRUE);              /* Returns the value from PostQuitMessage */
}

/****************************************************************************

    FUNCTION: MainWndProc(HWND, unsigned, WORD, LONG)

    PURPOSE:  Processes messages

    MESSAGES:

	WM_COMMAND    - application menu (About dialog box)
	WM_DESTROY    - destroy window

    COMMENTS:

	To process the IDM_ABOUT message, call MakeProcInstance() to get the
	current instance address of the About() function.  Then call Dialog
	box which will create the box according to the information in your
	Datest.rc file and turn control over to the About() function.   When
	it returns, free the intance address.

****************************************************************************/

long FAR PASCAL MainWndProc(hWnd, message, wParam, lParam)
HWND hWnd;                                /* window handle                   */
unsigned message;                         /* type of message                 */
WORD wParam;                              /* additional information          */
LONG lParam;                              /* additional information          */
{
   FARPROC lpProcAbout;                   /* pointer to the "About" function */
   LONG lResult;

    switch (message)
      {
	   case WM_COMMAND:           /* message: command from application menu */

	      switch (wParam)
	         {
	         case IDM_ABOUT:
		            lpProcAbout = MakeProcInstance(About, hInst);

		            DialogBox(hInst,             /* current instance         */
		            "AboutBox",                 /* resource to use          */
		            hWnd,                       /* parent handle            */
		            lpProcAbout);               /* About() instance address */

		            FreeProcInstance(lpProcAbout);
		            break;

	         case IDM_TEST:
		            TestDateRoutines(hWnd);    /* test primitive DATE.DLL */
		            break;

	         case IDM_OOP:
		            TestDateObj(hWnd);         /* test OOP DATE.DLL */
		            break;

	         case IDM_EXIT:
		            DestroyWindow(hWnd);       /* close main window */
		            break;

	         }
	         return (DefWindowProc(hWnd, message, wParam, lParam));

         case WM_DESTROY:                  /* message: window being destroyed */
	            KillTimer(hTestDialog, TIMER_DATETEST); /* Stop the timer */
               if (hDateObj)
                  {
                  lResult = DateObj(hDateObj, DELETE, DATE, 0);
                  if (HIWORD(lResult) == DATE_ERROR)
                     MessageBox(hParent, "Cannot delete DateObj instance",
	                           "FATAL ERROR", MB_ICONSTOP | MB_OK);
                  hDateObj = 0;
                  }
	            PostQuitMessage(0);
	            break;

         default:                          /* Passes it on if unproccessed    */
	            return (DefWindowProc(hWnd, message, wParam, lParam));
      }
    return (NULL);
}


/****************************************************************************

    FUNCTION: About(HWND, unsigned, WORD, LONG)

    PURPOSE:  Processes messages for "About" dialog box

    MESSAGES:

	WM_INITDIALOG - initialize dialog box
	WM_COMMAND    - Input received

    COMMENTS:

	No initialization is needed for this particular dialog box, but TRUE
	must be returned to Windows.

	Wait for user to click on "Ok" button, then close the dialog box.

****************************************************************************/

BOOL FAR PASCAL About(hDlg, message, wParam, lParam)
HWND hDlg;                                /* window handle of the dialog box */
unsigned message;                         /* type of message                 */
WORD wParam;                              /* message-specific information    */
LONG lParam;
{
    switch (message) {
	case WM_INITDIALOG:                /* message: initialize dialog box */
	    return (TRUE);

	case WM_COMMAND:                      /* message: received a command */
	    if (wParam == IDOK                /* "OK" box selected?          */
		|| wParam == IDCANCEL) {      /* System menu close command? */
		EndDialog(hDlg, TRUE);        /* Exits the dialog box        */
		return (TRUE);
	    }
	    break;
    }
    return (FALSE);                           /* Didn't process a message    */
}


/****************************************************************************

    FUNCTION: VOID TestDateRoutines(VOID)

    PURPOSE:  Calls the DATE.DLL primitive functions to test their accuracy

    COMMENTS:

****************************************************************************/

VOID TestDateRoutines(HWND hWnd)
{
    #define FREQ_MILLISECONDS 250

    while (!SetTimer(hTestDialog, TIMER_DATETEST, FREQ_MILLISECONDS, NULL))
	{
	if (IDCANCEL == MessageBox(hParent,
	    "Too many clocks or timers!  Close other windows and retry.",
	    "RESOURCES UNAVAILABLE",MB_ICONEXCLAMATION | MB_RETRYCANCEL) )
	    return;
	}
    SendMessage(hTestDialog, WM_INITDIALOG, NULL, NULL);
    ShowWindow(hTestDialog, SW_SHOWNORMAL);
}


/****************************************************************************

    FUNCTION: DateTest(HWND, unsigned, WORD, LONG)

    PURPOSE:  Processes messages for "DateTest" dialog box

    MESSAGES:

	WM_INITDIALOG - initialize dialog box
	WM_COMMAND    - Input received

    COMMENTS:


****************************************************************************/

BOOL FAR PASCAL DateTest(hDlg, message, wParam, lParam)
HWND hDlg;                                /* window handle of the dialog box */
unsigned message;                         /* type of message                 */
WORD wParam;                              /* message-specific information    */
LONG lParam;
{
    static long lJulianDate = 0;            /* persistent julian date */
    static Date DateStruct = {0};           /* Date structure */
    static Date *DStruct = &DateStruct;     /* ptr to Date structure */
    #define MAXSTRSIZE 20                   /* Maximum size of string buffer */
    unsigned char szBuffer[MAXSTRSIZE];     /* string to hold stuff */
    long lWorkFld;                          /* general-purpose field */

    int nLoopCtr;                           /* temporary loop counter */
    #define ITERATIONS 6                    /* number of calc's per WM_TIMER */

    switch (message) {

	case WM_INITDIALOG:                /* message: initialize dialog box */
	    lJulianDate = 1;
	    return (TRUE);

	case WM_COMMAND:                      /* message: received a command */
	    if (wParam == IDD_DATETESTCANCEL  /* "CANCEL" box selected?      */
		|| wParam == IDCANCEL)        /* System menu close command? */
		{
		KillTimer(hDlg, TIMER_DATETEST); /* Stop the timer */
		ShowWindow(hTestDialog, SW_HIDE); /* hide the dialog box */
		return (TRUE);                /* return */
		}
	    break;

	case WM_TIMER:                      /* do some more iterations */
	    for (nLoopCtr=0; nLoopCtr < ITERATIONS; lJulianDate++, nLoopCtr++)
		{
		if (lJulianDate > MAX_JULIAN)
		    lJulianDate = 1;

		// Set up julian date
		DStruct->lJulianDate = lJulianDate;

		// Display julian date
		wsprintf(szBuffer, "%lu", lJulianDate);/* put val in string */
                                             /* change dlg text */
		SetDlgItemText(hDlg, IDD_DATETESTINT, (LPSTR) szBuffer);

		// Call DATE.DLL
		CalcNormalDate((Date far *) DStruct);   /* get normal date */
		if (DStruct->nError)                    /* if bug, */
		    {                                   /* prompt & exit */
		    KillTimer(hDlg, TIMER_DATETEST); /* Stop the timer */
		    MessageBox(hParent, "DATE.DLL error!", "FATAL ERROR",
			MB_ICONEXCLAMATION | MB_OK);
		    ShowWindow(hTestDialog, SW_HIDE); /* hide the dialog box */
		    break;
		    }

		// Display normal results
		// PERCENT DONE WITH TEST:
		lWorkFld = (lJulianDate * 100) / MAX_JULIAN;
		wsprintf(szBuffer, "%3lu%%", lWorkFld ? lWorkFld : 1);
		SetDlgItemText(hDlg, IDD_PCNTDONE, (LPSTR) szBuffer);
		// DATE:
		wsprintf(szBuffer, "%2d/%2d/%4d", DStruct->nMonth,
				DStruct->nDay, DStruct->nYear);
		SetDlgItemText(hDlg, IDD_MMDDYYYY, (LPSTR) szBuffer);
		// DAY OF WEEK:
		wsprintf(szBuffer, "%1d", DStruct->nDayOfWeek);
		SetDlgItemText(hDlg, IDD_DOWVAL, (LPSTR) szBuffer);
		// DAY # IN YEAR:
		wsprintf(szBuffer, "%3d", DStruct->nYearDays);
		SetDlgItemText(hDlg, IDD_YEARDAYVAL, (LPSTR) szBuffer);
		// LEAP YEAR?
		wsprintf(szBuffer, "%s", DStruct->nLeapYear ?
                  (LPSTR) "Yes" : (LPSTR) "No");
		SetDlgItemText(hDlg, IDD_LEAPVAL, (LPSTR) szBuffer);

		}
	    break;

    }
    return (FALSE);                           /* Didn't process a message    */
}


/****************************************************************************

    FUNCTION: VOID TestDateObj(VOID)

    PURPOSE:  Calls the DateObj() function to test its accuracy

    COMMENTS:

****************************************************************************/

VOID TestDateObj(HWND hWnd)
{
   SendMessage(hDateObjDialog, WM_INITDIALOG, NULL, NULL);
   ShowWindow(hDateObjDialog, SW_SHOWNORMAL);
   if (!UpdateDateObjLB(hDateObjLB, hDateObj))
      MessageBox(hParent, "Cannot initialize DateObj list box",
	            "FATAL ERROR", MB_ICONSTOP | MB_OK);
}


/****************************************************************************

    FUNCTION: DateObjDlg(HWND, unsigned, WORD, LONG)

    PURPOSE:  Processes messages for "DateObj" dialog box

    MESSAGES:

	WM_INITDIALOG - initialize dialog box
	WM_COMMAND    - Input received

    COMMENTS:


****************************************************************************/

BOOL FAR PASCAL DateObjDlg(hDlg, message, wParam, lParam)
HWND hDlg;                                /* window handle of the dialog box */
unsigned message;                         /* type of message                 */
WORD wParam;                              /* message-specific information    */
LONG lParam;
{
   WORD nMonth = 0;                       /* Normal-date fields */
   WORD nDay = 0;
   WORD nYear = 0;
   LONG lJulianDate = 0;                  /* Julian-date field */
   #define MAXJULEN 20
   BYTE szJulianDate[MAXJULEN+1];         /* Julian-date string */
   WORD nFromNormal = 1;                  /* Calc from Normal date ? */
   LONG lResult;                          /* DateObj() result field */

   switch (message)
     {
     case WM_INITDIALOG:                  /* message: initialize dialog box */
	     return (TRUE);

     case WM_COMMAND:                     /* message: received a command */
        switch (wParam)
           {
           case IDD_CANCEL:
           case IDCANCEL:                 /* hide the dialog box */
	           ShowWindow(hDateObjDialog, SW_HIDE);
	           return (TRUE);

           case IDD_UPDATE:               /* put fields into DateObj instance */
               nMonth = GetDlgItemInt(hDateObjDialog, IDD_MONTH, 0, 0);
               nDay = GetDlgItemInt(hDateObjDialog, IDD_DAY, 0, 0);
               nYear = GetDlgItemInt(hDateObjDialog, IDD_YEAR, 0, 0);
               GetDlgItemText(hDateObjDialog, IDD_JULIAN, (LPSTR) szJulianDate,
                              MAXJULEN);
               lJulianDate = strtol(szJulianDate, 0, 0);
               lResult = DateObj(hDateObj, SET, MONTH, (LONG) nMonth);
               lResult = DateObj(hDateObj, SET, DAY, (LONG) nDay);
               lResult = DateObj(hDateObj, SET, YEAR, (LONG) nYear);
               lResult = DateObj(hDateObj, SET, JULIANDATE, lJulianDate);
               UpdateDateObjLB(hDateObjLB, hDateObj);
               break;

            case IDD_CALCULATE:
               nFromNormal = IsDlgButtonChecked(hDateObjDialog,
                                                IDD_NORMALMODE);
               if (nFromNormal)
                  lResult = DateObj(hDateObj, CALCULATE, 0, NORMAL);
               else
                  lResult = DateObj(hDateObj, CALCULATE, 0, JULIAN);
               UpdateDateObjLB(hDateObjLB, hDateObj);
               break;
	        }
        break;
     }
   return (FALSE);                           /* Didn't process a message    */
}


/****************************************************************************

    FUNCTION: UpdateDateObjLB(HWND hLB, HANDLE hDateObj)

    PURPOSE:  Updates contents of hLB list box with hDateObj instance data

    COMMENTS:


****************************************************************************/

BOOL UpdateDateObjLB(HWND hLB, HANDLE hDateObj)
{
   #define MAXLEN 20
   LONG lResult = 0;
   BOOL bResult = 1;
   BYTE szString[MAXLEN+1];

   if (!hLB || !hDateObj)                    // If any NULL handles,
      return (0);                            //  abort w/failure
   SendMessage(hLB, LB_RESETCONTENT, 0, 0);  // Tell list box to clear itself
   SendMessage(hLB, WM_SETREDRAW, 0, 0);     // Turn off the redraw flag
   
   lResult = DateObj(hDateObj, GET, YEAR, 0);// Add nYear to list box
   if (HIWORD(lResult) != DATE_ERROR)
      {
      wsprintf(szString, "nYear: %d", (WORD) lResult);
      SendMessage(hLB, LB_ADDSTRING, 0, (LONG) ((LPSTR) szString));
      }
   else
      return(0);
                                             // Add nMonth to list box
   lResult = DateObj(hDateObj, GET, MONTH, 0);
   if (HIWORD(lResult) != DATE_ERROR)
      {
      wsprintf(szString, "nMonth: %d", (WORD) lResult);
      SendMessage(hLB, LB_ADDSTRING, 0, (LONG) ((LPSTR) szString));
      }
   else
      return(0);
   lResult = DateObj(hDateObj, GET, DAY, 0); // Add nDay to list box
   if (HIWORD(lResult) != DATE_ERROR)
      {
      wsprintf(szString, "nDay: %d", (WORD) lResult);
      SendMessage(hLB, LB_ADDSTRING, 0, (LONG) ((LPSTR) szString));
      }
   else
      return(0);
                                             // Add nLeapYear to list box
   lResult = DateObj(hDateObj, GET, LEAPYEAR, 0);
   if (HIWORD(lResult) != DATE_ERROR)
      {
      wsprintf(szString, "nLeapYear: %d", (WORD) lResult);
      SendMessage(hLB, LB_ADDSTRING, 0, (LONG) ((LPSTR) szString));
      }
   else
      return(0);
                                             // Add nYearDays to list box
   lResult = DateObj(hDateObj, GET, DAYOFYEAR, 0);
   if (HIWORD(lResult) != DATE_ERROR)
      {
      wsprintf(szString, "nYearDays: %d", (WORD) lResult);
      SendMessage(hLB, LB_ADDSTRING, 0, (LONG) ((LPSTR) szString));
      }
   else
      return(0);
                                             // Add lJulianDate to list box
   lResult = DateObj(hDateObj, GET, JULIANDATE, 0);
   if (HIWORD(lResult) != DATE_ERROR)
      {
      wsprintf(szString, "lJulianDate: %lu", lResult);
      SendMessage(hLB, LB_ADDSTRING, 0, (LONG) ((LPSTR) szString));
      }
   else
      return(0);
   SendMessage(hLB, WM_SETREDRAW, 1, 0);     // Turn on the redraw flag
                                             // Add nDayOfWeek to list box
   lResult = DateObj(hDateObj, GET, DAYOFWEEK, 0);
   if (HIWORD(lResult) != DATE_ERROR)
      {
      wsprintf(szString, "nDayOfWeek: %d", (WORD) lResult);
      SendMessage(hLB, LB_ADDSTRING, 0, (LONG) ((LPSTR) szString));
      }
   else
      return(0);

   return(bResult);                          // return BOOL result
}
