/*
 * File:     graphics.h
 * Purpose:  Object graphics library for wxWindows.
 *           Defines a canvas which repaints its own graphics objects.
 *           Sorry, this library is not documented but you should be able to work
 *           most of it out, with help from the `objects' demo.
 *
 *                       wxWindows 1.40
 * Copyright (c) 1993 Artificial Intelligence Applications Institute,
 *                   The University of Edinburgh
 *
 *                     Author: Julian Smart
 *                        Date: 18-4-93
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose is hereby granted without fee, provided
 * that the above copyright notice, author statement and this permission
 * notice appear in all copies of this software and related documentation.
 *
 * THE SOFTWARE IS PROVIDED "AS-IS" AND WITHOUT WARRANTY OF ANY KIND, EXPRESS,
 * IMPLIED OR OTHERWISE, INCLUDING WITHOUT LIMITATION, ANY WARRANTY OF
 * MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * IN NO EVENT SHALL THE ARTIFICIAL INTELLIGENCE APPLICATIONS INSTITUTE OR THE
 * UNIVERSITY OF EDINBURGH BE LIABLE FOR ANY SPECIAL, INCIDENTAL, INDIRECT OR
 * CONSEQUENTIAL DAMAGES OF ANY KIND, OR ANY DAMAGES WHATSOEVER RESULTING FROM
 * LOSS OF USE, DATA OR PROFITS, WHETHER OR NOT ADVISED OF THE POSSIBILITY OF
 * DAMAGE, AND ON ANY THEORY OF LIABILITY, ARISING OUT OF OR IN CONNECTION WITH
 * THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 */

#ifndef hy_graphicsh
#define hy_graphicsh

// Key identifiers
#define KEY_SHIFT 1
#define KEY_CTRL  2

#define CONTROL_POINT_SIZE       6

// Control point types
// Rectangle and most other shapes
#define CONTROL_POINT_VERTICAL   1
#define CONTROL_POINT_HORIZONTAL 2
#define CONTROL_POINT_DIAGONAL   3

// Line
#define CONTROL_POINT_ENDPOINT_TO 4
#define CONTROL_POINT_ENDPOINT_FROM 5
#define CONTROL_POINT_LINE       6


class ObjectCanvas;
class LineObject;
class ControlPoint;

/*
 * The basic canvas object, defining all the shared members
 *
 */

class CanvasObject: public wxObject
{
  wxObject *ClientData;
 public:
  Bool draggable;
  Bool formatted;
  float xpos, ypos;
  wxPen *pen;
  wxBrush *brush;
  wxFont *font;
  wxColour *text_colour;
  ObjectCanvas *canvas;
  wxDC *dc;
  wxList lines;
  wxList text;
  wxList control_points; // Control points for when the object is selected
  Bool visible;
  Bool disable_label;
  long id;
  Bool selected;
  Bool attachment_mode;  // TRUE if using attachments, FALSE otherwise

  CanvasObject(void);
  CanvasObject(ObjectCanvas *can);
  virtual ~CanvasObject(void);
  virtual void GetBoundingBox(float *width, float *height);
  virtual Bool GetPerimeterPoint(float x1, float y1,
                                 float x2, float y2,
                                 float *x3, float *y3);
  ObjectCanvas *GetCanvas(void);
  void SetCanvas(ObjectCanvas *canvas);
  float GetX(void);
  float GetY(void);

  virtual void OnDraw(void);
  virtual void OnDrawContents(void);
  virtual void OnMoveLinks(void);
  virtual void OnErase(void);
  virtual void OnHighlight(void);
  virtual void OnLeftClick(float x, float y, int keys = 0, int attachment = 0);
  virtual void OnRightClick(float x, float y, int keys = 0, int attachment = 0);
  virtual void OnSize(float x, float y);
  virtual void OnMove(float x, float y, float old_x, float old_y);

  virtual void OnDragLeft(Bool draw, float x, float y, int keys=0, int attachment = 0); // Erase if draw false
  virtual void OnBeginDragLeft(float x, float y, int keys=0, int attachment = 0);
  virtual void OnEndDragLeft(float x, float y, int keys=0, int attachment = 0);
  virtual void OnDragRight(Bool draw, float x, float y, int keys=0, int attachment = 0); // Erase if draw false
  virtual void OnBeginDragRight(float x, float y, int keys=0, int attachment = 0);
  virtual void OnEndDragRight(float x, float y, int keys=0, int attachment = 0);
  virtual void OnDrawOutline(void);
  virtual void OnDrawControlPoints(void);
  virtual void OnEraseControlPoints(void);

  virtual void MakeControlPoints(void);
  virtual void DeleteControlPoints(void);
  virtual void ResetControlPoints(void);
  virtual void Select(Bool select = TRUE);
  virtual Bool Selected(void);

  virtual Bool HitTest(float x, float y, int *attachment, float *distance);

  void SetPen(wxPen *pen);
  void SetBrush(wxBrush *brush);
  void SetFont(wxFont *font);
  void SetClientData(wxObject *client_data);
  wxObject *GetClientData(void);

  virtual void Show(Bool show);
  virtual void Draggable(Bool truth);
  virtual void Move(float x1, float y1);
  virtual void Erase(void);
  virtual void Draw(void);
  virtual void Flash(void);
  virtual void MoveLinks(void);
  virtual void DrawContents(void);  // E.g. for drawing text label
  virtual void SetSize(float x, float y);
  void Attach(ObjectCanvas *can);
  void Detach(void);
  void SetDC(wxDC *the_dc);

  void AddLine(LineObject *line, CanvasObject *other);
  void AddText(char *string);
  virtual void FormatText(char *s);
  void ClearText(void);
  void RemoveLine(LineObject *line);

  // Make a copy
  virtual void Copy(CanvasObject& copy);

  // Attachment code
  virtual void GetAttachmentPosition(int attachment, float *x, float *y,
                                     int nth = 0, int no_arcs = 1);
  virtual int GetNumberOfAttachments(void);
  virtual void SetAttachmentMode(Bool flag);

  virtual void EraseLinks(int attachment = -1);
  virtual void DrawLinks(int attachment = -1);

  virtual void MoveLineToNewAttachment(LineObject *to_move,
                                       float x, float y);
};

/*
 * Polygon object - has an arbitrary number of vertices
 *
 */

class PolygonObject: public CanvasObject
{
  wxList *points;
  wxList *original_points;
  float bound_width;
  float bound_height;
  float original_width;
  float original_height;
 public:

  PolygonObject(void);
  ~PolygonObject(void);

  // Takes a list of wxPoints; each point is an OFFSET from the centre.
  // Deletes user's points in destructor.
  virtual void Create(wxList *points);

  void GetBoundingBox(float *w, float *h);
  void CalculateBoundingBox(void);
  Bool GetPerimeterPoint(float x1, float y1,
                                 float x2, float y2,
                                 float *x3, float *y3);
  virtual void SetSize(float x, float y);
  virtual void OnDraw(void);

  // Make a copy
  void Copy(PolygonObject& copy);

  int GetNumberOfAttachments(void);
  void GetAttachmentPosition(int attachment, float *x, float *y,
                                     int nth = 0, int no_arcs = 1);
};

/*
 * Rectangle object (rounded/non-rounded)
 *
 */

class RectangleObject: public CanvasObject
{
 public:
  float width;
  float height;
  float corner_radius;

  RectangleObject(float w, float h);
  void GetBoundingBox(float *w, float *h);
  Bool GetPerimeterPoint(float x1, float y1,
                                 float x2, float y2,
                                 float *x3, float *y3);
  void OnDraw(void);
  void SetSize(float x, float y);
  virtual void SetCornerRadius(float rad); // If > 0, rounded corners

  // Make a copy
  virtual void Copy(RectangleObject& copy);

  int GetNumberOfAttachments(void);
  void GetAttachmentPosition(int attachment, float *x, float *y,
                                     int nth = 0, int no_arcs = 1);
};

/*
 * Text object - text with no visible outline
 *
 */

class TextObject: public RectangleObject
{
 public:

  TextObject(float width, float height);

  void OnDraw(void);
};


/*
 * Ellipse object
 *
 */

class EllipseObject: public CanvasObject
{
 public:
  float width;
  float height;

  EllipseObject(float w, float h);
  void GetBoundingBox(float *w, float *h);
  Bool GetPerimeterPoint(float x1, float y1,
                                 float x2, float y2,
                                 float *x3, float *y3);

  void OnDraw(void);
  void SetSize(float x, float y);

  // Make a copy
  virtual void Copy(EllipseObject& copy);

  int GetNumberOfAttachments(void);
  void GetAttachmentPosition(int attachment, float *x, float *y,
                                     int nth = 0, int no_arcs = 1);
};

/*
 * Circle object
 *
 */

class CircleObject: public EllipseObject
{
 public:
  CircleObject(float w);
  virtual Bool GetPerimeterPoint(float x1, float y1,
                                 float x2, float y2,
                                 float *x3, float *y3);

};

// Arrow styles
#define ARROW_NONE         0
#define ARROW_ONE          1
#define ARROW_BOTH         2
#define ARROW_MIDDLE       3

/*
 * Line object
 *
 */

class LineObject: public CanvasObject
{
 public:
  // These define the segmented line - not to be confused with temporary control
  // points which appear when object is selected (although in this case they'll
  // probably be the same)
  wxList *line_control_points;

  float xpos1, ypos1, xpos2, ypos2;
  Bool linked_up; // Flag used when loading from file

  // Arrow head styles
  int start_style;
  int end_style;
  int middle_style;
  float arrow_length;
  float arrow_width;
  CanvasObject *to;
  CanvasObject *from;
  float actual_text_width;  // Space the text takes up
  float actual_text_height; // (depends on text content unlike nodes)

  int attachment_to;   // Attachment point at one end
  int attachment_from; // Attachment point at other end

  LineObject(void);
  LineObject(wxList *list);
  ~LineObject(void);

  // Called when a connected object has moved, to move the link to
  // correct position
  void OnMoveLink(void);
  void OnMove(float x, float y, float old_x, float old_y);
  void OnDraw(void);
  void OnDrawContents(void);
  void OnErase(void);
  void OnDrawOutline(void);
  void GetBoundingBox(float *w, float *h);
  virtual void SetEnds(float x1, float y1, float x2, float y2);
  virtual void GetEnds(float *x1, float *y1, float *x2, float *y2);
  virtual CanvasObject *GetFrom(void);
  virtual CanvasObject *GetTo(void);
  virtual void SetFrom(CanvasObject *object);
  virtual void SetTo(CanvasObject *object);
  virtual void DrawArrows(void);
  void FormatText(char *s);

  // Straighten verticals and horizontals
  virtual void Straighten(void);

  // Make handle control points
  void MakeControlPoints(void);
  void ResetControlPoints(void);

  // Make a given number of control points
  virtual void MakeLineControlPoints(int n);
  virtual void InsertLineControlPoint(void);
  virtual Bool DeleteLineControlPoint(void);
  virtual void Initialise(void);

  // Override dragging behaviour - don't want to be able to drag lines!
  void OnDragLeft(Bool draw, float x, float y, int keys=0, int attachment = 0);
  void OnBeginDragLeft(float x, float y, int keys=0, int attachment = 0);
  void OnEndDragLeft(float x, float y, int keys=0, int attachment = 0);

  void SetArrowSize(float length, float width);
  void SetStartArrow(int style);
  void SetMiddleArrow(int style);
  void SetEndArrow(int style);
  void Unlink(void);
  void SetAttachments(int from_attach, int to_attach);

  Bool HitTest(float x, float y, int *attachment, float *distance);

  // Make a copy
  virtual void Copy(LineObject& copy);

  virtual void FindNth(CanvasObject *image, int *nth, int *no_arcs);
};

/*
 * Spline object
 *
 */

class SplineObject: public LineObject
{
 public:
  SplineObject(void);
  // Supply a list of initial control points. SplineObject will delete list
  // when object deleted.
  SplineObject(wxList *list);
  ~SplineObject(void);

  virtual void OnDraw(void);

  Bool HitTest(float x, float y, int *attachment, float *distance);

  // Make a copy
  virtual void Copy(SplineObject& copy);
};

/*
 * Control point. These black squares are ordinary canvas objects,
 * so their behaviour may be defined in the same way as usual.
 * However, they contain a pointer to the canvas object they are
 * meant to control.
 */

class ControlPoint: public RectangleObject
{
 public:

  int type;
  float xoffset;
  float yoffset;
  CanvasObject *canvas_object;
  wxCursor *old_cursor;

  ControlPoint(ObjectCanvas *the_canvas, CanvasObject *object, float size, float the_xoffset, float the_yoffset, int the_type);
  ~ControlPoint(void);

  void OnDraw(void);
  void OnErase(void);
  void OnDrawContents(void);
  void OnDragLeft(Bool draw, float x, float y, int keys=0, int attachment = 0);
  void OnBeginDragLeft(float x, float y, int keys=0, int attachment = 0);
  void OnEndDragLeft(float x, float y, int keys=0, int attachment = 0);

  void GetAttachmentPosition(int attachment, float *x, float *y,
                                     int nth = 0, int no_arcs = 1);
  int GetNumberOfAttachments(void);
};

/*
 * Line control point
 *
 */

class LineControlPoint: public ControlPoint
{
 public:

  int type;
  wxPoint *point;  // Line point

  LineControlPoint(ObjectCanvas *the_canvas, CanvasObject *object, float size, float x, float y, int the_type);
  ~LineControlPoint(void);

  void OnDraw(void);
  void OnDragLeft(Bool draw, float x, float y, int keys=0, int attachment = 0);
  void OnBeginDragLeft(float x, float y, int keys=0, int attachment = 0);
  void OnEndDragLeft(float x, float y, int keys=0, int attachment = 0);

  void OnDragRight(Bool draw, float x, float y, int keys=0, int attachment = 0);
  void OnBeginDragRight(float x, float y, int keys=0, int attachment = 0);
  void OnEndDragRight(float x, float y, int keys=0, int attachment = 0);
};

// Drag states
#define NoDragging             0
#define StartDraggingLeft      1
#define ContinueDraggingLeft   2
#define StartDraggingRight     3
#define ContinueDraggingRight  4

/*
 * Object canvas; maintains a list of objects
 *
 */

class ObjectCanvas: public wxCanvas
{
 public:
  Bool quick_edit_mode;
  Bool snap_to_grid;
  float grid_spacing;

  wxList *object_list;
  int DragState;
  float old_drag_x, old_drag_y;  // Previous drag coordinates
  float OutlineStartX;
  float OutlineStartY;

  CanvasObject *DraggedObject;
  int DraggedAttachment;

  ObjectCanvas(wxFrame *frame, int x = -1, int y = -1, int width = -1, int height = -1,
               int style = wxRETAINED);
  ~ObjectCanvas(void);

  void DrawOutline(float x1, float y1, float x2, float y2);

  virtual void OnPaint(void);
  // This sends higher-level events to images or canvas
  virtual void OnEvent(wxEvent& event);

  virtual void OnLeftClick(float x, float y, int keys = 0);
  virtual void OnRightClick(float x, float y, int keys = 0);

  virtual void OnDragLeft(Bool draw, float x, float y, int keys=0); // Erase if draw false
  virtual void OnBeginDragLeft(float x, float y, int keys=0);
  virtual void OnEndDragLeft(float x, float y, int keys=0);

  virtual void OnDragRight(Bool draw, float x, float y, int keys=0); // Erase if draw false
  virtual void OnBeginDragRight(float x, float y, int keys=0);
  virtual void OnEndDragRight(float x, float y, int keys=0);

  virtual void Redraw(void); // Called automatically by default OnPaint handler
  virtual void Clear(void);

  // Add object to end of object list
  virtual void AddObject(CanvasObject *object);

  // Add object to front of object list
  virtual void InsertObject(CanvasObject *object);

  void SetSnapToGrid(Bool snap);
  void SetGridSpacing(float spacing);
  void Snap(float *x, float *y);

  virtual void RemoveObject(CanvasObject *object);
  virtual void RemoveAllObjects(void);
  virtual void ShowAll(Bool show);
  virtual CanvasObject *FindObject(float x, float y, int *attachment);  // Find object for mouse click

};

/*
 * Defines a string plus a position
 *
 */

class CanvasObjectTextLine: public wxObject
{
 public:
   float x;
   float y;
   char *line;
   CanvasObjectTextLine(float the_x, float the_y, char *the_line);
   ~CanvasObjectTextLine(void);
};

/*
 * Utility functions
 *
 */

void CentreText(wxDC *dc, wxList *text, float xpos, float ypos,
                          float width, float height);

wxList *FormatText(wxDC *dc, char *text, float width, float height);
  
void CentreTextNoClipping(wxDC *dc, wxList *text_list,
                              float xpos, float ypos, float width, float height);
void GetCentredTextExtent(wxDC *dc, wxList *text_list,
                              float xpos, float ypos, float width, float height,
                              float *actual_width, float *actual_height);
void DrawFormattedText(wxDC *context, wxList *text_list,
                              float xpos, float ypos, float width, float height);

// Give it a list of points, finds the centre.
void find_polyline_centroid(wxList *points, float *x, float *y);

void check_line_intersection(float x1, float y1, float x2, float y2, 
                             float x3, float y3, float x4, float y4,
                             float *ratio1, float *ratio2);

void find_end_for_polyline(float n, float xvec[], float yvec[], 
                           float x1, float y1, float x2, float y2, float *x3, float *y3);


void find_end_for_box(float width, float height, 
                      float x1, float y1,         // Centre of box (possibly)
                      float x2, float y2,         // other end of line
                      float *x3, float *y3);      // End on box edge

void find_end_for_circle(float radius, 
                         float x1, float y1,  // Centre of circle
                         float x2, float y2,  // Other end of line
                         float *x3, float *y3);

void get_arrow_points(float x1, float y1, float x2, float y2,
                      float length, float width,
                      float *tip_x, float *tip_y,
                      float *side1_x, float *side1_y,
                      float *side2_x, float *side2_y);

extern wxFont *normal_font;
extern wxFont *italic_font;
extern wxFont *swiss_font_4;
extern wxFont *swiss_font_6;
extern wxFont *swiss_font_8;
extern wxFont *swiss_font_10;
extern wxFont *swiss_font_12;
extern wxFont *swiss_font_14;
extern wxFont *swiss_font_18;
extern wxFont *swiss_font_24;
extern wxPen *red_pen;
extern wxPen *cyan_pen;
extern wxPen *green_pen;
extern wxPen *black_pen;

extern wxBrush *blue_brush;
extern wxBrush *green_brush;
extern wxBrush *white_brush;
extern wxBrush *black_brush;
extern wxBrush *cyan_brush;
extern wxBrush *red_brush;

extern wxPen *white_background_pen;
extern wxBrush *transparent_brush;
extern wxBrush *white_background_brush;
extern wxPen *black_foreground_pen;
extern wxPen *black_dashed_pen;

void UpdateListBox(wxListBox *item, wxList *list);
void GraphicsInitialize(void);
wxFont *FontSizeDialog(wxFrame *parent);
wxFont *MatchFont(int point_size);

#endif // hy_graphicsh
