/*
 * File:     wx_dc.cc
 * Purpose:  Device context implementation
 *
 *                       wxWindows 1.40
 * Copyright (c) 1993 Artificial Intelligence Applications Institute,
 *                   The University of Edinburgh
 *
 *                     Author: Julian Smart
 *                       Date: 18-4-93
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose is hereby granted without fee, provided
 * that the above copyright notice, author statement and this permission
 * notice appear in all copies of this software and related documentation.
 *
 * THE SOFTWARE IS PROVIDED "AS-IS" AND WITHOUT WARRANTY OF ANY KIND, EXPRESS,
 * IMPLIED OR OTHERWISE, INCLUDING WITHOUT LIMITATION, ANY WARRANTY OF
 * MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * IN NO EVENT SHALL THE ARTIFICIAL INTELLIGENCE APPLICATIONS INSTITUTE OR THE
 * UNIVERSITY OF EDINBURGH BE LIABLE FOR ANY SPECIAL, INCIDENTAL, INDIRECT OR
 * CONSEQUENTIAL DAMAGES OF ANY KIND, OR ANY DAMAGES WHATSOEVER RESULTING FROM
 * LOSS OF USE, DATA OR PROFITS, WHETHER OR NOT ADVISED OF THE POSSIBILITY OF
 * DAMAGE, AND ON ANY THEORY OF LIABILITY, ARISING OUT OF OR IN CONNECTION WITH
 * THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */

#include <windows.h>
#include <iostream.h>
#include <fstream.h>
#include <math.h>
#include <stdio.h>
#include <stdlib.h>

#ifdef wx_msw
#include <commdlg.h>
#endif
#include <time.h>

#include "common.h"
#include "wx_frame.h"
#include "wx_dc.h"
#include "wx_event.h"
#include "wx_utils.h"
#include "wx_canvs.h"
#include "wx_dialg.h"
#include "wx_main.h"
#include "wx_privt.h"

#ifdef wx_x
#include <netdb.h>
#include <sys/file.h>
#include <unistd.h>
#include <pwd.h>
// For AT&T C++ users
int gethostname(char *host, unsigned const len);
#endif

// Some global printer parameters
char *wx_printer_command = "lpr";
char *wx_preview_command = "ghostview";
char *wx_printer_flags = NULL;
char wx_printer_file[200];
Bool wx_preview = TRUE;
Bool wx_portrait = TRUE;
float wx_printer_scale_x = 1.0;
float wx_printer_scale_y = 1.0;
float wx_printer_translate_x = 0.0;
float wx_printer_translate_y = 0.0;
Bool wx_print_to_file = FALSE;

// Declarations local to this file

#define YSCALE(y) (yorigin - (y))

#ifdef wx_x
Bool XPrinterDialog(void);
#endif

#define     wx_round(a)    (int)((a)+.5)

void wx_draw_open_spline(wxDC *dc, wxSpline *spline);
void wx_draw_open_spline_ps(wxDC *dc, wxSpline *s);

void wx_quadratic_spline(float a1, float b1, float a2, float b2,
                         float a3, float b3, float a4, float b4);
void wx_clear_stack(void);
int wx_spline_pop(float *x1, float *y1, float *x2, float *y2, float *x3,
        float *y3, float *x4, float *y4);
void wx_spline_push(float x1, float y1, float x2, float y2, float x3, float y3,
          float x4, float y4);
static Bool wx_spline_add_point(int x, int y);
static void wx_spline_draw_point_array(wxDC *dc);
wxSpline *wx_make_spline(int x1, int y1, int x2, int y2, int x3, int y3);


// Default constructor
wxDC::wxDC(void)
{
}

/*
 * Create a new dc from an old dc
 *
 */

wxDC::wxDC(wxDC *old_dc)
{
#ifdef wx_msw
  clipping = FALSE;
  dont_delete = FALSE;
  window_ext_x = VIEWPORT_EXTENT;
  window_ext_y = VIEWPORT_EXTENT;
  device = wxDEVICE_WINDOWS;
  old_bitmap = NULL;
  old_pen = NULL;
  old_brush = NULL;
  old_font = NULL;
#endif
#ifdef wx_x
  device = wxDEVICE_PIXMAP;
  current_colour = NULL;
#endif
  canvas = NULL;

  ok = FALSE;
  pstream = NULL;
  title = NULL;
  filename = NULL;

  current_logical_function = -1;
  font = NULL;
  logical_origin_x = 0;
  logical_origin_y = 0;

  device_origin_x = 0;
  device_origin_y = 0;

  logical_scale_x = 1.0;
  logical_scale_y = 1.0;

  user_scale_x = 1.0;
  user_scale_y = 1.0;

  mapping_mode = MM_TEXT;

  current_pen = NULL;
  current_brush = NULL;
  current_background_brush = wxWHITE_BRUSH;
  current_text_foreground = wxBLACK;
  current_text_background = NULL;


#ifdef wx_x
#ifdef wx_xview
  Xv_Screen screen = xv_get(xview_server, SERVER_NTH_SCREEN, 0);
  Xv_Window root_window = xv_get(screen, XV_ROOT);
  Display *dpy = (Display *)xv_get(root_window, XV_DISPLAY);
#endif
#ifdef wx_motif
  Display *dpy = XtDisplay(wxTheApp->topLevel);
#endif

  XGCValues gcvalues;
  gcvalues.foreground = BlackPixel(dpy,
				   DefaultScreen(dpy));
  gcvalues.background = WhitePixel(dpy,
				   DefaultScreen(dpy));
  gcvalues.graphics_exposures = False;
  gcvalues.line_width = 1;
  gc = XCreateGC(dpy, RootWindow(dpy,
		 DefaultScreen(dpy)),
    GCForeground | GCBackground | GCGraphicsExposures | GCLineWidth, 
    &gcvalues);

#ifdef wx_motif
  gcBacking = XCreateGC(dpy, RootWindow(dpy,
		 DefaultScreen(dpy)),
    GCForeground | GCBackground | GCGraphicsExposures | GCLineWidth, 
    &gcvalues);
#endif

  background_pixel = gcvalues.background;
  ok = TRUE;
  current_colour = NULL;
  selected_pixmap = NULL;
#endif
#ifdef wx_msw
  wxWnd *wnd = NULL;
  if (old_dc->canvas)
    wnd = (wxWnd *)old_dc->canvas->handle;

  HDC dc = NULL;
  if (old_dc->cdc)
    dc = old_dc->cdc;
  else
    dc = GetDC(wnd->handle);

  cdc = ::CreateCompatibleDC(dc);
  ok = (cdc != NULL);

  if (!old_dc->cdc)
    ReleaseDC(wnd->handle, dc);
#endif

  Colour = wxColourDisplay();
  SetBrush(wxWHITE_BRUSH);
  SetPen(wxBLACK_PEN);
}

wxDC::wxDC(wxCanvas *the_canvas)
{
  canvas = the_canvas;
#ifdef wx_msw
  device = wxDEVICE_WINDOWS;
#endif
#ifdef wx_x
  device = wxDEVICE_CANVAS;
#endif
  font = NULL;

  logical_origin_x = 0;
  logical_origin_y = 0;

  device_origin_x = 0;
  device_origin_y = 0;

  logical_scale_x = 1.0;
  logical_scale_y = 1.0;

  user_scale_x = 1.0;
  user_scale_y = 1.0;

  mapping_mode = MM_TEXT;

  pstream = NULL;
  title = NULL;
  filename = NULL;
#ifdef wx_x
  XGCValues gcvalues;
  gcvalues.foreground = BlackPixel(the_canvas->display,
				   DefaultScreen(the_canvas->display));
  gcvalues.background = WhitePixel(the_canvas->display,
				   DefaultScreen(the_canvas->display));
  gcvalues.graphics_exposures = False;
  gcvalues.line_width = 1;
  gc = XCreateGC(the_canvas->display, RootWindow(the_canvas->display,
		 DefaultScreen(the_canvas->display)),
    GCForeground | GCBackground | GCGraphicsExposures | GCLineWidth, 
    &gcvalues);
#ifdef wx_motif
  gcBacking = XCreateGC(the_canvas->display, RootWindow(the_canvas->display,
		 DefaultScreen(the_canvas->display)),
    GCForeground | GCBackground | GCGraphicsExposures | GCLineWidth, 
    &gcvalues);
#endif

  background_pixel = gcvalues.background;
  ok = TRUE;
  current_colour = NULL;
#endif
#ifdef wx_msw
  old_bitmap = NULL;
  old_pen = NULL;
  old_brush = NULL;
  old_font = NULL;
  dont_delete = FALSE;
  cdc = NULL;
  clipping = FALSE;
  ok = TRUE;
  window_ext_x = VIEWPORT_EXTENT;
  window_ext_y = VIEWPORT_EXTENT;

  wxWnd *wnd = (wxWnd *)the_canvas->handle;
  HDC dc = GetDC(wnd->handle);
  SetMapMode(MM_TEXT);

  ReleaseDC(wnd->handle, dc);
#endif
  current_logical_function = -1;

  current_pen = NULL;
  current_brush = NULL;
  current_background_brush = wxWHITE_BRUSH;
  current_text_foreground = wxBLACK;
  current_text_background = NULL;
  Colour = wxColourDisplay();
  SetBrush(wxWHITE_BRUSH);
  SetPen(wxBLACK_PEN);
}

wxDC::wxDC(char *driver_name, char *device_name, char *file, Bool interactive)
{
  wx_interactive = interactive;
  canvas = NULL;
  font = NULL;
#ifdef wx_msw
  if (device_name && strcmp(device_name, "PostScript") == 0)
    device = wxDEVICE_EPS;
  else device = wxDEVICE_WINDOWS;

  cdc = NULL;
  old_bitmap = NULL;
  old_pen = NULL;
  old_brush = NULL;
  old_font = NULL;
  dont_delete = FALSE;
  clipping = FALSE;
#endif
#ifdef wx_x
  if (!device_name || strcmp(device_name, "PostScript") == 0)
    device = wxDEVICE_EPS;
  else { ok = FALSE; return; }
#endif

  logical_origin_x = 0;
  logical_origin_y = 0;

  device_origin_x = 0;
  device_origin_y = 0;

  logical_scale_x = 1.0;
  logical_scale_y = 1.0;

  user_scale_x = 1.0;
  user_scale_y = 1.0;

  mapping_mode = MM_TEXT;

  yorigin = 792;  // For EPS output
  min_x = 1000.0;
  min_y = 1000.0;
  max_x = -1000.0;
  max_y = -1000.0;
  title = NULL;
  if (file)
    filename = copystring(file);
  else filename = NULL;

  pstream = NULL;

#ifdef wx_x
  if (interactive)
    ok = XPrinterDialog();
  if (!ok)
    return;

  if (interactive && (wx_preview || (!file && !wx_print_to_file)))
  {
    struct passwd *who = getpwuid(getuid());
    strcpy(wx_printer_file, "/tmp/preview_");
    strcat(wx_printer_file, who->pw_name);
    strcat(wx_printer_file, ".ps");
    file = wx_printer_file;
  }
#endif
  if (device == wxDEVICE_EPS)
  {
    if (!file)
    {
      file = wxFileSelector("Enter PostScript file name", NULL, "", "ps", "*.ps");
      if (!file)
      {
        ok = FALSE;
        return;
      }
      strcpy(wx_printer_file, file);
      ok = TRUE;
    }
    else strcpy(wx_printer_file, file);

    pstream = new ofstream(wx_printer_file);
    if (!pstream)
    {
      (void)wxMessageBox("Error", "Cannot open file!", wxOK);
      ok = FALSE;
      return;
    }
    ok = TRUE;
  }
#ifdef wx_msw
  else
  {
   if (!driver_name || !device_name || !file)
   {
     PRINTDLG pd;
	
     pd.lStructSize = sizeof( PRINTDLG );
     pd.hwndOwner=NULL;
     pd.hDevMode=(HANDLE)NULL;
     pd.hDevNames=(HANDLE)NULL;
     pd.Flags=PD_RETURNDC | PD_NOSELECTION | PD_NOPAGENUMS;
     pd.nFromPage=0;
     pd.nToPage=0;
     pd.nMinPage=0;
     pd.nMaxPage=0;
     pd.nCopies=1;
     pd.hInstance=(HINSTANCE)NULL;

     if ( PrintDlg( &pd ) != 0 )
     {
       cdc = pd.hDC;
       ok = TRUE;
     }
     else
     {
       ok = FALSE;
       return;
     }

     dont_delete = TRUE; // ??? WHY???
   }
   else
   {
     cdc = CreateDC(driver_name, device_name, file, NULL);
     ok = TRUE;
   }
   if (cdc)
   {
     int width = GetDeviceCaps(cdc, VERTRES);
     int height = GetDeviceCaps(cdc, HORZRES);
     SetMapMode(MM_TEXT);
   }
  }
#endif
  current_logical_function = -1;
  current_pen = NULL;
  current_brush = NULL;
  current_background_brush = wxWHITE_BRUSH;
  current_text_foreground = wxBLACK;
  current_text_background = NULL;
  Colour = wxColourDisplay();
#ifdef wx_x
  current_colour = NULL;
#endif
  SetBrush(wxBLACK_BRUSH);
  SetPen(wxBLACK_PEN);
}

wxDC::~wxDC(void)
{
  if (pstream)
    delete pstream;
  if (title)
    delete title;
  if (filename)
    delete filename;
#ifdef wx_msw
  if (cdc)
  {
    SelectOldObjects(cdc);
    DeleteDC(cdc);
  }
#endif
}

#ifdef wx_msw
// This will select current objects out of the DC,
// which is what you have to do before deleting the
// DC.
void wxDC::SelectOldObjects(HDC dc)
{
  if (dc)
  {
    if (old_bitmap)
      ::SelectObject(dc, old_bitmap);
    if (old_pen)
      ::SelectObject(dc, old_pen);
    if (old_brush)
      ::SelectObject(dc, old_brush);
    if (old_font)
      ::SelectObject(dc, old_font);
  }
}
#endif

void wxDC::SetClippingRegion(float cx, float cy, float cw, float ch)
{
  switch (device)
  {
#ifdef wx_x
    case wxDEVICE_CANVAS:
    {
      if (canvas)
      {
        XRectangle rects[1];

        rects[0].x = XLOG2DEV(cx); rects[0].y = YLOG2DEV(cy);
        rects[0].width = XLOG2DEVREL(cw); rects[0].height = YLOG2DEVREL(ch);
        XSetClipRectangles(canvas->display, gc, 0, 0, rects, 1, Unsorted);
#ifdef wx_motif
        if (canvas->is_retained)
	{
          rects[0].x = XLOG2DEV_2(cx); rects[0].y = YLOG2DEV_2(cy);
          rects[0].width = XLOG2DEVREL(cw); rects[0].height = YLOG2DEVREL(ch);
          XSetClipRectangles(canvas->display, gcBacking, 0, 0, rects, 1, Unsorted);
	}
#endif
      }
      break;
    }
#endif
    case wxDEVICE_EPS:
    {
      clipping = TRUE;
      *pstream << "gsave\n";
      *pstream << "newpath\n";
      *pstream << cx << " " << YSCALE(cy) << " moveto\n";
      *pstream << cx+cw << " " << YSCALE(cy) << " lineto\n";
      *pstream << cx+cw << " " << YSCALE(cy + ch) << " lineto\n";
      *pstream << cx << " " << YSCALE(cy + ch) << " lineto\n";
      *pstream << "closepath clip newpath\n";
      break;
    }
#ifdef wx_msw
    case wxDEVICE_WINDOWS:
    {
      clipping = TRUE;
      clip_x1 = (int)cx;
      clip_y1 = (int)cy;
      clip_x2 = (int)(cx + cw);
      clip_y2 = (int)(cy + ch);

      HDC dc = NULL;
      wxWnd *wnd = NULL;
      if (canvas) wnd = (wxWnd *)canvas->handle;
      if (cdc)
        dc = cdc;
      else if (wnd)
        dc = GetDC(wnd->handle);

      DoClipping(dc);

      if (!cdc)
        ReleaseDC(wnd->handle, dc);
      break;
    }
#endif
  }
}

#ifdef wx_msw
void wxDC::DoClipping(HDC dc)
{
  if (clipping && dc)
  {
    int x_off = 0;
    int y_off = 0;
    if (canvas)
    {
      wxWnd *wnd = (wxWnd *)canvas->handle;

      wnd->CalcScrolledPosition(0, 0, &x_off, &y_off);
    }
//    HRGN rgn = CreateRectRgn(XLOG2DEV(clip_x1 + x_off), YLOG2DEV(clip_y1 + y_off),
//                          XLOG2DEV(clip_x2 + x_off), YLOG2DEV(clip_y2 + y_off));

//    SelectClipRgn(dc, rgn);
//    DeleteObject(rgn);
    IntersectClipRect(dc, XLOG2DEV(clip_x1 + x_off), YLOG2DEV(clip_y1 + y_off),
                          XLOG2DEV(clip_x2 + x_off), YLOG2DEV(clip_y2 + y_off));
  }
}
#endif

void wxDC::DestroyClippingRegion(void)
{
  switch (device)
  {
#ifdef wx_x
    case wxDEVICE_CANVAS:
    {
      if (canvas)
      {
        XGCValues gc_val;
        gc_val.clip_mask = None;
        XChangeGC(canvas->display, gc, GCClipMask, &gc_val);
#ifdef wx_motif
        if (canvas->is_retained)
	{
          XChangeGC(canvas->display, gcBacking, GCClipMask, &gc_val);
	}
#endif
#ifdef wx_xview
        if (canvas->xrects)
          XSetClipRectangles(canvas->display, gc, 0, 0, canvas->xrects->rect_array,
                           canvas->xrects->count, Unsorted);
#endif
      }
      break;
    }
#endif
    case wxDEVICE_EPS:
    {
      if (clipping)
      {
        clipping = FALSE;
        *pstream << "grestore\n";
      }
      break;
    }
#ifdef wx_msw
    case wxDEVICE_WINDOWS:
    {
      HDC dc = NULL;
      wxWnd *wnd = NULL;
      if (canvas)
        wnd = (wxWnd *)canvas->handle;
      if (cdc)
        dc = cdc;
      else if (wnd)
        dc = GetDC(wnd->handle);

      if (clipping && dc)
      {
//        SelectClipRgn(dc, NULL);
        HRGN rgn = CreateRectRgn(0, 0, 32000, 32000);
        SelectClipRgn(dc, rgn);
        DeleteObject(rgn);
      }
      clipping = FALSE;

      if (!cdc)
        ReleaseDC(wnd->handle, dc);
      break;
    }
#endif
  }
}

void wxDC::Clear(void)
{
  switch (device)
  {
#ifdef wx_x
    case wxDEVICE_CANVAS:
    {
      if (canvas)
      {
#ifdef wx_xview
        // If we're not in the middle of a paint event...
        if (!canvas->xrects || !canvas->is_retained)
          XClearWindow(canvas->display, canvas->xwindow);
#endif
#ifdef wx_motif
        int w, h;
        canvas->GetSize(&w, &h);

        if (canvas->hScroll)
          w = canvas->hExtent;
        if (canvas->vScroll)
          h = canvas->vExtent;

        if (canvas->is_retained)
        {
          w = canvas->pixmapWidth;
          h = canvas->pixmapHeight;
        }

        SetBrush(current_background_brush);
        XFillRectangle(canvas->display, canvas->xwindow, gc, 0, 0, w, h);

        if (canvas->is_retained)
          XFillRectangle(canvas->display, canvas->backingPixmap, gcBacking, 0, 0, w, h);
#endif
      }
      break;
    }
#endif
    case wxDEVICE_EPS:
    {
      break;
    }
#ifdef wx_msw
    case wxDEVICE_WINDOWS:
    {
      if (canvas)
      {
        HDC dc = NULL;
        wxWnd *wnd = (wxWnd *)canvas->handle;
        if (cdc)
          dc = cdc;
        else
          dc = GetDC(wnd->handle);

        RECT rect;

        GetClientRect(wnd->handle, &rect);
        int mode = ::SetMapMode(dc, MM_TEXT);

        DWORD colour = GetBkColor(dc);
        HBRUSH brush = CreateSolidBrush(colour);
        FillRect(dc, &rect, brush);
        DeleteObject(brush);

        ::SetMapMode(dc, MM_ANISOTROPIC);
        SetViewportExtEx(dc, VIEWPORT_EXTENT, VIEWPORT_EXTENT, NULL);
        SetWindowExtEx(dc, window_ext_x, window_ext_y, NULL);

        if (!cdc)
          ReleaseDC(wnd->handle, dc);
      }
      break;
    }
#endif
  }
}

void wxDC::DrawLine(float x1, float y1, float x2, float y2)
{
  switch (device)
  {
#ifdef wx_x
    case wxDEVICE_CANVAS:
    {
      if (canvas)
      {
        if (current_pen)
          SetPen(current_pen);
        XDrawLine(canvas->display, canvas->xwindow, gc, XLOG2DEV(x1), YLOG2DEV(y1),
                  XLOG2DEV(x2), YLOG2DEV(y2));
#ifdef wx_motif
        if (canvas->is_retained)
          XDrawLine(canvas->display, canvas->backingPixmap, gcBacking, XLOG2DEV_2(x1), YLOG2DEV_2(y1),
                    XLOG2DEV_2(x2), YLOG2DEV_2(y2));
#endif
      }
      break;
    }
#endif
    case wxDEVICE_EPS:
    {
      if (current_pen) SetPen(current_pen);
      *pstream << "newpath\n";
      *pstream << x1 << " " << YSCALE(y1) << " moveto\n";
      *pstream << x2 << " " << YSCALE(y2) << " lineto\n";
      *pstream << "stroke\n";
      CalcBoundingBox(x1, YSCALE(y1));
      CalcBoundingBox(x2, YSCALE(y2));
      break;
    }
#ifdef wx_msw
    case wxDEVICE_WINDOWS:
    {
      int xx1 = (int)x1;
      int yy1 = (int)y1;
      int xx2 = (int)x2;
      int yy2 = (int)y2;

      wxWnd *wnd = NULL;
      if (canvas)
      {
        wnd = (wxWnd *)canvas->handle;
        wnd->CalcScrolledPosition((int)x1, (int)y1, &xx1, &yy1);
        wnd->CalcScrolledPosition((int)x2, (int)y2, &xx2, &yy2);
      }
      HDC dc = NULL;
      if (cdc)
        dc = cdc;
      else if (wnd)
        dc = GetDC(wnd->handle);

      (void)MoveToEx(dc, XLOG2DEV(xx1), YLOG2DEV(yy1), NULL);
      (void)LineTo(dc, XLOG2DEV(xx2), YLOG2DEV(yy2));

      if (!cdc)
        ReleaseDC(wnd->handle, dc);
      break;
    }
#endif
  }
}

void wxDC::DrawPoint(float x, float y)
{
  switch (device)
  {
#ifdef wx_x
    case wxDEVICE_CANVAS:
    {
      if (canvas)
      {
        if (current_pen)
          SetPen(current_pen);

        XDrawPoint(canvas->display, canvas->xwindow, gc, XLOG2DEV(x), YLOG2DEV(y));
#ifdef wx_motif
        if (canvas->is_retained)
          XDrawPoint(canvas->display, canvas->backingPixmap, gcBacking, XLOG2DEV_2(x), YLOG2DEV_2(y));
#endif
      }
      break;
    }
#endif
    case wxDEVICE_EPS:
    {
      if (current_pen) SetPen(current_pen);
      *pstream << "newpath\n";
      *pstream << x << " " << YSCALE(y) << " moveto\n";
      *pstream << x << " " << YSCALE(y) << " lineto\n";
      *pstream << "stroke\n";
      CalcBoundingBox(x, YSCALE(y));
      break;
    }
#ifdef wx_msw
    case wxDEVICE_WINDOWS:
    {
      wxWnd *wnd = NULL;

      int xx1 = (int)x;
      int yy1 = (int)y;
      if (canvas)
      {
        wnd = (wxWnd *)canvas->handle;
        wnd->CalcScrolledPosition((int)x, (int)y, &xx1, &yy1);
      }

      HDC dc = NULL;
      if (cdc)
        dc = cdc;
      else
        dc = GetDC(wnd->handle);

      (void)MoveTo(dc, XLOG2DEV(xx1), YLOG2DEV(yy1));
      (void)LineTo(dc, XLOG2DEV(xx1), YLOG2DEV(yy1));

      if (!cdc)
        ReleaseDC(wnd->handle, dc);
      break;
    }
#endif
  }
}

void wxDC::DrawPolygon(int n, wxPoint points[], float xoffset, float yoffset)
{
  switch (device)
  {
#ifdef wx_x
    case wxDEVICE_CANVAS:
    {
      if (canvas)
      {
        XPoint *xpoints1 = new XPoint[n];
        XPoint *xpoints2 = new XPoint[n];
        int i;
        for (i = 0; i < n; i++)
        {
          xpoints1[i].x = XLOG2DEV(points[i].x + xoffset);
          xpoints1[i].y = YLOG2DEV(points[i].y + yoffset);
          xpoints2[i].x = XLOG2DEV_2(points[i].x + xoffset);
          xpoints2[i].y = YLOG2DEV_2(points[i].y + yoffset);
        }

        if (current_brush && current_brush->style != wxTRANSPARENT)
        {
          SetBrush(current_brush);
          XFillPolygon(canvas->display, canvas->xwindow, gc, xpoints1, n, Convex, 0);
#ifdef wx_motif
          if (canvas->is_retained)
            XFillPolygon(canvas->display, canvas->backingPixmap, gcBacking, xpoints2, n, Convex, 0);
#endif
        }

        if (current_pen && current_pen->style != wxTRANSPARENT)
        {
          SetPen(current_pen);
          XDrawLines(canvas->display, canvas->xwindow, gc, xpoints1, n, 0);
#ifdef wx_motif
          if (canvas->is_retained)
            XDrawLines(canvas->display, canvas->backingPixmap, gcBacking, xpoints2, n, 0);
#endif
        }

        delete[] xpoints1;
        delete[] xpoints2;
      }
      break;
    }
#endif
    case wxDEVICE_EPS:
    {
      if (n > 0)
      {
        if (current_brush && current_brush->style != wxTRANSPARENT)
        {
          SetBrush(current_brush);
          *pstream << "newpath\n";

          float xx = points[0].x + xoffset;
          float yy = YSCALE(points[0].y + yoffset);
          *pstream << xx << " " << yy << " moveto\n";
          CalcBoundingBox(xx, yy);

          int i;
          for (i = 1; i < n; i++)
	  {
            xx = points[i].x + xoffset;
            yy = YSCALE(points[i].y + yoffset);
            *pstream << xx << " " << yy << " lineto\n";
            CalcBoundingBox(xx, yy);
	  }
          *pstream << "fill\n";
	}

        if (current_pen && current_pen->style != wxTRANSPARENT)
        {
          SetPen(current_pen);
          *pstream << "newpath\n";

          float xx = points[0].x + xoffset;
          float yy = YSCALE(points[0].y + yoffset);
          *pstream << xx << " " << yy << " moveto\n";
          CalcBoundingBox(xx, yy);

          int i;
          for (i = 1; i < n; i++)
	  {
            xx = points[i].x + xoffset;
            yy = YSCALE(points[i].y + yoffset);
            *pstream << xx << " " << yy << " lineto\n";
            CalcBoundingBox(xx, yy);
	  }
          *pstream << "stroke\n";
	}

      }
      break;
    }
#ifdef wx_msw
    case wxDEVICE_WINDOWS:
    {
      int xoffset1 = 0;
      int yoffset1 = 0;
      wxWnd *wnd = NULL;

      if (canvas)
      {
        wnd = (wxWnd *)canvas->handle;
        wnd->CalcScrolledPosition(0, 0, &xoffset1, &yoffset1);
      }

      xoffset1 += (int)xoffset; yoffset1 += (int)yoffset;

      HDC dc = NULL;
      if (cdc)
        dc = cdc;
      else
        dc = GetDC(wnd->handle);

      POINT *cpoints = new POINT[n];
      int i;
      for (i = 0; i < n; i++)
      {
        cpoints[i].x = (int)(XLOG2DEV(points[i].x + xoffset1));
        cpoints[i].y = (int)(YLOG2DEV(points[i].y + yoffset1));
      }

      (void)Polygon(dc, cpoints, n);

      if (!cdc)
        ReleaseDC(wnd->handle, dc);

      delete[] cpoints;
      break;
    }
#endif
  }
}

void wxDC::DrawPolygon(wxList *list, float xoffset, float yoffset)
{
  switch (device)
  {
#ifdef wx_x
    case wxDEVICE_CANVAS:
    {
      if (canvas)
      {
        if (current_pen)
          SetPen(current_pen);

        int n = list->Number();
        wxNode *current = list->First();
    
        XPoint *xpoints1 = new XPoint[n];
        XPoint *xpoints2 = new XPoint[n];
        int i;
        for (i = 0; i < n; i++)
        {
          wxPoint *point = (wxPoint *)current->Data();
          xpoints1[i].x = XLOG2DEV(point->x + xoffset);
          xpoints1[i].y = YLOG2DEV(point->y + yoffset);
          xpoints2[i].x = XLOG2DEV_2(point->x + xoffset);
          xpoints2[i].y = YLOG2DEV_2(point->y + yoffset);
          current = current->Next();
        }

        if (current_brush && current_brush->style != wxTRANSPARENT)
        {
          SetBrush(current_brush);
          XFillPolygon(canvas->display, canvas->xwindow, gc, xpoints1, n, Convex, 0);
#ifdef wx_motif
          if (canvas->is_retained)
            XFillPolygon(canvas->display, canvas->backingPixmap, gcBacking, xpoints2, n, Convex, 0);
#endif
        }

        if (current_pen && current_pen->style != wxTRANSPARENT)
        {
          SetPen(current_pen);
          XDrawLines(canvas->display, canvas->xwindow, gc, xpoints1, n, 0);
#ifdef wx_motif
          if (canvas->is_retained)
            XDrawLines(canvas->display, canvas->backingPixmap, gcBacking, xpoints2, n, 0);
#endif
        }

        delete[] xpoints1;
        delete[] xpoints2;
      }
      break;
    }
#endif
    case wxDEVICE_EPS:
    {
      int n = list->Number();

      if (n > 0)
      {
        if (current_brush && current_brush->style != wxTRANSPARENT)
        {
          SetBrush(current_brush);

          *pstream << "newpath\n";

          wxNode *node = list->First();
          wxPoint *point = (wxPoint *)node->Data();
          float xx = point->x + xoffset;
          float yy = YSCALE(point->y + yoffset);

          *pstream << xx << " " << yy << " moveto\n";
          CalcBoundingBox(xx, yy);

          node = node->Next();

          while (node)
          {
            point = (wxPoint *)node->Data();
            xx = point->x + xoffset;
            yy = YSCALE(point->y + yoffset);
            *pstream << xx << " " << yy << " lineto\n";
            CalcBoundingBox(xx, yy);
            node = node->Next();
          }
          *pstream << "closepath\nfill\n";
        }

        if (current_pen && current_pen->style != wxTRANSPARENT)
        {
          SetPen(current_pen);

          *pstream << "newpath\n";

          wxNode *node = list->First();
          wxPoint *point = (wxPoint *)node->Data();

          float xx = point->x + xoffset;
          float yy = YSCALE(point->y + yoffset);
          *pstream << xx << " " << yy << " moveto\n";
          CalcBoundingBox(xx, yy);

          node = node->Next();

          while (node)
          {
            point = (wxPoint *)node->Data();
            xx = point->x + xoffset;
            yy = YSCALE(point->y + yoffset);
            *pstream << xx << " " << yy << " lineto\n";
            CalcBoundingBox(xx, yy);

            node = node->Next();
          }
          *pstream << "closepath\nstroke\n";
        }
      }
      break;
    }
#ifdef wx_msw
    case wxDEVICE_WINDOWS:
    {
      int xoffset1 = 0;
      int yoffset1 = 0;
      wxWnd *wnd = NULL;

      if (canvas)
      {
        wnd = (wxWnd *)canvas->handle;
        wnd->CalcScrolledPosition(0, 0, &xoffset1, &yoffset1);
      }
      xoffset1 += (int)xoffset; yoffset1 += (int)yoffset;

      HDC dc = NULL;
      if (cdc)
        dc = cdc;
      else
        dc = GetDC(wnd->handle);

      POINT *cpoints = new POINT[list->Number()];
      int i;
      wxNode *node = list->First();
      for (i = 0; i < list->Number(); i++)
      {
        wxPoint *point = (wxPoint *)node->Data();
        cpoints[i].x = (int)(XLOG2DEV(point->x + xoffset1));
        cpoints[i].y = (int)(YLOG2DEV(point->y + yoffset1));
        node = node->Next();
      }

      (void)Polygon(dc, cpoints, list->Number());

      if (!cdc)
        ReleaseDC(wnd->handle, dc);

      delete[] cpoints;
      break;
    }
#endif
  }
}

void wxDC::DrawLines(int n, wxPoint points[], float xoffset, float yoffset)
{
  switch (device)
  {
#ifdef wx_x
    case wxDEVICE_CANVAS:
    {
      if (canvas)
      {
        XPoint *xpoints1 = new XPoint[n];
        XPoint *xpoints2 = new XPoint[n];
        int i;
        for (i = 0; i < n; i++)
        {
          xpoints1[i].x = XLOG2DEV(points[i].x + xoffset);
          xpoints1[i].y = YLOG2DEV(points[i].y + yoffset);
          xpoints2[i].x = XLOG2DEV_2(points[i].x + xoffset);
          xpoints2[i].y = YLOG2DEV_2(points[i].y + yoffset);
        }

        if (current_pen && current_pen->style != wxTRANSPARENT)
        {
          SetPen(current_pen);
          XDrawLines(canvas->display, canvas->xwindow, gc, xpoints1, n, 0);
#ifdef wx_motif
          if (canvas->is_retained)
            XDrawLines(canvas->display, canvas->backingPixmap, gcBacking, xpoints2, n, 0);
#endif
        }

        delete[] xpoints1;
        delete[] xpoints2;
      }
      break;
    }
#endif
    case wxDEVICE_EPS:
    {
      if (n > 0)
      {
        if (current_pen) SetPen(current_pen);

        *pstream << "newpath\n";

        float xx = points[0].x + xoffset;
        float yy = YSCALE(points[0].y + yoffset);
        *pstream << xx << " " << yy << " moveto\n";
        CalcBoundingBox(xx, yy);

        int i;
        for (i = 1; i < n; i++)
	{
          xx = points[i].x + xoffset;
          yy = YSCALE(points[i].y + yoffset);
          *pstream << xx << " " << yy << " lineto\n";
          CalcBoundingBox(xx, yy);
	}
        *pstream << "stroke\n";

      }
      break;
    }
#ifdef wx_msw
   case wxDEVICE_WINDOWS:
   {
      int xoffset1 = 0;
      int yoffset1 = 0;
      wxWnd *wnd = NULL;

      if (canvas)
      {
        wnd = (wxWnd *)canvas->handle;
        wnd->CalcScrolledPosition(0, 0, &xoffset1, &yoffset1);
      }
      xoffset1 += (int)xoffset; yoffset1 += (int)yoffset;

      HDC dc = NULL;
      if (cdc)
        dc = cdc;
      else
        dc = GetDC(wnd->handle);

      POINT *cpoints = new POINT[n];
      int i;
      for (i = 0; i < n; i++)
      {
        cpoints[i].x = (int)(XLOG2DEV(points[i].x + xoffset1));
        cpoints[i].y = (int)(YLOG2DEV(points[i].y + yoffset1));
      }

      (void)Polyline(dc, cpoints, n);

      if (!cdc)
        ReleaseDC(wnd->handle, dc);

      delete[] cpoints;
      break;
    }
#endif
  }
}

void wxDC::DrawLines(wxList *list, float xoffset, float yoffset)
{
  switch (device)
  {
#ifdef wx_x
    case wxDEVICE_CANVAS:
    {

      if (canvas)
      {
        if (current_pen)
          SetPen(current_pen);

        int n = list->Number();
        wxNode *current = list->First();

        XPoint *xpoints1 = new XPoint[n];
        XPoint *xpoints2 = new XPoint[n];
        int i;
        for (i = 0; i < n; i++)
        {
          wxPoint *point = (wxPoint *)current->Data();
          xpoints1[i].x = XLOG2DEV(point->x + xoffset);
          xpoints1[i].y = YLOG2DEV(point->y + yoffset);
          xpoints2[i].x = XLOG2DEV_2(point->x + xoffset);
          xpoints2[i].y = YLOG2DEV_2(point->y + yoffset);
          current = current->Next();
        }

        XDrawLines(canvas->display, canvas->xwindow, gc, xpoints1, n, 0);
#ifdef wx_motif
        if (canvas->is_retained)
          XDrawLines(canvas->display, canvas->backingPixmap, gcBacking, xpoints2, n, 0);
#endif
        delete[] xpoints1;
        delete[] xpoints2;
      }
      break;
    }
#endif
    case wxDEVICE_EPS:
    {
      int n = list->Number();

      if (n > 0)
      {
        if (current_pen) SetPen(current_pen);
        *pstream << "newpath\n";

        wxNode *node = list->First();
        wxPoint *point = (wxPoint *)node->Data();

        float xx = point->x + xoffset;
        float yy = YSCALE(point->y + yoffset);
        CalcBoundingBox(xx, yy);

        *pstream << xx << " " << yy << " moveto\n";
        node = node->Next();

        while (node)
	{
          point = (wxPoint *)node->Data();

          xx = point->x + xoffset;
          yy = YSCALE(point->y + yoffset);
          CalcBoundingBox(xx, yy);

          *pstream << xx << " " << yy << " lineto\n";
          node = node->Next();
	}
        *pstream << "stroke\n";
      }
      break;
    }
#ifdef wx_msw
    case wxDEVICE_WINDOWS:
    {
      int xoffset1 = 0;
      int yoffset1 = 0;
      wxWnd *wnd = NULL;

      if (canvas)
      {
        wnd = (wxWnd *)canvas->handle;
        wnd->CalcScrolledPosition(0, 0, &xoffset1, &yoffset1);
      }
      xoffset1 += (int)xoffset; yoffset1 += (int)yoffset;

      HDC dc = NULL;
      if (cdc)
        dc = cdc;
      else
        dc = GetDC(wnd->handle);

      POINT *cpoints = new POINT[list->Number()];
      int i;
      wxNode *node = list->First();
      for (i = 0; i < list->Number(); i++)
      {
        wxPoint *point = (wxPoint *)node->Data();
        cpoints[i].x = (int)(XLOG2DEV(point->x + xoffset1));
        cpoints[i].y = (int)(YLOG2DEV(point->y + yoffset1));
        node = node->Next();
      }

      (void)Polyline(dc, cpoints, list->Number());

      if (!cdc)
        ReleaseDC(wnd->handle, dc);

      delete[] cpoints;
      break;
    }
#endif
  }
}

void wxDC::DrawRectangle(float x, float y, float width, float height)
{
  switch (device)
  {
#ifdef wx_x
    case wxDEVICE_CANVAS:
    {
      if (canvas)
      {
        if (current_brush && current_brush->style != wxTRANSPARENT)
        {
          SetBrush(current_brush);
          XFillRectangle(canvas->display, canvas->xwindow, gc, XLOG2DEV(x), YLOG2DEV(y),
                         XLOG2DEVREL(width), YLOG2DEVREL(height));
#ifdef wx_motif
          if (canvas->is_retained)
            XFillRectangle(canvas->display, canvas->backingPixmap, gcBacking, XLOG2DEV_2(x), YLOG2DEV_2(y),
                           XLOG2DEVREL(width), YLOG2DEVREL(height));
#endif
        }

        if (current_pen && current_pen->style != wxTRANSPARENT)
        {
          SetPen(current_pen);
          XDrawRectangle(canvas->display, canvas->xwindow, gc, XLOG2DEV(x), YLOG2DEV(y),
                         XLOG2DEVREL(width), YLOG2DEVREL(height));
#ifdef wx_motif
          if (canvas->is_retained)
            XDrawRectangle(canvas->display, canvas->backingPixmap, gcBacking, XLOG2DEV_2(x), YLOG2DEV_2(y),
                         XLOG2DEVREL(width), YLOG2DEVREL(height));
#endif
        }
      }
      break;
    }
#endif
    case wxDEVICE_EPS:
    {
      if (current_brush && current_brush->style != wxTRANSPARENT)
      {
        SetBrush(current_brush);
      
        *pstream << "newpath\n";
        *pstream << x << " " << YSCALE(y) << " moveto\n";
        *pstream << x+width << " " << YSCALE(y) << " lineto\n";
        *pstream << x+width << " " << YSCALE(y+height) << " lineto\n";
        *pstream << x << " " << YSCALE(y+height) << " lineto\n";
        *pstream << "closepath\n";
        *pstream << "fill\n";

        CalcBoundingBox(x, YSCALE(y));
        CalcBoundingBox(x+width, YSCALE(y+height));
      }
      if (current_pen && current_pen->style != wxTRANSPARENT)
      {
        SetPen(current_pen);
      
        *pstream << "newpath\n";
        *pstream << x << " " << YSCALE(y) << " moveto\n";
        *pstream << x+width << " " << YSCALE(y) << " lineto\n";
        *pstream << x+width << " " << YSCALE(y+height) << " lineto\n";
        *pstream << x << " " << YSCALE(y+height) << " lineto\n";
        *pstream << "closepath\n";
        *pstream << "stroke\n";

        CalcBoundingBox(x, YSCALE(y));
        CalcBoundingBox(x+width, YSCALE(y+height));
      }
      break;
    }
#ifdef wx_msw
    case wxDEVICE_WINDOWS:
    {
      wxWnd *wnd = NULL;

      int x1 = (int)x;
      int y1 = (int)y;
      int x2 = (int)(x+width);
      int y2 = (int)(y+height);

      if (canvas)
      {
        wnd = (wxWnd *)canvas->handle;
        wnd->CalcScrolledPosition((int)x, (int)y, &x1, &y1);
        wnd->CalcScrolledPosition((int)(x+width), (int)(y+height), &x2, &y2);
      }

      HDC dc = NULL;
      if (cdc)
        dc = cdc;
      else
        dc = GetDC(wnd->handle);

      (void)Rectangle(dc, XLOG2DEV(x1), YLOG2DEV(y1), XLOG2DEV(x2), YLOG2DEV(y2));

      if (!cdc)
        ReleaseDC(wnd->handle, dc);
      break;
    }
#endif
  }
}

void wxDC::DrawRoundedRectangle(float x, float y, float width, float height, float radius)
{
  switch (device)
  {
#ifdef wx_x
    case wxDEVICE_CANVAS:
    {
      if (canvas)
      {
        float rwidth = radius*2;
        float rheight = rwidth;

        if (current_brush && current_brush->style != wxTRANSPARENT)
        {
          SetBrush(current_brush);
          XFillRectangle(canvas->display, canvas->xwindow, gc, 
                XLOG2DEV(x+radius), YLOG2DEV(y), 
                XLOG2DEVREL(width - rwidth), YLOG2DEVREL(height));
          XFillRectangle(canvas->display, canvas->xwindow, gc, 
                XLOG2DEV(x), YLOG2DEV(y+radius),
                XLOG2DEVREL(width), YLOG2DEVREL(height - rheight));

          XFillArc(canvas->display, canvas->xwindow, gc, 
                XLOG2DEV(x), YLOG2DEV(y), 
                XLOG2DEVREL(rwidth), YLOG2DEVREL(rheight), 90*64, 90*64);
          XFillArc(canvas->display, canvas->xwindow, gc, 
                XLOG2DEV(x+width - rwidth), YLOG2DEV(y), 
                XLOG2DEVREL(rwidth), YLOG2DEVREL(rheight), 0, 90*64);
          XFillArc(canvas->display, canvas->xwindow, gc, 
                XLOG2DEV(x+width - rwidth), 
                YLOG2DEV(y+height - rheight),
    	        XLOG2DEVREL(rwidth), YLOG2DEVREL(rheight), 270*64, 90*64);
          XFillArc(canvas->display, canvas->xwindow, gc, 
                XLOG2DEV(x), YLOG2DEV(y+height - rheight),
	        XLOG2DEVREL(rwidth), YLOG2DEVREL(rheight), 180*64, 90*64);
#ifdef wx_motif
          if (canvas->is_retained)
	  {
          XFillRectangle(canvas->display, canvas->backingPixmap, gcBacking, 
                XLOG2DEV_2(x+radius), YLOG2DEV_2(y), 
                XLOG2DEVREL(width - rwidth), YLOG2DEVREL(height));
          XFillRectangle(canvas->display, canvas->backingPixmap, gcBacking, 
                XLOG2DEV_2(x), YLOG2DEV_2(y+radius),
                XLOG2DEVREL(width), YLOG2DEVREL(height - rheight));

          XFillArc(canvas->display, canvas->backingPixmap, gcBacking, 
                XLOG2DEV_2(x), YLOG2DEV_2(y), 
                XLOG2DEVREL(rwidth), YLOG2DEVREL(rheight), 90*64, 90*64);
          XFillArc(canvas->display, canvas->backingPixmap, gcBacking, 
                XLOG2DEV_2(x+width - rwidth), YLOG2DEV_2(y), 
                XLOG2DEVREL(rwidth), YLOG2DEVREL(rheight), 0, 90*64);
          XFillArc(canvas->display, canvas->backingPixmap, gcBacking, 
                XLOG2DEV_2(x+width - rwidth), 
                YLOG2DEV_2(y+height - rheight),
    	        XLOG2DEVREL(rwidth), YLOG2DEVREL(rheight), 270*64, 90*64);
          XFillArc(canvas->display, canvas->backingPixmap, gcBacking, 
                XLOG2DEV_2(x), YLOG2DEV_2(y+height - rheight),
	        XLOG2DEVREL(rwidth), YLOG2DEVREL(rheight), 180*64, 90*64);
	  }
#endif
        }

        if (current_pen && current_pen->style != wxTRANSPARENT)
        {
          SetPen(current_pen);
          XDrawLine(canvas->display, canvas->xwindow, gc, 
               XLOG2DEV(x+radius), YLOG2DEV(y),
               XLOG2DEV(x+width-radius), YLOG2DEV(y));
          XDrawLine(canvas->display, canvas->xwindow, gc, 
               XLOG2DEV(x+radius), YLOG2DEV(y+height), 
               XLOG2DEV(x+width-radius), YLOG2DEV(y+height));

          XDrawLine(canvas->display, canvas->xwindow, gc, 
               XLOG2DEV(x), YLOG2DEV(y+radius),
               XLOG2DEV(x), YLOG2DEV(y+height-radius));
          XDrawLine(canvas->display, canvas->xwindow, gc, 
               XLOG2DEV(x+width), YLOG2DEV(y+radius),
               XLOG2DEV(x+width), YLOG2DEV(y+height-radius));
          XDrawArc(canvas->display, canvas->xwindow, gc, 
                XLOG2DEV(x), YLOG2DEV(y), 
                XLOG2DEVREL(rwidth), YLOG2DEVREL(rheight), 90*64, 90*64);
          XDrawArc(canvas->display, canvas->xwindow, gc, 
                XLOG2DEV(x+width - rwidth), YLOG2DEV(y), 
                XLOG2DEVREL(rwidth), YLOG2DEVREL(rheight), 0, 90*64);
          XDrawArc(canvas->display, canvas->xwindow, gc, 
                XLOG2DEV(x+width - rwidth), 
                YLOG2DEV(y+height - rheight),
    	        XLOG2DEVREL(rwidth), YLOG2DEVREL(rheight), 270*64, 90*64);
          XDrawArc(canvas->display, canvas->xwindow, gc, 
                XLOG2DEV(x), YLOG2DEV(y+height - rheight),
	        XLOG2DEVREL(rwidth), YLOG2DEVREL(rheight), 180*64, 90*64);
#ifdef wx_motif
          if (canvas->is_retained)
	  {
          XDrawLine(canvas->display, canvas->backingPixmap, gcBacking, 
               XLOG2DEV_2(x+radius), YLOG2DEV_2(y),
               XLOG2DEV_2(x+width-radius), YLOG2DEV_2(y));
          XDrawLine(canvas->display, canvas->backingPixmap, gcBacking, 
               XLOG2DEV_2(x+radius), YLOG2DEV_2(y+height), 
               XLOG2DEV_2(x+width-radius), YLOG2DEV_2(y+height));

          XDrawLine(canvas->display, canvas->backingPixmap, gcBacking, 
               XLOG2DEV_2(x), YLOG2DEV_2(y+radius),
               XLOG2DEV_2(x), YLOG2DEV_2(y+height-radius));
          XDrawLine(canvas->display, canvas->backingPixmap, gcBacking, 
               XLOG2DEV(x+width), YLOG2DEV_2(y+radius),
               XLOG2DEV_2(x+width), YLOG2DEV_2(y+height-radius));
          XDrawArc(canvas->display, canvas->backingPixmap, gcBacking, 
                XLOG2DEV_2(x), YLOG2DEV_2(y), 
                XLOG2DEVREL(rwidth), YLOG2DEVREL(rheight), 90*64, 90*64);
          XDrawArc(canvas->display, canvas->backingPixmap, gcBacking, 
                XLOG2DEV_2(x+width - rwidth), YLOG2DEV_2(y), 
                XLOG2DEVREL(rwidth), YLOG2DEVREL(rheight), 0, 90*64);
          XDrawArc(canvas->display, canvas->backingPixmap, gcBacking, 
                XLOG2DEV_2(x+width - rwidth), 
                YLOG2DEV_2(y+height - rheight),
    	        XLOG2DEVREL(rwidth), YLOG2DEVREL(rheight), 270*64, 90*64);
          XDrawArc(canvas->display, canvas->backingPixmap, gcBacking, 
                XLOG2DEV_2(x), YLOG2DEV_2(y+height - rheight),
	        XLOG2DEVREL(rwidth), YLOG2DEVREL(rheight), 180*64, 90*64);
	  }
#endif
        }
      }
      break;
    }
#endif
    case wxDEVICE_EPS:
    {
      if (current_brush && current_brush->style != wxTRANSPARENT)
      {
        SetBrush(current_brush);
        // Draw rectangle anticlockwise
        *pstream << "newpath\n";
        *pstream << x+radius << " " << YSCALE(y+radius) << " " << radius << " 90 180 arc\n";

        *pstream << x << " " << YSCALE(y + radius) << " moveto\n";

        *pstream << x+radius << " " << YSCALE(y+height-radius) << " " << radius << " 180 270 arc\n";
        *pstream << x+width-radius << " " << YSCALE(y+height) << " lineto\n";

        *pstream << x+width-radius << " " << YSCALE(y+height-radius) << " " << radius << " 270 0 arc\n";
        *pstream << x+width << " " << YSCALE(y+radius) << " lineto\n";

        *pstream << x+width-radius << " " << YSCALE(y+radius) << " " << radius << " 0 90 arc\n";

        *pstream << x+radius << " " << YSCALE(y) << " lineto\n";

        *pstream << "closepath\n";

        *pstream << "fill\n";

        CalcBoundingBox(x, YSCALE(y));
        CalcBoundingBox(x+width, YSCALE(y+height));
      }
      if (current_pen && current_pen->style != wxTRANSPARENT)
      {
        SetPen(current_pen);
        // Draw rectangle anticlockwise
        *pstream << "newpath\n";
        *pstream << x+radius << " " << YSCALE(y+radius) << " " << radius << " 90 180 arc\n";

        *pstream << x << " " << YSCALE(y + height - radius) << " lineto\n";

        *pstream << x+radius << " " << YSCALE(y+height-radius) << " " << radius << " 180 270 arc\n";
        *pstream << x+width-radius << " " << YSCALE(y+height) << " lineto\n";

        *pstream << x+width-radius << " " << YSCALE(y+height-radius) << " " << radius << " 270 0 arc\n";
        *pstream << x+width << " " << YSCALE(y+radius) << " lineto\n";

        *pstream << x+width-radius << " " << YSCALE(y+radius) << " " << radius << " 0 90 arc\n";

        *pstream << x+radius << " " << YSCALE(y) << " lineto\n";

        *pstream << "closepath\n";

        *pstream << "stroke\n";

        CalcBoundingBox(x, YSCALE(y));
        CalcBoundingBox(x+width, YSCALE(y+height));
      }
      break;
    }
#ifdef wx_msw
    case wxDEVICE_WINDOWS:
    {
      wxWnd *wnd = NULL;

      int x1 = (int)x;
      int y1 = (int)y;
      int x2 = (int)(x+width);
      int y2 = (int)(y+height);

      if (canvas)
      {
        wnd = (wxWnd *)canvas->handle;
        wnd->CalcScrolledPosition((int)x, (int)y, &x1, &y1);
        wnd->CalcScrolledPosition((int)(x+width), (int)(y+height), &x2, &y2);
      }

      HDC dc = NULL;
      if (cdc)
        dc = cdc;
      else
        dc = GetDC(wnd->handle);

      (void)RoundRect(dc, XLOG2DEV(x1), YLOG2DEV(y1), XLOG2DEV(x2),
                          YLOG2DEV(y2), XLOG2DEV(radius), YLOG2DEV(radius));

      if (!cdc)
        ReleaseDC(wnd->handle, dc);
      break;
    }
#endif
  }
}

void wxDC::DrawEllipse(float x, float y, float width, float height)
{
  switch (device)
  {
#ifdef wx_x
    case wxDEVICE_CANVAS:
    {
      if (canvas)
      {
        static const int angle = 23040;

        if (current_brush && current_brush->style != wxTRANSPARENT)
        {
          SetBrush(current_brush);
          XFillArc(canvas->display, canvas->xwindow, gc, 
              XLOG2DEV(x), YLOG2DEV(y), XLOG2DEVREL(width), YLOG2DEVREL(height), 0, angle);
#ifdef wx_motif
          if (canvas->is_retained)
            XFillArc(canvas->display, canvas->backingPixmap, gcBacking, 
                XLOG2DEV(x), YLOG2DEV(y), XLOG2DEVREL(width), YLOG2DEVREL(height), 0, angle);
#endif
        }

        if (current_pen && current_pen->style != wxTRANSPARENT)
        {
          SetPen(current_pen);
          XDrawArc(canvas->display, canvas->xwindow, gc, 
              XLOG2DEV(x), YLOG2DEV(y), XLOG2DEVREL(width), YLOG2DEVREL(height), 0, angle);
#ifdef wx_motif
          if (canvas->is_retained)
            XDrawArc(canvas->display, canvas->backingPixmap, gcBacking, 
              XLOG2DEV_2(x), YLOG2DEV_2(y), XLOG2DEVREL(width), YLOG2DEVREL(height), 0, angle);
#endif
        }
      }
      break;
    }
#endif
    case wxDEVICE_EPS:
    {
      if (current_brush && current_brush->style != wxTRANSPARENT)
      {
        SetBrush(current_brush);

        *pstream << "newpath\n";
        *pstream << x + width/2 << " " << YSCALE(y + height/2) << " ";
        *pstream << width/2 << " " << height/2 << " 0 360 ellipse\n";
        *pstream << "fill\n";

        CalcBoundingBox(x-width, YSCALE(y-height));
        CalcBoundingBox(x+width, YSCALE(y+height));
      }
      if (current_pen && current_pen->style != wxTRANSPARENT)
      {
        SetPen(current_pen);

        *pstream << "newpath\n";
        *pstream << x + width/2 << " " << YSCALE(y + height/2) << " ";
        *pstream << width/2 << " " << height/2 << " 0 360 ellipse\n";
        *pstream << "stroke\n";

        CalcBoundingBox(x-width, YSCALE(y-height));
        CalcBoundingBox(x+width, YSCALE(y+height));
      }
      break;
    }
#ifdef wx_msw
    case wxDEVICE_WINDOWS:
    {
      wxWnd *wnd = NULL;

      int x1 = (int)x;
      int y1 = (int)y;
      int x2 = (int)(x+width);
      int y2 = (int)(y+height);

      if (canvas)
      {
        wnd = (wxWnd *)canvas->handle;
        wnd->CalcScrolledPosition((int)x, (int)y, &x1, &y1);
        wnd->CalcScrolledPosition((int)(x+width), (int)(y+height), &x2, &y2);
      }

      HDC dc = NULL;
      if (cdc)
        dc = cdc;
      else
        dc = GetDC(wnd->handle);

      (void)Ellipse(dc, XLOG2DEV(x1), YLOG2DEV(y1), XLOG2DEV(x2), YLOG2DEV(y2));

      if (!cdc)
        ReleaseDC(wnd->handle, dc);
      break;
    }
#endif
  }
}

void wxDC::DrawIcon(wxIcon *icon, float x, float y)
{
  switch (device)
  {
#ifdef wx_x
    case wxDEVICE_CANVAS:
    {
      if (canvas)
      {
#ifdef wx_xview
        Xv_Screen screen = xv_get(xview_server, SERVER_NTH_SCREEN, 0);
        Xv_Window root_window = xv_get(screen, XV_ROOT);
/*
        Canvas xview_canvas = (Canvas)canvas->handle;
        Xv_Window paint_window = (Xv_Window)xv_get(xview_canvas, CANVAS_NTH_PAINT_WINDOW, 0);
        Window canvas_window = (Window)xv_get(paint_window, XV_XID);
*/
        Display *dpy = (Display *)xv_get(root_window, XV_DISPLAY);
        Pixmap pixmap = (Pixmap)xv_get(icon->x_image, SERVER_IMAGE_PIXMAP);
        int width = (int)xv_get(icon->x_image, XV_WIDTH);
        int height = (int)xv_get(icon->x_image, XV_HEIGHT);
#endif
#ifdef wx_motif
        Display *dpy = XtDisplay(wxTheApp->topLevel);
        Pixmap pixmap = icon->x_pixmap;
        int width = icon->iconWidth;
        int height = icon->iconHeight;
#endif
        XCopyPlane(dpy, pixmap, canvas->xwindow, gc,
                   0, 0, width, height,
                   (int)XLOG2DEV(x), (int)YLOG2DEV(y), 1);
#ifdef wx_motif
        if (canvas->is_retained)
          XCopyPlane(dpy, pixmap, canvas->backingPixmap, gcBacking,
                     0, 0, width, height,
                     (int)XLOG2DEV_2(x), (int)YLOG2DEV_2(y), 1);
#endif
      }
      break;
    }
#endif
    case wxDEVICE_EPS:
      break;

#ifdef wx_msw
    case wxDEVICE_WINDOWS:
    {
      wxWnd *wnd = NULL;

      int x1 = (int)x;
      int y1 = (int)y;

      if (canvas)
      {
        wnd = (wxWnd *)canvas->handle;
        wnd->CalcScrolledPosition((int)x, (int)y, &x1, &y1);
      }

      HDC dc = NULL;
      if (cdc)
        dc = cdc;
      else
        dc = GetDC(wnd->handle);

      ::DrawIcon(dc, x1, y1, icon->ms_icon);

      if (!cdc)
        ReleaseDC(wnd->handle, dc);
      break;
    }
#endif
  }
}


void wxDC::SetFont(wxFont *the_font)
{
  font = the_font;
  switch (device)
  {
#ifdef wx_motif
#endif
#ifdef wx_x
    case wxDEVICE_CANVAS:
    {
      if (canvas)
      {
#ifdef wx_xview
        // Set the font according to the current scaling
        int scaled_size = (int)(user_scale_y*font->point_size + 0.5);
        Xv_Font xfont = wxFontPool->FindNearestFont(font->family, font->style,
                                                  font->weight, scaled_size, 0, 0);
        font->x_font = xfont;
        Font theFont = (Font)xv_get(xfont, XV_XID);
#endif
#ifdef wx_motif
//        Display *dpy = XtDisplay(wxTheApp->topLevel);
//        int screen = DefaultScreen(dpy);

/*
        int res_x = (int)(DisplayWidth(dpy, screen)/(DisplayWidthMM(dpy, screen)/25.4));
        int res_y = (int)(DisplayHeight(dpy, screen)/(DisplayHeightMM(dpy, screen)/25.4));
*/
        int res_x = 100;
        int res_y = 100;

        int scaled_size = (int)(10 * ((int)(user_scale_y*font->point_size + 0.5)));

        XFontStruct *fontStruct = wxFontPool->FindNearestFont(font->family, font->style,
                                                   font->weight, scaled_size,
                                                   res_x, res_y);
        font->xFont = fontStruct;
        Font theFont = fontStruct->fid;
#endif
        XSetFont(canvas->display, gc, theFont);
#ifdef wx_motif
        if (canvas->is_retained)
          XSetFont(canvas->display, gcBacking, theFont);
#endif
      }
      break;
    }
#endif
    case wxDEVICE_EPS:
    {
      char buf[100];
      char *name;
      char *style = "";
      int Style = font->style;
      int Weight = font->weight;

      switch (font->family)
      {

        case wxMODERN:     name = "/Courier";
                           break;
        case wxSWISS:      name = "/Helvetica";
                           break;
        default:
        case wxDEFAULT:
        case wxROMAN:      name = "/Times";
                           break;
      }

      if (Style == wxNORMAL && (Weight == wxNORMAL || Weight == wxLIGHT))
      {
        if (font->family == wxROMAN)
          style = "-Roman";
        else
          style = "";
      }
      else if (Style == wxNORMAL && Weight == wxBOLD)
        style = "-Bold";

      else if (Style == wxITALIC && (Weight == wxNORMAL || Weight == wxLIGHT))
      {
        if (font->family == wxROMAN)
          style = "-Italic";
        else
          style = "-Oblique";
      }
      else if (Style == wxITALIC && Weight == wxBOLD)
      {
        if (font->family == wxROMAN)
          style = "-BoldItalic";
        else
          style = "-BoldOblique";
      }
      else if (Style == wxSLANT && (Weight == wxNORMAL || Weight == wxLIGHT))
      {
        if (font->family == wxROMAN)
          style = "-Italic";
        else
          style = "-Oblique";
      }
      else if (Style == wxSLANT && Weight == wxBOLD)
      {
        if (font->family == wxROMAN)
          style = "-BoldItalic";
        else
          style = "-BoldOblique";
      }
      else style = "";

      strcpy(buf, name);
      strcat(buf, style);
      *pstream << buf << " findfont\n";
      *pstream << font->point_size << " scalefont setfont\n";

      break;
    }
#ifdef wx_msw
    case wxDEVICE_WINDOWS:
    {
      wxWnd *wnd = NULL;
      if (canvas)
        wnd = (wxWnd *)canvas->handle;

      HDC dc = NULL;
      if (cdc)
        dc = cdc;
      else
        dc = GetDC(wnd->handle);

      if (font && !font->cfont)
      {

        BYTE italic;
        int weight = 0;
        int family = 0;
        char *face = NULL;

        switch (font->family)
        {
          case wxDECORATIVE: family = FF_DECORATIVE;
                             break;
          case wxROMAN:      family = FF_ROMAN;
                             break;
          case wxMODERN:     family = FF_MODERN;
                             break;
          case wxSWISS:      family = FF_SWISS;
                             face = "Arial";
                             break;
          case wxDEFAULT:
          default:           family = FF_DONTCARE;
        }

        if (font->style == wxITALIC || font->style == wxSLANT)
          italic = 1;
        else
          italic = 0;

        if (font->weight == wxNORMAL)
          weight = FW_NORMAL;
        else if (font->weight == wxLIGHT)
          weight = FW_LIGHT;
        else if (font->weight == wxBOLD)
          weight = FW_BOLD;

        int nHeight = font->point_size*GetDeviceCaps(dc, LOGPIXELSY)/72;
        font->cfont = CreateFont(nHeight, 0, 0, 0, weight, italic, 0, 0,
                    ANSI_CHARSET, OUT_DEFAULT_PRECIS, CLIP_DEFAULT_PRECIS,
                    PROOF_QUALITY, DEFAULT_PITCH | family,
                    face);
      }

      if (font && font->cfont)
      {
        HFONT f = ::SelectObject(dc, font->cfont);
        if (!old_font)
          old_font = f;
      }
      if (!cdc)
        ReleaseDC(wnd->handle, dc);
      break;
    }
#endif
  }
}

void wxDC::SetPen(wxPen *pen)
{
#ifdef wx_x
  wxColour *old_colour = current_colour;
  int old_pen_style = -1;
  int old_pen_width = -1;
  if (current_pen)
  {
    old_pen_style = current_pen->style;
    old_pen_width = current_pen->width;
  }
#endif
  current_pen = pen;
  if (!pen)
  {
#ifdef wx_x
    current_colour = NULL;
#endif
    return;
  }
#ifdef wx_x
  current_colour = pen->colour;
#endif
  switch (device)
  {
#ifdef wx_x
    case wxDEVICE_CANVAS:
    {
      if (canvas)
      {
        Bool same_style = (old_pen_style == pen->style && old_pen_width == pen->width);
        if (!same_style)
	{
          int scaled_width = (int)XLOG2DEVREL(pen->width);
          if (scaled_width < 0)
            scaled_width = 0;

          int style;
          static char dotted[] = {2, 5};
          static char short_dashed[] = {4, 4};
          static char long_dashed[] = {4, 8};
          static char dotted_dashed[] = {6, 6, 2, 6};

          switch (pen->style)
          {
            case wxDOT:
              XSetDashes(canvas->display, gc, 0, dotted, 2);
#ifdef wx_motif
              if (canvas->is_retained)
                XSetDashes(canvas->display, gcBacking, 0, dotted, 2);
#endif
              style = LineOnOffDash;
              break;
            case wxSHORT_DASH:
              XSetDashes(canvas->display, gc, 0, short_dashed, 2);
#ifdef wx_motif
              if (canvas->is_retained)
                XSetDashes(canvas->display, gcBacking, 0, short_dashed, 2);
#endif
              style = LineOnOffDash;
	      break;
            case wxLONG_DASH:
              XSetDashes(canvas->display, gc, 0, long_dashed, 2);
#ifdef wx_motif
              if (canvas->is_retained)
                XSetDashes(canvas->display, gcBacking, 0, long_dashed, 2);
#endif
              style = LineOnOffDash;
              break;
            case wxDOT_DASH:
              XSetDashes(canvas->display, gc, 0, dotted_dashed, 4);
#ifdef wx_motif
              if (canvas->is_retained)
                XSetDashes(canvas->display, gcBacking, 0, dotted_dashed, 2);
#endif
	      style = LineOnOffDash;
	      break;
            case wxSOLID:
            case wxTRANSPARENT:
            default:
	      style = LineSolid;
          }

          XSetLineAttributes(canvas->display, gc, scaled_width, style, CapRound,
	                    JoinRound);
#ifdef wx_motif
          if (canvas->is_retained)
            XSetLineAttributes(canvas->display, gcBacking, scaled_width, style, CapRound,
	                       JoinRound);
#endif

	}

        Bool same_colour = (old_colour &&
                            (old_colour->red == pen->colour->red) &&
                            (old_colour->blue == pen->colour->blue) &&
                            (old_colour->green == pen->colour->green));

        if (!same_colour)
	{
        int pixel = -1;
        if (pen->style == wxTRANSPARENT)
          pixel = background_pixel;
        else if (!Colour)
        {
          unsigned char red = pen->colour->Red();
          unsigned char blue = pen->colour->Blue();
          unsigned char green = pen->colour->Green();
          if (red == (unsigned char )255 && blue == (unsigned char)255
               && green == (unsigned char)255)
	  {
            pixel = WhitePixel(canvas->display, DefaultScreen(canvas->display));
            current_colour = wxWHITE;
	  }
          else
	  {
            pixel = BlackPixel(canvas->display, DefaultScreen(canvas->display));
            current_colour = wxBLACK;
	  }
        }
        else if (pen->colour)
          {
            XColor exact_def;
	    exact_def.red = pen->colour->Red();
	    exact_def.green = pen->colour->Green();
	    exact_def.blue = pen->colour->Blue();
	    exact_def.flags = DoRed | DoGreen | DoBlue;

	    Colormap cmap = DefaultColormap(canvas->display, DefaultScreen(canvas->display));
            if (!XAllocColor(canvas->display, cmap, &exact_def))
	    {
              pixel = BlackPixel(canvas->display, DefaultScreen(canvas->display));
	      fprintf(stderr, "wxWindows warning: cannot allocate pen colour; using black instead\n");
	    }
	    else
	      pixel = exact_def.pixel;
          }

         // Finally, set the GC to the required colour
        if (pixel > -1)
	{
          XSetForeground(canvas->display, gc, pixel);
#ifdef wx_motif
          if (canvas->is_retained)
            XSetForeground(canvas->display, gcBacking, pixel);
#endif
	}
        }
       }
      break;
    }
#endif
    case wxDEVICE_EPS:
    {
      // Line width
      *pstream << pen->width << " setlinewidth\n";

      // Line style
      static char *dotted = "[2 5] 2";
      static char *short_dashed = "[4 4] 2";
      static char *long_dashed = "[4 8] 2";
      static char *dotted_dashed = "[6 6 2 6] 4";

      char *psdash = NULL;
      switch (pen->style)
      {
        case wxDOT:
          psdash = dotted;
          break;
        case wxSHORT_DASH:
          psdash = short_dashed;
          break;
        case wxLONG_DASH:
          psdash = long_dashed;
          break;
        case wxDOT_DASH:
          psdash = dotted_dashed;
          break;
        case wxSOLID:
        case wxTRANSPARENT:
        default:
          psdash = "[] 0";
          break;
      }
      *pstream << psdash << " setdash\n";

      // Line colour
      unsigned char red = pen->colour->Red();
      unsigned char blue = pen->colour->Blue();
      unsigned char green = pen->colour->Green();

      if (!Colour)
      {
        // Anything not white is black
        if (!(red == (unsigned char )255 && blue == (unsigned char)255
                       && green == (unsigned char)255))
        {
          red = (unsigned char )0;
          green = (unsigned char )0;
          blue = (unsigned char )0;
        }
      }
      *pstream << (int)red << " " << (int)green << " " << (int)blue << " setrgbcolor\n";

      break;
    }
#ifdef wx_msw
    case wxDEVICE_WINDOWS:
    {
      HDC dc = NULL;
      wxWnd *wnd = NULL;
      if (canvas)
        wnd = (wxWnd *)canvas->handle;
      if (cdc)
        dc = cdc;
      else if (wnd)
        dc = GetDC(wnd->handle);

      HPEN p = ::SelectObject(dc, pen->cpen);
      if (!old_pen)
        old_pen = p;

      if (!cdc)
        ReleaseDC(wnd->handle, dc);
      break;
    }
#endif
  }
}

void wxDC::SetTextForeground(wxColour *colour)
{
  current_text_foreground = colour;
}

void wxDC::SetTextBackground(wxColour *colour)
{
  current_text_background = colour;
}

void wxDC::SetBrush(wxBrush *brush)
{
#ifdef wx_x
  wxColour *old_colour = current_colour;
  int old_brush_style = -1;
  if (current_brush)
  {
    old_brush_style = current_brush->style;
  }
#endif
  current_brush = brush;
  if (!brush)
  {
#ifdef wx_x
    current_colour = NULL;
#endif
    return;
  }
#ifdef wx_x
  current_colour = brush->colour;
#endif
  switch (device)
  {
#ifdef wx_x
    case wxDEVICE_CANVAS:
    {
      if (canvas)
      {
        if (old_brush_style != brush->style)
	{
          switch (brush->style)
          {
            case wxTRANSPARENT:
              break;
            case wxSOLID:
            case wxSTIPPLE:
            default:
              XSetFillStyle(canvas->display, gc, FillSolid);
#ifdef wx_motif
              if (canvas->is_retained)
                XSetFillStyle(canvas->display, gcBacking, FillSolid);
#endif
          }
	}

        Bool same_colour = (old_colour &&
                            (old_colour->red == brush->colour->red) &&
                            (old_colour->blue == brush->colour->blue) &&
                            (old_colour->green == brush->colour->green));
        if (!same_colour)
	{
        int pixel = -1;
        if (!Colour)
	{
          // Policy - on a monochrome screen, all brushes are white.
          pixel = WhitePixel(canvas->display, DefaultScreen(canvas->display));
          current_colour = wxWHITE;

          // N.B. comment out the above line and uncomment the following lines
          // if you want non-white colours to be black on a monochrome display.
/*
          unsigned char red = brush->colour->Red();
          unsigned char blue = brush->colour->Blue();
          unsigned char green = brush->colour->Green();
          if (red == (unsigned char )255 && blue == (unsigned char)255
               && green == (unsigned char)255)
            pixel = WhitePixel(canvas->display, DefaultScreen(canvas->display));
          else
            pixel = BlackPixel(canvas->display, DefaultScreen(canvas->display));
*/
        }
        else if (brush->style != wxTRANSPARENT && brush->colour)
          {
            XColor exact_def;
	    exact_def.red = brush->colour->Red();
	    exact_def.green = brush->colour->Green();
	    exact_def.blue = brush->colour->Blue();
	    exact_def.flags = DoRed | DoGreen | DoBlue;

            Colormap cmap = DefaultColormap(canvas->display, DefaultScreen(canvas->display));

	    if (!XAllocColor(canvas->display, cmap, &exact_def))
	    {
	      pixel = BlackPixel(canvas->display, DefaultScreen(canvas->display));
	      fprintf(stderr, "wxWindows warning: cannot allocate brush colour; using black instead\n");
	    }
	    else
	      pixel = exact_def.pixel;
          }
//        if (pixel > -1) cout << "Setting foreground to " << pixel << "\n";
        if (pixel > -1)
	{
          // Finally, set the GC to the required colour
          XSetForeground(canvas->display, gc, pixel);
#ifdef wx_motif
          if (canvas->is_retained)
            XSetForeground(canvas->display, gcBacking, pixel);
#endif
	}
        }
      }
      break;
    }
#endif
    case wxDEVICE_EPS:
    {
      // Brush colour
      unsigned char red = brush->colour->Red();
      unsigned char blue = brush->colour->Blue();
      unsigned char green = brush->colour->Green();

      if (!Colour)
      {
        // Anything not black is white
        if (!(red == (unsigned char )0 && blue == (unsigned char)0
                       && green == (unsigned char)0))
        {
          red = (unsigned char )255;
          green = (unsigned char )255;
          blue = (unsigned char )255;
        }
      }
      *pstream << (int)red << " " << (int)green << " " << (int)blue << " setrgbcolor\n";

      break;
    }
#ifdef wx_msw
    case wxDEVICE_WINDOWS:
    {
      HDC dc = NULL;
      wxWnd *wnd = NULL;
      if (canvas)
        wnd = (wxWnd *)canvas->handle;
      if (cdc)
        dc = cdc;
      else
        dc = GetDC(wnd->handle);

      if (current_brush && current_brush->cbrush)
      {
        HBRUSH b = ::SelectObject(dc, current_brush->cbrush);
        if (!old_brush)
          old_brush = b;
      }
      if (!cdc)
        ReleaseDC(wnd->handle, dc);
      break;
    }
#endif
  }
}

void wxDC::DrawText(char *text, float x, float y)
{
  switch (device)
  {
#ifdef wx_motif
#endif
#ifdef wx_x
    case wxDEVICE_CANVAS:
    {
      if (canvas)
      {
        // Since X draws from the baseline of the text, must
        // add the text height
        int cx = 0;
        int cy = 0;
        int ascent = 0;
        if (font)
	{
#ifdef wx_xview
          XFontStruct *font_info = (XFontStruct *)xv_get(font->x_font, FONT_INFO);
#endif
#ifdef wx_motif
          XFontStruct *font_info = font->xFont;
#endif
          int direction, descent;
          XCharStruct overall_return;
          (void)XTextExtents(font_info, text, strlen(text), &direction,
                             &ascent, &descent, &overall_return);
          cx = overall_return.width;
          cy = ascent + descent;
	}

        // First draw a rectangle representing the text background,
        // if a text background is specified
        if (current_text_background)
	{
          wxColour *old_colour = current_colour;
          current_colour = current_text_background;
          Bool same_colour = (old_colour && current_text_background &&
                              (old_colour->red == current_text_background->red) &&
                              (old_colour->blue == current_text_background->blue) &&
                              (old_colour->green == current_text_background->green));

    
          if (!same_colour)
          {
            int pixel = -1;
            if (current_text_background)
            {
              XColor exact_def;
              exact_def.red = current_text_background->Red();
              exact_def.green = current_text_background->Green();
              exact_def.blue = current_text_background->Blue();
              exact_def.flags = DoRed | DoGreen | DoBlue;

	      Colormap cmap = DefaultColormap(canvas->display, DefaultScreen(canvas->display));
	      if (!XAllocColor(canvas->display, cmap, &exact_def))
	      {
	        pixel = WhitePixel(canvas->display, DefaultScreen(canvas->display));
	        fprintf(stderr, "wxWindows warning: cannot allocate text background colour; using white instead\n");
	      }
	      else
	        pixel = exact_def.pixel;
	    }

            // Set the GC to the required colour
           if (pixel > -1)
	   {
             XSetForeground(canvas->display, gc, pixel);
#ifdef wx_motif
             if (canvas->is_retained)
               XSetForeground(canvas->display, gcBacking, pixel);
#endif
	   }
	  }
          XFillRectangle(canvas->display, canvas->xwindow, gc, 
                         XLOG2DEV(x), YLOG2DEV(y),
                         cx, cy);
#ifdef wx_motif
          if (canvas->is_retained)
            XFillRectangle(canvas->display, canvas->backingPixmap, gcBacking, 
                           XLOG2DEV_2(x), YLOG2DEV_2(y),
                           cx, cy);
#endif
	}

        // Now set the text foreground and draw the text
        if (current_text_foreground)
	{
          wxColour *old_colour = current_colour;
          current_colour = current_text_foreground;
          Bool same_colour = (old_colour && current_text_foreground &&
                        (old_colour->red == current_text_foreground->red) &&
                        (old_colour->blue == current_text_foreground->blue) &&
                        (old_colour->green == current_text_foreground->green));


    
          if (!same_colour)
          {
              int pixel = -1;
              if (!Colour)
              {
                // Unless foreground is really white, draw it in black
                unsigned char red = current_text_foreground->Red();
                unsigned char blue = current_text_foreground->Blue();
                unsigned char green = current_text_foreground->Green();
                if (red == (unsigned char )255 && blue == (unsigned char)255
                     && green == (unsigned char)255)
                {
                  pixel = WhitePixel(canvas->display, DefaultScreen(canvas->display));
                  current_colour = wxWHITE;
                }
                else
                {
                  pixel = BlackPixel(canvas->display, DefaultScreen(canvas->display));
                  current_colour = wxBLACK;
                }
              }
              else
	      {
                XColor exact_def;
                exact_def.red = current_text_foreground->Red();
	        exact_def.green = current_text_foreground->Green();
	        exact_def.blue = current_text_foreground->Blue();
	        exact_def.flags = DoRed | DoGreen | DoBlue;

	        Colormap cmap = DefaultColormap(canvas->display, DefaultScreen(canvas->display));
	        if (!XAllocColor(canvas->display, cmap, &exact_def))
	        {
	          pixel = BlackPixel(canvas->display, DefaultScreen(canvas->display));
	          fprintf(stderr, "wxWindows warning: cannot allocate text colour; using black instead\n");
	        }
	        else
	          pixel = exact_def.pixel;
	      }
             // Set the GC to the required colour
             if (pixel > -1)
	     {
               XSetForeground(canvas->display, gc, pixel);
#ifdef wx_motif
               if (canvas->is_retained)
                 XSetForeground(canvas->display, gcBacking, pixel);
#endif
	     }
	  }
	}

        // We need to add the ascent, not the whole height, since X draws
        // at the point above the descender.
        XDrawString(canvas->display, canvas->xwindow, gc, 
                      XLOG2DEV(x), YLOG2DEV(y)+ascent, text, strlen(text));
#ifdef wx_motif
        if (canvas->is_retained)
          XDrawString(canvas->display, canvas->backingPixmap, gcBacking, 
                        XLOG2DEV_2(x), YLOG2DEV_2(y)+ascent, text, strlen(text));
#endif
      }
      break;
    }
#endif
    case wxDEVICE_EPS:
    {
      if (font)
        SetFont(font);

      if (current_text_foreground)
      {
        unsigned char red = current_text_foreground->Red();
        unsigned char blue = current_text_foreground->Blue();
        unsigned char green = current_text_foreground->Green();

        if (!Colour)
        {
          // Anything not white is black
          if (!(red == (unsigned char )255 && blue == (unsigned char)255
                         && green == (unsigned char)255))
          {
            red = (unsigned char )0;
            green = (unsigned char )0;
            blue = (unsigned char )0;
          }
        }
        *pstream << (int)red << " " << (int)green << " " << (int)blue << " setrgbcolor\n";
      }

      int size = 10;
      if (font) size = font->point_size;

      *pstream << x << " " << YSCALE(y + size) << " moveto\n";

      *pstream << "(" << text << ")" << " show\n";

      CalcBoundingBox(x, YSCALE(y+size));
      CalcBoundingBox(x + size*strlen(text), YSCALE(y));
      break;
    }
#ifdef wx_msw
    case wxDEVICE_WINDOWS:
    {
      wxWnd *wnd = NULL;

      int xx1 = (int)x;
      int yy1 = (int)y;

      if (canvas)
      {
        wnd = (wxWnd *)canvas->handle;
        wnd->CalcScrolledPosition((int)x, (int)y, &xx1, &yy1);
      }

      HDC dc = NULL;
      if (cdc)
        dc = cdc;
      else
        dc = GetDC(wnd->handle);

      if (font && font->cfont)
      {
        HFONT f = ::SelectObject(dc, font->cfont);
        if (!old_font)
          old_font = f;
      }

      if (current_text_foreground)
        SetTextColor(dc, RGB(current_text_foreground->Red(),
                             current_text_foreground->Green(),
                             current_text_foreground->Blue()));

      DWORD old_background;
      if (current_text_background)
        old_background = SetBkColor(dc, RGB(current_text_background->Red(),
                                            current_text_background->Green(),
                                            current_text_background->Blue()));

      (void)TextOut(dc, XLOG2DEV(xx1), YLOG2DEV(yy1), text, strlen(text));

      if (current_text_background)
        (void)SetBkColor(dc, old_background);

      if (!cdc)
        ReleaseDC(wnd->handle, dc);

      break;
    }
#endif
  }
}

void wxDC::SetBackground(wxBrush *brush)
{
  current_background_brush = brush;
  switch (device)
  {
#ifdef wx_motif
#endif
#ifdef wx_x
    case wxDEVICE_CANVAS:
    {
      if (canvas)
      {
        XColor exact_def;
        exact_def.red = brush->colour->Red();
        exact_def.green = brush->colour->Green();
        exact_def.blue = brush->colour->Blue();
        exact_def.flags = DoRed | DoGreen | DoBlue;
        int pixel;

        Colormap cmap = DefaultColormap(canvas->display, DefaultScreen(canvas->display));
          
        if (!XAllocColor(canvas->display, cmap, &exact_def))
        {
          pixel = WhitePixel(canvas->display, DefaultScreen(canvas->display));
          cerr << "wxWindows warning: cannot allocate background colour; using white instead\n";
        }
        else
          pixel = exact_def.pixel;

#ifdef wx_xview
        // Finally, set the background to the required colour
        Xv_Window pw = canvas_paint_window((Canvas)(canvas->handle));

        Window win = (Window)xv_get(pw, XV_XID);
        XSetWindowBackground(canvas->display, win, pixel);
#endif
#ifdef wx_motif
        XSetWindowBackground(canvas->display, canvas->xwindow, pixel);
        if (canvas->is_retained)
          XSetWindowBackground(canvas->display, canvas->backingPixmap, pixel);
#endif
      }
      break;
    }
#endif
    case wxDEVICE_EPS:
    {
      break;
    }
#ifdef wx_msw
    case wxDEVICE_WINDOWS:
    {
      break;
    }
#endif
  }
}

void wxDC::SetLogicalFunction(int function)
{
  switch (device)
  {
#ifdef wx_x
    case wxDEVICE_CANVAS:
    {
      if (canvas)
      {
        int x_function;
        switch (function)
        {
          case wxXOR:  x_function = GXxor; break;
          case wxINVERT: x_function = GXinvert; break;
          case wxOR_REVERSE: x_function = GXorReverse; break;
          case wxAND_REVERSE: x_function = GXandReverse; break;
          case wxCOPY:
          default:
            x_function = GXcopy; break;
        }

        XSetFunction(canvas->display, gc, x_function);
#ifdef wx_motif
        if (canvas->is_retained)
          XSetFunction(canvas->display, gcBacking, x_function);
#endif
      }
      break;
    }
#endif
    case wxDEVICE_EPS:
    {
      break;
    }
#ifdef wx_msw
    case wxDEVICE_WINDOWS:
    {
      current_logical_function = function;

      HDC dc = NULL;
      wxWnd *wnd = NULL;
      if (canvas)
        wnd = (wxWnd *)canvas->handle;
      if (cdc)
        dc = cdc;
      else
        dc = GetDC(wnd->handle);

      SetRop(dc);

      if (!cdc)
        ReleaseDC(wnd->handle, dc);
    }
#endif
  }
}

#ifdef wx_msw
void wxDC::SetRop(HDC dc)
{
  if (!dc || current_logical_function < 0)
    return;

  int c_rop;
  // These may be wrong
  switch (current_logical_function)
  {
    case wxXOR: c_rop = R2_XORPEN; break;
    case wxINVERT: c_rop = R2_NOT; break;
    case wxOR_REVERSE: c_rop = R2_MERGEPENNOT; break;
    case wxAND_REVERSE: c_rop = R2_MASKNOTPEN; break;
    case wxCOPY:
    default:
      c_rop = R2_COPYPEN; break;
  }
  SetROP2(dc, c_rop);
}
#endif

// Make a 3-point spline
void wxDC::DrawSpline(float x1, float y1, float x2, float y2, float x3, float y3)
{

  wxList *point_list = new wxList;

  wxPoint *point1 = new wxPoint;
  point1->x = x1; point1->y = y1;
  point_list->Append(point1);

  wxPoint *point2 = new wxPoint;
  point2->x = x2; point2->y = y2;
  point_list->Append(point2);

  wxPoint *point3 = new wxPoint;
  point3->x = x3; point3->y = y3;
  point_list->Append(point3);

  wxSpline spline(point_list);

  if (device == wxDEVICE_EPS)
  {
    wx_draw_open_spline_ps(this, &spline);
  }
  else wx_draw_open_spline(this, &spline);
  spline.DeletePoints();
}

void wxDC::DrawSpline(wxList *list)
{
  wxSpline spline(list);

  if (device == wxDEVICE_EPS)
    wx_draw_open_spline_ps(this, &spline);
  else wx_draw_open_spline(this, &spline);
}


int wxDC::GetTextFamily(void)
{
  switch (device)
  {
#ifdef wx_x
    case wxDEVICE_CANVAS:
    {
#ifdef wx_motif
    if (!font)
      return -1;
    else return font->family;
#endif
#ifdef wx_xview
      if (!(font && font->x_font))
        return -1;

      // Is this the right syntax????
      char *family = (char *)xv_get((Xv_opaque)NULL, font->x_font, FONT_FAMILY);
      int wx_family;

      if (strcmp(family, FONT_FAMILY_LUCIDA) == 0)
        wx_family = wxDECORATIVE;
      else if (strcmp(family, FONT_FAMILY_ROMAN) == 0)
        wx_family = wxROMAN;
      else if (strcmp(family, FONT_FAMILY_HELVETICA) == 0)
        wx_family = wxSWISS;
     else if (strcmp(family, FONT_FAMILY_COUR) == 0)
        wx_family = wxMODERN;
      else
        wx_family = wxDEFAULT;

      return wx_family;
#endif
      break;
    }
#endif
    case wxDEVICE_EPS:
    {
      if (font)
        return font->family;
      else return wxDEFAULT;
      break;
    }
#ifdef wx_msw
    case wxDEVICE_WINDOWS:
    {
      TEXTMETRIC lpTextMetric;
      wxWnd *wnd = NULL;

      if (canvas)
        wnd = (wxWnd *)canvas->handle;

      HDC dc = NULL;
      if (cdc)
        dc = cdc;
      else
        dc = GetDC(wnd->handle);

      GetTextMetrics(dc, &lpTextMetric);

      if (!cdc)
        ReleaseDC(wnd->handle, dc);

      int family;
      switch (lpTextMetric.tmPitchAndFamily & 0xF0)
      {
        case FF_DECORATIVE:
          family = wxDECORATIVE;
          break;
        case FF_MODERN:
          family = wxMODERN;
          break;
        case FF_ROMAN:
          family = wxROMAN;
          break;
        case FF_SWISS:
          family = wxSWISS;
          break;
        default:
        case FF_DONTCARE:
          family = wxDEFAULT;
      }
      return family;
      break;
    }
#endif
  }
  return wxDEFAULT;
}

int wxDC::GetTextStyle(void)
{
  switch (device)
  {
#ifdef wx_x
    case wxDEVICE_CANVAS:
    {
#ifdef wx_motif
    if (!font)
      return -1;
    else return font->style;
#endif
#ifdef wx_xview
      if (!(font && font->x_font))
        return -1;

      // Is this the right syntax????
      char *style = (char *)xv_get((Xv_opaque)NULL, font->x_font, FONT_STYLE);
      int wx_style;

      if (strcmp(style, FONT_STYLE_NORMAL) == 0)
        wx_style = wxNORMAL;
      else if (strcmp(style, FONT_STYLE_BOLD) == 0)
        wx_style = wxNORMAL;
      else if (strcmp(style, FONT_STYLE_ITALIC) == 0)
        wx_style = wxITALIC;
      else if (strcmp(style, FONT_STYLE_BOLD_ITALIC) == 0)
        wx_style = wxITALIC;
      else if (strcmp(style, FONT_STYLE_OBLIQUE) == 0)
        wx_style = wxSLANT;
      else if (strcmp(style, FONT_STYLE_BOLD_OBLIQUE) == 0)
        wx_style = wxSLANT;
      else wx_style = wxDEFAULT;

      return wx_style;
#endif
      break;
    }
#endif
    case wxDEVICE_EPS:
    {
      if (font)
        return font->style;
      else return wxDEFAULT;
      break;
    }
#ifdef wx_msw
    case wxDEVICE_WINDOWS:
    {
      TEXTMETRIC lpTextMetric;
      wxWnd *wnd = NULL;

      if (canvas)
      {
        wnd = (wxWnd *)canvas->handle;
      }

      HDC dc = NULL;

      if (cdc)
        dc = cdc;
      else dc = GetDC(wnd->handle);

      GetTextMetrics(dc, &lpTextMetric);

      if (!cdc)
        ReleaseDC(wnd->handle, dc);

      int wx_style;
      if (lpTextMetric.tmItalic)
        wx_style = wxITALIC;
      else wx_style = wxNORMAL;

      return wx_style;
      break;
    }
#endif
  }
  return wxNORMAL;
}

int wxDC::GetTextWeight(void)
{
  switch (device)
  {
#ifdef wx_x
    case wxDEVICE_CANVAS:
    {
#ifdef wx_motif
      if (!font)
        return -1;
      else return font->weight;
#endif
#ifdef wx_xview
      if (!(font && font->x_font))
        return -1;

      char *style = (char *)xv_get((Xv_opaque)NULL, font->x_font, FONT_STYLE);
      int wx_weight;

      if (strcmp(style, FONT_STYLE_NORMAL) == 0)
        wx_weight = wxNORMAL;
      else if (strcmp(style, FONT_STYLE_BOLD) == 0)
        wx_weight = wxBOLD;
      else if (strcmp(style, FONT_STYLE_ITALIC) == 0)
        wx_weight = wxNORMAL;
      else if (strcmp(style, FONT_STYLE_BOLD_ITALIC) == 0)
        wx_weight = wxBOLD;
      else if (strcmp(style, FONT_STYLE_OBLIQUE) == 0)
        wx_weight = wxNORMAL;
      else if (strcmp(style, FONT_STYLE_BOLD_OBLIQUE) == 0)
        wx_weight = wxBOLD;
      else wx_weight = wxNORMAL;

      return wx_weight;
#endif
      break;
    }
#endif
    case wxDEVICE_EPS:
    {
      if (font)
        return font->weight;
      else return wxNORMAL;
      break;
    }
#ifdef wx_msw
    case wxDEVICE_WINDOWS:
    {
      TEXTMETRIC lpTextMetric;
      wxWnd *wnd = NULL;

      if (canvas)
      {
        wnd = (wxWnd *)canvas->handle;
      }

      HDC dc = NULL;

      if (cdc)
        dc = cdc;
      else dc = GetDC(wnd->handle);

      GetTextMetrics(dc, &lpTextMetric);

      if (!cdc)
        ReleaseDC(wnd->handle, dc);

      int wx_weight;
      int cweight = lpTextMetric.tmWeight;
      if (cweight > 0 && cweight < 350)
        wx_weight = wxLIGHT;  // light
      else if (cweight >= 350 && cweight < 500)
        wx_weight = wxNORMAL; // normal
      else if (cweight >= 500)
        wx_weight = wxBOLD; // bold
      else wx_weight = wxNORMAL;

      return wx_weight;
      break;
    }
#endif
  }
  return wxNORMAL;
}

float wxDC::GetTextHeight(void)
{
  switch (device)
  {
#ifdef wx_x
    case wxDEVICE_CANVAS:
    {
#ifdef wx_xview
      if (!(font && font->x_font))
        return YDEV2LOGREL(12);

      return YDEV2LOGREL((int)xv_get((Xv_opaque)NULL, font->x_font, FONT_DEFAULT_CHAR_HEIGHT));
#endif
#ifdef wx_motif
      if (!(font && font->xFont))
        return XDEV2LOGREL(12);

      int direction, ascent, descent;
      XCharStruct overall;
      XTextExtents(font->xFont, "x", 1, &direction, &ascent,
                   &descent, &overall);
      return overall.ascent + overall.descent;
#endif
      break;
    }
#endif
    case wxDEVICE_EPS:
    {
      if (font)
        return font->point_size;
      else return 12.0;
      break;
    }
#ifdef wx_msw
    case wxDEVICE_WINDOWS:
    {
      TEXTMETRIC lpTextMetric;
      wxWnd *wnd = NULL;

      if (canvas)
      {
        wnd = (wxWnd *)canvas->handle;
      }

      HDC dc = NULL;

      if (cdc)
        dc = cdc;
      else dc = GetDC(wnd->handle);

      GetTextMetrics(dc, &lpTextMetric);

      if (!cdc)
        ReleaseDC(wnd->handle, dc);

      return YDEV2LOGREL(lpTextMetric.tmHeight);
      break;
    }
#endif
  }
  return 12.0;
}

float wxDC::GetTextWidth(void)
{
  switch (device)
  {
#ifdef wx_x
    case wxDEVICE_CANVAS:
    {
#ifdef wx_xview
      if (!(font && font->x_font))
        return XDEV2LOGREL(16);

      return XDEV2LOGREL((int)xv_get((Xv_opaque)NULL, font->x_font, FONT_DEFAULT_CHAR_WIDTH));
#endif
#ifdef wx_motif
      if (!(font && font->xFont))
        return XDEV2LOGREL(16);

      int direction, ascent, descent;
      XCharStruct overall;
      XTextExtents(font->xFont, "x", 1, &direction, &ascent,
                   &descent, &overall);
      return overall.width;
#endif
      break;
    }
#endif
    case wxDEVICE_EPS:
    {
      return 0;
      break;
    }
#ifdef wx_msw

    case wxDEVICE_WINDOWS:
    {
      TEXTMETRIC lpTextMetric;
      wxWnd *wnd = NULL;

      if (canvas)
      {
        wnd = (wxWnd *)canvas->handle;
      }

      HDC dc = NULL;

      if (cdc)
        dc = cdc;
      else dc = GetDC(wnd->handle);

      GetTextMetrics(dc, &lpTextMetric);

      if (!cdc)
        ReleaseDC(wnd->handle, dc);

      return XDEV2LOGREL(lpTextMetric.tmAveCharWidth);
      break;
    }
#endif
  }
  return 0.0;
}

void wxDC::GetTextExtent(char *string, float *x, float *y)
{
  switch (device)
  {
#ifdef wx_x
    case wxDEVICE_CANVAS:
    {
#ifdef wx_xview
      if (!(font && font->x_font))
      {
        cerr << "wxWindows warning - set a font before calling GetTextExtent!\n";
        *x = -1;
        *y = -1;
        return;
      }

      Font_string_dims dims;

      (void)xv_get(font->x_font, FONT_STRING_DIMS, string, &dims);
      *x = XDEV2LOGREL(dims.width);
      *y = YDEV2LOGREL(dims.height);
#endif
#ifdef wx_motif
      if (!(font && font->xFont))
      {
        cerr << "wxWindows warning - set a font before calling GetTextExtent!\n";
        *x = -1;
        *y = -1;
        return;
      }
      int direction, ascent, descent;
      XCharStruct overall;
      XTextExtents(font->xFont, string, strlen(string), &direction, &ascent,
                   &descent, &overall);
//      *x = XDEV2LOGREL(overall.width);
//      *y = YDEV2LOGREL(overall.ascent + overall.descent);
      *x = XDEV2LOGREL(overall.width);
      *y = YDEV2LOGREL(ascent + descent);
#endif
      break;
    }
#endif
    case wxDEVICE_EPS:
    {
      // Provide a VERY rough estimate (avoid using it)
      int width = 12;
      int height = 12;

      if (font)
        { height = font->point_size; width = height; }
      *x = strlen(string)*width;
      *y = strlen(string)*height;
      break;
    }
#ifdef wx_msw
    case wxDEVICE_WINDOWS:
    {
      TEXTMETRIC lpTextMetric;
      wxWnd *wnd = NULL;

      if (canvas)
      {
        wnd = (wxWnd *)canvas->handle;
      }

      HDC dc = NULL;

      if (cdc)
        dc = cdc;
      else dc = GetDC(wnd->handle);

      GetTextMetrics(dc, &lpTextMetric);

      SIZE sizeRect;
      GetTextExtentPoint(dc, string, strlen(string), &sizeRect);

      if (!cdc)
        ReleaseDC(wnd->handle, dc);

      *x = XDEV2LOGREL(sizeRect.cx);
      *y = YDEV2LOGREL(sizeRect.cy);
      break;
    }
#endif
  }
}

Bool wxDC::Ok(void)
{
  return ok;
}

#define ELLIPSE_PS  "\
/ellipsedict 8 dict def\n\
ellipsedict /mtrx matrix put\n\
/ellipse\n\
{ ellipsedict begin\n\
  /endangle exch def\n\
  /startangle exch def\n\
  /yrad exch def\n\
  /xrad exch def\n\
  /y exch def\n\
  /x exch def\n\
  /savematrix mtrx currentmatrix def\n\
  x y translate\n\
  xrad yrad scale\n\
  0 0 1 startangle endangle arc\n\
  savematrix setmatrix\n\
  end\n\
  } def\n\
"

#define		SPLINE_PS	" \
/DrawSplineSection {\n\
	/y3 exch def\n\
	/x3 exch def\n\
	/y2 exch def\n\
	/x2 exch def\n\
	/y1 exch def\n\
	/x1 exch def\n\
	/xa x1 x2 x1 sub 0.666667 mul add def\n\
	/ya y1 y2 y1 sub 0.666667 mul add def\n\
	/xb x3 x2 x3 sub 0.666667 mul add def\n\
	/yb y3 y2 y3 sub 0.666667 mul add def\n\
	x1 y1 lineto\n\
	xa ya xb yb x3 y3 curveto\n\
	} def\n\
"

Bool wxDC::StartDoc(char *message)
{
  Bool flag = FALSE;
  switch (device)
  {
    case wxDEVICE_EPS:
    {
      if (message) title = copystring(message);
      flag = TRUE;
      break;
    }
#ifdef wx_msw
    case wxDEVICE_WINDOWS:
    {
      DOCINFO docinfo;
      docinfo.cbSize = sizeof(DOCINFO);
      docinfo.lpszDocName = message;
      docinfo.lpszOutput = filename;
      if (cdc) flag = ::StartDoc(cdc, &docinfo);
      else flag = FALSE;
      break;
    }
#endif
    default:
      break;
  }
  return flag;
}

void wxDC::EndDoc(void)
{
  switch (device)
  {
    case wxDEVICE_EPS:
    {
      if (clipping) { clipping = FALSE; *pstream << "grestore\n"; }

      // Will reuse pstream for header
      if (pstream) { delete pstream; pstream = NULL; }

#ifdef wx_x
      char buf[300];
      long when;
#endif
#ifdef wx_msw
      time_t when;
#endif
      (void) time(&when);

      // Write header now
#ifdef wx_x
      char *header_file = "/tmp/_header.ps";
#endif
#ifdef wx_msw
      char *header_file = "_header.ps";
#endif
      pstream = new ofstream(header_file);

      *pstream << "%!\n";   /* PostScript magic strings */
      if (title) *pstream << "%%Title: " << title << "\n";
      *pstream << "%%Creator: " << wxTheApp->argv[0] << "\n";
      *pstream << "%%CreationDate: " << ctime(&when);
#ifdef wx_x
      char host[256];
      struct passwd *who = getpwuid(getuid());
      if (-1 == gethostname(host, sizeof(host)))
        (void)strcpy(host, "unknown-host!");
      *pstream << "%%For: " << who->pw_name << "@" << host << " (" <<
        who->pw_gecos << ")\n";
#endif
      // In landscape mode, what should bounding box be changed to????
      // Bounding box is currently wrong in landscape mode.

      *pstream << "%%BoundingBox: " << min_x << " " << min_y << " " << max_x << " " << max_y << "\n";
      *pstream << "%%EndComments\n\n";

      // Output scaling
      float real_translate_y = wx_printer_translate_y;
      if (!wx_portrait)
      {
        real_translate_y -= max_y;
        *pstream << "90 rotate\n";
      }

      *pstream << wx_printer_scale_x << " " << wx_printer_scale_y << " scale\n";
      *pstream << wx_printer_translate_x << " " << real_translate_y << " translate\n";

      *pstream << ELLIPSE_PS;
      *pstream << SPLINE_PS;

      delete pstream;
      pstream = NULL;

#ifdef wx_x
      char *concat_name = "/tmp/_concat.tmp";
#endif
#ifdef wx_msw
      char *concat_name = "concat.tmp";
#endif

      // Would be better not to write to wx_printer_file initially,
      // clear up sometime
      wxConcatFiles(header_file, wx_printer_file, concat_name);
      wxRemoveFile(wx_printer_file);
      wxCopyFile(concat_name, wx_printer_file);
      wxRemoveFile(concat_name);
      wxRemoveFile(header_file);
#ifdef wx_x
      if (wx_interactive)
      {
        if (ok && wx_preview)
        {
          strcpy(buf, wx_preview_command);
          strcat(buf, " ");
          strcat(buf, wx_printer_file);
          strcat(buf, " &");
          system(buf);
        }
        else if (ok && !wx_preview && !wx_print_to_file)
        {
          strcpy(buf, wx_printer_command);
          strcat(buf, " ");
          strcat(buf, wx_printer_flags);
          strcat(buf, " ");
          strcat(buf, wx_printer_file);
          strcat(buf, " &");
          system(buf);
        }
      }
#endif
      break;
    }
#ifdef wx_msw
    case wxDEVICE_WINDOWS:
    {
      if (cdc) ::EndDoc(cdc);
      break;
    }
#endif
  }
}

void wxDC::GetSize(float *width, float *height)
{
  switch (device)
  {
    case wxDEVICE_EPS:
    {
      if (!(min_x == 1000.0 && min_y == 1000.0 && max_x == -1000.0 && max_y == -1000.0))
      {
        *width = (float)(max_x - min_x);
        *height = (float)(max_y - min_y);
      }
      else
      {
        *width = 0.0;
        *height = 0.0;
      }
      break;
    }
    default:
    {
      *width = 0.0; *height = 0.0;
      break;
    }
  }
}

void wxDC::StartPage(void)
{
  switch (device)
  {
#ifdef wx_msw
    case wxDEVICE_WINDOWS:
    {
      if (cdc)
        ::StartPage(cdc);
      break;
    }
#endif
    default:
      break;
  }
}

void wxDC::EndPage(void)
{
  switch (device)
  {
    case wxDEVICE_EPS:
    {
      *pstream << "showpage\n";
      break;
    }
#ifdef wx_msw
    case wxDEVICE_WINDOWS:
    {
      if (cdc)
        ::EndPage(cdc);
      break;
    }
#endif
    default:
      break;
  }
}

void wxDC::SetMapMode(int mode)
{
  mapping_mode = mode;

  int pixel_width = 0;
  int pixel_height = 0;
  int mm_width = 0;
  int mm_height = 0;

  switch (device)
  {
#ifdef wx_x
    case wxDEVICE_CANVAS:
    {
      // First, calculate how to scale from mm to pixels.
      // Then we just need to find the scaling factor from ? to mm and multiply
      // by the first scaling factor.
#ifdef wx_xview
      Xv_Screen screen = xv_get(xview_server, SERVER_NTH_SCREEN, 0);
      Xv_Window root_window = xv_get(screen, XV_ROOT);

      Display *dpy = (Display *)xv_get(root_window, XV_DISPLAY);
#endif
#ifdef wx_motif
      Display *dpy = NULL;
      if (wxTheApp->wx_frame)
      {
        dpy = XtDisplay(wxTheApp->wx_frame->frameShell);
      } else if (canvas)
      {
        dpy = XtDisplay(canvas->scrolledWindow);
      }
#endif

      int screen_no = DefaultScreen(dpy);
      pixel_width = DisplayWidth(dpy, screen_no);
      pixel_height = DisplayHeight(dpy, screen_no);
      mm_width = DisplayWidthMM(dpy, screen_no);
      mm_height = DisplayHeightMM(dpy, screen_no);
      break;
    }
#endif
    case wxDEVICE_EPS:
    {
      return;
      break;
    }
#ifdef wx_msw
    case wxDEVICE_WINDOWS:
    {
      HDC dc = NULL;
      wxWnd *wnd = NULL;
      if (canvas)
        wnd = (wxWnd *)canvas->handle;

      if (cdc)
        dc = cdc;
      else if (wnd)
        dc = GetDC(wnd->handle);

      pixel_width = GetDeviceCaps(dc, HORZRES);
      pixel_height = GetDeviceCaps(dc, VERTRES);
      mm_width = GetDeviceCaps(dc, HORZSIZE);
      mm_height = GetDeviceCaps(dc, VERTSIZE);
      if (!cdc)
        ReleaseDC(wnd->handle, dc);
      break;
    }
#endif
  }
  float mm2pixelsX = pixel_width/mm_width;
  float mm2pixelsY = pixel_height/mm_height;

  switch (mode)
  {
    case MM_TWIPS:
    {
      logical_scale_x = (float)(twips2mm * mm2pixelsX);
      logical_scale_y = (float)(twips2mm * mm2pixelsY);
      break;
    }
    case MM_POINTS:
    {
      logical_scale_x = (float)(pt2mm * mm2pixelsX);
      logical_scale_y = (float)(pt2mm * mm2pixelsY);
      break;
    }
    case MM_METRIC:
    {
      logical_scale_x = mm2pixelsX;
      logical_scale_y = mm2pixelsY;
      break;
    }
    case MM_LOMETRIC:
    {
      logical_scale_x = (float)(mm2pixelsX/10.0);
      logical_scale_y = (float)(mm2pixelsY/10.0);
      break;
    }
    default:
    case MM_TEXT:
    {
      logical_scale_x = 1.0;
      logical_scale_y = 1.0;
      break;
    }
  }
#ifdef wx_msw
  if (device == wxDEVICE_WINDOWS)
  {
    HDC dc = NULL;
    wxWnd *wnd = NULL;
    if (canvas)
      wnd = (wxWnd *)canvas->handle;

    if (cdc)
      dc = cdc;
    else if (wnd)
      dc = GetDC(wnd->handle);

    ::SetMapMode(dc, MM_ANISOTROPIC);
    SetViewportExtEx(dc, VIEWPORT_EXTENT, VIEWPORT_EXTENT, NULL);
    window_ext_x = MS_XDEV2LOG(VIEWPORT_EXTENT);
    window_ext_y = MS_YDEV2LOG(VIEWPORT_EXTENT);
    SetWindowExtEx(dc, window_ext_x, window_ext_y, NULL);

    if (!cdc)
      ReleaseDC(wnd->handle, dc);
  }
#endif
}

int wxDC::GetMapMode(void)
{
  return mapping_mode;
}

void wxDC::SetLogicalOrigin(float x, float y)
{
  logical_origin_x = x;
  logical_origin_y = y;
}

void wxDC::SetDeviceOrigin(float x, float y)
{
  device_origin_x = x;
  device_origin_y = y;
}

// For use by wxWindows only, unless custom units are required.
void wxDC::SetLogicalScale(float x, float y)
{
  logical_scale_x = x;
  logical_scale_y = y;
}

void wxDC::SetUserScale(float x, float y)
{
  user_scale_x = x;
  user_scale_y = y;

#ifdef wx_msw
  SetMapMode(mapping_mode);
#endif
#ifdef wx_x
  // Force recalculation of line width
  wxPen *old_pen = current_pen;
  if (old_pen)
  {
    current_pen = NULL;
    SetPen(old_pen);
  }
#endif
}

float wxDC::DeviceToLogicalX(int x)
{
#ifdef wx_x
  return XDEV2LOG(x);
#endif
#ifdef wx_msw
  return MS_XDEV2LOG(x);
#endif
}

float wxDC::DeviceToLogicalY(int y)
{
#ifdef wx_x
  return YDEV2LOG(y);
#endif
#ifdef wx_msw
  return MS_YDEV2LOG(y);
#endif
}

int wxDC::LogicalToDeviceX(float x)
{
#ifdef wx_x
  return XLOG2DEV(x);
#endif
#ifdef wx_msw
  return MS_XLOG2DEV(x);
#endif
}

int wxDC::LogicalToDeviceY(float y)
{
#ifdef wx_x
  return YLOG2DEV(y);
#endif
#ifdef wx_msw
  return MS_YLOG2DEV(y);
#endif
}

void wxDC::CalcBoundingBox(float x, float y)
{
  if (x < min_x)
    min_x = x;
  if (y < min_y)
    min_y = y;

  if (x > max_x)
    max_x = x;

  if (y > max_y)
    max_y = y;
}

// Creates a memory DC
wxDC *wxDC::CreateCompatibleDC(void)
{
  wxDC *wxdc = new wxDC(this);
  return wxdc;
}

void wxDC::SelectObject(wxBitmap *bitmap)
{
  switch (device)
  {
#ifdef wx_x
    case wxDEVICE_CANVAS:
      return;
    case wxDEVICE_PIXMAP:
    {
      selected_pixmap = bitmap;
      return;
    }
#endif
    case wxDEVICE_EPS:
      return;
#ifdef wx_msw
    case wxDEVICE_WINDOWS:
    {
      HBITMAP bm = ::SelectObject(cdc, bitmap->ms_bitmap);
      if (!old_bitmap)
        old_bitmap = bm;
      return;
    }
#endif
  }
}

Bool wxDC::Blit(float xdest, float ydest, float width, float height,
                wxDC *source, float xsrc, float ysrc, int rop)
{
  switch (device)
  {
#ifdef wx_x
    case wxDEVICE_CANVAS:
    {
      if (source->device == wxDEVICE_PIXMAP)
      {
        wxBitmap *bitmap = source->selected_pixmap;
        if (bitmap && bitmap->x_pixmap)
	{
#ifdef wx_xview
          Xv_Screen screen = xv_get(xview_server, SERVER_NTH_SCREEN, 0);
          Xv_Window root_window = xv_get(screen, XV_ROOT);
          Canvas xview_canvas = (Canvas)canvas->handle;
          Xv_Window paint_window = (Xv_Window)xv_get(xview_canvas, CANVAS_NTH_PAINT_WINDOW, 0);
          Window canvas_window = (Window)xv_get(paint_window, XV_XID);

          Display *dpy = (Display *)xv_get(root_window, XV_DISPLAY);
#endif
#ifdef wx_motif
          Display *dpy = canvas->display;
          Window canvas_window = canvas->xwindow;

          if (canvas->is_retained)
            XCopyPlane(dpy, bitmap->x_pixmap, canvas->backingPixmap, gcBacking,
                       (int)source->LogicalToDeviceX(xsrc), (int)source->LogicalToDeviceY(ysrc), (int)width, (int)height,
                       (int)XLOG2DEV_2(xdest), (int)YLOG2DEV_2(ydest), 1);
#endif
          XCopyPlane(dpy, bitmap->x_pixmap, canvas_window, gc,
                     (int)source->LogicalToDeviceX(xsrc), (int)source->LogicalToDeviceY(ysrc), (int)width, (int)height,
                     (int)XLOG2DEV(xdest), (int)YLOG2DEV(ydest), 1);

          return TRUE;
	}
      }
      return FALSE;
    }

    case wxDEVICE_PIXMAP:
      return FALSE;
#endif
    case wxDEVICE_EPS:
      return FALSE;
#ifdef wx_msw
    case wxDEVICE_WINDOWS:
    {
      wxWnd *wnd = NULL;
      wxWnd *wnd_src = NULL;
      if (canvas)
        wnd = (wxWnd *)canvas->handle;
      if (source->canvas)
        wnd_src = (wxWnd *)source->canvas->handle;

      HDC dc = NULL;
      if (cdc)
        dc = cdc;
      else
        dc = GetDC(wnd->handle);

      HDC dc_src = NULL;
      if (source->cdc)
        dc_src = source->cdc;
      else if (wnd)
        dc_src = GetDC(wnd_src->handle);

      int xdest1 = (int)xdest;
      int ydest1 = (int)ydest;
      int xsrc1 = (int)xsrc;
      int ysrc1 = (int)ysrc;

      if (canvas)
      {
        wnd->CalcScrolledPosition((int)xdest, (int)ydest, &xdest1, &ydest1);
      }
      if (source->canvas)
      {
        wnd_src->CalcScrolledPosition((int)xsrc, (int)ysrc, &xsrc1, &ysrc1);
      }

      Bool success = BitBlt(dc, xdest1, ydest1, (int)width, (int)height, dc_src,
                                xsrc1, ysrc1, 
                rop == wxCOPY ? SRCCOPY :
                rop == wxINVERT ? NOTSRCCOPY :
                rop == wxXOR ? SRCINVERT :
                rop == wxOR_REVERSE ? MERGEPAINT :
                rop == wxAND_REVERSE ? SRCERASE :
                SRCCOPY);

      if (!cdc)
        ReleaseDC(wnd->handle, dc);
      if (!source->cdc)
        ReleaseDC(wnd_src->handle, dc_src);
      return success;
      break;
    }
#endif
  }
  return FALSE;
}

/***************************************************************************
 * Spline code taken from xfig - see below   
 *
 * FIG : Facility for Interactive Generation of figures
 * Copyright (c) 1985 by Supoj Sutanthavibul
 *
 * "Permission to use, copy, modify, distribute, and sell this software and its
 * documentation for any purpose is hereby granted without fee, provided that
 * the above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation, and that the name of M.I.T. not be used in advertising or
 * publicity pertaining to distribution of the software without specific,
 * written prior permission.  M.I.T. makes no representations about the
 * suitability of this software for any purpose.  It is provided "as is"
 * without express or implied warranty."
 *
 **************************************************************************/

wxList wx_spline_point_list;

void wx_draw_open_spline(wxDC *dc, wxSpline *spline)
{
    wxPoint *p;
    float           cx1, cy1, cx2, cy2, cx3, cy3, cx4, cy4;
    float           x1, y1, x2, y2;

    wxNode *node = spline->points->First();
    p = (wxPoint *)node->Data();

    x1 = p->x;
    y1 = p->y;

    node = node->Next();
    p = (wxPoint *)node->Data();

    x2 = p->x;
    y2 = p->y;
    cx1 = (float)((x1 + x2) / 2);
    cy1 = (float)((y1 + y2) / 2);
    cx2 = (float)((cx1 + x2) / 2);
    cy2 = (float)((cy1 + y2) / 2);

    wx_spline_add_point((int) x1, (int) y1);

    node = node->Next();

    while (node)
    {
        p = (wxPoint *)node->Data();
	x1 = x2;
	y1 = y2;
	x2 = p->x;
	y2 = p->y;
        cx4 = (float)(x1 + x2) / 2;
        cy4 = (float)(y1 + y2) / 2;
        cx3 = (float)(x1 + cx4) / 2;
        cy3 = (float)(y1 + cy4) / 2;

        wx_quadratic_spline(cx1, cy1, cx2, cy2, cx3, cy3, cx4, cy4);

	cx1 = cx4;
	cy1 = cy4;
        cx2 = (float)(cx1 + x2) / 2;
        cy2 = (float)(cy1 + y2) / 2;

        node = node->Next();
    }

    wx_spline_add_point(wx_round(cx1), wx_round(cy1));
    wx_spline_add_point((int) x2, (int) y2);

    wx_spline_draw_point_array(dc);

}

void wx_draw_open_spline_ps(wxDC *dc, wxSpline *s)
{
	double		a, b, c, d, x1, y1, x2, y2, x3, y3;
        wxPoint *p, *q;
        dc->SetPen(dc->current_pen);

        wxNode *node = s->points->First();
        p = (wxPoint *)node->Data();

	x1 = p->x; y1 = p->y;

        node = node->Next();
        p = (wxPoint *)node->Data();
	c = p->x; d = p->y;
        x3 = a = (float)(x1 + c) / 2;
        y3 = b = (float)(y1 + d) / 2;

        *(dc->pstream) << "newpath " << x1 << " " << dc->yorigin - y1 << " moveto " << x3 << " " << dc->yorigin - y3;
        *(dc->pstream) << " lineto\n";
        dc->CalcBoundingBox((float)x1, (float)(dc->yorigin - y1));
        dc->CalcBoundingBox((float)x3, (float)(dc->yorigin - y3));

        node = node->Next();

        while (node)
	{
          q = (wxPoint *)node->Data();

	  x1 = x3; y1 = y3;
	  x2 = c;  y2 = d;
	  c = q->x; d = q->y;
          x3 = (float)(x2 + c) / 2;
          y3 = (float)(y2 + d) / 2;
          *(dc->pstream) << x1 << " " << dc->yorigin - y1 << " " << x2 << " " << dc->yorigin - y2 << " ";
          *(dc->pstream) << x3 << " " << dc->yorigin - y3 << " DrawSplineSection\n";

          dc->CalcBoundingBox((float)x1, (float)(dc->yorigin - y1));
          dc->CalcBoundingBox((float)x3, (float)(dc->yorigin - y3));


         node = node->Next();
        }
	/*
	* At this point, (x2,y2) and (c,d) are the position of the 
	* next-to-last and last point respectively, in the point list
	*/
        *(dc->pstream) << c << " " << dc->yorigin - d << " lineto stroke\n";
}

/********************* CURVES FOR SPLINES *****************************

	The following spline drawing routine is from

	"An Algorithm for High-Speed Curve Generation"
	by George Merrill Chaikin,
	Computer Graphics and Image Processing, 3, Academic Press,
	1974, 346-349.

	and

	"On Chaikin's Algorithm" by R. F. Riesenfeld,
	Computer Graphics and Image Processing, 4, Academic Press,
	1975, 304-310.

***********************************************************************/

#define		half(z1, z2)	((z1+z2)/2.0)
#define		THRESHOLD	5

/* iterative version */

void wx_quadratic_spline(float a1, float b1, float a2, float b2, float a3, float b3, float a4,
                 float b4)
{
    register float  xmid, ymid;
    float           x1, y1, x2, y2, x3, y3, x4, y4;

    wx_clear_stack();
    wx_spline_push(a1, b1, a2, b2, a3, b3, a4, b4);

    while (wx_spline_pop(&x1, &y1, &x2, &y2, &x3, &y3, &x4, &y4)) {
        xmid = (float)half(x2, x3);
        ymid = (float)half(y2, y3);
	if (fabs(x1 - xmid) < THRESHOLD && fabs(y1 - ymid) < THRESHOLD &&
	    fabs(xmid - x4) < THRESHOLD && fabs(ymid - y4) < THRESHOLD) {
            wx_spline_add_point(wx_round(x1), wx_round(y1));
            wx_spline_add_point(wx_round(xmid), wx_round(ymid));
	} else {
            wx_spline_push(xmid, ymid, (float)half(xmid, x3), (float)half(ymid, y3),
                 (float)half(x3, x4), (float)half(y3, y4), x4, y4);
            wx_spline_push(x1, y1, (float)half(x1, x2), (float)half(y1, y2),
                 (float)half(x2, xmid), (float)half(y2, ymid), xmid, ymid);
	}
    }
}


/* utilities used by spline drawing routines */


typedef struct wx_spline_stack_struct {
    float           x1, y1, x2, y2, x3, y3, x4, y4;
}
                Stack;

#define         SPLINE_STACK_DEPTH             20
static Stack    wx_spline_stack[SPLINE_STACK_DEPTH];
static Stack   *wx_stack_top;
static int      wx_stack_count;

void wx_clear_stack(void)
{
    wx_stack_top = wx_spline_stack;
    wx_stack_count = 0;
}

void wx_spline_push(float x1, float y1, float x2, float y2, float x3, float y3, float x4, float y4)
{
    wx_stack_top->x1 = x1;
    wx_stack_top->y1 = y1;
    wx_stack_top->x2 = x2;
    wx_stack_top->y2 = y2;
    wx_stack_top->x3 = x3;
    wx_stack_top->y3 = y3;
    wx_stack_top->x4 = x4;
    wx_stack_top->y4 = y4;
    wx_stack_top++;
    wx_stack_count++;
}

int wx_spline_pop(float *x1, float *y1, float *x2, float *y2,
                  float *x3, float *y3, float *x4, float *y4)
{
    if (wx_stack_count == 0)
	return (0);
    wx_stack_top--;
    wx_stack_count--;
    *x1 = wx_stack_top->x1;
    *y1 = wx_stack_top->y1;
    *x2 = wx_stack_top->x2;
    *y2 = wx_stack_top->y2;
    *x3 = wx_stack_top->x3;
    *y3 = wx_stack_top->y3;
    *x4 = wx_stack_top->x4;
    *y4 = wx_stack_top->y4;
    return (1);
}

static Bool wx_spline_add_point(int x, int y)
{
  wxPoint *point = new wxPoint(x, y);
  wx_spline_point_list.Append(point);
  return TRUE;
}

static void wx_spline_draw_point_array(wxDC *dc)
{
  dc->DrawLines(&wx_spline_point_list);
  wxNode *node = wx_spline_point_list.First();
  while (node)
  {
    wxPoint *point = (wxPoint *)node->Data();
    delete point;
    delete node;
    node = wx_spline_point_list.First();
  }
}

wxSpline::wxSpline(wxList *list)
{
  points = list;
}

wxSpline::~wxSpline(void)
{
}

void wxSpline::DeletePoints(void)
{
  wxNode *node = points->First();
  while (node)
  {
    wxPoint *point = (wxPoint *)node->Data();
    delete point;
    delete node;
    node = points->First();
  }
  delete points;
}


#ifdef wx_x
class wxPrinterDialogBox: public wxDialogBox
{
 public:
  wxPrinterDialogBox(wxFrame *frame, char *title, Bool modal = FALSE,
              int x = -1, int y = -1, int
              width = -1, int height = -1);
};

wxPrinterDialogBox::wxPrinterDialogBox(wxFrame *frame, char *title, Bool modal,
              int x, int y, int width, int height):
  wxDialogBox(frame, title, modal, x, y, width, height)
{
}

Bool wxPrinterDialogAnswer = TRUE;

void wxPrinterDialogOk(wxButton& button, wxEvent& event)
{
  wxPrinterDialogAnswer = TRUE;
  wxPrinterDialogBox *dialog = (wxPrinterDialogBox *)button.GetParent();
  dialog->Show(FALSE);
}

void wxPrinterDialogCancel(wxButton& button, wxEvent& event)
{
  wxPrinterDialogAnswer = FALSE;
  wxPrinterDialogBox *dialog = (wxPrinterDialogBox *)button.GetParent();
  dialog->Show(FALSE);
}

Bool XPrinterDialog(void)
{
  char buf[100];
  wxPrinterDialogBox dialog(NULL, "Printer Settings", TRUE, 150, 150, 400, 400);

  (void)new wxButton(&dialog, (wxFunction)wxPrinterDialogOk, "Ok");
  (void)new wxButton(&dialog, (wxFunction)wxPrinterDialogCancel, "Cancel");
  dialog.NewLine();
  dialog.NewLine();

  wxText text_prt(&dialog, (wxFunction)NULL, "Printer Command: ", wx_printer_command, -1, -1, 100, -1);
  dialog.NewLine();

  wxText text0(&dialog, (wxFunction)NULL, "Printer Options: ", wx_printer_flags, -1, -1, 200, -1);
  dialog.NewLine();

  wxCheckBox checkbox0(&dialog, (wxFunction)NULL, "Portrait");
  checkbox0.SetValue(wx_portrait);

  wxCheckBox checkbox1(&dialog, (wxFunction)NULL, "Print to file");
  checkbox1.SetValue(wx_print_to_file);

  wxCheckBox checkbox2(&dialog, (wxFunction)NULL, "Preview only");
  checkbox2.SetValue(wx_preview);
  dialog.NewLine();

  sprintf(buf, "%.2f", wx_printer_scale_x);
  wxText text1(&dialog, (wxFunction)NULL, "X Scaling: ", buf, -1, -1, 100, -1);

  sprintf(buf, "%.2f", wx_printer_scale_y);
  wxText text2(&dialog, (wxFunction)NULL, "Y Scaling: ", buf, -1, -1, 100, -1);

  dialog.NewLine();

  sprintf(buf, "%.2f", wx_printer_translate_x);
  wxText text3(&dialog, (wxFunction)NULL, "X Translation: ", buf, -1, -1, 100, -1);

  sprintf(buf, "%.2f", wx_printer_translate_y);
  wxText text4(&dialog, (wxFunction)NULL, "Y Translation: ", buf, -1, -1, 100, -1);

  dialog.NewLine();

  dialog.Fit();

  dialog.Show(TRUE);

  if (wxPrinterDialogAnswer)
  {
    StringToFloat(text1.GetValue(), &wx_printer_scale_x);
    StringToFloat(text2.GetValue(), &wx_printer_scale_y);
    StringToFloat(text3.GetValue(), &wx_printer_translate_x);
    StringToFloat(text4.GetValue(), &wx_printer_translate_y);

    wx_printer_flags = copystring(text0.GetValue());
    wx_printer_command = copystring(text_prt.GetValue());

    wx_portrait = checkbox0.GetValue();
    wx_print_to_file = checkbox1.GetValue();
    wx_preview = checkbox2.GetValue();
  }

  return wxPrinterDialogAnswer;
}

#endif

/*
 * Metafiles - Windows 3.1 only
 * Currently, the only purpose for making a metafile is to put
 * it on the clipboard.
 */

wxMetaFile::wxMetaFile(void)
{
#ifdef wx_msw
  metafile = NULL;
#endif
}

wxMetaFile::~wxMetaFile(void)
{
#ifdef wx_msw
  if (metafile)
    { DeleteMetaFile(metafile); metafile = NULL; }
#endif
}

Bool wxMetaFile::SetClipboard(int width, int height)
{
#ifdef wx_msw
  HANDLE data = GlobalAlloc(GHND, sizeof(METAFILEPICT) + 1);
  METAFILEPICT *mf = (METAFILEPICT *)GlobalLock(data);
  mf->mm = MM_ANISOTROPIC;
  mf->xExt = width;
  mf->yExt = height;
  mf->hMF = metafile;
  GlobalUnlock(data);
  metafile = NULL;

  wxFrame *frame = wxTheApp->wx_frame;
  if (!frame)
    return FALSE;

  wxWnd *wnd = (wxWnd *)frame->handle;
  OpenClipboard(wnd->handle);
  EmptyClipboard();
  HANDLE success = SetClipboardData(CF_METAFILEPICT, data);
  CloseClipboard();
  return (Bool)success;
#endif
#ifdef wx_x
  return FALSE;
#endif
}

/*
 * Metafile device context
 *
 */

wxMetaFileDC::wxMetaFileDC(char *file)
{
#ifdef wx_x
  ok = FALSE;
#endif
#ifdef wx_msw
  old_bitmap = NULL;
  old_pen = NULL;
  old_brush = NULL;
  old_font = NULL;

  metafile = NULL;
  wx_interactive = FALSE;
  canvas = NULL;
  font = NULL;

  device = wxDEVICE_WINDOWS;

  cdc = NULL;
  dont_delete = FALSE;
  clipping = FALSE;

  logical_origin_x = 0;
  logical_origin_y = 0;

  device_origin_x = 0;
  device_origin_y = 0;

  logical_scale_x = 1.0;
  logical_scale_y = 1.0;

  user_scale_x = 1.0;
  user_scale_y = 1.0;

  mapping_mode = MM_TEXT;

  min_x = 1000.0;
  min_y = 1000.0;
  max_x = -1000.0;
  max_y = -1000.0;
  title = NULL;

  pstream = NULL;
  filename = NULL;

  cdc = CreateMetaFile(file);
  ok = TRUE;
  SetMapMode(MM_TEXT);

  current_logical_function = -1;
  current_pen = wxBLACK_PEN;
  current_brush = wxBLACK_BRUSH;
  current_background_brush = wxWHITE_BRUSH;
  current_text_foreground = wxBLACK;
  current_text_background = wxWHITE;
  Colour = wxColourDisplay();
#endif
}

wxMetaFileDC::~wxMetaFileDC(void)
{
}

wxMetaFile *wxMetaFileDC::Close(void)
{
#ifdef wx_x
  return NULL;
#endif
#ifdef wx_msw
  HANDLE mf = CloseMetaFile(cdc);
  if (mf)
  {
    wxMetaFile *wx_mf = new wxMetaFile;
    wx_mf->metafile = mf;
    return wx_mf;
  }
  return NULL;
#endif
}

void wxMetaFileDC::SetMapMode(int mode)
{
#ifdef wx_msw
  mapping_mode = mode;

  int pixel_width = 0;
  int pixel_height = 0;
  int mm_width = 0;
  int mm_height = 0;

  float mm2pixelsX = 10.0;
  float mm2pixelsY = 10.0;

  switch (mode)
  {
    case MM_TWIPS:
    {
      logical_scale_x = (float)(twips2mm * mm2pixelsX);
      logical_scale_y = (float)(twips2mm * mm2pixelsY);
      break;
    }
    case MM_POINTS:
    {
      logical_scale_x = (float)(pt2mm * mm2pixelsX);
      logical_scale_y = (float)(pt2mm * mm2pixelsY);
      break;
    }
    case MM_METRIC:
    {
      logical_scale_x = mm2pixelsX;
      logical_scale_y = mm2pixelsY;
      break;
    }
    case MM_LOMETRIC:
    {
      logical_scale_x = (float)(mm2pixelsX/10.0);
      logical_scale_y = (float)(mm2pixelsY/10.0);
      break;
    }
    default:
    case MM_TEXT:
    {
      logical_scale_x = 1.0;
      logical_scale_y = 1.0;
      break;
    }
  }
  ::SetMapMode(cdc, MM_ANISOTROPIC);
  SetViewportExtEx(cdc, VIEWPORT_EXTENT, VIEWPORT_EXTENT, NULL);
  window_ext_x = MS_XDEV2LOG(VIEWPORT_EXTENT);
  window_ext_y = MS_YDEV2LOG(VIEWPORT_EXTENT);
  SetWindowExtEx(cdc, window_ext_x, window_ext_y, NULL);
#endif
}
