/*
 * File:     wx_dialg.cc
 * Purpose:  wxDialogBox and miscellaneous dialog functions
 *
 *                       wxWindows 1.40
 * Copyright (c) 1993 Artificial Intelligence Applications Institute,
 *                   The University of Edinburgh
 *
 *                     Author: Julian Smart
 *                       Date: 18-4-93
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose is hereby granted without fee, provided
 * that the above copyright notice, author statement and this permission
 * notice appear in all copies of this software and related documentation.
 *
 * THE SOFTWARE IS PROVIDED "AS-IS" AND WITHOUT WARRANTY OF ANY KIND, EXPRESS,
 * IMPLIED OR OTHERWISE, INCLUDING WITHOUT LIMITATION, ANY WARRANTY OF
 * MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * IN NO EVENT SHALL THE ARTIFICIAL INTELLIGENCE APPLICATIONS INSTITUTE OR THE
 * UNIVERSITY OF EDINBURGH BE LIABLE FOR ANY SPECIAL, INCIDENTAL, INDIRECT OR
 * CONSEQUENTIAL DAMAGES OF ANY KIND, OR ANY DAMAGES WHATSOEVER RESULTING FROM
 * LOSS OF USE, DATA OR PROFITS, WHETHER OR NOT ADVISED OF THE POSSIBILITY OF
 * DAMAGE, AND ON ANY THEORY OF LIABILITY, ARISING OUT OF OR IN CONNECTION WITH
 * THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */

#include <windows.h>
#include <iostream.h>
#include <stdio.h>
#include "common.h"
#include "wx_dialg.h"
#include "wx_utils.h"
#include "wx_frame.h"
#include "wx_privt.h"
#include "wx_main.h"

#define wxDIALOG_DEFAULT_X 300
#define wxDIALOG_DEFAULT_Y 300

#ifdef wx_motif
#include <Xm/Label.h>
#include <Xm/Text.h>
#include <Xm/FileSB.h>
#include <Xm/RowColumn.h>
char *wxMotifFileSelector(char *message,
                     char *default_path, char *default_filename, 
                     char *default_extension, char *wildcard, int flags,
                     wxFrame *parent, int x, int y);
#endif

#ifdef wx_xview
char *wxXFileSelector(wxFrame *parent, char *path, char *file, char *message, int flags,
  char *wild_card);
#endif

#ifdef wx_xview
#include <dirent.h>
#include <unistd.h>
#include <xview/panel.h>
extern "C" int xv_window_loop(Frame);
extern "C" int xv_window_return(int);
extern void wxPanelEventProc(Panel panel, Event *event); // See wx_item.cc
#endif

#ifdef wx_msw
#include <commdlg.h>

char *wxmswFileSelector(char *message,
                     char *default_path, char *default_filename, 
                     char *default_extension, char *wildcard, int flags,
                     wxFrame *parent, int x, int y);

// Lists to keep track of windows, so we can disable/enable them
// for modal dialogs
wxList wxModalDialogs;
wxList wxModelessWindows;  // Frames and modeless dialogs

class wxDialogWnd : public wxSubWnd
{
public:
  wxDialogWnd(wxWnd *parent, wxWindow *wx_win,
              int x, int y, int width, int height,
              char *dialog_template);

  // Handlers
  LONG DefWindowProc(UINT nMsg, UINT wParam, LONG lParam);
  BOOL ProcessMessage(MSG* pMsg);
  BOOL OnEraseBkgnd(HDC pDC);
  BOOL OnClose(void);
};

wxDialogWnd::wxDialogWnd(wxWnd *parent, wxWindow *wx_win,
              int x, int y, int width, int height,
              char *dialog_template):
  wxSubWnd(parent, NULL, wx_win, x, y, width, height, 0, dialog_template)
{
}

LONG wxDialogWnd::DefWindowProc(UINT nMsg, UINT wParam, LONG lParam)
{
  return ::DefWindowProc(handle, nMsg, wParam, lParam);
}

BOOL wxDialogWnd::ProcessMessage(MSG* pMsg)
{
  return ::IsDialogMessage(handle, pMsg);
}

BOOL wxDialogWnd::OnClose(void)
{
  if (wx_window)
  {
    if (wx_window->OnClose())
    {
      ((wxDialogBox *)wx_window)->Show(FALSE);
      delete wx_window;
      return TRUE;
    } else return FALSE;
  }
  return FALSE;
}


BOOL wxDialogWnd::OnEraseBkgnd(HDC pDC)
{
  if (background_brush)
  {
    RECT rect;
    GetClientRect(handle, &rect);
    int mode = SetMapMode(pDC, MM_TEXT);
    FillRect(pDC, &rect, background_brush);
    SetMapMode(pDC, mode);
    return TRUE;
  }
  else return FALSE;
}

#endif



// Dialog box - like panel but doesn't need a frame, and is modal or
// non-modal

wxDialogBox::wxDialogBox(wxFrame *Parent, char *Title, Bool Modal, 
                         int x, int y, int width, int height):wxPanel(NULL)
{
  if (Parent) Parent->AddChild(this);
  window_parent = Parent;

  if (x < 0) x = wxDIALOG_DEFAULT_X;
  if (y < 0) y = wxDIALOG_DEFAULT_Y;

#ifdef wx_motif
  borderWidget = 0;
  wxType = 1;
  hSpacing = PANEL_HSPACING;
  vSpacing = PANEL_VSPACING;

  int real_y = y;

  new_line = TRUE;
  firstRowWidget = NULL;
  lastWidget = NULL;

  dialogShell = XtAppCreateShell(NULL, "Shell", 
         topLevelShellWidgetClass, XtDisplay(wxTheApp->topLevel), NULL, 0);

  panelWidget = XtVaCreateManagedWidget("panel",
                  xmFormWidgetClass, dialogShell,
                  XmNx, 0,
                  XmNy, 0,
                  NULL);

  if (Title)
    XtVaSetValues(dialogShell, 
                  XmNtitle, Title,
                  NULL);

  XtVaSetValues(dialogShell, 
                 XmNallowShellResize, True,
                 XmNdeleteResponse, XmDO_NOTHING,
                 XmNmappedWhenManaged, False,
                 NULL);

  /*
   * CAN ANYONE explain why the border of a dialog box is messed up by widgets,
   * to the bottom and right of the box? Why does the form allow this to happen?
   * This attempt below doesn't work, it obliterates the bottom and right
   * borders, but at least it looks more consistent than mouse-bites.
   *
   */
  // Put a rowcolumn to the right and along the bottom as a border
  XtVaCreateManagedWidget("rowcol",
                  xmRowColumnWidgetClass, panelWidget,
                  XmNorientation, XmVERTICAL,
                  XmNtopAttachment,        XmATTACH_FORM,
                  XmNbottomAttachment,     XmATTACH_FORM,
                  XmNrightAttachment,      XmATTACH_FORM,
                  XmNwidth, 2,
                  NULL);
  XtVaCreateManagedWidget("rowcol",
                  xmRowColumnWidgetClass, panelWidget,
                  XmNorientation, XmHORIZONTAL,
                  XmNleftAttachment,        XmATTACH_FORM,
                  XmNbottomAttachment,     XmATTACH_FORM,
                  XmNrightAttachment,      XmATTACH_FORM,
                  XmNheight, 2,
                  NULL);


  wxWidgetHashTable->Put((long)panelWidget, this);

  XtOverrideTranslations(panelWidget,
              XtParseTranslationTable("<Configure>: resize()"));

  if (x > -1)
    XtVaSetValues(panelWidget, XmNx, x, 
                  NULL);
  if (y > -1)
    XtVaSetValues(panelWidget, XmNy, real_y,
                  NULL);
  if (width > -1)
    XtVaSetValues(panelWidget, XmNwidth, width, NULL);
  if (height > -1)
    XtVaSetValues(panelWidget, XmNheight, height, NULL);

  XtRealizeWidget(dialogShell);
  handle = (char *)panelWidget;
#endif
#ifdef wx_xview
  Frame parent;
  if (Parent == (wxFrame *)NULL)
    {
      parent = (Frame)NULL;
    }
  else
    {
      parent = (Frame)(Parent->GetHandle());
    };
  frame = (Frame) xv_create(parent, FRAME, FRAME_LABEL, Title,
			      XV_X, x, XV_Y, y,
                              WIN_CLIENT_DATA, (char *)this,
                              NULL);

  Panel x_panel = (Panel)xv_create(frame, PANEL,
                                   PANEL_BACKGROUND_PROC, wxPanelEventProc,
                                   PANEL_ACCEPT_KEYSTROKE, TRUE,
                                   WIN_CLIENT_DATA, (char *)this,
                                   NULL);

  if (width > -1)
  {
    xv_set(frame, XV_WIDTH, width, NULL);
    xv_set(x_panel, XV_WIDTH, width, NULL);
  }

  if (height > -1)
  {
    xv_set(frame, XV_HEIGHT, height, NULL);
    xv_set(x_panel, XV_HEIGHT, height, NULL);
  }

  handle = (char *)x_panel;
#endif
#ifdef wx_msw
  wxWinType = wxTYPE_XWND;
  wxWnd *cparent = NULL;
  if (Parent)
    cparent = (wxWnd *)Parent->handle;

  if (width < 0)
    width = 500;
  if (height < 0)
    height = 500;

  wxDialogWnd *wnd = new wxDialogWnd(cparent, this,
                                      x, y, width, height,
                                      "wxDummyDialog");

  handle = (char *)wnd;
  SetWindowText(wnd->handle, Title);

  if (!Modal)
    wxModelessWindows.Append(this);
#endif

  modal = Modal;

}

wxDialogBox::~wxDialogBox()
{
#ifdef wx_motif
  modal_showing = FALSE;
//  XtUnmapWidget(dialogShell);
#endif
#ifdef wx_xview
 xv_destroy_safe(frame);
#endif
#ifdef wx_msw
  wxWnd *wnd = (wxWnd *)handle;
  modal_showing = FALSE;
  ShowWindow(wnd->handle, SW_HIDE);
  if (!modal)
    wxModelessWindows.DeleteObject(this);
#endif
}

#ifdef wx_motif
void wxWindow::PostDestroyChildren(void)
{
  if (wxType == 1)
  {
    wxDialogBox *box = (wxDialogBox *)this;
    XtDestroyWidget(box->dialogShell);

    // Now process all events, because otherwise
    // this might remain on the screen
    XSync(XtDisplay(wxTheApp->topLevel), FALSE);
    XEvent event;
    while (XtAppPending(wxTheApp->appContext))
    {
      XFlush(XtDisplay(wxTheApp->topLevel));
      XtAppNextEvent(wxTheApp->appContext, &event);
      XtDispatchEvent(&event);
    }
  }
}
#endif

void wxDialogBox::Fit(void)
{
#ifdef wx_motif
#endif
#ifdef wx_xview
  Panel panel = (Panel)handle;
  window_fit(panel);
  window_fit(frame);
#endif
#ifdef wx_msw
  wxPanel::Fit();
#endif
}

void wxDialogBox::Centre(int direction)
{
  int display_width, display_height, width, height, x, y;
  wxDisplaySize(&display_width, &display_height);

  GetSize(&width, &height);
  GetPosition(&x, &y);

  if (direction == wxBOTH || direction == wxHORIZONTAL)
    x = (int)((display_width - width)/2);
  if (direction == wxBOTH || direction == wxVERTICAL)
    y = (int)((display_height - height)/2);

  SetSize(x, y, width, height);
}

void wxDialogBox::Iconize(Bool iconize)
{
#ifdef wx_motif
  XtVaSetValues(dialogShell, XmNiconic, iconize, NULL);
#endif
#ifdef wx_xview
  xv_set(frame, FRAME_CLOSED, iconize, NULL);
#endif
#ifdef wx_msw
  // Windows dialog boxes can't be iconized
#endif
}

Bool wxDialogBox::Iconized(void)
{
#ifdef wx_motif
  Bool iconic;
  XtVaGetValues(dialogShell, XmNiconic, &iconic, NULL);
  return iconic;
#endif
#ifdef wx_xview
  return xv_get(frame, FRAME_CLOSED);
#endif
#ifdef wx_msw
  return FALSE;
#endif
}

void wxDialogBox::SetSize(int x, int y, int width, int height)
{
#ifdef wx_motif
  if (x > -1)
    XtVaSetValues(dialogShell, XmNx, x, NULL);
  if (y > -1)
    XtVaSetValues(dialogShell, XmNy, y, NULL);
  if (width > -1)
    XtVaSetValues((Widget)handle, XmNwidth, width, NULL);
  if (height > -1)
    XtVaSetValues((Widget)handle, XmNheight, height, NULL);
  OnSize(width, height);
#endif
#ifdef wx_xview
  (void)xv_set(frame, XV_X, x, XV_Y, y, XV_WIDTH, width, XV_HEIGHT, height, NULL);
  OnSize(width, height);
#endif
#ifdef wx_msw
  wxWindow::SetSize(x, y, width, height);
#endif
}

void wxDialogBox::SetClientSize(int width, int height)
{
#ifdef wx_motif
  if (width > -1)
    XtVaSetValues((Widget)handle, XmNwidth, width, NULL);
  if (height > -1)
    XtVaSetValues((Widget)handle, XmNheight, height, NULL);
  OnSize(width, height);
#endif
#ifdef wx_xview
  (void)xv_set(frame, XV_WIDTH, width, XV_HEIGHT, height, NULL);
#endif
#ifdef wx_msw
  wxWnd *wnd = (wxWnd *)handle;
  RECT rect;
  GetClientRect(wnd->handle, &rect);

  RECT rect2;
  GetWindowRect(wnd->handle, &rect2);

  // Find the difference between the entire window (title bar and all)
  // and the client area; add this to the new client size to move the
  // window
  int actual_width = rect2.right - rect2.left - rect.right + width;
  int actual_height = rect2.bottom - rect2.top - rect.bottom + height;

  MoveWindow(wnd->handle, rect2.left, rect2.top, actual_width, actual_height, TRUE);
  OnSize(actual_width, actual_height);
#endif
}

void wxDialogBox::GetPosition(int *x, int *y)
{
#ifdef wx_motif
  Dimension xx, yy;
  XtVaGetValues(dialogShell, XmNx, &xx, XmNy, &yy, NULL);
  *x = xx; *y = yy;
#endif
#ifdef wx_xview
  *x = (int)xv_get(frame, XV_X);
  *y = (int)xv_get(frame, XV_Y);
#endif
#ifdef wx_msw
  wxWindow::GetPosition(x, y);
#endif
}

void wxDialogBox::Show(Bool show)
{
#ifdef wx_motif
  if (show)
  {
    XtMapWidget(dialogShell);
    XRaiseWindow(XtDisplay(dialogShell), XtWindow(dialogShell));
    if (modal)
    {
      modal_showing = TRUE;
      XtAddGrab(dialogShell, TRUE, FALSE);
      XEvent event;
      while (modal_showing || XtAppPending(wxTheApp->appContext))
      {
        XtAppNextEvent(wxTheApp->appContext, &event);
        XtDispatchEvent(&event);
      }
    }
  }
  else
  {
    if (modal_showing)
      XtRemoveGrab(dialogShell);

    XtUnmapWidget(dialogShell);
    modal_showing = FALSE;
  }
#endif
#ifdef wx_xview
  if (show)
    {
      xv_set(frame, XV_SHOW, TRUE, NULL);

      if (modal)
      {
        xv_set(frame, WIN_GRAB_ALL_INPUT, FALSE, NULL);
        xv_window_loop(frame);
       }
    }
  else
    {
     xv_set(frame, WIN_GRAB_ALL_INPUT, FALSE, NULL);

     xv_set(frame, XV_SHOW, FALSE, NULL);
     if (modal)
     {
       xv_window_return(0);
     }
    }
#endif
#ifdef wx_msw
  wxWnd *dialog = (wxWnd *)handle;
  if (modal)
  {
    if (show)
    {
      EnableWindow(dialog->handle, TRUE);
      ShowWindow(dialog->handle, SW_SHOW);
      BringWindowToTop(dialog->handle);
      modal_showing = TRUE;

      wxNode *node = wxModalDialogs.First();
      while (node)
      {
        wxDialogBox *box = (wxDialogBox *)node->Data();
        wxWnd *the_dialog = (wxWnd *)box->handle;
        if (box != this)
          EnableWindow(the_dialog->handle, FALSE);
        node = node->Next();
      }
      node = wxModelessWindows.First();
      while (node)
      {
        wxWindow *win = (wxWindow *)node->Data();
        wxWnd *wnd = (wxWnd *)win->handle;
        EnableWindow(wnd->handle, FALSE);
        node = node->Next();
      }

      wxModalDialogs.Append(this);

      MSG msg;
      while (modal_showing && GetMessage(&msg, NULL, 0, 0))
      {
        if (!IsDialogMessage(dialog->handle, &msg))
        {
          TranslateMessage(&msg);
          DispatchMessage(&msg);
        }
      }
    }
    else
    {
      wxModalDialogs.DeleteObject(this);

      wxNode *first = wxModalDialogs.First();

      // If there's still a modal dialog active, we
      // enable it, else we enable all modeless windows

      if (first)
      {
        wxDialogBox *box = (wxDialogBox *)first->Data();
        wxWnd *the_dialog = (wxWnd *)box->handle;
        EnableWindow(the_dialog->handle, TRUE);
        BringWindowToTop(dialog->handle);
      }
      else
      {
        wxNode *node = wxModelessWindows.First();
        while (node)
        {
          wxWindow *win = (wxWindow *)node->Data();
          wxWnd *wnd = (wxWnd *)win->handle;
          EnableWindow(wnd->handle, TRUE);
          node = node->Next();
        }
      }
      ShowWindow(dialog->handle, SW_HIDE);
      modal_showing = FALSE;
    }
  }
  else
  {
    if (show)
    {
      ShowWindow(dialog->handle, SW_SHOW);
      BringWindowToTop(dialog->handle);
    }
    else
    {
      ShowWindow(dialog->handle, SW_HIDE);
    }
  }
#endif
}

/*
 * Common dialogs code
 *
 */

int wxDialogButtonPressed = 0;

void wxDialogOkButton(wxButton& but, wxEvent& event)
{
  wxDialogBox *dialog = (wxDialogBox *)but.GetParent();
  wxDialogButtonPressed = wxOK;
  dialog->Show(FALSE);
}

void wxDialogCancelButton(wxButton& but, wxEvent& event)
{
  wxDialogBox *dialog = (wxDialogBox *)but.GetParent();
  wxDialogButtonPressed = wxCANCEL;
  dialog->Show(FALSE);
}

void wxDialogYesButton(wxButton& but, wxEvent& event)
{
  wxDialogBox *dialog = (wxDialogBox *)but.GetParent();
  wxDialogButtonPressed = wxYES;
  dialog->Show(FALSE);
}

void wxDialogNoButton(wxButton& but, wxEvent& event)
{
  wxDialogBox *dialog = (wxDialogBox *)but.GetParent();
  wxDialogButtonPressed = wxNO;
  dialog->Show(FALSE);
}

// Return NULL if cancel pressed
char *wxGetTextFromUser(char *message, char *caption, char *default_value,
                        wxFrame *parent, int x, int y)
{
  if (x < 0) x = wxDIALOG_DEFAULT_X;
  if (y < 0) y = wxDIALOG_DEFAULT_Y;

  wxDialogBox dialog(parent, caption, TRUE, x, y, 500, 160);
  dialog.SetClientSize(320, 160);

  (void)new wxMessage(&dialog, message);
  dialog.NewLine();

  wxText text(&dialog, NULL, NULL, default_value, -1, -1, 320);
  dialog.NewLine();

  wxButton *ok = new wxButton(&dialog, (wxFunction)&wxDialogOkButton, "OK");
  (void)new wxButton(&dialog, (wxFunction)&wxDialogCancelButton, "Cancel");

  wxDialogButtonPressed = 0;

  ok->SetDefault();
  text.SetFocus();

  dialog.Fit();
  dialog.Centre();
  dialog.Show(TRUE);

  if (wxDialogButtonPressed == wxOK)
    return text.GetValue();
  else
    return NULL;
}

#define wxCHOICE_HEIGHT 150
#define wxCHOICE_WIDTH 200

char *wxGetSingleChoice(char *message, char *caption, int n, char *choices[],
                        wxFrame *parent, int x, int y)
{
  if (x < 0) x = wxDIALOG_DEFAULT_X;
  if (y < 0) y = wxDIALOG_DEFAULT_Y;

  // Really want to centre on screen
  wxDialogBox dialog(parent, caption, TRUE, x, y, 300, 250);
  dialog.SetClientSize(300, 250);
  (void)new wxMessage(&dialog, message);
  dialog.NewLine();

  wxListBox listbox(&dialog, NULL, NULL, wxSINGLE,
                     -1, -1, wxCHOICE_WIDTH, wxCHOICE_HEIGHT, n, choices);

  dialog.NewLine();
  wxButton *ok = new wxButton(&dialog, (wxFunction)&wxDialogOkButton, "OK");
  (void)new wxButton(&dialog, (wxFunction)&wxDialogCancelButton, "Cancel");
  ok->SetDefault();

  wxDialogButtonPressed = 0;
  dialog.Fit();
  dialog.Centre();
  dialog.Show(TRUE);
  if (wxDialogButtonPressed == wxOK)
  {
    return listbox.String(listbox.GetSelection());
  }
  else
    return NULL;
}

int wxGetSingleChoiceIndex(char *message, char *caption, int n, char *choices[],
                           wxFrame *parent, int x, int y)
{
  // Really want to centre on screen
  if (x < 0) x = wxDIALOG_DEFAULT_X;
  if (y < 0) y = wxDIALOG_DEFAULT_Y;


  wxDialogBox dialog(parent, caption, TRUE, x, y, 300, 250);
  dialog.SetClientSize(300, 250);
  (void)new wxMessage(&dialog, message);
  dialog.NewLine();

  wxListBox listbox(&dialog, NULL, NULL, wxSINGLE,
                     -1, -1, wxCHOICE_WIDTH, wxCHOICE_HEIGHT, n, choices);
  dialog.NewLine();

  wxButton *ok = new wxButton(&dialog, (wxFunction)&wxDialogOkButton, "OK");
  (void)new wxButton(&dialog, (wxFunction)&wxDialogCancelButton, "Cancel");
  ok->SetDefault();

  wxDialogButtonPressed = 0;
  dialog.Fit();
  dialog.Centre();
  dialog.Show(TRUE);
  if (wxDialogButtonPressed == wxOK)
  {
    return listbox.GetSelection();
  }
  else
    return -1;
}

char *wxGetSingleChoiceData(char *message, char *caption, int n,
                            char *choices[], char *client_data[],
                            wxFrame *parent, int x, int y)
{
  // Really want to centre on screen
  if (x < 0) x = wxDIALOG_DEFAULT_X;
  if (y < 0) y = wxDIALOG_DEFAULT_Y;

  wxDialogBox dialog(parent, caption, TRUE, x, y, 300, 250);
  dialog.SetClientSize(300, 250);
  (void)new wxMessage(&dialog, message);
  dialog.NewLine();

  wxListBox listbox(&dialog, NULL, NULL, wxSINGLE, -1, -1,
                     wxCHOICE_WIDTH, wxCHOICE_HEIGHT);
  int i;
  for (i = 0; i < n; i++)
    listbox.Append(choices[i], client_data[i]);

  dialog.NewLine();

  wxButton *ok = new wxButton(&dialog, (wxFunction)&wxDialogOkButton, "OK");
  (void)new wxButton(&dialog, (wxFunction)&wxDialogCancelButton, "Cancel");
  ok->SetDefault();

  wxDialogButtonPressed = 0;
  dialog.Fit();
  dialog.Centre();
  dialog.Show(TRUE);
  if (wxDialogButtonPressed == wxOK)
  {
    int sel = listbox.GetSelection();
    return listbox.GetClientData(sel);
  }
  else
    return NULL;
}

// Pop up a message box
int wxMessageBox(char *message, char *caption, int type,
                 wxFrame *parent, int x, int y)
{
  wxDialogBox dialog(parent, caption, TRUE, x, y, 500, 500);

  // Format and centre the message
  wxList messageList;
  char *copyMessage = copystring(message);
  int i = 0;
  int len = strlen(copyMessage);
  char *currentMessage = copyMessage;
  while (i < len)
  {
    while ((i < len) && (copyMessage[i] != '\n')) i ++;
    if (i < len) copyMessage[i] = 0;
    wxMessage *mess = new wxMessage(&dialog, currentMessage);
    messageList.Append(mess);
    dialog.NewLine();

    currentMessage = copyMessage + i + 1;
  }
  delete copyMessage;
  dialog.NewLine();

  wxButton *ok = NULL;
  wxButton *cancel = NULL;
  wxButton *yes = NULL;
  wxButton *no = NULL;

  if (type & wxYES_NO)
  {
    yes = new wxButton(&dialog, (wxFunction)&wxDialogYesButton, "Yes");
    no = new wxButton(&dialog, (wxFunction)&wxDialogNoButton, "No");
  }

  if (type & wxOK)
  {
    ok = new wxButton(&dialog, (wxFunction)&wxDialogOkButton, "OK");
  }

  if (type & wxCANCEL)
  {
    cancel = new wxButton(&dialog, (wxFunction)&wxDialogCancelButton, "Cancel");
  }

  dialog.Fit();

  // Do the message centering
  wxNode *node = messageList.First();
  while (node)
  {
    wxMessage *mess = (wxMessage *)node->Data();
    mess->Centre();
    node = node->Next();
  }
  if (ok && !cancel && !yes && !no)
    ok->Centre();

  if ((x < 0) || (y < 0))
    dialog.Centre();

  if (ok)
    ok->SetDefault();
  else if (yes)
    yes->SetDefault();

  wxDialogButtonPressed = 0;
  dialog.Show(TRUE);

  if (ok)
    delete ok;
  if (cancel)
    delete cancel;
  if (yes)
    delete yes;
  if (no)
    delete no;

  return wxDialogButtonPressed;
}

char *wxFileSelector(char *message,
                     char *default_path, char *default_filename, 
                     char *default_extension, char *wildcard, int flags,
                     wxFrame *parent, int x, int y)
{
  if (x < 0) x = wxDIALOG_DEFAULT_X;
  if (y < 0) y = wxDIALOG_DEFAULT_Y;
#ifdef wx_motif
  return wxMotifFileSelector(message, default_path, default_filename, 
                     default_extension, wildcard, flags,
                     parent, x, y);
#endif
#ifdef wx_msw
  return wxmswFileSelector(message, default_path, default_filename,
                  default_extension, wildcard, flags, parent, x, y);
#endif
#ifdef wx_xview
  char buf[400];
  if (!default_path)
    default_path = getcwd(buf, 400);
  if (!default_filename)
    default_filename = "";

  return wxXFileSelector(parent, default_path, default_filename, message, flags, wildcard);
#endif
}

#ifdef wx_xview
void wxReadDir(wxListBox *filebox, wxListBox *dirbox, char *path, char *filename, DIR *dirp)
{
  filebox->Show(FALSE);
  filebox->Clear();
  dirbox->Show(FALSE);
  dirbox->Clear();
  struct dirent *dir;

  dirbox->Append("..");

  DIR *dirstream = NULL;
  if ((dir=readdir(dirp))==NULL)
  {
	printf("There has been an error in opening the directory\n");
	return;
  }

  char buf[400];

  // Skip the first directory (.)

  wxStringList files;
  wxStringList dirs;

  while((dir=readdir(dirp))!=NULL)
  {
    strcpy(buf, path);
    int len = strlen(buf);

    if (buf[len - 1] != '/')
      strcat(buf, "/");

    strcat(buf, dir->d_name);
    if (dirstream = opendir(buf))
    {
      closedir(dirstream);

      if (dir->d_name[0] != '.')
        dirs.Add(dir->d_name);
    }
    else if (!wxIsWild(filename) || wxMatchWild(filename, dir->d_name))
      files.Add(dir->d_name);

  }
  files.Sort();
  dirs.Sort();

  wxNode *node = files.First();
  while (node)
  {
    char *s = (char *)node->Data();
    filebox->Append(s);
    delete s;
    delete node;
    node = files.First();
  }

  node = dirs.First();
  while (node)
  {
    char *s = (char *)node->Data();
    dirbox->Append(s);
    delete s;
    delete node;
    node = dirs.First();
  }

  filebox->Show(TRUE);
  dirbox->Show(TRUE);
}

class wxXFileSelDialog: public wxDialogBox
{
 public:
  char *wild_card;
  wxText *name_item;
  wxText *path_item;
  wxListBox *list_item;
  wxListBox *file_item;
  wxListBox *dir_item;
  wxXFileSelDialog(wxFrame *frame, char *title, Bool modal = FALSE,
              int x = -1, int y = -1, int
              width = -1, int height = -1);
};

wxXFileSelDialog::wxXFileSelDialog(wxFrame *frame, char *title, Bool modal,
              int x, int y, int width, int height):
  wxDialogBox(frame, title, modal, x, y, width, height)
{
}

void wxXFileSelPath(wxText& text, wxEvent& event)
{
}

void wxStripOneDir(char *dir)
{
  int len = strlen(dir);
  if (len > 1)
  {
    int i = len - 1;
    char ch = 0;
    while (ch != '/' && i > 0)
    {
      ch = dir[i];
      i --;
    }
    i ++;
    if (i == 0)
      i ++;
    dir[i] = 0;
  }
}


void wxXFileSelDirList(wxListBox& listbox, wxEvent& event)
{
  char *name = copystring(listbox.GetStringSelection());
  wxXFileSelDialog *dialog = (wxXFileSelDialog *)listbox.GetParent();

  char *path = copystring(dialog->path_item->GetValue());
  char buf[500];
  strcpy(buf, path);

  int len = strlen(buf);

  if (strcmp(name, "..") == 0)
  {
    wxStripOneDir(buf);
  }
  else
  {
    if (buf[len - 1] != '/')
      strcat(buf, "/");
    strcat(buf, name);
  }

  DIR *dirstream = NULL;
  if (dirstream = opendir(buf))
  {
    dialog->path_item->SetValue(buf);
    wxReadDir(dialog->file_item, &listbox, buf, dialog->name_item->GetValue(), dirstream);
    closedir(dirstream);
  }
  else dialog->name_item->SetValue(name);
}

void wxXFileSelFileList(wxListBox& listbox, wxEvent& event)
{
  char *name = copystring(listbox.GetStringSelection());
  wxXFileSelDialog *dialog = (wxXFileSelDialog *)listbox.GetParent();

  dialog->name_item->SetValue(name);
}

int wxXFileSelResponse = 0;
char *wxXFileSelAnswer = NULL;

void wxXFileSelOK(wxButton& ok, wxEvent& event)
{
  wxXFileSelDialog *dialog = (wxXFileSelDialog *)ok.GetParent();
  char *nameval = dialog->name_item->GetValue();
  char *pathval = dialog->path_item->GetValue();
  if (wxIsWild(nameval))
  {
    DIR *dirstream = NULL;
    if (dirstream = opendir(pathval))
    {
      wxReadDir(dialog->file_item, dialog->dir_item, pathval, nameval, dirstream);
      closedir(dirstream);
    }
    return;
  }

  char *name = copystring(nameval);
  char *path = copystring(pathval);

  char buf[400];
  strcpy(buf, path);
  int len = strlen(buf);
  if (buf[len - 1] != '/')
    strcat(buf, "/");
  strcat(buf, name);

  delete name;
  delete path;

  if (wxXFileSelAnswer)
    delete wxXFileSelAnswer;

  wxXFileSelAnswer = copystring(buf);  

  dialog->Show(FALSE);
  delete dialog;

  wxXFileSelResponse = 1;
}

void wxXFileSelCancel(wxButton& cancel, wxEvent& event)
{
  wxXFileSelDialog *dialog = (wxXFileSelDialog *)cancel.GetParent();
  wxXFileSelResponse = 0;
  dialog->Show(FALSE);
  delete dialog;
}

char *wxXFileSelector(wxFrame *parent, char *path, char *file, char *message, int flags, char *wild_card)
{
  wxXFileSelResponse = 0;
  DIR *dirstream;
  if (!path)
  {
    char buf[400];
    path = getcwd(buf, 400);
  }
  if (!file)
    file = "";

  if ((dirstream=opendir(path))!=NULL)
  {
    wxXFileSelDialog *dialog = new wxXFileSelDialog(parent, "File selector", TRUE, 150, 150, 800, 800);

    (void)new wxMessage(dialog, message);
    dialog->NewLine();
    wxText *name_text = new wxText(dialog, (wxFunction)NULL, "Name", file, -1, -1, 300, -1);
    dialog->NewLine();
    wxText *path_text = new wxText(dialog, (wxFunction)wxXFileSelPath, "Path", path, -1, -1, 350, -1);
    dialog->NewLine();
    wxListBox *filebox = new wxListBox(dialog, (wxFunction)wxXFileSelFileList, "Files", wxSINGLE,
                                       4, 80, 200, 300);
    wxListBox *dirbox = new wxListBox(dialog, (wxFunction)wxXFileSelDirList, "Directories", wxSINGLE,
                                      -1, -1, 200, 300);
    dialog->NewLine();

    (void)new wxButton(dialog, (wxFunction)wxXFileSelOK, "OK");
    (void)new wxButton(dialog, (wxFunction)wxXFileSelCancel, "Cancel");

    dialog->path_item = path_text;
    dialog->name_item = name_text;
    dialog->dir_item = dirbox;
    dialog->file_item = filebox;
    dialog->wild_card = wild_card;

    dialog->Fit();

    char *initial_name = file;
    if (((initial_name == NULL) || (strcmp(initial_name, "") == 0)) &&
        wild_card)
    {
      initial_name = wild_card;
      dialog->name_item->SetValue(initial_name);
    }

    wxReadDir(filebox, dirbox, path, initial_name, dirstream);
    dialog->Show(TRUE);

    if (wxXFileSelResponse == 0)
      return NULL;
    else
    {
      if ((flags & wxOVERWRITE_PROMPT) && FileExists(wxXFileSelAnswer))
      {
        char buf[200];
        sprintf(buf, "Overwrite existing file %s?", wxXFileSelAnswer);
        int ans = wxMessageBox(buf, "Warning", wxYES_NO);
        if (ans == wxYES)
          return copystring(wxXFileSelAnswer);
        else return NULL;
      } else return copystring(wxXFileSelAnswer);
    }
  }
  else return NULL;
}
#endif

#ifdef wx_msw
char *wxmswFileSelector(char *message,
                     char *default_path, char *default_filename, 
                     char *default_extension, char *wildcard, int flags,
                     wxFrame *parent, int x, int y)
{
  wxWnd *wnd = NULL;
  HWND hwnd = NULL;
  if (parent)
  {
    wnd = (wxWnd *)parent->handle;
    hwnd = wnd->handle;
  }
  char file_buffer[400];

  if (default_filename)
    strcpy(file_buffer, default_filename);
  else file_buffer[0] = 0;

  char title_buffer[50];
  title_buffer[0] = 0;

  char filter_buffer[200];

  if (!wildcard)
    wildcard = "*.*";

  if (wildcard)
  {
    sprintf(filter_buffer, "Files (%s)", wildcard);
    int len1 = strlen(filter_buffer);
    int len2 = strlen(wildcard);

    filter_buffer[len1] = 0;
    int i;
    for (i = 0; i < len2; i++)
      filter_buffer[len1 + 1 + i] = wildcard[i];
    filter_buffer[len1 + 1 + len2] = 0;
    filter_buffer[len1 + 2 + len2] = 0;
  }

  OPENFILENAME of;
  memset(&of, 0, sizeof(OPENFILENAME));

  of.lStructSize = sizeof(OPENFILENAME);
  of.hwndOwner = hwnd;

  if (wildcard)
  {
    of.lpstrFilter = (LPSTR)filter_buffer;
    of.nFilterIndex = 1L;
  }
  else
  {
    of.lpstrFilter = NULL;
    of.nFilterIndex = 0L;
  }
  of.lpstrCustomFilter = NULL;
  of.nMaxCustFilter = 0L;
  of.lpstrFile = file_buffer;
  of.nMaxFile = 400;
  of.lpstrFileTitle = title_buffer;
  of.nMaxFileTitle = 50;
  of.lpstrInitialDir = default_path;
  of.lpstrTitle = message;
  of.nFileOffset = 0;
  of.nFileExtension = 0;
  of.lpstrDefExt = default_extension;

  int msw_flags = 0;

  if (flags & wxOVERWRITE_PROMPT)
    msw_flags |= OFN_OVERWRITEPROMPT;
  if (flags & wxHIDE_READONLY)
    msw_flags |= OFN_HIDEREADONLY;
  of.Flags = msw_flags;

  Bool success;
  if (flags & wxSAVE)
    success = GetSaveFileName(&of);
  else
    success = GetOpenFileName(&of);

  DWORD error = CommDlgExtendedError();
  if (success)
    return copystring(file_buffer);
  else
    return NULL;
}

#endif

#ifdef wx_motif
char *wxFileSelectorAnswer = NULL;
Bool wxFileSelectorReturned = FALSE;

void wxFileSelCancel(Widget fs, XtPointer client_data, XmFileSelectionBoxCallbackStruct *cbs)
{
  wxFileSelectorAnswer = NULL;
  wxFileSelectorReturned = TRUE;
}

void wxFileSelOk(Widget fs, XtPointer client_data, XmFileSelectionBoxCallbackStruct *cbs)
{
  char *filename;
  if (!XmStringGetLtoR(cbs->value, XmSTRING_DEFAULT_CHARSET, &filename))
  {
    wxFileSelectorAnswer = NULL;
    wxFileSelectorReturned = TRUE;
  }
  else
  {
    wxFileSelectorAnswer = copystring(filename);
    XtFree(filename);
    wxFileSelectorReturned = TRUE;
  }
}

char *wxMotifFileSelector(char *message,
                     char *default_path, char *default_filename, 
                     char *default_extension, char *wildcard, int flags,
                     wxFrame *parent, int x, int y)
{
  Widget parentWidget;
  if (parent)
    parentWidget = parent->frameShell;
  else
    parentWidget = wxTheApp->wx_frame->frameShell;

  Widget fileSel = XmCreateFileSelectionDialog(parentWidget, "file_selector", NULL, 0);
  XtUnmanageChild(XmFileSelectionBoxGetChild(fileSel, XmDIALOG_HELP_BUTTON));

  Widget shell = XtParent(fileSel);

  if (message)
    XtVaSetValues(shell, XmNtitle, message, NULL);

  char *entirePath = NULL;

  if (default_path && default_filename)
  {
    sprintf(wxBuffer, "%s/%s", default_path, default_filename);
    entirePath = copystring(wxBuffer);
  }
  else if (default_path && !default_filename)
  {
    sprintf(wxBuffer, "%s/", default_path);
    entirePath = copystring(wxBuffer);
  }
  else if ((!default_path) && default_filename)
  {
    sprintf(wxBuffer, "%s", default_filename);
    entirePath = copystring(wxBuffer);
  }

  if (entirePath)
  {
    Widget selectionWidget = XmFileSelectionBoxGetChild(fileSel, XmDIALOG_TEXT);
    XmTextSetString(selectionWidget, entirePath);
    delete entirePath;
  }

  if (wildcard)
  {
    Widget filterWidget = XmFileSelectionBoxGetChild(fileSel, XmDIALOG_FILTER_TEXT);
    XmTextSetString(filterWidget, wildcard);
  }

  XtAddCallback(fileSel, XmNcancelCallback, wxFileSelCancel, NULL);
  XtAddCallback(fileSel, XmNokCallback, wxFileSelOk, NULL);
  XtManageChild(fileSel);

  if (wxFileSelectorAnswer)
    delete wxFileSelectorAnswer;

  wxFileSelectorAnswer = NULL;
  wxFileSelectorReturned = FALSE;

  XtAddGrab(XtParent(fileSel), TRUE, FALSE);
  XEvent event;
  while (!wxFileSelectorReturned)
  {
    XtAppNextEvent(wxTheApp->appContext, &event);
    XtDispatchEvent(&event);
  }
  XtRemoveGrab(XtParent(fileSel));

  XtDestroyWidget(fileSel);

  return wxFileSelectorAnswer;
}
#endif
