/*
 * File:     wx_help.cc
 * Purpose:  API for invoking wxHelp
 *
 *                       wxWindows 1.40
 * Copyright (c) 1993 Artificial Intelligence Applications Institute,
 *                   The University of Edinburgh
 *
 *                     Author: Julian Smart
 *                       Date: 18-4-93
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose is hereby granted without fee, provided
 * that the above copyright notice, author statement and this permission
 * notice appear in all copies of this software and related documentation.
 *
 * THE SOFTWARE IS PROVIDED "AS-IS" AND WITHOUT WARRANTY OF ANY KIND, EXPRESS,
 * IMPLIED OR OTHERWISE, INCLUDING WITHOUT LIMITATION, ANY WARRANTY OF
 * MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * IN NO EVENT SHALL THE ARTIFICIAL INTELLIGENCE APPLICATIONS INSTITUTE OR THE
 * UNIVERSITY OF EDINBURGH BE LIABLE FOR ANY SPECIAL, INCIDENTAL, INDIRECT OR
 * CONSEQUENTIAL DAMAGES OF ANY KIND, OR ANY DAMAGES WHATSOEVER RESULTING FROM
 * LOSS OF USE, DATA OR PROFITS, WHETHER OR NOT ADVISED OF THE POSSIBILITY OF
 * DAMAGE, AND ON ANY THEORY OF LIABILITY, ARISING OUT OF OR IN CONNECTION WITH
 * THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */

#include <windows.h>
#include <wx.h>
#include <time.h>

#ifdef wx_x
#include <netdb.h>
#endif

// Timeout in seconds
#define WX_HELP_TIMEOUT 30

#include "wx_help.h"

Bool wxHelpInstance::Initialize(char *filename, int server)
{
#ifdef wx_x
  char host_buf[400];
  if (-1 != gethostname(host_buf, sizeof(host_buf)))
    helpHost = copystring(host_buf);
  else helpHost = NULL;
#endif
#ifdef wx_msw
  helpHost = copystring("dummy");
#endif

  helpFile = copystring(filename);
  helpServer = server;
  wxIPCInitialize();
  return TRUE;
}

Bool wxHelpInstance::LoadFile(char *file)
{
  if (!file)
    file = helpFile;

  if (!helpRunning)
  {
    if (!Run())
      return FALSE;
  }
  char buf[200];
  sprintf(buf, "f %s", file);
  if (helpConnection)
    return helpConnection->Execute(buf);
  else return FALSE;
}

Bool wxHelpInstance::DisplayContents(void)
{
  if (!helpRunning)
  {
    if (!Run())
      return FALSE;
  }
  if (helpConnection)
    return helpConnection->Execute("s -1");
  else return FALSE;
}

Bool wxHelpInstance::DisplaySection(int section)
{
  if (!helpRunning)
  {
    if (!Run())
      return FALSE;
  }
  char buf[200];
  sprintf(buf, "s %d", section);
  if (helpConnection)
    return helpConnection->Execute(buf);
  else return FALSE;
}

Bool wxHelpInstance::DisplayBlock(long block)
{
  if (!helpRunning)
  {
    if (!Run())
      return FALSE;
  }
  char buf[200];
  sprintf(buf, "b %l", block);
  if (helpConnection)
    return helpConnection->Execute(buf);
  else return FALSE;
}

Bool wxHelpInstance::KeywordSearch(char *k)
{
  if (!helpRunning)
  {
    if (!Run())
      return FALSE;
  }
  char buf[500];
  sprintf(buf, "k %s", k);
  if (helpConnection)
    return helpConnection->Execute(buf);
  else return FALSE;
}

Bool wxHelpInstance::Quit(void)
{
  if (helpConnection)
    return helpConnection->Disconnect(); // Calls OnQuit via OnDisconnect
  else return FALSE;
}

void wxHelpInstance::OnQuit(void)
{
}

Bool wxHelpInstance::Run(void)
{
#ifdef wx_x
  if (!helpFile || !helpHost || helpRunning)
    return FALSE;
#endif
#ifdef wx_msw
  if (!helpFile || helpRunning)
    return FALSE;
#endif

  char buf[500];
  char buf1[100];
  time_t start_time;
  time_t current_time;

  (void)time(&current_time);

  // Invent a server name that's likely to be unique but different from
  // last time
#ifdef wx_x
  if (helpServer == -1)
    helpServer = 4000 + (current_time % 4000);
#endif
#ifdef wx_msw // Only one instance of wxHelp at a time
  helpServer = 4000;
#endif

  sprintf(buf1, "%d", helpServer);

#ifdef wx_x
  sprintf(buf, "wxhelp -server %d &", helpServer);
#endif
#ifdef wx_msw
  // Only one instance of wxHelp under Windows.
  // See if there's already an instance of wxHelp
  if (helpConnection = (wxHelpConnection *)MakeConnection(helpHost, buf1, "WXHELP"))
  {
    helpRunning = TRUE;
    return TRUE;
  }

  sprintf(buf, "wxhelp -server %d", helpServer);
#endif
  wxExecute(buf);

  (void)time(&start_time);
  (void)time(&current_time);

  // Give it some time to respond
  while (!helpConnection && ((current_time - start_time) < WX_HELP_TIMEOUT))
  {
    wxSleep(1);
    helpConnection = (wxHelpConnection *)MakeConnection(helpHost, buf1, "WXHELP");
    (void)time(&current_time);
  }

  if (helpConnection)
  {
    helpRunning = TRUE;
    return TRUE;
  }
  else
  {
    sprintf(buf, "Connection to wxHelp timed out in %d seconds", WX_HELP_TIMEOUT);
    (void)wxMessageBox(buf);
    return FALSE;
  }
}

Bool wxHelpConnection::OnDisconnect(void)
{
  helpInstance->OnQuit();
  helpInstance->helpRunning = FALSE;
  helpInstance->helpConnection = NULL;
  helpInstance->helpServer = -1;
  delete this;
  return TRUE;
}

