/*
 * File:     wx_win.cc
 * Purpose:  wxWindow class implementation
 *
 *                       wxWindows 1.40
 * Copyright (c) 1993 Artificial Intelligence Applications Institute,
 *                   The University of Edinburgh
 *
 *                     Author: Julian Smart
 *                       Date: 18-4-93
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose is hereby granted without fee, provided
 * that the above copyright notice, author statement and this permission
 * notice appear in all copies of this software and related documentation.
 *
 * THE SOFTWARE IS PROVIDED "AS-IS" AND WITHOUT WARRANTY OF ANY KIND, EXPRESS,
 * IMPLIED OR OTHERWISE, INCLUDING WITHOUT LIMITATION, ANY WARRANTY OF
 * MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * IN NO EVENT SHALL THE ARTIFICIAL INTELLIGENCE APPLICATIONS INSTITUTE OR THE
 * UNIVERSITY OF EDINBURGH BE LIABLE FOR ANY SPECIAL, INCIDENTAL, INDIRECT OR
 * CONSEQUENTIAL DAMAGES OF ANY KIND, OR ANY DAMAGES WHATSOEVER RESULTING FROM
 * LOSS OF USE, DATA OR PROFITS, WHETHER OR NOT ADVISED OF THE POSSIBILITY OF
 * DAMAGE, AND ON ANY THEORY OF LIABILITY, ARISING OUT OF OR IN CONNECTION WITH
 * THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */

#include <windows.h>
#include <iostream.h>

#include "common.h"
#include "wx_win.h"
#include "wx_main.h"
#include "wx_utils.h"
#include "wx_privt.h"

#ifdef wx_xview
// Intercept focus events
extern "C" int win_set_kbd_focus(Xv_Window, XID);
#endif

#ifdef wx_msw
// Find an item given the MS Windows id
wxWindow *wxWindow::FindItem(int id)
{
  if (!children)
    return NULL;
  wxNode *current = children->First();
  wxWindow *the_item = NULL;

  while (current)
  {
    wxWindow *item = (wxWindow *)current->Data();
    if (item->windows_id == id)
    {
      the_item = item;
      break;
    }
    else
      current = current->Next();
  }
  return the_item;
}

// Default command handler
BOOL wxWindow::Command(UINT param)
{
  return FALSE;
}

void wxWindow::PreDelete(HDC dc)
{
}
#endif

// Constructor
wxWindow::wxWindow(void)
{
  wx_client_data = NULL;
  window_parent = NULL;
  font = NULL;
  handle = NULL;
  callback = 0;
  wx_cursor = wxSTANDARD_CURSOR;
  children = new wxList;
#ifdef wx_motif
  wxType = 0;
#endif
}

// Destructor
wxWindow::~wxWindow(void)
{
  if (window_parent)
    window_parent->RemoveChild(this);

#ifdef wx_motif
  if (handle)
  {
    DestroyChildren();

    wxWidgetHashTable->Delete((long)handle);
    Widget w = (Widget)handle;
    XtDestroyWidget(w);

    PostDestroyChildren();

  }
#endif
#ifdef wx_xview
  if (handle)
  {
    // Reset client data
    xv_set((Xv_opaque)handle, WIN_CLIENT_DATA, NULL, NULL);
    DestroyChildren();
    xv_destroy_safe((Xv_opaque)handle);
  }
#endif
#ifdef wx_msw
  DestroyChildren();
  switch (wxWinType)
  {
    case wxTYPE_XWND:
    {
      if (handle)
      {
        wxWnd *wnd = (wxWnd *)handle;
        HDC dc = GetDC(wnd->handle);
        PreDelete(dc);
        ReleaseDC(wnd->handle, dc);

        wnd->DestroyWindow();
        delete wnd;
        handle = NULL;
      }
      break;
    }
    case wxTYPE_HWND:
    {
      DestroyWindow((HWND)ms_handle);
      handle = NULL;
      break;
    }
    default:
      break;
    }
#endif

  delete children;
  children = NULL;
}

char *wxWindow::GetHandle(void)
{
  return handle;
}

void wxWindow::SetFocus(void)
{
#ifdef wx_motif
  XmProcessTraversal((Widget)handle, XmTRAVERSE_CURRENT);
  XmProcessTraversal((Widget)handle, XmTRAVERSE_CURRENT);
#endif
#ifdef wx_xview
  Xv_opaque win = (Xv_opaque)handle;
  win_set_kbd_focus(win, xv_get(win, XV_XID));
#endif
#ifdef wx_msw
  switch (wxWinType)
  {
    case wxTYPE_XWND:
    {
      if (handle != NULL)
      {
        wxWnd *wnd = (wxWnd *)handle;
        ::SetFocus(wnd->handle);
      }
      break;
    }
    case wxTYPE_HWND:
    {
      if (ms_handle != NULL)
        ::SetFocus((HWND)ms_handle);
      break;
    }
    default:
      break;
  }
#endif
}

void wxWindow::OnPaint(void)
{
}

void wxWindow::OnSize(int width, int height)
{
}

void wxWindow::OnEvent(wxEvent& event)
{
}

void wxWindow::OnChar(int ch)
{
}

Bool wxWindow::OnClose(void)
{
  return FALSE;
}

void wxWindow::OnActivate(Bool active)
{
}

// Get total size
void wxWindow::GetSize(int *x, int *y)
{
#ifdef wx_motif
  Widget widget = (Widget)handle;
  Dimension xx, yy;
  XtVaGetValues(widget, XmNwidth, &xx, XmNheight, &yy, NULL);
  *x = xx; *y = yy;
#endif
#ifdef wx_xview
  Xv_opaque x_win = (Xv_opaque)handle;

  *x = (int)xv_get(x_win, XV_WIDTH);
  *y = (int)xv_get(x_win, XV_HEIGHT);
#endif
#ifdef wx_msw
  wxWnd *wnd = (wxWnd *)handle;
  RECT rect;
  GetWindowRect(wnd->handle, &rect);
  *x = rect.right - rect.left;
  *y = rect.bottom - rect.top;
#endif
}

void wxWindow::GetPosition(int *x, int *y)
{
#ifdef wx_motif
  Widget widget = (Widget)handle;
  Dimension xx, yy;
  XtVaGetValues(widget, XmNx, &xx, XmNy, &yy, NULL);
  *x = xx; *y = yy;
#endif
#ifdef wx_xview
  Xv_opaque x_win = (Xv_opaque)handle;

  *x = (int)xv_get(x_win, XV_X);
  *y = (int)xv_get(x_win, XV_Y);
#endif
#ifdef wx_msw
  wxWnd *wnd = (wxWnd *)handle;
  wxWindow *parent = GetParent();

  RECT rect;
  GetWindowRect(wnd->handle, &rect);

  // Since we now have the absolute screen coords,
  // if there's a parent we must subtract its top left corner
  POINT point;
  point.x = rect.left;
  point.y = rect.top;
  if (parent)
  {
    wxWnd *cparent = (wxWnd *)(parent->handle);
    ScreenToClient(cparent->handle, &point);
  }
  *x = point.x;
  *y = point.y;

#endif
}

wxCursor *wxWindow::SetCursor(wxCursor *cursor)
{
  wxCursor *old_cursor = wx_cursor;
  wx_cursor = cursor;
#ifdef wx_motif
  if (cursor && cursor->x_cursor)
  {
    Widget w = (Widget)handle;
    Window win = XtWindow(w);
    Display *dpy = XtDisplay(wxTheApp->topLevel);
    XDefineCursor(dpy, win, cursor->x_cursor);
  }
#endif
#ifdef wx_xview
  Xv_opaque x_win = (Xv_opaque)handle;
  Xv_Window win = xv_get(x_win, CANVAS_NTH_PAINT_WINDOW, 0);
  if (cursor && cursor->x_cursor)
  {
    if (cursor->use_raw_x_cursor)
    {
      Xv_Screen screen = xv_get(xview_server, SERVER_NTH_SCREEN, 0);
      Xv_Window root_window = xv_get(screen, XV_ROOT);
      Display *dpy = (Display *)xv_get(root_window, XV_DISPLAY);
      Window win2 = xv_get(win, XV_XID);

      XDefineCursor(dpy, win2, cursor->x_cursor);
    }
    else
      xv_set(win, WIN_CURSOR, cursor->x_cursor, NULL);
  }

#endif
#ifdef wx_msw
  if (wx_cursor)
    ::SetCursor(wx_cursor->ms_cursor);
#endif
  return old_cursor;
}

// Get size *available for subwindows* i.e. excluding menu bar etc.
// For XView, this is the same as GetSize
void wxWindow::GetClientSize(int *x, int *y)
{
#ifdef wx_motif
  Widget widget = (Widget)handle;
  Dimension xx, yy;
  XtVaGetValues(widget, XmNwidth, &xx, XmNheight, &yy, NULL);
  *x = xx; *y = yy;
#endif
#ifdef wx_xview
  Xv_opaque x_win = (Xv_opaque)handle;

  *x = (int)xv_get(x_win, XV_WIDTH);
  *y = (int)xv_get(x_win, XV_HEIGHT);
#endif
#ifdef wx_msw
  wxWnd *wnd = (wxWnd *)handle;
  RECT rect;
  GetClientRect(wnd->handle, &rect);
  *x = rect.right;
  *y = rect.bottom;
#endif
}

void wxWindow::SetSize(int x, int y, int width, int height)
{
#ifdef wx_motif
  Widget widget = (Widget)handle;

  if (x > -1)
    XtVaSetValues(widget, XmNx, x, NULL);
  if (y > -1)
    XtVaSetValues(widget, XmNy, y, NULL);
  if (width > -1)
    XtVaSetValues(widget, XmNwidth, width, NULL);
  if (height > -1)
    XtVaSetValues(widget, XmNheight, height, NULL);
  OnSize(width, height);
#endif
#ifdef wx_xview
  Xv_opaque x_win = (Xv_opaque)handle;

  (void)xv_set(x_win, XV_X, x, XV_Y, y, XV_WIDTH, width, XV_HEIGHT, height, NULL);
  OnSize(width, height);
#endif
#ifdef wx_msw
  int currentX, currentY;
  GetPosition(&currentX, &currentY);
  if (x == -1)
    x = currentX;
  if (y == -1)
    y = currentY;

  wxWnd *wnd = (wxWnd *)handle;
  if (wnd)
    MoveWindow(wnd->handle, x, y, width, height, TRUE);
  OnSize(width, height);
#endif
}

void wxWindow::SetClientSize(int width, int height)
{
#ifdef wx_motif
  Widget widget = (Widget)handle;

    XtVaSetValues(widget, XmNwidth, width, NULL);
  if (height > -1)
    XtVaSetValues(widget, XmNheight, height, NULL);
  OnSize(width, height);
#endif
#ifdef wx_xview
  Xv_opaque x_win = (Xv_opaque)handle;

  (void)xv_set(x_win, XV_WIDTH, width, XV_HEIGHT, height, NULL);
#endif
#ifdef wx_msw
  wxWindow *parent = GetParent();
  wxWnd *wnd = (wxWnd *)handle;
  RECT rect;
  GetClientRect(wnd->handle, &rect);

  RECT rect2;
  GetWindowRect(wnd->handle, &rect2);

  // Find the difference between the entire window (title bar and all)
  // and the client area; add this to the new client size to move the
  // window
  int actual_width = rect2.right - rect2.left - rect.right + width;
  int actual_height = rect2.bottom - rect2.top - rect.bottom + height;

  // If there's a parent, must subtract the parent's top left corner
  // since MoveWindow moves relative to the parent

  POINT point;
  point.x = rect2.left;
  point.y = rect2.top;
  if (parent)
  {
    wxWnd *cparent = (wxWnd *)(parent->handle);
    ScreenToClient(cparent->handle, &point);
  }

  MoveWindow(wnd->handle, point.x, point.y, actual_width, actual_height, TRUE);
  OnSize(actual_width, actual_height);
#endif
}

// General callback setting
void wxWindow::Callback(wxFunction Function)
{
  if (Function)
    callback = Function;
}

// Client data handling (any window, item etc.)
void wxWindow::SetClientData(char *data)
{
  wx_client_data = data;
}

char *wxWindow::GetClientData(void)
{
  return wx_client_data;
}

void wxWindow::Show(Bool show)
{
#ifdef wx_motif
#endif
#ifdef wx_xview
  Xv_opaque window = (Xv_opaque)handle;
  xv_set(window, XV_SHOW, show, NULL);
#endif
#ifdef wx_msw
  wxWnd *wnd = (wxWnd *)handle;
  int cshow;
  if (show)
    cshow = SW_SHOW;
  else
    cshow = SW_HIDE;
  ShowWindow(wnd->handle, cshow);
  if (show)
    BringWindowToTop(wnd->handle);
#endif
}

wxWindow *wxWindow::GetParent(void)
{
  return window_parent;
}

wxWindow *wxWindow::GetGrandParent(void)
{
  if (window_parent)
    return window_parent->window_parent;
  else return NULL;
}

void wxWindow::AddChild(wxObject *child)
{
  children->Append(child);
}

void wxWindow::RemoveChild(wxObject *child)
{
  if (children)
    children->DeleteObject(child);
}

void wxWindow::DestroyChildren(void)
{
  if (children)
  {
    wxNode *node = children->First();
    while (node)
    {
      wxWindow *child = (wxWindow *)node->Data();
      child->DestroyChildren();
      delete child;
      node = children->First();
    }
  }
}

// Default menu command handler
void wxWindow::OnMenuCommand(int Id)
{
}

// Default handler for OnMenuSelect
void wxWindow::OnMenuSelect(int Id)
{
}

void wxWindow::SetTitle(char *title)
{
}

int wxWindow::GetTextFamily(void)
{
#ifdef wx_motif
  return 0;
#endif
#ifdef wx_xview
  Xv_Font the_font;

  if (!(font && font->x_font))
  {
    Xv_opaque thing = (Xv_opaque)handle;
    the_font = (Xv_Font)xv_get(thing, XV_FONT);
  }
  else the_font = font->x_font;


  char *family = (char *)xv_get(the_font, FONT_FAMILY);
  int wx_family;

  if (strcmp(family, FONT_FAMILY_LUCIDA) == 0)
    wx_family = wxDECORATIVE;
  else if (strcmp(family, FONT_FAMILY_ROMAN) == 0)
    wx_family = wxROMAN;
  else if (strcmp(family, FONT_FAMILY_HELVETICA) == 0)
    wx_family = wxSWISS;
 else if (strcmp(family, FONT_FAMILY_COUR) == 0)
    wx_family = wxMODERN;
  else
    wx_family = wxDEFAULT;

  return wx_family;
#endif
#ifdef wx_msw
  TEXTMETRIC lpTextMetric;
  wxWnd *wnd = (wxWnd *)handle;
  HDC dc = GetDC(wnd->handle);

  GetTextMetrics(dc, &lpTextMetric);
  ReleaseDC(wnd->handle, dc);
  int family;
  switch (lpTextMetric.tmPitchAndFamily & 0xF0)
  {
    case FF_DECORATIVE:
      family = wxDECORATIVE;
      break;
    case FF_MODERN:
      family = wxMODERN;
      break;
    case FF_ROMAN:
      family = wxROMAN;
      break;
    case FF_SWISS:
      family = wxSWISS;
      break;
    default:
    case FF_DONTCARE:
      family = wxDEFAULT;
  }
  return family;
#endif
}

int wxWindow::GetTextStyle(void)
{
#ifdef wx_motif
  return 0;
#endif
#ifdef wx_xview
  Xv_Font the_font;

  if (!(font && font->x_font))
  {
    Xv_opaque thing = (Xv_opaque)handle;
    the_font = (Xv_Font)xv_get(thing, XV_FONT);
  }
  else the_font = font->x_font;

  char *style = (char *)xv_get(the_font, FONT_STYLE);
  int wx_style;

  if (strcmp(style, FONT_STYLE_NORMAL) == 0)
    wx_style = wxNORMAL;
  else if (strcmp(style, FONT_STYLE_BOLD) == 0)
    wx_style = wxNORMAL;
  else if (strcmp(style, FONT_STYLE_ITALIC) == 0)
    wx_style = wxITALIC;
  else if (strcmp(style, FONT_STYLE_BOLD_ITALIC) == 0)
    wx_style = wxITALIC;
  else if (strcmp(style, FONT_STYLE_OBLIQUE) == 0)
    wx_style = wxSLANT;
  else if (strcmp(style, FONT_STYLE_BOLD_OBLIQUE) == 0)
    wx_style = wxSLANT;
  else wx_style = wxDEFAULT;

  return wx_style;
#endif
#ifdef wx_msw
  TEXTMETRIC lpTextMetric;
  wxWnd *wnd = (wxWnd *)handle;
  HDC dc = GetDC(wnd->handle);

  GetTextMetrics(dc, &lpTextMetric);
  ReleaseDC(wnd->handle, dc);
  int wx_style;
  if (lpTextMetric.tmItalic)
    wx_style = wxITALIC;
  else wx_style = wxNORMAL;

  return wx_style;
#endif
}

int wxWindow::GetTextWeight(void)
{
#ifdef wx_motif
  return 0;
#endif
#ifdef wx_xview
  Xv_Font the_font;

  if (!(font && font->x_font))
  {
    Xv_opaque thing = (Xv_opaque)handle;
    the_font = (Xv_Font)xv_get(thing, XV_FONT);
  }
  else the_font = font->x_font;

  char *style = (char *)xv_get(the_font, FONT_STYLE);
  int wx_weight;

  if (strcmp(style, FONT_STYLE_NORMAL) == 0)
    wx_weight = wxNORMAL;
  else if (strcmp(style, FONT_STYLE_BOLD) == 0)
    wx_weight = wxBOLD;
  else if (strcmp(style, FONT_STYLE_ITALIC) == 0)
    wx_weight = wxNORMAL;
  else if (strcmp(style, FONT_STYLE_BOLD_ITALIC) == 0)
    wx_weight = wxBOLD;
  else if (strcmp(style, FONT_STYLE_OBLIQUE) == 0)
    wx_weight = wxNORMAL;
  else if (strcmp(style, FONT_STYLE_BOLD_OBLIQUE) == 0)
    wx_weight = wxBOLD;
  else wx_weight = wxNORMAL;

  return wx_weight;
#endif
#ifdef wx_msw
  TEXTMETRIC lpTextMetric;
  wxWnd *wnd = (wxWnd *)handle;
  HDC dc = GetDC(wnd->handle);

  GetTextMetrics(dc, &lpTextMetric);
  ReleaseDC(wnd->handle, dc);
  int wx_weight;
  int cweight = lpTextMetric.tmWeight;
  if (cweight > 0 && cweight < 350)
    wx_weight = wxLIGHT;  // light
  else if (cweight >= 350 && cweight < 500)
    wx_weight = wxNORMAL; // normal
  else if (cweight >= 500)
    wx_weight = wxBOLD; // bold
  else wx_weight = wxNORMAL;

  return wx_weight;
#endif
}

float wxWindow::GetTextHeight(void)
{
#ifdef wx_motif
  return 0.0;
#endif
#ifdef wx_xview
  Xv_Font the_font;

  if (!(font && font->x_font))
  {
    Xv_opaque thing = (Xv_opaque)handle;
    the_font = (Xv_Font)xv_get(thing, XV_FONT);
  }
  else the_font = font->x_font;

  return (float)xv_get(the_font, FONT_DEFAULT_CHAR_HEIGHT);
#endif
#ifdef wx_msw
  TEXTMETRIC lpTextMetric;
  wxWnd *wnd = (wxWnd *)handle;
  HDC dc = GetDC(wnd->handle);

  GetTextMetrics(dc, &lpTextMetric);
  ReleaseDC(wnd->handle, dc);

  return (float)lpTextMetric.tmHeight;
#endif
}

float wxWindow::GetTextWidth(void)
{
#ifdef wx_motif
  return 0.0;
#endif
#ifdef wx_xview
  Xv_Font the_font;

  if (!(font && font->x_font))
  {
    Xv_opaque thing = (Xv_opaque)handle;
    the_font = (Xv_Font)xv_get(thing, XV_FONT);
  }
  else the_font = font->x_font;

  return (float)xv_get(the_font, FONT_DEFAULT_CHAR_WIDTH);
#endif
#ifdef wx_msw
  TEXTMETRIC lpTextMetric;
  wxWnd *wnd = (wxWnd *)handle;
  HDC dc = GetDC(wnd->handle);

  GetTextMetrics(dc, &lpTextMetric);
  ReleaseDC(wnd->handle, dc);

  return (float)lpTextMetric.tmAveCharWidth;
#endif
}

void wxWindow::GetTextExtent(char *string, float *x, float *y)
{
#ifdef wx_motif
  *x = 0; *y = 0;
#endif
#ifdef wx_xview
  Xv_Font the_font;

  if (!(font && font->x_font))
  {
    Xv_opaque thing = (Xv_opaque)handle;
    the_font = (Xv_Font)xv_get(thing, XV_FONT);
  }
  else the_font = font->x_font;

  Font_string_dims dims;

  (void)xv_get(the_font, FONT_STRING_DIMS, string, &dims);
  *x = (float)dims.width;
  *y = (float)dims.height;
#endif
#ifdef wx_msw
  wxWnd *wnd = (wxWnd *)handle;
  HDC dc = GetDC(wnd->handle);

  SIZE sizeRect;
  GetTextExtentPoint(dc, string, strlen(string), &sizeRect);

  ReleaseDC(wnd->handle, dc);

  *x = (float)sizeRect.cx;
  *y = (float)sizeRect.cy;

#endif
}

// Return the number of chars that this pixel dimension represents
int wxWindow::XChar(int pixels)
{
#ifdef wx_motif
  return 0;
#endif
#ifdef wx_xview
  Xv_Font the_font;

  if (!(font && font->x_font))
  {
    Xv_opaque thing = (Xv_opaque)handle;
    the_font = (Xv_Font)xv_get(thing, XV_FONT);
  }
  else the_font = font->x_font;

  return (int)(pixels/((int)xv_get(the_font, FONT_DEFAULT_CHAR_WIDTH)));
#endif
#ifdef wx_msw
  wxWnd *wnd = (wxWnd *)handle;
  int x, y;
  wxGetCharSize(wnd->handle, &x, &y);
  return (int)(pixels/x);
#endif
}

int wxWindow::YChar(int pixels)
{
#ifdef wx_motif
  return 0;
#endif
#ifdef wx_xview
  Xv_Font the_font;

  if (!(font && font->x_font))
  {
    Xv_opaque thing = (Xv_opaque)handle;
    the_font = (Xv_Font)xv_get(thing, XV_FONT);
  }
  else the_font = font->x_font;

  return (int)(pixels/((int)xv_get(the_font, FONT_DEFAULT_CHAR_HEIGHT)));
#endif
#ifdef wx_msw
  wxWnd *wnd = (wxWnd *)handle;
  int x, y;
  wxGetCharSize(wnd->handle, &x, &y);
  return (int)(pixels/y);
#endif
}

// Fit window to contents
void wxWindow::Fit(void)
{
}

// Centre on panel/screen
void wxWindow::Centre(int direction)
{
}

#ifdef wx_motif
void wxWindow::PreResize(void)
{
}
#endif

#ifdef wx_motif
// All widgets should have this as their resize proc.
// OnSize sent to wxWindow via client data.
void wxWidgetResizeProc(Widget w, XConfigureEvent *event, String args[], int *num_args)
{
//  cout << "Motif widget resize: " << (long)w << "\n";
  wxWindow *win = (wxWindow *)wxWidgetHashTable->Get((long)w);
  if (!win)
    return;

  int width, height;
  win->GetSize(&width, &height);
  win->PreResize();
  win->OnSize(width, height);
}
#endif

#ifdef wx_msw
// Hook for new window just as it's being created,
// when the window isn't yet associated with the handle
wxWnd *wxWndHook = NULL;

// Main Windows 3 window proc
LONG FAR PASCAL _export
  wxWndProc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam)
{
  wxWnd *wnd = (wxWnd *)GetWindowLong(hWnd, 0);
  if (!wnd)
  {
    if (wxWndHook)
    {
      wnd = wxWndHook;
      wnd->handle = hWnd;
    }
    else wnd = wxFindWinFromHandle(hWnd);
  }

  if (wnd)
  {
    wnd->last_msg = message;
    wnd->last_wparam = wParam;
    wnd->last_lparam = lParam;

    if (message == WM_SETFONT)
      return 0;
    else if (message == WM_INITDIALOG)
      return TRUE;
  }

  switch (message)
  {
        case WM_ACTIVATE:
        {
            WORD state = LOWORD(wParam);
#ifdef WIN32
            WORD minimized = HIWORD(wParam);
            HWND hwnd = (HWND)lParam;
#else
            WORD minimized = LOWORD(lParam);
            HWND hwnd = (HWND)HIWORD(lParam);
#endif
            if (!wnd->OnActivate(state, minimized, hwnd))
              return wnd->DefWindowProc(message, wParam, lParam );
            break;
        }
        case WM_SETFOCUS:
        {
            HWND hwnd = (HWND)wParam;
            if (!wnd->OnSetFocus(hwnd))
              return wnd->DefWindowProc(message, wParam, lParam );
            break;
        }
        case WM_KILLFOCUS:
        {
            HWND hwnd = (HWND)lParam;
            if (!wnd->OnKillFocus(hwnd))
              return wnd->DefWindowProc(message, wParam, lParam );
            break;
        }
	case WM_CREATE:
            if (wnd)
              wnd->OnCreate((LPCREATESTRUCT)lParam);
            return 0;
	    break;

	case WM_PAINT:
            return (long)wnd->OnPaint();
            break;

        case WM_SIZE:
        {
            if (wnd)
            {
              int width = LOWORD(lParam);
              int height = HIWORD(lParam);
              wnd->OnSize(width, height, wParam);
            }
            else return DefWindowProc( hWnd, message, wParam, lParam );
            break;
        }

        case WM_RBUTTONDOWN:
        {
            int x = LOWORD(lParam);
            int y = HIWORD(lParam);
            wnd->OnRButtonDown(x, y, wParam);
            break;
        }
        case WM_RBUTTONUP:
        {
            int x = LOWORD(lParam);
            int y = HIWORD(lParam);
            wnd->OnRButtonUp(x, y, wParam);
            break;
        }
        case WM_LBUTTONDOWN:
        {
            int x = LOWORD(lParam);
            int y = HIWORD(lParam);
            wnd->OnLButtonDown(x, y, wParam);
            break;
        }
        case WM_LBUTTONUP:
        {
            int x = LOWORD(lParam);
            int y = HIWORD(lParam);
            wnd->OnLButtonUp(x, y, wParam);
            break;
        }
        case WM_MOUSEMOVE:
        {
            int x = LOWORD(lParam);
            int y = HIWORD(lParam);
            wnd->OnMouseMove(x, y, wParam);
            break;
        }
        case WM_DESTROY:
            if (wnd)
            {
              if (wnd->OnDestroy())
                return 0;
              else return wnd->DefWindowProc(message, wParam, lParam );
            }
            else return DefWindowProc( hWnd, message, wParam, lParam );
            break;
/*
        case WM_SYSCOMMAND:
            break;
*/
        case WM_COMMAND:
	{
            WORD id = LOWORD(wParam);
            HWND hwnd = (HWND)(UINT)lParam;
#ifdef WIN32
            WORD cmd = HIWORD(wParam);
#else
            WORD cmd = HIWORD(lParam);
#endif
            if (!wnd->OnCommand(id, cmd, hwnd))
              return wnd->DefWindowProc(message, wParam, lParam );
            break;
	 }
        case WM_MENUSELECT:
        {
            WORD id = LOWORD(wParam);
#ifdef WIN32
            WORD flags = HIWORD(wParam);
            HMENU sysmenu = (HMENU)lParam;
#else
            WORD flags = LOWORD(lParam);
            HMENU sysmenu = (HMENU)HIWORD(lParam);
#endif
            wnd->OnMenuSelect(wParam, flags, sysmenu);
            break;
        }
        case WM_KEYDOWN:
            break;
        case WM_CHAR:
            wnd->OnChar(wParam);
            break;
        case WM_HSCROLL:
        {
            WORD code = LOWORD(wParam);
#ifdef WIN32
            WORD pos = HIWORD(wParam);
            HWND control = (HWND)lParam;
#else
            WORD pos = LOWORD(lParam);
            HWND control = (HWND)HIWORD(lParam);
#endif
            wnd->OnHScroll(code, pos, control);
            break;
        }
        case WM_VSCROLL:
        {
            WORD code = LOWORD(wParam);
#ifdef WIN32
            WORD pos = HIWORD(wParam);
            HWND control = (HWND)lParam;
#else
            WORD pos = LOWORD(lParam);
            HWND control = (HWND)HIWORD(lParam);
#endif
            wnd->OnVScroll(code, pos, control);
            break;
        }
#ifdef WIN32
        case WM_CTLCOLORBTN:
	{
          int nCtlColor = CTLCOLOR_BTN;
          HWND control = (HWND)LOWORD(lParam);
          HDC pDC = (HDC)HIWORD(wParam);
          return (DWORD)wnd->OnCtlColor(pDC, control, nCtlColor);
          break;
	}
        case WM_CTLCOLORDLG:
	{
          int nCtlColor = CTLCOLOR_DLG;
          HWND control = (HWND)LOWORD(lParam);
          HDC pDC = (HDC)HIWORD(wParam);
          return (DWORD)wnd->OnCtlColor(pDC, control, nCtlColor);
          break;
	}
        case WM_CTLCOLORLISTBOX:
	{
          int nCtlColor = CTLCOLOR_LISTBOX;
          HWND control = (HWND)LOWORD(lParam);
          HDC pDC = (HDC)HIWORD(wParam);
          return (DWORD)wnd->OnCtlColor(pDC, control, nCtlColor);
          break;
	}
        case WM_CTLCOLORMSGBOX:
	{
          int nCtlColor = CTLCOLOR_MSGBOX;
          HWND control = (HWND)LOWORD(lParam);
          HDC pDC = (HDC)HIWORD(wParam);
          return (DWORD)wnd->OnCtlColor(pDC, control, nCtlColor);
          break;
	}
        case WM_CTLCOLORSCROLLBAR:
	{
          int nCtlColor = CTLCOLOR_SCROLLBAR;
          HWND control = (HWND)LOWORD(lParam);
          HDC pDC = (HDC)HIWORD(wParam);
          return (DWORD)wnd->OnCtlColor(pDC, control, nCtlColor);
          break;
	}
        case WM_CTLCOLORSTATIC:
	{
          int nCtlColor = CTLCOLOR_STATIC;
          HWND control = (HWND)LOWORD(lParam);
          HDC pDC = (HDC)HIWORD(wParam);
          return (DWORD)wnd->OnCtlColor(pDC, control, nCtlColor);
          break;
	}
        case WM_CTLCOLOREDIT:
	{
          int nCtlColor = CTLCOLOR_EDIT;
          HWND control = (HWND)LOWORD(lParam);
          HDC pDC = (HDC)HIWORD(wParam);
          return (DWORD)wnd->OnCtlColor(pDC, control, nCtlColor);
          break;
	}
#else
        case WM_CTLCOLOR:
        {
          HWND control = (HWND)LOWORD(lParam);
          int nCtlColor = (int)HIWORD(lParam);
          HDC pDC = (HDC)wParam;
          return (DWORD)wnd->OnCtlColor(pDC, control, nCtlColor);
          break;
        }
#endif
        case WM_ERASEBKGND:
        {
            if (!wnd->OnEraseBkgnd((HDC)wParam))
              return wnd->DefWindowProc(message, wParam, lParam );
            else return 1;
            break;
        }
        case WM_MDIACTIVATE:
        {
#ifdef WIN32
            // Is this right???
            HWND hWndActivate = (HWND)LOWORD(wParam);
            HWND hWndDeactivate = (HWND)LOWORD(lParam);
            BOOL activate = (hWndActivate == hWnd);
            return wnd->OnMDIActivate(activate, hWndActivate, hWndDeactivate);
#else
            return wnd->OnMDIActivate((BOOL)wParam, (HWND)LOWORD(lParam),
                                               (HWND)HIWORD(lParam));
#endif
        }
        case WM_CLOSE:
        {
            if (wnd->OnClose())
              return 0L;
            else
              return 1L;
            break;
        }

        default:
            if (wnd)
              return wnd->DefWindowProc(message, wParam, lParam );
            else return DefWindowProc( hWnd, message, wParam, lParam );
    }
    return 0;
}

// Dialog window proc
LONG FAR PASCAL _export
  wxDlgProc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam)
{
  wxWnd *wnd = wxFindWinFromHandle(hWnd);

  if (!wnd && wxWndHook)
  {
    wnd = wxWndHook;
    wnd->handle = hWnd;
  }

  if (wnd)
  {
    wnd->last_msg = message;
    wnd->last_wparam = wParam;
    wnd->last_lparam = lParam;
  }

  if (message == WM_SETFONT)
    return 0;
  else if (message == WM_INITDIALOG)
    return TRUE;

  switch (message)
  {
        case WM_ACTIVATE:
        {
            WORD state = LOWORD(wParam);
#ifdef WIN32
            WORD minimized = HIWORD(wParam);
            HWND hwnd = (HWND)lParam;
#else
            WORD minimized = LOWORD(lParam);
            HWND hwnd = (HWND)HIWORD(lParam);
#endif
            if (!wnd->OnActivate(state, minimized, hwnd))
              return wnd->DefWindowProc(message, wParam, lParam );
            break;
	 }

        case WM_SETFOCUS:
        {
            HWND hwnd = (HWND)wParam;
            if (!wnd->OnSetFocus(hwnd))
              return wnd->DefWindowProc(message, wParam, lParam );
            break;
        }
        case WM_KILLFOCUS:
        {
            HWND hwnd = (HWND)lParam;
            if (!wnd->OnKillFocus(hwnd))
              return wnd->DefWindowProc(message, wParam, lParam );
            break;
        }

        case WM_CREATE:
            if (wnd)
              wnd->OnCreate((LPCREATESTRUCT)lParam);
            return 0;
	    break;

        case WM_SIZE:
        {
            if (wnd)
            {
              int width = LOWORD(lParam);
              int height = HIWORD(lParam);
              wnd->OnSize(width, height, wParam);
            }
            else return FALSE;
            break;
        }
/*
        case WM_DESTROY:
            if (wnd)
            {
              if (wnd->OnDestroy())
                return 0;
            }
            return FALSE;
            break;
*/
        case WM_COMMAND:
	{
            WORD id = LOWORD(wParam);
            HWND hwnd = (HWND)(UINT)lParam;
#ifdef WIN32
            WORD cmd = HIWORD(wParam);
#else
            WORD cmd = HIWORD(lParam);
#endif
            if (!wnd->OnCommand(id, cmd, hwnd))
              return wnd->DefWindowProc(message, wParam, lParam );
            break;
	}
        case WM_HSCROLL:
        {
            WORD code = LOWORD(wParam);
#ifdef WIN32
            WORD pos = HIWORD(wParam);
            HWND control = (HWND)lParam;
#else
            WORD pos = LOWORD(lParam);
            HWND control = (HWND)HIWORD(lParam);
#endif
            wnd->OnHScroll(code, pos, control);
            break;
        }
        case WM_VSCROLL:
        {
            WORD code = LOWORD(wParam);
#ifdef WIN32
            WORD pos = HIWORD(wParam);
            HWND control = (HWND)lParam;
#else
            WORD pos = LOWORD(lParam);
            HWND control = (HWND)HIWORD(lParam);
#endif
            wnd->OnVScroll(code, pos, control);
            break;
        }
#ifdef WIN32
        case WM_CTLCOLORBTN:
	{
          int nCtlColor = CTLCOLOR_BTN;
          HWND control = (HWND)LOWORD(lParam);
          HDC pDC = (HDC)HIWORD(wParam);
          return (DWORD)wnd->OnCtlColor(pDC, control, nCtlColor);
          break;
	}
        case WM_CTLCOLORDLG:
	{
          int nCtlColor = CTLCOLOR_DLG;
          HWND control = (HWND)LOWORD(lParam);
          HDC pDC = (HDC)HIWORD(wParam);
          return (DWORD)wnd->OnCtlColor(pDC, control, nCtlColor);
          break;
	}
        case WM_CTLCOLORLISTBOX:
	{
          int nCtlColor = CTLCOLOR_LISTBOX;
          HWND control = (HWND)LOWORD(lParam);
          HDC pDC = (HDC)HIWORD(wParam);
          return (DWORD)wnd->OnCtlColor(pDC, control, nCtlColor);
          break;
	}
        case WM_CTLCOLORMSGBOX:
	{
          int nCtlColor = CTLCOLOR_MSGBOX;
          HWND control = (HWND)LOWORD(lParam);
          HDC pDC = (HDC)HIWORD(wParam);
          return (DWORD)wnd->OnCtlColor(pDC, control, nCtlColor);
          break;
	}
        case WM_CTLCOLORSCROLLBAR:
	{
          int nCtlColor = CTLCOLOR_SCROLLBAR;
          HWND control = (HWND)LOWORD(lParam);
          HDC pDC = (HDC)HIWORD(wParam);
          return (DWORD)wnd->OnCtlColor(pDC, control, nCtlColor);
          break;
	}
        case WM_CTLCOLORSTATIC:
	{
          int nCtlColor = CTLCOLOR_STATIC;
          HWND control = (HWND)LOWORD(lParam);
          HDC pDC = (HDC)HIWORD(wParam);
          return (DWORD)wnd->OnCtlColor(pDC, control, nCtlColor);
          break;
	}
        case WM_CTLCOLOREDIT:
	{
          int nCtlColor = CTLCOLOR_EDIT;
          HWND control = (HWND)LOWORD(lParam);
          HDC pDC = (HDC)HIWORD(wParam);
          return (DWORD)wnd->OnCtlColor(pDC, control, nCtlColor);
          break;
	}
#else
        case WM_CTLCOLOR:
        {
          HWND control = (HWND)LOWORD(lParam);
          int nCtlColor = (int)HIWORD(lParam);
          HDC pDC = (HDC)wParam;
          return (DWORD)wnd->OnCtlColor(pDC, control, nCtlColor);
          break;
        }
#endif
        case WM_ERASEBKGND:
        {
            return wnd->OnEraseBkgnd((HDC)wParam);
            break;
        }
        case WM_CLOSE:
        {
            return !wnd->OnClose();
            break;
        }
        default:
          return FALSE;
    }
    return FALSE;
}

wxList *wxWinHandleList = NULL;
wxWnd *wxFindWinFromHandle(HWND hWnd)
{
  wxNode *node = wxWinHandleList->Find((long)hWnd);
  if (!node)
    return NULL;
  return (wxWnd *)node->Data();
}

/* wxWnd class used to implement all Windows 3 windows
 */
wxWnd::wxWnd(void)

{
  x_scrolling_enabled = TRUE;
  y_scrolling_enabled = TRUE;
  last_msg = 0;
  last_wparam = 0;
  last_lparam = 0;
  accelerator_table = NULL;

  xscroll_pixels_per_line = 0;
  yscroll_pixels_per_line = 0;
  xscroll_lines = 0;
  yscroll_lines = 0;
  xscroll_lines_per_page = 0;
  yscroll_lines_per_page = 0;
  xscroll_position = 0;
  yscroll_position = 0;
  background_brush = GetStockObject( LTGRAY_BRUSH );
  last_x_pos = -1.0;
  last_y_pos = -1.0;
  last_event = -1;
  is_canvas = FALSE;
  cdc = NULL;
}

wxWnd::~wxWnd(void)
{
  wxWinHandleList->DeleteObject(this);
}

// Default destroyer - override if you destroy it in some other way
// (e.g. with MDI child windows)
void wxWnd::DestroyWindow(void)
{
  ::DestroyWindow(handle);
  SetWindowLong(handle, 0, (long)0);
  handle = NULL;
}

void wxWnd::Create(wxWnd *parent, char *wclass, wxWindow *wx_win, char *title,
                    int x, int y, int width, int height,
                    DWORD style, char *dialog_template)
{
  wx_window = wx_win;
  is_dialog = (dialog_template != NULL);
  int x1 = 0;
  int y1 = 0;
  int x2 = 100;
  int y2 = 100;

  // Find parent's size, if it exists, to set up a possible default
  // panel size the size of the parent window
  RECT parent_rect;
  if (parent)
  {
    GetWindowRect(parent->handle, &parent_rect);

    // Convert from screen coordinates to parent coordinates
    x2 = parent_rect.right - parent_rect.left;
    y2 = parent_rect.bottom - parent_rect.top;
  }

  if (x > -1) x1 = x;
  if (y > -1) y1 = y;
  if (width > -1) x2 = x1 + width;
  if (height > -1) y2 = y1 + height;

  HWND hParent = NULL;
  if (parent)
    hParent = parent->handle;

  wxWndHook = this;

  if (is_dialog)
  {
    // MakeProcInstance doesn't seem to be needed in C7. Is it needed for
    // other compilers???
//    DLGPROC dlgproc = (DLGPROC)MakeProcInstance(wxWndProc, wxhInstance);

    handle = ::CreateDialog(wxhInstance, dialog_template, hParent,
                            (DLGPROC)wxDlgProc);
    MoveWindow(handle, x1, y1, x2 - x1, y2 - y1, FALSE);
  }
  else
    handle = CreateWindow(wclass,
                title,
                style,
                x1, y1,
                x2 - x1, y2 - y1,
                hParent, NULL, wxhInstance,
                NULL);

  wxWndHook = NULL;
  wxWinHandleList->Append((long)handle, this);

  // Can't do this for dialogs!!!!
  if (!is_dialog) SetWindowLong(handle, 0, (long)this);
}

void wxWnd::OnCreate(LPCREATESTRUCT cs)
{
}

BOOL wxWnd::OnPaint(void)
{
  return 1;
}

BOOL wxWnd::OnClose(void)
{
  return FALSE;
}

BOOL wxWnd::OnDestroy(void)
{
  return FALSE;
}

void wxWnd::OnSize(int x, int y, UINT flag)
{
}

// Deal with child commands from buttons etc.

BOOL wxWnd::OnCommand(WORD id, WORD cmd, HWND control)
{
  return FALSE;
}

void wxWnd::OnMenuSelect(WORD item, WORD flags, HMENU sysmenu)
{
}

BOOL wxWnd::OnActivate(BOOL state, BOOL minimized, HWND activate)
{
  if (wx_window)
  {
    wx_window->OnActivate(((state == WA_ACTIVE) || (state == WA_CLICKACTIVE)));
    return TRUE;
  }
  else return FALSE;
}

BOOL wxWnd::OnSetFocus(HWND hwnd)
{
  if (wx_window)
  {
    wx_window->OnSetFocus();
    return TRUE;
  }
  else return FALSE;
}

BOOL wxWnd::OnKillFocus(HWND hwnd)
{
  if (wx_window)
  {
    wx_window->OnKillFocus();
    return TRUE;
  }
  else return FALSE;
}

void wxWnd::OnVScroll(WORD code, WORD pos, HWND control)
{
}

void wxWnd::OnHScroll(WORD code, WORD pos, HWND control)
{
}

void wxWnd::CalcScrolledPosition(int x, int y, int *xx, int *yy)
{
  *xx = x - xscroll_position * xscroll_pixels_per_line;
  *yy = y - yscroll_position * yscroll_pixels_per_line;
}

void wxWnd::CalcUnscrolledPosition(int x, int y, float *xx, float *yy)
{
  *xx = (float)(x + xscroll_position * xscroll_pixels_per_line);
  *yy = (float)(y + yscroll_position * yscroll_pixels_per_line);
}

HBRUSH wxWnd::OnCtlColor(HDC pDC, HWND pWnd, UINT nCtlColor)
{
  if ((nCtlColor == CTLCOLOR_STATIC || nCtlColor == CTLCOLOR_BTN) && background_brush)
  {
    SetBkColor(pDC, RGB(200, 200, 200));
    return background_brush;
  }
  else return NULL;
}

BOOL wxWnd::OnEraseBkgnd(HDC pDC)
{
  return FALSE;
}

void wxWnd::OnLButtonDown(int x, int y, UINT flags)
{
}

void wxWnd::OnLButtonUp(int x, int y, UINT flags)
{
}

void wxWnd::OnRButtonDown(int x, int y, UINT flags)
{
}

void wxWnd::OnRButtonUp(int x, int y, UINT flags)
{
}

void wxWnd::OnMouseMove(int x, int y, UINT flags)
{
}

void wxWnd::OnChar(WORD wParam)
{
}

LONG wxWnd::DefWindowProc(UINT nMsg, WPARAM wParam, LPARAM lParam)
{
  return ::DefWindowProc(handle, nMsg, wParam, lParam);
}

BOOL wxWnd::ProcessMessage(MSG* pMsg)
{
  return FALSE;
}

BOOL wxWnd::OnMDIActivate(BOOL flag, HWND activate, HWND deactivate)
{
  return 1;
}

/*
 * Subwindow - used for panels and canvases
 *
 */

wxSubWnd::wxSubWnd(wxWnd *parent, char *wclass, wxWindow *wx_win,
                           int x, int y, int width, int height,
                           DWORD style, char *dialog_template)

{
  Create(parent, wclass, wx_win, NULL, x, y, width, height, style, dialog_template);
}

wxSubWnd::~wxSubWnd(void)
{
}


BOOL wxSubWnd::OnPaint(void)
{
  RECT rect;
  if (GetUpdateRect(handle, &rect, FALSE))
  {
    PAINTSTRUCT ps;
    // Hold a pointer to the dc so long as the OnPaint() message
    // is being processed
    cdc = BeginPaint(handle, &ps);
    if (wx_window)
      wx_window->OnPaint();

    cdc = NULL;
    EndPaint(handle, &ps);
    return 0;
  }
  return 1;
}

void wxSubWnd::OnSize(int x, int y, UINT flag)
{
  // Store DC for duration of size message
  cdc = GetDC(handle);
  if (wx_window)
    wx_window->OnSize(x, y);

  ReleaseDC(handle, cdc);
  cdc = NULL;
}



// Deal with child commands from buttons etc.
BOOL wxSubWnd::OnCommand(WORD id, WORD cmd, HWND control)
{
  wxWindow *item = wx_window->FindItem(id);
  if (item)
  {
    Bool value = item->Command(cmd);
    return value;
  }
  else
    return FALSE;
}

void wxSubWnd::OnLButtonDown(int x, int y, UINT flags)
{
  wxEvent event;
  float px = (float)x;
  float py = (float)y;

  DeviceToLogical(&px, &py);

  CalcUnscrolledPosition((int)px, (int)py, &event.x, &event.y);

  event.event = wxLEFTDOWN;
  event.flags = flags;

  last_x_pos = event.x; last_y_pos = event.y; last_event = wxLEFTDOWN;
  if (wx_window) wx_window->OnEvent(event);
}

void wxSubWnd::OnLButtonUp(int x, int y, UINT flags)
{
  wxEvent event;
  float px = (float)x;
  float py = (float)y;

  DeviceToLogical(&px, &py);

  CalcUnscrolledPosition((int)px, (int)py, &event.x, &event.y);

  event.event = wxLEFTUP;
  event.flags = flags;
  last_x_pos = event.x; last_y_pos = event.y; last_event = wxLEFTUP;
  if (wx_window) wx_window->OnEvent(event);
}

void wxSubWnd::OnRButtonDown(int x, int y, UINT flags)
{
  wxEvent event;
  float px = (float)x;
  float py = (float)y;

  DeviceToLogical(&px, &py);

  CalcUnscrolledPosition((int)px, (int)py, &event.x, &event.y);

  event.event = wxRIGHTDOWN;
  event.flags = flags;

  last_x_pos = event.x; last_y_pos = event.y; last_event = wxRIGHTDOWN;
  if (wx_window) wx_window->OnEvent(event);
}

void wxSubWnd::OnRButtonUp(int x, int y, UINT flags)
{
  wxEvent event;
  float px = (float)x;
  float py = (float)y;

  DeviceToLogical(&px, &py);

  CalcUnscrolledPosition((int)px, (int)py, &event.x, &event.y);

  event.event = wxRIGHTUP;
  event.flags = flags;
  last_x_pos = event.x; last_y_pos = event.y; last_event = wxRIGHTUP;
  if (wx_window) wx_window->OnEvent(event);
}

void wxSubWnd::OnMouseMove(int x, int y, UINT flags)
{
  // Set cursor
  if (wx_window->wx_cursor)
    ::SetCursor(wx_window->wx_cursor->ms_cursor);

  wxEvent event;
  float px = (float)x;
  float py = (float)y;

  DeviceToLogical(&px, &py);

  CalcUnscrolledPosition((int)px, (int)py, &event.x, &event.y);

  event.event = wxMOVE;
  event.flags = flags;

  // Window gets a click down message followed by a mouse move
  // message even if position isn't changed!  We want to discard
  // the trailing move event if x and y are the same.
  if ((last_event == wxRIGHTDOWN || last_event == wxLEFTDOWN) &&
      (last_x_pos == event.x && last_y_pos == event.y))
  {
    last_x_pos = event.x; last_y_pos = event.y;
    last_event = wxMOVE;
    return;
  }

  last_event = wxMOVE;
  last_x_pos = event.x; last_y_pos = event.y;
  if (wx_window) wx_window->OnEvent(event);
}

void wxSubWnd::OnChar(WORD wParam)
{
  if (wx_window) wx_window->OnChar((int)wParam);
}

void wxSubWnd::OnVScroll(WORD wParam, WORD pos, HWND control)
{
	short nScrollInc;
	
	switch ( wParam )
	{
		case SB_TOP:
                        nScrollInc = -yscroll_position;
			break;

		case SB_BOTTOM:
                        nScrollInc = yscroll_lines - yscroll_position;
			break;
			
		case SB_LINEUP:
			nScrollInc = -1;
			break;

		case SB_LINEDOWN:
			nScrollInc = 1;
			break;

		case SB_PAGEUP:
                        nScrollInc = min(-1, -yscroll_lines_per_page);
			break;

		case SB_PAGEDOWN:
                        nScrollInc = max(1, yscroll_lines_per_page);
			break;

		case SB_THUMBTRACK:
                        nScrollInc = pos - yscroll_position;
			break;

		default:
			nScrollInc = 0;
	}

        int w, h;
        RECT rect;
        GetWindowRect(handle, &rect);
        w = rect.right - rect.left;
        h = rect.bottom - rect.top;

        int nVscrollMax = max(0, (int)(yscroll_lines + 2 -  h/yscroll_pixels_per_line));
        if ( nScrollInc = max( -yscroll_position,
                        min( nScrollInc, nVscrollMax - yscroll_position ) ) )
	{
          yscroll_position += nScrollInc;

          if (y_scrolling_enabled)
            ScrollWindow(handle, 0, -yscroll_pixels_per_line * nScrollInc, NULL, NULL );
          else
            InvalidateRect(handle, NULL, FALSE);
          SetScrollPos(handle, SB_VERT, yscroll_position, TRUE );
	}
}

void wxSubWnd::OnHScroll( WORD wParam, WORD pos, HWND control)
{
  if (control)
  {
    wxSliderEvent(control, wParam, pos);
  }
  else
  {
	int nScrollInc;
	switch ( wParam )
	{
		case SB_LINEUP:
			nScrollInc = -1;
			break;

		case SB_LINEDOWN:
			nScrollInc = 1;
			break;

		case SB_PAGEUP:
                        nScrollInc = -xscroll_lines_per_page;
			break;

		case SB_PAGEDOWN:
                        nScrollInc = xscroll_lines_per_page;
			break;

                case SB_THUMBTRACK:
                        nScrollInc = pos - xscroll_position;
			break;

		default:
			nScrollInc = 0;
	}
        int w, h;
        RECT rect;
        GetWindowRect(handle, &rect);
        w = rect.right - rect.left;
        h = rect.bottom - rect.top;
        int nMaxWidth = xscroll_lines*xscroll_pixels_per_line;
        int nHscrollMax = max(0, (int)(2 + (nMaxWidth - w)/xscroll_pixels_per_line));

        if ( nScrollInc = max( -xscroll_position,
                         min( nScrollInc, nHscrollMax - xscroll_position ) ) )
	{
          xscroll_position += nScrollInc;
          if (x_scrolling_enabled)
            ScrollWindow(handle, -xscroll_pixels_per_line * nScrollInc, 0, NULL, NULL );
          else
            InvalidateRect(handle, NULL, FALSE);
          SetScrollPos(handle, SB_HORZ, xscroll_position, TRUE );
	}
  }
}

void wxGetCharSize(HWND wnd, int *x, int *y)
{
  TEXTMETRIC tm;
  HDC dc = GetDC(wnd);
  GetTextMetrics(dc, &tm);
  ReleaseDC(wnd, dc);
  *x = tm.tmAveCharWidth;
  *y = tm.tmHeight + tm.tmExternalLeading;
}

#endif

