{**************************************************************************}
{*  BitSoft Developmnet, L.L.C.                                           *}
{*  Copyright (C) 1995, 1996 BitSoft Development, L.L.C.                  *}
{*  All rights reserved.                                                  *}
{**************************************************************************}

unit MtdTests;

{$X+,B-}

interface

uses BsdTypes,
     Containr, ctArrays, ctQueues, ctTrees, ctStacks, ctBiTree;

{ TContainer methods }

procedure TestContainerInsert (Container: PContainer; TotalItems : LongInt);
{ Inserts TotalItems items into the container.  This method is used for
  inserting in the container the data that will be used in other tests. }

procedure TestStaticSequenceInsert (Sequence: PSequence; TotalItems :
  LongInt);
{ Inserts TotalItems items into the container, but using AtPut instead of
  the standard Insert method.  This test is useful when working with
  non-dynamically sized data structures, like for example, arrays. }

procedure TestContainerForEach (Container: PContainer);
{ Tests the ForEach method. }

procedure TestContainerForEachThat (Container: PContainer);
{ Tests the ForEachThat method. }

procedure TestSequenceDelete (Sequence: PSequence);
{ Tests the Delete method in TSequence descendants.  Separate methods for
  sequences and graphs are needed because of the way items to be deleted
  are selected.  The Delete method is common to all containers. }

procedure TestGraphDelete (Graph: PGraph);
{ Tests the Delete method in TGraph descendants.  Separate methods for
  sequences and graphs are needed because of the way items to be deleted
  are selected.  The Delete method is common to all containers. }

procedure TestSequenceDeleteAll (Sequence : PSequence);
{ Tests the DeleteAll method in sequences.  Separate methods for sequences
  and graphs are needed because of the way the list of deleted items is
  built.  The DeleteAll method is common to all containers.

  Note:  There must be only TotalDeleteItems items (see utils.pas) in the
  container.  Otherwise, there will be a memory leak when the items are
  deleted from the container.}

procedure TestGraphDeleteAll (Graph : PGraph);
{ Tests the DeleteAll method in graphs. Separate methods for sequences
  and graphs are needed because of the way the list of deleted items is
  built.  The DeleteAll method is common to all containers.

  Note:  There must be only TotalDeleteItems items (see utils.pas) in the
  container.  Otherwise, there will be a memory leak when the items are
  deleted from the container.}

procedure TestContainerDeleteAllThat (Container: PContainer);
{ Tests the DeleteAllThat method. }

procedure TestSequenceFree (Sequence: PSequence);
{ Tests the Free method in TSequence descendants.  Separate methods for
  sequences and graphs are needed because of the way items to be freed
  are selected.  The Free method is common to all containers. }

procedure TestGraphFree (Graph: PGraph);
{ Tests the Free method in TGraph descendants.  Separate methods for
  sequences and graphs are needed because of the way items to be freed
  are selected.  The Free method is common to all containers. }

procedure TestContainerFreeAll (Container : PContainer);

procedure TestContainerFreeAllThat (Container: PContainer);
{ Tests the FreeAllThat method. }

procedure TestContainerPack (Container : PContainer);

{ TSequence methods }

procedure TestSequenceAt (Sequence : PSequence);
procedure TestSequenceAtDelete (Sequence : PSequence);
procedure TestSequenceAtFree (Sequence : PSequence);
procedure TestSequenceAtInsert (Sequence : PSequence);

procedure TestStaticSequenceAtInsert (Sequence : PSequence);
{ Tests the AtInsert method in static (i.e. not dynamically sized) sequences.
  In static sequences, the last item in the sequence gets deleted but not
  disposed of.  This test takes this into account and frees the last
  item in the sequence before inserting a new one. }

procedure TestSequenceAtPut (Sequence : PSequence);
procedure TestSequenceFirst (Sequence : PSequence);
procedure TestSequenceNext (Sequence : PSequence);
procedure TestSequenceLast (Sequence : PSequence);
procedure TestSequencePrev (Sequence : PSequence);
procedure TestSequenceFirstThat (Sequence : PSequence);
procedure TestSequenceNextThat (Sequence : PSequence);
procedure TestSequenceLastThat (Sequence : PSequence);
procedure TestSequencePrevThat (Sequence : PSequence);
procedure TestSequenceSearch (Sequence : PSequence);

{ TGraph methods }

procedure TestGraphFirst (Graph : PGraph);
procedure TestGraphLast (Graph : PGraph);
procedure TestGraphNext (Graph : PGraph);
procedure TestGraphPrev (Graph : PGraph);
procedure TestGraphFirstThat (Graph : PGraph; var Key: String5);
procedure TestGraphLastThat (Graph : PGraph; var Key: String5);
procedure TestGraphNextThat (Graph : PGraph; var Key: String5);
procedure TestGraphPrevThat (Graph : PGraph; var Key: String5);
procedure TestGraphDuplicates (Graph : PGraph; var DuplicateKey: String5);
procedure TestGraphKeyFirst (Graph : PGraph; Key : String5);
procedure TestGraphNextExactMatch(Graph : PGraph; DuplicateKey: String5);
procedure TestGraphPrevExactMatch(Graph : PGraph; DuplicateKey: String5);
procedure TestGraphKeyLast (Graph : PGraph; Key : String5);
procedure TestGraphKeyFirstThat(Graph : PGraph; DuplicateKey : String5);
procedure TestGraphKeyLastThat(Graph : PGraph; DuplicateKey : String5);
procedure TestGraphItemPut (Graph : PGraph; Key : String5);
procedure TestGraphItemReplace (Graph : PGraph; Key : String5);
procedure TestGraphFind (Graph : PGraph; Key : String5);
procedure TestGraphFindThat (Graph : PGraph; Key : String5);

{ Array-specific methods }

procedure TestArrayAtClear (TestArray : PDynamicArray);

{ Queue-specific methods }

procedure TestQueueEnqueue (Queue : PQueue);
procedure TestQueueRemove (Queue : PQueue);
procedure TestQueueFront (Queue : PQueue);
procedure TestQueueRear (Queue : PQueue);
procedure TestDoubleEndedQueueRemoveFirst (Queue : PDoubleEndedQueue);
procedure TestDoubleEndedQueueRemoveLast (Queue : PDoubleEndedQueue);

{ Stack-specific methods }

procedure TestHugeCollectionStackPush (Stack : PHugeCollectionStack);
procedure TestHugeCollectionStackPop (Stack : PHugeCollectionStack);
procedure TestHugeCollectionStackTop (Stack : PHugeCollectionStack);
procedure TestHugeCollectionStackBottom (Stack : PHugeCollectionStack);

procedure TestArrayStackPush (Stack : PArrayStack);
procedure TestArrayStackPop (Stack : PArrayStack);
procedure TestArrayStackTop (Stack : PArrayStack);
procedure TestArrayStackBottom (Stack : PArrayStack);

procedure TestHugeArrayStackPush (Stack : PHugeArrayStack);
procedure TestHugeArrayStackPop (Stack : PHugeArrayStack);
procedure TestHugeArrayStackTop (Stack : PHugeArrayStack);
procedure TestHugeArrayStackBottom (Stack : PHugeArrayStack);

procedure TestLinkedStackPush (Stack : PLinkedStack);
procedure TestLinkedStackPop (Stack : PLinkedStack);
procedure TestLinkedStackTop (Stack : PLinkedStack);
procedure TestLinkedStackBottom (Stack : PLinkedStack);

procedure TestStreamStackPush (Stack : PStreamStack);
procedure TestStreamStackPop (Stack : PStreamStack);
procedure TestStreamStackTop (Stack : PStreamStack);
procedure TestStreamStackBottom (Stack : PStreamStack);

{ Binary Tree-specific methods }

procedure TestTreeTraverse(Tree: PBinaryTree);
procedure TestTreeTraverseThat(Tree: PBinaryTree);

implementation

uses Objects, Drivers, Memory,
     BsdTest,
     ctCollec,
     ObjTests, Utils, Types, Data;

{****************************************************************************}
{ TestArrayAtClear                                                           }
{****************************************************************************}
procedure TestArrayAtClear (TestArray : PDynamicArray);
var
  Item : Pointer;
  i : Integer;
  Index : LongInt;
  StrIndex : string;
begin
  if ExitTesting
    then Exit;
  StartTest('AtClear', 'Clearing items at random...');
  Writeln(TestWindow^.T);
  Randomize;
  for i := 1 to 20 do
  begin
    Item := nil;
    repeat
      TestArray^.DoneItem(Item);
      Index := Random(Pred(TestArray^.Count));
      Item := TestArray^.At(Index);
    until (Item <> nil)  and (TestReader^.ExtractText(Item) <> nil);
    Str(Index, StrIndex);
    WriteSubHeader('Clearing '+TestReader^.ExtractText(Item)^+' at index '+
      StrIndex);
    TestArray^.DoneItem(Item);
    SetInitTime;
    TestArray^.AtClear(Index);
    SetFinalTime;
    WriteTime;
  end; { for }
  NotifyDataChange;
  PauseTest;
end;

{****************************************************************************}
{ TestArrayStackBottom                                                       }
{****************************************************************************}
procedure TestArrayStackBottom (Stack : PArrayStack);
var
  Item : Pointer;
begin
  if ExitTesting
    then Exit;
  StartTest('Bottom', 'Getting item at the bottom of the stack...');
  Item := Stack^.Bottom;
  StopTest;
  TestReader^.ShowItem(Item);
  PauseTest;
end;

{****************************************************************************}
{ TestArrayStackPop                                                          }
{****************************************************************************}
procedure TestArrayStackPop (Stack : PArrayStack);
var
  Item : Pointer;
  i : Integer;
begin
  if ExitTesting
    then Exit;
  StartTest('Pop', 'Getting all items out of the stack...');
  Writeln(TestWindow^.T);
  for i := 1 to Stack^.Count do
  begin
    SetInitTime;
    Item := Stack^.Pop;
    SetFinalTime;
    WriteSubHeader('Popping '+TestReader^.ExtractText(Item)^);
    WriteTime;
    Stack^.FreeItem(Item);
  end; { for }
  PauseTest;
end;

{****************************************************************************}
{ TestArrayStackPush                                                         }
{****************************************************************************}
procedure TestArrayStackPush (Stack : PArrayStack);
var
  i : Integer;
  Item : Pointer;
  Key : String5;
  F : Text;
begin
  if ExitTesting
    then Exit;
  StartTest('Push', 'Adding 20 items to the stack...');
  Writeln(TestWindow^.T);
  Assign(F, 'Items.Dat');
  Reset(F);
  for i := 1 to 20 do
  begin
    Readln(F, Key);
    if LowMemory
      then begin
             Writeln(TestWindow^.T);
             Writeln(TestWindow^.T);
             Writeln(TestWindow^.T, 'Not enough memory... aborting test.');
             ExitTesting := True;
             Close(F);
             ResetApplication;
             Exit;
           end { if }
      else begin
             CreateNonDynamicTestRec(Key+' ', 0, NonDynamicRec);
             Item := @NonDynamicRec;
           end; { else }
    SetInitTime;
    Stack^.Push(Item);
    SetFinalTime;
    WriteSubHeader('Pushing '+Key);
    WriteTime;
  end; { for }
  PauseTest;
end;

{****************************************************************************}
{ TestArrayStackTop                                                          }
{****************************************************************************}
procedure TestArrayStackTop (Stack : PArrayStack);
var
  Item : Pointer;
begin
  if ExitTesting
    then Exit;
  StartTest('Top', 'Getting item at the top of the stack...');
  Item := Stack^.Top;
  StopTest;
  TestReader^.ShowItem(Item);
  PauseTest;
end;

{****************************************************************************}
{ TestContainerDeleteAllThat                                                 }
{****************************************************************************}
procedure TestContainerDeleteAllThat (Container : PContainer);
var
  Item : Pointer;
  DeletedItems : PStreamCollection;
  Counter : Integer;

  function Match(Item : Pointer) : Boolean; far;
  begin
    if (Item <> nil) and (TestReader^.ExtractText(Item) <> nil)
      then if TestReader^.ExtractText(Item)^[3] = 'J'
             then begin
                    if TestingMemArray
                      then Container^.FreeItem(Item)
                      else DeletedItems^.Insert(Item);
                    Inc(Counter);
                    Match := True;
                  end { if }
             else Match := False
      else Match := False;
  end; { Match }

  procedure FreeItem (Item : Pointer); far;
  begin
    Container^.FreeItem(Item);
  end;

begin
  if ExitTesting
    then Exit;
  DeletedItems := New(PStreamCollection, Init(500, 500));
  if DeletedItems = nil
    then begin
           Writeln(TestWindow^.T);
           Writeln(TestWindow^.T);
           Writeln(TestWindow^.T, 'Not enough memory... aborting test.');
           ExitTesting := True;
           ResetApplication;
           Exit;
         end; { if }
  Counter := 0;
  StartTest('DeleteAllThat', 'Deleting all items with 3rd character equal '+
    'to ''J''...');
  Container^.DeleteAllThat(@Match);
  StopTest;
  if not TestingMemArray
    then begin
           DeletedItems^.ForEach(@FreeItem);
           DeletedItems^.DeleteAll;
         end; { if }
  Dispose(DeletedItems, Done);
  WriteNumResult('Total items deleted:', Counter);
  NotifyDataChange;
  PauseTest;
end;

{****************************************************************************}
{ TestContainerForEach                                                       }
{****************************************************************************}
procedure TestContainerForEach (Container: PContainer);

  procedure ChangeLastCharacter(Item : Pointer); far;
  begin
    if (Item <> nil) and (TestReader^.ExtractText(Item) <> nil)
      then TestReader^.ExtractText(Item)^[5] := '-';
  end; { ChangeLastCharacter }

begin
  if ExitTesting
    then Exit;
  StartTest('ForEach', 'Appending a ''-'' character to all keys...');
  Container^.ForEach(@ChangeLastCharacter);
  StopTest;
  PauseTest;
end;

{****************************************************************************}
{ TestContainerForEachThat                                                   }
{****************************************************************************}
procedure TestContainerForEachThat (Container: PContainer);

  function Match(Item : Pointer): Boolean; far;
  begin
    if (Item <> nil) and (TestReader^.ExtractText(Item) <> nil)
      then if TestReader^.ExtractText(Item)^[3] = 'R'
             then Match := True
             else Match := False
      else Match := False;
  end; { Match }

  procedure ChangeLastCharacter(Item : Pointer); far;
  begin
    if (Item <> nil) and (TestReader^.ExtractText(Item) <> nil)
      then TestReader^.ExtractText(Item)^[5] := '@';
  end; { ChangeLastCharacter }

begin
  if ExitTesting
    then Exit;
  StartTest('ForEachThat',
    'Changing the last character of all items with ''R'' as the 3rd '+
    'character to @...');
  Container^.ForEachThat(@Match, @ChangeLastCharacter);
  StopTest;
  PauseTest;
end;

{****************************************************************************}
{ TestContainerFreeAll                                                       }
{****************************************************************************}
procedure TestContainerFreeAll (Container : PContainer);
begin
  if ExitTesting
    then Exit;
  StartTest('FreeAll', 'Disposing of all items in the container...');
  Container^.FreeAll;
  StopTest;
  NotifyDataChange;
  PauseTest;
end;

{****************************************************************************}
{ TestContainerFreeAllThat                                                   }
{****************************************************************************}
procedure TestContainerFreeAllThat (Container : PContainer);
var
  Item : Pointer;
  Counter : Integer;

  function Match(Item : Pointer) : Boolean; far;
  begin
    if (Item <> nil) and (TestReader^.ExtractText(Item) <> nil)
      then if TestReader^.ExtractText(Item)^[3] = 'B'
             then begin
                    Inc(Counter);
                    Match := True
                  end { if }
             else Match := False
      else Match := False;
  end; { Match }

begin
  if ExitTesting
    then Exit;
  Counter := 0;
  StartTest('FreeAllThat', 'Freeing all items with 3rd character equal '+
    'to ''B''...');
  Container^.FreeAllThat(@Match);
  StopTest;
  WriteNumResult('Total items freed:', Counter);
  NotifyDataChange;
  PauseTest;
end;

{****************************************************************************}
{ TestContainerInsert                                                        }
{****************************************************************************}
procedure TestContainerInsert (Container: PContainer; TotalItems: LongInt);
var
  SubHeader : string;
  F : Text;
  i : LongInt;
  Key : String5;
  Item : Pointer;
begin
  if ExitTesting
    then Exit;
  Assign(F, 'items.dat');
  Reset(F);
  FormatStr(SubHeader, 'Inserting %d items into the container...',
    TotalItems);
  StartTest('Insert', SubHeader);
  for i := 0 to Pred(TotalItems) do
  begin
    Readln(F, Key);
    if LowMemory
      then begin
             Writeln(TestWindow^.T);
             Writeln(TestWindow^.T);
             Writeln(TestWindow^.T, 'Not enough memory... aborting test.');
             ExitTesting := True;
             Close(F);
             ResetApplication;
             Exit;
           end { if }
      else if UseNonDynamicTestRec
             then begin
                    CreateNonDynamicTestRec(Key+' ', 0, NonDynamicRec);
                    Item := @NonDynamicRec;
                  end { if }
      else if UseNonDynamicTestObject
             then begin
                    CreateNonDynamicTestObject(Key+' ', 0, NonDynamicObject);
                    Item := @NonDynamicObject;
                  end { if }
      else if UseNonDynamicTestStaticObject
             then begin
                    CreateNonDynamicTestStaticObject(Key+' ', 0,
                      NonDynamicStaticObject);
                    Item := @NonDynamicStaticObject;
                  end { if }
      else Item := CreateItem(Key+' ', 0);
    Container^.Insert(Item);
  end; { for }
  StopTest;
  Close(F);
  NotifyDataChange;
  PauseTest;
end;

{****************************************************************************}
{ TestContainerPack                                                          }
{****************************************************************************}
procedure TestContainerPack (Container : PContainer);
begin
  if ExitTesting
    then Exit;
  StartTest('Pack', 'Packing the container...');
  Container^.Pack;
  StopTest;
  NotifyDataChange;
  PauseTest;
end;

{****************************************************************************}
{ TestDoubleEndedQueueRemoveFirst                                            }
{****************************************************************************}
procedure TestDoubleEndedQueueRemoveFirst (Queue : PDoubleEndedQueue);
var
  Item : Pointer;
begin
  if ExitTesting
    then Exit;
  StartTest('RemoveFirst', 'Removing first item in the queue...');
  Item := PDoubleEndedQueue(Queue)^.RemoveFirst;
  StopTest;
  TestReader^.ShowItem(Item);
  Queue^.FreeItem(Item);
  NotifyDataChange;
  PauseTest;
end;

{****************************************************************************}
{ TestDoubleEndedQueueRemoveLast                                             }
{****************************************************************************}
procedure TestDoubleEndedQueueRemoveLast (Queue : PDoubleEndedQueue);
var
  Item : Pointer;
begin
  if ExitTesting
    then Exit;
  StartTest('RemoveLast', 'Removing last item in the queue...');
  Item := PDoubleEndedQueue(Queue)^.RemoveLast;
  StopTest;
  TestReader^.ShowItem(Item);
  Queue^.FreeItem(Item);
  NotifyDataChange;
  PauseTest;
end;

{****************************************************************************}
{ TestGraphDelete                                                            }
{****************************************************************************}
procedure TestGraphDelete (Graph : PGraph);
var
  Count : Byte;
  Item, Hold : Pointer;

  function Match(Item : Pointer) : Boolean; far;
  begin
    if (Item <> nil) and (TestReader^.ExtractText(Item) <> nil)
      then if TestReader^.ExtractText(Item)^[3] = 'X'
             then Match := True
             else Match := False
      else Match := False;
  end; { Match }

begin
  if ExitTesting
    then Exit;
  WriteHeader('Delete');
  WriteSubHeader('Deleting the first 20 items with ''X'' as the 3rd '+
    'character...');
  Writeln(TestWindow^.T);
  Item := Graph^.FirstThat(@Match);
  Count := 1;
  while (Item <> nil) and (Count <= 20) do
  begin
    WriteSubHeader('Deleting '+TestReader^.ExtractText(Item)^);
    SetInitTime;
    Graph^.Delete(Item);
    SetFinalTime;
    WriteTime;
    Graph^.FreeItem(Item);
    Item := Graph^.FirstThat(@Match);
    Inc(Count)
  end; { while }
  WriteNumResult('Total items deleted:', Pred(Count));
  NotifyDataChange;
  PauseTest;
end;

{****************************************************************************}
{ TestGraphDeleteAll                                                         }
{****************************************************************************}
procedure TestGraphDeleteAll (Graph : PGraph);
var
  Items : array [1..TotalDeleteItems] of Pointer;
  Item : Pointer;
  i : Integer;
  GraphCount : LongInt;
begin
  if ExitTesting
    then Exit;
  GraphCount := Graph^.Count;
  Item := Graph^.First;
  i := 1;
  while Item <> nil do
  begin
    Items[i] := Item;
    Inc(i);
    Item := Graph^.Next(Item);
  end; { while }
  StartTest('DeleteAll', 'Deleting all items in the container...');
  Graph^.DeleteAll;
  StopTest;
  WriteSubHeader('Disposing of deleted items...');
  if (TypeOf(Graph^) <> TypeOf(TTestObjectBTree))
      and (TypeOf(Graph^) <> TypeOf(TTestObjectBPlusTree))
    then for i := 1 to GraphCount do
           Graph^.FreeItem(Items[i]);
  Writeln(TestWindow^.T, '  done.');
  NotifyDataChange;
  PauseTest;
end;

{****************************************************************************}
{ TestGraphDuplicates                                                        }
{****************************************************************************}
procedure TestGraphDuplicates (Graph : PGraph; var DuplicateKey : String5);
var
  Hold, Item : Pointer;
  Counter : LongInt;

  function Match(Item : Pointer) : Boolean; far;
  begin
    if (Item <> nil) and (TestReader^.ExtractText(Item) <> nil)
      then if TestReader^.ExtractText(Item)^ > 'GRKT'
             then Match := True
             else Match := False
      else Match := False;
  end; { Match }

  function MatchDuplicate(Item : Pointer) : Boolean; far;
  begin
    if (Item <> nil) and (TestReader^.ExtractText(Item) <> nil)
      then if TestReader^.ExtractText(Item)^ = DuplicateKey
             then MatchDuplicate := True
             else MatchDuplicate := False
      else MatchDuplicate := False;
  end; { MatchDuplicate }

begin
  if ExitTesting
    then Exit;
  Item := Graph^.FirstThat(@Match);
  DuplicateKey := TestReader^.ExtractText(Item)^;
  StartTest('Duplicates:', 'Testing duplicates in tree using the key:');
  writeln(TestWindow^.T, DuplicateKey:13);
  WriteSubHeader('(Duplicates is set to FALSE)');
  Writeln(TestWindow^.T);
  WriteSubHeader('Inserting first duplicate key...');
  Writeln(TestWindow^.T);
  if UseNonDynamicTestRec
    then begin
           CreateNonDynamicTestRec(DuplicateKey, 1, NonDynamicRec);
           Item := @NonDynamicRec;
         end { if }
    else Item := CreateItem(DuplicateKey, 1);
  Graph^.Insert(Item);
  Graph^.Reset;
  WriteSubHeader('Setting Duplicates to TRUE and trying again...');
  SetInitTime;
  Graph^.Duplicates := True;
  Graph^.Insert(Item);
  for Counter := 2 to Pred(TotalDuplicates) do
  begin
    if UseNonDynamicTestRec
      then begin
             CreateNonDynamicTestRec(DuplicateKey, Counter, NonDynamicRec);
             Item := @NonDynamicRec;
           end { if }
      else Item := CreateItem(DuplicateKey, Counter);
    Graph^.Insert(Item);
  end; { for }
  StopTest;
  NotifyDataChange;
  PauseTest;
end;

{****************************************************************************}
{ TestGraphFind                                                              }
{****************************************************************************}
procedure TestGraphFind (Graph : PGraph; Key : String5);
var
  Hits : LongInt;
begin
  if ExitTesting
    then Exit;
  StartTest('Find', 'Finding all items with a '+ Key + ' key...');
  Graph^.Find(@Key, Hits);
  StopTest;
  WriteNumResult('Items found:', Hits);
  PauseTest;
end;

{****************************************************************************}
{ TestGraphFindThat                                                          }
{****************************************************************************}
procedure TestGraphFindThat (Graph : PGraph; Key : String5);
var
  Hits : LongInt;

  function Match(Item : Pointer) : Boolean; far;
  begin
    if (Item <> nil) and ((TestReader^.ExtractIndex(Item) mod 2) = 0) and
       (TestReader^.ExtractIndex(Item) <> 0)
      then Match := True
      else Match := False;
  end; { Match }

begin
  if ExitTesting
    then Exit;
  StartTest('FindThat', 'Finding all '+ Key +
    ' items with even Index field...');
  Graph^.FindThat(@Key, @Match, Hits);
  StopTest;
  WriteNumResult('Items found...', Hits);
  PauseTest;
end;

{****************************************************************************}
{ TestGraphFree                                                              }
{****************************************************************************}
procedure TestGraphFree (Graph : PGraph);
var
  Count : Byte;
  Item : Pointer;

  function Match(Item : Pointer) : Boolean; far;
  begin
    if (Item <> nil) and (TestReader^.ExtractText(Item)^[3] = 'W')
      then Match := True
      else Match := False;
  end; { Match }

begin
  if ExitTesting
    then Exit;
  WriteHeader('Free');
  WriteSubHeader('Freeing the first 20 items with ''W'' as the 3rd '+
    'character...');
  Writeln(TestWindow^.T);
  Item := Graph^.FirstThat(@Match);
  Count := 1;
  while (Item <> nil) and (Count <= 20) do
  begin
    WriteSubHeader('Freeing '+TestReader^.ExtractText(Item)^);
    SetInitTime;
    Graph^.Free(Item);
    SetFinalTime;
    WriteTime;
    Item := Graph^.FirstThat(@Match);
    Inc(Count);
  end; { while }
  WriteNumResult('Total items freed:', Pred(Count));
  NotifyDataChange;
  PauseTest;
end;

{****************************************************************************}
{ TestGraphFirst                                                             }
{****************************************************************************}
procedure TestGraphFirst (Graph : PGraph);
var
  Item : Pointer;
begin
  if ExitTesting
    then Exit;
  StartTest('First', 'Retrieving the first item in the container...');
  Item := Graph^.First;
  StopTest;
  TestReader^.ShowItem(Item);
  Graph^.DoneItem(Item);
  PauseTest;
end;

{****************************************************************************}
{ TestGraphFirstThat                                                         }
{****************************************************************************}
procedure TestGraphFirstThat (Graph : PGraph; var Key: String5);
var
  Item : Pointer;

  function Match(Item : Pointer) : Boolean; far;
  begin
    if (Item <> nil) and (TestReader^.ExtractText(Item) <> nil)
      then if TestReader^.ExtractText(Item)^ > 'UXVT'
             then Match := True
             else Match := False
      else Match := False;
  end; { Match }

begin
  if ExitTesting
    then Exit;
  StartTest('FirstThat', 'Retrieving first item with key > ''UXVT''');
  Item := Graph^.FirstThat(@Match);
  StopTest;
  if Item <> nil
    then begin
           TestReader^.ShowItem(Item);
           Key := TestReader^.ExtractText(Item)^;
         end { if }
    else begin
           WriteResult('Not found');
           Item := Graph^.First;
           Key := TestReader^.ExtractText(Item)^;
           Graph^.DoneItem(Item);
         end; { else }
  Graph^.DoneItem(Item);
  PauseTest;
end;

{****************************************************************************}
{ TestGraphKeyFirstThat                                                      }
{****************************************************************************}
procedure TestGraphKeyFirstThat(Graph : PGraph; DuplicateKey : String5);
var
  Item : Pointer;

  function Match(Item : Pointer) : Boolean; far;
  begin
    Match := (Item <> nil) and (TestReader^.ExtractIndex(Item) < 5);
  end; { Match }

begin
  if ExitTesting
    then Exit;
  StartTest('KeyFirstThat', 'Retrieving the first duplicate key item with '+
    'an index lower than 5');
  Item := Graph^.KeyFirstThat(@Match, @DuplicateKey);
  StopTest;
  if Item <> nil
    then TestReader^.ShowItem(Item)
    else WriteResult('... not found ...');
  Graph^.DoneItem(Item);
  PauseTest;
end;

{****************************************************************************}
{ TestGraphKeyLastThat                                                       }
{****************************************************************************}
procedure TestGraphKeyLastThat(Graph : PGraph; DuplicateKey : String5);
var
  Item : Pointer;

  function Match(Item : Pointer) : Boolean; far;
  begin
    Match := (Item <> nil) and (TestReader^.ExtractIndex(Item) > 5);
  end; { Match }

begin
  if ExitTesting
    then Exit;
  StartTest('KeyFirstThat', 'Retrieving the last duplicate key item with '+
    'an index higher than 5');
  Item := Graph^.KeyLastThat(@Match, @DuplicateKey);
  StopTest;
  if Item <> nil
    then TestReader^.ShowItem(Item)
    else WriteResult('... not found ...');
  Graph^.DoneItem(Item);
  PauseTest;
end;

{****************************************************************************}
{ TestGraphKeyFirst                                                          }
{****************************************************************************}
procedure TestGraphKeyFirst (Graph : PGraph; Key : String5);
var
  Item : Pointer;
begin
  if ExitTesting
    then Exit;
  StartTest('KeyFirst', 'Retrieving the first duplicate '+ Key +
    ' key...');
  Item := Graph^.KeyFirst(@Key);
  StopTest;
  TestReader^.ShowItem(Item);
  Graph^.DoneItem(Item);
  PauseTest;
end;

{****************************************************************************}
{ TestGraphKeyLast                                                           }
{****************************************************************************}
procedure TestGraphKeyLast (Graph : PGraph; Key : String5);
var
  Item : Pointer;
begin
  if ExitTesting
    then Exit;
  StartTest('KeyLast', 'Retrieving the last duplicate '+ Key +
    ' key...');
  Item := Graph^.KeyLast(@Key);
  StopTest;
  TestReader^.ShowItem(Item);
  Graph^.DoneItem(Item);
  PauseTest;
end;

{****************************************************************************}
{ TestGraphNextExactMatch(Graph                                              }
{****************************************************************************}
procedure TestGraphNextExactMatch(Graph : PGraph; DuplicateKey: String5);
var
  Hold, Item : Pointer;
begin
  if ExitTesting
    then Exit;
  WriteHeader('Next (ExactMatch = True)');
  WriteSubHeader('Displaying in order the duplicate items...');
  Writeln(TestWindow^.T);
  Writeln(TestWindow^.T);
  writeln('1');
  Graph^.ExactMatch := True;
  Item := Graph^.KeyFirst(@DuplicateKey);
  while Item <> nil do
  begin
    TestReader^.ShowItem(Item);
    Hold := Item;
    Item := Graph^.Next(Item);
    Graph^.DoneItem(Hold);
  end; { while }
  PauseTest;
end;

{****************************************************************************}
{ TestGraphKeyPrev                                                           }
{****************************************************************************}
procedure TestGraphPrevExactMatch(Graph : PGraph; DuplicateKey: String5);
var
  Hold, Item : Pointer;
begin
  if ExitTesting
    then Exit;
  WriteHeader('Prev (ExactMatch = True)');
  WriteSubHeader('Displaying in reverse order the duplicate items...');
  Writeln(TestWindow^.T);
  Writeln(TestWindow^.T);
  Graph^.ExactMatch := True;
  Item := Graph^.KeyLast(@DuplicateKey);
  while Item <> nil do
  begin
    TestReader^.ShowItem(Item);
    Hold := Item;
    Item := Graph^.Prev(Item);
    Graph^.DoneItem(Hold);
  end; { while }
  PauseTest;
end;

{****************************************************************************}
{ TestGraphItemPut                                                           }
{****************************************************************************}
procedure TestGraphItemPut (Graph : PGraph; Key : String5);
var
  OldItem, NewItem : Pointer;
begin
  if ExitTesting
    then Exit;
  if UseNonDynamicTestRec
    then begin
           CreateNonDynamicTestRec('****', 1, NonDynamicRec);
           NewItem := @NonDynamicRec;
         end { if }
    else NewItem := CreateItem('****', 1);
  OldItem := Graph^.KeyFirst(@Key);
  StartTest('ItemPut', 'Replacing the first item with key '+ Key +
    ' with a new item with **** as its key...');
  Graph^.ItemPut(OldItem, NewItem);
  StopTest;
  Graph^.FreeItem(OldItem);
  NotifyDataChange;
  PauseTest;
end;

{****************************************************************************}
{ TestGraphItemReplace                                                       }
{****************************************************************************}
procedure TestGraphItemReplace (Graph : PGraph; Key : String5);
var
  OldItem, NewItem : Pointer;
begin
  if ExitTesting
    then Exit;
  if UseNonDynamicTestRec
    then begin
           CreateNonDynamicTestRec(Copy(Key, 1, 4) + '*', 1, NonDynamicRec);
           NewItem := @NonDynamicRec;
         end { if }
    else NewItem := CreateItem(Copy(Key, 1, 4) + '*', 1);
  OldItem := Graph^.KeyFirst(@Key);
  StartTest('ItemReplace', 'Replacing the first item with key '+ Key +
    ' with a new item with '+Copy(Key, 1, 4) + '* as its key...');
  Graph^.ItemReplace(OldItem, NewItem);
  StopTest;
  NotifyDataChange;
  PauseTest;
end;

{****************************************************************************}
{ TestGraphLast                                                              }
{****************************************************************************}
procedure TestGraphLast (Graph : PGraph);
var
  Item : Pointer;
begin
  if ExitTesting
    then Exit;
  StartTest('Last', 'Retrieving the last item in the container...');
  Item := Graph^.Last;
  StopTest;
  TestReader^.ShowItem(Item);
  Graph^.DoneItem(Item);
  PauseTest;
end;

{****************************************************************************}
{ TestGraphLastThat                                                          }
{****************************************************************************}
procedure TestGraphLastThat (Graph : PGraph; var Key: String5);
var
  Item : Pointer;

  function Match(Item : Pointer) : Boolean; far;
  begin
    if (Item <> nil) and (TestReader^.ExtractText(Item)^ < 'DRTG')
      then Match := True
      else Match := False;
  end; { Match }

begin
  if ExitTesting
    then Exit;
  StartTest('LastThat', 'Retrieving last item with key < ''DRTG''');
  Item := Graph^.LastThat(@Match);
  StopTest;
  if Item <> nil
    then begin
           TestReader^.ShowItem(Item);
           Key := TestReader^.ExtractText(Item)^;
         end { if }
    else begin
           WriteResult('Not found');
           Item := Graph^.First;
           Key := TestReader^.ExtractText(Item)^;
           Graph^.DoneItem(Item);
         end; { else }
  Graph^.DoneItem(Item);
  PauseTest;
end;

{****************************************************************************}
{ TestGraphNext                                                              }
{****************************************************************************}
procedure TestGraphNext (Graph : PGraph);
var
  Counter : LongInt;
  Item : Pointer;
begin
  if ExitTesting
    then Exit;
  StartTest('Next', 'Traversing the graph using First and Next...');
  Counter := 0;
  Graph^.ExactMatch := False;
  Item := Graph^.First;
  while Item <> nil do
  begin
    Graph^.DoneItem(Item);
    Inc(Counter);
    Item := Graph^.Next(Item);
  end; { while }
  StopTest;
  WriteNumResult('Total nodes visited:', Counter);
  PauseTest;
end;

{****************************************************************************}
{ TestGraphNextThat                                                          }
{****************************************************************************}
procedure TestGraphNextThat (Graph : PGraph; var Key: String5);
var
  Hold, Item : Pointer;

  function MatchFirst(Item : Pointer) : Boolean; far;
  begin
    if (Item <> nil) and (TestReader^.ExtractText(Item)^ > 'UXVT')
      then MatchFirst := True
      else MatchFirst := False;
  end; { MatchFirst }

  function MatchNext(Item : Pointer) : Boolean; far;
  begin
    if (Item <> nil) and (TestReader^.ExtractText(Item)^[3] = 'Q')
     then MatchNext := True
     else MatchNext := False;
  end; { MatchNext }

begin
  if ExitTesting
    then Exit;
  Graph^.ExactMatch := False;
  Item := Graph^.FirstThat(@MatchFirst);
  StartTest('NextThat', 'Retrieving next item with ''Q'' as the 3rd '+
    'character after first item with key > ''UXVT''');
  Hold := Item;
  Item := Graph^.NextThat(@MatchNext, Item);
  StopTest;
  if Item <> nil
    then begin
           TestReader^.ShowItem(Item);
           Key := TestReader^.ExtractText(Item)^;
         end { if }
    else begin
           WriteResult('Not found');
           Item := Graph^.First;
           Key := TestReader^.ExtractText(Item)^;
           Graph^.DoneItem(Item);
         end; { else }
  Graph^.DoneItem(Hold);
  Graph^.DoneItem(Item);
  PauseTest;
end;


{****************************************************************************}
{ TestGraphPrev                                                              }
{****************************************************************************}
procedure TestGraphPrev (Graph : PGraph);
var
  Counter : LongInt;
  Item : Pointer;
begin
  if ExitTesting
    then Exit;
  StartTest('Prev', 'Traversing the graph using Last and Prev...');
  Counter := 0;
  Graph^.ExactMatch := False;
  Item := Graph^.Last;
  while Item <> nil do
  begin
    Graph^.DoneItem(Item);
    Inc(Counter);
    Item := Graph^.Prev(Item);
  end; { while }
  StopTest;
  WriteNumResult('Total nodes visited:', Counter);
  PauseTest;
end;

{****************************************************************************}
{ TestGraphPrevThat                                                          }
{****************************************************************************}
procedure TestGraphPrevThat (Graph : PGraph; var Key: String5);
var
  Hold, Item : Pointer;

  function MatchFirst(Item : Pointer) : Boolean; far;
  begin
    if (Item <> nil) and (TestReader^.ExtractText(Item)^ < 'DRTG')
      then MatchFirst := True
      else MatchFirst := False;
  end; { MatchFirst }

  function MatchNext(Item : Pointer) : Boolean; far;
  begin
    if (Item <> nil) and (TestReader^.ExtractText(Item)^[3] = 'F')
     then MatchNext := True
     else MatchNext := False;
  end; { MatchNext }

begin
  if ExitTesting
    then Exit;
  Graph^.ExactMatch := False;
  Item := Graph^.LastThat(@MatchFirst);
  StartTest('PrevThat', 'Retrieving first item with ''F'' as the 3rd '+
    'character before last item with key < ''DRTG''');
  Hold := Item;
  Item := Graph^.PrevThat(@MatchNext, Item);
  StopTest;
  if Item <> nil
    then begin
           TestReader^.ShowItem(Item);
           Key := TestReader^.ExtractText(Item)^;
         end { if }
    else begin
           WriteResult('Not found');
           Item := Graph^.Last;
           Key := TestReader^.ExtractText(Item)^;
           Graph^.DoneItem(Item);
         end; { else }
  Graph^.DoneItem(Hold);
  Graph^.DoneItem(Item);
  PauseTest;
end;

{****************************************************************************}
{ TestHugeArrayStackBottom                                                   }
{****************************************************************************}
procedure TestHugeArrayStackBottom (Stack : PHugeArrayStack);
var
  Item : Pointer;
begin
  if ExitTesting
    then Exit;
  StartTest('Bottom', 'Getting item at the bottom of the stack...');
  Item := Stack^.Bottom;
  StopTest;
  TestReader^.ShowItem(Item);
  PauseTest;
end;

{****************************************************************************}
{ TestHugeArrayStackPop                                                      }
{****************************************************************************}
procedure TestHugeArrayStackPop (Stack : PHugeArrayStack);
var
  Item : Pointer;
  i : Integer;
begin
  if ExitTesting
    then Exit;
  StartTest('Pop', 'Getting all items out of the stack...');
  Writeln(TestWindow^.T);
  for i := 1 to Stack^.Count do
  begin
    SetInitTime;
    Item := Stack^.Pop;
    SetFinalTime;
    WriteSubHeader('Popping '+TestReader^.ExtractText(Item)^);
    WriteTime;
    Stack^.FreeItem(Item);
  end; { for }
  PauseTest;
end;

{****************************************************************************}
{ TestHugeArrayStackPush                                                     }
{****************************************************************************}
procedure TestHugeArrayStackPush (Stack : PHugeArrayStack);
var
  i : Integer;
  Item : Pointer;
  Key : String5;
  F : Text;
begin
  if ExitTesting
    then Exit;
  StartTest('Push', 'Adding 20 items to the stack...');
  Writeln(TestWindow^.T);
  Assign(F, 'Items.Dat');
  Reset(F);
  for i := 1 to 20 do
  begin
    Readln(F, Key);
    if LowMemory
      then begin
             Writeln(TestWindow^.T);
             Writeln(TestWindow^.T);
             Writeln(TestWindow^.T, 'Not enough memory... aborting test.');
             ExitTesting := True;
             Close(F);
             ResetApplication;
             Exit;
           end { if }
      else begin
             CreateNonDynamicTestRec(Key+' ', 0, NonDynamicRec);
             Item := @NonDynamicRec;
           end; { else }
    SetInitTime;
    Stack^.Push(Item);
    SetFinalTime;
    WriteSubHeader('Pushing '+Key);
    WriteTime;
  end; { for }
  PauseTest;
end;

{****************************************************************************}
{ TestHugeArrayStackTop                                                      }
{****************************************************************************}
procedure TestHugeArrayStackTop (Stack : PHugeArrayStack);
var
  Item : Pointer;
begin
  if ExitTesting
    then Exit;
  StartTest('Top', 'Getting item at the top of the stack...');
  Item := Stack^.Top;
  StopTest;
  TestReader^.ShowItem(Item);
  PauseTest;
end;

{****************************************************************************}
{ TestHugeCollectionStackBottom                                              }
{****************************************************************************}
procedure TestHugeCollectionStackBottom (Stack : PHugeCollectionStack);
var
  Item : Pointer;
begin
  if ExitTesting
    then Exit;
  StartTest('Bottom', 'Getting item at the bottom of the stack...');
  Item := Stack^.Bottom;
  StopTest;
  TestReader^.ShowItem(Item);
  PauseTest;
end;

{****************************************************************************}
{ TestHugeCollectionStackPop                                                 }
{****************************************************************************}
procedure TestHugeCollectionStackPop (Stack : PHugeCollectionStack);
var
  Item : Pointer;
  i : Integer;
begin
  if ExitTesting
    then Exit;
  StartTest('Pop', 'Getting all items out of the stack...');
  Writeln(TestWindow^.T);
  for i := 1 to Stack^.Count do
  begin
    SetInitTime;
    Item := Stack^.Pop;
    SetFinalTime;
    WriteSubHeader('Popping '+TestReader^.ExtractText(Item)^);
    WriteTime;
    Stack^.FreeItem(Item);
  end; { for }
  PauseTest;
end;

{****************************************************************************}
{ TestHugeCollectionStackPush                                                }
{****************************************************************************}
procedure TestHugeCollectionStackPush (Stack : PHugeCollectionStack);
var
  i : Integer;
  Item : Pointer;
  Key : String5;
  F : Text;
begin
  if ExitTesting
    then Exit;
  StartTest('Push', 'Adding 20 items to the stack...');
  Writeln(TestWindow^.T);
  Assign(F, 'Items.Dat');
  Reset(F);
  for i := 1 to 20 do
  begin
    Readln(F, Key);
    if LowMemory
      then begin
             Writeln(TestWindow^.T);
             Writeln(TestWindow^.T);
             Writeln(TestWindow^.T, 'Not enough memory... aborting test.');
             ExitTesting := True;
             Close(F);
             ResetApplication;
             Exit;
           end { if }
      else Item := CreateItem(Key, 0);
    SetInitTime;
    Stack^.Push(Item);
    SetFinalTime;
    WriteSubHeader('Pushing '+Key);
    WriteTime;
  end; { for }
  PauseTest;
end;

{****************************************************************************}
{ TestHugeCollectionStackTop                                                 }
{****************************************************************************}
procedure TestHugeCollectionStackTop (Stack : PHugeCollectionStack);
var
  Item : Pointer;
begin
  if ExitTesting
    then Exit;
  StartTest('Top', 'Getting item at the top of the stack...');
  Item := Stack^.Top;
  StopTest;
  TestReader^.ShowItem(Item);
  PauseTest;
end;

{****************************************************************************}
{ TestLinkedStackBottom                                                      }
{****************************************************************************}
procedure TestLinkedStackBottom (Stack : PLinkedStack);
var
  Item : Pointer;
begin
  if ExitTesting
    then Exit;
  StartTest('Bottom', 'Getting item at the bottom of the stack...');
  Item := Stack^.Bottom;
  StopTest;
  TestReader^.ShowItem(Item);
  PauseTest;
end;

{****************************************************************************}
{ TestLinkedStackPop                                                         }
{****************************************************************************}
procedure TestLinkedStackPop (Stack : PLinkedStack);
var
  Item : Pointer;
  i : Integer;
begin
  if ExitTesting
    then Exit;
  StartTest('Pop', 'Getting all items out of the stack...');
  Writeln(TestWindow^.T);
  for i := 1 to Stack^.Count do
  begin
    SetInitTime;
    Item := Stack^.Pop;
    SetFinalTime;
    WriteSubHeader('Popping '+TestReader^.ExtractText(Item)^);
    WriteTime;
    Stack^.FreeItem(Item);
  end; { for }
  PauseTest;
end;

{****************************************************************************}
{ TestLinkedStackPush                                                        }
{****************************************************************************}
procedure TestLinkedStackPush (Stack : PLinkedStack);
var
  i : Integer;
  Item : Pointer;
  Key : String5;
  F : Text;
begin
  if ExitTesting
    then Exit;
  StartTest('Push', 'Adding 20 items to the stack...');
  Writeln(TestWindow^.T);
  Assign(F, 'Items.Dat');
  Reset(F);
  for i := 1 to 20 do
  begin
    Readln(F, Key);
    if LowMemory
      then begin
             Writeln(TestWindow^.T);
             Writeln(TestWindow^.T);
             Writeln(TestWindow^.T, 'Not enough memory... aborting test.');
             ExitTesting := True;
             Close(F);
             ResetApplication;
             Exit;
           end { if }
      else Item := CreateItem(Key, 0);
    SetInitTime;
    Stack^.Push(Item);
    SetFinalTime;
    WriteSubHeader('Pushing '+Key);
    WriteTime;
  end; { for }
  PauseTest;
end;

{****************************************************************************}
{ TestLinkedStackTop                                                         }
{****************************************************************************}
procedure TestLinkedStackTop (Stack : PLinkedStack);
var
  Item : Pointer;
begin
  if ExitTesting
    then Exit;
  StartTest('Top', 'Getting item at the top of the stack...');
  Item := Stack^.Top;
  StopTest;
  TestReader^.ShowItem(Item);
  PauseTest;
end;

{****************************************************************************}
{ TestQueueEnqueue                                                           }
{****************************************************************************}
procedure TestQueueEnqueue (Queue : PQueue);
var
  F : Text;
  i : Integer;
  Key : String5;
  Item : Pointer;
begin
  if ExitTesting
    then Exit;
  StartTest('EnQueue', 'Adding 20 items to the queue...');
  Writeln(TestWindow^.T);
  Assign(F, 'items.dat');
  Reset(F);
  for i := 1 to 20 do
  begin
    Readln(F, Key);
    if LowMemory
      then begin
             Writeln(TestWindow^.T);
             Writeln(TestWindow^.T);
             Writeln(TestWindow^.T, 'Not enough memory... aborting test.');
             ExitTesting := True;
             Close(F);
             ResetApplication;
             Exit;
           end { if }
      else Item := CreateItem(Key, 0);
    SetInitTime;
    Queue^.Enqueue(Item);
    SetFinalTime;
    WriteSubHeader('Enqueueing '+Key);
    WriteTime;
  end; { for }
  PauseTest;
end;

{****************************************************************************}
{ TestQueueFront                                                             }
{****************************************************************************}
procedure TestQueueFront (Queue : PQueue);
var
  Item : Pointer;
begin
  if ExitTesting
    then Exit;
  StartTest('Front', 'Getting item at the front of the queue...');
  Item := Queue^.Front;
  StopTest;
  TestReader^.ShowItem(Item);
  PauseTest;
end;

{****************************************************************************}
{ TestQueueRear                                                              }
{****************************************************************************}
procedure TestQueueRear (Queue : PQueue);
var
  Item : Pointer;
begin
  if ExitTesting
    then Exit;
  StartTest('Rear', 'Getting item at the rear of the queue...');
  Item := Queue^.Rear;
  StopTest;
  TestReader^.ShowItem(Item);
  PauseTest;
end;

{****************************************************************************}
{ TestQueueRemove                                                            }
{****************************************************************************}
procedure TestQueueRemove (Queue : PQueue);
var
  i : Integer;
  Item : Pointer;
begin
  if ExitTesting
    then Exit;
  StartTest('Remove', 'Getting all items out of the queue...');
  Writeln(TestWindow^.T);
  for i := 1 to Queue^.Count do
  begin
    SetInitTime;
    Item := Queue^.Remove;
    SetFinalTime;
    WriteSubHeader('Removing '+TestReader^.ExtractText(Item)^);
    WriteTime;
    Queue^.FreeItem(Item);
  end; { for }
  PauseTest;
end;

{****************************************************************************}
{ TestSequenceAt                                                             }
{****************************************************************************}
procedure TestSequenceAt (Sequence : PSequence);
var
  Item : Pointer;
  i : Integer;
  Index : LongInt;
  StrIndex : string;
begin
  if ExitTesting
    then Exit;
  StartTest('At', 'Displaying items at random using At...');
  Writeln(TestWindow^.T);
  Randomize;
  for i := 1 to 20 do
  begin
    Item := nil;
    repeat
      Sequence^.DoneItem(Item);
      Index := Random(Pred(Sequence^.Count));
      SetInitTime;
      Item := Sequence^.At(Index);
      SetFinalTime;
    until (Item <> nil) and (TestReader^.ExtractText(Item) <> nil);
    Str(Index, StrIndex);
    WriteSubHeader('Retrieving '+TestReader^.ExtractText(Item)^+' at index '+
      StrIndex);
    Sequence^.DoneItem(Item);
    WriteTime;
  end; { for }
  PauseTest;
end;

{****************************************************************************}
{ TestSequenceAtDelete                                                       }
{****************************************************************************}
procedure TestSequenceAtDelete (Sequence : PSequence);
var
  Item : Pointer;
  i : Integer;
  Index : LongInt;
  StrIndex : string;
begin
  if ExitTesting
    then Exit;
  StartTest('AtDelete', 'Deleting items at random...');
  Writeln(TestWindow^.T);
  Randomize;
  for i := 1 to 20 do
  begin
    Item := nil;
    repeat
      Sequence^.DoneItem(Item);
      Index := Random(Pred(Sequence^.Count));
      Item := Sequence^.At(Index);
    until (Item <> nil) and (TestReader^.ExtractText(Item) <> nil);
    Str(Index, StrIndex);
    WriteSubHeader('Deleting '+TestReader^.ExtractText(Item)^+' at index '+
      StrIndex);
    if TestingMemArray
      then Sequence^.FreeItem(Item);
    SetInitTime;
    Sequence^.AtDelete(Index);
    SetFinalTime;
    if not TestingMemArray
      then Sequence^.FreeItem(Item);
    WriteTime;
  end; { for }
  NotifyDataChange;
  PauseTest;
end;

{****************************************************************************}
{ TestSequenceAtFree                                                         }
{****************************************************************************}
procedure TestSequenceAtFree (Sequence : PSequence);
var
  Item : Pointer;
  i : Integer;
  Index : LongInt;
  StrIndex : string;
begin
  if ExitTesting
    then Exit;
  StartTest('AtFree', 'Freeing items at random...');
  Writeln(TestWindow^.T);
  Randomize;
  for i := 1 to 20 do
  begin
    Item := nil;
    repeat
      Sequence^.DoneItem(Item);
      Index := Random(Pred(Sequence^.Count));
      Item := Sequence^.At(Index);
    until (Item <> nil) and (TestReader^.ExtractText(Item) <> nil);
    Str(Index, StrIndex);
    WriteSubHeader('Freeing '+TestReader^.ExtractText(Item)^+' at index '+
      StrIndex);
    Sequence^.DoneItem(Item);
    SetInitTime;
    Sequence^.AtFree(Index);
    SetFinalTime;
    WriteTime;
  end; { for }
  NotifyDataChange;
  PauseTest;
end;

{****************************************************************************}
{ TestSequenceAtInsert                                                       }
{****************************************************************************}
procedure TestSequenceAtInsert (Sequence : PSequence);
var
  Item : Pointer;
  i : Integer;
  Key : String5;
  Index : LongInt;
  StrIndex : string;
  F : Text;
begin
  if ExitTesting
    then Exit;
  StartTest('AtInsert', 'Inserting items at random...');
  Writeln(TestWindow^.T);
  Assign(F, 'items.dat');
  Reset(F);
  Randomize;
  for i := 1 to 20 do
  begin
    Readln(F, Key);
    if UseNonDynamicTestRec
      then begin
             CreateNonDynamicTestRec(Key, 0, NonDynamicRec);
             Item := @NonDynamicRec;
           end { if }
      else if UseNonDynamicTestObject
             then begin
                    CreateNonDynamicTestObject(Key+' ', 0, NonDynamicObject);
                    Item := @NonDynamicObject;
                  end { if }
      else if UseNonDynamicTestStaticObject
             then begin
                    CreateNonDynamicTestStaticObject(Key+' ', 0,
                      NonDynamicStaticObject);
                    Item := @NonDynamicStaticObject;
                  end { if }
      else Item := CreateItem(Key, 0);
    Index := Random(Pred(Sequence^.Count));
    Str(Index, StrIndex);
    WriteSubHeader('Inserting '+Key+' at index '+StrIndex);
    SetInitTime;
    Sequence^.AtInsert(Index, Item);
    SetFinalTime;
    WriteTime;
  end; { for }
  Close(F);
  NotifyDataChange;
  PauseTest;
end;

{****************************************************************************}
{ TestSequenceAtPut                                                          }
{****************************************************************************}
procedure TestSequenceAtPut (Sequence : PSequence);
var
  New, Item : Pointer;
  i : Integer;
  Key : String5;
  Index : LongInt;
  StrIndex : string;
  F : Text;
begin
  if ExitTesting
    then Exit;
  StartTest('AtPut', 'Replacing items at random...');
  Writeln(TestWindow^.T);
  Assign(F, 'items.dat');
  Reset(F);
  Randomize;
  for i := 1 to 20 do
  begin
    Readln(F, Key);
    if UseNonDynamicTestRec
      then begin
             CreateNonDynamicTestRec(Key, 0, NonDynamicRec);
             New := @NonDynamicRec;
           end { if }
      else if UseNonDynamicTestObject
             then begin
                    CreateNonDynamicTestObject(Key+' ', 0, NonDynamicObject);
                    New := @NonDynamicObject;
                  end { if }
      else if UseNonDynamicTestStaticObject
             then begin
                    CreateNonDynamicTestStaticObject(Key+' ', 0,
                      NonDynamicStaticObject);
                    Item := @NonDynamicStaticObject;
                  end { if }
      else New := CreateItem(Key, 0);
    Item := nil;
    repeat
      Sequence^.DoneItem(Item);
      Index := Random(Pred(Sequence^.Count));
      Item := Sequence^.At(Index)
    until (Item <> nil) and (TestReader^.ExtractText(Item) <> nil);
    Str(Index, StrIndex);
    WriteSubHeader('Replacing '+TestReader^.ExtractText(Item)^+' at index '
      +StrIndex+' with '+Key);
    if TestingMemArray
      then Sequence^.FreeItem(Item);
    SetInitTime;
    Sequence^.AtPut(Index, New);
    SetFinalTime;
    if not TestingMemArray
      then Sequence^.FreeItem(Item);
    WriteTime;
  end; { for }
  Close(F);
  NotifyDataChange;
  PauseTest;
end;

{****************************************************************************}
{ TestSequenceDelete                                                         }
{****************************************************************************}
procedure TestSequenceDelete (Sequence: PSequence);
var
  Count : Byte;
  Index : LongInt;
  Item : Pointer;

  function Match(Item : Pointer) : Boolean; far;
  begin
    if (Item <> nil) and (TestReader^.ExtractText(Item) <> nil)
      then if TestReader^.ExtractText(Item)^[3] = 'X'
             then Match := True
             else Match := False
      else Match := False;
  end; { Match }

begin
  if ExitTesting
    then Exit;
  WriteHeader('Delete');
  WriteSubHeader('Deleting first 20 items with ''X'' as the 3rd '+
   'character...');
  Writeln(TestWindow^.T);
  Item := Sequence^.FirstThat(@Match, Index);
  Count := 1;
  while (Item <> nil) and (Count <= 20) do
  begin
    WriteSubHeader('Deleting '+TestReader^.ExtractText(Item)^);
    if TestingMemArray
      then Sequence^.FreeItem(Item);
    SetInitTime;
    Sequence^.Delete(Item);
    SetFinalTime;
    WriteTime;
    if not TestingMemArray
      then Sequence^.FreeItem(Item);
    Item := Sequence^.NextThat(@Match, Index);
    Inc(Count);
  end; { while }
  Sequence^.DoneItem(Item);
  WriteNumResult('Total items deleted:', Pred(Count));
  NotifyDataChange;
  PauseTest;
end;

{****************************************************************************}
{ TestSequenceDeleteAll                                                      }
{****************************************************************************}
procedure TestSequenceDeleteAll (Sequence : PSequence);
var
  Items : array [1..TotalDeleteItems] of Pointer;
  i : Integer;
  SequenceCount : LongInt;
begin
  if ExitTesting
    then Exit;
  SequenceCount := Sequence^.Count;
  if SequenceCount > TotalDeleteItems
    then SequenceCount := TotalDeleteItems;
  for i := 1 to SequenceCount do
    Items[i] := Sequence^.At(Pred(i));
  if TestingMemArray
    then for i := 1 to SequenceCount do
           Sequence^.FreeItem(Items[i]);
  StartTest('DeleteAll', 'Deleting all items in the container...');
  Sequence^.DeleteAll;
  StopTest;
  WriteSubHeader('Disposing of deleted items...');
  if not TestingMemArray
    then for i := 1 to SequenceCount do
           Sequence^.FreeItem(Items[i]);
  Writeln(TestWindow^.T, '  done.');
  NotifyDataChange;
  PauseTest;
end;

{****************************************************************************}
{ TestSequenceFree                                                           }
{****************************************************************************}
procedure TestSequenceFree (Sequence: PSequence);
var
  Count : Byte;
  Index : LongInt;
  Item : Pointer;

  function Match(Item : Pointer) : Boolean; far;
  begin
    if (Item <> nil) and (TestReader^.ExtractText(Item) <> nil)
      then if TestReader^.ExtractText(Item)^[3] = 'W'
             then Match := True
             else Match := False
      else Match := False;
  end; { Match }

begin
  if ExitTesting
    then Exit;
  WriteHeader('Free');
  WriteSubHeader('Freeing first 20 items with ''W'' as the 3rd '+
    'character...');
  Writeln(TestWindow^.T);
  Item := Sequence^.FirstThat(@Match, Index);
  Count := 1;
  while (Item <> nil) and (Count <= 20) do
  begin
    WriteSubHeader('Freeing '+TestReader^.ExtractText(Item)^);
    SetInitTime;
    Sequence^.Free(Item);
    SetFinalTime;
    WriteTime;
    Item := Sequence^.NextThat(@Match, Index);
    Inc(Count);
  end; { while }
  Sequence^.DoneItem(Item);
  WriteNumResult('Total items freed:', Pred(Count));
  NotifyDataChange;
  PauseTest;
end;

{****************************************************************************}
{ TestSequenceFirst                                                          }
{****************************************************************************}
procedure TestSequenceFirst (Sequence : PSequence);
var
  Index : LongInt;
  Item : Pointer;
begin
  if ExitTesting
    then Exit;
  StartTest('First', 'Retrieving the first item in the container...');
  Item := Sequence^.First(Index);
  StopTest;
  if Item <> nil
    then TestReader^.ShowItem(Item)
    else WriteResult('nil/deleted');
  Sequence^.DoneItem(Item);
  PauseTest;
end;

{****************************************************************************}
{ TestSequenceFirstThat                                                      }
{****************************************************************************}
procedure TestSequenceFirstThat (Sequence : PSequence);
var
  Item : Pointer;
  Index : LongInt;

  function Match(Item : Pointer) : Boolean; far;
  begin
    if (Item <> nil) and (TestReader^.ExtractText(Item) <> nil)
      then if TestReader^.ExtractText(Item)^ > 'UXVT'
             then Match := True
             else Match := False
      else Match := False;
  end; { Match }

begin
  if ExitTesting
    then Exit;
  StartTest('FirstThat', 'Retrieving first item with key > ''UXVT''');
  Item := Sequence^.FirstThat(@Match, Index);
  StopTest;
  if Item <> nil
    then TestReader^.ShowItem(Item)
    else WriteResult('Not found');
  Sequence^.DoneItem(Item);
  PauseTest;
end;

{****************************************************************************}
{ TestSequenceLast                                                           }
{****************************************************************************}
procedure TestSequenceLast (Sequence : PSequence);
var
  Index : LongInt;
  Item : Pointer;
begin
  if ExitTesting
    then Exit;
  StartTest('Last', 'Retrieving the last item in the container...');
  Item := Sequence^.Last(Index);
  StopTest;
  if Item <> nil
    then TestReader^.ShowItem(Item)
    else WriteResult('nil/deleted');
  Sequence^.DoneItem(Item);
  PauseTest;
end;

{****************************************************************************}
{ TestSequenceLastThat                                                       }
{****************************************************************************}
procedure TestSequenceLastThat (Sequence : PSequence);
var
  Item : Pointer;
  Index : LongInt;

  function Match(Item : Pointer) : Boolean; far;
  begin
    if (Item <> nil) and (TestReader^.ExtractText(Item) <> nil)
      then if TestReader^.ExtractText(Item)^ < 'DRTG'
             then Match := True
             else Match := False
      else Match := False;
  end; { Match }

begin
  if ExitTesting
    then Exit;
  StartTest('LastThat', 'Retrieving last item with key < ''DRTG''');
  Item := Sequence^.LastThat(@Match, Index);
  StopTest;
  if Item <> nil
    then TestReader^.ShowItem(Item)
    else WriteResult('Not found');
  Sequence^.DoneItem(Item);
  PauseTest;
end;

{****************************************************************************}
{ TestSequenceNext                                                           }
{****************************************************************************}
procedure TestSequenceNext (Sequence : PSequence);
var
  Index, Counter : LongInt;
  Item : Pointer;
begin
  if ExitTesting
    then Exit;
  StartTest('Next', 'Traversing the container using First and Next '+
    'methods...');
  if Sequence^.Status > ctOk
   then Sequence^.Status := ctOk;
  Counter := 0;
  Item := Sequence^.First(Index);
  while Sequence^.Status = ctOk do
  begin
    Sequence^.DoneItem(Item);
    Inc(Counter);
    Item := Sequence^.Next(Index);
  end; { while }
  StopTest;
  WriteNumResult('Total nodes visited:', Counter);
  PauseTest;
end;

{****************************************************************************}
{ TestSequenceNextThat                                                       }
{****************************************************************************}
procedure TestSequenceNextThat (Sequence : PSequence);
var
  Item : Pointer;
  Index : LongInt;

  function MatchFirst(Item : Pointer) : Boolean; far;
  begin
    if (Item <> nil) and (TestReader^.ExtractText(Item) <> nil)
      then if TestReader^.ExtractText(Item)^ > 'UXVT'
             then MatchFirst := True
             else MatchFirst := False
      else MatchFirst := False;
  end; { MatchFirst }

  function MatchNext(Item : Pointer) : Boolean; far;
  begin
    if (Item <> nil) and (TestReader^.ExtractText(Item) <> nil)
     then if TestReader^.ExtractText(Item)^[3] = 'Q'
            then MatchNext := True
            else MatchNext := False
     else MatchNext := False;
  end; { MatchNext }

begin
  if ExitTesting
    then Exit;
  Item := Sequence^.FirstThat(@MatchFirst, Index);
  Sequence^.DoneItem(Item);
  StartTest('NextThat', 'Retrieving next item with ''Q'' as the 3rd '+
    'character after first item with key > ''UXVT''');
  Item := Sequence^.NextThat(@MatchNext, Index);
  StopTest;
  if Item <> nil
    then TestReader^.ShowItem(Item)
    else WriteResult('Not found');
  Sequence^.DoneItem(Item);
  PauseTest;
end;

{****************************************************************************}
{ TestSequencePrev                                                           }
{****************************************************************************}
procedure TestSequencePrev (Sequence : PSequence);
var
  Index, Counter : LongInt;
  Item : Pointer;
begin
  if ExitTesting
    then Exit;
  StartTest('Prev', 'Traversing the container using Last and Prev '+
    'methods...');
  if Sequence^.Status > ctOk
   then Sequence^.Status := ctOk;
  Counter := 0;
  Item := Sequence^.Last(Index);
  while Sequence^.Status = ctOk do
  begin
    Sequence^.DoneItem(Item);
    Inc(Counter);
    Item := Sequence^.Prev(Index);
  end; { while }
  StopTest;
  WriteNumResult('Total nodes visited:', Counter);
  PauseTest;
end;

{****************************************************************************}
{ TestSequencePrevThat                                                       }
{****************************************************************************}
procedure TestSequencePrevThat (Sequence : PSequence);
var
  Item : Pointer;
  Index : LongInt;

  function MatchFirst(Item : Pointer) : Boolean; far;
  begin
    if (Item <> nil) and (TestReader^.ExtractText(Item) <> nil)
      then if TestReader^.ExtractText(Item)^ < 'DRTG'
             then MatchFirst := True
             else MatchFirst := False
      else MatchFirst := False;
  end; { MatchFirst }

  function MatchNext(Item : Pointer) : Boolean; far;
  begin
    if (Item <> nil) and (TestReader^.ExtractText(Item) <> nil)
     then if TestReader^.ExtractText(Item)^[3] = 'F'
            then MatchNext := True
            else MatchNext := False
     else MatchNext := False;
  end; { MatchNext }

begin
  if ExitTesting
    then Exit;
  Item := Sequence^.LastThat(@MatchFirst, Index);
  Sequence^.DoneItem(Item);
  StartTest('PrevThat', 'Retrieving first item with ''F'' as the 3rd '+
    'character before last item with key < ''DRTG''');
  Item := Sequence^.PrevThat(@MatchNext, Index);
  StopTest;
  if Item <> nil
    then TestReader^.ShowItem(Item)
    else WriteResult('Not found');
  Sequence^.DoneItem(Item);
  PauseTest;
end;

{****************************************************************************}
{ TestSequenceSearch                                                         }
{****************************************************************************}
procedure TestSequenceSearch (Sequence : PSequence);
var
  F : Text;
  i : Integer;
  Key : String5;
  Index : LongInt;
begin
  if ExitTesting
    then Exit;
  Assign(F, 'items.dat');
  Reset(F);
  for i := 1 to 20 do
  begin
    Readln(F, Key);
    WriteSubHeader('Searching for '+key);
    SetInitTime;
    Sequence^.Search(@Key, Index);
    Writeln(TestWindow^.T, Index:11);
  end; { for }
  Close(F);
  PauseTest;
end;

{****************************************************************************}
{ TestStaticSequenceAtInsert                                                 }
{****************************************************************************}
procedure TestStaticSequenceAtInsert (Sequence : PSequence);
var
  Item : Pointer;
  i : Integer;
  Key : String5;
  Index : LongInt;
  StrIndex : string;
  F : Text;
begin
  if ExitTesting
    then Exit;
  StartTest('AtInsert', 'Inserting items at random...');
  Writeln(TestWindow^.T);
  Assign(F, 'items.dat');
  Reset(F);
  Randomize;
  for i := 1 to 20 do
  begin
    Readln(F, Key);
    if UseNonDynamicTestRec
      then begin
             CreateNonDynamicTestRec(Key, 0, NonDynamicRec);
             Item := @NonDynamicRec;
           end { if }
      else if UseNonDynamicTestObject
             then begin
                    CreateNonDynamicTestObject(Key+' ', 0, NonDynamicObject);
                    Item := @NonDynamicObject;
                  end { if }
      else if UseNonDynamicTestStaticObject
             then begin
                    CreateNonDynamicTestStaticObject(Key+' ', 0,
                      NonDynamicStaticObject);
                    Item := @NonDynamicStaticObject;
                  end { if }
      else Item := CreateItem(Key, 0);
    Index := Random(Pred(Sequence^.Count));
    Str(Index, StrIndex);
    WriteSubHeader('Inserting '+Key+' at index '+StrIndex);
    Sequence^.AtFree(Sequence^.LastIndex);
    SetInitTime;
    Sequence^.AtInsert(Index, Item);
    SetFinalTime;
    WriteTime;
  end; { for }
  Close(F);
  NotifyDataChange;
  PauseTest;
end;

{****************************************************************************}
{ TestStaticSequenceInsert                                                   }
{****************************************************************************}
procedure TestStaticSequenceInsert (Sequence: PSequence; TotalItems:
  LongInt);
var
  SubHeader : string;
  F : Text;
  i : LongInt;
  Key : String5;
  Item : Pointer;
begin
  if ExitTesting
    then Exit;
  Assign(F, 'items.dat');
  Reset(F);
  FormatStr(SubHeader, 'Inserting %d items into the container...',
    TotalItems);
  StartTest('AtPut', SubHeader);
  for i := 0 to Pred(TotalItems) do
  begin
    Readln(F, Key);
    if LowMemory
      then begin
             Writeln(TestWindow^.T);
             Writeln(TestWindow^.T);
             Writeln(TestWindow^.T, 'Not enough memory... aborting test.');
             ExitTesting := True;
             Close(F);
             ResetApplication;
             Exit;
           end { if }
      else if UseNonDynamicTestRec
             then begin
                    CreateNonDynamicTestRec(Key+' ', 0, NonDynamicRec);
                    Item := @NonDynamicRec;
                  end { if }
      else if UseNonDynamicTestObject
             then begin
                    CreateNonDynamicTestObject(Key+' ', 0, NonDynamicObject);
                    Item := @NonDynamicObject;
                  end { if }
      else if UseNonDynamicTestStaticObject
             then begin
                    CreateNonDynamicTestStaticObject(Key+' ', 0,
                      NonDynamicStaticObject);
                    Item := @NonDynamicStaticObject;
                  end { if }
      else Item := CreateItem(Key+' ', 0);
    Sequence^.AtPut(i, Item);
  end; { for }
  StopTest;
  Close(F);
  NotifyDataChange;
  PauseTest;
end;

{****************************************************************************}
{ TestStreamStackBottom                                                      }
{****************************************************************************}
procedure TestStreamStackBottom (Stack : PStreamStack);
var
  Item : Pointer;
begin
  if ExitTesting
    then Exit;
  StartTest('Bottom', 'Getting item at the bottom of the stack...');
  Item := Stack^.Bottom;
  StopTest;
  TestReader^.ShowItem(Item);
  PauseTest;
end;

{****************************************************************************}
{ TestStreamStackPop                                                         }
{****************************************************************************}
procedure TestStreamStackPop (Stack : PStreamStack);
var
  Item : Pointer;
  i : Integer;
begin
  if ExitTesting
    then Exit;
  StartTest('Pop', 'Getting all items out of the stack...');
  Writeln(TestWindow^.T);
  for i := 1 to Stack^.Count do
  begin
    SetInitTime;
    Item := Stack^.Pop;
    SetFinalTime;
    WriteSubHeader('Popping '+TestReader^.ExtractText(Item)^);
    WriteTime;
    Stack^.FreeItem(Item);
  end; { for }
  PauseTest;
end;

{****************************************************************************}
{ TestStreamStackPush                                                        }
{****************************************************************************}
procedure TestStreamStackPush (Stack : PStreamStack);
var
  i : Integer;
  Item : Pointer;
  Key : String5;
  F : Text;
begin
  if ExitTesting
    then Exit;
  StartTest('Push', 'Adding 20 items to the stack...');
  Writeln(TestWindow^.T);
  Assign(F, 'Items.Dat');
  Reset(F);
  for i := 1 to 20 do
  begin
    Readln(F, Key);
    if LowMemory
      then begin
             Writeln(TestWindow^.T);
             Writeln(TestWindow^.T);
             Writeln(TestWindow^.T, 'Not enough memory... aborting test.');
             ExitTesting := True;
             Close(F);
             ResetApplication;
             Exit;
           end { if }
      else Item := CreateItem(Key, 0);
    SetInitTime;
    Stack^.Push(Item);
    SetFinalTime;
    WriteSubHeader('Pushing '+Key);
    WriteTime;
  end; { for }
  PauseTest;
end;

{****************************************************************************}
{ TestStreamStackTop                                                         }
{****************************************************************************}
procedure TestStreamStackTop (Stack : PStreamStack);
var
  Item : Pointer;
begin
  if ExitTesting
    then Exit;
  StartTest('Top', 'Getting item at the top of the stack...');
  Item := Stack^.Top;
  StopTest;
  TestReader^.ShowItem(Item);
  PauseTest;
end;

{****************************************************************************}
{ TestTreeTraverse                                                           }
{****************************************************************************}
procedure TestTreeTraverse(Tree: PBinaryTree);
var
  Counter : Integer;

  procedure DisplayItem(Item : Pointer); far;
  begin
    Inc(Counter);
    if Counter <= 20
      then if Item <> nil
             then TestReader^.ShowItem(Item);
  end; { DisplayItem }

begin
  if ExitTesting
    then Exit;
  Counter := 0;
  StartTest('Traverse', 'Traversing items in order...');
  Writeln(TestWindow^.T);
  WriteSubHeader('(displaying first 20 items)');
  Writeln(TestWindow^.T);
  Tree^.Traverse(@DisplayItem, InOrder);
  WriteSubHeader('done...');
  StopTest;
  PauseTest;

  if ExitTesting
    then Exit;
  Counter := 0;
  StartTest('Traverse', 'Traversing items in pre-order...');
  Writeln(TestWindow^.T);
  WriteSubHeader('(displaying first 20 items)');
  Writeln(TestWindow^.T);
  Tree^.Traverse(@DisplayItem, PreOrder);
  WriteSubHeader('done...');
  StopTest;
  PauseTest;

  if ExitTesting
    then Exit;
  Counter := 0;
  StartTest('Traverse', 'Traversing items in post-order...');
  Writeln(TestWindow^.T);
  WriteSubHeader('(displaying first 20 items)');
  Writeln(TestWindow^.T);
  Tree^.Traverse(@DisplayItem, PostOrder);
  WriteSubHeader('done...');
  StopTest;
  PauseTest;
end;

{****************************************************************************}
{ TestTreeTraverseThat                                                       }
{****************************************************************************}
procedure TestTreeTraverseThat(Tree: PBinaryTree);
var
  Counter : Integer;

  procedure DisplayItem(Item : Pointer); far;
  begin
    Inc(Counter);
    if Counter <= 20
      then if Item <> nil
             then TestReader^.ShowItem(Item);
  end; { DisplayItem }

  function Match(Item : Pointer): Boolean; far;
  begin
    if (Item <> nil) and (TestReader^.ExtractText(Item)^[3] = 'Q')
      then Match := True
      else Match := False;
  end; { Match }

begin
  if ExitTesting
    then Exit;
  Counter := 0;
  StartTest('TraverseThat', 'Displaying in-order items with 3rd char=Q...');
  Writeln(TestWindow^.T);
  WriteSubHeader('(displaying first 20 items)');
  Writeln(TestWindow^.T);
  Tree^.TraverseThat(@Match, @DisplayItem, InOrder);
  WriteSubHeader('done...');
  StopTest;
  PauseTest;

  if ExitTesting
    then Exit;
  Counter := 0;
  StartTest('TraverseThat', 'Displaying in pre-order items with 3rd char=Q...');
  Writeln(TestWindow^.T);
  WriteSubHeader('(displaying first 20 items)');
  Writeln(TestWindow^.T);
  Tree^.TraverseThat(@Match, @DisplayItem, PreOrder);
  WriteSubHeader('done...');
  StopTest;
  PauseTest;

  if ExitTesting
    then Exit;
  Counter := 0;
  StartTest('TraverseThat', 'Displaying in post-order items with 3rd char=Q...');
  Writeln(TestWindow^.T);
  WriteSubHeader('(displaying first 20 items)');
  Writeln(TestWindow^.T);
  Tree^.TraverseThat(@Match, @DisplayItem, PostOrder);
  WriteSubHeader('done...');
  StopTest;
  PauseTest;
end;

begin
  LowMemSize := 5120 div 16;
end.