{ ----------------------------------------------------------------------------}
{ Data aware components for Delphi.                                           }
{ Copyright 1995, Curtis White.  All Rights Reserved.                         }
{ These components can be freely used and distributed in commercial and       }
{ private environments, provied this notice is not modified in any way.       }
{ ----------------------------------------------------------------------------}
{ Feel free to contact me if you have any questions, comments or suggestions  }
{ at cwhite@teleport.com                                                      }
{ ----------------------------------------------------------------------------}
{ Date last modified:  08/12/95                                               }
{ ----------------------------------------------------------------------------}
{ ----------------------------------------------------------------------------}
{ TDBButton v1.00                                                             }
{ ----------------------------------------------------------------------------}
{ Description:                                                                }
{   A data aware button.                                                      }
{ Features:                                                                   }
{   Same as a normal button except the caption can be set by a database field.}
{ ----------------------------------------------------------------------------}
{ ----------------------------------------------------------------------------}
{ Revision History:                                                           }
{ 1.00:  Initial release                                                      }
{ ----------------------------------------------------------------------------}

{ ----------------------------------------------------------------------------}
{ TDBBitBtn v1.00                                                             }
{ ----------------------------------------------------------------------------}
{ Description:                                                                }
{   A data aware bitmap button.                                               }
{ Features:                                                                   }
{   Same as a normal bitmap button except the caption can be set by a database}
{   field.}
{ ----------------------------------------------------------------------------}
{ ----------------------------------------------------------------------------}
{ Revision History:                                                           }
{ 1.00:  Initial release                                                      }
{ ----------------------------------------------------------------------------}

{ ----------------------------------------------------------------------------}
{ TDBRadioButton v1.00                                                        }
{ ----------------------------------------------------------------------------}
{ Description:                                                                }
{   A data aware radio button.                                                }
{ Features:                                                                   }
{   Same as a normal radio button except the caption can be set by a database }
{   field.}
{ ----------------------------------------------------------------------------}
{ ----------------------------------------------------------------------------}
{ Revision History:                                                           }
{ 1.00:  Initial release                                                      }
{ ----------------------------------------------------------------------------}

{ ----------------------------------------------------------------------------}
{ TDBLabel3D v1.00                                                            }
{ ----------------------------------------------------------------------------}
{ Description:                                                                }
{   A data aware 3-D label.                                                   }
{ Features:                                                                   }
{   The label style can be set to Raised, Lowered, or None.                   }
{   The label caption can be set by a database field.                         }
{ ----------------------------------------------------------------------------}
{ ----------------------------------------------------------------------------}
{ Revision History:                                                           }
{ 1.00:  Initial release                                                      }
{ ----------------------------------------------------------------------------}

{ ----------------------------------------------------------------------------}
{ TDBMaskEdit v1.02                                                           }
{ ----------------------------------------------------------------------------}
{ Description:                                                                }
{   A data aware masked edit box.                                             }
{ Features:                                                                   }
{   Same as a normal masked edit box except box can be filled by a database   }
{     field.  It only masks data being input into the database.  Data from    }
{     the database is displayed exactly as it is stored on the database.      }
{ ----------------------------------------------------------------------------}
{ ----------------------------------------------------------------------------}
{ Revision History:                                                           }
{ 1.00:  Initial release                                                      }
{ 1.01:  Fixed bug to allow mask to be used in DB set.  Created constructor   }
{        that sets the MaskState to msDBSetText.  Also enabled the EditText   }
{        property.  This will allow a masked edit to be posted to a database  }
{        field.                                                               }
{ 1.02   Had to completely change the code to derive from TCustomMaskEdit     }
{        instead of TDBEdit.  TDBEdit had problems where it forced the Mask   }
{        to Nil if the TField mask was Nil.  It now sets the mask to match    }
{        the TField mask only if the TField mask is not Nil.  If it is Nil    }
{        then the TDBMaskEdit uses it's own mask.  The database fields are    }
{        displayed exactly as they appear on the database.  The TDBMaskEdit   }
{        only limits data being input to whatever the mask is set to.  It     }
{        does not limit the data coming from the database to the mask.        }
{ ----------------------------------------------------------------------------}

unit Dabtns;

interface

uses
  SysUtils, WinTypes, WinProcs, Messages, Classes, Graphics, Controls, Menus,
  Forms, Dialogs, StdCtrls, ExtCtrls, DB, DBCtrls, DBTables, Buttons, Mask,
  DsgnIntf, DABtnAbt, DABBtAbt, DARadAbt, DA3DLAbt, DAMEAbt;

type
  { TDBButton }
  { Data Aware Button }
  TDBButton = class(TButton)
  private
    { Variables for properties }
    FDataLink: TFieldDataLink;
    FButtonAbout: TDBBtnAboutBox;

    { Procedures for setting property values }
    function GetDataField: string;
    function GetDataSource: TDataSource;
    procedure SetDataField(const Value: string);
    procedure SetDataSource(Value: TDataSource);
    procedure DataChange(Sender: TObject);
  protected
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
  published
    { AboutBox }
    property About: TDBBtnAboutBox read FButtonAbout write FButtonAbout;
    { Field in the database table }
    property DataField: string read GetDataField write SetDataField;
    { Datasource attached to the database table }
    property DataSource: TDataSource read GetDataSource write SetDataSource;
  end;


  { TDBBitBtn }
  { Data Aware BitBtn }
  TDBBitBtn = class(TBitBtn)
  private
    { Variables for properties }
    FDataLink: TFieldDataLink;
    FBitBtnAbout: TDBBitBtnAboutBox;

    { Procedures for setting property values }
    function GetDataField: string;
    function GetDataSource: TDataSource;
    procedure SetDataField(const Value: string);
    procedure SetDataSource(Value: TDataSource);
    procedure DataChange(Sender: TObject);
  protected
  public
    { Public declarations }
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
  published
    { AboutBox }
    property About: TDBBitBtnAboutBox read FBitBtnAbout write FBitBtnAbout;
    { Field in the database table }
    property DataField: string read GetDataField write SetDataField;
    { Datasource attached to the database table }
    property DataSource: TDataSource read GetDataSource write SetDataSource;
  end;


  { TDBRadioButton }
  { Data Aware Radio Button }
  TDBRadioButton = class(TRadioButton)
  private
    { Variables for properties }
    FDataLink: TFieldDataLink;
    FRadioButtonAbout: TDBRadBtnAboutBox;

    { Procedures for setting property values }
    function GetDataField: string;
    function GetDataSource: TDataSource;
    procedure SetDataField(const Value: string);
    procedure SetDataSource(Value: TDataSource);
    procedure DataChange(Sender: TObject);
  protected
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
  published
    { AboutBox }
    property About: TDBRadBtnAboutBox read FRadioButtonAbout write FRadioButtonAbout;
    { Field in the database table }
    property DataField: string read GetDataField write SetDataField;
    { Datasource attached to the database table }
    property DataSource: TDataSource read GetDataSource write SetDataSource;
  end;


  { TDBLabel3D }
  { Data Aware 3-D Label }
  { Style of 3-D label }
  TTextStyle = ( tsNone, tsRaised, tsLowered );

  TDBLabel3D = class(TLabel)
  private
    { Variables for properties }
    FTextStyle: TTextStyle;
    FDataLink: TFieldDataLink;
    F3DLabelAbout: TDB3DLabAboutBox;

    { Procedures for setting property values }
    function GetDataField: string;
    function GetDataSource: TDataSource;
    procedure SetDataField(const Value: string);
    procedure SetDataSource(Value: TDataSource);
    procedure DataChange(Sender: TObject);

    { Procedure to draw the 3-D label }
    procedure DrawLabel3D(var TextRect: TRect; SpecialFlags: Word);
  protected
    procedure Paint; override;
    { Procedure to set the text style }
    procedure SetTextStyle( Value : TTextStyle );
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
  published
    { AboutBox }
    property About: TDB3DLabAboutBox read F3DLabelAbout write F3DLabelAbout;
    { Style of the 3-D label }
    property TextStyle: TTextStyle read FTextStyle write SetTextStyle
        default tsLowered;
    { Field in the database table }
    property DataField: string read GetDataField write SetDataField;
    { Datasource attached to the database table }
    property DataSource: TDataSource read GetDataSource write SetDataSource;
  end;

{--------------------------------------------------------------------}

  { TDBMaskEdit }
  { Data Aware Mask Edit }
  TDBMaskEdit = class(TCustomMaskEdit)
  private
    FMaskEditAbout: TDBMaskEdAboutBox;
    FDataLink: TFieldDataLink;
    FCanvas: TControlCanvas;
    FAlignment: TAlignment;
    FFocused: Boolean;
    FTextMargin: Integer;
    procedure SetDataField(const Value: string);
    procedure SetDataSource(Value: TDataSource);
    procedure SetFocused(Value: Boolean);
    procedure SetReadOnly(Value: Boolean);
    procedure UpdateData(Sender: TObject);
    procedure CalcTextMargin;
    procedure DataChange(Sender: TObject);
    procedure EditingChange(Sender: TObject);
    procedure WMCut(var Message: TMessage); message WM_CUT;
    procedure WMPaste(var Message: TMessage); message WM_PASTE;
    procedure CMEnter(var Message: TCMEnter); message CM_ENTER;
    procedure CMExit(var Message: TCMExit); message CM_EXIT;
    procedure WMPaint(var Message: TWMPaint); message WM_PAINT;
    procedure CMFontChanged(var Message: TMessage); message CM_FONTCHANGED;
    function GetDataField: string;
    function GetDataSource: TDataSource;
    function GetField: TField;
    function GetReadOnly: Boolean;
  protected
    procedure Change; override;
    procedure Reset; override;
    procedure Notification(AComponent: TComponent; Operation: TOperation); override;
    procedure KeyDown(var Key: Word; Shift: TShiftState); override;
    procedure KeyPress(var Key: Char); override;
    function EditCanModify: Boolean; override;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    property Field: TField read GetField;
  published
    { AboutBox }
    property About: TDBMaskEdAboutBox read FMaskEditAbout write FMaskEditAbout;
    property DataField: string read GetDataField write SetDataField;
    property DataSource: TDataSource read GetDataSource write SetDataSource;
    property ReadOnly: Boolean read GetReadOnly write SetReadOnly default False;
    property AutoSelect;
    property AutoSize;
    property BorderStyle;
    property CharCase;
    property Color;
    property Ctl3D;
    property DragCursor;
    property DragMode;
    { Publish EditMask and EditText properties.  EditText property does not }
    { need to be published if so desired.                                   }
    property EditMask;
    property EditText;
    property Enabled;
    property Font;
    property MaxLength;
    property ParentColor;
    property ParentCtl3D;
    property ParentFont;
    property ParentShowHint;
    property PasswordChar;
    property PopupMenu;
    property ShowHint;
    property TabOrder;
    property TabStop;
    { Publish Text property.  This property does not need to be published if }
    { so desired.                                                            }
    property Text;
    property Visible;
    property OnChange;
    property OnClick;
    property OnDblClick;
    property OnDragDrop;
    property OnDragOver;
    property OnEndDrag;
    property OnEnter;
    property OnExit;
    property OnKeyDown;
    property OnKeyPress;
    property OnKeyUp;
    property OnMouseDown;
    property OnMouseMove;
    property OnMouseUp;
  end;

{--------------------------------------------------------------------}

  { TDBButton Component AboutBox }
  TDBBtnCompAbout = class(TComponentEditor)
    procedure ExecuteVerb(Index: Integer); override;
    function GetVerb(Index: Integer): string; override;
    function GetVerbCount: Integer; override;
    procedure ShowAbout;
  end;

{--------------------------------------------------------------------}

  { TDBButton Property AboutBox }
  TDBBtnPropAbout = class(TClassProperty)
    procedure Edit; override;
    function GetAttributes: TPropertyAttributes; override;
    procedure ShowAbout;
  end;

{--------------------------------------------------------------------}

  { TDBBitBtn Component AboutBox }
  TDBBitBtnCompAbout = class(TComponentEditor)
    procedure ExecuteVerb(Index: Integer); override;
    function GetVerb(Index: Integer): string; override;
    function GetVerbCount: Integer; override;
    procedure ShowAbout;
  end;

{--------------------------------------------------------------------}

  { TDBBitBtn Property AboutBox }
  TDBBitBtnPropAbout = class(TClassProperty)
    procedure Edit; override;
    function GetAttributes: TPropertyAttributes; override;
    procedure ShowAbout;
  end;

{--------------------------------------------------------------------}

  { TDBRadioButton Component AboutBox }
  TDBRadBtnCompAbout = class(TComponentEditor)
    procedure ExecuteVerb(Index: Integer); override;
    function GetVerb(Index: Integer): string; override;
    function GetVerbCount: Integer; override;
    procedure ShowAbout;
  end;

{--------------------------------------------------------------------}

  { TDBRadioButton Property AboutBox }
  TDBRadBtnPropAbout = class(TClassProperty)
    procedure Edit; override;
    function GetAttributes: TPropertyAttributes; override;
    procedure ShowAbout;
  end;

{--------------------------------------------------------------------}

  { TDB3DLabel Component AboutBox }
  TDB3DLabCompAbout = class(TComponentEditor)
    procedure ExecuteVerb(Index: Integer); override;
    function GetVerb(Index: Integer): string; override;
    function GetVerbCount: Integer; override;
    procedure ShowAbout;
  end;

{--------------------------------------------------------------------}

  { TDB3DLabel Property AboutBox }
  TDB3DLabPropAbout = class(TClassProperty)
    procedure Edit; override;
    function GetAttributes: TPropertyAttributes; override;
    procedure ShowAbout;
  end;

{--------------------------------------------------------------------}

  { TDBMaskEdit Component AboutBox }
  TDBMaskEdCompAbout = class(TComponentEditor)
    procedure ExecuteVerb(Index: Integer); override;
    function GetVerb(Index: Integer): string; override;
    function GetVerbCount: Integer; override;
    procedure ShowAbout;
  end;

{--------------------------------------------------------------------}

  { TDBMaskEdit Property AboutBox }
  TDBMaskEdPropAbout = class(TClassProperty)
    procedure Edit; override;
    function GetAttributes: TPropertyAttributes; override;
    procedure ShowAbout;
  end;


procedure Register;

implementation

var
  DBBtnAboutBox: TDBBtnAboutBox;
  DBBitBtnAboutBox: TDBBitBtnAboutBox;
  DBRadBtnAboutBox: TDBRadBtnAboutBox;
  DB3DLabAboutBox: TDB3DLabAboutBox;
  DBMaskEdAboutBox: TDBMaskEdAboutBox;

{ TDBButton }
{ Data Aware Button }
{ Override the constructor to initialize variables }
constructor TDBButton.Create(AOwner: TComponent);
begin
  { Inherit original constructor }
  inherited Create(AOwner);
  { Add new initializations }
  FDataLink := TFieldDataLink.Create;
  FDataLink.OnDataChange := DataChange;
end;

{ Override the destructor to destroy variables }
destructor TDBButton.Destroy;
begin
  FDataLink.OnDataChange := nil;
  FDataLink.Free;
  inherited Destroy;
end;

{ Get database field }
function TDBButton.GetDataField: string;
begin
  Result := FDataLink.FieldName;
end;

{ Get datasource }
function TDBButton.GetDataSource: TDataSource;
begin
  Result := FDataLink.DataSource;
end;

{ Set database field }
procedure TDBButton.SetDataField(const Value: string);
begin
  FDataLink.FieldName := Value;
end;

{ Set datasource }
procedure TDBButton.SetDataSource(Value: TDataSource);
begin
  FDataLink.DataSource := Value;
end;

{ Set caption when data changes }
procedure TDBButton.DataChange(Sender: TObject);
begin
  if FDataLink.Field = nil then
    Caption := ''
  else
    Caption := FDataLink.Field.AsString;
end;

{ ----------------------------------------------------------------------------}

{ TDBBitBtn }
{ Data Aware BitBtn }
{ Override the constructor to initialize variables }
constructor TDBBitBtn.Create(AOwner: TComponent);
begin
  { Inherit original constructor }
  inherited Create(AOwner);
  { Add new initializations }
  FDataLink := TFieldDataLink.Create;
  FDataLink.OnDataChange := DataChange;
end;

{ Override the destructor to destroy variables }
destructor TDBBitBtn.Destroy;
begin
  FDataLink.OnDataChange := nil;
  FDataLink.Free;
  inherited Destroy;
end;

{ Get database field }
function TDBBitBtn.GetDataField: string;
begin
  Result := FDataLink.FieldName;
end;

{ Get datasource }
function TDBBitBtn.GetDataSource: TDataSource;
begin
  Result := FDataLink.DataSource;
end;

{ Set database field }
procedure TDBBitBtn.SetDataField(const Value: string);
begin
  FDataLink.FieldName := Value;
end;

{ Set datasource }
procedure TDBBitBtn.SetDataSource(Value: TDataSource);
begin
  FDataLink.DataSource := Value;
end;

{ Set caption when data changes }
procedure TDBBitBtn.DataChange(Sender: TObject);
begin
  if FDataLink.Field = nil then
    Caption := ''
  else
    Caption := FDataLink.Field.AsString;
end;

{ ----------------------------------------------------------------------------}

{ TDBRadioButton }
{ Data Aware Radio Button }
{ Override the constructor to initialize variables }
constructor TDBRadioButton.Create(AOwner: TComponent);
begin
  { Inherit original constructor }
  inherited Create(AOwner);
  { Add new initializations }
  FDataLink := TFieldDataLink.Create;
  FDataLink.OnDataChange := DataChange;
end;

{ Override the destructor to destroy variables }
destructor TDBRadioButton.Destroy;
begin
  FDataLink.OnDataChange := nil;
  FDataLink.Free;
  inherited Destroy;
end;

{ Get database field }
function TDBRadioButton.GetDataField: string;
begin
  Result := FDataLink.FieldName;
end;

{ Get datasource }
function TDBRadioButton.GetDataSource: TDataSource;
begin
  Result := FDataLink.DataSource;
end;

{ Set database field }
procedure TDBRadioButton.SetDataField(const Value: string);
begin
  FDataLink.FieldName := Value;
end;

{ Set datasource }
procedure TDBRadioButton.SetDataSource(Value: TDataSource);
begin
  FDataLink.DataSource := Value;
end;

{ Set caption when data changes }
procedure TDBRadioButton.DataChange(Sender: TObject);
begin
  if FDataLink.Field = nil then
    Caption := ''
  else
    Caption := FDataLink.Field.AsString;
end;

{ ----------------------------------------------------------------------------}

{ TDBLabel3D }
{ Data Aware 3D Label }
{ Override the constructor to initialize variables }
constructor TDBLabel3D.Create(AOwner: TComponent);
begin
  { Inherit original constructor }
  inherited Create(AOwner);
  { Add new initializations }
  FTextStyle := tsLowered;
  FDataLink := TFieldDataLink.Create;
  FDataLink.OnDataChange := DataChange;
end;

{ Override the destructor to destroy variables }
destructor TDBLabel3D.Destroy;
begin
  FDataLink.OnDataChange := nil;
  FDataLink.Free;
  inherited Destroy;
end;

{ Set the 3-D style of the label }
procedure TDBLabel3D.SetTextStyle( Value : TTextStyle );
begin
  if Value <> FTextStyle then
  begin
    FTextStyle := Value;
    Invalidate;
  end;
end;

{ Draw the text and the 3-D bevel }
procedure TDBLabel3D.DrawLabel3D( var TextRect : TRect; SpecialFlags : Word );
var
  Text        : array[ 0..255 ] of Char;
  WorkRect    : TRect;
  TopColor    : TColor;
  BottomColor : TColor;
begin
  { Copy the text to a buffer }
  GetTextBuf(Text, SizeOf(Text));

  { Process special characters }
  if (SpecialFlags and DT_CALCRECT <> 0) and
     ((Text[0] = #0) or ShowAccelChar and
      (Text[0] = '&') and
      (Text[1] = #0)) then
    StrCopy(Text, ' ');

  if not ShowAccelChar then
    SpecialFlags := SpecialFlags or DT_NOPREFIX;

  { Set the font }
  Canvas.Font := Font;

  { Set highlight and shadow colors }
  TopColor := clBtnHighlight;
  BottomColor := clBtnShadow;

  { If the text style is lowered then reverse the highlight and shadow colors }
  if FTextStyle = tsLowered then
  begin
    TopColor := clBtnShadow;
    BottomColor := clBtnHighlight;
  end;

  { If the text style is lowered or raised then draw it with its bevel }
  if FTextStyle in [ tsLowered, tsRaised ] then
  begin
    { Draw the bottom color }
    { Set the working rectangle coordinates }
    WorkRect := TextRect;
    { Offset it by 1 pixel }
    OffsetRect( WorkRect, 1, 1 );
    { Set color to the bottom color }
    Canvas.Font.Color := BottomColor;
    { Draw the text }
    DrawText(Canvas.Handle, Text, StrLen(Text), WorkRect, SpecialFlags);

    { Draw the top color }
    { Set the working rectangle coordinates }
    WorkRect := TextRect;
    { Offset it by -1 pixel }
    OffsetRect( WorkRect, -1, -1 );
    { Set color to the top color }
    Canvas.Font.Color := TopColor;
    { Draw the text }
    DrawText(Canvas.Handle, Text, StrLen(Text), WorkRect, SpecialFlags);
  end;

  { Set the actual font color }
  Canvas.Font.Color := Font.Color;
  { If it is disabled then set the font color to gray }
  if not Enabled then
    Canvas.Font.Color := clGrayText;
  { Draw the text }
  DrawText(Canvas.Handle, Text, StrLen(Text), TextRect, SpecialFlags);
end;

{ Paint 3-D Label }
procedure TDBLabel3D.Paint;
const
  Alignments: array[TAlignment] of Word = (DT_LEFT, DT_RIGHT, DT_CENTER);
var
  Rect: TRect;
begin
  with Canvas do
  begin
    { If transparent property is off then draw background }
    if not Transparent then
    begin
      { Set brush color and style then paint the background }
      Brush.Color := Self.Color;
      Brush.Style := bsSolid;
      FillRect(ClientRect);
    end;

    { Set the brush style and text rectangle coordinates then paint the text }
    Brush.Style := bsClear;
    Rect := ClientRect;
    DrawLabel3D(Rect, (DT_EXPANDTABS or DT_WORDBREAK ) or
                Alignments[Alignment]);
  end;
end;

{ Get database field }
function TDBLabel3D.GetDataField: string;
begin
  Result := FDataLink.FieldName;
end;

{ Get datasource }
function TDBLabel3D.GetDataSource: TDataSource;
begin
  Result := FDataLink.DataSource;
end;

{ Set database field }
procedure TDBLabel3D.SetDataField(const Value: string);
begin
  FDataLink.FieldName := Value;
end;

{ Set datasource }
procedure TDBLabel3D.SetDataSource(Value: TDataSource);
begin
  FDataLink.DataSource := Value;
end;

{ Set caption when data changes }
procedure TDBLabel3D.DataChange(Sender: TObject);
begin
  if FDataLink.Field = nil then
    Caption := ''
  else
    Caption := FDataLink.Field.AsString;
end;

{---------------------------------------------------------------------}

{ TDBMaskEdit }
{ Data Aware Masked Edit }
{ Override the constructor to initialize variables }
constructor TDBMaskEdit.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  inherited ReadOnly := True;
  MaskState := [msDBSetText];
  FDataLink := TFieldDataLink.Create;
  FDataLink.Control := Self;
  FDataLink.OnDataChange := DataChange;
  FDataLink.OnEditingChange := EditingChange;
  FDataLink.OnUpdateData := UpdateData;
  CalcTextMargin;
  EditMask := '';
end;

destructor TDBMaskEdit.Destroy;
begin
  FDataLink.Free;
  FDataLink := nil;
  FCanvas.Free;
  inherited Destroy;
end;

procedure TDBMaskEdit.Notification(AComponent: TComponent; Operation: TOperation);
begin
  inherited Notification(AComponent, Operation);
  if (Operation = opRemove) and (FDataLink <> nil) and
     (AComponent = DataSource) then DataSource := nil;
end;

procedure TDBMaskEdit.KeyDown(var Key: Word; Shift: TShiftState);
begin
  inherited KeyDown(Key, Shift);
  if (Key = VK_DELETE) or ((Key = VK_INSERT) and (ssShift in Shift)) then
    FDataLink.Edit;
end;

procedure TDBMaskEdit.KeyPress(var Key: Char);
begin
  inherited KeyPress(Key);
  if (Key in [#32..#255]) and (FDataLink.Field <> nil) and
     not FDataLink.Field.IsValidChar(Key) then
  begin
    MessageBeep(0);
    Key := #0;
  end;
  case Key of
    ^H, ^V, ^X, #32..#255:
      FDataLink.Edit;
    #27:
      begin
        FDataLink.Reset;
        SelectAll;
        Key := #0;
      end;
  end;
end;

function TDBMaskEdit.EditCanModify: Boolean;
begin
  Result := FDataLink.Edit;
end;

procedure TDBMaskEdit.Reset;
begin
  FDataLink.Reset;
  SelectAll;
end;

procedure TDBMaskEdit.SetFocused(Value: Boolean);
begin
  if FFocused <> Value then
  begin
    FFocused := Value;
    if (FAlignment <> taLeftJustify) and not IsMasked then
      Invalidate;
    FDataLink.Reset;
  end;
end;

procedure TDBMaskEdit.Change;
begin
  FDataLink.Modified;
  inherited Change;
end;

function TDBMaskEdit.GetDataSource: TDataSource;
begin
  Result := FDataLink.DataSource;
end;

procedure TDBMaskEdit.SetDataSource(Value: TDataSource);
begin
  FDataLink.DataSource := Value;
end;

function TDBMaskEdit.GetDataField: string;
begin
  Result := FDataLink.FieldName;
end;

procedure TDBMaskEdit.SetDataField(const Value: string);
begin
  FDataLink.FieldName := Value;
end;

function TDBMaskEdit.GetReadOnly: Boolean;
begin
  Result := FDataLink.ReadOnly;
end;

procedure TDBMaskEdit.SetReadOnly(Value: Boolean);
begin
  FDataLink.ReadOnly := Value;
end;

function TDBMaskEdit.GetField: TField;
begin
  Result := FDataLink.Field;
end;

procedure TDBMaskEdit.DataChange(Sender: TObject);
begin
  if FDataLink.Field <> nil then
  begin
    if FAlignment <> FDataLink.Field.Alignment then
    begin
      EditText := '';  {forces update}
      FAlignment := FDataLink.Field.Alignment;
    end;
    if FDataLink.Field.EditMask <> '' then
      EditMask := FDataLink.Field.EditMask;

    if FDataLink.Field.DataType = ftString then
      MaxLength := FDataLink.Field.Size else
      MaxLength := 0;
    if FFocused and FDataLink.CanModify then
      Text := FDataLink.Field.Text
    else
      EditText := FDataLink.Field.DisplayText;
  end
  else
  begin
    FAlignment := taLeftJustify;
    EditMask := '';
    MaxLength := 0;
    if csDesigning in ComponentState then
      EditText := Name
    else
      EditText := '';
  end;
end;

procedure TDBMaskEdit.EditingChange(Sender: TObject);
begin
  inherited ReadOnly := not FDataLink.Editing;
end;

procedure TDBMaskEdit.UpdateData(Sender: TObject);
begin
  ValidateEdit;
  FDataLink.Field.Text := Text;
end;

procedure TDBMaskEdit.WMPaste(var Message: TMessage);
begin
  FDataLink.Edit;
  inherited;
end;

procedure TDBMaskEdit.WMCut(var Message: TMessage);
begin
  FDataLink.Edit;
  inherited;
end;

procedure TDBMaskEdit.CMEnter(var Message: TCMEnter);
begin
  SetFocused(True);
  inherited;
end;

procedure TDBMaskEdit.CMExit(var Message: TCMExit);
begin
  try
    FDataLink.UpdateRecord;
  except
    SelectAll;
    SetFocus;
    raise;
  end;
  SetFocused(False);
  SetCursor(0);
  DoExit;
end;

procedure TDBMaskEdit.WMPaint(var Message: TWMPaint);
var
  Width, Indent, Left, I: Integer;
  R: TRect;
  DC: HDC;
  PS: TPaintStruct;
  S: string;
begin
  if (FAlignment = taLeftJustify) or FFocused then
  begin
    inherited;
    Exit;
  end;
  { Since edit controls do not handle justification unless multi-line (and
  then only poorly) we will draw right and center justify manually unless
  the edit has the focus. }
  if FCanvas = nil then
  begin
    FCanvas := TControlCanvas.Create;
    FCanvas.Control := Self;
  end;
  DC := Message.DC;
  if DC = 0 then
    DC := BeginPaint(Handle, PS);
  FCanvas.Handle := DC;
  try
    FCanvas.Font := Font;
    with FCanvas do
    begin
      R := ClientRect;
      if (BorderStyle = bsSingle) then
      begin
        Brush.Color := clWindowFrame;
        FrameRect(R);
        InflateRect(R, -1, -1);
      end;
      Brush.Color := Color;
      S := EditText;
      if PasswordChar <> #0 then
      begin
        for I := 1 to Length(S) do
          S[I] := PasswordChar;
      end;
      Width := TextWidth(S);
      if BorderStyle = bsNone then
        Indent := 0
      else
        Indent := FTextMargin;
      if FAlignment = taRightJustify then
        Left := R.Right - Width - Indent
      else
        Left := (R.Left + R.Right - Width) div 2;
      TextRect(R, Left, Indent, S);
    end;
  finally
    FCanvas.Handle := 0;
    if Message.DC = 0 then
      EndPaint(Handle, PS);
  end;
end;

procedure TDBMaskEdit.CMFontChanged(var Message: TMessage);
begin
  inherited;
  CalcTextMargin;
end;

procedure TDBMaskEdit.CalcTextMargin;
var
  DC: HDC;
  SaveFont: HFont;
  I: Integer;
  SysMetrics, Metrics: TTextMetric;
begin
  DC := GetDC(0);
  GetTextMetrics(DC, SysMetrics);
  SaveFont := SelectObject(DC, Font.Handle);
  GetTextMetrics(DC, Metrics);
  SelectObject(DC, SaveFont);
  ReleaseDC(0, DC);
  I := SysMetrics.tmHeight;
  if I > Metrics.tmHeight then
    I := Metrics.tmHeight;
  FTextMargin := I div 4;
end;

{---------------------------------------------------------------------}

{ TDBButton Component AboutBox }
procedure TDBBtnCompAbout.ShowAbout;
begin
  DBBtnAboutBox := TDBBtnAboutBox.Create(Application);
  DBBtnAboutBox.ShowModal;
  DBBtnAboutBox.Free;
end;

function TDBBtnCompAbout.GetVerbCount: Integer;
begin
  Result := 1;
end;

function TDBBtnCompAbout.GetVerb(Index: Integer): string;
begin
  Result := 'About';
end;

procedure TDBBtnCompAbout.ExecuteVerb(Index: Integer);
begin
  ShowAbout;
end;

{---------------------------------------------------------------------}

{ TDBButton Property AboutBox }
procedure TDBBtnPropAbout.ShowAbout;
begin
  DBBtnAboutBox := TDBBtnAboutBox.Create(Application);
  DBBtnAboutBox.ShowModal;
  DBBtnAboutBox.Free;
end;

procedure TDBBtnPropAbout.Edit;
begin
  ShowAbout;
end;

function TDBBtnPropAbout.GetAttributes: TPropertyAttributes;
begin
  Result := [paMultiSelect, paDialog, paReadOnly];
end;

{---------------------------------------------------------------------}

{ TDBBitBtn Component AboutBox }
procedure TDBBitBtnCompAbout.ShowAbout;
begin
  DBBitBtnAboutBox := TDBBitBtnAboutBox.Create(Application);
  DBBitBtnAboutBox.ShowModal;
  DBBitBtnAboutBox.Free;
end;

function TDBBitBtnCompAbout.GetVerbCount: Integer;
begin
  Result := 1;
end;

function TDBBitBtnCompAbout.GetVerb(Index: Integer): string;
begin
  Result := 'About';
end;

procedure TDBBitBtnCompAbout.ExecuteVerb(Index: Integer);
begin
  ShowAbout;
end;

{---------------------------------------------------------------------}

{ TDBBitBtn Property AboutBox }
procedure TDBBitBtnPropAbout.ShowAbout;
begin
  DBBitBtnAboutBox := TDBBitBtnAboutBox.Create(Application);
  DBBitBtnAboutBox.ShowModal;
  DBBitBtnAboutBox.Free;
end;

procedure TDBBitBtnPropAbout.Edit;
begin
  ShowAbout;
end;

function TDBBitBtnPropAbout.GetAttributes: TPropertyAttributes;
begin
  Result := [paMultiSelect, paDialog, paReadOnly];
end;

{---------------------------------------------------------------------}

{ TDBRadioButton Component AboutBox }
procedure TDBRadBtnCompAbout.ShowAbout;
begin
  DBRadBtnAboutBox := TDBRadBtnAboutBox.Create(Application);
  DBRadBtnAboutBox.ShowModal;
  DBRadBtnAboutBox.Free;
end;

function TDBRadBtnCompAbout.GetVerbCount: Integer;
begin
  Result := 1;
end;

function TDBRadBtnCompAbout.GetVerb(Index: Integer): string;
begin
  Result := 'About';
end;

procedure TDBRadBtnCompAbout.ExecuteVerb(Index: Integer);
begin
  ShowAbout;
end;

{---------------------------------------------------------------------}

{ TDBRadioButton Property AboutBox }
procedure TDBRadBtnPropAbout.ShowAbout;
begin
  DBRadBtnAboutBox := TDBRadBtnAboutBox.Create(Application);
  DBRadBtnAboutBox.ShowModal;
  DBRadBtnAboutBox.Free;
end;

procedure TDBRadBtnPropAbout.Edit;
begin
  ShowAbout;
end;

function TDBRadBtnPropAbout.GetAttributes: TPropertyAttributes;
begin
  Result := [paMultiSelect, paDialog, paReadOnly];
end;

{---------------------------------------------------------------------}

{ TDB3DLabel Component AboutBox }
procedure TDB3DLabCompAbout.ShowAbout;
begin
  DB3DLabAboutBox := TDB3DLabAboutBox.Create(Application);
  DB3DLabAboutBox.ShowModal;
  DB3DLabAboutBox.Free;
end;

function TDB3DLabCompAbout.GetVerbCount: Integer;
begin
  Result := 1;
end;

function TDB3DLabCompAbout.GetVerb(Index: Integer): string;
begin
  Result := 'About';
end;

procedure TDB3DLabCompAbout.ExecuteVerb(Index: Integer);
begin
  ShowAbout;
end;

{---------------------------------------------------------------------}

{ TDB3DLabel Property AboutBox }
procedure TDB3DLabPropAbout.ShowAbout;
begin
  DB3DLabAboutBox := TDB3DLabAboutBox.Create(Application);
  DB3DLabAboutBox.ShowModal;
  DB3DLabAboutBox.Free;
end;

procedure TDB3DLabPropAbout.Edit;
begin
  ShowAbout;
end;

function TDB3DLabPropAbout.GetAttributes: TPropertyAttributes;
begin
  Result := [paMultiSelect, paDialog, paReadOnly];
end;

{---------------------------------------------------------------------}

{ TDBMaskEdit Component AboutBox }
procedure TDBMaskEdCompAbout.ShowAbout;
begin
  DBMaskEdAboutBox := TDBMaskEdAboutBox.Create(Application);
  DBMaskEdAboutBox.ShowModal;
  DBMaskEdAboutBox.Free;
end;

function TDBMaskEdCompAbout.GetVerbCount: Integer;
begin
  Result := 1;
end;

function TDBMaskEdCompAbout.GetVerb(Index: Integer): string;
begin
  Result := 'About';
end;

procedure TDBMaskEdCompAbout.ExecuteVerb(Index: Integer);
begin
  ShowAbout;
end;

{---------------------------------------------------------------------}

{ TDBMaskEdit Property AboutBox }
procedure TDBMaskEdPropAbout.ShowAbout;
begin
  DBMaskEdAboutBox := TDBMaskEdAboutBox.Create(Application);
  DBMaskEdAboutBox.ShowModal;
  DBMaskEdAboutBox.Free;
end;

procedure TDBMaskEdPropAbout.Edit;
begin
  ShowAbout;
end;

function TDBMaskEdPropAbout.GetAttributes: TPropertyAttributes;
begin
  Result := [paMultiSelect, paDialog, paReadOnly];
end;

{ ----------------------------------------------------------------------------}

{ Register the components }
procedure Register;
begin
  RegisterComponents('Data Controls', [TDBButton, TDBBitBtn, TDBRadioButton,
                                       TDBLabel3D, TDBMaskEdit]);
  RegisterComponentEditor(TDBButton, TDBBtnCompAbout);
  RegisterPropertyEditor(TypeInfo(TDBBtnAboutBox), TDBButton, 'About', TDBBtnPropAbout);
  RegisterComponentEditor(TDBBitBtn, TDBBitBtnCompAbout);
  RegisterPropertyEditor(TypeInfo(TDBBitBtnAboutBox), TDBBitBtn, 'About', TDBBitBtnPropAbout);
  RegisterComponentEditor(TDBRadioButton, TDBRadBtnCompAbout);
  RegisterPropertyEditor(TypeInfo(TDBRadBtnAboutBox), TDBRadioButton, 'About', TDBRadBtnPropAbout);
  RegisterComponentEditor(TDBLabel3D, TDB3DLabCompAbout);
  RegisterPropertyEditor(TypeInfo(TDB3DLabAboutBox), TDBLabel3D, 'About', TDB3DLabPropAbout);
  RegisterComponentEditor(TDBMaskEdit, TDBMaskEdCompAbout);
  RegisterPropertyEditor(TypeInfo(TDBMaskEdAboutBox), TDBMaskEdit, 'About', TDBMaskEdPropAbout);
end;

end.
