unit DataConnection;

{ ****************************************************************************
  Component TDataConnection
  Version 1.0 - 27 April 1997 - for Delphi 2.x

  (c) 1997 by Wolfgang Junker

  See file README.TXT for reference and licensing details.
  The file README.TXT is an inseparable constituent of this Component.
  Distribution without the file README.TXT is illegal.
  **************************************************************************** }



interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs, DB;

type
  TConnectionDataLink = class;

  TCheckReadOnlyEvent = procedure(Sender: TObject; var Value: Boolean) of object;

  TDataConnection = class(TComponent)
  private
    FDataLink: TConnectionDataLink;
    FOnEditingChange: TNotifyEvent;
    FOnDataChange: TDataChangeEvent;
    FOnReadOnlyChange: TNotifyEvent;
    FOnCheckReadOnly: TCheckReadOnlyEvent;
    FReadOnly: Boolean;
    function GetDataSet: TDataSet;
    function GetDataSource: TDataSource;
    procedure SetDataSource(Value: TDataSource);
    function GetEditing: Boolean;
    function GetState: TDataSetState;
  protected
    procedure EditingChanged;
    procedure RecordChanged(Field: TField);
    procedure ActiveChanged;
    procedure ReadOnlyChanged;
    procedure Loaded; override;
    procedure Notification(AComponent: TComponent;
      Operation: TOperation); override;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    procedure CheckReadOnly;
    property DataSet: TDataSet read GetDataSet;
    property Editing: Boolean read GetEditing;
    property ReadOnly: Boolean read FReadOnly;
    property State: TDataSetState read GetState;
  published
    property DataSource: TDataSource read GetDataSource write SetDataSource;
    property OnEditingChange: TNotifyEvent read FOnEditingChange
      write FOnEditingChange;
    property OnDataChange: TDataChangeEvent read FOnDataChange
      write FOnDataChange;
    property OnReadOnlyChange: TNotifyEvent read FOnReadOnlyChange
      write FOnReadOnlyChange;
    property OnCheckReadOnly: TCheckReadOnlyEvent read FOnCheckReadOnly
      write FOnCheckReadOnly;
  end;

  TConnectionDataLink = class(TDataLink)
  private
    FDataConnection: TDataConnection;
  protected
    procedure EditingChanged; override;
    procedure RecordChanged(Field: TField); override;
    procedure ActiveChanged; override;
  public
    constructor Create(AConnection: TDataConnection);
    destructor Destroy; override;
  end;

procedure Register;

implementation

procedure Register;
begin
  RegisterComponents('Datenzugriff', [TDataConnection]);
end;


{ TDataConnection }

constructor TDataConnection.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FDataLink := TConnectionDataLink.Create(Self);
  FReadOnly := True;
end;

destructor TDataConnection.Destroy;
begin
  FDataLink.Free;
  FDataLink := nil;
  inherited Destroy;
end;

procedure TDataConnection.Loaded;
begin
  inherited Loaded;
  ActiveChanged;
end;

procedure TDataConnection.Notification(AComponent: TComponent;
  Operation: TOperation);
begin
  inherited Notification(AComponent, Operation);
  if (Operation = opRemove) and (FDataLink <> nil) and
    (AComponent = DataSource) then DataSource := nil;
end;

function TDataConnection.GetDataSet: TDataSet;
begin
  if DataSource <> nil then Result := DataSource.DataSet else Result := nil;
end;

function TDataConnection.GetDataSource: TDataSource;
begin
  Result := FDataLink.DataSource;
end;

procedure TDataConnection.SetDataSource(Value: TDataSource);
begin
  FDataLink.DataSource := Value;
  if not (csLoading in ComponentState) then
    ActiveChanged;
  if Value <> nil then Value.FreeNotification(Self);
end;

function TDataConnection.GetEditing: Boolean;
begin
  Result := FDataLink.Editing;
end;

function TDataConnection.GetState: TDataSetState;
begin
  if DataSource <> nil then Result := DataSource.State else Result := dsInactive;
end;

procedure TDataConnection.EditingChanged;
begin
  if Assigned(FOnEditingChange) then FOnEditingChange(Self);
end;

procedure TDataConnection.RecordChanged(Field: TField);
begin
  if Field = nil then CheckReadOnly;
  if Assigned(FOnDataChange) then FOnDataChange(Self, Field);
end;

procedure TDataConnection.ActiveChanged;
begin
  if FDataLink.Active then begin
    RecordChanged(nil);
    EditingChanged;
  end else CheckReadOnly;
end;

procedure TDataConnection.ReadOnlyChanged;
begin
  if Assigned(FOnReadOnlyChange) then FOnReadOnlyChange(Self);
end;

procedure TDataConnection.CheckReadOnly;
var
  RO: Boolean;
begin
  try
    if FDataLink.Active then begin
      RO := FDataLink.ReadOnly;
      if Assigned(FOnCheckReadOnly) then FOnCheckReadOnly(Self, RO);
    end else RO := True;
  finally
    if RO <> fReadOnly then begin
      fReadOnly := RO;
      if Assigned(FOnReadOnlyChange) then FOnReadOnlyChange(Self);
    end;
  end;
end;


{ TConnectionDataLink }

constructor TConnectionDataLink.Create(AConnection: TDataConnection);
begin
  inherited Create;
  FDataConnection := AConnection;
end;

destructor TConnectionDataLink.Destroy;
begin
  FDataConnection := nil;
  inherited Destroy;
end;

procedure TConnectionDataLink.EditingChanged;
begin
  inherited EditingChanged;
  if FDataConnection <> nil then FDataConnection.EditingChanged;
end;

procedure TConnectionDataLink.RecordChanged(Field: TField);
begin
  inherited RecordChanged(Field);
  if FDataConnection <> nil then FDataConnection.RecordChanged(Field);
end;

procedure TConnectionDataLink.ActiveChanged;
begin
  inherited ActiveChanged;
  if FDataConnection <> nil then FDataConnection.ActiveChanged;
end;

end.
