unit FTP;

interface

Uses SysUtils, Classes, Winsock, SockHelp, WinTypes, WinProcs, Dialogs,
     Messages;

Const
  { Commands } 
  // Maximum buffer size for FTP reads
  MaxFTPBuffer = 32767;
  MaxFTPLine = 80;    

  wm_Results = wm_User + 1;

  // Additional file attributes
  faRead      = $00000100;
  faWrite     = $00000200;
  faExecute   = $00000400;
  faLink      = $00001000;
  faBlock     = $00002000;
  faCharacter = $00004000;
  faPipe      = $00008000;

Type
  PFTPDirListing = ^TFTPDirListing;
  TFTPDirListing = record
    Attributes: LongInt;
    FilePermissions: String;
    CreationTime: String;
    FileSize: LongInt;
    FileName: String;
    Links: Integer;
    Owner: String;
    Group: String;
  end;


  TFTPDirList = class(TList)
    procedure Delete(Index: Integer);
    function IsDirectory(Index: Integer): Boolean;
    function FileAttributes(Index: Integer): LongInt;
    function FilePermissions(Index: Integer): String;
    function CreationTime(Index: Integer): String;
    function FileSize(Index: Integer): LongInt;
    function FileName(Index: Integer): String;
    function Links(Index: Integer): Integer;
    function Owner(Index: Integer): String;
    function Group(Index: Integer): String;
  end;

  FTPTransferType = (ftpASCII, ftpBinary);
  FTPAccessType = (acPre_Config,     { use default }
                   acLocal,          { direct to Internet }
                   acGateway,        { Internet via gateway }
                   acCERN_Proxy      { Internet via CERN proxy });


  TFTP = class(TComponent)
  private
    { Private declarations }
    fTransfer: FTPTransferType;
    fAccess: FTPAccessType;
    fPeerName: String;
    fHostName,
    fUserName,
    fPassword,
    fAccount: String;
    fPeerPort: Word;
    fLastLine,
    fLastMessage: String;
    fFTPCmdSocket: TSOCKET;     // Socket handle (or descriptor)
    fFTPDataSocket: TSocket;
    fReadSocket: TSocket;
    fWriteSocket: TSocket;
    fLastError: LongInt;
    fCaller: THandle;

    function LoginFTP: Boolean;
    function ConnectCmdSocket: Boolean;
    procedure SetTransferType(Value: FTPTransferType);
  protected
    { Protected declarations }
  public
    { Public declarations }
    FTPBuffer: Array[0..MaxFTPBuffer] Of Byte;
    FTPLines: TStringList;
    constructor Create(AOwner: TComponent);
    procedure Free;
    function Disconnect: Boolean;
    function Connect: Boolean;
    function LogonUser: Boolean;
    function LogonPassword: Boolean;
    function LogonAccount: Boolean;

    function ReadLines(WaitTime: LongInt): Boolean;
    function SendCommand(Command: String): Boolean;
    function SendCommandResult(strCommand: string; WaitTime: LongInt): Boolean;
    function CreateListenSocket(var strResult: String): TSocket;
    function ReadBuffer(var BufRead: Integer; WaitTime: LongInt): Boolean;
    function WriteBuffer(var BufWritten: Integer; WaitTime: LongInt): Boolean;

    function Get(RemoteFile, LocalFile: String): Boolean;
    function Put(RemoteFile, LocalFile: String): Boolean;

    function DeleteFile(FileName: String): Boolean;
    function RenameFile(FromFileName, ToFileName: String): Boolean;

    function DirectoryList: TFTPDirList;
    function ChDir(PathName: String): Boolean;
    function MkDir(PathName: String): Boolean;
    function RmDir(PathName: String): Boolean;
    function PWD(var PathName: String): Boolean;

    procedure SendListToCaller(strList: TStringList);
    function ConnectionInfo(hSocket: TSocket): String;

    function CreateDirEntry(strInput: String): PFTPDirListing;
    property Caller: THandle read fCaller write fCaller;
  published
    { Published declarations }
    property Transfer: FTPTransferType read fTransfer write SetTransferType;
    property Access: FTPAccessType read fAccess write fAccess;
    property HostName: String read fHostName write fHostName;
    property UserName: String read fUserName write fUserName;
    property Password: String read fPassword write fPassword;
    property Account: String read fAccount write fAccount;
    property LastLine: String read fLastLine;
    property LastMessage: String read fLastMessage;
    property FTPCmdSocket: TSOCKET read fFTPCmdSocket;
    property FTPDataSocket: TSocket read fFTPDataSocket;
    property ReadSocket: TSocket read fReadSocket;
    property WriteSocket: TSocket read fWriteSocket;
    property LastError: LongInt read fLastError;
  end;

function ParseFTPReply(Reply: String; var Pos1, Pos2: Char): Boolean;
procedure StringToList(var strInput: String; strlList: TStringList);
function ConvertFileTime(FT: TFileTime): TDateTime;

implementation

procedure TFTPDirList.Delete(Index: Integer);
begin
  Dispose(PFTPDirListing(Items[Index]));
  Items[Index] := Nil;
  inherited Delete(Index);
end;

function TFTPDirList.IsDirectory(Index: Integer): Boolean;
begin
  If Index >= Count THEN
     Result := False
  ELSE
     Result := (PFTPDirListing(Items[Index])^.Attributes AND
                faDirectory) <> 0;
end;

function TFTPDirList.FileAttributes(Index: Integer): LongInt;
begin
  If Index >= Count THEN
     Result := 0
  ELSE
     Result := PFTPDirListing(Items[Index])^.Attributes;
end;

function TFTPDirList.FilePermissions(Index: Integer): String;
begin
  If Index >= Count THEN
     Result := ''
  ELSE
     Result := PFTPDirListing(Items[Index])^.FilePermissions;
end;

function TFTPDirList.CreationTime(Index: Integer): String;
begin
  If Index >= Count THEN
     Result := ''
  ELSE
     Result := PFTPDirListing(Items[Index])^.CreationTime;
end;

function TFTPDirList.FileSize(Index: Integer): LongInt;
begin
  If Index >= Count THEN
     Result := 0
  ELSE
     Result := PFTPDirListing(Items[Index])^.FileSize;
end;

function TFTPDirList.FileName(Index: Integer): String;
begin
  If Index >= Count THEN
     Result := ''
  ELSE
     Result := PFTPDirListing(Items[Index])^.FileName;
end;

function TFTPDirList.Links(Index: Integer): Integer;
begin
  If Index >= Count THEN
     Result := -1
  ELSE
     Result := PFTPDirListing(Items[Index])^.Links;
end;

function TFTPDirList.Owner(Index: Integer): String;
begin
  If Index >= Count THEN
     Result := ''
  ELSE
     Result := PFTPDirListing(Items[Index])^.Owner;
end;

function TFTPDirList.Group(Index: Integer): String;
begin
  If Index >= Count THEN
     Result := ''
  ELSE
     Result := PFTPDirListing(Items[Index])^.Group;
end;



function ConvertFileTime(FT: TFileTime): TDateTime;
var
  ST: TSystemTime;
begin
  Result := 0;
  If FileTimeToSystemTime(FT, ST) THEN
  With ST DO
    Result := EncodeDate(wYear, wMonth, wDay) +
              EncodeTime(wHour, wMinute, wSecond, wMilliseconds);
end;


// Checks the reply from the FTP Remote Host
// Returns TRUE if the reply starts in '0'..'5'
function ParseFTPReply(Reply: String; var Pos1, Pos2: Char): Boolean;
begin
  Pos1 := #0;
  Pos2 := #0;
  If Length(Reply) < 2 THEN
     Result := False
  ELSE
  Begin
    Result := True;
    Pos1 := Reply[1];
    Pos2 := Reply[2];
    If Not (Pos1 IN ['1'..'5']) THEN
       Result := False;
    If Not (Pos2 IN ['0'..'5']) THEN
       Result := False;
  End;
end;

constructor TFTP.Create(AOwner: TComponent);
var
  WSData: TWSAData;
begin
  inherited Create(AOwner);
  FTPLines := TStringList.Create;
  fAccess :=  acLocal;
  fHostName := '';
  fUserName := '';
  fPassword := '';
  fPeername := '';
  fPeerPort := 0;
  fCaller := 0;
  fLastError := 0;
  fLastLine := '';
  fLastMessage := '';
  fLastError := WSAStartup($11, WSData);
  SetTransferType(ftpBinary);
end;

procedure TFTP.Free;
begin
  DisConnect;
  FTPLines.Free;
  WSACleanup;
  inherited Free;
end;

// Connect to host
function TFTP.Connect: Boolean;
var
  Reply: String;
  strlList: TStringList;
  Name: TSockAddr;
  NameLen: Integer;
begin
  Result := False;
  If ConnectCmdSocket THEN
  Begin
    NameLen := SizeOf(Name);
    // Get the name of the host we are connected to
    If getpeername(fFTPCmdSocket, Name, NameLen) = 0 THEN
    Begin
      fPeerName := StrPas(inet_ntoa(name.sin_addr));
      fPeerPort := ntohs(name.sin_port);
    End;
    // Now see if the remote host is talking to us (3 seconds to time out)
    If ReadLines(3) THEN
       Result := (fLastMessage = '220'); // Ready for new user
    If Result THEN
       Result := LoginFTP;
  End;
end;

// Login with username, password, and account
function TFTP.LoginFTP: Boolean;
var
  I: Integer;
begin
  LogonUser;
  // Now see if the remote host is talking to us again
  fLastMessage := '';
  fLastLine := '';
  If ReadLines(3) THEN
  Begin
    If (fLastMessage = '331') AND LogonPassword THEN
    Begin
      fLastMessage := '';
      ReadLines(3);
      If (fLastMessage = '332') AND LogonAccount THEN
      Begin
        fLastMessage := '';
        ReadLines(3);
      End;
    End;
    Result := (fLastMessage = '230');
  End;
  If Result THEN
     SendCommandResult('MODE S', 3);
end;

// Disconnect from host
function TFTP.Disconnect: Boolean;
begin
  FTPLines.Clear;
  If fFTPCmdSocket <> INVALID_SOCKET THEN
     Result := CloseSocket(fFTPCmdSocket) <> 0;
  If Result THEN
     fFTPCmdSocket := INVALID_SOCKET
  ELSE
     fLastError := WSAGetLastError;
end;

// Connect to the remote hosts FTP socket
function TFTP.ConnectCmdSocket: Boolean;
var
  iFTPPort: Integer;
  pServEntry: pServENT;  // Internet protocol data structure
begin
  Result := False;
  pServEntry := getservbyname('FTP', Nil);
  If (pServEntry = Nil) THEN
     iFTPPort := IPPORT_FTP
  ELSE
     iFTPPort := pServEntry^.s_port;

  fFTPCmdSocket := ConnectSocket(fHostName, iFTPPort);
  If fFTPCmdSocket <> INVALID_SOCKET THEN
     Result := TRUE;
end;

// Send user name
function TFTP.LogonUser: Boolean;
var
  iCharSent: Integer;
  strUser: String;
begin
  Result := False;
  If fFTPCmdSocket = INVALID_SOCKET THEN
     Exit;
  strUser := 'USER ' + fUserName + #13 + #10;
  iCharSent := Winsock.send(fFTPCmdSocket, strUser[1], Length(strUser), 0);

  If iCharSent = SOCKET_ERROR THEN
     fLastError := WSAGetLastError
  ELSE
     Result := true;
end;

// Send password
function TFTP.LogonPassword: Boolean;
var
  iCharSent: Integer;
  strOwner: String;
begin
  Result := False;
  If fFTPCmdSocket = INVALID_SOCKET THEN
     Exit;
  strOwner := 'pass ' + fPassword + #13 + #10;
  iCharSent := Winsock.send(fFTPCmdSocket, strOwner[1], Length(strOwner), 0);

  If iCharSent = SOCKET_ERROR THEN
     fLastError := WSAGetLastError
  ELSE
     Result := true;
end;

// Send account
function TFTP.LogonAccount: Boolean;
var
  iCharSent: Integer;
  strOwner: String;
begin
  Result := False;
  If fFTPCmdSocket = INVALID_SOCKET THEN
     Exit;
  strOwner := 'ACCT ' + fAccount + #13 + #10;
  iCharSent := Winsock.send(fFTPCmdSocket, strOwner[1], Length(strOwner), 0);

  If iCharSent = SOCKET_ERROR THEN
     fLastError := WSAGetLastError
  ELSE
     Result := true;
end;

// Parses a line into a string list based on the deliminator specified
function ParseFTPLine(Input: String; Delim: String): TStringList;
var
  L,
  P: Integer;
  S,
  A: String;
begin
  Result := TStringList.Create;
  S := Input;
  L := Length(Delim);
  P := Pos(Delim, S);
  While P > 0 DO
  Begin
    If P > 1 THEN
    Begin
      A := Copy(S, 1, P - 1);
      Result.Add(Trim(A));
    End;
    S := Copy(S, P + L, Length(S));
    S := Trim(S);
    P := Pos(Delim, S);
  End;
  If (Length(S) > 0) AND (S <> '') THEN
     Result.Add(S);
end;

// Send a command to the FTP host
// Do not wait for reply
function TFTP.SendCommand(Command: String): Boolean;
var
  iCharSent: Integer;
  strOwner: String;
begin
  Result := False;
  If fFTPCmdSocket = INVALID_SOCKET THEN
     Exit;
  strOwner := Command + #13 + #10;
  iCharSent := Winsock.send(fFTPCmdSocket, strOwner[1], Length(strOwner), 0);

  If iCharSent = SOCKET_ERROR THEN
     fLastError := WSAGetLastError
  ELSE
     Result := true;
end;

// Sends command and gets host's response
// Response is sent to user-defined Handle by message
// Result is TRUE if reply was 1xx, 2xx, or 3xx
function TFTP.SendCommandResult(strCommand: string; WaitTime: LongInt): Boolean;
begin
  Result := SendCommand(strCommand);
  If Result THEN
     If ReadLines(WaitTime) THEN
        Result := (fLastMessage[1] IN ['1'..'3']);
end;

procedure TFTP.SetTransferType(Value: FTPTransferType);
begin
  fTransfer := Value;
  Case Value OF
       ftpASCII: SendCommandResult('TYPE ASCII', 2);
       ftpBinary: SendCommandResult('TYPE IMAGE', 2);
  End;
end;

// This will loop until all the data is read
// All lines (CRLF-delimited) are returned in the string list
function TFTP.ReadLines(WaitTime: LongInt): Boolean;
var
  strResult,
  strLeftOver: String;
  P,
  iConnect,
  iCharRecv,
  iLength: Integer;
  ReadFDS: TFDSet;
  timeout: TTimeVal;
begin
  strResult := '';
  Result := False;

  If fFTPCmdSocket = INVALID_SOCKET THEN
     Exit;

  ReadFDS.fd_Count := 1;
  ReadFDS.fd_array[0] := fFTPCmdSocket;
  // If nothing to read in WaitTime seconds, then exit
  timeout.tv_sec := WaitTime;
  timeout.tv_usec := 0;

  FTPLines.Clear;
  If select(0, @Readfds, Nil, Nil, @timeout) > 0 THEN
  Begin
    iLength := MaxFTPLine;
    SetLength(strResult, iLength + 1);
    timeout.tv_sec := 1;
    timeout.tv_usec := 0;
    iCharRecv := select(0, @Readfds, Nil, Nil, @timeout);
    While (iCharRecv > 0) DO
    Begin
      SetLength(strResult, iLength + 1);
      iCharRecv := Winsock.recv(fFTPCmdSocket, strResult[1], iLength, 0);
      If iCharRecv > 0 THEN
      Begin
        // Fix all CRLF pairs
        SetLength(strResult, iCharRecv);
        If Length(strLeftOver) > 0 THEN
           strResult := strLeftOver + strResult;
        // Now parse it into a string list
        StringToList(strResult, FTPLines);
        If Length(strResult) > 0 THEN
           strLeftover := strResult
        ELSE
           strLeftOver := '';
        strResult := '';
      End;
      iCharRecv := select(0, @Readfds, Nil, Nil, @timeout);
    End;
    If Length(strLeftover) > 0 THEN
       FTPLines.Add(strLeftOver);
    If Length(strResult) > 0 THEN
       FTPLines.Add(strResult);
    If FTPLines.Count > 0 THEN
    Begin
      fLastLine := FTPLines[FTPLines.Count-1];
      fLastMessage := Copy(fLastLine, 1, 3);
    End;
    Result := True;
  End;
  SendListToCaller(FTPLines);
end;

function TFTP.CreateListenSocket(var strResult: String): TSocket;
var
  Name: TSockAddr;
  strCommand: String;
  strList: TStringList;
  P: Word;
begin
  Result := INVALID_SOCKET;
  If fFTPCmdSocket = INVALID_SOCKET THEN
     Exit;

  Result := socket(AF_INET, SOCK_STREAM, IPPROTO_TCP);
  If Result = INVALID_SOCKET THEN
  Begin
    fLastError := WSAGetLastError;
    Exit;
  End;
  If GetHostConnectionInfo(fFTPCmdSocket, Name) THEN
  Begin
    With Name DO
    Begin
      sin_port := htons(0);
      sin_family := AF_INET;
    End;

    // Bind any socket
    If NOT BindToPort(Result, Name) THEN
    Begin
      CloseSocket(Result);
      Result := INVALID_SOCKET;
      Exit
    End
    ELSE
    If GetHostConnectionInfo(Result, Name) THEN
    Begin
      // Now we have our socket.  Send info to the other end
      P := Name.sin_port;
      With Name.sin_addr.s_un_b DO
           strCommand := 'port ' + IntToStr(Ord(s_b1)) +
                         ',' + IntToStr(Ord(s_b2)) +
                         ',' + IntToStr(Ord(s_b3)) +
                         ',' + IntToStr(Ord(s_b4)) +
                         ',' + IntToStr(Lo(P)) +
                         ',' + IntToStr(Hi(P));

      If Not SendCommandResult(strCommand, 5) THEN
      Begin
        CloseSocket(Result);
        Result := INVALID_SOCKET;
        Exit;
      End
      ELSE
        Listen(Result, 1);
    End;
  End;
end;

// Data returned in FTPBuffer
// Caller should clear FTPBuffer before calling again
// Returns False if no more data, True if more
function TFTP.ReadBuffer(var BufRead: Integer; WaitTime: LongInt): Boolean;
var
  iCharRecv: Integer;
  ReadFDS: TFDSet;
  timeout: TTimeVal;
begin
  Result := False;

  If (fFTPCmdSocket = INVALID_SOCKET) OR
     (fFTPDataSocket = INVALID_SOCKET) THEN
     Exit;

  If fReadSocket = INVALID_SOCKET THEN
     Exit;

  ReadFDS.fd_Count := 1;
  ReadFDS.fd_array[0] := fReadSocket;
  // If nothing to read in WaitTime seconds, then exit
  timeout.tv_sec := WaitTime;
  timeout.tv_usec := 0;

  If select(0, @Readfds, Nil, Nil, @timeout) > 0 THEN
  Begin
    iCharRecv := Winsock.recv(fReadSocket, FTPBuffer, BufRead, 0);
    BufRead := iCharRecv;
    If iCharRecv <= 0  THEN
    Begin
      If iCharRecv < 0 THEN
         fLastError := WSAGetLastError;
      Result :=  False; // Got it all
    End
    ELSE
       Result := True; // More to go

  End
  ELSE
    Result := False;
end;


function TFTP.Get(RemoteFile, LocalFile: String): Boolean;
var
  F: File;
  bResult: Boolean;
  BufWritten,
  BufRead: Integer;
  strResult: String;
  Name: TSockAddr;
  NameLen: Integer;
begin
  Result := False;

  AssignFile(F, LocalFile);
  {$I-}
  ReWrite(F, 1);
  {$I+}
  If IOResult <> 0 THEN
     Exit;

  fFTPDataSocket := CreateListenSocket(strResult);

  If fFTPDataSocket <> INVALID_SOCKET THEN
  Begin
    If SendCommandResult('RETR ' + RemoteFile, 5) THEN
    Begin
      NameLen := SizeOf(Name);
      fReadSocket := accept(fFTPDataSocket, Name, NameLen);
      // Read the buffer
      bResult := True;
      While bResult DO
      Begin
        BufRead := MaxFTPBuffer;
        bResult := ReadBuffer(BufRead, 2);
        If bResult AND (BufRead > 0) THEN
        Begin
          BlockWrite(F, FTPBuffer[0], BufRead, BufWritten);
          If BufWritten = BufRead THEN
             Result := True;
        End;
      End;
      CloseSocket(fReadSocket);
    End;
  End;
  CloseSocket(fFTPDataSocket);
  fFTPDataSocket := INVALID_SOCKET;
  CloseFile(F);
end;


// Place data in FTPBuffer before calling
function TFTP.WriteBuffer(var BufWritten: Integer; WaitTime: LongInt): Boolean;
var
  iCharSent: Integer;
  WriteFDS: TFDSet;
  timeout: TTimeVal;
begin
  Result := False;

  If (fFTPCmdSocket = INVALID_SOCKET) OR
     (fFTPDataSocket = INVALID_SOCKET) THEN
     Exit;

  If fWriteSocket = INVALID_SOCKET THEN
     Exit;

  WriteFDS.fd_Count := 1;
  WriteFDS.fd_array[0] := fWriteSocket;
  // If nothing to read in WaitTime seconds, then exit
  timeout.tv_sec := WaitTime;
  timeout.tv_usec := 0;
  iCharSent := -1;
  If select(0, Nil, @WriteFDS, Nil, @timeout) > 0 THEN
  Begin
    timeout.tv_sec := 1;
    timeout.tv_usec := 0;
    iCharSent := Winsock.send(fWriteSocket, FTPBuffer, BufWritten, 0);
  End;
  Result := (iCharSent = BufWritten);
  BufWritten := iCharSent;
end;

function TFTP.Put(RemoteFile, LocalFile: String): Boolean;
var
  F: File;
  bResult: Boolean;
  BufRead: Integer;
  strlResult: TStringList;
  strResult: String;
  Name: TSockAddr;
  NameLen: Integer;
begin
  Result := False;

  AssignFile(F, LocalFile);
  {$I-}
  Reset(F, 1);
  {$I+}
  If IOResult <> 0 THEN
     Exit;

  fFTPDataSocket := CreateListenSocket(strResult);

  If fFTPDataSocket <> INVALID_SOCKET THEN
  Begin
    If SendCommandResult('STOR ' + RemoteFile, 5) THEN
    Begin
      NameLen := SizeOf(Name);
      fWriteSocket := accept(fFTPDataSocket, Name, NameLen);
      // Read the buffer
      bResult := True;
      While bResult DO
      Begin
        BlockRead(F, FTPBuffer[0], MaxFTPBuffer, BufRead);
        If BufRead > 0 THEN
           bResult := WriteBuffer(BufRead, 2);
        If BufRead < MaxFTPBuffer THEN
           bResult := False;
      End;
      If BufRead < MaxFTPBuffer THEN
         Result := True;
      CloseSocket(fWriteSocket);
      If Result THEN
         ReadLines(3);
    End;
  End;
  CloseSocket(fFTPDataSocket);
  fFTPDataSocket := INVALID_SOCKET;
  CloseFile(F);
end;


procedure TFTP.SendListToCaller(strList: TStringList);
begin
  If fCaller > 0 THEN
     SendMessage(fCaller, wm_Results, 0, LongInt(strList));
end;

function TFTP.ConnectionInfo(hSocket: TSocket): String;
var
  Name: TSockAddr;
  P: Word;
begin
  If GetPeerConnectionInfo(fFTPCmdSocket, Name) THEN
  Begin
    P := Name.sin_port;
    With Name.sin_addr.s_un_b DO
         Result := 'Connected to ' + IntToStr(Ord(s_b1)) +
                   '.' + IntToStr(Ord(s_b2)) +
                   '.' + IntToStr(Ord(s_b3)) +
                   '.' + IntToStr(Ord(s_b4)) +
                   ' on Port ' + IntToStr(Lo(P)) +
                   IntToStr(Hi(P));
  End;
end;

function TFTP.DirectoryList: TFTPDirList;
var
  bResult: Boolean;
  P, I,
  BufRead: Integer;
  strlParse,
  strlList: TStringList;
  strLeftover,
  strInput: String;
  Name: TSockAddr;
  NameLen: Integer;
begin
  strlList := Nil;
  fFTPDataSocket := CreateListenSocket(strInput);

  If fFTPDataSocket <> INVALID_SOCKET THEN
  Begin
    If SendCommandResult('LIST', 3) THEN
    Begin
      NameLen := SizeOf(Name);
      fReadSocket := accept(fFTPDataSocket, Name, NameLen);
      strlList := TStringList.Create;
      Result := TFTPDirList.Create;
      bResult := True;
      strLeftOver := '';
      strInput := '';
      // Read the buffer
      While bResult DO
      Begin
        SetLength(strInput, MaxFTPBuffer);
        BufRead := MaxFTPBuffer;
        bResult := ReadBuffer(BufRead, 2);
        SetLength(strInput, BufRead + 1);
        Move(FTPBuffer, strInput[1], BufRead);
        strInput[BufRead + 1] := #0;
        strInput := strLeftOver + strInput;
         strInput := AdjustLineBreaks(strInput);
        P := Pos(#13#10, strInput);
        While P > 0 DO
        Begin
          strLeftover :=Copy(strInput, 1, P - 1);
          strlList.Add(strLeftover);
          strInput := Copy(strInput, P + 2, Length(strInput));
          P := Pos(#13#10, strInput);
        End;
        strLeftover := strInput;
      End;
      CloseSocket(fReadSocket);
      // Now convert list to FTPDirListing Entries
      // Should be nine strings
      If strlList.Count > 0 THEN
      Begin
        For I := 0 TO strlList.Count - 1 DO
        Begin
          strInput := strlList[I];
          If strInput <> '' THEN
             Result.Add(CreateDirEntry(strInput));
        End;
      End;
    End;
  End;
  CloseSocket(fFTPDataSocket);
  fFTPDataSocket := INVALID_SOCKET;
end;

// Parses a string into a list
// strInput returns with any "leftover" text (no CRLF pair at end)
procedure StringToList(var strInput: String; strlList: TStringList);
var
  P: Integer;
  strWork: String;
begin
  If Not Assigned(strlList) THEN
     Exit;
     
  // Now parse it into a string list
  strInput := AdjustLineBreaks(strInput);
  P := Pos(#13#10, strInput);
  While P > 0 DO
  Begin
    strWork :=Copy(strInput, 1, P - 1);
    strlList.Add(strWork);
    strInput := Copy(strInput, P + 2, Length(strInput));
    P := Pos(#13#10, strInput);
  End;
end;

// Create a directory entry for the FTPDirectoryList
function TFTP.CreateDirEntry(strInput: String): PFTPDirListing;
var
  strlParse: TStringList;
  Ch: Char;
begin
  Result := New(PFTPDirListing);
  With Result^ DO
  Begin
    Attributes := 0;
    FilePermissions := '';
    CreationTime := '';
    FileSize := 0;
    FileName := '';
    Links := 0;
    Owner := '';
    Group := '';
  End;
  strlParse := ParseFTPLine(strInput, ' ');
  If strlParse.Count > 8 THEN
  Begin
     With Result^ DO
     Begin
       FileName := strlParse[8];
       If FileName[1] = '.' THEN
          Attributes := Attributes OR faDirectory;
       FilePermissions := strlParse[0];
       Ch := UpCase(FilePermissions[1]);
       Case Ch OF
         'D': Attributes := Attributes OR faDirectory;
         'L': Attributes := Attributes OR faLink;
         'B': Attributes := Attributes OR faBlock;
         'C': Attributes := Attributes OR faCharacter;
         'P': Attributes := Attributes OR faPipe;
       End;
       Ch := FilePermissions[2];
       If Ch = 'R' THEN
          Attributes := Attributes OR faRead;
       Ch := FilePermissions[3];
       If Ch = 'W' THEN
          Attributes := Attributes OR faWrite;
       Ch := FilePermissions[4];
       If Ch = 'X' THEN
          Attributes := Attributes OR faExecute;

       Links := StrToIntDef(strlParse[1], 0);
       Owner := strlParse[2];
       Group := strlParse[3];
       FileSize := StrToIntDef(strlParse[4], 0);
       CreationTime := strlParse[5] + ' ' +
                       strlParse[6] + ' ' +
                       strlParse[7];

     End;
  End;
  strlParse.Free;
end;

// Delete file
function TFTP.DeleteFile(FileName: String): Boolean;
var
  strCommand: String;
begin
  strCommand := 'DELE ' + FileName;
  Result := SendCommandResult(strCommand, 5);
end;

// Rename file
function TFTP.RenameFile(FromFileName, ToFileName: String): Boolean;
var
  strCommand: String;
begin
  strCommand := 'RNFR ' + FromFileName;
  Result := SendCommandResult(strCommand, 5);
  If Result THEN
  Begin
    strCommand := 'RNTO ' + ToFileName;
    Result := SendCommandResult(strCommand, 5);
  End;
end;

// Change directories
function TFTP.ChDir(PathName: String): Boolean;
var
  strCommand: String;
begin
  strCommand := 'CWD ' + PathName;
  Result := SendCommandResult(strCommand, 5);
end;

// Make a directory
function TFTP.MkDir(PathName: String): Boolean;
var
  strCommand: String;
begin
  strCommand := 'MKD ' + PathName;
  Result := SendCommandResult(strCommand, 5);
end;

// Remove the directory
function TFTP.RmDir(PathName: String): Boolean;
var
  strCommand: String;
begin
  strCommand := 'RMD ' + PathName;
  Result := SendCommandResult(strCommand, 5);
end;

// return the present working directory
function TFTP.PWD(var PathName: String): Boolean;
var
  strlList: TStringList;
  P: Integer;
  strCommand: String;
begin
  strCommand := 'PWD';
  Result := SendCommand(strCommand);
  If Result THEN
  Begin
    If ReadLines(3) THEN
    Begin
      Result := (fLastMessage[1] IN ['1'..'3']);
      If Result THEN
      Begin
        SetLength(PathName, Length(fLastLine));
        PathName := Copy(fLastLine, 5, Length(fLastLine));
        P := Pos(' ', PathName);
        If P > 0 THEN
           PathName := Copy(PathName, 1, P - 1);
        P := Pos('"', PathName);
        If P = 1 THEN
           PathName := Copy(PathName, 2, Length(PathName));
        P := Pos('"', PathName);
        If P = Length(PathName) THEN
           SetLength(PathName, Length(PathName) - 1);
      End;
    End;
  End
  ELSE
    Result := False;
end;

end.

