Unit SockHelp;

Interface

Uses SysUtils, WinSock, WinTypes, WinProcs, Dialogs, Classes;

Const
  // IP and ICMP data structures for raw sockets. 

  ICMP_ECHO       = 8;			         // An ICMP echo message 
  ICMP_ECHOREPLY  = 0;			         // An ICMP echo reply message 
  ICMP_HEADERSIZE = 8;			         // ICMP header size ("echo messages" only) 

Type
  PICMP = ^ICMP;
  ICMP = Record					            // Structure for an ICMP header
    icmp_type,						          // Type of message
    icmp_code: Byte;				   	    // Type "sub code" (zero for echos)
    icmp_cksum,					            // 1's complement checksum
    icmp_id,							          // Unique ID (our handle)
    icmp_seq: Word;						      // Datagram sequence number
    icmp_data: Array[0..1] Of Byte;	// Start of the optional data
  End;

  PIP = ^IP;
  IP = Record                       // Structure for IP datagram header
     ip_verlen: Byte;		            // Version and header length
     ip_tos: Byte;			            // Type of service
     ip_len: Word;			            // Total packet length
     ip_id: Word;			              // Datagram identification
     ip_fragoff: Word;		          // Fragment offset
     ip_ttl,					              // Time to live
     ip_proto: Byte;		            // Protocol
     ip_chksum: Word;		            // Checksum
     ip_src_addr,			              // Source address
     ip_dst_addr: TInAddr;          // Destination address
     ip_data: Array[0..1] Of Byte;	// Variable length data area
  End;

var
  LastWsaError: Integer;

function IsDottedDecimal(strAddress: String): Boolean;
function LookupHostBlocking(UserEntry: String): PHostEnt;
function SocketErrorString(Error: integer) : string;

function ParseLine(Input: String; Delim: Char): TStringList;

function LoadRemoteAddress(HostName: String; var HostAddr: TSockAddrIn): Boolean;
function ConnectSocket(HostName : String; PortNo: Integer): TSocket;
function BindToPort(hSocket: TSocket; Name: TSockAddr) : Boolean;
procedure GetLocalHostInfo(var pLocalHostInfo : PhostEnt);
function GetHostConnectionInfo(hSocket: TSocket; var Name: TSockAddr): Boolean;
function GetPeerConnectionInfo(hSocket: TSocket; var Name: TSockAddr): Boolean;

implementation

function IsDottedDecimal(strAddress: String): Boolean;
var
  DotList: TStringList;
  I: Integer;
begin
  I := 1;
  While Result AND (I <= Length(strAddress)) DO
  Begin
    Result := strAddress[I] IN ['0'..'9', '.'];
    Inc(I);
  End;
  If Result THEN
  Begin
    DotList := ParseLine(strAddress, '.');
    If DotList.Count = 4 THEN
       Result := True
    ELSE
       Result := False;
    DotList.Free;
  End;
end;


function LookupHostBlocking(UserEntry: String): PHostEnt;
var
  dwIPAddr: LongInt;        // IP address as an unsigned long 
  pHostEntry: PHostEnt;     // Pointer to an Internet host data structure 
  szTemp,
  szUserEntry: PChar;
  Len: Integer;
begin
	pHostEntry := Nil;
  Result := Nil;

  Len := Length(UserEntry);
  Inc(Len);
  GetMem(szUserEntry, Len);

  StrPCopy(szUserEntry, UserEntry);
  // Check if a dotted-decimal address 
  If IsDottedDecimal(UserEntry) THEN
  Begin
    dwIPAddr := inet_addr(szUserEntry);
    pHostEntry := gethostbyaddr(PChar(@dwIPAddr), 4, PF_INET);
  End
  ELSE
    // If it wasn't a dotted-decimal address, assume it's a host name 
    pHostEntry := gethostbyname(szUserEntry);

	// If the host entry is valid, copy it to the global
  // variable before calling any other Winsock functions 

  If (pHostEntry <> Nil) THEN
     // Do not dispose of this value.  Winsock will clean it up
     Result := pHostEntry;
  FreeMem(szUserEntry, Len);
end;

//  Returns a descriptive string based on the supplied error code
function SocketErrorString(Error: integer) : string;
begin
  case Error of
    WSAEINTR:
      Result := 'Interrupted system call';
    WSAEBADF:
      Result := 'Bad file number';
    WSAEACCES:
      Result := 'Permission denied';
    WSAEFAULT:
      Result := 'Bad address';
    WSAEINVAL:
      Result := 'Invalid argument';
    WSAEMFILE:
      Result := 'Too many open files';
    WSAEWOULDBLOCK:
      Result := 'Operation would block';
    WSAEINPROGRESS:
      Result := 'Operation now in progress';
    WSAEALREADY:
      Result := 'Operation already in progress';
    WSAENOTSOCK:
      Result := 'Socket operation on non-socket';
    WSAEDESTADDRREQ:
      Result := 'Destination address required';
    WSAEMSGSIZE:
      Result := 'Message too long';
    WSAEPROTOTYPE:
      Result := 'Protocol wrong type for socket';
    WSAENOPROTOOPT:
      Result := 'Protocol not available';
    WSAEPROTONOSUPPORT:
      Result := 'Protocol not supported';
    WSAESOCKTNOSUPPORT:
      Result := 'Socket type not supported';
    WSAEOPNOTSUPP:
      Result := 'Operation not supported on socket';
    WSAEPFNOSUPPORT:
      Result := 'Protocol family not supported';
    WSAEAFNOSUPPORT:
      Result := 'Address family not supported by protocol family';
    WSAEADDRINUSE:
      Result := 'Address already in use';
    WSAEADDRNOTAVAIL:
      Result := 'Can''t assign requested address';
    WSAENETDOWN:
      Result := 'Network is down';
    WSAENETUNREACH:
      Result := 'Network is unreachable';
    WSAENETRESET:
      Result := 'Network dropped connection on reset';
    WSAECONNABORTED:
      Result := 'Software caused connection abort';
    WSAECONNRESET:
      Result := 'Connection reset by peer';
    WSAENOBUFS:
      Result := 'No buffer space available';
    WSAEISCONN:
      Result := 'Socket is already connected';
    WSAENOTCONN:
      Result := 'Socket is not connected';
    WSAESHUTDOWN:
      Result := 'Can''t send after socket shutdown';
    WSAETOOMANYREFS:
      Result := 'Too many references: can''t splice';
    WSAETIMEDOUT:
      Result := 'Connection timed out';
    WSAECONNREFUSED:
      Result := 'Connection refused';
    WSAELOOP:
      Result := 'Too many levels of symbolic links';
    WSAENAMETOOLONG:
      Result := 'File name too long';
    WSAEHOSTDOWN:
      Result := 'Host is down';
    WSAEHOSTUNREACH:
      Result := 'No route to host';
    WSAENOTEMPTY:
      Result := 'Directory not empty';
    WSAEPROCLIM:
      Result := 'Too many processes';
    WSAEUSERS:
      Result := 'Too many users';
    WSAEDQUOT:
      Result := 'Disc quota exceeded';
    WSAESTALE:
      Result := 'Stale NFS file handle';
    WSAEREMOTE:
      Result := 'Too many levels of remote in path';
    WSASYSNOTREADY:
      Result := 'Network sub-system is unusable';
    WSAVERNOTSUPPORTED:
      Result := 'WinSock DLL cannot support this application';
    WSANOTINITIALISED:
      Result := 'WinSock not initialized';
    WSAHOST_NOT_FOUND:
      Result := 'Host not found';
    WSATRY_AGAIN:
      Result := 'Non-authoritative host not found';
    WSANO_RECOVERY:
      Result := 'Non-recoverable error';
    WSANO_DATA:
      Result := 'No Data';
    ELSE
      Result := 'Not a WinSock error';
  end;
end;

function ConnectSocket(HostName: String; PortNo: Integer): TSocket;
  // Connect to the HostName:PortNo given.
var
  RemoteConnectAddress : TSockAddrIn; // The remote address.
  sResult: Integer;
begin
  RemoteConnectAddress.sin_family := AF_INET;
  RemoteConnectAddress.sin_port := htons(PortNo);

  // Load the sin_addr portion of RemoteConnectAddress.
  If Not LoadRemoteAddress(HostName, RemoteConnectAddress) THEN
  Begin
    LastWsaError := WsaGetLastError;
    Result := INVALID_SOCKET;
  End
  ELSE
  Begin

    // Get a socket to connect with.
    Result := socket(PF_INET, SOCK_STREAM, IPPROTO_IP);
    If Result = INVALID_SOCKET THEN
       LastWsaError := WsaGetLastError
    ELSE
    Begin
      // Connect to the remote host
      sResult := connect(Result, RemoteConnectAddress,
                         SizeOf(RemoteConnectAddress));
      If sResult <> 0 THEN
      Begin
        LastWsaError := WsaGetLastError;
        CloseSocket(Result);
        Result := INVALID_SOCKET;
      End;
    End;
  End;
end;

// Returns a list of tokens from the string
function ParseLine(Input: String; Delim: Char): TStringList;
var
  P: Integer;
  S,
  A: String;
begin
  Result := TStringList.Create;
  S := Input;
  P := Pos(Delim, S);
  While P > 0 DO
  Begin
    A := Copy(S, 1, P - 1);
    Result.Add(Trim(A));
    S := Copy(S, P + 1, Length(S));
    P := Pos(Delim, S);
  End;
  If Length(S) > 0 THEN
     Result.Add(Trim(S));
end;

function LoadRemoteAddress(HostName: String; var HostAddr: TSockAddrIn): Boolean;
  // Accept HostName as "foo.com" or "206.181.78.36" and put the correct
  // address into HostAddr.
var
  pRemoteHostInfo      : PHostEnt;
  pRemoteHostName      : PChar;
begin
  Result := False;
  pRemoteHostInfo := LookupHostBlocking(HostName);
  If pRemoteHostInfo <> NIL THEN
  Begin
    // Get the address from pRemoteHostInfo.
    HostAddr.sin_addr.S_un_B.s_b1 := pRemoteHostInfo^.h_addr_list^[0];
    HostAddr.sin_addr.S_un_B.s_b2 := pRemoteHostInfo^.h_addr_list^[1];
    HostAddr.sin_addr.S_un_B.s_b3 := pRemoteHostInfo^.h_addr_list^[2];
    HostAddr.sin_addr.S_un_B.s_b4 := pRemoteHostInfo^.h_addr_list^[3];
    Result := true;
  End;
end;


// Binds socket to a port.  Name contains the local address and
// port number
function BindToPort(hSocket: TSocket;
                    Name: TSockAddr) : Boolean;
// Binds the Socket to a port.
var
  iResult: Integer;
begin
  Result := False;
  If hSocket = INVALID_SOCKET THEN
     Exit;

  iResult := bind(hSocket, Name, SizeOf(Name));
  If iResult <> -1 THEN
    Result := True
  ELSE
    LastWsaError := WsaGetLastError;
end;

procedure GetLocalHostInfo(var pLocalHostInfo : PhostEnt);
// Retrieves the local host information.
var
  iResult: Integer;
  pLocalHostName : PChar;
begin
  GetMem( pLocalHostName, 100);
  iResult := getHostName(pLocalHostName, 100);
  pLocalHostInfo := gethostbyname(pLocalHostName);
  FreeMem(pLocalHostName, 100);
end;

// Returns the connection info for a socket connected to a
// particular host
function GetHostConnectionInfo(hSocket: TSocket; var Name: TSockAddr): Boolean;
var
  NameLen: Integer;
begin
  NameLen := SizeOf(TSockAddr);
  Result := getsockname(hSocket, Name, NameLen) = 0;
end;

function GetPeerConnectionInfo(hSocket: TSocket; var Name: TSockAddr): Boolean;
var
  NameLen: Integer;
begin
  NameLen := SizeOf(TSockAddr);
  Result := getpeername(hSocket, Name, NameLen) = 0;
end;

end.

