{++

  TAPI Support Kit for Delphi
  Version 3.0  Copyright (c) 1995-1997 by Alexander Staubo

  TAPI utility functions.

  Conditional Compiler Defines
  ----------------------------

     Tapi_NoErrorStrings   When defined, this unit will obtain error
                           descriptions from the TAPI DLL itself. For some
                           reason, these error strings are not present
                           in the Windows95 TAPI binaries. These built-in
                           strings (obtained from the NT 4.0 binaries) will
                           only be used in cases where the error string could
                           not be obtained from TAPI.
 --}

unit Tapiutil;

interface

uses
  SysUtils,
  Tapih;

{ Exception classes }

type
  ETapiError =
    class(Exception)
    public
      ErrorCode : Longint;
    end;

{ Helper functions }

function TapiErrorMsg (ResultCode : Longint) : string;
  { Returns error message for a result code }

procedure TapiCheck (ResultCode : Longint);
  { Raises an exception if ResultCode indicates an error }

procedure RaiseTapiError (ErrorCode : Longint);
  { Raises a TAPI exception }

implementation

uses
  Windows;

{ Publics }

function TapiErrorMsg (ResultCode : Longint) : string;
{$IFNDEF Tapi_NoErrorStrings}
const
  LineErrorStrings : array[$80000001..$8000005C] of string =
    (
      'The line device is already in use',
      'Invalid line device ID',
      'The requested bearer mode is unavailable',
      '',
      'No call appearance available',
      'Too many call completions outstanding',
      'The conference is full',
      'The ''''$'''' dial modifier is not supported',
      'The ''''W'''' dial modifier is not supported',
      'The ''''?'''' dial modifier is not supported',
      'The ''''@'''' dial modifier is not supported',
      'Incompatible API version',
      'Incompatible extension version',
      'The TAPI configuration information is unusable',
      'The line device is already in use',
      'The phone number is invalid or not properly formatted',
      'Invalid address ID',
      'Invalid address mode',
      'Operation not permitted in current address state',
      'Invalid TAPI line application handle',
      'Invalid application name',
      'Invalid bearer mode',
      'Invalid call completion mode',
      'Invalid call handle',
      'Invalid LINECALLPARAMS structure',
      'Invalid call privilege',
      'Invalid call select parameter',
      'Operation not permitted in current call state',
      'Invalid call state list',
      'Invalid calling card ID',
      'Invalid call completion ID',
      'Invalid conference call handle',
      'Invalid consultation call handle',
      'Invalid country code',
      'Invalid device class identifier',
      'Invalid device handle',
      'Invalid dialing parameters',
      'Invalid digit list',
      'Invalid digit mode',
      'Invalid digits',
      'Invalid extension version',
      'Invalid group pickup ID',
      'Invalid line handle',
      'Operation not permitted in current line state',
      'Invalid location ID',
      'Invalid media list',
      'Invalid media mode',
      'Invalid message ID',
      '',
      'Invalid parameter',
      'Invalid park ID',
      'Invalid park mode',
      'Invalid pointer',
      'Invalid call privilege selection',
      'Invalid rate',
      'Invalid request mode',
      'Invalid terminal ID',
      'Invalid terminal mode',
      'Invalid timeout value',
      'Invalid tone',
      'Invalid tone list',
      'Invalid tone mode',
      'Invalid transfer mode',
      'No device matches the specified requirements',
      'The call is not part of a conference',
      'The device was removed, or the device class is not recognized',
      'The service provider was removed',
      'Insufficient memory available to complete the operation',
      'No Assisted Telephony requests are pending',
      'The application is does not have OWNER privilege on the call',
      'The application is not registered to handle requests',
      'The operation failed for unspecified reasons',
      'The operation is not supported by the underlying service provider',
      'The requested data rate is not available',
      'A resource needed to fulfill the request is not available',
      'The request queue is already full',
      'The application failed to allocate sufficient memory for the minimum structure size',
      'The call handoff failed because the specified target was not found',
      'No higher priority target exists for the call handoff',
      'The telephony service has not been initialized',
      'The amount of user-user info exceeds the maximum permitted',
      'The operation cannot be completed until all TAPI applications call lineShutdown',
      'You are not permitted to call this number',
      'The calling card number or other billing information was rejected',
      'Invalid device-specific feature',
      'You cannot have two instances of the same service provider',
      'Invalid agent ID',
      'Invalid agent group',
      'Invalid agent password',
      'Invalid agent state',
      'Invalid agent activity',
      'The '''':'''' dial modifier is not supported'
    );

  PhoneErrorStrings : array[$90000001..$90000023] of string =
    (
      'The phone device is already in use',
      'Invalid phone device ID',
      'Incompatible API version',
      'Incompatible extension version',
      'The TAPI configuration information is unusable',
      'The phone device is already in use',
      'Invalid TAPI phone application handle',
      'Invalid application name',
      'Invalid button or lamp ID',
      'Invalid button mode',
      'Invalid button state',
      'Invalid data segment ID',
      'Invalid device class identifier',
      'Invalid extension version',
      'Invalid hookswitch device ID',
      'Invalid hookswitch mode',
      'Invalid lamp mode',
      'Invalid parameter',
      'Invalid phone handle',
      'Operation not permitted in current phone state',
      'Invalid pointer',
      'Invalid privilege',
      'Invalid ring mode',
      'The device was removed, or the device class is not recognized',
      'The service provider was removed',
      'Insufficient memory available to complete the operation',
      'The application is does not have OWNER privilege on the phone',
      'The operation failed for unspecified reasons',
      'The operation is not supported by the underlying service provider',
      '',
      'A resource needed to fulfill the request is not available',
      'The request queue is already full',
      'The application failed to allocate sufficient memory for the minimum structure size',
      'The telephony service has not been initialized',
      'The operation cannot be completed until all TAPI applications call phoneShutdown'
    );

  TapiErrorStrings : array[-18..-1] of string =
    (
      'Invalid pointer',
      'The request was cancelled',
      'The request failed for unspecified reasons',
      'Unknown request ID',
      'Unknown window handle',
      'The called number could not be reached',
      'The called party does not answer',
      'The called number is busy',
      'The device is already in use',
      'The specified device is unavailable',
      'The device class is unavailable',
      'Invalid device ID',
      'Invalid device class',
      'Invalid window handle',
      'The phone number is invalid or improperly formatted',
      'The queue of call requests is full',
      'No program is available to handle the request',
      'The call was disconnected'
    );
{$ENDIF}
var
  Buffer : array[0..512] of Char;
begin
  if TapiFailed(ResultCode) then
    begin
      { Convert error code to a string }
      if FormatMessage(FORMAT_MESSAGE_FROM_HMODULE,
                       Pointer(GetModuleHandle(TapiDll)),
                       TAPIERROR_FORMATMESSAGE(ResultCode),
                       0,
                       @Buffer,
                       SizeOf(Buffer),
                       nil) <> 0 then
        Result:=string(Buffer)
      else
        begin
          {$IFNDEF Tapi_NoErrorStrings}
          if ResultCode > $FFFF0000 then
            Result:=TapiErrorStrings[ResultCode]
          else if ResultCode >= $80000000 then
            begin
              if ResultCode and $10000000 <> 0 then
                Result:=PhoneErrorStrings[ResultCode]
              else
                Result:=LineErrorStrings[ResultCode];
            end
          else
            Result:='';
          if Result = '' then
          {$ENDIF}
            Result:=Format('Unknown TAPI error %.8x', [ResultCode]);
        end;
    end
  else
    begin
      Result:=SysErrorMessage(0);
    end;
end;

procedure TapiCheck (ResultCode : Longint);
begin
  if TapiFailed(ResultCode) then
    RaiseTapiError(ResultCode);
end;

procedure RaiseTapiError (ErrorCode : Longint);
var
  E : ETapiError;
begin
  E:=ETapiError.CreateFmt('TAPI error: %d', [TapiErrorMsg(ErrorCode)]);
  E.ErrorCode:=ErrorCode;
  raise E;
end;

end.

