unit MSPing;
{
  Written By:    Martien Verbruggen
  Date:          12 July, 1996 - 21 August, 1996
  Version:       1.12
  Copyright:     (C) Copyright Martien Verbruggen, 1996. All Rights Reserved
  Email:         tgtcmv@chem.tue.nl
  www:           http://www.tcp.chem.tue.nl/~tgtcmv/
  Contents:      Components TMSPing and TMSTrace.

  Description:   These components provide an interface to the MS Winsock
                 icmp.dll functions, needed to perform ICMP Echo requests.
                 Because of the blocking nature of these icmp calls, these
                 components can spawn a thread to do the sending of and waiting
                 for the package.

  Prerequisites: You must have the Microsoft 32 bit Winsock installed and
                 the WSOCK32.DLL and ICMP.DLL available. This code has been
                 tested under Windows 95.

  Modifications: Version 1.12 - 21 August 1996
                 - Improved stability, here and there
                 - Resolved problem with global object
                 - moved initialisation of winsock back into component to
                   avoid unnecessary winsock calls while designing
                 Version 1.1 - 22 July 1996
                 - Improved Error Handling
                 - Streamlined source a lot
                 - Added some blocking routines.. Dunno if anyone would want it.
                 - Moved initialisation of the icmp library to the initialisation
                   part of this unit.
                 Version 1.0 - 19 July 1996
                 - Added static DNS object
                 - Changed WSExtra so I could initiate winsock only once,
                   instead of having it done for each object. Added globals
                   there, WSInitData (Data returned from Winsock) and
                   WSInitNumber (Number of times that Winsock was tried to load).
                 - Split TMSicmp in smaller class, for just one packet, and
                   Class TMSicmpTask to wrap Trace and Ping.
                 Version 0.98 - 18 July 1996
                 - Improved Error Handling
                 - Tested several things
                 - Increased version number :)
                 Version 0.95 - 17 July 1996
                 - Changed WSExtra.pas to do DNS lookups, so I could remove
                   Any Winsock stuff from this unit
                 Version 0.9 - 16 July, 1996
                 - Added Properties SendTestpacket and ExpectedHops and
                   OnTest Event to TMSTrace.
                 - Put OnPingFinished and OnTraceFinished in TMSicmp as
                   OnFinished, changed type to simple TNotifyEvent.

  Properties:    BOTH
                   Address:     (Read Only) IP network byte order address
                   Cancelled:   (Read Only) True if the process is cancelled,
                                useful in the error event handler.
                   Destination: Sets the hostname of the machine to contact and
                                then starts contacting it.
                   IPAddress:   (Read Only) dotted decimal Address of the host
                                to contact
                   InProgress:  (Read Only) True if a process is in progress
                                Component uses this to see if it's already
                                working, and will refuse to start a second cycle.
                   Mode:        wsmBlocking or wsmAsync.
                   PacketSize:  The size of the icmp data packets to be sent out.
                   Resolve:     True if you want the responding host IP's
                                resolved back to hostnames.
                   ReturnName:  (Read Only) Name of Host that responded
                   ReturnIP:    (Read Only) IP of host that responded
                   TimeOut:     Time to wait for a packet return (millisecs)
                   TimeToLive:  Maximum number of hops to take.
                   TTL:         (Read Only) TTL of packet that returned
                   RTT:         (Read Only) Round Trip Time of packet that returned

                 TMSPing
                   Average      (Read Only) Average RTT
                   Max:         (Read Only) Max RTT
                   Min:         (Read Only) Minimum RTT
                   Number:      Number of Packets to send out
                   Received:    (Read Only) How Many Received
                   Sent:        (Read Only) How many sent

                 TMSTrace
                   ExpectedHops:
                                (Read Only) Set to expected hop count if
                                SendTestPacket = True, else 0.
                   HopCount:    (Read Only) Number of hops so far
                   SendTestPacket:
                                TMSTrace will send a regular ping first, to
                                determine if the host is reachable, and to
                                determine the expected hop count;

  Events:        BOTH
                   OnData:      Called when a packet returns
                   OnError:     Called when an error occurs
                   OnFinished:  Called when the request is at the end or
                                cancelled
                 TMSTrace
                   OnTest:      Called when a test packet returns. This only
                                happens of course when the property
                                SendtestPacket = True.

  Methods:       BOTH
                   Cancel:      Cancels and terminates the request
                   Create:      registers with winsock, sets defaults
                   Destroy:     unregisters with winsock
                   GetData:     Gets a string describing the result of the last
                                packet (Use with OnData)
                   GetStats:    Gets a string describing the total result
                                (Use with OnXXXXFinished)
                   Connect(Host: String):
                                Start the process by setting Destination.

  ToDo:          - Write an implementation for RAW_SOCKET support, for for example
                   Trumpet Winsock, or future releases of the MS Winsock. Problem:
                   Where do I test it? :)
                 - Make Sure TMSPing and TMSTrace don't do things TMSicmpTask
                   should do, like setting the InProgress.
                 - Create better property handlers. e.g don't allow any
                   ridiculous numbers.
}

interface

uses
  WinTypes, Classes, Controls, Winsock, WSExtra, MSIcmp;

const
  IP_TRACE            = IP_TTL_EXPIRED_TRANSIT;
  IP_REACHED          = IP_SUCCESS;
  IP_TIMEOUT          = IP_REQ_TIMED_OUT;
  ICMP_STATUS_BASE    = $80000100;
  ICMP_INVALID        = ICMP_STATUS_BASE + 0;
  ICMP_NO_ADDRESS     = ICMP_STATUS_BASE + 1;
  ICMP_CANCEL         = ICMP_STATUS_BASE + 3;
  ICMP_BUSY           = ICMP_STATUS_BASE + 4;
  TTL_MAX             = 255;
  MIN_PACKET_SIZE     = 8;

type
  // Will do one ping, in a separate Thread, and then return
  TPingThread = class(TThread)
  private
    hICMP:      THandle;              // Handle for the ICMP Calls
    FAddress:   DWord;                // destination address
    FReqSize:   DWord;                // Size of request packet
    FpER:       PIcmpEchoReply;       // Echo Reply
    FIPOptions: TIPOptionInformation; // IP options of incoming packet
    FTimeOut:   DWord;                // Time to wait for a packet
    FpStatus:   ^Integer;
  protected
    procedure Execute; override;
  public
    constructor Create(var Status: Integer; IPOpt: TIPOptionInformation;
                       Addr, Size, Timeout: DWord; var IPE: TIcmpEchoReply );
    destructor Destroy; override;
  end;

  TError = procedure (Sender: TObject; Error: Integer; Msg: String) of object;
  TDataRt = procedure (Sender: TObject; Status: Integer) of object;

  TMSicmp = Class(TComponent)
  private
    FAddress:      DWord;                 // Host to ping
    FDataRt:       TDataRt;               // Called on return of ping packet
    FError:        TError;                // Called on Error
    FIPOptions:    TIPOptionInformation;  // IP option Information
    FMessage:      String;                // Message describing Status
    FMode:         TWSMode;               // Blocking or Async Mode
    FPacketSize:   DWord;                 // Length of Ping Data packet
    FTimeOut:      Word;                  // How long will we wait?
    FTTL:          Byte;                  // Maximum number of hops
    procedure IcmpError(Errno: Integer; Msg: String);
    procedure SetPacketSize(Size: DWord);
    function  GetAsync: Boolean;
    function  CheckPacket: Boolean;
    procedure NotifyData;
    procedure SendPacket;
  protected
    FStatus:       Integer;               // Status of the return, if not 0 something went seriously wrong
    FIPEcho:       TIcmpEchoReply;        // Echo reply
    procedure GotPacket(Sender: TObject); virtual;
    procedure Reset; virtual;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    property  Address: DWord        read FAddress    write FAddress;
    property  Async: Boolean        read GetAsync;
    property  RoundTripTime:Integer read FIPEcho.RTT;
    property  TTL: Byte             read FIPEcho.Options.TTL;
  published
    property  Mode: TWSMode         read FMode       write FMode         default wsmAsync;
    property  OnError: TError       read FError      write FError;
    property  PacketSize: DWord     read FPacketSize write SetPacketSize default 56;
    property  TimeOut: Word         read FTimeOut    write FTimeOut      default 2500;
    property  TimeToLive: Byte      read FTTL        write FTTL          default 32;
  end;

  TMSicmpTask = Class(TMSicmp)
  private
    FHostName:    String;           // Name of the host to ping
    FHostIP:      String;           // IP of the Host to ping
    FResolve:     Boolean;          // Resolve names of answering hosts?
    FReturnName:  String;           // Name of the returning Host
    FReturnIP:    String;           // IP of the Returning host
    FCancelled:   Boolean;          // Set to cancel cycle
    FFinished:    TNotifyEvent;     // Called when finished
    FInProgress:  Boolean;          // True if currently working
    procedure GotHost(Sender: TObject);
    procedure GotPacket(Sender: TObject); override;
    procedure ResolveAnswer;
    procedure SetHostName(InName: String);
    procedure SetBusy;
    procedure UnSetBusy;
  protected
    procedure Initialise; virtual; abstract;
    procedure Initialised;
    procedure ProcessPacket; virtual;
    function  Increment: Boolean; virtual;
    procedure Finish; virtual;
  public
    FDNS:         TDNS;             // Used for lookups
    constructor Create(AOwner: TComponent); override;
    constructor CreateWithParent(AOwner: TComponent; AParent: TWinControl);
    procedure Connect(Host: String); virtual;
    procedure Cancel; // Cancels the cycle when the next packet returns
    property  Cancelled: Boolean       read FCancelled;
    property  Destination: String      read FHostName write SetHostName;
    property  InProgress: Boolean      read FInProgress;
    property  IPAddress: String        read FHostIP;
    property  ReturnName: String       read FReturnName;
    property  ReturnIP: String         read FReturnIP;
  published
    property  Resolve: Boolean         read FResolve  write FResolve default False;
    property  OnData: TDataRt          read FDataRt   write FDataRt;
    property  OnFinished: TNotifyEvent read FFinished write FFinished;
  end;

  TMSPing = Class(TMSicmpTask)
  private
    FNumber:  Word;               // Number of Ping Packets to send
    FSent:    Word;               // How Many Sent?
    FRcvd:    Word;               // How Many received back?
    FMin:     Word;               // Minimal Ping time
    FMax:     Word;               // Maximal Ping time
    FAverage: Word;               // Mean Ping time
    procedure Initialise; override;
    procedure ProcessPacket; override;
    function  Increment: Boolean; override;
  public
    constructor Create(AOwner: TComponent); override;
    function  GetData: String;
    function  GetStats: String;
    property  Average: Word  read FAverage;
    property  Max: Word      read FMax;
    property  Min: Word      read FMin;
    property  Received: Word read FRcvd;
    property  Sent: Word     read FSent;
  published
    property  Number: Word   read FNumber write FNumber default 5;
  end;

  TMSTrace = Class(TMSicmpTask)
  private
    FExpected: Byte;               // Expected number of hops
    FSendTest: Boolean;            // Send a Test packet to see if it's reachable, and determine expected hops?
    FOnTest:   TNotifyEvent;       // Called when test packet returns
    procedure GotTest(Sender: TObject);
    procedure NotifyTest;
    procedure Initialise; override;
    function  Increment: Boolean; override;
  public
    constructor Create(AOwner: TComponent); override;
    function  GetData: String;
    function  GetStats: String;
    property  ExpectedHops: Byte      read FExpected;
    property  HopCount: Byte          read FIPOptions.TTL;
  published
    property  SendTestPacket: Boolean read FSendTest write FSendtest default False;
    property  OnTest: TNotifyEvent    read FOnTest   write FOnTest;
  end;

function DoPing(hIP: THandle; pIPOpt: PIPOptionInformation; pEcho: PIcmpEchoreply; Address, Size, TimeOut: DWord): Integer;
procedure Register;

implementation

uses
  SysUtils, Dialogs, Forms;

procedure Register;
begin
  RegisterComponents('WinSock', [TMSPing, TMSTrace]);
end;

var
  IcmpCreateFile : TIcmpCreateFile;
  IcmpCloseHandle: TIcmpCloseHandle;
  IcmpSendEcho:    TIcmpSendEcho;
  IcmpValid:       Boolean;

// Ping Thread Stuff

constructor TPingThread.Create(var Status: Integer;
                               IPOpt: TIPOptionInformation;
                               Addr, Size, Timeout: DWord;
                               var IPE: TIcmpEchoReply );
begin
  inherited Create(False);
  FpStatus := @Status; FIPOptions := IPOpt; FAddress := Addr;
  FReqSize := Size; FTimeOut := TimeOut; FpER := @IPE;
  hICMP := IcmpCreateFile;
  if hICMP = INVALID_HANDLE_VALUE then begin
    Status := ICMP_INVALID; Terminate;
  end;
end;

destructor TPingThread.Destroy;
begin
  IcmpCloseHandle(hICMP);
  inherited Destroy;
end;

procedure TPingThread.Execute;
begin
  if Terminated then exit;
  FpStatus^ := DoPing(hICMP, @FIPOptions, FpER, FAddress, FReqSize, FTimeOut);
end;

// Icmp Class

constructor TMSicmp.Create(AOwner: TComponent);
begin
  inherited;
  FPacketSize := 56; FTimeOut := 2500; TimeToLive := 32;
  FAddress := INADDR_NONE; Mode := wsmAsync; FIPOptions.TTL := TimeToLive;
  if not ( csDesigning in ComponentState ) and not WSInitialised then begin
    if WSAStartup(WSVersionRqrd,WSInitdata) = 0 then begin
      wsInitialised := True;
      wsInitialiser := Self;
    end else
      if MessageDlg(WSErrorString(WSAGetLastError), mtError,
         [mbIgnore, mbAbort],0) = mrAbort then halt;
  end;
end;

destructor TMSIcmp.Destroy;
begin
  if wsInitialiser = Self then
    if WSACleanup = 0 then begin
      wsInitialised := False;
      wsINitialiser := nil;
    end else
      ShowMessage(WSErrorString(WSAGetLastError));
  inherited;
end;

procedure TMSicmp.SendPacket;
var
  hICMP: THandle;
begin
  Reset;
  if Async then begin
    with TPingThread.Create(FStatus, FIPOptions, Address, PacketSize, TimeOut, FIPEcho ) do begin
      OnTerminate := GotPacket;
      FreeOnTerminate := True;
    end;
  end else begin
    hICMP := IcmpCreateFile;
    if hICMP <> INVALID_HANDLE_VALUE then begin
      FStatus := DoPing(hICMP, @FIPOptions, @FIPEcho, Address, PacketSize, TimeOut);
      IcmpCloseHandle(hICMP);
    end else FStatus := ICMP_INVALID;
    GotPacket(Self);
  end;
end;

procedure TMSicmp.Reset;
begin
  FStatus := 0;
  with FIPEcho do begin
    Address := INADDR_NONE; Status := 0; RTT := 0; DataSize := 0; Data := nil;
    with Options do begin
      TTL := 0; TOS := 0; Flags := 0; OptionsSize := 0; OptionsData := nil;
    end;
  end
end;

procedure TMSicmp.GotPacket(Sender:TObject);
begin
  if CheckPacket then Notifydata  else ICMPError(FStatus, FMessage);
end;

// Tricky Error messages. If it returned with an error, it could be
// critical and non critical. hard to distinguish. One thing I do know here,
// is that if the error is a winsock error, that we're in trouble. I'll
// have to assume that all other errors are non critical, and that I
// programmed the ping part well enough.
function TMSicmp.CheckPacket;
begin
  case FStatus of
    0: begin  // A Packet returned
      Result := True;
      FStatus := FIPEcho.Status; FMessage := IPErrorString(FStatus);
    end;
    // Nothing returned
    WSABASEERR..IP_STATUS_BASE: begin
      Result := False;
      FMessage := WSErrorString(FStatus);
    end;
    ICMP_INVALID: begin
      Result := False;
      FMessage := 'Unable to get a handle for the Ping';
    end;
    // All the rest should be ok, and handled by the rest of the component
    else begin
      Result := True;
      FIPEcho.Status := FStatus; FMessage := IPErrorString(FStatus);
    end;
  end;
end;

procedure TMSicmp.NotifyData;
begin
  if Assigned(FDataRt) then FDataRt(Self, FStatus);
end;

procedure TMSicmp.IcmpError(Errno: Integer; Msg: String);
begin
  if Assigned(FError) then FError(Self, Errno, Msg)
                      else ShowMessage('Ping Error ' + IntToStr(Errno) + ': ' + Msg);
end;

Procedure TMSicmp.SetPacketSize(Size: DWord);
begin
  if Size < MIN_PACKET_SIZE then FPacketSize := MIN_PACKET_SIZE
                            else FPacketSize := Size;
end;

function TMSicmp.GetAsync;
begin
  if FMode = wsmAsync then Result := True else Result := False;
end;

// ICMP Task Class

constructor TMSicmpTask.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FCancelled := False; FInProgress := False; FHostName := ''; FHostIP := '';
  if FDNS = nil then begin
    FDNS := TDNS.Create(Self);
    if (AOwner <> nil) and (AOwner.InheritsFrom(TWinControl)) then
      FDNS.Parent := TWinControl(AOwner);
  end;
end;

constructor TMSicmpTask.CreateWithParent(AOwner: TComponent; AParent: TWinControl);
begin
  Create(AOwner);
  FDNS := TDNS.Create(Self);
  FDNS.Parent := AParent;
end;

procedure TMSicmpTask.Connect(Host: String);
begin
  Destination := Host;
end;

procedure TMSicmpTask.Cancel;
begin
  FCancelled := True; FDNS.Cancel;
end;

procedure TMSicmpTask.SetHostName(InName: String);
begin
  if not IcmpValid then begin
    IcmpError(ICMP_INVALID, 'ICMP disabled: Winsock or ' + icmpDLL + ' not found');
    Finish; exit;
  end;
  if InProgress then
    IcmpError(ICMP_BUSY, 'Already in Progress')
  else begin
    SetBusy; FCancelled := False; FStatus := 0; FMessage := '';
    with FDNS do begin
      Mode := FMode;
      Onresolved := GotHost;
      FDNS.HostName := InName;
    end;
  end;
end;

procedure TMSicmpTask.GotHost(Sender: TObject);
begin
  with FDNS do begin
    FAddress := Address; FHostName := HostName; FHostIP := HostIP;
    FStatus := Status; FMessage := Msg;
  end;
  if FStatus = 0 then
    Initialise
  else begin
    IcmpError(FStatus, FMessage); UnSetBusy;
  end;
  Application.Processmessages;
end;

procedure TMSicmpTask.Initialised;
begin
  SendPacket;
end;

procedure TMSicmpTask.GotPacket(Sender:TObject);
begin
  Application.Processmessages;
  if CheckPacket then begin
    ResolveAnswer;
    ProcessPacket;
    NotifyData;
    if not Cancelled and Increment then SendPacket
                                   else Finish;
  end else
    ICMPError(FStatus, FMessage);
end;

procedure TMSicmpTask.ResolveAnswer;
var
  Phe: pHostEnt;
begin
  if FIPEcho.Address <> INADDR_NONE then begin
    FReturnIP := StrPas(inet_ntoa(TInAddr(FIPEcho.Address)));
    FReturnName := ReturnIP;
    if FResolve then begin
      Phe := GetHostByAddr(@FIPEcho.Address, 4, PF_INET);
      if Phe <> Nil then FReturnName := Phe.h_name;
    end;
  end else begin
    FReturnIP := FHostIP; FReturnName := FHostName;
  end;
end;

procedure TMSicmpTask.ProcessPacket;
begin
  if Cancelled then FStatus := ICMP_CANCEL;
end;

function TMSicmpTask.Increment;
begin
  Result := False;
end;

procedure TMSicmpTask.Finish;
begin
  if Assigned(FFinished) then FFinished(Self); UnsetBusy;
end;

procedure TMSicmpTask.SetBusy;
begin
  FInProgress := True;
end;

procedure TMSicmpTask.UnSetBusy;
begin
  FInProgress := False;
end;

// Ping Class

constructor TMSPing.Create(AOwner: TComponent);
begin
  inherited Create(AOwner); FNumber := 5;
end;

procedure TMSPing.Initialise;
begin
  FSent := 1; FRcvd := 0; FMin := 0; FMax := 0; FAverage := 0;
  FIPOptions.TTL := TimeToLive;
  Initialised;
end;

procedure TMSPing.ProcessPacket;
begin
  if FStatus = 0 then begin
    with FIPEcho do begin
      Inc(FRcvd);
      if FRcvd = 1 then begin Fmin := RTT; FMax := RTT; end;
      FAverage := Round((FAverage * (FRcvd-1) + RTT)/FRcvd);
      if RTT < FMin then FMin := RTT;
      if RTT > FMax then FMax := RTT;
    end;
  end;
  inherited ProcessPacket;
end;

function TMSPing.Increment: Boolean;
begin
  if (FSent < Number) then begin Inc(FSent); Result := True; end
                      else Result := False;
end;

function TMSPing.GetData: String;
begin
  if not InProgress then exit;
  case FStatus of
    IP_REACHED:
      Result := IntToStr(FIPEcho.DataSize) + ' bytes received from ' + ReturnName + '. time = ' + IntToStr(FIPEcho.RTT) + 'ms';
    ICMP_CANCEL:
      if FIPEcho.Status = IP_REACHED then
        Result := IntToStr(FIPEcho.DataSize) + ' bytes received from ' + ReturnName + '. time = ' + IntToStr(FIPEcho.RTT) + 'ms'
      else
        Result := ReturnName + ': ' + FMessage;
    else
      Result := ReturnName + ': ' + FMessage;
  end;
end;

function TMSPing.GetStats: String;
begin
  if InProgress then
    Result := 'Stats: Rcvd/Sent: ' + IntToStr(Received) + '/' + IntToStr(Sent) + ', Min/Avg/Max: ' + IntToStr(Min) + '/' + IntToStr(Average) + '/' + IntToStr(Max);
end;

// TMSTrace

constructor TMSTrace.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  Resolve := False; FExpected := 0; SendTestPacket := False;
end;

procedure TMSTrace.Initialise;
begin
  FExpected := 0;
  if FSendTest then begin
    if Faddress <> $100007F then begin
      FIPOptions.TTL := TimeToLive;
      with TPingThread.Create(FStatus, FIPOptions, Address, PacketSize, TimeOut, FIPEcho ) do begin
        OnTerminate := GotTest;
        FreeOnTerminate := True;
      end;
    end else
      with FIPEcho do begin
        FStatus := IP_REACHED; Options.TTL := TTL_MAX; GotTest(Self);
      end;
  end
  else begin
    FIPOptions.TTL := 1; Initialised;
  end;
end;

procedure TMSTrace.GotTest(Sender: TObject);
begin
  if CheckPacket and (FStatus = IP_REACHED) then begin
    with FIPEcho do begin
      FExpected := TTL_MAX + 1 - Options.TTL; NotifyTest; FIPOptions.TTL := 1;
      Initialised;
    end;
  end else begin
    IcmpError(FStatus, FMessage); UnSetBusy;
  end;
end;

function TMSTrace.Increment: Boolean;
begin
  if (FStatus = IP_SUCCESS) or (FIPOptions.TTL >= TimeToLive) then
    Result := False
  else begin
    Inc(FIPOptions.TTL); Result := True;
  end;
end;

procedure TMSTrace.NotifyTest;
begin
  if Assigned(FOnTest) then FOnTest(Self);
end;

function TMSTrace.GetData: String;
var
  Number: Integer;
begin
  if not InProgress then exit;
  Number := FIPOptions.TTL;
  with FIPEcho do
    case FStatus of
      IP_REACHED:
        Result := IntToStr(Number) + #09 + IntToStr(RTT) + #09 + ReturnName + #09 + 'reached';
      IP_TRACE:
        Result := IntToStr(Number) + #09 + IntToStr(RTT) + #09 + ReturnName;
      IP_TIMEOUT:
        Result := IntToStr(Number) + #09 + '*'+ #09 + 'timed out';
      ICMP_CANCEL:
        Result := '<cancel>';
      IP_DEST_HOST_UNREACHABLE:
        Result := 'From ' + ReturnName + ': ' + FMessage;
      else
       Result := ReturnName + ': ' + FMessage;
    end;
end;

function TMSTrace.GetStats: String;
var
  Number: Integer;
begin
  Number := FIPOptions.TTL;
  with FIPEcho do
    case FStatus of
      IP_REACHED:
        if FExpected = 0 then
          Result := ReturnName + ' reached in ' + IntToStr(Number) + ' hops.'
        else
          Result := ReturnName + ' reached in ' + IntToStr(Number) + ' hops. (expected: ' + IntToStr(FExpected) + ')';
      ICMP_CANCEL:
        Result := 'Trace Cancelled after ' + IntToStr(Number - 1) + ' hops.';
      else
       Result := ReturnName + ': ' + FMessage;
    end;
end;

function DoPing(hIP: THandle; pIPOpt: PIPOptionInformation; pEcho: PIcmpEchoreply; Address, Size, TimeOut: DWord): Integer;
var
  BufferSize, nPkts: Integer;
  pReqData, pData: Pointer;
  pIPE: PIcmpEchoReply;
begin
  Result := 0; BufferSize := SizeOf(TICMPEchoReply) + Size;
  GetMem(pReqData, Size); GetMem(pData, Size); GetMem(pIPE, BufferSize);
  FillChar(pReqData^, Size, $AA);  pIPE^.Data := pData;
  NPkts := IcmpSendEcho(hIP, Address, pReqData, Size, pIPOpt, pIPE, BufferSize, TimeOut);
  if NPkts = 0 then Result := GetLastError
               else pEcho^  := pIPE^;
  FreeMem(pIPE); FreeMem(pData); FreeMem(pReqData);
end;

var
  LibLoad: Integer;
  hICMPlib: HModule;

Initialization
begin
  if LibLoad = 0 then begin
    IcmpValid := True;
    hICMPlib := loadlibrary(icmpDLL);
    if hICMPlib <> null then begin
      @ICMPCreateFile := GetProcAddress(hICMPlib, 'IcmpCreateFile');
      @IcmpCloseHandle:= GetProcAddress(hICMPlib, 'IcmpCloseHandle');
      @IcmpSendEcho:= GetProcAddress(hICMPlib, 'IcmpSendEcho');
      if (@ICMPCreateFile  = Nil) or
         (@IcmpCloseHandle = Nil) or
         (@IcmpSendEcho    = Nil) then begin
        IcmpValid := False;
        if MessageDlg('Unable to load functions from ' + icmpDLL, mtError,
                      [mbIgnore, mbAbort],0) = mrAbort then halt;
      end;
    end else begin
      IcmpValid := False;
      if MessageDlg('Unable to find ' + icmpDLL, mtError,
                    [mbIgnore, mbAbort],0) = mrAbort then halt
    end;
  end;
  Inc(LibLoad);
end;

Finalization
begin
  Dec(LibLoad);
  if LibLoad = 0 then FreeLibrary(hICMPlib);
end;

end.

