unit WSExtra;
{
  Written By:    Martien Verbruggen
  Date:          12 July, 1996 - 21 August, 1996
  Copyright:     (C) Copyright Martien Verbruggen, 1996. All Rights Reserved
  Email:         tgtcmv@chem.tue.nl
  www:           http://www.tcp.chem.tue.nl/~tgtcmv/
  Contents:      Components TDNS and Some Winsock procs.

  Description:   TDNS provides an object to resolve hostnames. Create an
                 object, set to Async or blocking, create an event handler
                 for OnResolved, and read out the data you need.

  Prerequisites: You must have a working Winsock installed. This code has been
                 tested under Windows 95.

  Properties:    Address:     (Read Only) IP network byte order address
                 Async:       True for Try Asynchronous lookup (set a handler
                              for the OnResolved Event)
                 HostIP:      (Read Only) dotted decimal Address
                 HostName:    Hostname or dotted decimal address to resolve
                 Msg:         Message describing error, if any.
                 Status:      Status code of the lookup

  Events:        OnResolved:  Called when the data is available
                 OnError:     Called when an error occurs

  Methods:       Cancel:      Cancels an Asynchronous lookup
                 Create:      registers with winsock, sets defaults
                 Destroy:     unregisters with winsock

  Known Problems:
                 Needs a parent, sometimes hard when creating dynamically,
                 need to make sure it's owned by a TWinControl descendant,
                 or that the Handle is set manually.

  Globals:
    constants:
                 WSVersionRqrd:
                              Required version of Winsock;
    variables:
                 WSAData:     Winsock data as filled by WSAStartup
                 WSAInitialised:
                              Boolean to make sure winsock
                              doesn't get initialised more than
                              necessary (once per application
                              is enough)
                 WSAInitialiser:
                              Object that initialised Winsock,
                              should also take care of freeing
                              it again.
    procedures and functions:
                 GetWSDescription:
                 GetWSVersion:
                 GetWSMaxSockets:
                 GetWSMaxUdpDg:
                              Return info about the winsock
                 WSErrorString:
                              input: WS error code, output
                              descriptive string
}

interface

uses Messages, Classes, WinTypes, Controls, Winsock;

const
  WSVersionRqrd      = $101;
  WM_GotAsyncLookup  = WM_USER + 312;
  DNS_STATUS_BASE    = $80000000;
  DNS_IN_PROGRESS    = DNS_STATUS_BASE + 0;
  DNS_CANCELLED      = DNS_STATUS_BASE + 1;
  DNS_NO_ADDRESS     = DNS_STATUS_BASE + 2;

type
  TWSMode = (wsmBlocking, wsmAsync);

  TDNS = Class(TWinControl)    // Need a Window handle for Async Messages
  private
    FAddress:    DWord;        // Address of the host
    FMode:       TWSMode;      // Blocking or Asynchronous lookup
    FHostName:   String;       // Name of the host
    FHostIP:     String;       // Dotted decimal IP address
    FMessage:    String;       // Used to store error messages
    FResolved:   TNotifyEvent; // Called when Async completes
    FStatus:     Integer;      // Status of the lookup. 0 on success
    FInProgress: Boolean;      // True if Async call in progress
    pDNSBuf: PChar;
    hndDNS: THandle;
    function  GetAsync: Boolean;
    procedure Notify;
    procedure Finish;
    procedure SetHost(Name: String);
    procedure SetHostAsync(Name: String);
    procedure SetHostBlocking(Name: String);
    procedure WMGotAsyncLookup(var Msg: TMessage); message WM_GotAsyncLookup;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    procedure Cancel; // Only for Async call
    property  Address: DWord read FAddress;
    property  HostName: String read FHostName write SetHost;
    property  HostIP: String read FHostIP write SetHost;
    property  Msg: String read FMessage;
    property  Status: Integer read FStatus;
    property  Async: Boolean read GetAsync;
  published
    property  Mode: TWSMode read FMode write FMode default wsmBlocking;
    property  OnResolved: TNotifyEvent read FResolved write FResolved;
  end;

function GetWSDescription: string;
function GetWSVersion: string;
function GetWSMaxSockets: u_short;
function GetWSMaxUdpDg: u_short;
function WSErrorString(nErr: Integer): String;

var
  wsInitdata: TWSAData;
  wsInitialised: Boolean;
  wsInitialiser: TObject;

implementation

uses
  SysUtils, Dialogs;

// TDNS Class

constructor TDNS.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FHostName := ''; FAddress := INADDR_NONE;
  FInProgress := False;
  Mode := wsmBlocking;
  if not ( csDesigning in ComponentState ) and not WSInitialised then begin
    if WSAStartup(WSVersionRqrd,WSInitdata) = 0 then begin
      wsInitialised := True;
      wsInitialiser := Self;
    end else
      if MessageDlg(WSErrorString(WSAGetLastError), mtError,
         [mbIgnore, mbAbort],0) = mrAbort then halt;
  end;
end;

destructor TDNS.Destroy;
begin
  if wsInitialiser = Self then
    if WSACleanup = 0 then begin
      wsInitialised := False;
      wsInitialiser := nil;
    end else
      ShowMessage(WSErrorString(WSAGetLastError));
  inherited;
end;

procedure TDNS.SetHost(Name: String);
begin
  FHostIP := ''; FHostName := ''; FAddress := INADDR_NONE;
  FStatus := 0;
  if Mode = wsmAsync then try HandleNeeded except Mode := wsmBlocking end;
  if FInProgress then begin
    FStatus := DNS_IN_PROGRESS; FMessage := 'Lookup already in progress';
    Notify;
  end else begin
    if Name = '' then begin
      FStatus := DNS_NO_ADDRESS; FMessage := 'No Address Specified';
      Finish;
    end else begin
      FInprogress := True;
      FAddress := inet_addr(PChar(Name));
      if Async then SetHostAsync(Name)
               else SetHostBlocking(Name);
    end;
  end;
end;

procedure TDNS.SetHostBlocking(Name: String);
var
  Phe: PHostEnt;
begin
  if (FAddress = INADDR_NONE) then begin
    Phe := GetHostByName(PChar(Name));
    if Phe = Nil then begin
      FStatus := WSAGetLastError; FMessage := WSErrorString(FStatus);
    end;
  end else begin
    Phe := GetHostByAddr(@FAddress, 4, PF_INET);
    if Phe = Nil then begin
      FStatus := WSAGetLastError; FMessage := WSErrorString(FStatus);
    end;
  end;
  if FStatus = 0 then begin
    FAddress := longint(plongint(Phe^.h_addr_list^)^);
    FHostName := Phe^.h_name;
    FHostIP := StrPas(inet_ntoa(TInAddr(FAddress)));
  end;
  Finish;
end;

procedure TDNS.SetHostAsync(Name: String);
begin
  GetMem(pDNSBuf, MAXGETHOSTSTRUCT);
  if (FAddress = INADDR_NONE) then begin
    hndDNS := WSAAsyncGetHostByName(Handle, WM_GotAsyncLookup, PChar(Name), pDNSBuf, MAXGETHOSTSTRUCT);
    if hndDNS = 0 then begin
      FStatus := WSAGetLastError; FMessage := WSErrorString(FStatus);
    end;
  end else begin
    hndDNS := WSAAsyncGetHostByAddr(Handle, WM_GotAsyncLookup, @FAddress, 4, PF_INET, pDNSBuf, MAXGETHOSTSTRUCT);
    if hndDNS = 0 then begin
      FStatus := WSAGetLastError; FMessage := WSErrorString(FStatus);
    end;
  end;
  // If the status <> 0 then something went wrong firing off the lookup
  if FStatus <> 0 then Finish;
end;

procedure TDNS.WMGotAsyncLookup(var Msg: TMessage);
var
  Phe: PHostEnt;
begin
  FStatus := WSAGetAsyncError(Msg.lParam);
  if FStatus <> 0 then
    FMessage := WSErrorString(FStatus)
  else begin
    Phe := pHostEnt(pDNSBuf);
    FAddress := longint(plongint(pHE^.h_addr_list^)^);
    FHostName := Phe^.h_name;
    FHostIP := StrPas(inet_ntoa(TInAddr(FAddress)));
  end;
  FreeMem(pDNSBuf);
  Finish;
end;

procedure TDNS.Notify;
begin
  if Assigned(FResolved) then FResolved(Self);
end;

procedure TDNS.Finish;
begin
  if Assigned(FResolved) then FResolved(Self);
  FInProgress := False;
end;

// This only makes sense for an asynchronous call
procedure TDNS.Cancel;
begin
  if Async and FInProgress then begin
    FStatus := WSACancelAsyncRequest(hndDNS);
    if FStatus = 0 then begin
      FMessage := 'Cancelled Asynchronous lookup';
      FStatus := DNS_CANCELLED;
    end else FMessage := WSErrorString(FStatus);
    Finish;
  end;
end;

function TDNS.GetAsync;
begin
  if FMode = wsmAsync then Result := True
                      else Result := False;
end;

// General functions

function GetWSDescription: string;
begin
  Result := WSinitdata.szDescription;
end;

function GetWSVersion: string;
begin
  Result := IntToStr(Hi(WSinitdata.wVersion)) + '.' +
            IntToStr(Hi(WSinitdata.wVersion));
end;

function GetWSMaxSockets: u_short;
begin
  Result := WSinitdata.iMaxSockets;
end;

function GetWSMaxUdpDg: u_short;
begin
  Result := WSinitdata.iMaxUdpDg;
end;

function WSErrorString(nErr: Integer): String;
begin
  case nErr of
    WSAVERNOTSUPPORTED:
      Result := 'Version of WinSock not supported';
    WSASYSNOTREADY:
      Result := 'WinSock not present or not responding';
    WSAEINVAL:
      Result := 'App version not supported by DLL';
    WSAHOST_NOT_FOUND:
      Result := 'Authoritative Host not found';
    WSATRY_AGAIN:
      Result := 'Non-authoritative Host not found or server failure';
    WSANO_RECOVERY:
      Result := 'Non-recoverable: refused or not implemented';
    WSANO_DATA:
      Result := 'Valid name, no data record for type';
    // WSANO_ADDRESS:
    //   Result := 'Valid name, no MX record';
    WSANOTINITIALISED:
      Result := 'WSA Startup not initialized';
    WSAENETDOWN:
      Result := 'Network subsystem failed';
    WSAEINPROGRESS:
      Result := 'Blocking operation in progress';
    WSAEINTR:
      Result := 'Blocking call cancelled';
    WSAEAFNOSUPPORT:
      Result := 'Address family not supported';
    WSAEMFILE:
      Result := 'No file descriptors available';
    WSAENOBUFS:
      Result := 'No buffer space available';
    WSAEPROTONOSUPPORT:
      Result := 'Specified protocol not supported';
    WSAEPROTOTYPE:
      Result := 'Protocol wrong type for this socket';
    WSAESOCKTNOSUPPORT:
      Result := 'Socket type not supported for address family';
    WSAENOTSOCK:
      Result := 'Descriptor is not a socket';
    WSAEWOULDBLOCK:
      Result := 'Socket marked as non-blocking and SO_LINGER set not 0';
    WSAEADDRINUSE:
      Result := 'Address already in use';
    WSAECONNABORTED:
      Result := 'Connection aborted';
    WSAECONNRESET:
      Result := 'Connection reset';
    WSAENOTCONN:
      Result := 'Not connected';
    WSAETIMEDOUT:
      Result := 'Connection timed out';
    WSAECONNREFUSED:
      Result := 'Connection refused';
    WSAEHOSTDOWN:
      Result := 'Host down';
    WSAENETUNREACH:
      Result := 'Network unreachable';
    WSAEHOSTUNREACH:
      Result := 'Host unreachable';
    WSAEADDRNOTAVAIL:
      Result := 'Address not available';
    else
      Result := '';
  end;
end;

Initialization
begin
  WSInitialised := False;
  WSInitialiser := nil;
end;

end.
