unit Main;

interface

uses
  SysUtils, WinTypes, WinProcs, Messages, Classes, Graphics, Controls,
  Forms, Dialogs, AaWiz, StdCtrls, AaWizStd, ExtCtrls;

type
  TMainForm = class(TForm)
    Wizard: TAAWizard;
    WelcomeQuestion: TAAPromptQuestion;
    UnitNameQuestion: TAATextQuestion;
    ClassNameQuestion: TAATextQuestion;
    MenuNameQuestion: TAATextQuestion;
    GenerateQuestion: TAAPromptQuestion;
    LinksQuestion: TAAMemoQuestion;
    SourceListBox: TListBox;
    Button1: TButton;
    procedure Button1Click(Sender: TObject);
    procedure ClassNameQuestionValidate(Sender: TObject;
      var Continue: Boolean);
    procedure MenuNameQuestionValidate(Sender: TObject;
      var Continue: Boolean);
    procedure UnitNameQuestionValidate(Sender: TObject;
      var Continue: Boolean);
    procedure LinksQuestionValidate(Sender: TObject;
      var Continue: Boolean);
  private
    procedure BuildQuestionClass;
  public
  end;

var
  MainForm: TMainForm;

implementation

{$R *.DFM}

const
  idBase           = '{<<';
  idLink           = '{<<Link>>}';
  idClassName      = '{<<ClassName>>}';
  idMenuName       = '{<<MenuName>>}';
  idUnitName       = '{<<UnitName>>}';


{------------------------------------------------------------------------------}

procedure ShowError(msg: PChar);
begin
  Application.MessageBox(msg, 'Error', MB_ICONEXCLAMATION);
end;

{------------------------------------------------------------------------------}

function ShowConfirm(msg: PChar): boolean;
begin
  result := (Application.MessageBox(msg, 'Confirm',
        MB_ICONQUESTION or MB_YESNO) = IDYES);
end;

{------------------------------------------------------------------------------}

procedure TMainForm.Button1Click(Sender: TObject);
begin
  if Wizard.Run then
    BuildQuestionClass;
  Application.Terminate;
end;

{------------------------------------------------------------------------------}

procedure TMainForm.ClassNameQuestionValidate(Sender: TObject;
  var Continue: Boolean);
begin
  if not IsValidIdent((Sender as TAATextQuestion).Text) then begin
    ShowError('Invalid question class name!');
    Continue := false;
  end;
end;

{------------------------------------------------------------------------------}

procedure TMainForm.MenuNameQuestionValidate(Sender: TObject;
  var Continue: Boolean);
begin
  if MenuNameQuestion.Text = '' then begin
    ShowError('Please specify a menu name for the question.');
    Continue := false;
  end;
end;

{------------------------------------------------------------------------------}

procedure TMainForm.UnitNameQuestionValidate(Sender: TObject;
  var Continue: Boolean);
var
  s: TFileStream;
begin

  if FileExists(UnitNameQuestion.Text) then begin
    if not(ShowConfirm('This file already exists.  Overwrite?')) then
      Continue := false;
    exit;
  end;

  try
    s := TFileStream.Create(UnitNameQuestion.Text, fmCreate);
    s.Free;
    SysUtils.DeleteFile(UnitNameQuestion.Text);
  except
    on EStreamError do begin
      ShowError('Invalid file and/or path name.');
      Continue := false;
    end;
  end;


end;

{------------------------------------------------------------------------------}

procedure TMainForm.LinksQuestionValidate(Sender: TObject;
  var Continue: Boolean);
var
  i: integer;
begin

  if LinksQuestion.Lines.Count = 0 then begin
    ShowError('The question must have one or more links.');
    continue := false;
    Exit;
  end;

  for i := 0 to LinksQuestion.Lines.Count-1 do
    if not(IsValidIdent(LinksQuestion.Lines[i])) then begin
      ShowError('One or more of the link properties have invalid names.');
      continue := false;
      exit;
    end;

end;

{------------------------------------------------------------------------------}

function ReplaceString(const search, replace, src: string): string;
var
  idx: integer;
  len: integer;
begin
  idx := Pos(search, src);
  if idx = 0 then
    result := src
  else begin
    len := Length(search);
    result := Copy(src, 1, idx-1) + replace + Copy(src, idx+len, Length(src));
    {recurse for any other tokens in the same line }
    result := ReplaceString(search, replace, result);
  end;
end;

{------------------------------------------------------------------------------}


procedure TMainForm.BuildQuestionClass;
var
  src: TStrings;
  links: TStrings;
  line_no: integer;
  link_template: string;
  i: integer;
  filename: string;
  clsname: string;
  menuname: string;
  unitname: string;
begin

  src := SourceListBox.Items;
  links := TStringList.Create;


  links.Assign(LinksQuestion.Lines);
  filename := UnitNameQuestion.Text;
  clsname := ClassNameQuestion.Text;
  menuname := MenuNameQuestion.Text;
  unitname := ChangeFileExt(ExtractFileName(filename), '');

  if links.Count = 0 then
    raise Exception.Create('No links defined');


  try

    src.BeginUpdate;

    { on the first pass expand all link references }
    line_no := 0;
    while line_no < src.Count do begin
      if Pos(idLink, src[line_no]) > 0 then begin
        { delete the link template line }
        link_template := src[line_no];
        src.Delete(line_no);
        { add lines corresponding to the template for each link }
        for i := links.Count-1 downto 0 do
          src.Insert(line_no, ReplaceString(idLink, links[i], link_template));
      end;
      Inc(line_no);
    end;

    { on the second pass expand all other references }
    for i := 0 to src.Count-1 do
      if Pos(idBase, src[i]) > 0 then begin
        src[i] := ReplaceString(idClassName, clsname, src[i]);
        src[i] := ReplaceString(idMenuName, menuname, src[i]);
        src[i] := ReplaceString(idUnitName, unitname, src[i]);
      end;

  finally
    src.EndUpdate;
    links.Free;
  end;

  src.SaveToFile(filename);

  Application.MessageBox('The question class was created successfully.',
    'Success', MB_ICONINFORMATION);

end;

{------------------------------------------------------------------------------}



end.
