{----------------------------------------------------------------------}
{                                                                      }
{                  The TEyes component is Copyright  1997             }
{                       by Athena Software Amsterdam                   }
{                            All Rights Reserved                       }
{                          Written by Pepijn Smits                     }
{                           pepijn@compuserve.com                      }
{                                                                      }
{----------------------------------------------------------------------}

unit Eyes;

interface

uses
  SysUtils, Windows, Messages, Classes, Graphics,
  Forms, Controls, ExtCtrls, DsgnIntf, Dialogs;

type
  TCustomEyes = class(TGraphicControl)
  private
    FTimer: TTimer;
    FEyeColor,FPupilColor: TColor;
    CurMouse,OldMouse: TPoint;
    Pupil1,Pupil2: TPoint;
    PupilWidth: Integer;
    PupilDist:Integer;
    function GetActive: Boolean;
    procedure SetActive(Value: Boolean);
    procedure SetEyeColor(Value: TColor);
    procedure SetPupilColor(Value: TColor);
  protected
    procedure Paint; override;
    procedure Invalidate; override;
    procedure SetBounds(ALeft, ATop, AWidth, AHeight: Integer); override;
    procedure Timer(Sender: TObject); virtual;
    function EyeWidth: integer; virtual;
    function EyeRight: integer; virtual;
    function EyeCenter(Nr: integer): Integer; virtual;
    function CalcPupil(Nr: integer): TPoint; virtual;
    procedure DrawPupils(P1,P2: TPoint; Black: Boolean); virtual;
    procedure UpdatePupils; virtual;
    property Active:Boolean read GetActive write SetActive;
    property EyeColor: TColor read FEyeColor write SetEyeColor;
    property PupilColor: TColor read FPupilColor write SetPupilColor;
  public
    constructor Create(AOwner: TComponent); override;
  end;
  TEyes = Class(TCustomEyes)
  published
    property Active default True;
    property EyeColor default clWhite;
    property PupilColor default clBlack;
    property Align;
    property DragCursor;
    property DragMode;
    property ParentShowHint;
    property PopupMenu;
    property ShowHint;
    property Visible;
    property OnClick;
    property OnDblClick;
    property OnDragDrop;
    property OnDragOver;
    property OnEndDrag;
    property OnMouseDown;
    property OnMouseMove;
    property OnMouseUp;
    property OnStartDrag;
  end;
  TEyesEditor = class(TComponentEditor)
    function GetVerbCount: Integer; Override;
    function GetVerb(index: Integer): String; Override;
    procedure ExecuteVerb(index: Integer); Override;
  end;

procedure Register;

implementation

{ Local functions - Needed for Pupil Position }

function ArcTan2(X,Y:Double):Double;
begin
  Result := ArcTan(Y/X);
  if (X<0) then Result := Result + Pi;
end;

function Hypot(X,Y:Double):Double;
begin
  Result := Sqrt(Sqr(X)+Sqr(Y));
end;

function Sign(X:Double):integer;
begin
  Result := 0;
  if X<0 then Result := -1;
  if X>0 then Result := 1;
end;

{ TCustomEyes }

constructor TCustomEyes.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FTimer := TTimer.Create(Self);
  FTimer.Interval := 100;
  FTimer.OnTimer := Timer;
  FTimer.Enabled := True;
  Width := 100;
  Height := 100;
  FEyeColor := clWhite;
  FPupilColor := clBlack;
end;

procedure TCustomEyes.Paint;
begin
  With Canvas do
  begin
    Brush.Color := FEyeColor;
    Pen.Color := FEyeColor;
    Pen.Width := 1;
    Ellipse(0,0,EyeWidth,Height);
    Ellipse(EyeRight,0,Width,Height);
    OldMouse.X := -1;
    OldMouse.Y := -1;
  end;
end;

procedure TCustomEyes.Invalidate;
begin
  OldMouse.X := -1;
  OldMouse.Y := -1;
  Pupil1.X := -1;
  Pupil1.Y := -1;
  Pupil2.X := -1;
  Pupil2.Y := -1;
  if (Height>10) and (Width>10)
  Then
    PupilWidth := (Height+Width) div 20
  else
    PupilWidth := 1;
  PupilDist := Trunc(1.175*PupilWidth);
  inherited Invalidate;
end;

procedure TCustomEyes.SetBounds(ALeft, ATop, AWidth, AHeight: Integer);
begin
  if not (csLoading in ComponentState) then
    if (AHeight > AWidth) then AHeight := AWidth;
  inherited SetBounds(ALeft, ATop, AWidth, AHeight);
end;

procedure TCustomEyes.Timer(Sender: TObject);
begin
  GetCursorPos(CurMouse);
  if (CurMouse.X <> OldMouse.X) or (CurMouse.Y <> OldMouse.Y) then
  begin
    UpdatePupils;
    OldMouse := CurMouse;
  end;
end;

function TCustomEyes.EyeWidth:integer;
begin
  Result := (Width div 2)-1;
end;

function TCustomEyes.EyeRight:integer;
begin
  Result := (Width div 2)+1
end;

function TCustomEyes.EyeCenter(Nr:integer):integer;
begin
  if Nr=0 then Result := Width div 4 else Result := EyeRight + Width div 4;
end;

function TCustomEyes.CalcPupil(Nr:Integer):TPoint;
var
  PX,PY:Double; { The Result Coordinates }
  DX,DY:Double; { Delta X and Y }
  MX,MY:Double; { The mouse position }
  EX,EY:Double; { The Center of the Eye.. }
  Angle:Double; { The Angle of Mouse from the EyeCentre }
  SinAngle,     { Sinus and Cosinus of Angle }
  CosAngle:Double;
  Hypo,         { Hypo Calculated distance }
  Dist:Double;  { Temp. Distance Variable }
  W,H:Double;   { DrawWidth and DrawHeight }
  WH:Double;    { Width/2*Height/2 of Eye's Bouding Box }
  X,Y:Double;   { Temp. Variables }
begin
  MX := CurMouse.X;
  MY := CurMouse.Y;
  EX := ClientOrigin.X + EyeCenter(Nr);
  EY := ClientOrigin.Y + Height/2;
  W := EyeWidth/2-PupilDist;
  H := Height/2-PupilDist;
  WH := (W*H);
  DX := MX - EX;
  DY := MY - EY;
  if (DX<>0) and (DY<>0) then
  begin
    Angle := ArcTan2(DX,DY);
    SinAngle := Sin(Angle);
    CosAngle := Cos(Angle);
    Hypo := Hypot(H*CosAngle,W*SinAngle);
    X := (WH*CosAngle)/Hypo;
    Y := (WH*SinAngle)/Hypo;
    Dist := Hypot(X,Y);
    if Dist<Hypot(DX,DY) then
    begin
      PX := EX + Dist * CosAngle;
      PY := EY + Dist * SinAngle;
    end else
    begin
      PX := MX;
      PY := MY;
    end;
  end else
  begin
    if (Abs(DX)<W)
      then PX := MX else PX := EX + Sign(DX)*W;
    if (Abs(DY)<H)
      then PY := MY else PY := EY + Sign(DY)*H;
  end;
  Result.X := Trunc(PX);
  Result.Y := Trunc(PY);
end;

procedure TCustomEyes.DrawPupils(P1,P2:TPoint;Black:Boolean);
begin
  P1 := ScreenToClient(P1);
  P2 := ScreenToClient(P2);
  With Canvas do
  begin
    if Black then
    begin
      Brush.Color := FPupilColor;
      Pen.Color := FPupilColor;
    end else
    begin
      Brush.Color := FEyeColor;
      Pen.Color := FEyeColor;
    end;
    Pen.Width := 1;
    With P1 do Ellipse(X-PupilWidth,Y-PupilWidth,
                       X+PupilWidth,Y+PupilWidth);
    With P2 do Ellipse(X-PupilWidth,Y-PupilWidth,
                       X+PupilWidth,Y+PupilWidth);
  end;
end;

procedure TCustomEyes.UpdatePupils;
var
  P1,P2:TPoint;
begin
  if (Pupil1.X<>-1) Then DrawPupils(Pupil1,Pupil2,False);
  P1 := CalcPupil(0);
  P2 := CalcPupil(1);
  DrawPupils(P1,P2,True);
  Pupil1 := P1;
  Pupil2 := P2;
end;

function TCustomEyes.GetActive: Boolean;
begin
  Result := FTimer.Enabled;
end;

procedure TCustomEyes.SetActive(Value: Boolean);
begin
  if Value<>Active then
  begin
    FTimer.Enabled := Value;
    Invalidate;
  end;
end;

procedure TCustomEyes.SetEyeColor(Value: TColor);
begin
  if Value<>FEyeColor then
  begin
    FEyeColor := Value;
    Invalidate;
  end;
end;

procedure TCustomEyes.SetPupilColor(Value: TColor);
begin
  if Value<>FPupilColor then
  begin
    FPupilColor := Value;
    Invalidate;
  end;
end;

{ TEyesEditor }

function TEyesEditor.GetVerbCount: integer;
begin
  Result := 1;
end;

function TEyesEditor.GetVerb(index: Integer): string;
begin
  Result := 'About..';
end;

procedure TEyesEditor.ExecuteVerb(index:integer);
begin
  MessageDlg('"Eyes" Component v1.00 - for Delphi 2.0'+#13#13+
             'Written by Pepijn Smits - Athena Software Amsterdam'+#13#13+
             'E-Mail pepijn@compuserve.com'+#13#13+
             'or'+#13#13+
             'CompuServe 74750,733'+#13#13+
             'This Component is Freeware, All rights reserved.'+#13#13+
             'See EYES.HTM for more information.'
             ,mtInformation,[mbOk],0)
end;

{ Registration }

procedure Register;
begin
  RegisterComponents('Samples', [TEyes]);
  RegisterComponentEditor(TEyes, TEyesEditor);
end;

end.
