(********************************************************************
 *
 *  TdkwLabelEdit
 *
 *  The LabelEdit control is designed to function like a DBEdit
 *  control with a static label automatically associated with it.
 *
 *  Copyright (c) 1995 by DKW Systems Corporation, All Rights
 *  Reserved.  Changes originally written by Blake Stone,
 *  Sept 17, 1995
 *
 *  Change History:
 *
 *    Sept 17, 1995
 *      BWS - Original version written by Blake Stone
 *
 *    June 14, 1996
 *      BWS - Converted to Delphi32 and made into a proper subclass
 *            of TDBEdit
 *
 *******************************************************************)

unit DKWLEdit;

interface

uses
  SysUtils, WinTypes, WinProcs, Messages, Classes, Graphics, Controls,
  Forms, Dialogs, StdCtrls, Mask, DBCtrls, DB, DBTables, Menus;

type
  TdkwLabelPosition = ( leBeside, leAbove ) ;

  TdkwLabelEdit = class(TDBEdit)
  private
    FLabel: TLabel ;
    FLabelPosition: TdkwLabelPosition ;
    FOldOnDataChange: TNotifyEvent ;
    procedure UpdateLabel ;
    procedure LabelDataChange ( Sender: TObject ) ;
  protected
    procedure SetParent ( AParent: TWinControl ) ; override ;
    procedure SetBounds ( ALeft, ATop, AWidth, AHeight: Integer ) ; override ;
    procedure SetLabelPosition ( Value: TdkwLabelPosition ) ;
  public
    constructor Create(AOwner: TComponent); override ;
    destructor Destroy ; override ;
  published
    property LabelPosition: TdkwLabelPosition read FLabelPosition write SetLabelPosition default leBeside ;
  end;

procedure Register;

implementation

procedure Register;
begin
  RegisterComponents('DKW', [TdkwLabelEdit]);
end;

{ Create

  Preserve the reference to the old DataChange method, which
  is private, and replace it with our own }

constructor TdkwLabelEdit.Create(AOwner: TComponent);
var
  Message: TMessage ;
begin
  inherited ;
  FLabel := TLabel.Create ( nil ) ;
  Message.Msg := CM_GETDATALINK ;
  Dispatch ( Message ) ;
  with TFieldDataLink ( Message.Result ) do
  begin
    FOldOnDataChange := OnDataChange ;
    OnDataChange := LabelDataChange ;
  end ;
end ;

{ LabelDataChange

  Update the label and pass the change request on to the
  old DataChange method }

procedure TdkwLabelEdit.LabelDataChange ( Sender: TObject ) ;
begin
  UpdateLabel ;
  if Assigned ( FOldOnDataChange ) then FOldOnDataChange ( self ) ;
end ;

{ Destroy

  Cleanly destroy the TdkwLabelEdit }

destructor TdkwLabelEdit.Destroy;
begin

  { Note the serious workaround here ... we cannot
    destroy FLabel if the parent window is planning
    on destroying it, but we CAN count on it
    destroying the label for us, so this should be
    safe, albeit inelegant }

  if FLabel <> nil then
  begin
    if ( FLabel.Parent = nil ) then
      FLabel.Free
    else
      if not ( csDestroying in FLabel.Parent.ComponentState ) then
        FLabel.Free ;
    FLabel := nil ;
  end;

  inherited ;
end;

{ SetParent

  Move our label with us to a new parent }

procedure TdkwLabelEdit.SetParent ( AParent: TWinControl ) ;
begin
  inherited ;
  UpdateLabel ;
end;

{ SetBounds

  Move our label with us to a new location }

procedure TdkwLabelEdit.SetBounds ( ALeft, ATop, AWidth, AHeight: Integer ) ;
begin
  inherited ;
  UpdateLabel ;
end;

{ SetLabelPosition

  Move the label to a new position }

procedure TdkwLabelEdit.SetLabelPosition ( Value: TdkwLabelPosition ) ;
begin
  if ( FLabelPosition <> Value ) then
  begin
    FLabelPosition := Value ;
    UpdateLabel ;
  end ;
end ;

{ UpdateLabel

  Create a paired label at the correct location }

procedure TdkwLabelEdit.UpdateLabel ;
begin
  if FLabel <> nil then
  begin
    if FLabel.Parent <> Parent then FLabel.Parent := Parent ;

    if Field <> nil then
      FLabel.Caption := Field.DisplayName
    else
      FLabel.Caption := '' ;
    case FLabelPosition of
      leBeside: FLabel.SetBounds ( Left - FLabel.Width - 4,
          Top + ( Height - FLabel.Height ) div 2,
          FLabel.Width, FLabel.Height ) ;
      leAbove: FLabel.SetBounds ( Left,
          Top - FLabel.Height - 2,
          FLabel.Width, FLabel.Height ) ;
    end ;
  end ;
end ;

end.

