Unit Dol_edit;
{Implenmation of a TEdit component with dollar only entry, built in pop-up
 calculator, and required field validation.
 Copyright (c), 1996 by Bert Perez. All Rights Reserved.}

Interface

Uses
	SysUtils, WinTypes, WinProcs, Messages, Classes, Graphics, Controls,
 Forms, Dialogs, StdCtrls, ExtCtrls, Sendkey;

Type
 TDollarEdit = Class(TEdit)

 Private
 {Private declarations}

 Protected
 {Protected declarations}
 	FRequired    : Boolean;
   FMinusKey    : Boolean;
   FDollarDigits: Integer;
   FCentDigits  : Integer;

  	Procedure SetRequired(vValue: Boolean);
   Procedure SetMinusKey(vValue: Boolean);
   Procedure SetDollarDigits(vValue: Integer);
   Procedure SetCentDigits(vValue: Integer);

 Public
 {Public declarations}
   Constructor Create(AOwner: TComponent); Override;
   Procedure KeyPress(Var Key: Char); Override;
   Procedure DoExit; Override;
   Procedure DoEnter; Override;

 Published
 {Published declarations}
 	Property RequiredField: Boolean  Read FRequired
   																Write SetRequired;

		Property MinusKey: Boolean  Read FMinusKey
   													 Write SetMinusKey;

		Property DollarDigits: Integer  Read FDollarDigits
   															 Write SetDollarDigits;

		Property CentDigits: Integer  Read FCentDigits
   														 Write SetCentDigits;

 End;

 TCalcFrm = Class(TForm)

  Private
  {Private declarations}
   Oper1, Oper2: Double;     {Previously input operands.}
   NumOper: Integer;         {Number of operands.}
   LastInput: Integer;       {Holds type of last keypress.}
   OperFlag: Char;           {Holds pending operation.}
   DecFlag: Boolean;         {Holds the decimal point status.}
  	Button1: TButton;
   Button2: TButton;
   Button3: TButton;
   Button4: TButton;
   Button5: TButton;
   Button6: TButton;
   Button7: TButton;
   Button8: TButton;
   Button9: TButton;
   Button0: TButton;
   ButtonDec: TButton;
   ButtonEqu: TButton;
   ButtonMin: TButton;
   ButtonDiv: TButton;
   ButtonMul: TButton;
   ButtonAdd: TButton;
   ButtonC: TButton;
   ButtonAC: TButton;
   ButtonGet: TButton;
   ButtonPut: TButton;
   Panel1: TPanel;
   Display: TLabel;
   Procedure NumberClick(Sender: TObject);
   Procedure OperatorClick(Sender: TObject);
   Procedure FormShow(Sender: TObject);
   Procedure ButtonACClick(Sender: TObject);
   Procedure ButtonCClick(Sender: TObject);
   Procedure ButtonDecClick(Sender: TObject);
   Function FuncStr(vNum: Double): String;
   Function FuncVal(vStr: String): Double;
   Procedure FormKeyPress(Sender: TObject; Var Key: Char);
   Procedure ButtonGetClick(Sender: TObject);
   Procedure ButtonPutClick(Sender: TObject);
 End;

Procedure Register;
Procedure ProcDisplayCalculator;
Function FuncLeftTrim(Const vStr: String): String;
Function FuncUnFormat(vStr: String): String;

Implementation

	Var
 	CalcFrm : TCalcFrm;

   {Holds whether or not the calculator has been created.}
   vCreated: Boolean;

   {Holds whether or not the calculator put has been pressed.}
   vPutKey: Boolean;

   {Text to be used for assigning the calculator field with the dollar
    field, or visa versa.}
   vGlobalText: String;

   {Holds the maximum number of digits allowed prior to and after the
    decimal point.}
   vDollars, vCents: Integer;

	Const
	{These constants define the type of last calculator keypress.}
 	cDigit = 1;
 	cDecimal = 2;
		cClear = 3;
 	cOper = 4;
 	cMaxChars = 22;      {Maximum characters allowed in calculator display.}

Constructor TDollarEdit.Create(AOwner: TComponent);
{Initializes the component, and default variables.}

	Begin
 	Inherited Create(AOwner);
   FRequired := False;
   FMinusKey := True;
   FDollarDigits := 9;
   FCentDigits := 2;
   Text := '';
 End;
{------------------------------------------------------------------------------}
Procedure TDollarEdit.SetRequired(vValue: Boolean);
{Sets whether or not field is to be required.}

 Begin
   If vValue <> FRequired Then
     FRequired := vValue;
 End;
{------------------------------------------------------------------------------}
Procedure TDollarEdit.SetMinusKey(vValue: Boolean);
{Sets whether or not to accept a minuskey in the field.}

 Begin
   If vValue <> FMinusKey Then
     FMinusKey := vValue;
 End;
{------------------------------------------------------------------------------}
Procedure TDollarEdit.SetDollarDigits(vValue: Integer);
{Sets the maximum number of digits which can be entered prior to the
 decimal point.}

 Begin
   If vValue <> FDollarDigits Then
     FDollarDigits := vValue;

   If FDollarDigits > 17 Then
     FDollarDigits := 17;

   vDollars := FDollarDigits;
 End;
{------------------------------------------------------------------------------}
Procedure TDollarEdit.SetCentDigits(vValue: Integer);
{Sets the maximum number of digits which can be entered after the decimal
 point.}

 Begin
   If vValue <> FCentDigits Then
     FCentDigits := vValue;

   If FCentDigits > 4 Then
     FCentDigits := 4;

   vCents := FCentDigits;
 End;
{------------------------------------------------------------------------------}
Procedure TDollarEdit.KeyPress(Var Key: Char);
{Intercepts all key presses, and defines their behavior.}

 Begin
 	{Define the acceptable keys, and reject any invalid keys.}
   If Not (Key In ['0'..'9', '-', '.', 'C', #8]) Then Key := #0;

   {If a decimal point or minuskey key, and such key has already been
    entered, reject it.}
   If (Key In ['.', '-']) Then
     If (Pos(Key, Text) > 0) Then Key := #0;

   {If a minuskey, but the field does not allow one, reject it.}
   If Key = '-' Then
     If FMinusKey = False Then
       Key := #0;

   {If a minuskey, but not the first character in the field, reject it.}
   If Key = '-' Then
     If Text <> '' Then
       Key := #0;

		{If shift+c key, then assign the field text to the global text, and
    display the calculator.}
   If Key = 'C' Then
     If Key <> #8 Then
       Begin
         vGlobalText := Text;
         Color := clTeal;
         ProcDisplayCalculator;
         Key := #0
       End;

   {If the calculator put key equals true, then reset it to false, and
    assign the global text to the field text.}
   If vPutKey = True Then
     Begin
       vPutKey := False;
       Text := vGlobalText;
       Key := #0
     End;

   {If the minuskey is not allowed, and the calculator put a negative
    amount in the field, reject it.}
   If FMinusKey = False Then
   	If Pos('-', Text) > 0 Then
     	Begin
					Text := '';
         MessageDlg('Invalid entry for this field!', mtWarning, [mbOK],
                0);
       End;

   {If the maximum number of characters permitted prior to the decimal
    point has been met, but a decimal point has not yet been entered,
    then don't accept any more characters, and allow the decimal point
    or the backspace key only.}
   If Length(Text) = FDollarDigits Then
     If Pos('.', Text) = 0 Then
       If Not (Key In ['.', #8]) Then
         Key := #0;

		{If less than or equal to the maximum number of characters permitted
    prior to the decimal point has been met, and the decimal point has
    been entered, and the maximum number of characters permitted after
    the decimal point has also been entered, then reject all keys except
    for the backspace key.}
  	If Length(Text) <= FDollarDigits Then
    	If Pos('.', Text) > 0 Then
      	If Length(Text) - Pos('.', Text) = FCentDigits Then
       	If Key <> #8 Then
         	Key := #0;

   {Same as above, except this time the test involves greater than or
    equal to.}
  	If Length(Text) >= FDollarDigits Then
    	If Pos('.', Text) > 0 Then
      	If Length(Text) - Pos('.', Text) = FCentDigits Then
       	If Key <> #8 Then
         	Key := #0;

   {Inherit all other acceptable key presses, such as ENTER, HOME, DELETE,
    etc.}
 	Inherited KeyPress(Key);
 End;
{------------------------------------------------------------------------------}
Procedure TDollarEdit.DoExit;
{Determines whether the field is required upon exiting, and if so, it
 checks if the field is null, and if it is, it informs the user that
 the field is required, and resets focus to the field.}

 Var
   vFieldValue: Double;
   vErrCode: Integer;
   vTextFormat: String;

	Begin
 	Inherited DoExit;
   Val(Text, vFieldValue, vErrCode);

   Case vCents of
     0: vTextFormat := ',0';
     1: vTextFormat := ',0.0';
     2: vTextFormat := ',0.00';
     3: vTextFormat := ',0.000';
     4: vTextFormat := ',0.0000';
   End;

   If FRequired Then
     If vFieldValue = 0 Then
       Begin
         MessageDlg(' This field is required! ', mtWarning, [mbOK], 0);
         SetFocus;
       End
     Else
     	Begin
       	Text := FormatFloat(vTextFormat, vFieldValue);
       	Color := clWhite;
       End
   Else
   	Begin
     	Text := FormatFloat(vTextFormat, vFieldValue);
     	Color := clWhite;
     End;
 End;
{------------------------------------------------------------------------------}
Procedure TDollarEdit.DoEnter;
{Calls FuncUnFormat, and initializes vDollars and vCents.}

 Begin
   Inherited DoEnter;
   Text := FuncUnFormat(Text);
   vDollars := FDollarDigits;
   vCents := FCentDigits;
 End;
{------------------------------------------------------------------------------}
Function TCalcFrm.FuncVal(vStr: String): Double;
{Converts from text to numeric data.}

 Var
  vErrCode: Integer;
  vNum    : Double;

 Begin
  Val(vStr, vNum, vErrCode);
  FuncVal := vNum;
 End; {Function FuncVal}
{------------------------------------------------------------------------------}
Function TCalcFrm.FuncStr(vNum: Double): String;
{Converts from numeric to text data.}

 Const
  cCrLf = #10 + #13;
  cMsg = 'The calculated result will not ' + cCrLf +
         'fit into the calculator dispaly';
 Var
  vStr: String;

 Begin
  Str(vNum: vDollars : vCents, vStr);

  If Length(vStr) > cMaxChars Then
   Begin
    MessageDlg(cMsg, mtWarning, [mbOK], 0);
    FuncStr := '0';
    FormShow(ButtonAC);
   End
  Else
   FuncStr := vStr;
 End; {Function FuncStr}
{------------------------------------------------------------------------------}
Procedure TCalcFrm.NumberClick(Sender: TObject);
{Defines the behavior of the digit keys}

 Begin
  If LastInput <> cDigit Then
   Begin
    Display.Caption := '';
    DecFlag := False;
   End;

  With Sender as TButton Do
   If Length(Display.Caption) < cMaxChars Then
    Begin
     Display.Caption := Display.Caption + Caption;
     LastInput := cDigit;
    End;
 End; {Procedure TCalcFrm.NumberClick}
{------------------------------------------------------------------------------}
Procedure TCalcFrm.FormShow(Sender: TObject);
{Initializes all variables.}

 Begin
  Display.Caption := '0';
  Oper1 := 0;
  Oper2 := 0;
  NumOper := 0;
  LastInput := 0;
  OperFlag := ' ';
  DecFlag := False;
 End; {Procedure TCalcFrm.FormShow}
{------------------------------------------------------------------------------}
Procedure TCalcFrm.ButtonACClick(Sender: TObject);
{Calls FormShow to re-initialize all variables.}

 Begin
  FormShow(ButtonAC);
 End; {Procedure TCalcFrm.ButtonACClick}
{------------------------------------------------------------------------------}
Procedure TCalcFrm.ButtonCClick(Sender: TObject);
{Defines the behavior of the clear key. Clears the last entry.}

 Begin
  Display.Caption := '0';
  LastInput := cClear;
  DecFlag := False;

 End; {TCalcFrm.ButtonCClick}
{------------------------------------------------------------------------------}
Procedure TCalcFrm.ButtonDecClick(Sender: TObject);
{Defines the behavior of the decimal point key.}

 Begin
  If LastInput <> cDigit Then
   Display.Caption := '0.'
  Else If DecFlag = False Then
   If Length(Display.Caption) < cMaxChars Then
    Display.Caption := Display.Caption + '.';

  DecFlag := True;
  LastInput := cDigit;

 End; {TCalcFrm.ButtonDecClick}
{------------------------------------------------------------------------------}
Procedure TCalcFrm.OperatorClick(Sender: TObject);
{Defines the behavior of the operator keys (+, -, *, /, =).}

 Var
  vOperCaption: Char;

 Begin
  With Sender as TButton Do vOperCaption := Caption[1];
  If LastInput = cDigit Then Inc(NumOper);

  If NumOper = 1 Then
   Oper1 := FuncVal(Display.Caption)
  Else If NumOper = 2 Then
   Begin
    Oper2 := FuncVal(Display.Caption);

    Case OperFlag of
     '+': Oper1 := Oper1 + Oper2;
     '-': Oper1 := Oper1 - Oper2;
     '*': Oper1 := Oper1 * Oper2;
     '/': If Oper2 = 0 Then
           MessageDlg(' You cannot divide by zero ', mtError, [mbOK], 0)
          Else
           Oper1 := Oper1 / Oper2;
     '=': Oper1 := Oper2
    End;

    Display.Caption := FuncStr(Oper1);
    NumOper := 1;
   End;

   LastInput := cOper;
   OperFlag := vOperCaption;
 End; {Procedure TCalcFrm.OperatorClick}
{------------------------------------------------------------------------------}
Procedure TCalcFrm.FormKeyPress(Sender: TObject; var Key: Char);
{Handles user key press.}

 Begin
  Case Key of
   '0': Begin
         NumberClick(Button0);
         Button0.SetFocus;
        End;
   '1': Begin
         NumberClick(Button1);
         Button1.SetFocus;
        End;
   '2': Begin
         NumberClick(Button2);
         Button2.SetFocus;
        End;
   '3': Begin
         NumberClick(Button3);
         Button3.SetFocus;
        End;
   '4': Begin
         NumberClick(Button4);
         Button4.SetFocus;
        End;
   '5': Begin
         NumberClick(Button5);
         Button5.SetFocus;
        End;
   '6': Begin
         NumberClick(Button6);
         Button6.SetFocus;
        End;
   '7': Begin
         NumberClick(Button7);
         Button7.SetFocus;
        End;
   '8': Begin
         NumberClick(Button8);
         Button8.SetFocus;
        End;
   '9': Begin
         NumberClick(Button9);
         Button9.SetFocus;
        End;
   '.': Begin
         ButtonDecClick(ButtonDec);
         ButtonDec.SetFocus;
        End;
   '=': Begin
         OperatorClick(ButtonEqu);
         ButtonEqu.SetFocus;
        End;
   '-': Begin
         OperatorClick(ButtonMin);
         ButtonMin.SetFocus;
        End;
   '/': Begin
         OperatorClick(ButtonDiv);
         ButtonDiv.SetFocus;
        End;
   '*': Begin
         OperatorClick(ButtonMul);
         ButtonMul.SetFocus;
        End;
   '+': Begin
         OperatorClick(ButtonAdd);
         ButtonAdd.SetFocus;
        End;
   'C', 'c': Begin
              ButtonCClick(ButtonC);
              ButtonC.SetFocus;
             End;
   'A', 'a': Begin
              ButtonACClick(ButtonAC);
              ButtonAC.SetFocus;
             End;
   'P', 'p': ButtonPutClick(ButtonPut);
   'G', 'g': Begin
               ButtonGetClick(ButtonGet);
               ButtonGet.SetFocus;
             End;
  End;
 End;
{------------------------------------------------------------------------------}
Procedure TCalcFrm.ButtonPutClick(Sender: TObject);
{Defines the behavior of the Put button.}

 Const
  cCrLf = #10 + #13;
  cMsg = 'The calculator display will not' + cCrLf +
         'fit into the data entry field! ';

 Var
   vMaxChars: Integer;

 Begin
   vMaxChars := vDollars + vCents + 1;

   If Length(Display.Caption) <= vMaxChars Then
     Begin
       vPutKey := True;
       vGlobalText := FuncLeftTrim(Display.Caption);
       CalcFrm.Hide;

       {Simulate the ENTER key so as to activate the dollar field's
        OnKeyPress event. This will immediately update the dollar
        field without any further user response.}
       If SendKeys('ENTER') <> sk_None Then MessageBeep(0);
     End
   Else
     MessageDlg(cMsg, mtWarning, [mbOK], 0);
 End;
{------------------------------------------------------------------------------}
Procedure TCalcFrm.ButtonGetClick(Sender: TObject);
{Defines the behavior of the Get button.}

 Begin
   Display.Caption := vGlobalText;
   LastInput := cDigit;
   If Pos('.', Display.Caption) > 0 Then DecFlag := True;
 End;
{------------------------------------------------------------------------------}
Procedure ProcDisplayCalculator;
{Checks if the calculator form has already been created, and if not, it
 creates it, and displays it. Otherwise, it just display the calculator
 form.}

 Begin
   If Not vCreated Then
     Begin
       CalcFrm := TCalcFrm.CreateNew(Application);

       With CalcFrm Do
         Begin
           Left := 218;
           Top := 123;
           Width := 215;
           Height := 240;
           BorderStyle := bsDialog;
           Caption := 'CALCULATOR';
           Color := clTeal;
           Font.Color := clWindowText;
           Font.Height := -13;
           Font.Name := 'System';
           Font.Style := [];
           KeyPreview := True;
           PixelsPerInch := 96;
           OnKeyPress := FormKeyPress;
           OnShow := FormShow;

           Button1 := TButton.Create(CalcFrm);
           With Button1 Do
             Begin
               Parent := CalcFrm;
               Left := 8;
               Top := 128;
               Width := 30;
               Height := 30;
               Caption := '1';
               TabOrder := 10;
               OnClick := NumberClick;
             End;

           Button2 := TButton.Create(CalcFrm);
           With Button2 Do
             Begin
               Parent := CalcFrm;
               Left := 48;
               Top := 128;
               Width := 30;
               Height := 30;
               Caption := '2';
               TabOrder := 11;
               OnClick := NumberClick;
             End;

           Button3 := TButton.Create(CalcFrm);
           With Button3 Do
             Begin
               Parent := CalcFrm;
               Left := 88;
               Top := 128;
               Width := 30;
               Height := 30;
               Caption := '3';
               TabOrder := 12;
               OnClick := NumberClick;
             End;

           Button4 := TButton.Create(CalcFrm);
           With Button4 Do
             Begin
               Parent := CalcFrm;
               Left := 8;
               Top := 88;
               Width := 30;
               Height := 30;
               Caption := '4';
               TabOrder := 5;
               OnClick := NumberClick;
             End;

           Button5 := TButton.Create(CalcFrm);
           With Button5 Do
             Begin
               Parent := CalcFrm;
               Left := 48;
               Top := 88;
               Width := 30;
               Height := 30;
               Caption := '5';
               TabOrder := 6;
               OnClick := NumberClick;
             End;

           Button6 := TButton.Create(CalcFrm);
           With Button6 Do
             Begin
               Parent := CalcFrm;
               Left := 88;
               Top := 88;
               Width := 30;
               Height := 30;
               Caption := '6';
               TabOrder := 7;
               OnClick := NumberClick;
             End;

           Button7 := TButton.Create(CalcFrm);
           With Button7 Do
             Begin
               Parent := CalcFrm;
               Left := 8;
               Top := 48;
               Width := 30;
               Height := 30;
               Caption := '7';
               TabOrder := 0;
               OnClick := NumberClick;
             End;

           Button8 := TButton.Create(CalcFrm);
           With Button8 Do
             Begin
               Parent := CalcFrm;
               Left := 48;
               Top := 48;
               Width := 30;
               Height := 30;
               Caption := '8';
               TabOrder := 1;
               OnClick := NumberClick;
             End;

           Button9 := TButton.Create(CalcFrm);
           With Button9 Do
             Begin
               Parent := CalcFrm;
               Left := 88;
               Top := 48;
               Width := 30;
               Height := 30;
               Caption := '9';
               TabOrder := 2;
               OnClick := NumberClick;
             End;


           Button0 := TButton.Create(CalcFrm);
           With Button0 Do
             Begin
               Parent := CalcFrm;
               Left := 8;
               Top := 168;
               Width := 30;
               Height := 30;
               Caption := '0';
               TabOrder := 15;
               OnClick := NumberClick;
             End;

           ButtonDec := TButton.Create(CalcFrm);
           With ButtonDec Do
             Begin
               Parent := CalcFrm;
               Left := 48;
               Top := 168;
               Width := 30;
               Height := 30;
               Caption := '.';
               TabOrder := 16;
               OnClick := ButtonDecClick;
             End;

           ButtonEqu := TButton.Create(CalcFrm);
           With ButtonEqu Do
             Begin
               Parent := CalcFrm;
               Left := 88;
               Top := 168;
               Width := 30;
               Height := 30;
               Caption := '=';
               TabOrder := 17;
               OnClick := OperatorClick;
             End;

           ButtonMin := TButton.Create(CalcFrm);
           With ButtonMin Do
             Begin
               Parent := CalcFrm;
               Left := 128;
               Top := 168;
               Width := 30;
               Height := 30;
               Caption := '-';
               TabOrder := 18;
               OnClick := OperatorClick;
             End;

           ButtonDiv := TButton.Create(CalcFrm);
           With ButtonDiv Do
             Begin
               Parent := CalcFrm;
               Left := 128;
               Top := 48;
               Width := 30;
               Height := 30;
               Caption := '/';
               TabOrder := 3;
               OnClick := OperatorClick;
             End;

           ButtonMul := TButton.Create(CalcFrm);
           With ButtonMul Do
             Begin
               Parent := CalcFrm;
               Left := 128;
               Top := 88;
               Width := 30;
               Height := 30;
               Caption := '*';
               TabOrder := 8;
               OnClick := OperatorClick;
             End;

           ButtonAdd := TButton.Create(CalcFrm);
           With ButtonAdd Do
             Begin
               Parent := CalcFrm;
               Left := 128;
               Top := 128;
               Width := 30;
               Height := 30;
               Caption := '+';
               TabOrder := 13;
               OnClick := OperatorClick;
             End;

           ButtonC := TButton.Create(CalcFrm);
           With ButtonC Do
             Begin
               Parent := CalcFrm;
               Left := 168;
               Top := 48;
               Width := 30;
               Height := 30;
               Caption := 'C';
               TabOrder := 4;
               OnClick := ButtonCClick;
             End;

           ButtonAC := TButton.Create(CalcFrm);
           With ButtonAC Do
             Begin
               Parent := CalcFrm;
               Left := 168;
               Top := 88;
               Width := 30;
               Height := 30;
               Caption := 'AC';
               TabOrder := 9;
               OnClick := ButtonACClick;
             End;

           ButtonGet := TButton.Create(CalcFrm);
           With ButtonGet Do
             Begin
               Parent := CalcFrm;
               Left := 168;
               Top := 128;
               Width := 30;
               Height := 30;
               Caption := 'Get';
               TabOrder := 14;
               OnClick := ButtonGetClick;
             End;

           ButtonPut := TButton.Create(CalcFrm);
           With ButtonPut Do
             Begin
               Parent := CalcFrm;
               Left := 168;
               Top := 168;
               Width := 30;
               Height := 30;
               Caption := 'Put';
               TabOrder := 19;
               OnClick := ButtonPutClick;
             End;

           Panel1 := TPanel.Create(CalcFrm);
           With Panel1 Do
             Begin
               Parent := CalcFrm;
               Left := 8;
               Top := 8;
               Width := 189;
               Height := 24;
               BevelOuter := bvLowered;
             End;

           Display := TLabel.Create(CalcFrm);
           With Display Do
             Begin
               Parent := Panel1;
               Left := 180;
               Top := 5;
               Width := 0;
               Height := 16;
               Alignment := taRightJustify;
             End;
         End;

       vCreated := True;
     End;

   CalcFrm.Show;
 End;
{------------------------------------------------------------------------------}
Function FuncLeftTrim(Const vStr: String): String;
{Returns a string with leading spaces removed. This function is required
 when the calculator display has to be assigned to the global text.}

 Const
   vSpace = #$20;

 Var
   vLen: Byte Absolute vStr;
   vIndex: Integer;

 Begin
   vIndex := 1;
   While (vIndex <= vLen) And (vStr[vIndex] = vSpace) Do Inc(vIndex);
   FuncLeftTrim := Copy(vStr, vIndex, vLen)
 End;
{------------------------------------------------------------------------------}
Function FuncUnFormat(vStr: String): String;
{Unformats the field by removing the thousands separator.}

 Var
   vFilterStr, vTemp, T: String;
   vLen, vCount : Integer;

   {Isolate each character in the string.}
   Begin
    vFilterStr := '1234567890.-';
    vTemp := '';
    vLen := Length(vStr);

    For vCount := 1 to vLen Do
     Begin
      T := Copy(vStr, vCount, 1);
      {If the character is in the filter string, save it.}
      If Pos(T, vFilterStr) <> 0 Then
       vTemp := vTemp + T;
     End;
    FuncUnFormat := vTemp
   End; {Function FuncUnFormat}
{------------------------------------------------------------------------------}
Procedure Register;
	Begin
 	RegisterComponents('Samples', [TDollarEdit]);
	End;

End.
