Unit SendKey;

Interface

Uses
 SysUtils, WinTypes, WinProcs, Messages, Classes, KeyDefs;

Type
  { Error codes }
  TSendKeyError = (sk_None, sk_FailSetHook, sk_InvalidToken, sk_UnknownError);

  { exceptions }
  ESendKeyError = Class(Exception);
  ESetHookError = Class(ESendKeyError);
  EInvalidToken = Class(ESendKeyError);

  { a TList descendant that know how to dispose of its contents }
  TMessageList = Class(TList)
  Public
    Destructor Destroy; Override;
  End;

Function SendKeys(S: String): TSendKeyError;

Implementation

Destructor TMessageList.Destroy;
Var
  i: longint;
Begin
  { deallocate all the message records before discarding the list }
  For i := 0 To Count - 1 Do
    Dispose(PEventMsg(Items[i]));
  Inherited Destroy;
End;

Var
  { Variables global to the Unit }
  MsgCount: word;
  MessageBuffer: TEventMsg;
  HookHandle: hHook;
  Playing: Boolean;
  MessageList: TMessageList;
  AltPressed, ControlPressed, ShiftPressed: Boolean;
  NextSpecialKey: TKeyString;
{------------------------------------------------------------------------------}
Function MakeWord(L, H: Byte): Word;
{ macro creates a word from low and high bytes }
Inline(
  $5A/            { pop dx }
  $58/            { pop ax }
  $8A/$E2);       { mov ah, dl }
{------------------------------------------------------------------------------}
Procedure StopPlayback;
{ Unhook the hook, and clean up }
Begin
  { if Hook is currently active, then unplug it }
  If Playing Then
    UnhookWindowsHookEx(HookHandle);
  MessageList.Free;
  Playing := False;
End;
{------------------------------------------------------------------------------}
Function Play(Code: integer; wParam: word; lParam: Longint): Longint; export;
{ This is the JournalPlayback callback function.  It is called by Windows }
{ when Windows polls for hardware events.  The code parameter indicates what }
{ to do. }
Begin
  Case Code Of

    hc_Skip: Begin
    { hc_Skip means to pull the next message out of our list. If we }
    { are at the end of the list, it's okay to unhook the JournalPlayback }
    { hook from here. }
      { increment message counter }
      inc(MsgCount);
      { check to see if all messages have been played }
      If MsgCount >= MessageList.Count Then
        StopPlayback
      Else
      { copy next message from list into buffer }
      MessageBuffer := TEventMsg(MessageList.Items[MsgCount]^);
      Result := 0;
    End;

    hc_GetNext: Begin
    { hc_GetNext means to fill the wParam and lParam with the proper }
    { values so that the message can be played back.  DO NOT unhook }
    { hook from within here.  Return value indicates how much time until }
    { Windows should playback message.  We'll return 0 so that it's }
    { processed right away. }
      { move message in buffer to message queue }
      PEventMsg(lParam)^ := MessageBuffer;
      Result := 0  { process immediately }
    End

    Else
      { if Code isn't hc_Skip or hc_GetNext, then call next hook in chain }
      Result := CallNextHookEx(HookHandle, Code, wParam, lParam);
  End;
End;
{------------------------------------------------------------------------------}
Procedure StartPlayback;
{ Initializes globals and sets the hook }
Begin
  { grab first message from list and place in buffer in case we }
  { get a hc_GetNext before and hc_Skip }
  MessageBuffer := TEventMsg(MessageList.Items[0]^);
  { initialize message count and play indicator }
  MsgCount := 0;
  { initialize Alt, Control, and Shift key flags }
  AltPressed := False;
  ControlPressed := False;
  ShiftPressed := False;
  { set the hook! }
  HookHandle := SetWindowsHookEx(wh_JournalPlayback, Play, hInstance, 0);
  If HookHandle = 0 Then
    Raise ESetHookError.Create('Couldn''t set hook')
  Else
    Playing := True;
End;
{------------------------------------------------------------------------------}
Procedure MakeMessage(vKey: byte; M: word);
{ procedure builds a TEventMsg record that emulates a keystroke and }
{ adds it to message list }
Var
  E: PEventMsg;
Begin
  New(E);                                 { allocate a message record }
  With E^ Do Begin
    Message := M;                         { set message field }
    { high byte of ParamL is the vk code, low byte is the scan code }
    ParamL := MakeWord(vKey, MapVirtualKey(vKey, 0));
    ParamH := 1;                          { repeat count is 1 }
    Time := GetTickCount;                 { set time }
  End;
  MessageList.Add(E);
End;
{------------------------------------------------------------------------------}
Procedure KeyDown(vKey: byte);
{ Generates KeyDownMessage }
Begin
  { don't generate a "sys" key if the control key is pressed (Windows quirk) }
  If (AltPressed And (Not ControlPressed) And (vKey In [Ord('A')..Ord('Z')])) Or
     (vKey = vk_Menu) Then
    MakeMessage(vKey, wm_SysKeyDown)
  Else
    MakeMessage(vKey, wm_KeyDown);
End;
{------------------------------------------------------------------------------}
Procedure KeyUp(vKey: byte);
{ Generates KeyUp message }
Begin
  { don't generate a "sys" key if the control key is pressed (Windows quirk) }
  If AltPressed And (Not ControlPressed) And (vKey In [Ord('A')..Ord('Z')]) Then
    MakeMessage(vKey, wm_SysKeyUp)
  Else
    MakeMessage(vKey, wm_KeyUp);
End;
{------------------------------------------------------------------------------}
Procedure SimKeyPresses(VKeyCode: Word);
{ This function simulates keypresses for the given key, taking into }
{ account the current state of Alt, Control, and Shift keys }
Begin
  { press Alt key if flag has been set }
  If AltPressed Then
    KeyDown(vk_Menu);
  { press Control key if flag has been set }
  If ControlPressed Then
    KeyDown(vk_Control);
  { if shift is pressed, or shifted key and control is not pressed... }
  If (((Hi(VKeyCode) And 1) <> 0) And (Not ControlPressed)) Or ShiftPressed Then
    KeyDown(vk_Shift);    { ...press shift }
  KeyDown(Lo(VKeyCode));  { press key down }
  KeyUp(Lo(VKeyCode));    { release key }
  { if shift is pressed, or shifted key and control is not pressed... }
  If (((Hi(VKeyCode) And 1) <> 0) And (Not ControlPressed)) Or ShiftPressed Then
    KeyUp(vk_Shift);      { ...release shift }
  { if shift flag is set, reset flag }
  If ShiftPressed Then Begin
    ShiftPressed := False;
  End;
  { Release Control key if flag has been set, reset flag }
  If ControlPressed Then Begin
    KeyUp(vk_Control);
    ControlPressed := False;
  End;
  { Release Alt key if flag has been set, reset flag }
  If AltPressed Then Begin
    KeyUp(vk_Menu);
    AltPressed := False;
  End;
End;
{------------------------------------------------------------------------------}
Procedure ProcessKey(S: String);
{ This function parses each character in the string to create the message list }
Var
  KeyCode: word;
  Key: byte;
  index: integer;
  Token: TKeyString;
Begin
  index := 1;
  Repeat
    Case S[index] Of

      KeyGroupOpen : Begin
      { It's the beginning of a special token! }
        Token := '';
        inc(index);
        While S[index] <> KeyGroupClose Do Begin
          { add to Token until the end token symbol is encountered }
          Token := Token + S[index];
          inc(index);
          { check to make sure the token's not too long }
          If (Length(Token) = 7) And (S[index] <> KeyGroupClose) Then
            Raise EInvalidToken.Create('No closing brace');
        End;
        { look for token in array, Key parameter will }
        { contain vk code if successful }
        If Not FindKeyInArray(Token, Key) Then
          raise EInvalidToken.Create('Invalid token');
        { simulate keypress sequence }
        SimKeyPresses(MakeWord(Key, 0));
      End;

      AltKey : Begin
        { set Alt flag }
        AltPressed := True;
      End;

      ControlKey : Begin
        { set Control flag }
        ControlPressed := True;
      End;

      ShiftKey : Begin
        { set Shift flag }
        ShiftPressed := True;
      End;

      Else Begin
      { A normal character was pressed }
        { convert character into a word where the high byte contains }
        { the shift state and the low byte contains the vk code }
        KeyCode := vkKeyScan(MakeWord(Byte(S[index]), 0));
        { simulate keypress sequence }
        SimKeyPresses(KeyCode);
      End;
    End;
    inc(index);
  until index > Length(S);
End;
{------------------------------------------------------------------------------}
Function SendKeys(S: String): TSendKeyError;
{ This is the one entry point.  Based on the string passed in the S  }
{ parameter, this function creates a list of keyup/keydown messages, }
{ sets a JournalPlayback hook, and replays the keystroke messages.   }
Var
  i: byte;
Begin
  Try
    Result := sk_None;                   { assume success }
    MessageList := TMessageList.Create;  { create list of messages }
    ProcessKey(S);                       { create messages from string }
    StartPlayback;                       { set hook and play back messages }
  Except
    { if an exception occurs, return an error code, and clean up }
    On E:ESendKeyError Do Begin
      MessageList.Free;
      If E Is ESetHookError Then
        Result := sk_FailSetHook
      Else If E Is EInvalidToken Then
        Result := sk_InvalidToken;
    End
    Else
      { Catch-all exception handler ensures than an exception }
      { doesn't walk up into application stack }
      Result := sk_UnknownError;
  End;
End;

End.

