/* Main UnZip DLL module
 * Author: Eric W. Engler, May 5, 1997
 * Based on Mike White's DLL, but is dissimilar in many ways.
 * This module has the entry point for ALL calls from Delphi.
 */
#ifndef UNZIP_INTERNAL
#define UNZIP_INTERNAL
#endif

// EWE note: we get globals.h from this path:
// unzip.h -> unzpriv.h -> globals.h
#include <windows.h>
#include "unzip.h"
#include "version.h"
#include "wizunzip.h"
#include "consts.h"

BOOL FSetUpToProcessZipFile(LPDCL C);
void TakeDownFromProcessZipFile(void);

// see wizunzip.h
DLLCALLBK callb;
callbackstruct  callbackdata;

/* These are DLL specific globals. */
char ewemsg[2048];
char ewetmp[2048];  
int  global_trace_opt;
int  global_abort_sw;
int  global_error_code;
int  files_acted_on;
int  dll_handles_errors;
int  user_notified_of_abort;
long global_caller;
HWND global_handle;

// Declare Unzip Misc Buffers (UMB)  
UMB Buffers;

// Declare a pointer to Unzip Misc Buffers (UMB)  
LPUMB lpumb=&Buffers;

HANDLE hOutBuf;
HANDLE hInBuf;
HANDLE hZipFN;
HANDLE hwildZipFN;
HANDLE hFileName;


/* DLL Entry Point */
#ifdef __BORLANDC__
#pragma warn -par
#endif
BOOL WINAPI DllEntryPoint( HINSTANCE hinstDll,
                           DWORD fdwRreason,
                           LPVOID plvReserved)
{
   return 1;   /* Indicate that the DLL was initialized successfully. */
}

#ifdef __BORLANDC__
#pragma warn .par
#endif
extern int  zipmain OF((int, char **));

/* Main entry point to unzip files. */
long WINAPI UnzDllExec(DCL *C)
{
   int retcode;
   CONSTRUCTGLOBALS();    /* see globals.c/globals.h */


   globalsCtor();   /* init the globals */
   global_abort_sw=0;
   global_error_code=0;
   files_acted_on=0;
   user_notified_of_abort=0;
  
   global_handle=C->handle;  // handle can be 0
   global_caller=C->caller;  // we just give this back in the callback call

   /* point to the Delphi callback function */
   callb=C->callback;

   global_trace_opt=C->fTraceEnabled;
   diag("trace is on in UNZDLL.DLL");

   /*----------------------------------------------------------------*/
   if (C->lpszZipFN == NULL) /* Something screwed up, we don't have a filename */
      {
      diag("no zip filename received");
      return -1;
      }
   if (C->seven != 7)
      {
      diag("struct size mismatch");
      return -2;
      }
   if (C->version != UNZVERS)  // see version.h
      {
      // This msgbox won't appear if user didn't pass us a nonzero 
      // Window handle
      msgbox("Warning: UNZDLL.DLL version may be incorrect");
      // "fprintf" will report the error via the Delphi callback
      fprintf(stderr,"Warning: UNZDLL.DLL version may be incorrect");
      }

   /*----------------------------------------------------------------*/

   diag("ready to setup");
   FSetUpToProcessZipFile(C);          // parse our cmd line

   if (global_trace_opt)
      G.vflag=1;    // if tracing, make sure verbose is on

   diag("*** READY TO CALL process_zipfiles in process.c ***");
   retcode = process_zipfiles(__G);    // pass ptr to global bufs
   diag("*** BACK FROM CALL TO process_zipfiles ***");

   TakeDownFromProcessZipFile();

   DESTROYGLOBALS();        // in globals.c/globals.h */
   if (G.vflag)
      printf("Files acted on = %d", files_acted_on);
   return files_acted_on;
}

long WINAPI GetUnzDllVersion(void)
{  /* see version.h */
   return UNZVERS;
}

BOOL FSetUpToProcessZipFile(LPDCL C)  
{
    /* G.anything refers to globals in file globals.h */
    G.key = (char *)NULL;

    /* clear all global flags -- need to or not. */
    G.tflag=G.vflag=G.cflag=G.aflag=G.uflag=G.qflag=G.zflag=0;
    G.overwrite_all=G.overwrite_none=0;
    G.pfnames = &fnames[0]; // assign default file name vector 

    // These flags all have a fixed value in this version
    G.extract_flag = 1;
    G.cflag = 0;        // if true, write to stdout (N/A for us)
    G.C_flag = 1;       // if true, match filenames case-insensitively
    G.tflag = 0;        // if true, test zipfile
    G.zflag = 0;        // show zip comment (not yet supported)
    G.hflag = 0;        /* -h: header line (zipinfo) */
    G.lflag = 0;        /* -12slmv: listing format (zipinfo) */
    G.L_flag = 0;       /* -L: convert filenames from some OSes to lowercase */
    G.sflag = 0;        /* -s: convert spaces in filenames to underscores */
    G.volflag = 0;      /* -$: extract volume labels */
    G.T_flag = 1;       /* -T: timestamps (unzip) or dec. time fmt (zipinfo) */
    G.V_flag = 0;       /* -V: don't strip VMS version numbers */
    G.xfilespecs = 0;   /* number of excluded filespecs to be matched */
    G.zipinfo_mode=0;   /* behave like ZipInfo or like normal UnZip? */

    // set options from caller
    G.create_dirs = C->fDirectories;  // used by main(), mapname(), checkdir()
    G.dflag = C->fDirectories;        // "recreate dir structure"
    G.jflag = !(G.dflag);             // "junk pathnames"

    G.aflag = C->fConvert;       // do ASCII/EBCDIC or EOL conversions
    G.vflag = C->fVerbose;       // verbose flag
    G.qflag = C->fQuiet;         // quiet flag
    if (C->handle == 0)
       G.qflag = TRUE;           // if we don't have a window handle, then
                                 // we need to be quiet (no dialog boxes)

    G.uflag = C->fUpdate;        // "Update" - extract only newer files & brand new files
    G.fflag = C->fFreshen;       // "freshen" (extract only newer files)

    if (C->fOverwrite) {
        G.overwrite_all = TRUE;  // don't ask, always overwrite
        G.overwrite_none = FALSE;
    } else {
        G.overwrite_all = FALSE;
        G.overwrite_none = TRUE; // don't overwrite or ask; skip that file
    }

    G.filespecs = C->argc;  // number of fspecs

    G.local_hdr_sig[0] = G.central_hdr_sig[0] = G.end_central_sig[0] = '\120';
    G.local_hdr_sig[1] = G.central_hdr_sig[1] = G.end_central_sig[1] = '\0';

    G.cur_zipfile_bufstart = 0L;
    if ((hZipFN = GlobalAlloc(GMEM_MOVEABLE, FILNAMSIZ))== NULL)
        return FALSE;

    G.zipfn = (char far *)GlobalLock(hZipFN);
    lstrcpy(G.zipfn, C->lpszZipFN);

/* MW: G.wildzipfn needs to be initialized so that do_wild does not wind
   up clearing out the zip file name when it returns in process.c
*/
    if ((hwildZipFN = GlobalAlloc(GMEM_MOVEABLE, FILNAMSIZ))== NULL)
        return FALSE;
    G.wildzipfn = GlobalLock(hwildZipFN);
    lstrcpy(G.wildzipfn, C->lpszZipFN);

    if (stat(G.zipfn, &G.statbuf) || (G.statbuf.st_mode & S_IFMT) == S_IFDIR)
        strcat(G.zipfn, ZSUFX);
    if (stat(G.zipfn, &G.statbuf)) {  /* try again */
        fprintf(stdout, "error:  can't find zipfile [ %s ]\n", G.zipfn);
        return FALSE;
    } else
        G.ziplen = G.statbuf.st_size;

    if (C->argc == 0) 
       G.process_all_files = TRUE;   // skip fspec matching for speed
    else
       {
       G.pfnames = C->FNV;
       G.process_all_files = FALSE;  // we must check fspecs
       }

    Trace((stderr,"argc = %d, process_all_files = %d", C->argc, G.process_all_files));
/*---------------------------------------------------------------------------
    Ok, we have everything we need to get started.  
  ---------------------------------------------------------------------------*/
    if ((hInBuf = GlobalAlloc(GMEM_MOVEABLE, INBUFSIZ+4)) != NULL) {
        G.inbuf = (uch *) GlobalLock(hInBuf);
        WinAssert(G.inbuf);
    }
    if ((hOutBuf = GlobalAlloc(GMEM_MOVEABLE, OUTBUFSIZ+1)) != NULL) {
        G.outbuf = (uch *)GlobalLock(hOutBuf);
        WinAssert(G.outbuf);
      /*  outout = G.outbuf; */ /*  point to outbuf */
    }
    if ((hFileName = GlobalAlloc(GMEM_MOVEABLE, FILNAMSIZ)) != 0) {
        G.filename = GlobalLock(hFileName);
        WinAssert(G.filename);
    }
    if ((G.inbuf == NULL) || (G.outbuf == NULL) || 
        (G.zipfn == NULL) || (G.filename == NULL))
        return FALSE;

    G.hold = &G.inbuf[INBUFSIZ]; /* to check for boundary-spanning signatures */
    return TRUE;    /* set up was OK */
}

void TakeDownFromProcessZipFile(void)
{
    if (G.inbuf) {
        GlobalUnlock(hInBuf);
        G.inbuf = NULL;
    }
    if (hInBuf)
        hInBuf = GlobalFree(hInBuf);

    if (G.outbuf) {
        GlobalUnlock(hOutBuf);
        G.outbuf = NULL;
    }
    if (hOutBuf)
        hOutBuf = GlobalFree(hOutBuf);

    if (G.zipfn) {
        GlobalUnlock(hZipFN);
        G.zipfn = NULL;
    }
    if (hZipFN)
        hZipFN = GlobalFree(hZipFN);

    if (G.wildzipfn) {
        GlobalUnlock(hwildZipFN);
        G.wildzipfn = NULL;
    }
    if (hwildZipFN)
        hwildZipFN = GlobalFree(hwildZipFN);
    if (G.filename) {
        GlobalUnlock(hFileName);
        G.filename = NULL;
    }
    if (hFileName)
        hFileName = GlobalFree(hFileName);
}

/* fprintf clone to support DLL */
int __far __cdecl fprintf(FILE *file, const char *format, ...)
{
   va_list argptr;
   HANDLE hMemory;
   LPSTR pszBuffer;
   int len;

   va_start(argptr, format);
   hMemory = GlobalAlloc(GMEM_MOVEABLE, STDIO_BUF_SIZE);
   WinAssert(hMemory);
   if (!hMemory)
      return 0;
   pszBuffer = GlobalLock(hMemory);
   WinAssert(pszBuffer);
   len = wvsprintf(pszBuffer, format, argptr);
   va_end(argptr);
   WinAssert(strlen(pszBuffer) < STDIO_BUF_SIZE);

   // warning msg, or info msg only
   user_callback(4, 0, 0, pszBuffer);

   GlobalUnlock(hMemory);
   GlobalFree(hMemory);
   return len;
}

int __cdecl MyTrace(FILE *file, const char *format, ...)
{
   va_list argptr;
   HANDLE hMemory;
   LPSTR pszBuffer;
   int len;

   va_start(argptr, format);
   hMemory = GlobalAlloc(GMEM_MOVEABLE, STDIO_BUF_SIZE);
   WinAssert(hMemory);
   if (!hMemory)
      return 0;
   pszBuffer = GlobalLock(hMemory);
   WinAssert(pszBuffer);
   len = wvsprintf(pszBuffer, format, argptr);
   va_end(argptr);
   WinAssert(strlen(pszBuffer) < STDIO_BUF_SIZE);

   diag(pszBuffer);

   GlobalUnlock(hMemory);
   GlobalFree(hMemory);
   return len;
}

void msgbox(char *msg)
{
    if (global_handle == 0)
       // user didn't pass us a good window handle - we can't pop-up a box
       exit;
    /* bring up a dialog box */
    strcpy(ewetmp, "Msg From UNZIP DLL: ");
    strcat(ewetmp, msg);
    MessageBox(global_handle, ewetmp, "Message From UNZIP DLL", MB_OK);
}

void diag(char *msg)
{
   if (global_trace_opt)
      {
      /* log the message through the routine msg callback */
      strcpy(ewetmp, "Trace Msg: ");
      strcat(ewetmp, msg);
      /* specify a 0 error code to prevent a dialog box from coming 
         up in the Delphi applic pgm */
      user_callback(4, 0, 0, ewetmp);  
      }
}

#ifdef NEVER
   This provides the calling program with updated info on what the DLL 
   is doing.  Regardless of the type of call being made, the user's
   function must have a spin of the Windows message loop.  In fact, even
   if user isn't using a progress bar, he should still spin the msg 
   loop upon getting these callbacks (but he doesn't need to do anything 
   else).  In Delphi, "Application.ProcessMessages;" spins the loop.
   Here are the types of calls:

     ActionCode = 1, we're starting a zip operation on a new file
        error_code = N/A
        fsize = filesize of file we're going to operate on
        name_or_msg = pathname of file 
     IMPORTANT: The user's function must do the math for the progress
     bar upon getting this call.  See the Delphi sample application.

     ActionCode = 2, increment the progress bar
        These calls will occur after every 32K of input file has been
        processed. One additional call is made at the end of each file, 
        just to make sure the progress bar is max'ed out - this is also
        critical for files less than 32K in size (this last one will be 
        their only one).
        error_code = N/A
        fsize = N/A
        name_or_msg = N/A 

     ActionCode = 3, we're done with a batch of files - program flow
     will quickly return to the user's program.
     NOTE: the end of a every file will always be followed by an
           action of 1 or 3, so a separate call for end of one file
           isn't needed.
        error_code = N/A
        fsize = N/A 
        name_or_msg = N/A 

     ActionCode = 4, a routine message is being passed
        error_code = code corresponding to message (not widely used yet)
        fsize = N/A
        name_or_msg = text of message
#endif

// This calls the application program and passes status info
void user_callback(long action, long error_code, long fsize, char *name_or_msg)
{
   callbackdata.handle=global_handle;  /* Window handle of caller */
   callbackdata.caller=global_caller;  /* object instance pointer of caller */
                            /* (from the initial call to DllProcessZipFiles)*/
   callbackdata.version=UNZVERS;
   callbackdata.isoperationzip=FALSE; // true=zip, false=unzip
   callbackdata.actioncode=action;
   callbackdata.error_code=error_code;
   callbackdata.fsize=fsize;
   if (name_or_msg == NULL)
      callbackdata.filenameormsg[0]=0;
   else
      strcpy(callbackdata.filenameormsg, name_or_msg);

   /* make the call and check the return code.  If true is returned, user
      wants to abort the current batch job */
   global_abort_sw=callb(&callbackdata);  // call user's program

   // don't bother looking at the global_abort_sw here!
}

// Custom version of ziperr used by InfoZip's ZIP program.
void unzerr(c)
int c;                  /* error code from the ZE_ class - unzerr.h */
{
  static char errmsg[80];
     
  if (user_notified_of_abort)
     return;
  user_notified_of_abort=1;

  global_error_code = c;   /* last error code */

  switch(c) {
     case ZE_MISS:
     case ZE_OPEN:
        strcpy(errmsg,"File not found");
        break;
     case ZE_EOF:
        strcpy(errmsg,"Unexpected end of file");
        break;
     case ZE_MEM:
        strcpy(errmsg,"Out of memory");
        break;
     case ZE_ABORT:
        strcpy(errmsg,"Aborted by User");
        break;
     case ZE_READ:
        strcpy(errmsg, "Error reading a file");
        break;
     case ZE_WRITE:
        strcpy(errmsg, "Error writing to a file");
        break;
     case ZE_CREAT:
        strcpy(errmsg, "Error creating file");
        break;
     case ZE_NAME:
        strcpy(errmsg, "Missing or empty zip file");
        break;
     case ZE_PARMS:
        strcpy(errmsg, "Bad zip options specified");
        break;
     case ZE_FORM:
        strcpy(errmsg, "zip file structure error");
        break;

     /* The less likely errors: */
     case ZE_NONE:
        strcpy(errmsg, "Unknown error occured");
  } /* end switch */
             

  if (dll_handles_errors)
     {
     // I'm passing the error via the callback just to get it logged in
     // the status box - I'm sending it in with a 0 error code to avoid
     // a dialog box from the applic pgm.
     user_callback(4,0,0,errmsg);  
     sprintf(ewemsg, "%s   code=%d", errmsg, c);
     msgbox(ewemsg);
     }
  else
     user_callback(4,c,0,errmsg);  /* applic pgm handles errors */
}
