/* Modified Version for Delphi project, Eric W. Engler, Mar 5, 1997 */
/*---------------------------------------------------------------------------
  fileio.c
  This file contains routines for doing direct but relatively generic input/
  output, file-related sorts of things, plus some miscellaneous stuff.  Most
  of the stuff has to do with opening, closing, reading and/or writing files.
  Contains:  open_input_file()
             open_outfile()           (non-VMS, non-AOS/VS, non-CMS_MVS)
             undefer_input()
             defer_leftover_input()
             readbuf()
             readbyte()
             fillinbuf()
             flush()                  (non-VMS)
             disk_error()             (non-VMS)
             handler()
             dos_to_unix_time()       (non-VMS, non-OS/2, non-VM/CMS, non-MVS)
             check_for_newer()        (non-VMS, non-OS/2, non-VM/CMS, non-MVS)
             do_string()
             makeword()
             makelong()
             memset()                 (ZMEM only)
             memcpy()                 (ZMEM only)
             zstrnicmp()
  ---------------------------------------------------------------------------*/
#define FILEIO_C
#define UNZIP_INTERNAL

#include "unzip.h"
#include "crypt.h"
#include "ttyio.h"
/*
   Note: Under Windows, the maximum size of the buffer that can be used
   with any of the *printf calls is 16,384, so win_fprintf was used to
   feed the fprintf clone no more than 16K chunks at a time. This should
   be valid for anything up to 64K (and probably beyond, assuming your
   buffers are that big.)
*/
#ifdef USE_FWRITE
#  define WriteError(buf,len,strm) \
   ((extent)fwrite((char *)(buf),1,(extent)(len),strm) != (extent)(len))
#else
#  define WriteError(buf,len,strm) \
   ((extent)write(fileno(strm),(char *)(buf),(extent)(len)) != (extent)(len))
#endif

static int disk_error OF((__GPRO));

/******************************/
/* Function open_input_file() */
/******************************/
int open_input_file(__G)    /* return 1 if open failed */
    __GDEF
{
    /*
     *  open the zipfile for reading and in BINARY mode to prevent cr/lf
     *  translation, which would corrupt the bitstreams
     */
    Trace((stderr, "in open_input_file of fileio.c; fname=%s", G.zipfn));

    G.zipfd = open(G.zipfn, O_RDONLY | O_BINARY);
    if (G.zipfd < 0)
    {
        fprintf(stderr,"Error opening ZIP file: %s", G.zipfn);
        return 1;
    }
    Trace((stderr,"zip file %s opened OK",G.zipfn));
    return 0;
} /* end function open_input_file() */

/***************************/
/* Function open_outfile() */
/***************************/
int open_outfile(__G)         /* return 1 if fail */
    __GDEF
{
    if (stat(G.filename, &G.statbuf) == 0) {
        Trace((stderr, "open_outfile:  stat(%s) returns 0 (file exists)\n",
          G.filename));
        if (!(G.statbuf.st_mode & S_IWRITE)) {
            Trace((stderr, "open_outfile:  existing file %s is read-only\n",
              G.filename));
            chmod(G.filename, S_IREAD | S_IWRITE);
            Trace((stderr, "open_outfile:  %s now writable\n", G.filename));
        }
        if (unlink(G.filename) != 0)
            return 1;
        if (G.vflag)
           fprintf(stdout,"Deleting pre-existing file %s",G.filename);
        Trace((stderr, "open_outfile:  %s now deleted\n", G.filename));
    }

#ifdef NOVELL_BUG_FAILSAFE
    if (G.dne && ((G.outfile = fopen(G.filename, FOPR)) != (FILE *)NULL)) {
        fprintf(stderr,"File %s open failed due to Novell Bug", G.filename);
        fclose(G.outfile);
        return 1;   /* with "./" fix in checkdir(), should never reach here */
    }
#endif /* NOVELL_BUG_FAILSAFE */

    if ((G.outfile = fopen(G.filename, FOPW)) == (FILE *)NULL) {
        fprintf(stderr,"Error opening file %s for write", G.filename);
        return 1;
    }
    Trace((stderr, "open_outfile:  fopen(%s) for writing succeeded\n",
      G.filename));

#ifdef USE_FWRITE
    setbuf(G.outfile, (char *)NULL);   /* make output unbuffered */
#endif /* USE_FWRITE */
    return 0;
} /* end function open_outfile() */

/* These functions allow NEXTBYTE to function without needing two bounds
 * checks.  Call defer_leftover_input() if you ever have filled G.inbuf
 * by some means other than readbyte(), and you then want to start using
 * NEXTBYTE.  When going back to processing bytes without NEXTBYTE, call
 * undefer_input().  For example, extract_or_test_member brackets its
 * central section that does the decompression with these two functions.
 * If you need to check the number of bytes remaining in the current
 * file while using NEXTBYTE, check (G.csize + G.incnt), not G.csize.
 */
/****************************/
/* function undefer_input() */
/****************************/
void undefer_input(__G)
    __GDEF
{
    if (G.incnt > 0)
        G.csize += G.incnt;
    if (G.incnt_leftover > 0) {
        /* We know that "(G.csize < MAXINT)" so we can cast G.csize to int:
         * This condition was checked when G.incnt_leftover was set > 0 in
         * defer_leftover_input(), and it is NOT allowed to touch G.csize
         * before calling undefer_input() when (G.incnt_leftover > 0)
         * (single exception: see read_byte()'s  "G.csize <= 0" handling) !!
         */
        G.incnt = G.incnt_leftover + (int)G.csize;
        G.inptr = G.inptr_leftover - (int)G.csize;
        G.incnt_leftover = 0;
    } else if (G.incnt < 0)
        G.incnt = 0;
} /* end function undefer_input() */

/***********************************/
/* function defer_leftover_input() */
/***********************************/
void defer_leftover_input(__G)
    __GDEF
{
    if ((long)G.incnt > G.csize) {
        /* (G.csize < MAXINT), we can safely cast it to int !! */
        if (G.csize < 0L)
            G.csize = 0L;
        G.inptr_leftover = G.inptr + (int)G.csize;
        G.incnt_leftover = G.incnt - (int)G.csize;
        G.incnt = (int)G.csize;
    } else
        G.incnt_leftover = 0;
    G.csize -= G.incnt;
} /* end function defer_leftover_input() */

/**********************/
/* Function readbuf() */
/**********************/
unsigned readbuf(__G__ buf, size)   /* return number of bytes read into buf */
    __GDEF
    char *buf;
    register unsigned size;
{
    register unsigned count;
    unsigned n;

    n = size;
    while (size) {
        if (G.incnt <= 0) {
            if ((G.incnt = read(G.zipfd, (char *)G.inbuf, INBUFSIZ)) == 0)
                return (n-size);
            else if (G.incnt < 0) {
                fprintf(stderr,"Error reading ZIP file");
                return 0;  /* discard some data - better than lock-up */
            }
            /* buffer ALWAYS starts on a block boundary:  */
            G.cur_zipfile_bufstart += INBUFSIZ;
            G.inptr = G.inbuf;
        }
        count = MIN(size, (unsigned)G.incnt);
        memcpy(buf, G.inptr, count);
        buf += count;
        G.inptr += count;
        G.incnt -= count;
        size -= count;
    }
    Trace((stderr, "returning %u from readbuf",n));
    return n;
} /* end function readbuf() */

/***********************/
/* Function readbyte() */
/***********************/
int readbyte(__G)   /* refill inbuf and return a byte if available, else EOF */
    __GDEF
{
    if (G.mem_mode)
        return EOF;
    if (G.csize <= 0) {
        G.csize--;             /* for tests done after exploding */
        G.incnt = 0;
        return EOF;
    }
    if (G.incnt <= 0) {
        if ((G.incnt = read(G.zipfd, (char *)G.inbuf, INBUFSIZ)) == 0) {
            G.incnt = 0;       /* do not allow negative value to affect stuff */
            return EOF;
        } else if (G.incnt < 0) {  /* "fail" (abort, retry, ...) returns this */
            fprintf(stderr,"Error reading from ZIP file");
            DESTROYGLOBALS()
            exit(PK_BADERR);  /* totally bailing; better than lock-up */
        }
        G.cur_zipfile_bufstart += INBUFSIZ; /* always starts on block bndry */
        G.inptr = G.inbuf;
        defer_leftover_input(__G);           /* decrements G.csize */
    }

#ifdef CRYPT
    if (G.pInfo->encrypted) {
        uch *p;
        int n;

        /* This was previously set to decrypt one byte beyond G.csize, when
         * incnt reached that far.  GRR said, "but it's required:  why?"  This
         * was a bug in fillinbuf() -- was it also a bug here?
         */
        for (n = G.incnt, p = G.inptr;  n--;  p++)
            zdecode(*p);
    }
#endif /* CRYPT */

    --G.incnt;
    return *G.inptr++;
} /* end function readbyte() */

/********************/
/* Function flush() */
/********************/
int flush(__G__ rawbuf, size, unshrink)   /* cflag => always 0; 50 if write error */
    __GDEF
    uch *rawbuf;
    ulg size;
    int unshrink;
{
    register uch *p, *q;
    ulg crcwas;
    uch *transbuf;
    /* static int didCRlast = FALSE;    moved to globals.h */

    user_callback(2, 0, 0, NULL); // bump up progress bar 

/*---------------------------------------------------------------------------
    Compute the CRC first; if testing or if disk is full, that's it.
  ---------------------------------------------------------------------------*/
    crcwas=G.crc32val;
    G.crc32val = crc32(G.crc32val, rawbuf, (extent)size);
    // Trace((stderr, "fileio.c: G.crc32val updated from: %08X to: %08X", crcwas, G.crc32val));

    if (G.tflag || size == 0L)   /* testing or nothing to write:  all done */
        return 0;
    if (G.disk_full)
        return 50;            /* disk already full:  ignore rest of file */
/*---------------------------------------------------------------------------
    Write the bytes rawbuf[0..size-1] to the output device, first converting
    end-of-lines and ASCII/EBCDIC as needed.  If SMALL_MEM or MED_MEM are NOT
    defined, outbuf is assumed to be at least as large as rawbuf and is not
    necessarily checked for overflow.
  ---------------------------------------------------------------------------*/
    if (!G.pInfo->textmode) {   /* write raw binary data */
        /* GRR:  note that for standard MS-DOS compilers, size argument to
         * fwrite() can never be more than 65534, so WriteError macro will
         * have to be rewritten if size can ever be that large.  For now,
         * never more than 32K.  Also note that write() returns an int, which
         * doesn't necessarily limit size to 32767 bytes if write() is used
         * on 16-bit systems but does make it more of a pain; however, because
         * at least MSC 5.1 has a lousy implementation of fwrite() (as does
         * DEC Ultrix cc), write() is used anyway.
         */
        if (!G.cflag && WriteError(rawbuf, size, G.outfile))
            return disk_error(__G);
      // else if (G.cflag && (*G.message)((zvoid *)&G, rawbuf, size, 0))
        return 0;
    } else {
        if (unshrink) {
            /* rawbuf = outbuf */
            transbuf = G.outbuf2;
        } else {
            /* rawbuf = slide */
            transbuf = G.outbuf;
        }
        if (G.newfile) {
            G.didCRlast = FALSE;   /* no previous buffers written */
            G.newfile = FALSE;
        }
        p = rawbuf;
        if (*p == LF && G.didCRlast)
            ++p;

    /*-----------------------------------------------------------------------
        Algorithm:  CR/LF => native; lone CR => native; lone LF => native.
        This routine is only for non-raw-VMS, non-raw-VM/CMS files (i.e.,
        stream-oriented files, not record-oriented).
      -----------------------------------------------------------------------*/
        G.didCRlast = FALSE;
        for (q = transbuf;  p < rawbuf+(unsigned)size;  ++p) {
            if (*p == CR) {              /* lone CR or CR/LF: EOL either way */
                PutNativeEOL
                if (p == rawbuf+(unsigned)size-1)  /* last char in buffer */
                    G.didCRlast = TRUE;
                else if (p[1] == LF)     /* get rid of accompanying LF */
                    ++p;
            } else if (*p == LF)         /* lone LF */
                PutNativeEOL
            else
                *q++ = native(*p);
        }
    /*-----------------------------------------------------------------------
        Done translating:  write whatever we've got to file (or screen).
      -----------------------------------------------------------------------*/
        Trace((stderr, "p - rawbuf = %u   q-transbuf = %u   size = %lu\n",
          (unsigned)(p-rawbuf), (unsigned)(q-transbuf), size));
        if (q > transbuf)
            if (!G.cflag && WriteError(transbuf, (unsigned)(q-transbuf),G.outfile))
                return disk_error(__G);
         //  else if (G.cflag && (*G.message)((zvoid *)&G, transbuf,
         //      (ulg)(q-transbuf), 0))
            return 0;
    }
    return 0;
} /* end function flush() */

/*************************/
/* Function disk_error() */
/*************************/
static int disk_error(__G)
    __GDEF
{
    unzerr(ZE_WRITE);
    return PK_DISK;   /* disk is full */
} 

/**********************/
/* Function handler() */
/**********************/
void handler(signal)   /* upon interrupt, turn on echo and exit cleanly */
    int signal;
{
    GETGLOBALS();
#if !(defined(SIGBUS) || defined(SIGSEGV))      /* add a newline if not at */
  //  (*G.message)((zvoid *)&G, slide, 0L, 0x41); /*  start of line (to stderr; */
#endif                                          /*  slide[] should be safe) */
  //  echon();

#ifdef SIGBUS
    if (signal == SIGBUS) {
        unzerr(ZE_FORM); // Zipfile Corrupt
        DESTROYGLOBALS()
        EXIT(PK_BADERR);
    }
#endif /* SIGBUS */

#ifdef SIGSEGV
    if (signal == SIGSEGV) {
        Info(slide, 0x421, ((char *)slide, LoadFarString(ZipfileCorrupt),
          "segmentation violation"));
        DESTROYGLOBALS()
        EXIT(PK_BADERR);
    }
#endif /* SIGSEGV */
    /* probably ctrl-C */
    DESTROYGLOBALS()
    EXIT(PK_ERR);         /* was EXIT(0) */
}

#ifndef timezone
#  define timezone _timezone
#endif

/*******************************/
/* Function dos_to_unix_time() */   /* only used for freshening/updating */
/*******************************/
time_t dos_to_unix_time(ddate, dtime)
    unsigned ddate, dtime;
{
    time_t m_time;
    int yr, mo, dy, hh, mm, ss;
#define YRBASE  1970
    static short yday[]={0, 31, 59, 90, 120, 151, 181, 212, 243, 273, 304, 334};
    int leap;
    unsigned days;

    /* dissect date */
    yr = ((ddate >> 9) & 0x7f) + (1980 - YRBASE);
    mo = ((ddate >> 5) & 0x0f) - 1;
    dy = (ddate & 0x1f) - 1;

    /* dissect time */
    hh = (dtime >> 11) & 0x1f;
    mm = (dtime >> 5) & 0x3f;
    ss = (dtime & 0x1f) * 2;

    /* leap = # of leap yrs from YRBASE up to but not including current year */
    leap = ((yr + YRBASE - 1) / 4);   /* leap year base factor */

    /* calculate days from BASE to this year and add expired days this year */
    days = (yr * 365) + (leap - 492) + yday[mo];

    /* if year is a leap year and month is after February, add another day */
    if ((mo > 1) && ((yr+YRBASE)%4 == 0) && ((yr+YRBASE) != 2100))
        ++days;                 /* OK through 2199 */

    /* convert date & time to seconds relative to 00:00:00, 01/01/YRBASE */
    m_time = (time_t)((long)(days + dy) * 86400L + (long)hh * 3600L +
                      (long)(mm * 60 + ss));
      /* - 1;   MS-DOS times always rounded up to nearest even second */
    TTrace((stderr, "dos_to_unix_time:\n"));
    TTrace((stderr, "  m_time before timezone = %ld\n", (long)m_time));

    /* adjust for local standard timezone offset */
    {
        TIME_ZONE_INFORMATION tzinfo;
        DWORD res;

        /* account for timezone differences */
        res = GetTimeZoneInformation(&tzinfo);
        if (res == TIME_ZONE_ID_STANDARD)
            m_time += 60*(tzinfo.Bias + tzinfo.StandardBias);
        else if (res == TIME_ZONE_ID_DAYLIGHT)
            m_time += 60*(tzinfo.Bias + tzinfo.DaylightBias);
        /* GRR:  are other return-values possible? */
    }
    TTrace((stderr, "  m_time after timezone =  %ld\n", (long)m_time));

    return m_time;
} /* end function dos_to_unix_time() */

/******************************/
/* Function check_for_newer() */  /* used for overwriting/freshening/updating */
/******************************/
int check_for_newer(__G__ filename)  /* return 1 if existing file is newer */
    __GDEF                           /*  or equal; 0 if older; -1 if doesn't */
    char *filename;                  /*  exist yet */
{
    time_t existing, archive;
#ifdef USE_EF_UX_TIME
    ztimbuf z_utime;
#endif

    Trace((stderr, "check_for_newer:  doing stat(%s)\n", filename));
    if (stat(filename, &G.statbuf)) {
        Trace((stderr,
          "check_for_newer:  stat(%s) returns %d:  file does not exist\n",
          filename, stat(filename, &G.statbuf)));
        return DOES_NOT_EXIST;
    }
    Trace((stderr, "check_for_newer:  stat(%s) returns 0:  file exists\n",
      filename));

#ifdef USE_EF_UX_TIME
    /* The `Unix extra field mtime' should be used for comparison with the
     * time stamp of the existing file >>>ONLY<<< when the EF info is also
     * used to set the modification time of the extracted file.
     * Currently, this is only the case for systems that incorporate
     * the OS specific code from the "unix/" subdirectory.
     */
    if (G.extra_field &&
        ef_scan_for_izux(G.extra_field, G.lrec.extra_field_length,
                         &z_utime, NULL) > 0) {
        TTrace((stderr, "check_for_newer:  using Unix extra field mtime\n"));
        existing = G.statbuf.st_mtime;
        archive  = z_utime.modtime;
    } else {
        /* round up existing filetime to nearest 2 seconds for comparison */
        existing = G.statbuf.st_mtime & 1 ? G.statbuf.st_mtime + 1
                                          : G.statbuf.st_mtime;
        archive  = dos_to_unix_time(G.lrec.last_mod_file_date,
                                    G.lrec.last_mod_file_time);
    }
#else /* !USE_EF_UX_TIME */
    /* round up existing filetime to nearest 2 seconds for comparison */
    existing = G.statbuf.st_mtime & 1 ? G.statbuf.st_mtime + 1
                                      : G.statbuf.st_mtime;
    archive  = dos_to_unix_time(G.lrec.last_mod_file_date,
                                G.lrec.last_mod_file_time);
#endif /* ?USE_EF_UX_TIME */
    TTrace((stderr, "check_for_newer:  existing %ld, archive %ld, e-a %ld\n",
      existing, archive, existing-archive));
    return (existing >= archive);
} /* end function check_for_newer() */

/************************/
/* Function do_string() */
/************************/
int do_string(__G__ len, option)      /* return PK-type error code */
    __GDEF
    unsigned int len;           /* without prototype, ush converted to this */
    int option;
{
    long comment_bytes_left, block_length;
    int error=PK_OK;
    ush extra_len;

/*---------------------------------------------------------------------------
    This function processes arbitrary-length (well, usually) strings.  Four
    options are allowed:  SKIP, wherein the string is skipped (pretty logical,
    eh?); DISPLAY, wherein the string is printed to standard output after un-
    dergoing any necessary or unnecessary character conversions; DS_FN,
    wherein the string is put into the filename[] array after undergoing ap-
    propriate conversions (including case-conversion, if that is indicated:
    see the global variable pInfo->lcflag); and EXTRA_FIELD, wherein the
    `string' is assumed to be an extra field and is copied to the (freshly
    malloced) buffer G.extra_field.  The third option should be OK since
    filename is dimensioned at 1025, but we check anyway.

    The string, by the way, is assumed to start at the current file-pointer
    position; its length is given by len.  So start off by checking length
    of string:  if zero, we're already done.
  ---------------------------------------------------------------------------*/
    if (!len)
        return PK_COOL;
 
   
    switch (option) {
    /*
     * First case:  print string on standard output.  First set loop vari-
     * ables, then loop through the comment in chunks of OUTBUFSIZ bytes,
     * converting formats and printing as we go.  The second half of the
     * loop conditional was added because the file might be truncated, in
     * which case comment_bytes_left will remain at some non-zero value for
     * all time.  outbuf and slide are used as scratch buffers because they
     * are available (we should be either before or in between any file pro-
     * cessing).
     */
    case DISPLAY:
        comment_bytes_left = len;
        block_length = OUTBUFSIZ;    /* for the while statement, first time */
        while (comment_bytes_left > 0 && block_length > 0) {
            register uch *p = G.outbuf;
            register uch *q = G.outbuf;

            if ((block_length = readbuf(__G__ (char *)G.outbuf,
                   (unsigned) MIN((long)OUTBUFSIZ, comment_bytes_left))) == 0)
                return PK_EOF;
            comment_bytes_left -= block_length;

            /* this is why we allocated an extra byte for outbuf:  terminate
             *  with zero (ASCIIZ) */
            G.outbuf[(unsigned)block_length] = '\0';

            /* remove all ASCII carriage returns from comment before printing
             * (since used before A_TO_N(), check for CR instead of '\r')
             */
            while (*p) {
                while (*p == CR)
                    ++p;
                *q++ = *p++;
            }
            /* could check whether (p - outbuf) == block_length here */
            *q = '\0';

            A_TO_N(G.outbuf);   /* translate string to native */

#ifdef NATIVE          /* GRR:  can ANSI be used with EBCDIC? */
           // (*G.message)((zvoid *)&G, G.outbuf, (ulg)(q-G.outbuf), 0);
#else /* ASCII */
            p = G.outbuf - 1;
            q = slide;
            while (*++p) {
                int pause = FALSE;

                if (*p == 0x1B) {          /* ASCII escape char */
                    *q++ = '^';
                    *q++ = '[';
                } else if (*p == 0x13) {   /* ASCII ^S (pause) */
                    pause = TRUE;
                    if (p[1] == 0x0a)      /* ASCII LF */
                        *q++ = *++p;
                    else if (p[1] == 0x0d && p[2] == 0x0a) {  /* ASCII CR LF */
                        *q++ = *++p;
                        *q++ = *++p;
                    }
                } else
                    *q++ = *p;
                if ((unsigned)(q-slide) > WSIZE-3 || pause) {   /* flush */
                    // (*G.message)((zvoid *)&G, slide, (ulg)(q-slide), 0);
                    q = slide;
                }
            }
          // (*G.message)((zvoid *)&G, slide, (ulg)(q-slide), 0);
#endif /* ?NATIVE */
        }
        /* add '\n' if not at start of line */
        // (*G.message)((zvoid *)&G, slide, 0L, 0x40);
        break;
    /*
     * Second case:  read string into filename[] array.  The filename should
     * never ever be longer than FILNAMSIZ-1 (1024), but for now we'll check,
     * just to be sure.
     */
    case DS_FN:
        extra_len = 0;
        if (len >= FILNAMSIZ) {
            fprintf(stderr, "Warning - filename too long; truncated");
            error = PK_WARN;
            extra_len = (ush)(len - FILNAMSIZ + 1);
            len = FILNAMSIZ - 1;
        }
        if (readbuf(__G__  G.filename, len) == 0)
            return PK_EOF;
        G.filename[len] = '\0';   /* terminate w/zero:  ASCIIZ */

        A_TO_N(G.filename);       /* translate string to native */

        if (G.pInfo->lcflag)      /* replace with lowercase filename */
            TOLOWER(G.filename, G.filename);

        if (G.pInfo->vollabel && len > 8 && G.filename[8] == '.') {
            char *p = G.filename+8;
            while (*p++)
                p[-1] = *p;  /* disk label, and 8th char is dot:  remove dot */
        }
        if (!extra_len)         /* we're done here */
            break;
        /*
         * We truncated the filename, so fall
         * through to the SKIP routine.
         */
        len = extra_len;
        /*  FALL THROUGH...  */
    /*
     * Third case:  skip string, adjusting readbuf's internal variables
     * as necessary (and possibly skipping to and reading a new block of
     * data).
     */
    case SKIP:
        /* cur_zipfile_bufstart already takes account of extra_bytes, so don't
         * correct for it twice: */
        ZLSEEK(G.cur_zipfile_bufstart - G.extra_bytes +
               (G.inptr-G.inbuf) + len)
        break;
    /*
     * Fourth case:  assume we're at the start of an "extra field"; malloc
     * storage for it and read data into the allocated space.
     */
    case EXTRA_FIELD:
        if (G.extra_field != (uch *)NULL)
            free(G.extra_field);
        if ((G.extra_field = (uch *)malloc(len)) == (uch *)NULL) {
            fprintf(stderr,"Warning: extra field too long");
            /* cur_zipfile_bufstart already takes account of extra_bytes,
             * so don't correct for it twice: */
            ZLSEEK(G.cur_zipfile_bufstart - G.extra_bytes +
                   (G.inptr-G.inbuf) + len)
        } else
            if (readbuf(__G__  (char *)G.extra_field, len) == 0)
                return PK_EOF;
        break;
    } /* end switch (option) */
    return error;
} /* end function do_string() */

/***********************/
/* Function makeword() */
/***********************/
ush makeword(b)
    uch *b;
{
    /* Convert Intel style 'short' integer to non-Intel non-16-bit
     * host format.  This routine also takes care of byte-ordering.
     */
    return (ush)((b[1] << 8) | b[0]);
}

/***********************/
/* Function makelong() */
/***********************/
ulg makelong(sig)
    uch *sig;
{
    /*
     * Convert intel style 'long' variable to non-Intel non-16-bit
     * host format.  This routine also takes care of byte-ordering.
     */
    return (((ulg)sig[3]) << 24)
        + (((ulg)sig[2]) << 16)
        + (((ulg)sig[1]) << 8)
        + ((ulg)sig[0]);
}

#ifdef ZMEM  /* memset/memcpy for systems without either them or bzero/bcopy */
             /* (no known systems as of 960211) */
/*********************/
/* Function memset() */
/*********************/
zvoid *memset(buf, init, len)
    register zvoid *buf;        /* buffer location */
    register int init;          /* initializer character */
    register unsigned int len;  /* length of the buffer */
{
    zvoid *start;

    start = buf;
    while (len--)
        *((char *)buf++) = (char)init;
    return start;
}

/*********************/
/* Function memcpy() */
/*********************/
zvoid *memcpy(dst, src, len)
    register zvoid *dst;
    register const zvoid *src;
    register unsigned int len;
{
    zvoid *start;

    start = dst;
    while (len-- > 0)
        *((char *)dst)++ = *((char *)src)++;
    return start;
}
#endif /* ZMEM */

/************************/
/* Function zstrnicmp() */
/************************/
int zstrnicmp(s1, s2, n)
    register const char *s1, *s2;
    register unsigned n;
{
    for (; n > 0;  --n, ++s1, ++s2) {

        if (ToLower(*s1) != ToLower(*s2))
            /* test includes early termination of one string */
            return (ToLower(*s1) < ToLower(*s2))? -1 : 1;

        if (*s1 == '\0')   /* both strings terminate early */
            return 0;
    }
    return 0;
}

