/* Modified Version for Delphi project, Eric W. Engler, Mar 5, 1997 */
/*---------------------------------------------------------------------------
  process.c
  This file contains the top-level routines for processing multiple zipfiles.
  Contains:  process_zipfiles()
             do_seekable()
             find_ecrec()
             uz_end_central()
             process_cdir_file_hdr()
             get_cdir_ent()
             process_local_file_hdr()
             ef_scan_for_izux()
  ---------------------------------------------------------------------------*/
#define UNZIP_INTERNAL
#include "unzip.h"
#ifdef WIN32
#  include <windows.h>
#endif

static int    do_seekable        OF((__GPRO__ int lastchance));
static int    find_ecrec         OF((__GPRO__ long searchlen));

/*******************************/
/* Function process_zipfiles() */
/*******************************/
int process_zipfiles(__G)    /* return PK-type error code */
     __GDEF
{
    char *lastzipfn = (char *)NULL;
    int NumWinFiles, NumLoseFiles, NumWarnFiles;
    int NumMissDirs, NumMissFiles;
    int error=0, error_in_archive=0;

    diag("one in process.c");
/*---------------------------------------------------------------------------
    Start by allocating buffers and (re)constructing the various PK signature
    strings.
  ---------------------------------------------------------------------------*/
    G.inbuf = (uch *)malloc(INBUFSIZ + 4);    /* 4 extra for hold[] (below) */

    /* Normally realbuf and outbuf will be the same.  However, if the data
     * are redirected to a large memory buffer, realbuf will point to the
     * new location while outbuf will remain pointing to the malloc'd
     * memory buffer.  (The extra "1" is for string termination.) */
    G.realbuf = G.outbuf = (uch *)malloc(OUTBUFSIZ + 1);

    if ((G.inbuf == (uch *)NULL) || (G.outbuf == (uch *)NULL)) {
        unzerr(ZE_MEM);
        RETURN(PK_MEM);
    }
    G.hold = G.inbuf + INBUFSIZ;     /* to check for boundary-spanning sigs */

    diag("two");
    G.local_hdr_sig[0]  /* = extd_local_sig[0] */  = 0x50;   /* ASCII 'P', */
    G.central_hdr_sig[0] = G.end_central_sig[0] = 0x50;     /* not EBCDIC */

    G.local_hdr_sig[1]  /* = extd_local_sig[1] */  = 0x4B;   /* ASCII 'K', */
    G.central_hdr_sig[1] = G.end_central_sig[1] = 0x4B;     /* not EBCDIC */

    /* GRR FIX:  no need for these to be in global struct; OK to "overwrite" */
    strcpy(G.local_hdr_sig+2, LOCAL_HDR_SIG);
    strcpy(G.central_hdr_sig+2, CENTRAL_HDR_SIG);
    strcpy(G.end_central_sig+2, END_CENTRAL_SIG);
/*---------------------------------------------------------------------------
    Match (possible) wildcard zipfile specification with existing files and
    attempt to process each.  If no hits, try again after appending ".zip"
    suffix.  If still no luck, give up.
  ---------------------------------------------------------------------------*/
    NumWinFiles = NumLoseFiles = NumWarnFiles = 0;
    NumMissDirs = NumMissFiles = 0;
    diag("four in process.c - ready to dowild");

    // dowild() gives us the real name of zipfile, in case wildcard was used.
    while ((G.zipfn = do_wild(__G__ G.wildzipfn)) != (char *)NULL) {
        Trace((stderr, "back from do_wild( %s ) returned %s\n", G.wildzipfn, G.zipfn));
        lastzipfn = G.zipfn;

        if ((error = do_seekable(__G__ 0)) == PK_WARN)
            ++NumWarnFiles;
        else if (error == IZ_DIR)
            ++NumMissDirs;
        else if (error == PK_NOZIP)
            ++NumMissFiles;
        else if (error)
            ++NumLoseFiles;
        else
            ++NumWinFiles;

        Trace((stderr, "after do_seekable, warn=%d   missdir=%d  missfi=%d  losefi=%d  winfi=%d",
            NumWarnFiles,NumMissDirs,NumMissFiles,NumLoseFiles,NumWinFiles));

        if (error != IZ_DIR && error > error_in_archive)
            error_in_archive = error;
        Trace((stderr, "do_seekable(0) returns %d (see unzpriv.h, PK_*)\n", error));
    } /* end while-loop (wildcard zipfiles) */

    diag("five in process.c");
    if ((NumWinFiles + NumWarnFiles + NumLoseFiles) == 0  &&
        (NumMissDirs + NumMissFiles) == 1  &&  lastzipfn != (char *)NULL)
    {
        NumMissDirs = NumMissFiles = 0;
        if (error_in_archive == PK_NOZIP)
            error_in_archive = PK_COOL;

        if (iswild(G.wildzipfn))
            fprintf(stderr,"Error: No wildcard match on zipfile: %s", G.wildzipfn);
        else
        {
            char *p = lastzipfn + strlen(lastzipfn);
            G.zipfn = lastzipfn;
            strcpy(p, ZSUFX);

            error = do_seekable(__G__ 1);
            if (error == PK_WARN)   /* GRR: make this a switch/case stmt ... */
                ++NumWarnFiles;
            else if (error == IZ_DIR)
                ++NumMissDirs;
            else if (error == PK_NOZIP)
                /* increment again => bug: "1 file had no zipfile directory." */
                /* ++NumMissFiles */ ;
            else if (error)
                ++NumLoseFiles;
            else
                ++NumWinFiles;
            if (error > error_in_archive)
                error_in_archive = error;
            Trace((stderr, "do_seekable(1) returns %d\n", error));
        }
    }

    diag("six");
/*---------------------------------------------------------------------------
    Print summary of all zipfiles, assuming zipfile spec was a wildcard (no
    need for a summary if just one zipfile).
  ---------------------------------------------------------------------------*/
        if ((NumWinFiles > 1) || (NumWinFiles == 1 &&
          NumMissDirs + NumMissFiles + NumLoseFiles + NumWarnFiles > 0))
            fprintf(stderr,"files processed OK");
        if (NumWarnFiles > 0)
            fprintf(stderr,"warnings were given");
        if (NumLoseFiles > 0)
            fprintf(stderr,"file(s) not found");
        if (NumMissFiles > 0)
            fprintf(stderr, "file(s) not found");
        if (NumMissDirs == 1)
            fprintf(stderr, "zip file was dir");
        else if (NumMissDirs > 0)
            fprintf(stderr, "many zip file were dirs");
        if (NumWinFiles + NumLoseFiles + NumWarnFiles == 0)
            fprintf(stderr, "no files found");

    diag("seven");
    /* free allocated memory */
    inflate_free(__G);
    checkdir(__G__ (char *)NULL, END);
#ifdef DYNALLOC_CRCTAB
    if (G.extract_flag && CRC_32_TAB)
        nearfree((zvoid near *)CRC_32_TAB);
#endif /* DYNALLOC_CRCTAB */
    if (G.outbuf2)
        free(G.outbuf2);   /* malloc'd ONLY if unshrink and -a */
    free(G.outbuf);
    free(G.inbuf);
    return error_in_archive;
} /* end function process_zipfiles() */

/**************************/
/* Function do_seekable() */
/**************************/
static int do_seekable(__G__ lastchance)        /* return PK-type error code */
     __GDEF
    int lastchance;
{
    /* static int no_ecrec = FALSE;  SKM: moved to globals.h */
    int maybe_exe=FALSE;
    int too_weird_to_continue=FALSE;
    int error=0, error_in_archive;

    diag("starting do_seekable of process.c");
/*---------------------------------------------------------------------------
    Open the zipfile for reading in BINARY mode to prevent CR/LF translation,
    which would corrupt the bit streams.
  ---------------------------------------------------------------------------*/
    if (SSTAT(G.zipfn, &G.statbuf) ||
        (error = S_ISDIR(G.statbuf.st_mode)) != 0)
    {
        if (lastchance)
            if (G.no_ecrec)
                fprintf(stderr, "can't find zip file dir");
            else
                fprintf(stderr, "can't find either zip file");
        return error? IZ_DIR : PK_NOZIP;
    }
    G.ziplen = G.statbuf.st_size;

    if (G.statbuf.st_mode & S_IEXEC)   
        maybe_exe = TRUE;      /* might find unzip, not unzip.zip; etc. */

    diag("do_seekable, loc 2");
    if (open_input_file(__G))   /* this should never happen, given */
        return PK_NOZIP;        /*  the stat() test above, but... */
    diag("do_seekable, loc 3");
/*---------------------------------------------------------------------------
    Find and process the end-of-central-directory header.  UnZip need only
    check last 65557 bytes of zipfile:  comment may be up to 65535, end-of-
    central-directory record is 18 bytes, and signature itself is 4 bytes;
    add some to allow for appended garbage.  
  ---------------------------------------------------------------------------*/
    /* initialize the CRC table pointer (once) */
    // EWE: This "get" of CRC table will be done again by crc32.c.
    // I'm used a statically allocated table, so it doesn't matter.
    if (CRC_32_TAB == (ulg near *)NULL) {
        if ((CRC_32_TAB = (ulg near *)get_crc_table()) == (ulg near *)NULL)
            return PK_MEM2;
    }
    G.cur_zipfile_bufstart = 0;
    G.inptr = G.inbuf;

#ifdef TIMESTAMP
    if (!G.qflag && !G.T_flag && !G.zipinfo_mode)
#else
    if (!G.qflag && !G.zipinfo_mode)
#endif
        fprintf(stderr, "Archive:  %s", G.zipfn);

    diag("do_seekable, loc 4");
    if ((
          ((error_in_archive = find_ecrec(__G__ MIN(G.ziplen,66000L))) != 0 ||
          (error_in_archive = uz_end_central(__G)) > PK_WARN)))
    {
        close(G.zipfd);
        if (maybe_exe)
            fprintf(stderr,"maybe an EXE file: %s", G.zipfn);
        if (lastchance)
            return error_in_archive;
        else {
            G.no_ecrec = TRUE;    /* assume we found wrong file:  e.g., */
            return PK_NOZIP;      /*  unzip instead of unzip.zip */
        }
    }
    diag("do_seekable, loc 5");
    if ((G.zflag > 0) && !G.zipinfo_mode) {  /* unzip: zflag = comment ONLY */
        close(G.zipfd);
        return error_in_archive;
    }

/*---------------------------------------------------------------------------
    Test the end-of-central-directory info for incompatibilities (multi-disk
    archives) or inconsistencies (missing or extra bytes in zipfile).
  ---------------------------------------------------------------------------*/
#ifdef NO_MULTIPART
    error = !G.zipinfo_mode && (G.ecrec.number_this_disk == 1) &&
            (G.ecrec.num_disk_start_cdir == 1);
#else
    error = !G.zipinfo_mode && (G.ecrec.number_this_disk != 0);
#endif

    if (G.zipinfo_mode &&
        G.ecrec.number_this_disk != G.ecrec.num_disk_start_cdir)
    {
        if (G.ecrec.number_this_disk > G.ecrec.num_disk_start_cdir) {
            fprintf(stderr,"Central dir not in ZIP file");
            error_in_archive = PK_FIND;
            too_weird_to_continue = TRUE;
        } else {
            fprintf(stderr,"Central dir bogus");
            error_in_archive = PK_WARN;
        }
#ifdef NO_MULTIPART   /* concatenation of multiple parts works in some cases */
    } else if (!G.zipinfo_mode && !error && G.ecrec.number_this_disk != 0) {
        fprintf(stderr,"no multi-disk archives supported");
        error_in_archive = PK_FIND;
        too_weird_to_continue = TRUE;
#endif
    }
    
    diag("do_seekable, loc 6");
    if (!too_weird_to_continue) {  /* (relatively) normal zipfile:  go for it */
        if (error) {
            fprintf(stderr,"maybe a pak bug in %s", G.zipfn);
            error_in_archive = PK_WARN;
        }
        if ((G.extra_bytes = G.real_ecrec_offset-G.expect_ecrec_offset) <
            (LONGINT)0)
        {
            fprintf(stderr,"missing bytes in zipfile");
            error_in_archive = PK_ERR;
        } else if (G.extra_bytes > 0) {
            if ((G.ecrec.offset_start_central_directory == 0) &&
                (G.ecrec.size_central_directory != 0))   /* zip 1.5 -go bug */
            {
                fprintf(stderr, "NULL central dir");
                G.ecrec.offset_start_central_directory = G.extra_bytes;
                G.extra_bytes = 0;
                error_in_archive = PK_ERR;
            }
            else {
                fprintf(stderr,"Warning: extra bytes at start of zipfile");
                error_in_archive = PK_WARN;
            }
        }

    /*-----------------------------------------------------------------------
        Check for empty zipfile and exit now if so.
      -----------------------------------------------------------------------*/
        diag("do_seekable, loc 7");
        if (G.expect_ecrec_offset==0L && G.ecrec.size_central_directory==0) {
            fprintf(stderr,"Empty zipfile");
            close(G.zipfd);
            return (error_in_archive > PK_WARN)? error_in_archive : PK_WARN;
        }

    /*-----------------------------------------------------------------------
        Compensate for missing or extra bytes, and seek to where the start
        of central directory should be.  If header not found, uncompensate
        and try again (necessary for at least some Atari archives created
        with STZip, as well as archives created by J.H. Holm's ZIPSPLIT 1.1).
      -----------------------------------------------------------------------*/
        ZLSEEK( G.ecrec.offset_start_central_directory )
#ifdef OLD_SEEK_TEST
        if (readbuf(G.sig, 4) == 0) {
            close(G.zipfd);
            return PK_ERR;  /* file may be locked, or possibly disk error(?) */
        }
        if (strncmp(G.sig, G.central_hdr_sig, 4))
#else
        if ((readbuf(__G__ G.sig, 4) == 0) || strncmp(G.sig, G.central_hdr_sig, 4))
#endif
        {
            long tmp = G.extra_bytes;

            diag("central dir found");
            G.extra_bytes = 0;
            ZLSEEK( G.ecrec.offset_start_central_directory )
            if ((readbuf(__G__ G.sig, 4) == 0) ||
                strncmp(G.sig, G.central_hdr_sig, 4))
            {
                fprintf(stderr,"can't find start of central dir");
                close(G.zipfd);
                return PK_BADERR;
            }
            fprintf(stderr,"central dir too long");
            error_in_archive = PK_ERR;
        }
    /*-----------------------------------------------------------------------
        Seek to the start of the central directory one last time, since we
        have just read the first entry's signature bytes; then list, extract
        or test member files as instructed, and close the zipfile.
      -----------------------------------------------------------------------*/
        Trace((stderr, "about to extract/list files (error = %d)\n",
           error_in_archive));
        ZLSEEK( G.ecrec.offset_start_central_directory )

#ifdef TIMESTAMP
// EWE  do some work here???
// How can you timestamp a file before you extract it?
//          if (G.T_flag)
//              error = time_stamp(__G);              /* TIME-STAMP 'EM */
//          else
#endif
//          if (G.vflag && !G.tflag && !G.cflag)
//              error = list_files(__G);              /* LIST 'EM */
//          else
//              GO DO EXTRACT OR TEST
                error = extract_or_test_files(__G);   /* EXTRACT OR TEST 'EM */

        Trace((stderr, "Done with extract/list files (error = %d)\n", error));
        if (error > error_in_archive)   /* don't overwrite stronger error */
            error_in_archive = error;   /*  with (for example) a warning */
    } /* end if (!too_weird_to_continue) */
    close(G.zipfd);
    return error_in_archive;
} /* end function do_seekable() */

/*************************/
/* Function find_ecrec() */
/*************************/
static int find_ecrec(__G__ searchlen)          /* return PK-class error */
    long searchlen;
     __GDEF
{
    int i, numblks, found=FALSE;
    LONGINT tail_len;
    ec_byte_rec byterec;

    diag("in find_ecrec (end of centrral dir) in process.c");
/*---------------------------------------------------------------------------
    Treat case of short zipfile separately.
 ---------------------------------------------------------------------------*/
    if (G.ziplen <= INBUFSIZ) {
        lseek(G.zipfd, 0L, SEEK_SET);
        if ((G.incnt = read(G.zipfd,(char *)G.inbuf,(unsigned int)G.ziplen))
            == (int)G.ziplen)

            /* 'P' must be at least 22 bytes from end of zipfile */
            for (G.inptr = G.inbuf+(int)G.ziplen-22;  G.inptr >= G.inbuf;
                 --G.inptr)
                if ((native(*G.inptr) == 'P')  &&
                     !strncmp((char *)G.inptr, G.end_central_sig, 4)) {
                    G.incnt -= (int)(G.inptr - G.inbuf);
                    found = TRUE;
                    break;
                }
/*---------------------------------------------------------------------------
    Zipfile is longer than INBUFSIZ:  may need to loop.  Start with short
    block at end of zipfile (if not TOO short).
  ---------------------------------------------------------------------------*/
    } else {
        if ((tail_len = G.ziplen % INBUFSIZ) > ECREC_SIZE) {
#ifdef USE_STRM_INPUT
            fseek((FILE *)G.zipfd, G.ziplen-tail_len, SEEK_SET);
            G.cur_zipfile_bufstart = ftell((FILE *)G.zipfd);
#else /* !USE_STRM_INPUT */
            G.cur_zipfile_bufstart = lseek(G.zipfd, G.ziplen-tail_len,
              SEEK_SET);
#endif /* ?USE_STRM_INPUT */
            if ((G.incnt = read(G.zipfd, (char *)G.inbuf,
                (unsigned int)tail_len)) != (int)tail_len)
                goto fail;      /* it's expedient... */

            /* 'P' must be at least 22 bytes from end of zipfile */
            for (G.inptr = G.inbuf+(int)tail_len-22;  G.inptr >= G.inbuf;
                 --G.inptr)
                if ((native(*G.inptr) == 'P')  &&
                     !strncmp((char *)G.inptr, G.end_central_sig, 4)) {
                    G.incnt -= (int)(G.inptr - G.inbuf);
                    found = TRUE;
                    break;
                }
            /* sig may span block boundary: */
            strncpy((char *)G.hold, (char *)G.inbuf, 3);
        } else
            G.cur_zipfile_bufstart = G.ziplen - tail_len;

    /*-----------------------------------------------------------------------
        Loop through blocks of zipfile data, starting at the end and going
        toward the beginning.  In general, need not check whole zipfile for
        signature, but may want to do so if testing.
      -----------------------------------------------------------------------*/
        numblks = (int)((searchlen - tail_len + (INBUFSIZ-1)) / INBUFSIZ);
        /*               ==amount=   ==done==   ==rounding==    =blksiz=  */

        for (i = 1;  !found && (i <= numblks);  ++i) {
            G.cur_zipfile_bufstart -= INBUFSIZ;
            lseek(G.zipfd, G.cur_zipfile_bufstart, SEEK_SET);
            if ((G.incnt = read(G.zipfd,(char *)G.inbuf,INBUFSIZ)) != INBUFSIZ)
                break;          /* fall through and fail */

            for (G.inptr = G.inbuf+INBUFSIZ-1;  G.inptr >= G.inbuf; --G.inptr)
                if ((native(*G.inptr) == 'P')  &&
                     !strncmp((char *)G.inptr, G.end_central_sig, 4)) {
                    G.incnt -= (int)(G.inptr - G.inbuf);
                    found = TRUE;
                    break;
                }
            /* signature may span block boundary: */
            strncpy((char *)G.hold, (char *)G.inbuf, 3);
        }
    } /* end if (ziplen > INBUFSIZ) */

/*---------------------------------------------------------------------------
    Searched through whole region where signature should be without finding
    it.  Print informational message and die a horrible death.
  ---------------------------------------------------------------------------*/
fail:
    if (!found) {
        fprintf(stderr,"Fatal Error: central dir signature not found");
        return PK_ERR;   /* failed */
    }

/*---------------------------------------------------------------------------
    Found the signature, so get the end-central data before returning.  Do
    any necessary machine-type conversions (byte ordering, structure padding
    compensation) by reading data into character array and copying to struct.
  ---------------------------------------------------------------------------*/
    G.real_ecrec_offset = G.cur_zipfile_bufstart + (G.inptr-G.inbuf);

    Trace((stderr,"Found end-of-central-dir signature at offset %ld (%.8lXh)\n",
        G.real_ecrec_offset, G.real_ecrec_offset));
    Trace((stderr,"    from beginning of file; offset %d (%.4Xh) within block\n",
        G.inptr-G.inbuf, G.inptr-G.inbuf));

    if (readbuf(__G__ (char *)byterec, ECREC_SIZE+4) == 0)
        return PK_EOF;
    G.ecrec.number_this_disk =
      makeword(&byterec[NUMBER_THIS_DISK]);
    G.ecrec.num_disk_start_cdir =
      makeword(&byterec[NUM_DISK_WITH_START_CENTRAL_DIR]);
    G.ecrec.num_entries_centrl_dir_ths_disk =
      makeword(&byterec[NUM_ENTRIES_CENTRL_DIR_THS_DISK]);
    G.ecrec.total_entries_central_dir =
      makeword(&byterec[TOTAL_ENTRIES_CENTRAL_DIR]);
    G.ecrec.size_central_directory =
      makelong(&byterec[SIZE_CENTRAL_DIRECTORY]);
    G.ecrec.offset_start_central_directory =
      makelong(&byterec[OFFSET_START_CENTRAL_DIRECTORY]);
    G.ecrec.zipfile_comment_length =
      makeword(&byterec[ZIPFILE_COMMENT_LENGTH]);
    G.expect_ecrec_offset = G.ecrec.offset_start_central_directory +
                          G.ecrec.size_central_directory;
    return PK_COOL;
} /* end function find_ecrec() */

/*****************************/
/* Function uz_end_central() */
/*****************************/
int uz_end_central(__G)    /* return PK-type error code */
     __GDEF
{
    int error = PK_COOL;
/*---------------------------------------------------------------------------
    Get the zipfile comment (up to 64KB long), if any, and print it out.
    Then position the file pointer to the beginning of the central directory
    and fill buffer.
  ---------------------------------------------------------------------------*/
    if (G.ecrec.zipfile_comment_length && (G.zflag > 0 ||
        (G.zflag == 0 &&
#ifdef TIMESTAMP
                       !G.T_flag &&
#endif
                                  !G.qflag)))
    {
        if (do_string(__G__ G.ecrec.zipfile_comment_length,DISPLAY)) {
            fprintf(stderr,"zip file comment too long");
            error = PK_WARN;
        }
    }
    return error;
} /* end function uz_end_central() */

/************************************/
/* Function process_cdir_file_hdr() */   // cdir = central dir of zipfile
/************************************/
int process_cdir_file_hdr(__G)    /* return PK-type error code */
     __GDEF
{
    int error;

    diag("in process_cdir_file_hdr() in process.c");
/*---------------------------------------------------------------------------
    Get central directory info, save host and method numbers, and set flag
    for lowercase conversion of filename, depending on the OS from which the
    file is coming.
  ---------------------------------------------------------------------------*/
    if ((error = get_cdir_ent(__G)) != 0)
        {
        diag("Err returned by get_cdir_ent call in process.c");
        return error;
        }
    G.pInfo->hostnum = MIN(G.crec.version_made_by[1], NUM_HOSTS);
    G.pInfo->lcflag = 0;

    Trace((stderr,"Good entry; hostnum of file = %d", G.pInfo->hostnum));
    if (G.L_flag)             /* if user specified case-conversion */
        switch (G.pInfo->hostnum) {
            case FS_FAT_:     /* PKZIP and zip -k store in uppercase */
            case CPM_:        /* like MS-DOS, right? */
            case VM_CMS_:     /* all caps? */
            case MVS_:        /* all caps? */
            case TOPS20_:
            case VMS_:        /* our Zip uses lowercase, but asi's doesn't */
                G.pInfo->lcflag = 1;   /* convert filename to lowercase */
                break;

            default:     /* AMIGA_, FS_HPFS_, FS_NTFS_, MAC_, UNIX_, ATARI_, */
                break;   /*  FS_VFAT_, BEBOX_ (Z_SYSTEM_):  no conversion */
        }

    /* is there a volume label? */
    if (IS_VOLID(G.crec.external_file_attributes) &&
        (G.pInfo->hostnum == FS_FAT_ || G.pInfo->hostnum == FS_HPFS_ ||
         G.pInfo->hostnum == FS_NTFS_ || G.pInfo->hostnum == ATARI_))
    {
        G.pInfo->vollabel = TRUE;
        G.pInfo->lcflag = 0;        /* preserve case of volume labels */
    } else
        G.pInfo->vollabel = FALSE;
    diag("process_cdir_file_hdr returning PK_COOL");
    return PK_COOL;
} /* end function process_cdir_file_hdr() */

/***************************/
/* Function get_cdir_ent() */
/***************************/
int get_cdir_ent(__G)   /* return PK-type error code */
     __GDEF
{
    cdir_byte_hdr byterec;

    diag("in get_cdir_ent in process.c");
/*---------------------------------------------------------------------------
    Read the next central directory entry and do any necessary machine-type
    conversions (byte ordering, structure padding compensation--do so by
    copying the data from the array into which it was read (byterec) to the
    usable struct (crec)).
  ---------------------------------------------------------------------------*/
    if (readbuf(__G__ (char *)byterec, CREC_SIZE) == 0)
        {
        diag("no name found");
        return PK_EOF;
        }

    G.crec.version_made_by[0] = byterec[C_VERSION_MADE_BY_0];
    G.crec.version_made_by[1] = byterec[C_VERSION_MADE_BY_1];
    G.crec.version_needed_to_extract[0] =
      byterec[C_VERSION_NEEDED_TO_EXTRACT_0];
    G.crec.version_needed_to_extract[1] =
      byterec[C_VERSION_NEEDED_TO_EXTRACT_1];

    G.crec.general_purpose_bit_flag =
      makeword(&byterec[C_GENERAL_PURPOSE_BIT_FLAG]);
    G.crec.compression_method =
      makeword(&byterec[C_COMPRESSION_METHOD]);
    G.crec.last_mod_file_time =
      makeword(&byterec[C_LAST_MOD_FILE_TIME]);
    G.crec.last_mod_file_date =
      makeword(&byterec[C_LAST_MOD_FILE_DATE]);
    G.crec.crc32 =
      makelong(&byterec[C_CRC32]);
    G.crec.csize =
      makelong(&byterec[C_COMPRESSED_SIZE]);
    G.crec.ucsize =
      makelong(&byterec[C_UNCOMPRESSED_SIZE]);
    G.crec.filename_length =
      makeword(&byterec[C_FILENAME_LENGTH]);
    G.crec.extra_field_length =
      makeword(&byterec[C_EXTRA_FIELD_LENGTH]);
    G.crec.file_comment_length =
      makeword(&byterec[C_FILE_COMMENT_LENGTH]);
    G.crec.disk_number_start =
      makeword(&byterec[C_DISK_NUMBER_START]);
    G.crec.internal_file_attributes =
      makeword(&byterec[C_INTERNAL_FILE_ATTRIBUTES]);
    G.crec.external_file_attributes =
      makelong(&byterec[C_EXTERNAL_FILE_ATTRIBUTES]);  /* LONG, not word! */
    G.crec.relative_offset_local_header =
      makelong(&byterec[C_RELATIVE_OFFSET_LOCAL_HEADER]);
    Trace((stderr, "End of get_cdir_ent in process.c. Found filename of len %d",
          G.crec.filename_length));
    return PK_COOL;
} /* end function get_cdir_ent() */

/*************************************/
/* Function process_local_file_hdr() */
/*************************************/
int process_local_file_hdr(__G)    /* return PK-type error code */
     __GDEF
{
    local_byte_hdr byterec;
/*---------------------------------------------------------------------------
    Read the next local file header and do any necessary machine-type con-
    versions (byte ordering, structure padding compensation--do so by copy-
    ing the data from the array into which it was read (byterec) to the
    usable struct (lrec)).
  ---------------------------------------------------------------------------*/
    if (readbuf(__G__ (char *)byterec, LREC_SIZE) == 0)
        return PK_EOF;

    G.lrec.version_needed_to_extract[0] =
      byterec[L_VERSION_NEEDED_TO_EXTRACT_0];
    G.lrec.version_needed_to_extract[1] =
      byterec[L_VERSION_NEEDED_TO_EXTRACT_1];

    G.lrec.general_purpose_bit_flag =
      makeword(&byterec[L_GENERAL_PURPOSE_BIT_FLAG]);
    G.lrec.compression_method = makeword(&byterec[L_COMPRESSION_METHOD]);
    G.lrec.last_mod_file_time = makeword(&byterec[L_LAST_MOD_FILE_TIME]);
    G.lrec.last_mod_file_date = makeword(&byterec[L_LAST_MOD_FILE_DATE]);
    G.lrec.crc32 = makelong(&byterec[L_CRC32]);
    G.lrec.csize = makelong(&byterec[L_COMPRESSED_SIZE]);
    G.lrec.ucsize = makelong(&byterec[L_UNCOMPRESSED_SIZE]);
    G.lrec.filename_length = makeword(&byterec[L_FILENAME_LENGTH]);
    G.lrec.extra_field_length = makeword(&byterec[L_EXTRA_FIELD_LENGTH]);

    G.csize = (long) G.lrec.csize;
    G.ucsize = (long) G.lrec.ucsize;

    if ((G.lrec.general_purpose_bit_flag & 8) != 0) {
        /* can't trust local header, use central directory: */
        G.lrec.crc32 = G.pInfo->crc;
        G.csize = (long)(G.lrec.csize = G.pInfo->compr_size);
    }
    Trace((stderr, "process_local_file_hdr of process.c found fname len of %d",
        G.lrec.filename_length));
    return PK_COOL;
} /* end function process_local_file_hdr() */

#ifdef USE_EF_UX_TIME
/*************************************/
/* Function ef_scan_for_izux() */
/*************************************/
unsigned ef_scan_for_izux(ef_buf, ef_len, z_utim, z_uidgid)
    uch *ef_buf;                /* buffer containing extra field */
    unsigned ef_len;            /* total length of extra field */
    ztimbuf *z_utim;            /* return storage: atime and mtime */
    ush *z_uidgid;              /* return storage: uid and gid */
{
/* This function scans the extra field for a IZUNIX block containing
 * Unix style time_t (GMT) values for the entry's access and modification
 * time.  If a valid block is found, both time stamps are copied to the
 * ztimebuf structure (provided the z_utim pointer is not NULL).
 * If the IZUNIX block contains UID/GID fields and the z_uidgid array
 * pointer is valid (!= NULL), the owner info is transfered as well.
 * The return value is the size of the IZUNIX block found, or 0 in case
 * of failure.
 */
  unsigned r = 0;
  unsigned eb_id;
  unsigned eb_len;

  if (ef_len == 0 || ef_buf == NULL)
    return 0;

  TTrace((stderr,"\nef_scan_ux_: scanning extra field of length %u\n",
          ef_len));
  while (ef_len >= EB_HEADSIZE) {
    eb_id = makeword(EB_ID + ef_buf);
    eb_len = makeword(EB_LEN + ef_buf);
    if (eb_len > (ef_len - EB_HEADSIZE)) {
      /* Discovered some extra field inconsistency! */
      TTrace((stderr,"ef_scan_for_izux: block length %u > rest ef_size %u\n",
              eb_len, ef_len - EB_HEADSIZE));
      break;
    }
    if (eb_id == EF_IZUNIX && eb_len >= EB_UX_MINLEN) {
       TTrace((stderr,"ef_scan_ux_time: Found IZUNIX extra field\n"));
       if (z_utim != NULL) {
         z_utim->actime  = makelong((EB_HEADSIZE+EB_UX_ATIME) + ef_buf);
         z_utim->modtime = makelong((EB_HEADSIZE+EB_UX_MTIME) + ef_buf);
         TTrace((stderr,"  Unix EF access time = %ld\n",z_utim->actime));
         TTrace((stderr,"  Unix EF modif. time = %ld\n",z_utim->modtime));
       }
       if (eb_len >= EB_UX_FULLSIZE && z_uidgid != NULL) {
         z_uidgid[0] = makeword((EB_HEADSIZE+EB_UX_UID) + ef_buf);
         z_uidgid[1] = makeword((EB_HEADSIZE+EB_UX_GID) + ef_buf);
       }
       r = eb_len;              /* signal success */
       break;
    }
    /* Skip this extra field block */
    ef_buf += (eb_len + EB_HEADSIZE);
    ef_len -= (eb_len + EB_HEADSIZE);
  }
  return r;
}
#endif /* USE_EF_UX_TIME */
