/*
 Copyright (C) 1990-1996 Mark Adler, Richard B. Wales, Jean-loup Gailly,
 Kai Uwe Rommel, Onno van der Linden and Igor Mandrichenko.
 Modified by Eric W. Engler for the Delphi DLL project.
 Permission is granted to any individual or institution to use, copy, or
 redistribute this software so long as all of the original files are included,
 that it is not sold for profit, and that this copyright notice is retained.
*/

/* based on zip.c by Mark Adler. */

#define PROCNAME(n) (action==ADD||action==UPDATE?wild(n):procname(n))

#include "zip.h"
#include "crypt.h"
#include "ttyio.h"
#include <process.h>

#include <signal.h>

/* Local option flags */
#define DELETE  0
#define ADD     1
#define UPDATE  2
#define FRESHEN 3
int action = ADD; /* Must be ADD - the default action */
int latest = 0;   /* 1=set zip file time to time of newest file */
ulg before = 0;   /* 0=ignore, else exclude files before this time */
int tempdir = 0;  /* 1=use temp directory (-b) */
int junk_sfx = 0; /* 1=junk (remove) the sfx prefix */
FILE *x, *y;      /* input and output zip files */
FILE *tempzf;

local char *tempzip; /* name of tempfile */

#ifdef CRYPT
/* Pointer to crc_table, needed in crypt.c */
ulg near *crc_32_tab;
#endif

/* Local functions */

local void freeup  OF((void));
local void finish  OF((int));
local void get_filters OF((int argc, char **argv));
int  zipmain  OF((int, char **));

/* Free all allocations in the found list and the zfiles list */
local void freeup()
{
  struct flist far *f;  /* steps through found list */
  struct zlist far *z;  /* pointer to next entry in zfiles list */

  for (f = found; f != NULL; f = fexpel(f))
    ;
  while (zfiles != NULL)
  {
    z = zfiles->nxt;
    free((zvoid *)(zfiles->name));
    if (zfiles->zname != zfiles->name)
      free((zvoid *)(zfiles->zname));
    if (zfiles->ext)
      free((zvoid *)(zfiles->extra));
    if (zfiles->cext && zfiles->cextra != zfiles->extra)
      free((zvoid *)(zfiles->cextra));
    if (zfiles->com)
      free((zvoid *)(zfiles->comment));
    farfree((zvoid far *)zfiles);
    zfiles = z;
    zcount--;
  }
}

/* Process -o and -m options (if specified), free up malloc'ed stuff, and
   exit with the code e. */
local void finish(e)
int e;                  /* exit code */
{
  int r;                /* return value from trash() */
  ulg t;                /* latest time in zip file */
  struct zlist far *z;  /* pointer into zfile list */

  /* If latest, set time to zip file to latest file in zip file */
  if (latest && zipfile && strcmp(zipfile, "-"))
  {
    diag("changing time of zip file to time of latest file in it");
    /* find latest time in zip file */
    if (zfiles == NULL)
       zipwarn("zip file is empty, can't make it as old as latest entry", "");
    else {
      t = 0;
      for (z = zfiles; z != NULL; z = z->nxt)
        /* Ignore directories in time comparisons */
#ifdef USE_EF_UX_TIME
        if (z->zname[z->nam-1] != '/')
        {
          ztimbuf z_utim;
          ulg z_tim;

          z_tim = (get_ef_ux_ztime(z, &z_utim) ?
                   unix2dostime(&z_utim.modtime) : z->tim);
          if (t < z_tim)
            t = z_tim;
        }
#else /* !USE_EF_UX_TIME */
        if (z->zname[z->nam-1] != '/' && t < z->tim)
          t = z->tim;
#endif /* ?USE_EF_UX_TIME */
      /* set modified time of zip file to that time */
      if (t != 0)
        stamp(zipfile, t);
      else
        zipwarn(
         "zip file has only directories, can't make it as old as latest entry",
         "");
    }
  }
  if (tempath != NULL)
  {
    free((zvoid *)tempath);
    tempath = NULL;
  }
  if (zipfile != NULL)
  {
    free((zvoid *)zipfile);
    zipfile = NULL;
  }


  /* If dispose, delete all files in the zfiles list that are marked */
  if (dispose)
  {
    diag("deleting files that were added to zip file");
    if ((r = trash()) != ZE_OK)
      ziperr(r);
      return;
  }

  /* Done! */
  freeup();
}


/* Issue a message for an error, clean up files and memory, and exit. */
void ziperr(c)
int c;                  /* error code from the ZE_ class */
{
  static char errmsg[80];
     
  if (user_notified_of_abort)
     return;
  user_notified_of_abort=1;

  global_error_code = c;   /* last error code */

  switch(c) {
     case ZE_MISS:
     case ZE_OPEN:
        strcpy(errmsg,"File not found");
        break;
     case ZE_EOF:
        strcpy(errmsg,"Unexpected end of file");
        break;
     case ZE_MEM:
        strcpy(errmsg,"Out of memory");
        break;
     case ZE_ABORT:
        strcpy(errmsg,"Aborted by User");
        break;
     case ZE_READ:
        strcpy(errmsg, "Error reading a file");
        break;
     case ZE_WRITE:
        strcpy(errmsg, "Error writing to a file");
        break;
     case ZE_CREAT:
        strcpy(errmsg, "Error creating file");
        break;
     case ZE_NAME:
        strcpy(errmsg, "Missing or empty zip file");
        break;
     case ZE_PARMS:
        strcpy(errmsg, "Bad zip options specified");
        break;
     case ZE_FORM:
        strcpy(errmsg, "zip file structure error");
        break;

     /* The less likely errors: */
     case ZE_LOGIC:
     case ZE_BIG:
     case ZE_NOTE:
     case ZE_TEST:
     case ZE_TEMP:
     case ZE_NONE:
        strcpy(errmsg, "Unknown error occured");
  } /* end switch */
             

  if (dll_handles_errors)
     {
     // I'm passing the error via the callback just to get it logged in
     // the status box - I'm sending it in with a 0 error code to avoid
     // a dialog box from the applic pgm.
     user_callback(4,0,0,errmsg);  
     sprintf(ewemsg, "%s   code=%d", errmsg, c);
     msgbox(ewemsg);
     }
  else
     user_callback(4,c,0,errmsg);  /* applic pgm handles errors */

  freeup();  // free memory for file linked lists, etc.

  if (tempzip != zipfile) 
     {
     if (tempzf != NULL)
        fclose(tempzf);
     tempzf=NULL;
     destroy(tempzip);
     free((zvoid *)tempzip);
     }
  if (zipfile != NULL)
     {
     free((zvoid *)zipfile);
     zipfile=NULL;
     }
  if (tempath != NULL)
     {
     free((zvoid *)tempath);
     tempath = NULL;
     }
  if (tempzf != NULL)
     {
     fclose(tempzf);
     tempzf = NULL;
     }
  if (key != NULL)
    free((zvoid *)key);
  if (x != NULL)
     {
     fclose(x);
     x=NULL;
     }
  if (y != NULL)
     {
     fclose(y);
     y=NULL;
     }

#ifdef NEVER
// EWE NOTE: This code is problematic!!!
      /* -g option, attempt to restore the old file */
      int k = 0;            /* keep count for end header */
      ulg cb = cenbeg;      /* get start of central */
      struct zlist far *z;  /* steps through zfiles linked list */

      fprintf(stderr, "attempting to restore %s to its previous state\n",
                 zipfile);
      fseek(tempzf, cenbeg, SEEK_SET);
      tempzn = cenbeg;
      for (z = zfiles; z != NULL; z = z->nxt)
      {
        putcentral(z, tempzf);
        tempzn += 4 + CENHEAD + z->nam + z->cext + z->com;
        k++;
      }
      putend(k, tempzn - cb, cb, zcomlen, zcomment, tempzf);
      fclose(tempzf);
      tempzf = NULL;
#endif
}

/* Internal error, should never happen */
void error(h)
  char *h;
{
  ziperr(ZE_LOGIC);
}

/* Print a warning message to stderr and return. */
void zipwarn(a, b)
char *a, *b;  
{
   fprintf(stderr, "%s: %s\n", a, b); // combine both messages
}

/* Counts number of -i or -x patterns, sets patterns and pcount */
local void get_filters(argc, argv)
  int argc;               /* number of tokens in command line */
  char **argv;            /* command line tokens */
{
  int i;
  int flag = 0;

  pcount = 0;
  for (i = 1; i < argc; i++) {
    if (argv[i][0] == '-') {
      if (strrchr(argv[i], 'i') != NULL) {
        flag = 'i';
      } else if (strrchr(argv[i], 'x') != NULL) {
        flag = 'x';
      } else {
        flag = 0;
      }
    } else if (flag) {
      if (patterns != NULL) {
        patterns[pcount].zname = ex2in(argv[i], 0, (int *)NULL);
        patterns[pcount].select = flag;
        if (flag == 'i') icount++;
      }
      pcount++;
    }
  }
  if (pcount == 0 || patterns != NULL) return;
  patterns = (struct plist*) malloc(pcount * sizeof(struct plist));
  if (patterns == NULL) {
    ziperr(ZE_MEM);
    return;
    }
  get_filters(argc, argv);
}

/* Add, update, freshen, or delete zip entries in a zip file. */
int zipmain(argc, argv)
    int             argc;       /* number of tokens in command line */
    char          **argv;       /* command line tokens */
{
    int             a;          /* attributes of zip file */
    ulg             c;          /* start of central directory */
    int             d;          /* true if just adding to a zip file */
    struct flist far *f;        /* steps through "found" linked list */
    int             i;          /* arg counter, root directory flag */
    int             k;          /* next argument type, marked counter,
                                 * comment size, entry count */
    ulg             n;          /* total of entry len's */
    int             o;          /* true if there were any ZE_OPEN errors */
    char           *p;          /* steps through option arguments */
    int             r;          /* temporary variable */
    ulg             t;          /* file time, length of central directory */
    int             first_listarg = 0;  /* index of first arg of "process
                                         * these files" list */
    struct zlist far *v;        /* temporary variable */
    struct zlist far *far * w;  /* pointer to last link in "zfiles" list */
    struct zlist far *z;        /* steps through "zfiles" linked list */

    /*
     * Re-initialize global variables to make the zip dll re-entrant. It is
     * possible that we could get away with not re-initializing all of these
     * but better safe than sorry.
     */
    action = ADD;               /* must be ADD as default */
    latest = 0;                 /* 1=set zip file time to time of latest file */
    before = 0;                 /* 0=ignore, else exclude files before this
                                 * time */
    tempdir = 0;                /* 1=use temp directory (-b) */
    junk_sfx = 0;               /* 1=junk (ignore) the sfx prefix */
    tempzip = NULL;
    fcount = 0;
    recurse = 0;                /* 1=recurse into directories encountered */
    dispose = 0;                /* 1=remove orig files after put in zip file */

    pathput = 1;                /* 1=store with pathname by dflt */
    dirnames = 1;               /* include entries that only have a dirname by default */
    linkput = 0;                /* 1=store symbolic links as such */

    method = DEFLATE;           /* one of BEST, DEFLATE, or STORE */
    dosify = 0;                 /* 1=make new entries look like MSDOS (8x3) */
    verbose = 0;                /* 1=report oddities in zip file structure */
    adjust = 1;                 /* 1=adjust offsets for sfx'd file (keep
                                 * preamble) -A */
    level = 9;                  /* 0=no compression, 9=best compression */
    translate_eol = 0;          /* Translate end-of-line CRLF -> LF */
    use_longname_ea = 0;        /* 1=use the .LONGNAME EA as the file's name */
    hidden_files = 0;           /* process hidden and system files */
    volume_label = 0;           /* add volume label */
    noisy = 1;                  /* 0=quiet operation */

    /* extra fields are: unix timestamps, read-only, etc */
    extra_fields = 0;           /* 0=do not create extra fields */

    /* User can specify if he wants to skip compressing these types: */
    /* List of spcl suffixes */
    special = ".gif:.png:.Z:.zip:.zoo:.arc:.lzh:.arj:.taz:.tgz:.lha"; 

    key = NULL;                 /* Scramble password if scrambling */
    tempath = NULL;             /* Path for temporary files */

    /* linked list for new files to be added (not yet in ZIP) */
    found = NULL;         
    fnxt = &found;

    patterns = NULL;            /* List of patterns to be matched */
    pcount = 0;                 /* number of patterns */
    icount = 0;                 /* number of include only patterns */

    init_upper();               /* build case map table */

    if ((p = getenv("TZ")) == NULL || *p == '\0')
        extra_fields = 0;       /* disable storing "Unix" time stamps */

    /* Process arguments */
    diag("processing arguments");

    zipfile = tempzip = NULL;
    tempzf = NULL;
    d = 0;   /* disallow appending to a zip file */
    k = 0;   /* Next non-option argument type (zip filename by dflt) */

    // skip over pgm name in argv[0] 
    for (i = 1; i < argc; i++) {
        if ((argv[i] == NULL) || (argv[i][0] == 0)) {
           diag("breaking from outer loop");
           break;
        }

        if (argv[i][0] == '-') {  
            // found dash that likely preceeds a switch 

            if (argv[i][1]) {   /* if there is a char following the dash */
                for (p = argv[i] + 1; *p; p++) {
                    switch (*p) {
                    case '0':
                        method = STORE;
                        level = 0;
                        break;
                    case '1':
                    case '2':
                    case '3':
                    case '4':
                    case '5':
                    case '6':
                    case '7':
                    case '8':
                    case '9':
                        /* Set the compression level */
                        level = *p - '0';
                        sprintf(ewemsg, "setting compression level to %d", level);
                        diag(ewemsg);
                        break;
                    case 'A':  /* Adjust unzipsfx'd zipfile: adjust offsets
                                * only */
                        diag("A switch found");
                        adjust = 1;
                        break;
                    case 'b':   /* Specify path for temporary file */
                        diag("b switch found");
                        tempdir = 1;
                        /* if k isn't 0, then we have already parsed the
                           zip filename */
                        if (k != 0) {
                            ziperr(ZE_PARMS);
                            return 0;
                        } else
                            k = 1;      /* Next non-option is path */
                        break;
                    case 'd':   /* Delete files from zip file */
                        diag("d switch found");
                        if (action != ADD) {
                            ziperr(ZE_PARMS);
                            return 0;
                        }
                        action = DELETE;
                        break;
                    case 'D':   /* Do not add directory entries */
                        diag("D switch found");
                        dirnames = 0;
                        break;
                    case 'e':   /* Encrypt */
                        diag("e switch found");

#ifndef CRYPT
                        ziperr(ZE_PARMS);
                        return 0;
#else   /* CRYPT */
                        if (key == NULL) {
                            if ((key = malloc(PWLEN + 1)) == NULL) {
                                ziperr(ZE_MEM);
                                return 0;
                            }
                            /*
                             * DLL NOTE: If we support encryption, we need a
                             * 2-way dialog box to get password from user
                             * - not implemented yet
                             */
                            if (getp("Enter password: ", key, PWLEN + 1) == NULL) {
                                ziperr(ZE_PARMS);
                                return 0;
                            }
                            if ((e = malloc(PWLEN + 1)) == NULL) {
                                ziperr(ZE_MEM);
                                return 0;
                            }
                            if (getp("Verify password: ", e, PWLEN + 1) == NULL) {
                                ziperr(ZE_PARMS);
                                return 0;
                            }
                            r = strcmp(key, e);
                            free((zvoid *) e);
                            if (r) {
                                ziperr(ZE_PARMS);
                                return 0;
                            }
                            if (*key == '\0') {
                                ziperr(ZE_PARMS);
                                return 0;
                            }
                        }
#endif    /* ?CRYPT */
                        break;

                    case 'f':   /* Freshen zip file--overwrite only */
                        diag("f switch found");
                        if (action != ADD) {
                            ziperr(ZE_PARMS);
                            return 0;
                        }
                        action = FRESHEN;
                        break;
                    case 'g':   /* Allow appending to a zip file */
                        diag("g switch found");
                        d = 1;
                        break;
                    case 'j':   /* Junk (don't save) directory names */
                        diag("j switch found");
                        pathput = 0;
                        break;
                    case 'J':   /* Junk (remove) sfx prefix */
                        diag("J switch found");
                        junk_sfx = 1;
                        break;
                    case 'k':   /* Make entries using DOS names (8x3) */
                        diag("k switch found");
                        dosify = 1;
                        break;
                    case 'l':   /* Translate end-of-line */
                        diag("l switch found");
                        translate_eol++;
                        break;
                    case 'm':   /* Delete orig files added or updated in zip file */
                        diag("m switch found");
                        dispose = 1;
                        break;
                    case '!':   /* DO try to compress files with a special suffix */
                        diag("! switch found"); /* this is oppos of orig -n */
                        special = NULL;
                        break;
                    case 'o':   /* Set zip file time to time of latest file
                                 * in it */
                        diag("o switch found");
                        latest = 1;
                        break;
                    case 'p':   /* Store path with name */
                        diag("p switch found");
                        break;  /* do nothing - this is default */
                    case 'q':   /* Quiet operation */
                        diag("q switch found");
                        noisy = 0;     /* shut us up! */
                        verbose = 0;   /* override verbose option */
                        break;
                    case 'r':   /* Recurse into subdirectories */
                        /* we won't necessarily save dirnames */
                        diag("r switch found");
                        recurse = 1;
                        break;
                    case 'S':
                        diag("S switch found");
                        hidden_files = 1;
                        break;
                    case 't':   /* Exclude files earlier than specified date */
                        diag("t switch found");
                        if (before) {
                            ziperr(ZE_PARMS);
                            return 0;
                        }
                        k = 2;  // look for a date
                        break;
                    case 'u':   /* Update zip file--overwrite only if newer */
                        diag("u switch found");
                        if (action != ADD) {
                            ziperr(ZE_PARMS);
                            return 0;
                        }
                        action = UPDATE;
                        break;
                    case 'v':   /* Verbose messages */
                        diag("v switch found");
                        noisy = 1;
                        verbose++;
                        break;
                    case '$':   /* Include volume label */
                        diag("$ switch found");
                        volume_label = 1;
                        break;
                    case 'X':
                        diag("X switch found");
                        extra_fields = 0;
                        break;
                    default:
                        {
                        sprintf(errbuf, "no such option: %02X hex", *p);  
                        ziperr(ZE_PARMS);
                        return 0;
                        }
                    }  /* END switch/case  */
                }  /* end of inner for */
            } else {
                /* just a dash */
                switch (k) {
                case 0:   // need name of zipfile
                case 1:   // need name of tempdir
                    ziperr(ZE_PARMS);
                    return 0;
                case 2:   // need cutoff date
                    ziperr(ZE_PARMS);
                    return 0;
                case 3:   // need arg filename
                    // "procname" and "wild" are in win32zip.c
                    // for ADD, wild is used
                    if ((r = PROCNAME(argv[i])) != ZE_OK)
                        {
                        if (r == ZE_MISS)
                           /* this occurs if no file already in zip archive
                              was matched by the fspec */
                           zipwarn("Not found: ", argv[i]);
                        else 
                           {
                           diag("this can't happen");
                           ziperr(r);
                           return 0;
                           }
                        }
                } /* end switch */
            }   /* end inner for */

        } else {

            /* no dash was found in this arg */
            diag("reached \"no dash was found\"; checking for zip name or fname");

            /*  k controls the simulated state machine
            k = 0   get spec of zip filename
            k = 1   get spec for tempdir        
            k = 2   get spec for cutoff date    
            k = 3   get spec of first arg filename
            k = 4   get specs of 2nd thru "n" arg filenames  
            */

            switch (k) {
                case 0:
                    diag("k was 0, need zipfname");
                    /* in this state, we need the name of the zip file */
                    if ((zipfile = ziptyp(argv[i])) == NULL) {
                        ziperr(ZE_MEM);
                        return 0;
                    }

                    diag("ready to read zip file");
                    /* the read will be done in file: zipfile.c */

                    if ((r = readzipfile()) != ZE_OK) {
                        diag("err returned from \"readzipfile\"");
                        ziperr(r);
                        return 0;
                    }
                    /* change to state 3: looking for arg filenames */
                    k = 3;
                    break;
                case 1:
                    diag("k was 1");  // get tempdir spec
                    if ((tempath = malloc(strlen(argv[i]) + 1)) == NULL) {
                        ziperr(ZE_MEM);
                        return 0;
                    }
                    strcpy(tempath, argv[i]);
                    k = 0;  // tempdir spec OK, get zipfname
                    break;
                case 2:
                    { // get cutoff date
                    int  yy, mm, dd;        /* results of sscanf() */

                    diag("k was 2");
                    if (sscanf(argv[i], "%2d%2d%2d", &mm, &dd, &yy) != 3 ||
                      mm < 1 || mm > 12 || dd < 1 || dd > 31) {
                        ziperr(ZE_PARMS);
                        return 0;
                    }
                    // no "year 2000" problem here!
                    before = dostime(yy + (yy < 80 ? 2000 : 1900), mm, dd, 0, 0, 0);
                    k = 0;  // get zipfname 
                    break;
                    }
                case 3:
                case 4:
                    diag("k was 3 or 4, looking for arg fnames");
                    if ((r = PROCNAME(argv[i])) != ZE_OK)
                        {
                        if (r == ZE_MISS)
                           {
                           /* this occurs if new file wasn't found */
                           sprintf(ewemsg,"File Not Found: %s", argv[i]);
                           diag(ewemsg);
                           user_callback(4,100,0,ewemsg);
                           }
                        else 
                           {
                           ziperr(r);
                           return 0;
                           }
                        }
                    if (k == 3) {
                        diag("k was 3");
                        first_listarg = i; // the first fname's argv index
                        k = 4;  // look for arg fnames 2..n */
                    }
                }  /* end switch/case */
        } /* end of "no dash" processing */

    }  /* end of outer "for" loop */



    switch (action) {
       case ADD:  diag("action=ADD"); break;
       case UPDATE: diag("action=UPDATE"); break;
       case FRESHEN: diag("action=FRESHEN"); break;
       case DELETE: diag("action=DELETE"); break;
    }

    /* zcount is no. of entries in linked-list */
    /* zfiles is name of the linked-list of filenames for the archive */
    sprintf(ewemsg, "zcount=%d (no. of files in ZIP already)", zcount);
    diag(ewemsg);


    /* Clean up selections ("3 <= k <= 5" now) */
    /* if first_listarg is 0, then we didn't gat any fspecs on cmd line */
    if (k != 4 && first_listarg == 0 &&
     (action == UPDATE || action == FRESHEN)) 
        {
        /* if -update or -freshen with no args, do all, but, 
           when present, apply filters */
        for (z = zfiles; z != NULL; z = z->nxt) 
            {
            z->mark = pcount ? filter(z->zname) : 1;
            }
        }

    if ((r = check_dup()) != ZE_OK)  /* remove duplicates in list */
        if (r == ZE_PARMS) {
            ziperr(r);
            return 0;
        } else {
            ziperr(r);
            return 0;
        }

    if (zcount)
        free((zvoid *) zsort);

    /* Check option combinations */
    if (action == DELETE && ( dispose || recurse || key != NULL )) {
        ziperr(ZE_PARMS);
        return 0;
    }
    if (linkput && dosify) {
        zipwarn("can't use -y with -k, -y ignored", "");
        linkput = 0;
    }

    /* d is the "allow append" indicator */
    /* if zcount is 0, then zipfile doesn't exist, or is empty */
    if (zcount == 0 && ((action != ADD) || !d)) {
        zipwarn(zipfile, " not found or empty");
        return 0;
    }

    /* If -b not specified, set temporary path to zipfile path  */
    if (tempath == NULL && ((p = strrchr(zipfile, '/')) != NULL ||
                            (p = strrchr(zipfile, '\\')) != NULL ||
                            (p = strrchr(zipfile, ':')) != NULL)) {
        if (*p == ':')
            p++;
        if ((tempath = malloc((int) (p - zipfile) + 1)) == NULL) {
            ziperr(ZE_MEM);
            return 0;
        }
        r = *p;
        *p = 0;
        strcpy(tempath, zipfile);
        *p = (char) r;
    }

    // NOTE: "k" is being redefined below this point.  Now, it going to
    // track the no. of marked files in the "zfiles" linked list.

    /* For each marked entry in "zfiles" linked list, if not deleting,
     * check if a corresponding "external" file exists. 
     * If updating or freshening, compare date of "external" file with 
     * entry in orig zipfile.
     * Unmark if it doesn't exist or is too old, else mark it.
     * Entries that are marked will cause that file to be rezipped.
     */

     diag("checking marked entries");

     k = 0;   /* Initialize marked count */
     /* zfiles is name of the linked-list of filenames for the archive */
     for (z = zfiles; z != NULL; z = z->nxt)
        if (z->mark) {
#ifdef USE_EF_UX_TIME
            ztimbuf  f_utim, z_utim;
#endif  /* USE_EF_UX_TIME */

            if (action != DELETE &&
#ifdef USE_EF_UX_TIME
             ((t = filetime(z->name, (ulg *) NULL, (long *) NULL, &f_utim))
#else   /* !USE_EF_UX_TIME */
             ((t = filetime(z->name, (ulg *) NULL, (long *) NULL, (ztimbuf *) NULL))
#endif  /* ?USE_EF_UX_TIME */
               == 0 || t < before ||
               ((action == UPDATE || action == FRESHEN) &&
#ifdef USE_EF_UX_TIME
                (get_ef_ux_ztime(z, &z_utim) ?
                     f_utim.modtime <= z_utim.modtime : t <= z->tim)
#else   /* !USE_EF_UX_TIME */
                 t <= z->tim
#endif  /* ?USE_EF_UX_TIME */
                )))   {
                z->mark = 0;
                z->trash = t && t >= before;    /* delete if -um or -fm */
                if (verbose) {
                    printf("%s %s\n", z->name,
                          z->trash ? "up to date" : "missing or early");
                }
            } else
                k++;   /* incr. number of marked entries */
        }

    /* Remove entries from "found" linked list if no "external" matching 
       file is found, or if found, but are too old.  If filetime() returns
       any valid time, then at least we know the file was found. */
    diag("checking new entries");


    /*============ fileio.c built the found list ==============*/
    if (found == NULL)
       diag("found list empty - a");
    else
       diag("found list has at least one entry - a");

    for (f = found; f != NULL; )
       {
       if (action == DELETE || action == FRESHEN ||
        (t = filetime(f->name, (ulg *) NULL, (long *) NULL, (ztimbuf *) NULL))
        == 0 || t < before || (namecmp(f->name, zipfile) == 0 
        && strcmp(zipfile, "-")))
            {
            diag("fexpel being called");
            f = fexpel(f);  // delete an entry in the list
            }
       else
            /* file found, and not too old */
            f = f->nxt;     // save this one, link it up
       }

    /* Make sure there's something left to do */
    if (k == 0 && found == NULL && !(zfiles != NULL &&
     (latest || adjust || junk_sfx ))) {
        /* FOUND WAS NULL HERE, so just figure out which error
           message to show the user */
        if (action == UPDATE || action == FRESHEN) {
            finish(ZE_OK);
            return 0;
        } else if (zfiles == NULL && (latest || adjust || junk_sfx)) {
            ziperr(ZE_NAME);
            return 0;
        } else if (recurse && (pcount == 0) && (first_listarg > 0)) {
            /* add the list of filenames from cmd line to error msg */
            for (i = first_listarg; i < argc; i++)
                strcat(strcat(errbuf, " "), argv[i]);
            ziperr(ZE_NONE);
            return 0;
        } else {
            ziperr(ZE_NONE);
            return 0;
        }
    }
    d = (d && k == 0 && (zipbeg || zfiles != NULL)); /* d is true if appending */

    // continue on to add new files

#ifdef CRYPT
    /* Initialize the crc_32_tab pointer, when encryption was requested. */
    if (key != NULL)
        crc_32_tab = (ulg near *) get_crc_table();
#endif  /* CRYPT */

    /* Make sure zip file is writeable. This has the undesired side effect 
     * of leaving one empty junk file on a WORM, so when the zipfile does
     * not exist already and when -b is specified, the
     * writability check is made in replace().
     */
    if (strcmp(zipfile, "-")) {
        if (tempdir && zfiles == NULL && zipbeg == 0) {
            a = 0;
        } else {
            x = zfiles == NULL && zipbeg == 0 ? fopen(zipfile, FOPW) :
                fopen(zipfile, FOPM);
            /* Note: FOPW and FOPM expand to several parameters for VMS */
            if (x == NULL) {
                ziperr(ZE_CREAT);
                return 0;
            }
            fclose(x);
            a = getfileattr(zipfile);
            if (zfiles == NULL && zipbeg == 0)
                destroy(zipfile);
        }
    } else
        a = 0;

    /* Throw away the garbage in front of the zip file for -J */
    if (junk_sfx)
        zipbeg = 0;

    /* OPEN ZIP FILE and temporary output file */
    diag("opening zip file and creating temporary zip file");
    x = NULL;
    tempzn = 0;
#ifdef NEVER
    if (strcmp(zipfile, "-") == 0) {
        diag("STDOUT - Should never happen");
        /* Set stdout mode to binary for MSDOS systems */
        setmode(fileno(stdout), O_BINARY);
        tempzf = y = fdopen(fileno(stdout), FOPW);
        /* tempzip must be malloced so a later free won't barf */
        tempzip = malloc(4);
        if (tempzip == NULL) {
            ziperr(ZE_MEM);
            return 0;
        }
        strcpy(tempzip, "-");
    } else
#endif
     if (d) {
        /* zipfile is not stdout, and we are allowed to append */
        /* d is true if we're just appending (-g) */

        diag("in dllzip - ready to fopen in mode FOPM - point B");
        if ((y = fopen(zipfile, FOPM)) == NULL) {
            ziperr(ZE_NAME);
            return 0;
        }
        tempzip = zipfile;
        tempzf = y;
        if (fseek(y, cenbeg, SEEK_SET)) {
            ziperr(ferror(y) ? ZE_READ : ZE_EOF);
            return 0;
        }
        tempzn = cenbeg;
    } else {
        diag("in dllzip - ready to fopen for FOPR_EX");
        if ((zfiles != NULL || zipbeg) && (x = fopen(zipfile, FOPR_EX)) == NULL) {
            ziperr(ZE_NAME);
            return 0;
        }
        if ((tempzip = tempname(zipfile)) == NULL) {
            ziperr(ZE_MEM);
            return 0;
        }
        if ((tempzf = y = fopen(tempzip, FOPW)) == NULL) {
            ziperr(ZE_TEMP);
            return 0;
        }
    }

    if (strcmp(zipfile, "-") != 0 && !d) {  
        /* this must go after setvbuf */
        /* copy a compressed file from old archive to new archive */
        if (zipbeg && (r = fcopy(x, y, zipbeg)) != ZE_OK) {
            ziperr(r);
            return 0;
        }
        tempzn = zipbeg;
    }
    o = 0;      /* no ZE_OPEN errors yet */


    /* Process zip file, copying from old archive to new archive.
       Rezip any marked files */
    if (zfiles != NULL)
       diag("going through old zip file");
    w = &zfiles;

    while ((z = *w) != NULL) {
        if (global_abort_sw)
           {
           ziperr(ZE_ABORT);
           return 0;
           }
        if (z->mark == 1) {     
            /* This file is marked */
            /* if not deleting, rezip it */
            if (action != DELETE) {
                printf("updating: %s", z->zname);

                // zipup is in file zipup.c                
                if ((r = zipup(z, y)) != ZE_OK && r != ZE_OPEN && r != ZE_MISS) {
                    ziperr(r);
                    return 0;
                }
                if (r == ZE_OPEN || r == ZE_MISS) {
                    o = 1;
                    if (r == ZE_OPEN) {
                        zipwarn("could not open for reading: ", z->name);
                        user_callback(4, 2, 0, z->name);
                    } else {
                        zipwarn("file and directory with the same name: ", z->name);
                        user_callback(4, 4, 0, z->name);
                    }
                    zipwarn("will just copy entry over: ", z->zname);
                    if ((r = zipcopy(z, x, y)) != ZE_OK) {
                        sprintf(errbuf, "was copying %s", z->zname);
                        ziperr(r);
                        return 0;
                    }
                    z->mark = 0;
                }
                /* we know: r = ZE_OK  */
                w = &z->nxt;
#ifdef MSWIN
                diag("about to make final progress call for a file");
                user_callback(2,0,0,NULL);  // final progress call
#endif 
                files_acted_on++;
            } else {
                /* desired action is DELETE, this file marked  */
                // NOTE: no progress bar supt for DELETE yet
                printf("deleting: %s\n", z->zname);
                
                v = z->nxt;     /* delete entry from list */
                free((zvoid *) (z->name));
                free((zvoid *) (z->zname));
                if (z->ext)
                    free((zvoid *) (z->extra));
                if (z->cext && z->cextra != z->extra)
                    free((zvoid *) (z->cextra));
                if (z->com)
                    free((zvoid *) (z->comment));
                farfree((zvoid far *) z);
                *w = v;
                zcount--;
#ifdef MSWIN
             // progress not yet supported on DELETE
             // user_callback(2,0,0,NULL);  // final progress call
#endif 
                files_acted_on++;
            }
        } else {
            /* this file wasn't marked */
            /* copy the original entry verbatim */
            if (!d && (r = zipcopy(z, x, y)) != ZE_OK) {
                sprintf(errbuf, "was copying %s", z->zname);
                ziperr(r);
                return 0;
            }
            w = &z->nxt;
        }
    } /* end while */

    /* Process the "found" list, adding them to the zip file. */
    /* This is used to add files that weren't already in the archive. */
    diag("zipping up NEW entries, if any");

    if (found == NULL)
       diag("found list empty - b");
    else
       diag("found list has at least one entry - b");

    /* For each new file to add (src names in found list), make a new
       entry for it in the "zfiles" linked list, zip up the new file, 
       then remove the entry from the found list. */
    /* The last item in the for loop control deallocates spc for fname
       that was just zipped up */
    for (f = found; f != NULL; f = fexpel(f)) {
        /* add a new entry to "zfiles" list, before we zip up the
           file.  That way we'll be ready to update the ZIP file's
           directory later. */
        if (global_abort_sw) {
           ziperr(ZE_ABORT);
           return 0;
        }
        if ((z = (struct zlist far *) farmalloc(sizeof(struct zlist))) == NULL) {
            ziperr(ZE_MEM);
            return 0;
        }

        /* Similar names below are confusing: 
            f->name
            f->zname
            z->name
            z->zname
        */

        z->nxt = NULL;

        z->name = f->name;
        f->name = NULL;

        z->zname = f->zname;
        f->zname = NULL;

        z->ext = z->cext = z->com = 0;
        z->extra = z->cextra = NULL;
        z->mark = 1;
        z->dosflag = f->dosflag;

        /* zip it up */
        printf("  adding: %s", z->zname);

        /************  This is it - try to zip up new file  **************/     
        if ((r = zipup(z, y)) != ZE_OK && r != ZE_OPEN && r != ZE_MISS) {
            ziperr(r);
            return 0;
        }
        if (r == ZE_OPEN || r == ZE_MISS) {
            o = 1;
            if (r == ZE_OPEN) {
                zipwarn("could not open for reading: ", z->name);
            } else {
                zipwarn("file and directory with the same name: ", z->name);
            }
            free((zvoid *) (z->name));
            free((zvoid *) (z->zname));
            farfree((zvoid far *) z);
        } else {
            /* "zipup" of this file was good */
            *w = z;
            w = &z->nxt;
            zcount++;
#ifdef MSWIN
            user_callback(2,0,0,NULL);  // final progress call
#endif 
            files_acted_on++;
       }
    }
    if (key != NULL) {
       free((zvoid *) key);
       key = NULL;
    }

    /* Write central directory and end header to temporary zip */
    diag("writing central directory");
    k = 0;      /* keep count of new fnames for ZIPfile's end header */
    c = tempzn; /* get start of central directory */
    n = t = 0;
    for (z = zfiles; z != NULL; z = z->nxt) {
        if ((r = putcentral(z, y)) != ZE_OK) {
            ziperr(r);
            return 0;
        }
        tempzn += 4 + CENHEAD + z->nam + z->cext + z->com;
        n += z->len;
        t += z->siz;
        k++;
    }

    if (k == 0)
        zipwarn("zip file empty", "");

    if ((verbose) && (action == ADD) && (!global_error_code) 
     && (files_acted_on > 0))
        printf("Total Bytes=%lu, compr bytes=%lu -> %d%% savings",
                n, t, percent(n, t));

    t = tempzn - c;   /* compute length of central */

    diag("writing end of central directory");
    if ((r = putend(k, t, c, zcomlen, zcomment, y)) != ZE_OK) {
        ziperr(r);
        return 0;
    }

    tempzf = NULL;
    if (fclose(y)) {
        ziperr(d ? ZE_WRITE : ZE_TEMP);
        return 0;
    }
    if (x != NULL)
        fclose(x);

    lm_free();

    /* Replace old zip file with new zip file, leaving only the new one */
    if (strcmp(zipfile, "-") && !d) {
        diag("replacing old zip file with new zip file");
        if ((r = replace(zipfile, tempzip)) != ZE_OK) {
            zipwarn("new zip file left as: ", tempzip);
            free((zvoid *) tempzip);
            tempzip = NULL;
            ziperr(r);
            return 0;
        }
        free((zvoid *) tempzip);
    }
    tempzip = NULL;
    if (a && strcmp(zipfile, "-")) {
        setfileattr(zipfile, a);
    }
#ifdef MSWIN
    user_callback(3,0,0,NULL);  // done with a batch of files
#endif 
    /* Finish up (process -o, -m, clean up).  Exit code depends on o. */
    finish(o ? ZE_OPEN : ZE_OK);
    return 0;    /* just to avoid compiler warning */
}

