/*
 Copyright (C) 1990-1996 Mark Adler, Richard B. Wales, Jean-loup Gailly,
 Kai Uwe Rommel, Onno van der Linden and Igor Mandrichenko, Eric W. Engler.
 Permission is granted to any individual or institution to use, copy, or
 redistribute this software so long as all of the original files are included,
 that it is not sold for profit, and that this copyright notice is retained.
*/

/* util.c by Mark Adler. */
#include "zip.h"
#include <ctype.h>

/* Country-dependent case map table */
uch upper[256], lower[256];

#ifndef UTIL /* UTIL picks out namecmp code (all utils) */

/* Local functions */
local int recmatch OF((uch *, uch *));
local int count_args OF((char *s));

#ifdef NO_MKTIME
#include "mktime.c"
#endif

/* EWE NOTE: a shell expression is just a wildcard for fspec */
/* If p is a sh expression, a pointer to the first special 
   character is returned.  Otherwise, NULL is returned. */
char *isshexp(p)
char *p;        /* candidate sh expression */
{
  for ( ; *p; p++)
    if (*p == '\\' && *(p+1))
      p++;
    else if (*p == '?' || *p == '*' || *p == '[')   /* unix/dos wildcards */
      return p;
  return NULL;  /* no wildcards found */
}


/* Recursively compare the sh pattern p, with the string s,
   and return 1 if they match, and 0 or 2 if they don't (or if 
   there is a syntax error in the pattern).  This routine 
   recurses on itself no deeper than the number of characters
   in the pattern. */
local int recmatch(p, s)
uch *p;    /* sh pattern to match (ex: *.*) */
uch *s;    /* string to match it to (ex: FNAME.DAT) */
{
  unsigned int c;       /* pattern char or start of range in [-] loop */

  /* Get first character, the pattern for new recmatch calls follows */
  c = *p++;

  /* If that was the end of the pattern, match if string empty too */
  if (c == 0)
    return *s == 0;

  /* '?' matches any character (but not an empty string) */
  if (c == '?')
    return *s ? recmatch(p, s + 1) : 0;

  /* '*' matches any number of characters, including no char's! */
  /* EWE: todo: for MS-DOS/Win make sure it won't match period! */
  if (c == '*')
  {
    if (*p == 0)
      return 1;
    for ( ; *s; s++)
      if ((c = recmatch(p, s)) != 0)
        return (int)c;
    return 2;           /* 2 means give up--shmatch will return false */
  }

  /* If escape ('\'), just compare next character */
  if (c == '\\')
    if ((c = *p++) == 0)        /* if \ at end, then syntax error */
      return 0;

  /* Just a character--compare it */
  return case_map(c) == case_map(*s) ? recmatch(p, ++s) : 0;
}


/* Compare the sh pattern p with the string s and return true(1) 
   if they match, false(0) if they don't or if there is a syntax
   error in the pattern. */
/* sample args:  p=*.*(fspec user wants)  s=TEST.DAT(next file in dir) */
int shmatch(p, s)
char *p;                /* sh pattern to match */
char *s;                /* string to match it to */
{
  int ret;

  ret=recmatch((uch *) p, (uch *) s) == 1;
  sprintf(ewemsg,"in shmatch of util.c, pat=%s, str=%s, return=%d", p,s,ret);
  diag(ewemsg);
  return(ret);
}


/* Break the pattern and string into name and extension parts and match
   each separately using shmatch(). */
int dosmatch(p, s)
char *p;                /* dos pattern to match */
char *s;                /* string to match it to */
{
  char *p1, *p2;        /* pattern sections */
  char *s1, *s2;        /* string sections */
  int plen = strlen(p); /* length of pattern */
  int r, r1, r2;        /* result */

  sprintf(ewemsg,"in dosmatch of util.c, dos pattern=%s, string=%s",p,s);
  diag(ewemsg);

  if ((p1 = malloc(plen + 1)) == NULL ||
      (s1 = malloc(strlen(s) + 1)) == NULL)
  {
    if (p1 != NULL)
      free((zvoid *)p1);
    return 0;
  }
  strcpy(p1, p);  /* pattern to match */
  strcpy(s1, s);  /* string to match */

  /* is there a period in pattern? */
  if ((p2 = strrchr(p1, '.')) != NULL)
    *p2++ = '\0';  /* yes - wipe out extension */

  else if (plen && p1[plen - 1] == '*')  // if fname part ended in *
    p2 = "*";      /* put a zero in place of extension */

  else
    p2 = "";       /* no wild extension */

  /* is there a period in string? */
  if ((s2 = strrchr(s1, '.')) != NULL)
    *s2++ = '\0';
  else
    s2 = "";

  r1 = shmatch(p2, s2);
  sprintf(ewemsg,"in dosmatch of util.c, shmatch 1 pattern=%s, string=%s result=%d",p2,s2,r1);
  diag(ewemsg);

  r2 = shmatch(p1, s1);
  sprintf(ewemsg,"in dosmatch of util.c, shmatch 2 pattern=%s, string=%s result=%d",p1,s1,r2);
  diag(ewemsg);

  r = r1 && r2;

  free((zvoid *)p1);
  free((zvoid *)s1);
  return r;
}


/* Search for b in the pointer list a[0..n-1] using the compare function
   cmp(b, c) where c is an element of a[i] and cmp() returns negative if
   *b < *c, zero if *b == *c, or positive if *b > *c.  If *b is found,
   search returns a pointer to the entry in a[], else search() returns
   NULL.  The nature and size of *b and *c (they can be different) are
   left up to the cmp() function.  A binary search is used, and it is
   assumed that the list is sorted in ascending order. */
zvoid far **search(b, a, n, cmp)
zvoid *b;               /* pointer to value to search for */
zvoid far **a;          /* table of pointers to values, sorted */
extent n;               /* number of pointers in a[] */
int (*cmp) OF((const zvoid *, const zvoid far *)); /* comparison function */
{
  zvoid far **i;        /* pointer to midpoint of current range */
  zvoid far **l;        /* pointer to lower end of current range */
  int r;                /* result of (*cmp)() call */
  zvoid far **u;        /* pointer to upper end of current range */

  l = (zvoid far **)a;  u = l + (n-1);
  while (u >= l) {
    i = l + ((unsigned)(u - l) >> 1);
    if ((r = (*cmp)(b, (char *)*(struct zlist **)i)) < 0)
      u = i - 1;
    else if (r > 0)
      l = i + 1;
    else
      return (zvoid far **)i;
  }
  return NULL;          /* If b were in list, it would belong at l */
}
#endif /* !UTIL */


void init_upper()
{
  int c;
  for (c = 0; c < sizeof(upper); c++) upper[c] = lower[c] = c;
  for (c = 'a'; c <= 'z';        c++) upper[c] = c - 'a' + 'A';
  for (c = 'A'; c <= 'Z';        c++) lower[c] = c - 'A' + 'a';
}

/* Compare the two strings ignoring case, and correctly taking into
 * account national language characters. 
 */
int namecmp(string1, string2)
  char *string1, *string2;
{
  int d;

  for (;;)
  {
    d = (int) (uch) case_map(*string1)
      - (int) (uch) case_map(*string2);

    if (d || *string1 == 0 || *string2 == 0)
      return d;

    string1++;
    string2++;
  }
}
