unit FormFnp;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs, Registry,
  FnpFormProp;

type
  TFnpForm = class(TForm)
  private
    { Private declarations }
    PositionRead: Boolean;
    FAllowMaximized: Boolean;
    FAllowMinimized: Boolean;
    FMinSizeX: Integer;
    FMinSizeY: Integer;
    FMaxSizeX: Integer;
    FMaxSizeY: Integer;
    FRootKey: HKEY;
    FSavePosition: Boolean;
    FSubKey: String;
    FVersion: String;
    procedure DoShow; override;
    procedure WMGetMinMaxInfo(var Msg: TWMGetMinMaxInfo); message WM_GETMINMAXINFO;
    procedure SetAllowMaximized(Value: Boolean);
    procedure SetAllowMinimized(Value: Boolean);
    procedure SetMinSizeX(Value: Integer);
    procedure SetMinSizeY(Value: Integer);
    procedure SetMaxSizeX(Value: Integer);
    procedure SetMaxSizeY(Value: Integer);
    procedure SetRootKey(Value: HKEY);
    procedure SetSavePosition(Value: Boolean);
    procedure SetSubKey(Value: String);
    procedure DoSavePosition;
    procedure DoReadPosition;
  public
    { Public declarations }
    property AllowMaximized: Boolean read FAllowMaximized write SetAllowMaximized;
    property AllowMinimized: Boolean read FAllowMinimized write SetAllowMinimized;
    property MinSizeX: Integer read FMinSizeX write SetMinSizeX;
    property MinSizeY: Integer read FMinSizeY write SetMinSizeY;
    property MaxSizeX: Integer read FMaxSizeX write SetMaxSizeX;
    property MaxSizeY: Integer read FMaxSizeY write SetMaxSizeY;
    property RootKey: HKEY read FRootKey write SetRootKey default HKEY_CURRENT_USER;
    property SavePosition: Boolean read FSavePosition write SetSavePosition;
    property SubKey: String read FSubKey write SetSubKey;
    property Version: String read FVersion;
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
  end;

var
  FnpForm: TFnpForm;

implementation

{$R *.DFM}

constructor TFnpForm.Create(AOwner: TComponent);
var
  I: Integer ;
  FFP: TFnpFormProp;
begin
  inherited Create(AOwner);
  FFP := Nil;
  { Does this form contain a TFnpFormProp component? }
  For I := 0 to ComponentCount - 1 do
    if Components[I].ClassName = 'TFnpFormProp' then
    begin
      FFP := TFnpFormProp(Components[I]);
      Break;
    end;

  if Assigned(FFP) then
    { Transfer properties from TFnpFormProp }
    begin
      FAllowMaximized := FFP.AllowMaximized;
      FAllowMinimized := FFP.AllowMinimized;
      FMaxSizeX := FFP.MaxSizeX;
      FMaxSizeY := FFP.MaxSizeY;
      FMinSizeX := FFP.MinSizeX;
      FMinSizeY := FFP.MinSizeY;
      if FFP.RootKey = rkCurrentUser then
        FRootKey := HKEY_CURRENT_USER
      else
        FRootKey := HKEY_LOCAL_MACHINE;
      FSavePosition := FFP.SavePosition;
      FSubKey := FFP.SubKey;
      { Don't need it anymore! }
      FFP.Free;
      { We need to handle MDI-child forms specially since DoShow is called
        in the original constructor! Why? }
      if FormStyle = fsMDIChild	then
      begin
        DoReadPosition;
        PositionRead := True;
      end;
    end
  else
    { Set default properties }
    begin
      FAllowMaximized := True;
      FAllowMinimized := True;
      FRootKey := HKEY_CURRENT_USER;
    end;

  { Set version }
  FVersion := '1.00.00';

end;

destructor TFnpForm.Destroy;
begin
  if FSavePosition then
    DoSavePosition;
  inherited;
end;

procedure TFnpForm.DoShow;
begin
  inherited DoShow;
  { Do not handle MDI-child forms here, since OnShow is called in the
    original contructor! Why?}
  if FormStyle <> fsMDIChild	then
    if not PositionRead then
    begin
      { Read saved position? }
      if FSavePosition then
        DoReadPosition;
      PositionRead := True;
    end;
end;

procedure TFnpForm.WMGetMinMaxInfo(var Msg: TWMGetMinMaxInfo);
begin
  if BorderStyle = bsSizeable then
  begin
    if FMinSizeX > 0 then
      Msg.MinMaxInfo^.ptMinTrackSize.X := FMinSizeX;
    if FMinSizeY > 0 then
      Msg.MinMaxInfo^.ptMinTrackSize.Y := FMinSizeY;
    if FMaxSizeX > 0 then
      Msg.MinMaxInfo^.ptMaxTrackSize.X := FMaxSizeX;
    if FMaxSizeY > 0 then
      Msg.MinMaxInfo^.ptMaxTrackSize.Y := FMaxSizeY;
  end;
end;

procedure TFnpForm.SetAllowMaximized(Value: Boolean);
begin
  if FAllowMaximized <> Value then
    FAllowMaximized := Value;
end;

procedure TFnpForm.SetAllowMinimized(Value: Boolean);
begin
  if FAllowMinimized <> Value then
    FAllowMinimized := Value;
end;

procedure TFnpForm.SetMinSizeX(Value: Integer);
begin
  if FMinSizeX <> Value then
    if Value < 0 then
      FMinSizeX := 0
    else
      FMinSizeX := Value;
end;

procedure TFnpForm.SetMinSizeY(Value: Integer);
begin
  if FMinSizeY <> Value then
    if Value < 0 then
      FMinSizeY := 0
    else
      FMinSizeY := Value;
end;

procedure TFnpForm.SetMaxSizeX(Value: Integer);
begin
  if FMaxSizeX <> Value then
    if Value < 0 then
      FMaxSizeX := 0
    else
      FMaxSizeX := Value;
end;

procedure TFnpForm.SetMaxSizeY(Value: Integer);
begin
  if FMaxSizeY <> Value then
    if Value < 0 then
      FMaxSizeY := 0
    else
      FMaxSizeY := Value;
end;

procedure TFnpForm.SetRootKey(Value: HKEY);
begin
  if FRootKey <> Value then
    if (Value = HKEY_CURRENT_USER) or (Value = HKEY_LOCAL_MACHINE) then
      FRootKey := Value;
end;

procedure TFnpForm.SetSavePosition(Value: Boolean);
begin
  if FSavePosition <> Value then
    FSavePosition := Value;
end;

procedure TFnpForm.SetSubKey(Value: String);
begin
  if FSubKey <> Value then
    FSubKey := Value;
end;

procedure TFnpForm.DoReadPosition;
var
  Reg: TRegIniFile;
  Wp: TWINDOWPLACEMENT;
  WinState: TWindowState;
begin
  if Length(FSubKey) > 0 then
  begin
    try
      Reg := TRegIniFile.Create(FSubKey);
      Reg.RootKey := FRootKey;
      Wp.Length := SizeOf(Wp);
      { Is position saved? }
      if Reg.ReadInteger(TForm(Self).Name, 'Left', -9999) <> -9999 then
        begin
          GetWindowPlacement(Handle, @Wp);
          WinState := TWindowState(Reg.ReadInteger(TForm(Self).Name, 'WindowState', 0));
          if not FAllowMaximized and (WinState = wsMaximized) then
            WinState := wsNormal
          else if not FAllowMinimized and (WinState = wsMinimized) then
            WinState := wsNormal;
          Wp.rcNormalPosition.Left := Reg.ReadInteger(TForm(Self).Name, 'Left', Left);
          Wp.rcNormalPosition.Top := Reg.ReadInteger(TForm(Self).Name, 'Top', Top);
          Wp.rcNormalPosition.Right := Reg.ReadInteger(TForm(Self).Name, 'Right', Left + Width);
          Wp.rcNormalPosition.Bottom := Reg.ReadInteger(TForm(Self).Name, 'Bottom', Top + Height);
          case WinState of
            wsNormal:
              Wp.ShowCmd := SW_SHOWNORMAL;
            wsMinimized:
              Wp.ShowCmd := SW_SHOWMINIMIZED;
            wsMaximized:
              Wp.ShowCmd := SW_SHOWMAXIMIZED;
          end;
          SetWindowPlacement(Handle, @Wp);
        end;
    finally
      Reg.Free;
    end;
  end;
end;

procedure TFnpForm.DoSavePosition;
var
  Reg: TRegIniFile;
  Wp: TWINDOWPLACEMENT;
  WinState: TWindowState;
begin
  if Length(FSubKey) > 0 then
  begin
    try
      Reg := TRegIniFile.Create(FSubKey);
      Reg.RootKey := FRootKey;
      Wp.Length := SizeOf(Wp);
      GetWindowPlacement(Handle, @Wp);
      Reg.WriteInteger(TForm(Self).Name, 'Left', Wp.rcNormalPosition.Left);
      Reg.WriteInteger(TForm(Self).Name, 'Top', Wp.rcNormalPosition.Top);
      Reg.WriteInteger(TForm(Self).Name, 'Right', Wp.rcNormalPosition.Right);
      Reg.WriteInteger(TForm(Self).Name, 'Bottom', Wp.rcNormalPosition.Bottom);
      case Wp.ShowCmd of
        SW_SHOWNORMAL:
          WinState := wsNormal;
        SW_SHOWMINIMIZED:
          WinState := wsMinimized;
        SW_SHOWMAXIMIZED:
          WinState := wsMaximized;
      end;
      Reg.WriteInteger(TForm(Self).Name, 'WindowState', Integer(WinState));
    finally
      Reg.Free;
    end;
  end;
end;

end.
