Unit LZRW1;

{*****************************************************************************
*
*TLZRW1 file compression component.
*----------------------------------
*
*Compresses a file with :
*------------------------
*
*    either the LZRW1/KH or LZH compression algorithm,
*           with code posted by Kurt Haenen on the SWAG (lzrw1).
*    or the Japanese LZH compression algorithm
*           ( LZSS coded by Haruhiko OKUMURA
*             Adaptive Huffman Coding coded by Haruyasu YOSHIZAKI
*             Edited and translated to English by Kenji RIKITAKE
*             Translated from C to Turbo Pascal by Douglas Webb   2/18/91
*             posted by Doug Webb on the SWAG (preskit2\lzh).)
*
*

*Visual feedback on a Panel if so desired.

*

*All VCL code by D. Heijl , may 8-9 1996

*

*The Getblock/PutBlock procedures are based on the code in

*lzhtest.pas by Douglas Webb.

*

*

*The files lzh.pas and lzrw1kh.pas are essentially untouched

*(only some cosmetic changes, also added exceptions)

*

*----------------------------------------------------------------

*
* Feel free to use or give away this software as you see fit.
* Just leave the credits in place if you alter the source.
*
* This software is delivered to you "as is",
* no guarantees, it may blow up or trigger World War Three
* for all I know.
*
* If you find any bugs and let me know, I will try to fix them.
*
* I believe in programmers around the world helping each other
* out by distributing free source code.
*
*Danny Heijl, may 10 1996.

*Danny.Heijl@cevi.be
*

*----------------------------------------------------------------

*****************************************************************}

interface

uses SysUtils, WinTypes, WinProcs,  Classes, ExtCtrls, Controls, Forms,
     Graphics, Menus,
     Lzrw1kh, Lzh;


{$IFDEF WIN32}
type
    Int16   = SmallInt;
    SString = ShortString;
{$ELSE}
type
    Int16   = Integer;
{$ENDIF}

type
     ELzrw1Exception = class(Exception);
     TCompressMode = (Good, Fast, Auto);

type
  Tlzrw1 = class(TCustomPanel)

  private
    FIn  : String;
    FOut : String;
    FCompressMode : TCompressMode;

    procedure CheckWrite(Actual, Desired : Longint);
    procedure CheckRead(Actual, Desired : Longint);

    procedure GetBlock(VAR Target; NoBytes:Word; VAR Actual_Bytes:Word);
    procedure PutBlock(VAR Source; NoBytes:Word; VAR Actual_Bytes:Word);

    procedure LZrw1Compress;
    procedure LZrw1Decompress;
    procedure LZHCompress;
    procedure LZHDecompress;

  protected
    Function CompressFile   : Longint;
    Function DeCompressFile : Longint;
    function GetBestMode    : TcompressMode;
    procedure Loaded        ; override;

  public
    constructor Create(AOwner : TComponent); override;

  published
    Function Compress     : LongInt;
    Function Decompress   : Longint;
    Function Advise       : TcompressMode;
    property InputFile    : String read FIn  write FIn;
    property OutputFile   : String read FOut write FOut;
    property CompressMode : TCompressMode read  FCompressMode
                                          write FcompressMode default Good;
    property Align;
    property Alignment;
    property BevelInner;
    property BevelOuter;
    property BevelWidth;
    property BorderWidth;
    property BorderStyle;
    property DragCursor;
    property DragMode;
    property Enabled;
    {property Caption; }
    property Color;
    property Ctl3D;
    property Font;
    property Locked;
    property ParentColor;
    property ParentCtl3D;
    property ParentFont;
    property ParentShowHint;
    property PopupMenu;
    property ShowHint;
    property TabOrder;
    property TabStop;
    property Visible;
    property OnClick;
    property OnDblClick;
    property OnDragDrop;
    property OnDragOver;
    property OnEndDrag;
    property OnEnter;
    property OnExit;
    property OnMouseDown;
    property OnMouseMove;
    property OnMouseUp;
    property OnResize;

end;

procedure Register;

implementation

CONST
  LZRWIdentifier : LONGINT =
  ((((((ORD('L') SHL 8)+ORD('Z')) SHL 8)+ORD('R')) SHL 8)+ORD('W'));

  LZHIdentifier : LONGINT =
  ((((((ORD('L') SHL 8)+ORD('Z')) SHL 8)+ORD('H')) SHL 8)+ORD('!'));

  ChunkSize = 32768;
  IOBufSize = (ChunkSize + 16);

type

  LZHBuf  = Array[1..ChunkSize] OF BYTE;
  PLZHBuf = ^LZHBuf;

VAR
  LZHInBuf, LZHOutBuf : PLZHBuf;
  SRCBuf,DSTBuf       : BufferPtr;    { defined in LZRW1KH }

  SrcFh, DstFh        : Integer;
  SRCSize,DSTSize     : LongInt;

  Tmp                 : Longint;
  Identifier          : LONGINT;
  CompIdentifier      : LONGINT;
  InSize,OutSize      : LONGINT;

  Size : Longint;

  Buf : Longint; { getblock }
  PosnR : Word;  { getblock }
  PosnW : Word;  { putblock }


  ReadProc : TreadProc;     { must be passed to LZHPACK/UNPACK }

  WriteProc : TWriteProc;   { must be passed to LZHPACK/UNPACK }


procedure Register;
begin
  RegisterComponents('Compon',[Tlzrw1]);
end;

constructor TLzrw1.Create(AOwner : TComponent);
begin
  inherited Create(AOwner);
                            { initialize defaults }
  FcompressMode := Good;
end;

procedure TLzrw1.Loaded;
begin
  inherited Loaded;
                             { the caption is fixed }
  if (CsDesigning in ComponentState) then begin
    Caption := 'LZRW1/KH File Compressor/Decompressor';
  end else begin
    Caption := '';
  end;
end;

           { the 2 execute methods : compress and decompress }
           {-------------------------------------------------}

function TLzrw1.Compress : Longint;
begin
  Result := CompressFile;
end;

function TLzrw1.DeCompress : Longint;
begin
  Result := DeCompressFile;
end;

          { the 3d execute method : advise compression method }
          {---------------------------------------------------}

function TLzrw1.Advise : TcompressMode;
begin
  Result := GetBestMode;
end;

          { some common subroutine functions }
          {----------------------------------}

{ Check if Write was successfull, raise an exception if not }
procedure TLzrw1.CheckWrite(Actual, Desired : Longint);
begin
  if (Actual <> Desired) then begin
    Raise ELzrw1Exception.Create('Lzrw1 : ERROR WRITING TO ' + Fout + ' !');
  end;
  Application.ProcessMessages;
end;



{ check if Read was successfull, raise an exception if not }
procedure TLzrw1.CheckRead(Actual, Desired : Longint);
begin
  if (Actual <> Desired) then begin
    Raise ELzrw1Exception.Create('Lzrw1 : ERROR READING FROM ' + Fin + ' !');
  end;
  Application.ProcessMessages;
end;

          { the LZH reader and writer procedures }
          {--------------------------------------}

          { the reader : GetBlock }

Procedure TLzrw1.GetBlock(VAR Target; NoBytes:Word; VAR Actual_Bytes:Word);

BEGIN
  IF (PosnR > Buf) OR (PosnR + NoBytes > SUCC(Buf)) THEN BEGIN
    IF PosnR > Buf THEN BEGIN
       Buf := FileRead(SrcFh,LZHInBuf^,ChunkSize);
       if (Buf < 0) then begin
         Raise ELzrw1Exception.Create('Lzrw1/LZH : READ ERROR ON ' + Fin + ' !');
       end;
       Application.ProcessMessages;
       INC(InSize,Buf);
       if Visible then begin
         Caption := '(LZH) READ : ' + IntTostr(Insize) + ' WRITTEN : ' +
                 IntTostr(OutSize);
         Update;
       end;
    END
    ELSE BEGIN
      Move(LZHInBuf^[PosnR],LZHInBuf^[1],Buf - PosnR);
      Tmp := FileRead(SrcFh,LZHInBuf^[Buf-PosnR],ChunkSize - (Buf - PosnR));
      if (Tmp < 0) then begin
        Raise ELzrw1Exception.Create('Lzrw1/LZH : READ ERROR ON ' + Fin + ' !');
      end;
      Application.ProcessMessages;
      INC(InSize,Tmp);
      if Visible then begin
        Caption := '(LZH) READ : ' + IntTostr(Insize) + ' WRITTEN : ' +
                IntTostr(OutSize);
        Update;
      end;
      Buf := Buf - PosnR + Tmp;
    END;
    IF Buf = 0 THEN BEGIN
       Actual_Bytes := 0;
       Exit;
    END;
    PosnR := 1;
  END;

  Move(LZHInBuf^[PosnR],Target,NoBytes);
  INC(PosnR,NoBytes);

  IF PosnR > SUCC(Buf) THEN
    Actual_Bytes := NoBytes - (PosnR - SUCC(Buf))
  ELSE
    Actual_Bytes := NoBytes;

END;

          { and the writer : PutBlock }

Procedure TLzrw1.PutBlock(VAR Source; NoBytes:Word; VAR Actual_Bytes:Word);

BEGIN
  If NoBytes = 0 THEN begin   { Flush condition }
    Tmp := FileWrite(DstFh,LZHOutBuf^,PRED(PosnW));
    CheckWrite(Tmp, PRED(PosnW));
    Inc(OutSize, Tmp);
    if Visible then begin
      Caption := '(LZH) READ : ' + IntTostr(Insize) +
              ' WRITTEN : ' + IntToStr(OutSize);
      Update;
    end;
    EXIT;
  END;
  IF (PosnW > ChunkSize) OR (PosnW + NoBytes > SUCC(ChunkSize)) THEN BEGIN
    Tmp := FileWrite(DstFh,LZHOutBuf^,PRED(PosnW));
    CheckWrite(Tmp, PRED(PosnW));
    Inc(OutSize, Tmp);
    PosnW := 1;
    if Visible then begin
      Caption := '(LZH) READ : ' + IntTostr(Insize) +
              ' WRITTEN : ' + IntToStr(OutSize);
      Update;
    end;
  END;
  Move(Source,LZHOUTBuf^[PosnW],NoBytes);
  INC(PosnW,NoBytes);
  Actual_Bytes := NoBytes;
END;

                { compress a file with LZRW1/KH (FAST) }
                {--------------------------------------}

Procedure TLzrw1.LZRW1Compress;
begin                                           { start compressing }
  SRCSize := ChunkSize;
  InSize := 0;
  WHILE (SRCSize = ChunkSize) DO BEGIN
                                      { read a block af data }
    SrcSize := FileRead(SrcFh, SrcBuf^, ChunkSize);
    if (SrcSize < 0) then begin
      Raise ELzrw1Exception.Create('Lzrw1/KH : READ ERROR ON ' + Fin + ' !');
    end;
    Application.ProcessMessages;
    INC(InSize,SRCSize);
                                      { compress it }
    DSTSize := Compression(SRCBuf,DSTBuf,SRCSize);
                                      { write out compressed size }
    Tmp := FileWrite(DstFh, DstSize, SizeOf(Word));
    CheckWrite(Tmp, Sizeof(Word));
    INC(OutSize,Tmp);
                                       { write out compressed data }
    Tmp := FileWrite(DstFh, DstBuf^, DstSize);
    CheckWrite(Tmp, DstSize);
    INC(OutSize,Tmp);
    if Visible then begin
      Caption := '(LZRW1/KH) READ : ' + IntToStr(InSize) +
                 '  WRITTEN : ' + IntToStr(OutSize);
      Update;
    end;

  END;       { endwhile SRCSize = ChunkSize }
end;

                { compress a file with LZH (GOOD) }
                {---------------------------------}

procedure TLzrw1.LZHCompress;
var
  Bytes_Written : Longint;
  Temp          : Word;

begin

  ReadProc := GetBlock;
  WriteProc := PutBlock;

                    { initialize put/getblock variables }
  Buf := 0;
  PosnR := 1;
  PosnW := 1;
                    { pack the file with LZH }
  LZHPack(Bytes_written, ReadProc, WriteProc);

                    { flush last buffer }
  PutBlock(Size, 0, Temp);

end;

                { decompress a file with LZRW1 (FAST) }
                {-------------------------------------}

procedure TLzrw1.LZRW1Decompress;
var
  OrigSize : Longint;

begin
                    { read in uncompressed filesize }
  Tmp := FileRead(SrcFh, OrigSize, sizeof(Longint));
  CheckRead(Tmp, Sizeof(Longint));
  Inc(InSize,SIZEOF(LONGINT));
                                         { start decompression }
  WHILE (DSTSize = ChunkSize) DO BEGIN
                                         { read size of compressed block }
    Tmp := FileRead(SrcFh, SrcSize, SizeOf(Word));
    CheckRead(Tmp, Sizeof(Word));
                                          { read compressed block }
    Tmp := FileRead(SrcFh, SrcBuf^, SrcSize);
    Checkread(Tmp, SrcSize);
    INC(InSize,Tmp + SIZEOF(WORD));
                                           { decompress block }
    DSTSize := Decompression(SRCBuf,DstBuf,SRCSize);
                                           { write it out }
    Tmp := FileWrite(DstFh, DstBuf^, DstSize);
    CheckWrite(Tmp, DstSize);
    INC(OutSize,Tmp);
    if Visible then begin
      Caption := '(LZRW1/KH) READ : ' + IntToStr(InSize) +
                 '  WRITTEN : ' + IntToStr(OutSize);
      Update;
    end;
  END;   { endwhile data to read }

  If (OutSize <> OrigSize) then begin
    Raise ELzrw1Exception.Create('Lzrw1/KH : Sizes do not match !! Corruption');
  end;


end;

                { decompress a file with LZH (GOOD) }
                {-----------------------------------}


procedure TLzrw1.LZHDecompress;
var
  OrigSize      : Longint;
  Temp          : Word;

begin

  ReadProc := GetBlock;
  WriteProc := PutBlock;

                    { read in uncompressed filesize }
  Tmp := FileRead(SrcFh, OrigSize, sizeof(Longint));
  CheckRead(Tmp, Sizeof(Longint));
  Inc(InSize,SIZEOF(LONGINT));

                    { initialize put/getblock variables }
  PosnR := 1;
  Buf   := 0;
  PosnW := 1;
                    { unpack the file with LZH }
  LZHUnPack(OrigSize, ReadProc, WriteProc);

                    { flush last buffer }
  PutBlock(Size, 0, Temp);

  If (OutSize <> OrigSize) then begin
    Raise ELzrw1Exception.Create('Lzrw1/LZH : Sizes do not match !! Corruption');
  end;

end;

                { the main code common to both (de)compression methods  }
                {-------------------------------------------------------}

                { compress a file }
                {-----------------}

function TLzrw1.CompressFile : Longint;
var
  Infile : File;
  Mode : TcompressMode;

BEGIN

  Result := -1;

  If (FcompressMode = Auto) then
    Mode := GetBestMode
  else
    Mode := FcompressMode;

  try
    Getmem(SRCBuf, IOBufSize);
    Getmem(DSTBuf, IOBufSize);
    LZHInBuf := PLZHBuf(SRCBuf);
    LZHOutBuf := PLZHBuf(DSTBuf);
  except
    Raise ELzrw1Exception.Create('Lzrw1 : Cannot get memory for I/O Buffers');
    exit;
  end;


  if (Mode = Fast) then
    CompIdentifier := LZRWIdentifier
  else
    CompIdentifier := LZHIdentifier;

  try
    SrcFh := 0; DstFh := 0;
    try                    { need the filesize to start, get it }
      AssignFile(Infile, Fin);
      Reset(Infile,1);
      try
        Size := FileSize(Infile);
      finally;
        CloseFile(Infile);
      end;
    except
      Raise ELzrw1Exception.Create('Lzrw1 : Cannot obtain size of ' + Fin);
    end;

    try                    { tyry to open the files }
      SrcFh := FileOpen(Fin,fmOpenRead);
      if (SrcFh < 0) then begin
        Raise ELzrw1Exception.Create('LZRWKH1 : Inputfile '
              + Fin + ' not found !');
      end;
      DstFh := FileCreate(Fout);
      if (DstFh < 0) then begin
        Raise ELzrw1Exception.Create('Lzrw1 : Outputfile ' + Fout +
              ' can not be opened !');
      end;

      try               { try to compress the file }
                        { write out compression ID }
        Tmp := FileWrite(DstFh, CompIdentifier, sizeof(Longint));
        CheckWrite(Tmp, Sizeof(Longint));
        OutSize := SIZEOF(LONGINT);
        InSize := 0;
                          { write out uncompressed filesize }
        Tmp := FileWrite(DstFh, Size, sizeof(Longint));
        CheckWrite(Tmp, Sizeof(Longint));
        Inc(OutSize,SIZEOF(LONGINT));

        if (Mode = Fast) then
          LZRW1Compress
        else
          LZHCompress;

      except                  { error while compressing }
        on Exception do begin
          FileClose(DstFH); DstFH := 0;
                              { get rid of output file }
          {$IFDEF WIN32}             { bug in Delphi 2.0 WINAPI }
          DeleteFile(PChar(Fout));
          {$ELSE}
          DeleteFile(Fout);
          {$ENDIF}
          Raise;              { and reraise to inform user }
        end;
      end;

    finally
      if (SrcFh > 0) then FileClose(Srcfh);
      if (DstFh > 0) then FileClose(DstFh);
    end;

    if Visible then begin
      Caption := Fin + ' SUCCESFULLY COMPRESSED (' +
              IntToStr((Outsize * 100) div Insize) + '%)';
      Update;
    end;

  finally
    Freemem(SRCBuf,IOBufSize);
    Freemem(DSTBuf,IOBufSize);
  end;

  Result := OutSize;

end;


                { decompress a file }
                {-------------------}

function TLzrw1.DeCompressFile : Longint;

begin

  Result := -1;

  try
    Getmem(SRCBuf, IOBufSize);
    Getmem(DSTBuf, IOBufSize);
    LZHInBuf := PLZHBuf(SRCBuf);
    LZHOutBuf := PLZHBuf(DSTBuf);
  except
    Raise ELzrw1Exception.Create('Lzrw1 : no memory for I/O Buffers !!');
    exit;
  end;

  try
    SrcFh := 0; DstFh := 0;

    try
      SrcFh := FileOpen(Fin,fmOpenRead);
      if (SrcFh < 0) then begin
        Raise ELzrw1Exception.Create('Lzrw1 : Inputfile ' +
                                               Fin + ' not found !');
      end;
      DstFh := FileCreate(Fout);
      if (DstFh < 0) then begin
        Raise ELzrw1Exception.Create('Lzrw1 : Outputfile ' +
                                           Fout + ' can not be opened !');
      end;

      try
                                          { read compression ID }
        Tmp := FileRead(SrcFh, Identifier, Sizeof(Longint));
        CheckRead(Tmp, Sizeof(Longint));
        IF (Identifier <> LZRWIdentifier)
             and (Identifier <> LZHIdentifier) THEN BEGIN
          Raise ELzrw1Exception.Create('Lzrw1 : ' +
                                       Fin + ' : NOT A COMPRESSED FILE !');
        END;
        DSTSize := ChunkSize;
        InSize := SIZEOF(LONGINT);
        OutSize := 0;

        if (Identifier = LZRWIdentifier) then
          LZRW1Decompress
        else
          LZHDecompress;

        if Visible then begin
          Caption := Fin + ' SUCCESFULLY DECOMPRESSED (' +
                  IntToStr((Outsize * 100) div Insize) + '%)';
          Update;
        end;

      except
        on Exception do begin
          FileClose(DstFH); DstFH := 0;
                              { get rid of output file }
          {$IFDEF WIN32}             { bug in Delphi 2.0 WINAPI }
          DeleteFile(PChar(Fout));
          {$ELSE}
          DeleteFile(Fout);
          {$ENDIF}
          Raise;
        end;
      end;

    finally
      if (SrcFh > 0) then FileClose(SrcFh);
      if (DstFh > 0) then FileClose(DstFh);
    end;

  finally
    Freemem(SRCBuf,IOBufSize);
    Freemem(DSTBuf,IOBufSize);
  end;

  Result := OutSize;

end;

                { Guess the best compression mode }
                { returns Good or Fast }
function TLzrw1.GetBestMode: TCompressMode;
var
  Infile : File;
  CompressedSize : Longint;
  UncompressedSize : Longint;


  begin


  Result := Good;

  Caption := 'Guessing best compression for : ' + Fin;

  try
    Getmem(SRCBuf, IOBufSize);
    Getmem(DSTBuf, IOBufSize);
  except
    Raise ELzrw1Exception.Create('Lzrw1 : Cannot get memory for I/O Buffers');
    exit;
  end;

   try
    SrcFh := 0;
    try                    { need the filesize to start, get it }
      AssignFile(Infile, Fin);
      Reset(Infile,1);
      try
        Size := FileSize(Infile);
      finally;
        CloseFile(Infile);
      end;
    except
      Raise ELzrw1Exception.Create('Lzrw1 : Cannot obtain size of ' + Fin);
    end;

    try                    { try to open the inputfile }
      SrcFh := FileOpen(Fin,fmOpenRead);
      if (SrcFh < 0) then begin
        Raise ELzrw1Exception.Create('LZRWKH1 : Inputfile '
              + Fin + ' not found !');
      end;
                           { small files can afford LZH }
      if (Size < (3 * ChunkSize)) then begin
        FCompressMode := Good;
        Result := Good;
        exit;
      end;
                           { try 2 blocks with fast at 1/3 and 2/3 of file }
      FileSeek(Srcfh, (Size div 3) and $7FFF8000, 0);
      SrcSize := FileRead(SrcFh, SrcBuf^, ChunkSize);
      if (SrcSize < 0) then begin
        Raise ELzrw1Exception.Create('Lzrw1/KH : READ ERROR ON ' + Fin + ' !');
      end;
      UncompressedSize := SrcSize;
      Application.ProcessMessages;
      CompressedSize := Compression(SRCBuf,DSTBuf,SrcSize);
      FileSeek(Srcfh, ((Size * 2)div 3) and $7FFF8000, 0);
      SrcSize := FileRead(SrcFh, SrcBuf^, ChunkSize);
      if (SrcSize < 0) then begin
        Raise ELzrw1Exception.Create('Lzrw1/KH : READ ERROR ON ' + Fin + ' !');
      end;
      Inc(UncompressedSize, SrcSize);
      Application.ProcessMessages;
      Inc(CompressedSize,Compression(SRCBuf,DSTBuf,SRCSize));

      if (((UnCompressedSize * 40) div 100) > CompressedSize) then begin
        Result := Fast;
        Caption := 'My guess : Fast (' + IntToStr((CompressedSize * 100)
                               div UncompressedSize) + '%)';
      end
      else begin
        Result := Good;
        Caption := 'My guess : Good (' + IntToStr((CompressedSize * 100)
                               div UncompressedSize) + '%)';
      end;

    finally
      if (SrcFh > 0) then FileClose(Srcfh);
    end;

  finally
    Freemem(SRCBuf,IOBufSize);
    Freemem(DSTBuf,IOBufSize);
  end;

end;

end.
