{*******************************************************}
{                                                       }
{       Ole Automation Controller for Delphi 1.0        }
{       by Roger Headrick (Compserv 76114,1430)         }
{       contributed to public domain 12/23/95           }
{                                                       }
{*******************************************************}
unit Olectlr;

interface

uses
  SysUtils, WinTypes, WinProcs, Messages, Classes, Graphics, Controls,
  Forms, Dialogs, Ole2, Dispatch;

type

  { The IDispatch class in Borland's Dispatch unit is redeclared here
    to correct the Invoke method decaration.}
  IDispatch = class(IUnknown)
    function GetTypeInfoCount(var pctinfo: Integer): HResult; virtual; cdecl; export; abstract;
    function GetTypeInfo(itinfo: Integer; lcid: LCID; var pptinfo: ITypeInfo): HResult; virtual; cdecl; export; abstract;
    function GetIDsOfNames(const riid: IID; var rgszNames: PChar;
      cNames: Integer; lcid: LCID; var rgdispid: DISPID): HResult; virtual; cdecl; export; abstract;
    function Invoke(dispidMember: DISPID; const riid: IID; lcid: LCID;
      wFlags: Word; var pdispparams: DISPPARAMS; pvarResult: PVARIANT;
      var pexcepinfo: EXCEPINFO; var puArgErr: Integer): HResult; virtual; cdecl; export; abstract;
  end;


type
  TPasType = VARTYPE;
  TCurrency = CY;
  PSingle = ^Single;
  PDouble = ^Double;
  PWordBool = ^WordBool;
  PCurrency = ^TCurrency;
  PIDispatch = ^IDispatch;
  PPSAFEARRAY = ^PSAFEARRAY;
  PBStr = ^BStr;

const
  { map OLE types to Pascal types}
  PAS_BYTE           = VT_UI1;
  PAS_INTEGER        = VT_I2;
  PAS_LONGINT        = VT_I4;
  PAS_SINGLE         = VT_R4;
  PAS_DOUBLE         = VT_R8;
  PAS_WORDBOOL       = VT_BOOL;
  PAS_SCODE          = VT_ERROR;   { SCODE = Longint }
  PAS_CURRENCY       = VT_CY;      { = TCurrency }
  PAS_DATE           = VT_DATE;    { = Double }
  PAS_STRING         = VT_BSTR;    { requires conversion }
  PAS_IUNKNOWN       = VT_UNKNOWN; { ole interface Class }
  PAS_IDISPATCH      = VT_DISPATCH;{ ole interface Class }
  PAS_ARRAY          = VT_ARRAY;
  PAS_BYTE_PTR       = VT_UI1 or VT_BYREF;
  PAS_INTEGER_PTR    = VT_I2 or VT_BYREF;
  PAS_LONGINT_PTR    = VT_I4 or VT_BYREF;
  PAS_SINGLE_PTR     = VT_R4 or VT_BYREF;
  PAS_DOUBLE_PTR     = VT_R8 or VT_BYREF;
  PAS_WORDBOOL_PTR   = VT_BOOL or VT_BYREF;
  PAS_SCODE_PTR      = VT_ERROR or VT_BYREF;
  PAS_CURRENCY_PTR   = VT_CY or VT_BYREF;
  PAS_DATE_PTR       = VT_DATE or VT_BYREF;
  PAS_STRING_PTR     = VT_BSTR or VT_BYREF;
  PAS_IUNKNOWN_PTR   = VT_UNKNOWN or VT_BYREF;
  PAS_IDISPATCH_PTR  = VT_DISPATCH or VT_BYREF;
  PAS_ARRAY_PTR      = VT_ARRAY or VT_BYREF;
  PAS_PVARIANT       = VT_VARIANT or VT_BYREF; { ptr to VARIANT record }
  PAS_POINTER        = VT_BYREF;

type

  { error type raised by TOleCtlr methods }
  EOleAutoError = class(Exception);

  TOleCtlr = class(TComponent)
  private
    FOleInitialized: Boolean;
    FIDispatch: IDispatch;
  protected
    procedure AssignArgument( const vSource; ptArgType: TPasType;
      var varArg: VARIANT);
      (* Initializes varArg and fills with
         argument(vSource) and argument type(ptArgType).
         Strings are converted to BStr's *)
    procedure AssignResult( var varSource: VARIANT;
      ptResultType: TPasType; var vResult);
      (* Converts varSource to ptResultType and assigns
         resulting value to vResult *)
    function PasToBStr( const S: String): BStr;
      (* Copies a Pascal string to a new BStr and returns the new BStr *)
  public
    procedure CreateOleObject(const lpszProgID: LPCSTR);
      (* Initializes OLE, requests Object class to create an instance
         of itself, gets Object's IDispatch class and saves in FIDispatch.
         This procedure must be called before the Invoke procedure *)
    procedure Invoke( var szMemberName: PChar; wFlags: Word;
      pvarArgs: PVARIANT; cArgs: Word; pvarResult: PVARIANT);
      (* does some housekeeping and then calls the objects
         IDispatch.Invoke method
         PARAMS:
           szMemberName - name of method or property to be invoked.
           wFlags - DISPATCH_METHOD, DISPATCH_PROPERTYGET,
             or DISPATCH_PROPERTYPUT.
           pvarArgs - pointer to VARIANT which contains the argument to
             be passed. If more than one argument, pvarArgs should be the
             first element of an array of VARIANTs. If no arguments,
             pvarArgs should be nil.
           cArgs - number of arguments
           pvarResult - pointer to VARIANT to receive result value. Should
             be nil if no return value *)
    procedure FreeOleObject;
      (* releases the Object instance and uninitializes OLE *)
    destructor Destroy; override;
      (* calls FreeOleObject and then calls inherited Destroy *)
  end;

  TWordCtlr = class(TOleCtlr)
  public
    procedure CreateOleObject;
    { small subset of Word.Basic commands to show how OleCtlr works}
    procedure FileNewDefault;
    procedure FileSaveAs(const sFileName: String);
    procedure GetDocumentProperty( sPropName: String;
      ptResultType: TPasType; var vResult);
    procedure Insert(const sInsert: String);
  end;

procedure Register;

implementation

procedure Register;
begin
  RegisterComponents('Extensions', [TWordCtlr]);
end;

procedure TOleCtlr.CreateOleObject(const lpszProgID: LPCSTR);
var
  hr: HRESULT;
  ClassID: CLSID;
  pUnk: PIUNKNOWN;
  pDisp: IDispatch;
begin

  (* initialize OLE *)
  hr := OleInitialize( Nil );
  FOleInitialized := SUCCEEDEDHR(hr);
  if not FOleInitialized then
    raise EOleAutoError.Create('Ole Intialization Failed');

  (* Retrieve CLSID from the progID *)
  hr := CLSIDFromProgID(lpszProgID, ClassID);
  if FAILEDHR(hr) then
    raise EOleAutoError.Create('Object Not Registered');

  (* Create an instance of the automation object*)
  pUnk := nil;
  pDisp := nil;
  hr := CoCreateInstance(ClassID, nil, CLSCTX_SERVER, IID_IUnknown,
    pUnk);
  if FAILEDHR(hr) then
    raise EOleAutoError.Create('Object Creation Failed');

  (* get IDispatch interface*)
  try
    hr := pUnk.QueryInterface( IID_IDispatch, pDisp);
    if FAILEDHR(hr) then
      raise EOleAutoError.Create('Dispatch Interface Not Available');
    (* save IDispatch *)
    FIDispatch := pDisp;
  finally
    pUnk.Release;
  end;
end;

destructor TOleCtlr.Destroy;
begin
  FreeOleObject;
  inherited Destroy;
end;

procedure TOleCtlr.FreeOleObject;
begin
  if FIDispatch <> nil then
  begin
    FIDispatch.Release;
    FIDispatch := nil;
  end;
  if FOleInitialized then
  begin
    OleUninitialize;
    FOleInitialized := False;
  end;
end;

procedure TOleCtlr.AssignArgument( const vSource; ptArgType: TPasType;
  var varArg: VARIANT);
var
  pstringTemp: PString;
  bstrMem: BStr;
  pbstrMem: PBstr;
begin
  VariantInit( varArg);
  varArg.vt := ptArgType;
 { varArg.bstrVal := PasToBStr( sPropName );}
  with varArg do
    case ptArgType of
      PAS_BYTE:           bVal := Byte(vSource);
      PAS_INTEGER:        iVal := Integer(vSource);
      PAS_LONGINT:        lVal := Longint(vSource);
      PAS_SINGLE:         fltVal := Single(vSource);
      PAS_DOUBLE:         dblVal := Double(vSource);
      PAS_WORDBOOL:       vbool := WordBool(vSource);
      PAS_SCODE:          scode := Longint(vSource);
      PAS_CURRENCY:       cyVal := TCurrency(vSource);
      PAS_DATE:           date := Double(vSource);
      PAS_STRING:         bstrVal := PasToBStr(String(vSource));
      PAS_IUNKNOWN:       punkVal := IUnknown(vSource);
      PAS_IDISPATCH:      IDispatch(pdispVal) := IDispatch(vSource);
      PAS_ARRAY:          parray := PSAFEARRAY(vSource);
      PAS_BYTE_PTR:       PByte(pbVal) := PByte(vSource);
      PAS_INTEGER_PTR:    PInteger(piVal) := PInteger(vSource);
      PAS_LONGINT_PTR:    PLongint(plVal) := PLongInt(vSource);
      PAS_SINGLE_PTR:     PSingle(pfltVal) := PSingle(vSource);
      PAS_DOUBLE_PTR:     PDouble(pdblVal) := PDouble(vSource);
      PAS_WORDBOOL_PTR:   PWordBool(pbool) := PWordBool(vSource);
      PAS_SCODE_PTR:      PLongint(pscode) := PLongint(vSource);
      PAS_CURRENCY_PTR:   PCurrency(pcyVal) := PCurrency(vSource);
      PAS_DATE_PTR:       PDouble(pdate) := PDouble(vSource);
      PAS_STRING_PTR:     begin
                          pstringTemp := PString( vSource );
                          bstrMem := PasToBStr( pstringTemp^ );
                          GetMem( pbstrMem, SizeOf( bstrMem ));
                          pbstrMem^ := bstrMem;
                          PBStr(pbstrVal) := pbstrMem;
                          end;
      PAS_IUNKNOWN_PTR:   PIUnknown(ppunkVal) := PIUnknown(vSource);
      PAS_IDISPATCH_PTR:  PIDispatch(ppdispVal) := PIDispatch(vSource);
      PAS_ARRAY_PTR:      PPSAFEARRAY(pparray) := PPSAFEARRAY(vSource);
      PAS_PVARIANT:       pvarVal := PVARIANT(vSource);
      PAS_POINTER:        byRef := Pointer(vSource);
    end;
end;

procedure TOleCtlr.AssignResult( var varSource: VARIANT;
  ptResultType: TPasType; var vResult);
var
  hr: HRESULT;
begin
  hr := VariantChangeType(varSource, varSource, 0, ptResultType);
  if FAILEDHR( hr ) then
    raise EOleAutoError.Create( 'Invalid Result Type');
  with varSource do
    case ptResultType of
      PAS_BYTE:           Byte(vResult) := bVal;
      PAS_INTEGER:        SmallInt(vResult) := iVal;
      PAS_LONGINT:        Longint(vResult) := lVal;
      PAS_SINGLE:         Single(vResult) := fltVal;
      PAS_DOUBLE:         Double(vResult) := dblVal;
      PAS_WORDBOOL:       WordBool(vResult) := vbool;
      PAS_SCODE:          Longint(vResult) := scode;
      PAS_CURRENCY:       TCurrency(vResult) := cyVal;
      PAS_DATE:           Double(vResult) := date;
      PAS_STRING:         String(vResult) := StrPas(bstrVal);
      PAS_IUNKNOWN:       IUnknown(vResult) := punkVal;
      PAS_IDISPATCH:      IDispatch(vResult) := IDispatch(pdispVal);
      PAS_ARRAY:          PSAFEARRAY(vResult) := parray;
      PAS_BYTE_PTR:       PByte(vResult) := PByte(pbVal);
      PAS_INTEGER_PTR:    PInteger(vResult) := PInteger(piVal);
      PAS_LONGINT_PTR:    PLongint(vResult) := PLongInt(plVal);
      PAS_SINGLE_PTR:     PSingle(vResult) := PSingle(pfltVal);
      PAS_DOUBLE_PTR:     PDouble(vResult) := PDouble(pdblVal);
      PAS_WORDBOOL_PTR:   PWordBool(vResult) := PWordBool(pbool);
      PAS_SCODE_PTR:      PLongint(vResult) := PLongint(pscode);
      PAS_CURRENCY_PTR:   PCurrency(vResult) := PCurrency(pcyVal);
      PAS_DATE_PTR:       PDouble(vResult) := PDouble(pdate);
      PAS_STRING_PTR:     PString(vResult) := NewStr(StrPas(pbstrVal^));
      PAS_IUNKNOWN_PTR:   PIUnknown(vResult) := PIUnknown(ppunkVal);
      PAS_IDISPATCH_PTR:  PIDispatch(vResult) := PIDispatch(ppdispVal);
      PAS_ARRAY_PTR:      PPSAFEARRAY(vResult) := PPSAFEARRAY(pparray);
      PAS_PVARIANT:       PVARIANT(vResult) := pvarVal;
      PAS_POINTER:        Pointer(vResult) := byRef;
    end;
end;

procedure TOleCtlr.Invoke(var szMemberName: PChar; wFlags: Word;
  pVarArgs: PVARIANT; cArgs: Word; pVarResult: PVARIANT);
var
  hr: HRESULT;
  DispIDMember, DispIDNamed: DISPID;
  DispParms: DISPPARAMS;
  ErrInfo: EXCEPINFO;
  ArgErr: Integer;
begin

  (* make sure object has been created *)
  if FIDispatch = nil then
    raise EOleAutoError.Create( 'No Object');

  (* get DISPID of property/method *)
  hr := FIDispatch.GetIDsOfNames(IID_NULL, szMemberName, 1, STDOLE_LCID,
    DispIDMember);
  if FAILEDHR( hr ) then
    raise EOleAutoError.Create( 'Invalid Member Name');

  (* initialize & fill DISPPARAMS *)
  FillChar( DispParms, sizeof( DispParms ), #0);
  if cArgs <> 0 then
  begin
    if pVarArgs = nil then
      raise EOleAutoError.Create( 'Missing Arguments');
    DispParms.cArgs := cArgs;
    DispParms.rgvarg := pVarArgs;
  end;

  (* Property puts have a named argument that represents the value that
     the property is being assigned *)
  if (wFlags and DISPATCH_PROPERTYPUT) = DISPATCH_PROPERTYPUT then
  begin
    if DispParms.cArgs = 0 then
      raise EOleAutoError.Create( 'Invalid Argument Number');
    DispIDNamed := DISPID_PROPERTYPUT;
    DispParms.cNamedArgs := 1;
    DispParms.rgdispidNamedArgs := @DispIDNamed;
  end;

  (* Initialize return variant *)
  if pVarResult <> nil then
    VariantInit( pVarResult^ );

  (* Initialize error info buffer *)
  FillChar( ErrInfo, sizeof( ErrInfo ), #0);
  ArgErr := -1;

  (* make the call *)
  hr := FIDispatch.Invoke( DispIDMember, IID_NULL, STDOLE_LCID, wFlags,
    DispParms, pVarResult, ErrInfo, ArgErr);

  if FAILEDHR(hr) then
    raise EOleAutoError.Create( StrPas( ErrInfo.bstrDescription));
end;

function TOleCtlr.PasToBStr( const S: String): BStr;
var
  sz: PChar;
begin
  sz := StrPCopy( StrAlloc( Length(S) + 1), S);
  result := SysAllocString( sz );
  StrDispose( sz );
end;

procedure TWordCtlr.CreateOleObject;
begin
  inherited CreateOleObject( 'Word.Basic' );
end;

procedure TWordCtlr.FileNewDefault;
const szMemberName: PChar = 'FileNewDefault';
begin
  Invoke( szMemberName, DISPATCH_METHOD, nil, 0, nil);
end;

procedure TWordCtlr.FileSaveAs(const sFileName: String);
const szMemberName: PChar = 'FileSaveAs';
var
  varArg: VARIANT;
begin
  try
    AssignArgument( sFileName, PAS_STRING, varArg);
    Invoke( szMemberName, DISPATCH_METHOD, @varArg, 1, nil);
  finally
    VariantClear( varArg );
  end;
end;

procedure TWordCtlr.Insert( const sInsert: String);
const szMemberName: PChar = 'Insert';
var
  varArg: VARIANT;
begin
  try
    AssignArgument( sInsert, PAS_STRING, varArg);
    Invoke( szMemberName, DISPATCH_METHOD, @varArg, 1, nil);
  finally
    VariantClear( varArg );
  end;
end;

procedure TWordCtlr.GetDocumentProperty( sPropName: String;
   ptResultType: TPasType; var vResult);
const szMemberName: PChar = 'GetDocumentProperty';
var
  varArg, varResult: VARIANT;
begin
  try
    AssignArgument(sPropName, PAS_STRING, varArg);
    Invoke( szMemberName, DISPATCH_METHOD, @varArg, 1, @varResult);
    AssignResult( varResult, ptResultType, vResult);
  finally
    VariantClear( VarArg );
    VariantClear( VarResult);
  end;
end;

end.
