unit NoBDERP;

interface

uses
{$IFDEF WIN32}
  Windows, ComCtrls,
{$ELSE}
  WinTypes, WinProcs,
{$ENDIF}
{$IFDEF VER100}
  Toolwin, Isp3,
{$ENDIF}
  Classes, Controls, Graphics, Grids, SysUtils, TypInfo,
  Forms, StdCtrls, FileCtrl, OutLine, Tabs, Buttons, TabNotBk,
  ExtCtrls, CustomRP, DsgnIntf;

type
  TNoBDEResizerPanel = class(TCustomResizerPanel)
  protected
    procedure AddHandlers( Control: TControl; List: TList ); override;

  public
    constructor Create( AOwner: TComponent ); override;

  published
    property Active;
    property Warnings;
  end;


  TMRControlHandlerVCL = class(TMRControlHandler)
  public
    procedure CalcOldPosition( ParentWidth, ParentHeight: Integer ); override;
    function CalcNewPosition: TRect; override;
    procedure CalcPropertiesNewPositions; override;
  end;


  TMRLValue = record
    LastValue: Integer;
    CValue: Single;
  end;
  PMRLValue = ^TMRLValue;

  TMRLValues = class(TPersistent)
  private
    FList: TList;
    function GetLastValue(Index: Integer): Integer;
    procedure SetLastValue(Index: Integer; v: Integer);
    function GetCValue(Index: Integer): Single;
    procedure SetCValue(Index: Integer; v: Single);
    function GetCount: Integer;
    procedure NeedList;
  public
    destructor Destroy; override;
    procedure Add( lv: Integer; cv: Single );
    procedure Clear;

    property Count: Integer read GetCount;
    property LastValues[Index: Integer]: Integer read GetLastValue
      write SetLastValue;
    property CValues[Index: Integer]: Single read GetCValue write SetCValue;
  end;

  {$IFDEF WIN32}
  TMRLSection = class;
  TMRLSections = class;
  {$ENDIF}

  TMRTab = class(TMRPropertyHandler)
  private
    PInfoWidth, PInfoHeight: PPropInfo;
    LastWidth, LastHeight: Smallint;
    CWidth, Cheight: Single;
  public
    constructor Create( Control: TControl ); override;
    procedure BeforeResize; override;
    procedure AfterResize; override;
  end;

  (***)

  TMRScrollBox = class(TMRPropertyHandler)
  private
    FHorz, FVert: TControlScrollBar;
    LastHorz, LastVert: Integer;
    CHRange, CVRange: Single;
  public
    constructor Create( Control: TControl ); override;
    procedure BeforeResize; override;
    procedure AfterResize; override;
  end;

  (***)

  {$IFDEF WIN32}
  TMRStatusBar = class(TMRPropertyHandler)
  private
    FPanels: TStatusPanels;
    FData: TMRLValues;
  public
    constructor Create( Control: TControl ); override;
    destructor Destroy; override;
    procedure BeforeResize; override;
    procedure AfterResize; override;
  end;

  (***)

  TMRHeaderControl = class(TMRPropertyHandler)
  private
    FSections: THeaderSections;
    FData: TMRLSections;
  public
    constructor Create( Control: TControl ); override;
    destructor Destroy; override;
    procedure BeforeResize; override;
    procedure AfterResize; override;
  end;
  {$ENDIF}

  (***)

  TMRDrawGrid = class(TMRPropertyHandler)
  private
    PInfoWidth, PInfoHeight, PInfoLineWidth: PPropInfo;
    LastWidth, LastHeight, LastLineWidth: Integer;
    CWidth, CHeight, CLineWidth: Single;
    FCols, FRows: TMRLValues;
  public
    constructor Create( Control: TControl ); override;
    destructor Destroy; override;
    procedure BeforeResize; override;
    procedure AfterResize; override;
  end;

  (***)

  {$IFDEF WIN32}
  TMRListView = class(TMRPropertyHandler)
  private
    FColumns: TListColumns;
    FData: TMRLValues;
  public
    constructor Create( Control: TControl ); override;
    destructor Destroy; override;
    procedure BeforeResize; override;
    procedure AfterResize; override;
  end;
  {$ENDIF}

  (***)

  TMRTabSet = class(TMRPropertyHandler)
  private
    PInfoHeight, PInfoStart, PInfoEnd: PPropInfo;
    LastHeight, LastStart, LastEnd: Integer;
    Cheight, CStart, CEnd: Single;
  public
    constructor Create( Control: TControl ); override;
    procedure BeforeResize; override;
    procedure AfterResize; override;
  end;

  (***)

  {$IFDEF WIN32}
  TMRTreeView = class(TMRPropertyHandler)
  private
    PInfoIndent: PPropInfo;
    LastIndent: Integer;
    CIndent: Single;
  public
    constructor Create( Control: TControl ); override;
    procedure BeforeResize; override;
    procedure AfterResize; override;
  end;

  (***)

  TMRLSection = class(TCollectionItem)
  private
    FLastWidth, FLastMinWidth, FLastMaxWidth: Integer;
    CWidth, CMinWidth, CMaxWidth: Single;
  published
    property LastMaxWidth: Integer read FLastMaxWidth write FLastMaxWidth;
    property LastMinWidth: Integer read FLastMinWidth write FLastMinWidth;
    property LastWidth: Integer read FLastWidth write FLastWidth;
  end;

  TMRLSections = class(TCollection)
  private
    function GetItem(Index: Integer): TMRLSection;
    procedure SetItem(Index: Integer; Value: TMRLSection);
  public
    function Add: TMRLSection;
    property Items[Index: Integer]: TMRLSection read GetItem write SetItem; default;
  end;
  {$ENDIF}

  (***)

  TMRItemHeight = class(TMRPropertyHandler)
  private
    PInfoProperty: PPropInfo;
    LastValue: Integer;
    CValue: Single;
  public
    constructor Create( Control: TControl ); override;
    procedure BeforeResize; override;
    procedure AfterResize; override;
  end;

  (***)

  TMRMarginSpacing = class(TMRPropertyHandler)
  private
    PInfoMargin, PInfoSpacing: PPropInfo;
    LastMargin, LastSpacing: Integer;
    CMargin, CSpacing: Single;
  public
    constructor Create( Control: TControl ); override;
    procedure BeforeResize; override;
    procedure AfterResize; override;
  end;

  (***)

  TMRBorderWidth = class(TMRPropertyHandler)
  private
    PInfoProperty: PPropInfo;
    LastValue: Integer;
    CValue: Single;
  public
    constructor Create( Control: TControl ); override;
    procedure BeforeResize; override;
    procedure AfterResize; override;
  end;

  (***)

  TMRMinSize = class(TMRPropertyHandler)
  private
    PInfoSize: PPropInfo;
    LastSize: Integer;
    LastAlign: TAlign;
    CSize: Single;
  public
    constructor Create( Control: TControl ); override;
    procedure BeforeResize; override;
    procedure AfterResize; override;
  end;

  (***)

  TMRStretch = class(TMRPropertyHandler)
  public
    function GetWarning: string; override;
  end;

  (***)

  TMRAutoSize = class(TMRPropertyHandler)
  public
    function GetWarning: string; override;
  end;

  procedure Register;

var
  WarningListForm: TWarningListForm;

implementation

procedure Register;
begin
  RegisterComponents('Additional', [TNoBDEResizerPanel]);
  RegisterPropertyEditor( TypeInfo(TStringList), TNoBDEResizerPanel, '',
    TWarningListProperty );
end;


constructor TNoBDEResizerPanel.Create( AOwner: TComponent );
begin
  inherited Create( AOwner );

  ControlHandlerClass := TMRControlHandlerVCL;
end;


procedure TNoBDEResizerPanel.AddHandlers( Control: TControl; List: TList );
var
  pInfo: PPropInfo;
begin
  inherited AddHandlers( Control, List );

  pInfo := GetPropInfo( Control.ClassInfo, 'Font' );
  if assigned(pInfo) then
    List.Add(TMRFont.CreateHandler( Control, 'Font' ));

  if ((Control is TComboBox) or
  (Control is TDirectoryListBox) or
  {$IFDEF WIN32}
    (Control is TCustomListBox) or
  {$ELSE}
    (Control is TListBox) or
  {$ENDIF}
    (Control is TFileListBox) or (Control is TOutline)) then
    List.Add(TMRItemHeight.Create( Control ));

  if (Control is TScrollBox) then
    List.Add(TMRScrollBox.Create( Control ));

  if (Control is TDrawGrid) then
    List.Add(TMRDrawGrid.Create( Control ));

  if (Control is TTabSet) then
    List.Add(TMRTabSet.Create( Control ));

 if ((Control is TBitBtn) or (Control is TSpeedButton)) then
    List.Add(TMRMarginSpacing.Create( Control ));

  if (Control is TTabbedNotebook) then
    List.Add(TMRFont.CreateHandler( Control, 'TabFont' ));

  if (Control is TImage) then
    List.Add(TMRStretch.Create( Control ));

  if (Control is TLabel) then
    List.Add(TMRAutoSize.Create( Control ));

  {$IFDEF WIN32}
  if ((Control is TPageControl) or (Control is TTabControl)) then
    List.Add(TMRTab.Create( Control ));

  if (Control is TStatusBar) then
    List.Add(TMRStatusBar.Create( Control ));

  if (Control is THeaderControl) then
    List.Add(TMRHeaderControl.Create( Control ));

  if (Control is TListView) then
    List.Add(TMRListView.Create( Control ));

  if (Control is TTreeView) then
    List.Add(TMRTreeView.Create( Control ));
  {$ENDIF}

  {$IFDEF VER100}
  if (Control is TToolWindow) then
    List.Add(TMRBorderWidth.Create( Control ));

  if (Control is TSplitter) then
    List.Add(TMRMinSize.Create( Control ));

  if (Control is THTML) then
  begin
    List.Add(TMRFont.CreateHandler( Control, 'FixedFont' ));
    List.Add(TMRFont.CreateHandler( Control, 'Heading1Font' ));
    List.Add(TMRFont.CreateHandler( Control, 'Heading2Font' ));
    List.Add(TMRFont.CreateHandler( Control, 'Heading3Font' ));
    List.Add(TMRFont.CreateHandler( Control, 'Heading4Font' ));
    List.Add(TMRFont.CreateHandler( Control, 'Heading5Font' ));
    List.Add(TMRFont.CreateHandler( Control, 'Heading6Font' ));
  end;
  {$ENDIF}
end;


procedure TMRControlHandlerVCL.CalcOldPosition( ParentWidth, ParentHeight: Integer );
var
  Cnt: Integer;
begin
  (* This coding is due to the PageControl 'bug'
   get TabSheet dimensions from the active sheet *)
  {$IFDEF WIN32}
  if ((FControl is TTabSheet) and (FControl.Parent is TPageControl)) then
    FControl.BoundsRect := TPageControl(FControl.Parent).ActivePage.BoundsRect;
  {$ENDIF}

  (* Maintain all TTabbedNotebook tabpages with same size *)
  if (FControl is TTabbedNotebook) then
  with TTabbedNotebook(FControl) do
    for Cnt := Pages.Count-1 downto 0 do
      TTabPage(Pages.Objects[Cnt]).BoundsRect := TTabPage(Pages.Objects[PageIndex]).BoundsRect;

  inherited CalcOldPosition( ParentWidth, ParentHeight );
end;


function TMRControlHandlerVCL.CalcNewPosition: TRect;
var
  pInfo: PPropInfo;
  CFont: TFont;
  Cnt: Integer;
begin
  (* Maintain all TTabbedNotebook tabpages with same size *)
  if (FControl is TTabbedNotebook) then
  with TTabbedNotebook(FControl) do
    for Cnt := Pages.Count-1 downto 0 do
      TTabPage(Pages.Objects[Cnt]).BoundsRect := TTabPage(Pages.Objects[PageIndex]).BoundsRect;

  Result := inherited CalcNewPosition;

  (* patch for the ComboBox bug, the vertical size is determined by the
   font size *)
  if (FControl is TCustomComboBox) then
  begin
    pInfo := GetPropInfo( Control.ClassInfo, 'Font' );
    if (pInfo <>  nil) then
    begin
      CFont := TFont(GetOrdProp(Control, pInfo));
      if (CFont.Height < 0) then
        CFont.Height := CFont.Size * CFont.PixelsPerInch div 72;
      CFont.Height := (Round(FControl.Parent.ClientHeight*CHeight) -
        (FControl.Height - CFont.Height));
    end;
  end;

  (* This condition is due to the PageControl 'bug'
   set TabSheet dimensions through the active sheet *)
  {$IFDEF WIN32}
  if ((FControl is TTabSheet) and (FControl.Parent is TPageControl)) then
  begin
    TPageControl(FControl.Parent).ActivePage.BoundsRect := Result;
    FControl.BoundsRect := TPageControl(FControl.Parent).ActivePage.BoundsRect;
  end;
  {$ENDIF}

  (* Maintain all TTabbedNotebook tabpages with same size *)
  if (FControl is TTabbedNotebook) then
  with TTabbedNotebook(FControl) do
    for Cnt := Pages.Count-1 downto 0 do
      TTabPage(Pages.Objects[Cnt]).BoundsRect := TTabPage(Pages.Objects[PageIndex]).BoundsRect;
end;


procedure TMRControlHandlerVCL.CalcPropertiesNewPositions;
begin
  inherited CalcPropertiesNewPositions;

  (* Set dimensions through the active sheet *)
  {$IFDEF WIN32}
  if ((FControl is TTabSheet) and (FControl.Parent is TPageControl)) then
    FControl.BoundsRect := TPageControl(FControl.Parent).ActivePage.BoundsRect;
  {$ENDIF}
end;



{***********}
destructor TMRLValues.Destroy;
begin
  FList.Clear;
  FList.Free;

  inherited Destroy;
end;

procedure TMRLValues.NeedList;
begin
  if not Assigned(FList) then
    FList := TList.Create;
end;

procedure TMRLValues.Add( lv: Integer; cv: Single );
var
  p: PMRLValue;
begin
  NeedList;
  new(p);
  p^.LastValue := lv;
  p^.CValue := cv;
  FList.Add(p);
end;

procedure TMRLValues.Clear;
begin
  while (Assigned(FList) and (FList.Count > 0)) do
  begin
    Dispose( PMRLValue(FList[0]) );
    FList.Delete(0);
  end;
end;

function TMRLValues.GetLastValue( Index: Integer ): Integer;
begin
  if (Assigned(FList) and (Index >= 0) and (Index < FList.Count)) then
    Result := PMRLValue(FList[Index])^.LastValue
  else
    Result := 0;    { caso invlido }
end;

procedure TMRLValues.SetLastValue( Index: Integer; V: Integer );
begin
  if (Assigned(FList) and (Index >= 0) and (Index < FList.Count)) then
    PMRLValue(FList[Index])^.LastValue := V
end;

function TMRLValues.GetCValue( Index: Integer ): Single;
begin
  if (Assigned(FList) and (Index >= 0) and (Index < FList.Count)) then
    Result := PMRLValue(FList[Index])^.CValue
  else
    Result := 0;  { caso invlido }
end;

procedure TMRLValues.SetCValue( Index: Integer; V: Single );
begin
  if (Assigned(FList) and (Index >= 0) and (Index < FList.Count)) then
    PMRLValue(FList[Index])^.CValue := V;
end;

function TMRLValues.GetCount: Integer;
begin
  if Assigned(FList) then
    Result := FList.Count
  else
    Result := 0;
end;

(**************************************
**************************************
**************************************
 ******* Tab property handler ********)

constructor TMRTab.Create( Control: TControl );
begin
  inherited Create(Control);

  PInfoWidth := GetPropertyInfo( Control.ClassInfo, 'TabWidth' );
  PInfoHeight := GetPropertyInfo( Control.ClassInfo, 'TabHeight' );
end;


procedure TMRTab.BeforeResize;
var
  tw, th: Smallint;
  W, H: Single;
begin
  tw := GetOrdProp( FControl, PInfoWidth );
  th := GetOrdProp( FControl, PInfoHeight );

  (* verifies if the tab dimensions changed *)
  if ((LastWidth <> tw) or (LastHeight <> th) or (Invalid = True)) then
  begin
    Change; (* DEBUG *)

    W := FControl.ClientWidth;
    H := FControl.ClientHeight;

    if ((W = 0) or (H = 0)) then
    begin
      Invalid := True;
      exit;
    end;

    CWidth := tw;
    CWidth := CWidth / W;

    CHeight := th;
    CHeight := CHeight / H;
    Invalid := False;
  end;
end;

procedure TMRTab.AfterResize;
var
  tw, th: Smallint;
begin
  if (Invalid = True) then
    exit;

  tw := GetOrdProp( FControl, PInfoWidth );
  th := GetOrdProp( FControl, PInfoHeight );

  if (tw <> 0) then
  begin
    tw := Trunc(CWidth * FControl.ClientWidth);
    if (tw = 0) then
      tw := 1;

    SetOrdProp( FControl, PInfoWidth, tw );
  end;

  if (th <> 0) then
  begin
    th := Trunc(CHeight * FControl.ClientHeight);
    if (th = 0) then
      th := 1;

    SetOrdProp( FControl, PInfoHeight, th );
  end;

  LastWidth := GetOrdProp( FControl, PInfoWidth );
  LastHeight := GetOrdProp( FControl, PInfoHeight );
end;


(**************************************
**************************************
**************************************
 ******* ScrollBox property handler ********)

constructor TMRScrollBox.Create( Control: TControl );
var
  pInfo: PPropInfo;
begin
  inherited Create(Control);

  pInfo := GetPropertyInfo( Control.ClassInfo, 'HorzScrollBar' );
  FHorz := TControlScrollBar(GetOrdProp(Control, pInfo));
  pInfo := GetPropertyInfo( Control.ClassInfo, 'VertScrollBar' );
  FVert := TControlScrollBar(GetOrdProp(Control, pInfo));
end;

procedure TMRScrollBox.BeforeResize;
var
  Hr, Vr: Integer;
  W, H: Single;
begin
  (* reset position to avoid plenty of problems
   must be revised this part *)
  FHorz.Position := 0;
  FVert.Position := 0;

  Hr := FHorz.Range;
  Vr := FVert.Range;

  (* verifies if the tab dimensions changed *)
  if ((LastHorz <> Hr) or (LastVert <> Vr) or (Invalid = True)) then
  begin
    Change; (* DEBUG *)

    W := FControl.ClientWidth;
    H := FControl.ClientHeight;

    if ((W = 0) or (H = 0)) then
    begin
      Invalid := True;
      exit;
    end;

    CHRange := Hr;
    CHRange := CHRange / W;

    CVRange := Vr;
    CVRange := CVRange / H;
    Invalid := False;
  end;
end;

procedure TMRScrollBox.AfterResize;
begin
  if (Invalid = True) then
    exit;

  FHorz.Range := Trunc(CHRange * FControl.ClientWidth);
  FVert.Range := Trunc(CVRange * FControl.ClientHeight);

  LastHorz := FHorz.Range;
  LastVert := FVert.Range;
end;


{$IFDEF WIN32}
(**************************************
**************************************
**************************************
 ** Status Panels property handler ***)

constructor TMRStatusBar.Create( Control: TControl );
var
  pInfo: PPropInfo;
begin
  inherited Create(Control);

  FData := TMRLValues.Create;
  pInfo := GetPropertyInfo( Control.ClassInfo, 'Panels' );
  FPanels := TStatusPanels(GetOrdProp(Control, pInfo));
end;


destructor TMRStatusBar.Destroy;
begin
  FData.Free;

  inherited;
end;

procedure TMRStatusBar.BeforeResize;
var
  W, WC: Single;
  Cnt: Integer;
begin
  { verifies if the panels changed }
  if (FPanels.Count <> FData.Count) then
    Invalid := True;

  if (Invalid = False) then
    for Cnt := 0 to FPanels.Count-1 do
      if (FPanels.Items[Cnt].Width <> FData.LastValues[Cnt]) then
      begin
        Invalid := True;
        break;
      end;

  if (Invalid = False) then
    exit;

  Change; { DEBUG }

  WC := FControl.ClientWidth;
  if (WC = 0) then
  begin
    Invalid := True;
    exit;
  end;

  FData.Clear;
  for Cnt := 0 to FPanels.Count-1 do
  begin
    W := FPanels.Items[Cnt].Width;
    FData.Add( 0, W / WC );
  end;

  Invalid := False;
end;


procedure TMRStatusBar.AfterResize;
var
  Cnt: Integer;
  W: Single;
begin
  if (Invalid = True) then
    exit;

  W := FControl.ClientWidth;

  for Cnt := 0 to FData.Count-1 do
  begin
    FPanels.Items[Cnt].Width := Trunc(FData.CValues[Cnt] * W);
    FData.LastValues[Cnt] := FPanels.Items[Cnt].Width;
  end;
end;


(**************************************
**************************************
**************************************
 ** HeaderSections property handler **)

constructor TMRHeaderControl.Create( Control: TControl );
var
  pInfo: PPropInfo;
begin
  inherited;

  FData := TMRLSections.Create( TMRLSection );
  pInfo := GetPropertyInfo( Control.ClassInfo, 'Sections' );
  FSections := THeaderSections(GetOrdProp(Control, pInfo));
end;


destructor TMRHeaderControl.Destroy;
begin
  FData.Free;

  inherited;
end;

procedure TMRHeaderControl.BeforeResize;
var
  Changed: Byte;
  W, WC: Single;
  Cnt: Integer;
begin
  { verifies if the sections changed }
  Changed := $00;
  { verifies if number of sections differ }
  if ((FSections.Count <> FData.Count) or (Invalid = True)) then
    Changed := $FF
  else
  begin
    { verifies if sections width differ, which can happen by user action
      if so recalc at least this coeficient }
    for Cnt := 0 to FSections.Count-1 do
      if (FSections.Items[Cnt].Width <> FData.Items[Cnt].LastWidth) then
      begin
        Changed := $0F;
        break;
      end;

    { verifies if sections min/max widths differ
     if so recalc all coeficients }
    for Cnt := 0 to FSections.Count-1 do
      if ((FSections.Items[Cnt].MinWidth <> FData.Items[Cnt].LastMinWidth) or
        (FSections.Items[Cnt].MaxWidth <> FData.Items[Cnt].LastMaxWidth)) then
      begin
        Changed := $FF;
        break;
      end;
  end;

  if (Changed = $00) then
    exit;

  Change; { DEBUG }

  WC := FControl.ClientWidth;
  if (WC = 0) then
  begin
    Invalid := True;
    exit;
  end;

  { calc min/max width coeficients }
  if ((Changed and $F0) <> 0) then
  begin
    FData.Clear;
    for Cnt := 0 to FSections.Count-1 do
    begin
      FData.Add;  { BUGBUG it's assumed the item is added at end of array }
      W := FSections.Items[Cnt].MinWidth;
      FData.Items[Cnt].CMinWidth := W / WC;
      W := FSections.Items[Cnt].MaxWidth;
      FData.Items[Cnt].CMaxWidth := W / WC;
    end;
  end;

  { calc width coeficient }
  if ((Changed and $0F) <> 0) then
    for Cnt := 0 to FSections.Count-1 do
    begin
      W := FSections.Items[Cnt].Width;
      FData.Items[Cnt].CWidth := W / WC;
    end;

  Invalid := False;
end;

procedure TMRHeaderControl.AfterResize;
var
  Cnt: Integer;
  W: Single;
begin
  if (Invalid = True) then
    exit;

  W := FControl.ClientWidth;

  for Cnt := 0 to FData.Count-1 do
  begin
    FSections.Items[Cnt].Width := Trunc(FData.Items[Cnt].CWidth * W);
    FSections.Items[Cnt].MinWidth := Trunc(FData.Items[Cnt].CMinWidth * W);
    FSections.Items[Cnt].MaxWidth := Trunc(FData.Items[Cnt].CMaxWidth * W);

    FData.Items[Cnt].LastWidth := FSections.Items[Cnt].Width;
    FData.Items[Cnt].LastMinWidth := FSections.Items[Cnt].MinWidth;
    FData.Items[Cnt].LastMaxWidth := FSections.Items[Cnt].MaxWidth;
  end;
end;


{**************************************
**************************************
**************************************
 ** List Columns property handler ***}

constructor TMRListView.Create( Control: TControl );
var
  pInfo: PPropInfo;
begin
  inherited;

  FData := TMRLValues.Create;
  pInfo := GetPropertyInfo( Control.ClassInfo, 'Columns' );
  FColumns := TListColumns(GetOrdProp(Control, pInfo));
end;


destructor TMRListView.Destroy;
begin
  FData.Free;

  inherited;
end;

procedure TMRListView.BeforeResize;
var
  W, WC: Single;
  Cnt: Integer;
begin
  { verifies if the panels changed }
  if ((FColumns.Count <> FData.Count) or (Invalid = True)) then
    Invalid := True
  else
  begin
    for Cnt := 0 to FColumns.Count-1 do
      if (FColumns.Items[Cnt].Width <> FData.LastValues[Cnt]) then
      begin
        Invalid := True;
        break;
      end;
  end;

  if (Invalid = False) then
    exit;

  Change; { DEBUG }

  { certifies that the width isn't 0 }
  WC := FControl.ClientWidth;
  if (WC = 0) then
    exit;

  FData.Clear;
  for Cnt := 0 to FColumns.Count-1 do
  begin
    W := FColumns.Items[Cnt].Width;
    FData.Add( 0, W / WC );
  end;

  Invalid := False;
end;

procedure TMRListView.AfterResize;
var
  Cnt: Integer;
  W: Single;
begin
  if (Invalid = True) then
    exit;

  W := FControl.ClientWidth;

  for Cnt := 0 to FData.Count-1 do
  begin
    FColumns.Items[Cnt].Width := Trunc(FData.CValues[Cnt] * W);

    FData.LastValues[Cnt] := FColumns.Items[Cnt].Width;
  end;
end;

{**************************************
**************************************
**************************************
 ***** TreeView property handler *****}

constructor TMRTreeView.Create( Control: TControl );
begin
  inherited;

  PInfoIndent := GetPropertyInfo( Control.ClassInfo, 'Indent' );
end;

procedure TMRTreeView.BeforeResize;
var
  ind: Integer;
  W: Single;
begin
  { verifies if property changed }
  ind := GetOrdProp( FControl, PInfoIndent );

  if ((LastIndent <> ind) or (Invalid = True)) then
  begin
    Change; { DEBUG }

    W := FControl.ClientWidth;
    if (W = 0) then
    begin
      Invalid := True;
      exit;
    end;

    CIndent := ind;
    CIndent := CIndent / W;
  end;

  Invalid := False;
end;


procedure TMRTreeView.AfterResize;
begin
  if (Invalid = True) then
    exit;

  SetOrdProp( FControl, PInfoIndent, Trunc(CIndent * FControl.ClientWidth) );

  LastIndent := GetOrdProp( FControl, PInfoIndent );
end;


(**************************************
**************************************
**************************************)

function TMRLSections.Add: TMRLSection;
begin
  Result := TMRLSection(inherited Add);
end;

function TMRLSections.GetItem(Index: Integer): TMRLSection;
begin
  Result := TMRLSection(inherited GetItem(Index));
end;

procedure TMRLSections.SetItem(Index: Integer; Value: TMRLSection);
begin
  inherited SetItem(Index, Value);
end;


{$ENDIF}

(**************************************
**************************************
**************************************
 ** Col/Row Grid size property handler ***)

constructor TMRDrawGrid.Create( Control: TControl );
begin
  inherited Create(Control);

  FCols := TMRLValues.Create;
  FRows := TMRLValues.Create;

  PInfoWidth := GetPropertyInfo( Control.ClassInfo, 'DefaultColWidth' );
  PInfoHeight := GetPropertyInfo( Control.ClassInfo, 'DefaultRowHeight' );
  PInfoLineWidth := GetPropertyInfo( Control.ClassInfo, 'GridLineWidth' );
end;


destructor TMRDrawGrid.Destroy;
begin
  FCols.Free;
  FRows.Free;

  inherited Destroy;
end;

procedure TMRDrawGrid.BeforeResize;
var
  tw, th, lw: Integer;
  W, H, WC, HC: Single;
  Cnt: Integer;
begin
  (* verifies changes *)
  tw := GetOrdProp( FControl, PInfoWidth );
  th := GetOrdProp( FControl, PInfoHeight );
  lw := GetOrdProp( FControl, PInfoLineWidth );
  WC := FControl.ClientWidth;
  HC := FControl.ClientHeight;

  if ((LastWidth <> tw) or (LastHeight <> th)) then
    Invalid := True;

  with FControl as TDrawGrid do
  begin
    (* verifies if the cells dimensions changed *)
    if ((FCols.Count <> ColCount) or (FRows.Count <> RowCount)) then
      Invalid := True
    else
    begin
      for Cnt := 0 to FCols.Count-1 do
        if (FCols.LastValues[Cnt] <> ColWidths[Cnt]) then
        begin
          Invalid := True;
          break;
        end;

      for Cnt := 0 to FRows.Count-1 do
        if (FRows.LastValues[Cnt] <> RowHeights[Cnt]) then
        begin
          Invalid := True;
          break;
        end;
    end;
  end;

  if (Invalid = True) then
  begin
    Change; (* DEBUG *)

    if ((WC = 0) or (HC = 0)) then
      exit;

    CWidth := tw;
    CWidth := CWidth / WC;

    CHeight := th;
    CHeight := CHeight / HC;

    CLineWidth := lw;
    CLineWidth := CLineWidth / WC;

    with FControl as TDrawGrid do
    begin
      FCols.Clear;
      for Cnt := 0 to ColCount-1 do
      begin
        W := ColWidths[Cnt];
        FCols.Add( 0, W / WC );
      end;

      FRows.Clear;
      for Cnt := 0 to RowCount-1 do
      begin
        H := RowHeights[Cnt];
        FRows.Add( 0, H / HC );
      end;
    end;
  end;

  Invalid := False;
end;


procedure TMRDrawGrid.AfterResize;
var
  Cnt: Integer;
  W: Single;
begin
  if (Invalid = True) then
    exit;

  SetOrdProp( FControl, PInfoWidth, Trunc(CWidth * FControl.ClientWidth) );
  SetOrdProp( FControl, PInfoHeight, Trunc(CHeight * FControl.ClientHeight) );
  SetOrdProp( FControl, PInfoLineWidth, Trunc(CLineWidth * FControl.ClientHeight) );

  LastWidth := GetOrdProp( FControl, PInfoWidth );
  LastHeight := GetOrdProp( FControl, PInfoHeight );
  LastLineWidth := GetOrdProp( FControl, PInfoLineWidth );

  with FControl as TDrawGrid do
  begin
    W := FControl.ClientWidth;
    for Cnt := 0 to FCols.Count-1 do
      ColWidths[Cnt] := Trunc(FCols.CValues[Cnt] * W);

    W := FControl.ClientHeight;
    for Cnt := 0 to FRows.Count-1 do
      RowHeights[Cnt] := Trunc(FRows.CValues[Cnt] * W);

    for Cnt := 0 to FCols.Count-1 do
      FCols.LastValues[Cnt] := ColWidths[Cnt];
    for Cnt := 0 to FRows.Count-1 do
      FRows.LastValues[Cnt] := RowHeights[Cnt];
  end;
end;


(**************************************
**************************************
**************************************
 ******* TabSet property handler *****)

constructor TMRTabSet.Create( Control: TControl );
begin
  inherited Create(Control);

  PInfoHeight := GetPropertyInfo( Control.ClassInfo, 'TabHeight' );
  PInfoStart := GetPropertyInfo( Control.ClassInfo, 'StartMargin' );
  PInfoEnd := GetPropertyInfo( Control.ClassInfo, 'EndMargin' );
end;

procedure TMRTabSet.BeforeResize;
var
  th, sm, em: Integer;
  W, H: Single;
begin
  (* verifies if any property changed *)
  th := GetOrdProp( FControl, PInfoHeight );
  sm := GetOrdProp( FControl, PInfoStart );
  em := GetOrdProp( FControl, PInfoEnd );

  if ((LastHeight <> th) or (LastStart <> sm) or (LastEnd <> em)
    or (Invalid = True)) then
  begin
    Change; (* DEBUG *)

    H := FControl.ClientHeight;
    W := FControl.ClientWidth;
    if ((H = 0) or (W = 0)) then
    begin
      Invalid := True;
      exit;
    end;

    CHeight := th;
    CHeight := CHeight / H;

    CStart := sm;
    CStart := CStart / W;

    CEnd := em;
    CEnd := CEnd / W;
  end;

  Invalid := False;
end;

procedure TMRTabSet.AfterResize;
begin
  if (Invalid = True) then
    exit;

  SetOrdProp( FControl, PInfoHeight, Trunc(CHeight * FControl.ClientHeight) );
  SetOrdProp( FControl, PInfoStart, Trunc(CStart * FControl.ClientWidth) );
  SetOrdProp( FControl, PInfoEnd, Trunc(CEnd * FControl.ClientWidth) );

  LastHeight := GetOrdProp( FControl, PInfoHeight );
  LastStart := GetOrdProp( FControl, PInfoStart );
  LastEnd := GetOrdProp( FControl, PInfoEnd );
end;


(**************************************
**************************************
**************************************
**** ItemHeight property handler *****)

constructor TMRItemHeight.Create( Control: TControl );
begin
  inherited Create(Control);

  PInfoProperty := GetPropertyInfo( Control.ClassInfo, 'ItemHeight' );
end;

procedure TMRItemHeight.BeforeResize;
var
  th: Integer;
  H: Single;
begin
  (* verifies if the property changed *)
  th := GetOrdProp( FControl, PInfoProperty );

  if ((LastValue <> th) or (Invalid = True)) then
  begin
    Change; (* DEBUG *)

    H := FControl.ClientHeight;
    if (H = 0) then
    begin
      Invalid := True;
      exit;
    end;

    CValue := th;
    CValue := CValue / H;
  end;

  Invalid := False;
end;

                     
procedure TMRItemHeight.AfterResize;
begin
  if (Invalid = True) then
    exit;

  SetOrdProp( FControl, PInfoProperty, Trunc(CValue * FControl.ClientHeight) );

  LastValue := GetOrdProp( FControl, PInfoProperty );
end;


(**************************************
**************************************
**************************************
* Space & Margin properties handler **)

constructor TMRMarginSpacing.Create( Control: TControl );
begin
  inherited Create(Control);

  PInfoMargin := GetPropertyInfo( Control.ClassInfo, 'Margin' );
  PInfoSpacing := GetPropertyInfo( Control.ClassInfo, 'Spacing' );
end;


procedure TMRMarginSpacing.BeforeResize;
var
  Margin, Spacing: Integer;
  W: Single;
begin
  (* verifies if the properties changed *)
  Margin := GetOrdProp( FControl, PInfoMargin );
  Spacing := GetOrdProp( FControl, PInfoSpacing );

  W := FControl.ClientWidth;

  if (Margin = -1) then
  begin
    Invalid := True;
    exit;
  end;

  if ((LastMargin <> Margin) or (LastSpacing <> Spacing)) then
    Invalid := True;

  if (Invalid = True) then
  begin
    Change; (* DEBUG *)

    if (W = 0) then
      exit;

    CMargin := Margin;
    CMargin := CMargin / W;

    CSpacing := Spacing;
    CSpacing := CSpacing / W;
  end;

  Invalid := False;
end;


procedure TMRMarginSpacing.AfterResize;
begin
  if (Invalid = False) then
  begin
    SetOrdProp( FControl, PInfoMargin, Trunc(CMargin * FControl.ClientWidth) );
    LastMargin := GetOrdProp( FControl, PInfoMargin );

    SetOrdProp( FControl, PInfoSpacing, Trunc(CSpacing * FControl.ClientWidth) );
    LastSpacing := GetOrdProp( FControl, PInfoSpacing );
  end;
end;


(**************************************
**************************************
**************************************
**** BorderWidth property handler *****)

constructor TMRBorderWidth.Create( Control: TControl );
begin
  inherited Create(Control);

  PInfoProperty := GetPropertyInfo( Control.ClassInfo, 'BorderWidth' );
end;

procedure TMRBorderWidth.BeforeResize;
var
  th: Integer;
  H: Single;
begin
  (* verifies if the property changed *)
  th := GetOrdProp( FControl, PInfoProperty );

  if ((LastValue <> th) or (Invalid = True)) then
  begin
    Change; (* DEBUG *)

    H := FControl.ClientWidth;
    if (H = 0) then
    begin
      Invalid := True;
      exit;
    end;

    CValue := th;
    CValue := CValue / H;
  end;

  Invalid := False;
end;


procedure TMRBorderWidth.AfterResize;
begin
  if (Invalid = True) then
    exit;

  SetOrdProp( FControl, PInfoProperty, Trunc(CValue * FControl.ClientWidth) );

  LastValue := GetOrdProp( FControl, PInfoProperty );
end;


(**************************************
**************************************
**************************************
**** MinSize property handler *****)

constructor TMRMinSize.Create( Control: TControl );
begin
  inherited Create(Control);

  PInfoSize := GetPropertyInfo( Control.ClassInfo, 'MinSize' );
end;

procedure TMRMinSize.BeforeResize;
var
  ts: Integer;
  ta: TAlign;
  H: Single;
begin
  (* verifies if the property changed *)
  ts := GetOrdProp( FControl, PInfoSize );
  ta := FControl.Align;

  if ((LastSize <> ts) or (LastAlign <> ta) or (Invalid = True)) then
  begin
    Change; (* DEBUG *)

    if ((ta = alTop) or (ta = alBottom)) then
      H := FControl.ClientHeight
    else
      if ((ta = alLeft) or (ta = alRight)) then
        H := FControl.ClientWidth
      else
        H := 0;

    if (H = 0) then
    begin
      Invalid := True;
      exit;
    end;

    CSize := ts;
    CSize := CSize / H;
    LastAlign := ta;
  end;

  Invalid := False;
end;


procedure TMRMinSize.AfterResize;
begin
  if (Invalid = True) then
    exit;

  if ((LastAlign = alTop) or (LastAlign = alBottom)) then
    SetOrdProp( FControl, PInfoSize, Trunc(CSize * FControl.ClientHeight) )
  else
    if ((LastAlign = alLeft) or (LastAlign = alRight)) then
      SetOrdProp( FControl, PInfoSize, Trunc(CSize * FControl.ClientWidth) );

  LastSize := GetOrdProp( FControl, PInfoSize );
end;


(**************************************
**************************************
**************************************
 ******* Stretch property handler *******)


function TMRStretch.GetWarning: string;
var
  pInfo: PPropInfo;
  b: LongBool;
begin
  pInfo := GetPropertyInfo( Control.ClassInfo, 'Stretch' );
  b := LongBool(GetOrdProp(Control, pInfo));

  if (b = False) then
    Result := 'Set Stretch to True'
  else
    Result := '';
end;

(**************************************
**************************************
**************************************
******* AutoSize property handler *******)


function TMRAutoSize.GetWarning: string;
var
  pInfo: PPropInfo;
  b: LongBool;
begin
  pInfo := GetPropertyInfo( Control.ClassInfo, 'AutoSize' );
  b := LongBool(GetOrdProp(Control, pInfo));

  if (b = True) then
    Result := 'Set AutoSize to False'
  else
    Result := '';
end;


end.
