unit SCTree;

{-------------------------------------------------------------------------}
{ CREATED  ON : 14-may-1997                                               }
{ HISTORY     : __-___-____                                               }
{             : 13-may-1997 :.create TScrollableTreeView, from DBTree.pas }
{                                                                         }
{ GOAL IS     : Defines and manipulates a scrollable TTreeView            }
{                                                                         }
{-------------------------------------------------------------------------}

interface

Uses
    SysUtils, WinTypes, WinProcs, Messages, Classes, Graphics, Controls,
    ComCtrls, Forms, Dialogs;

Type

{-------------------------------------------------------------------------}
{ COMPONENT   : TSCROLLABLETREEVIEW                                       }
{               ================                                          }
{ GOAL        : Defines the basic methods for a future DBTreeview. Just   }
{             : add the scrollable capability.                            }
{ MAIN INFOS  :                                                           }
{-------------------------------------------------------------------------}
TScrollableTreeView = Class( TTreeView )
private
   bAutoScroll       : Boolean;             { b  = boolean                }
protected
   Procedure   Compute_TreeMoves( X, Y : Integer ); dynamic;
   Procedure   MouseMove( Shift : TShiftState; X, Y : Integer ) ; override;
public
   Constructor Create( AOwner : TComponent ); override;
published
   property    AutoScroll        : Boolean     read  bAutoScroll
                                               write bAutoScroll;
   end;


Procedure Register;


implementation


Constructor TScrollableTreeView.Create;
{-------------------------------------------------------------------------}
{ GOAL        : Initializes AUTOSCROLL                                    }
{             :                                                           }
{ IN/OUT      : in    AOwner            = Parent                          }
{ SIDE EFFECT : <none>                                                    }
{-------------------------------------------------------------------------}
Begin
     inherited Create( AOwner );

     AutoScroll := True;
End;


Procedure TSCrollableTreeView.Compute_TreeMoves;
{-------------------------------------------------------------------------}
{ GOAL        : Computes if tree must be moved up or down, left or right, }
{             : depending on mouse position.                              }
{ IN/OUT      : in    X, Y              = the mouse position              }
{ SIDE EFFECT : <none>                                                    }
{-------------------------------------------------------------------------}
Var
   NbPixels   : Integer;
   RMin, RMax : Integer;
   HOffset,
   VOffset    : Integer;
Begin
     {--------------------------------------------------------------------}
     { Algorithm :                                                        }
     { -----------                                                        }
     { . If not Autoscroll then Exit                                      }
     { . Detect scroll bars (horizontal and/or vertical) to set offsets   }
     { . If mouse is near upper edge or lower edge, scroll the control to }
     {   up or down by one line                                           }
     { . If mouse is near left or right edge, scroll the control to one   }
     {   page left or one page right                                      }
     {--------------------------------------------------------------------}

     If Not AutoScroll Then Exit;

     {--------------------------------------------------------------------}
     { Retrieve the scroll bar ranges, if such scroll bars exist (either  }
     { horizontal or vertical). An offset must be set if there is a       }
     { scroll bar, i-e if there is a range (RMin <> RMax).                }
     {--------------------------------------------------------------------}
     GetScrollRange( Handle, SB_HORZ, RMin, RMax );
     If RMin = RMax Then HOffset := 0 Else HOffset := 16;
     GetScrollRange( Handle, SB_VERT, RMin, RMax );
     If RMin = RMax Then VOffset := 0 Else VOffset := 16;

     {--------------------------------------------------------------------}
     { Near an edge means at a maximum of half a line, i-e half the pixels}
     { of the current font.                                               }
     {--------------------------------------------------------------------}
     NbPixels := Abs( ( Font.Height ) Div 2 );

     If ( Y < NbPixels ) Then
        Perform( WM_VSCROLL, SB_LINEUP, 0 );

     If ( Y > Height - VOffset - NbPixels ) Then
        Perform( WM_VSCROLL, SB_LINEDOWN, 0 );

     If ( X < NbPixels ) Then
        Perform( WM_HSCROLL, SB_PAGELEFT, 0 );

     If ( X > Width - HOffset - NbPixels ) Then
        Perform( WM_HSCROLL, SB_PAGERIGHT, 0 );
End;


Procedure TScrollableTreeView.MouseMove;
{-------------------------------------------------------------------------}
{ GOAL        : Overrides the MOUSEMOVE.                                  }
{             :                                                           }
{ IN/OUT      :                         =                                 }
{ SIDE EFFECT : <none>                                                    }
{-------------------------------------------------------------------------}
Begin
     {--------------------------------------------------------------------}
     { Algorithm :                                                        }
     { -----------                                                        }
     { . Sees if tree needs to be moved.                                  }
     { . Call the inherited method                                        }
     {--------------------------------------------------------------------}
     Compute_TreeMoves( X, Y );

     inherited MouseMove( Shift, X, Y );
End;


Procedure Register;
{-------------------------------------------------------------------------}
{ GOAL        : Registers TScrollableTreeView                             }
{             :                                                           }
{ SIDE EFFECT : <none>                                                    }
{-------------------------------------------------------------------------}
Begin
     RegisterComponents( 'DDO Org.', [ TScrollableTreeView ] );
End;


end.
