unit AS_Registry;

{ ==============================================================
  AS_Registry 1.0
{ ==============================================================

  SUMMARY

  The object inherits TRegistry and introduces additonal registry
  access methods.
  It allows you to use a single method to read from, and a single
  method to write to the registry, no matter what type variable
  you have. Instead of exceptions, appropriate null values are
  returned when value was not found.

  Author:       1997, Andy Schmidt
  Email:       Andy_Schmidt@CompuServe.com
  Compiler:    Delphi 2.01
  Runtime:     Win32

{ --------------------------------------------------------------


  USAGE

  1. replace "uses Registry" with "uses AS_Registry" in your
     uses clause.

  2. replace references to the TRegistry object and TAS_Registry

  3. replace ReadString, ReadBinary, ReadDate, etc. methods with
     ReadValue. Replace the various Writexxxx methods with
     WriteValue.

  4. remove your error handling for "not found" condition during
     registry read.

  Remark: You will still be able to use the existing TRegistry
  methods and properties with TAS_Registry.


{ --------------------------------------------------------------


  NEW METHODS

  OpenKeyVerbose   Opens a key, and reports whether the key was
                   created, or was already existing.

  CleanUpKey       Determines whether the current key contains
                   no values. If so, the empty key is deleted.

  ReadValue
  WriteValue       Reads or writes the value to the property using
                   the appropriate TRegistry method. The type of
                   value passed to the method will determine,
                   which format is used to store the data in the
                   registry.
                   Remark: If the value does not exist, the ReadValue
                   method will NOT cause an exception, but rather
                   return the appropriate "low" value for the
                   particular data type (e.g. an empty string,
                   or the number 0).


{ --------------------------------------------------------------

  ACKNOWLEDGEMENTS

  Dennis Passmore - his DPREGSTR.ZIP gave me the idea to use
                    a variant as the key value.

{ --------------------------------------------------------------

  CHANGE HISTORY

  1.0.0 31-Jan-97 (AS)  Initial Development

  -------------------------------------------------------------- }


interface

uses
    Windows, Registry;

type
    Tasr_OpenResult = set of (Failed, Created, OK);

    TAS_Registry = class(TRegistry)
    public
    { Public declarations: visible only at run-time }
    function OpenKeyVerbose(const RootHKEY: HKEY; const RegistryKey: string; const Create: boolean): Tasr_OpenResult;
    function ReadValue(const ValueName: string; const Value: variant): variant;
    procedure WriteValue(const ValueName: string; const Value: variant);
    procedure CleanupKey(const RegistryKey: string);
    end;

implementation


{ This open method will indicate, whether a new key had to be created }
function TAS_Registry.OpenKeyVerbose(const RootHKEY: HKEY; const RegistryKey: string; const Create: boolean): Tasr_OpenResult;
begin
    RootKey := RootHKEY;
    if KeyExists(RegistryKey) then
        result := []                        // Key already existed
    else
        result := [Created];                // key may get created
    if OpenKey(RegistryKey, Create) then
        result := result + [OK]             // Open or create registry key
    else result := [Failed];
end;


{ Read registry value into a variant }
function TAS_Registry.ReadValue(const ValueName: string; const Value: variant): variant;
begin
    { Check if value exists with length > 0 }
    if ValueExists(Valuename) then
        { Remark: if we still get ERegistryException,
          then the type or length did not match }
        case (VarType(Value) and varTypeMask) of
            { retrieve data depending on type in variant }
            varEmpty, varNull, varByte, varSmallint, varInteger, varError:
                Result := ReadInteger(ValueName);
            varSingle, varDouble:
                Result := ReadFloat(ValueName);
            varCurrency:
                Result := ReadCurrency(ValueName);
            varDate:
                Result := ReadDateTime(ValueName);
            varBoolean:
                Result := ReadBool(ValueName);
            varString, varOleStr:
                Result := ReadString(ValueName);
        end
    else
        { Value did not exist }
        Result := unassigned;
end;


{ Write registry value from a variant }
procedure TAS_Registry.WriteValue(const ValueName: string; const Value: variant);
begin
    case (VarType(Value) and varTypeMask) of
    { store data depending on type in variant }
        varEmpty, varNull, varByte, varSmallint, varInteger, varError:
            WriteInteger(ValueName, Value);
        varSingle, varDouble:
            WriteFloat(ValueName, Value);
        varCurrency:
            WriteCurrency(ValueName, Value);
        varDate:
            WriteDateTime(ValueName, Value);
        varBoolean:
            WriteBool(ValueName, Value);
        varString, varOleStr:
            WriteString(ValueName, Value);
    end
end;


{ Close key, and delete if key has no values }
procedure TAS_Registry.CleanupKey(const RegistryKey: string);
var RegistryInfo: TRegKeyInfo;
begin
    if GetKeyInfo(RegistryInfo) then
        if RegistryInfo.NumValues <= 0 then
            { remove any empty key we created }
            DeleteKey(RegistryKey);
end;

end.
