unit LabelEllipsis;

{ ==============================================================
  TLabelEllipsis 1.0
{ ==============================================================

  SUMMARY

  This modification to TLabel will shorten the text in the Caption
  property so that it fits the available space. It will either
  insert an ellipsis (...) at the end of the string, or insert
  the ellipsis in the beginning of a path (if that's what the
  Caption is supposed to display)

  Author:       1997, Andy Schmidt
  Email:       Andy_Schmidt@CompuServe.com
  Compiler:    Delphi 2.01
  Palette:     "Custom"
  Runtime:     Win32 (will not run under NT 3.51 or lower)


{ --------------------------------------------------------------

  PROPERTIES

  Ellipsis         Defines where to insert the ellipsis
                   elNone - string is not modified.
                   elEnd  - '...' shown at end of shortened string
                   elPath - '...' inserted in beginning of a path

  Alignment        Remark: This property is modified to accept
                   only left-justified as a possible option.

  WordWrap         Remark: This property is modified to not permit
                   WordWrap - as they are mutually exclusive.


  FUNCTIONS

  StringEllipsis   Can be called directly from a program to shorten
                   the string intended for an existing TLabel. This
                   is an alternative way if one does not wish to
                   use the TLabelEllipsis component on one's form.

{ --------------------------------------------------------------


  USAGE AS A COMPONENT

  1. add the LabelEllipsis component to your component library
     and rebuild.

  2. from the "Custom" palette, use the TLabelEllipsis instead
     of a TLabel.

  3. set the new "Ellipsis" property as required.


  USAGE AS A FUNCTION

  1. add LabelEllipsis to your USES clause.

  2. design a form with a TLabel (e.g. MyLabel).

  3. call the function to shorten the text to fit the available
     space for your TLabel object:

     MyLabel.Caption := StringEllipsis(MyLabel, Application.ExeName, elPath);

     This will display the path to your .EXE file in the available space.
     See what happens if you make your TLabel smaller than required to
     display the full path.


  REMARKS

  Unfortunately, Win32 does NOT expand any imbedded tab characters
  before calculating the size of the string. So you can NOT include
  any #09 tab characters inside your string.


{ --------------------------------------------------------------

  LICENSE

  The Author hereby grants to you a nonexclusive license to use
  this software and the accompanying Instructions, only as
  authorized in this License.

  You agree that you will not assign, sublicense, transfer,
  pledge, lease, rent, or share your rights under this License
  in return for compensation of any kind. You may include this
  object in executable form with your own product, but before
  you use this software in source code format for commercial
  purposes, you are required to pay a license fee of $20.00
  to the Author.

  You acknowledge and agree that the Software and the
  accompanying Instructions are intellectual property of
  the Author, protected under U.S. copyright law. You further
  acknowledge and agree that all right, title and interest in
  and to the Software, are and shall remain with the Author.
  This License does not convey to you an interest in or to the
  Software, but only a limited and revocable right of use.

  THIS SOFTWARE IS LICENSED "AS IS," AND LICENSOR DISCLAIMS ANY
  AND ALL WARRANTIES, WHETHER EXPRESS OR IMPLIED, INCLUDING,
  WITHOUT LIMITATION, ANY IMPLIED WARRANTIES OF MERCHANTABILITY
  OR FITNESS FOR A PARTICULAR PURPOSE.

  Author's cumulative liability to you or any other party for
  any loss or damages resulting from any claims, demands, or
  actions arising out of or relating to this License shall not
  exceed the license fee paid (if any) to Author for the use of
  the Software. In no event shall Author be liable for any
  indirect, incidental, consequential, special, or exemplary
  damages or lost profits, even if Author has been advised of
  the possibility of such damages.

  This software and accompanying instructions are provided with
  restricted rights. Use, duplication or disclosure by the
  Government is subject to restrictions as set forth in
  subparagraph (c)(1)(ii) of The Rights in Technical Data and
  Computer Software clause at DFARS 252.227-7013 or
  subparagraphs (c)(1) and (2) of the Commercial Computer
  Software - Restricted Rights 48 CFR 52.227-19, as applicable.

{ --------------------------------------------------------------

  CHANGE HISTORY

  1.0.0 01-Feb-97 (AS)  Initial Development
  1.0.1 09-Feb-97 (AS)  WordWrap property set to elFalse to avoid
                        conflict with 'flase' reserved word

  -------------------------------------------------------------- }


interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, StdCtrls;

type
  TEllipsisStyle = (elNone, elPath, elEnd);
  TE_Alignment = (taLeftJustify);
  TE_WordWrap = (elFalse);

  TLabelEllipsis = class(TLabel)
  private
    { Private declarations }
    FEllipsisStyle: TEllipsisStyle;
    FAlignment: TE_Alignment;
    FWordWrap: TE_WordWrap;
  protected
    { Protected declarations }
    function GetLabelText: string; override;
  published
    { Override existing properties to limit to appropriate options }
    property Alignment: TE_Alignment read FAlignment write FAlignment default taLeftJustify;
    property WordWrap: TE_WordWrap read FWordWrap write FWordWrap default elFalse;
    { New property to choose the type of ellipsis }
    property Ellipsis: TEllipsisStyle read FEllipsisStyle write FEllipsisStyle default elNone;
  end;

function StringEllipsis(Component: TLabel; const Text: string; const EStyle: TEllipsisStyle):string;

procedure Register;

implementation

procedure Register;
begin
  RegisterComponents('Custom', [TLabelEllipsis]);
end;


{ Shorten the Caption to fit available space }
function TLabelEllipsis.GetLabelText: string;
begin
  if csDesigning in self.ComponentState then
      { during design time display the caption }
      Result := Caption
  else
      { during runtime shorten the current caption }
      Result := StringEllipsis(Self, Caption, FEllipsisStyle);
end;


{ Shorten a Text to fit into the caption of a TLabel object }
function StringEllipsis(Component: TLabel; const Text: string; const EStyle: TEllipsisStyle):string;

var
  TempRect: TRect;
  TempFlags: cardinal;

begin
    Result := Text;
    if EStyle <> elNone then
        begin
        TempRect := Component.ClientRect;               // get dimensions of lable
        Component.Canvas.Font := Component.Font;        // Font from properties
        TempFlags := DT_LEFT or DT_MODIFYSTRING         // left align - we will use all space anyway
                             or DT_EXPANDTABS;          // unfortunately, ExpandTabs has no effect!
        if EStyle = elPath then
            { shorten beginning of path to fit available space }
            TempFlags := TempFlags or DT_PATH_ELLIPSIS
        else if EStyle = elEnd then
            { truncate end of text to fit space }
            TempFlags := TempFlags or DT_END_ELLIPSIS;
        Windows.DrawTextEx(Component.Canvas.Handle,     // Have Win32 fit the string
                           PChar(Result),
                           Length(Result),
                           TempRect,
                           TempFlags,
                           nil                     );
        SetLength( Result, StrLen( PChar(Result) ) );   // adjust string length accordingly
        end;
end;

end.
