unit LicenseInput;

{ ==============================================================
  LicenseInput 1.0.0
{ ==============================================================

  SUMMARY

  Standard registration screen for a software use to enter his
  license information or to start a free trial period.

  Author:       1997, Andy Schmidt
  Email:       Andy_Schmidt@CompuServe.com
  Compiler:    Delphi 2.01
  Runtime:     Win32


{ --------------------------------------------------------------

  USAGE

  It is best to use this form through the LicenseCode method
  'LicenseCode.Prompt'.


  SAMPLES

  TestLicenseSetup - demonstrates use of this form at setup time.
  TestLicenseInput - demonstrates use of this form inside a typical
                     application.

{ --------------------------------------------------------------


  PROPERTIES

  ObjLicense       (R/W) Pointer to LicenseCode object.
                   Must be provided by programmer as soon as this
                   form object is created.

  CodeUpdate       (R/W) Indicates that the code is currently
                   updating one of the writeable input fields.

  EnforceSetup     Set to 'true' by programmer, if you don't want
                   user to be able to initialize missing license
                   information during the first run of the
                   application.
                   Remark: See TestLicenseSetup sample on how to
                   create the license information at setup time.

 { --------------------------------------------------------------

  CHANGE HISTORY

  1.0.0 25-Jan-97 (AS)  Initial Development

  -------------------------------------------------------------- }


interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  StdCtrls, ExtCtrls,
  LicenseCode;

type
  TLicenseInput = class(TForm)
    FCompany: TLabel;
    FProduct: TLabel;
    LUserName: TLabel;
    LCode: TLabel;
    FUserName: TEdit;
    FCode: TEdit;
    FOptions: TEdit;
    LComment1: TLabel;
    BtnApply: TButton;
    BtnRegister: TButton;
    FStatus: TLabel;
    LShape: TShape;
    procedure FormShow(Sender: TObject);
    procedure Changed(Sender: TObject);
    procedure BtnApplyClick(Sender: TObject);
    procedure BtnRegisterClick(Sender: TObject);
  private
    { Private declarations }
    CtrBadCode: byte;
    FUpdating: boolean;
    FEnforceSetup: boolean;
    FObjLicense: TLicenseCode;
    function ProcessRegistration: boolean;
    function SetupRequired: boolean;
  protected
    { Protected declarations: visible only to derived objects }
    Registering: boolean;
  public
    { Public declarations }
    property ObjLicense: TLicenseCode read FObjLicense write FObjLicense;
    property CodeUpdate: boolean read FUpdating write FUpdating;
    property EnforceSetup: boolean write FEnforceSetup;
  end;

implementation

{$R *.DFM}


{ Initialize the display }
procedure TLicenseInput.FormShow(Sender: TObject);
begin
    FUpdating := true;
    { Initialize display }
    FObjLicense.Load;                // load current license info from disk
    with FObjLicense do
        begin
        FCompany.Caption := Company;
        FProduct.Caption := TrimRight( Product + ' ' + Version );
        FUserName.Text := UserName;  // display default user name
        if SetUpRequired then
            { Application mandates Setup - no defaults available }
            begin
            FOptions.Text := '';
            FCode.Text := '';
            end
        else
            { Application doesn't mandate Setup, create trial if necessary }
            if FirstTime then
                begin                         // first use on this machine
                FUserName.ReadOnly := False;  // permit user to change his name
                BtnApply.Enabled := True;
                FOptions.Text := IntToStr(TrialOptions);
                FCode.Text := Calculate( UserName, TrialOptions, Date+TrialDays );
                Validate(UserName, TrialOptions, Date+TrialDays, FCode.Text);
                end
            else
                begin                         // otherwise use information from disk
                FOptions.Text := IntToStr(Options);
                FCode.Text := Code;
                end;
        FStatus.Caption := LicenseStatus;
        end;
    FUpdating := false;
end;


{ Invalidate old data on screen }
procedure TLicenseInput.Changed(Sender: TObject);
begin
    if FUpdating then exit;                // ignore updating by program itself

    BtnApply.Enabled := True;
    if FCode.Readonly then FCode.Text := '';
    FStatus.Caption := '';
end;


{ Use the current input to calculate code and store to disk }
procedure TLicenseInput.BtnApplyClick(Sender: TObject);
begin
    FUpdating := true;
    with FObjLicense do
        begin
        if Registering then
            begin
            { We are processing a registered user. }
            if ProcessRegistration then
                { If registration was successful, lock down fields }
                begin
                FUserName.Text := UserName;   // in case a default user was set
                FCode.Text := Code;           // properly formatted code
                BtnRegister.Enabled := true;  // user may try again
                FOptions.ReadOnly := true;    // lock down fields
                FCode.ReadOnly := true;
                if (not FirstTime) or FEnforceSetup then
                    { Software has been used before, or setup is required }
                    begin                     // lock down user name
                    BtnApply.Enabled := false;
                    FUserName.ReadOnly := true;
                    end;
                Registering := false;
                end;
            end
        else{ We are processing a trial user }
            if not SetUpRequired then
                begin
                Store(FUserName.Text, TrialOptions, Date+TrialDays, '');
                FUserName.Text := UserName;   // in case a default user was set
                FOptions.Text := IntToStr(TrialOptions);
                FCode.Text := Code;           // new registration code
                FStatus.Caption := LicenseStatus; // registration result
                end;
        end;
    FUpdating := false
end;


{ User wants to register the software }
procedure TLicenseInput.BtnRegisterClick(Sender: TObject);
begin
    Registering := true;
    BtnRegister.Enabled := false;         // prevent iterations
    BtnApply.Enabled := true;             // user must press "Apply"
    FUserName.ReadOnly := false;          // permit user to change data
    FOptions.ReadOnly := false;
    FCode.ReadOnly := false;
end;


{ Validate input fields and store registration }
function TLicenseInput.ProcessRegistration: boolean;

var
    TempMessage: string;
    TempOptions: byte;
    TempStatus: TLicense_Status;

begin
    result := true;                   // in case everything goes well
    TempMessage := '';                // display all error messages at once

    { Validate User Name }
    if length(FUserName.Text) < 5 then
        begin
        TempMessage := 'Enter a User Name with 5 or more letters. ';
        if result then FUserName.SetFocus;
        result := false;
        end;
    { Validate Options }
    try TempOptions := StrToInt(FOptions.Text);
        except on EConvertError do
            begin
            TempOptions := 0;
            TempMessage := TempMessage + 'Enter Options with 1 to 3 digits. ';
            if result then FOptions.SetFocus;
            result := false;
            end;
        end;
    { Validate Code }
    if FCode.Text = '' then
        begin
        TempMessage := TempMessage + 'You must enter a license code. ';
        if result then FUserName.SetFocus;
        result := false;
        end;

    if result = false then
        { Display error message }
        begin
        FStatus.Caption := 'Invalid Input';
        Application.MessageBox( PChar(TempMessage), PChar(FStatus.Caption), MB_ICONERROR+MB_OK);
        end
    else
        { Validate and store the new registration code }
        begin
        with FObjLicense do
            begin
            TempStatus := Store(FUserName.Text, TempOptions, Date + TrialDays, FCode.Text);
            { We must specify the Registered Options in case this a registered code,
              and we must specify the trial date, in case this is a trial.
              The license object will ignore the Options for trials, and the expiry date
              when checking for registered codes }
            FStatus.Caption := LicenseStatus;
            end;
        if (TempStatus <> License_OK) and (TempStatus <> License_Trial) then
            begin
            result := false;
            { Abort if user guesses wrong 3 or more times }
            if CtrBadCode < 3 then
                inc(CtrBadCode)
            else
                begin
                Application.MessageBox( 'You entered three invalid registration codes.',
                                        'Access Denied', MB_ICONERROR+MB_OK);
                Application.Terminate;
                end;
            end;
        end;
end;


{ Check if product requires a setup before use }
function TLicenseInput.SetUpRequired: boolean;
begin
    if FObjLicense.FirstTime and FEnforceSetup then
        { Product mandates setup before first use }
        begin
        Application.MessageBox( 'Please run Setup to permit trial use.',
                                'Access Denied', MB_ICONERROR+MB_OK);
        BtnApply.Enabled := false;
        FUserName.ReadOnly := true;
        Result := true;
        end
    else Result := false;
end;


end.
