unit ProjectInfo;

{ ==============================================================
  ProjectInfo 1.0.0
{ ==============================================================

  SUMMARY

  Centrally stores and provides project data and services that
  are commonly required during the startup of an application or
  in the Help About box.

  Author:       1996, Andy Schmidt
  Email:       Andy_Schmidt@CompuServe.com
  Compiler:    Delphi 2.01
  Palette:     "Custom"
  Runtime:     Win32


{ --------------------------------------------------------------


  USAGE

  1. install ProjectInfo.pas to your Component palette.

  2. create a form and drop ProjectInfo from the 'Custom' palette
     onto the form.

  3. Set Company, Title and Version properties or create a
     VersionInfo resource.


{ --------------------------------------------------------------

  PROPERTIES

  Company          Name of the company releasing this software
                   Runtime Default: CompanyName from the VersionInfo
                   resource of the executable.

  Title            Name of the application
                   Runtime Default: ProductName from the VersionInfo
                   resource of the executable, or the Application.Title
                   defined in Project|Options at design time.
                   Comment: This project title is used as the unique
                   identifier to check for multiple instances under
                   Win32 and Win32s.

  Version          Version/Release/Modification/Patch numbers as elements
                   vv, rr, mm, pp.
                   Runtime Default: ProductVersion from the VersionInfo
                   resource of the executable.

  VersionString    [Read] contents of Version property in format
                   vv.rr or vv.rr.mm.pp

  ReleaseStatus    Optional release comments
                   (e.g. Alpha, Beta, Preview)
                   If set to '(RES)', it will be set from FileFlags
                   of the VersionInfo resource of the executable.

  ExeFileInfo      [Read] Attributes of the .EXE file, such as
                   Name (no path or extension), timestamp, full path,
                   version information resource.
  
  Copyright        Legal copyright notice
                   If set to '(RES)', it will be set from LegalCopyright
                   of the VersionInfo resource of the executable.

  UserName         [Read] Registered User's Name
                   Read from registry.

  Options          [Read] Registered Product Features
                   Read from registry.

  LicenseStatus    [Read] Registration Message string
                   Displays remaining trial dates or any other
                   license status.

  RegistryKey      [Read] Unique registry key for this application.
                   Created from \'Software'\Company\Title\Version

  NagScreen        True = developer wants to display a NagScreen with
                   license data during trial period.

  EnforceSetup     True = developer requires that registry entries already
                   exist from a prior "setup" step.
                   False = registry entries will be created on first
                   invocation of this product.

  InstanceCheck    True = developer wants to enforce single instance

  FirstInstance    [Read] Indicates if the current instance of this
                   application was the first to be created.


  METHODS

  -none-


  EVENTS

  OnFoundInstance  Receives control when a prior instance was detected.
                   FirstInstance will be false. If InstanceCheck = Enabled, then
                   application will shut down immediately after the event handler.

{ --------------------------------------------------------------

  ACKNOWLEDGEMENTS
  Xavier Pacheco   His 'MultiInst' introduces 'Messages' to handle applications
                   which change their title bar.

{ --------------------------------------------------------------

  LICENSE

  The Author hereby grants to you a nonexclusive license to use
  this software and the accompanying Instructions, only as
  authorized in this License.

  You agree that you will not assign, sublicense, transfer,
  pledge, lease, rent, or share your rights under this License
  in return for compensation of any kind. You may include this
  object in executable form with your own product, but before
  you use this software in source code format for commercial
  purposes, you are required to pay a license fee of $20.00
  to the Author.

  You acknowledge and agree that the Software and the
  accompanying Instructions are intellectual property of
  the Author, protected under U.S. copyright law. You further
  acknowledge and agree that all right, title and interest in
  and to the Software, are and shall remain with the Author.
  This License does not convey to you an interest in or to the
  Software, but only a limited and revocable right of use.

  THIS SOFTWARE IS LICENSED "AS IS," AND LICENSOR DISCLAIMS ANY
  AND ALL WARRANTIES, WHETHER EXPRESS OR IMPLIED, INCLUDING,
  WITHOUT LIMITATION, ANY IMPLIED WARRANTIES OF MERCHANTABILITY
  OR FITNESS FOR A PARTICULAR PURPOSE.

  Author's cumulative liability to you or any other party for
  any loss or damages resulting from any claims, demands, or
  actions arising out of or relating to this License shall not
  exceed the license fee paid (if any) to Author for the use of
  the Software. In no event shall Author be liable for any
  indirect, incidental, consequential, special, or exemplary
  damages or lost profits, even if Author has been advised of
  the possibility of such damages.

  This software and accompanying instructions are provided with
  restricted rights. Use, duplication or disclosure by the
  Government is subject to restrictions as set forth in
  subparagraph (c)(1)(ii) of The Rights in Technical Data and
  Computer Software clause at DFARS 252.227-7013 or
  subparagraphs (c)(1) and (2) of the Commercial Computer
  Software - Restricted Rights 48 CFR 52.227-19, as applicable.

{ --------------------------------------------------------------

  CHANGE HISTORY

  1.0.0 10-Jan-97 (AS)  Initial Development

  -------------------------------------------------------------- }

interface

uses
  Classes, Forms, SysUtils, DsgnIntf, Windows, Messages,
  FileInfo, LicenseCode;

type

  // Custom Windows Message
  TCMRestore = record
    Msg: cardinal;                        // Message ID
    wParam: word;                         // unused
    Handle: integer;                      // Sender's hWnd
    Result: longint;
    end;

  // Project Info (Main object)
  TProjectInfo = class(TComponent)
  private
  { Private declarations: visible only in this unit }
    // Fields to store property values
    FCompany: string;
    FTitle: string;
    FVersion: TProjVersion;
    FExeFileInfo: TFileInfo;
    FReleaseStatus: string;
    FCopyright: string;
    FUserName: string;
    FOptions: byte;
    FLicenseStatus: string;
    FRegistryKey: string;
    FInstanceCheck: boolean;
    FNagScreen: boolean;
    FEnforceSetup: boolean;
    FFirstInstance: boolean;
    FOnFoundInstance: TNotifyEvent;
    // Other instance variables
    Mutex: THandle;
    CM_Restore: cardinal;
    PWndMethod: pointer;
    POldWndMethod: pointer;
    LoadedDone: boolean;
    // Methods to maintain properties and handle custom messages
    function GetVersionString: string;
    procedure WndMethod(var WinMsg: TMessage);
  protected
  { Protected declarations: visible only to derived objects }
    function FoundInstance: boolean; dynamic;
    procedure RuntimeDefaults; dynamic;
  public
  { Public declarations: visible only at run-time }
    property UserName: string read FUserName;
    property Options: byte read FOptions;
    property LicenseStatus: string read FLicenseStatus;
    property VersionString: string read GetVersionString;
    property RegistryKey: string read FRegistryKey;
    property FirstInstance: boolean read FFirstInstance;
    property ExeFileInfo: TFileInfo read FExeFileInfo;
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    procedure Loaded; override;
  published
  { Published declarations: visible in Object Inspector }
    property Company: string read FCompany write FCompany;
    property Title: string read FTitle write FTitle;
    property Version: TProjVersion read FVersion write FVersion;
    property ReleaseStatus: string read FReleaseStatus write FReleaseStatus;
    property Copyright: string read FCopyright write FCopyright;
    property InstanceCheck: boolean read FInstanceCheck write FInstanceCheck default true;
    property NagScreen: boolean read FNagScreen write FNagScreen default false;
    property EnforceSetup: boolean read FEnforceSetup write FEnforceSetup default false;
    property OnFoundInstance: TNotifyEvent read FOnFoundInstance write FOnFoundInstance;
  end;

  // Property Editor for Version type
  TProjVersionProperty = class(TClassProperty)
  public
    function GetValue: string; override;
  end;

procedure Register;
function BuildVersionString(PProjVersion: TProjVersion): string;


implementation

const
    UniquePrefix: string = 'InstanceCheck_';
    // Default strings used for property editor
    DReleaseStatus: string = 'Development';
    DAuthor: string = '*Author*';


{ Register class on component palette }
procedure Register;
begin
    RegisterComponents('Custom', [TProjectInfo]);
    RegisterPropertyEditor(TypeInfo(TProjVersion), nil, '', TProjVersionProperty);
end;


{ Initialize the object properties, create nested objects }
constructor TProjectInfo.Create(AOwner: TComponent);

var
    CurrentYear: string;

begin
    inherited Create(AOwner);                   // Call the inherited constructor first
    FVersion := TProjVersion.Create;            // Create the version object
    FExeFileInfo := TFileInfo.Create;           // Create the file information object
    FInstanceCheck := true;                     // Permit only single instance
    FEnforceSetup := false;                     // Require registry settings to be present
    FNagScreen := false;                        // Display registration screen on every startup

    { Set design time defaults, before persistent data is loaded from the form.
      This way we don't overwrite user's settings. }
    if csDesigning in self.ComponentState then
        begin                                   // Build default copyright string
        DateTimeToString(CurrentYear, 'yyyy', Now);
        FCopyright := ' ' + CurrentYear + ', ' + DAuthor + '. All Rights Reserved!';
        FReleaseStatus := DReleaseStatus;       // Default release status
        end;
end;


{ Startup processing, before form is displayed }
procedure TProjectInfo.Loaded;

begin
    inherited Loaded;
    { Attempt to set runtime defaults on first path of "Loaded" method}
    if not ((csDesigning in self.ComponentState) or LoadedDone) then RunTimeDefaults;
    LoadedDone := True;                         // Loaded is called TWICE for form inheritance
end;


{ Set runtime defaults - in case design-time defaults were deleted by user.
  Check for multiple instances.
  Check for valid license registration }
procedure TProjectInfo.RuntimeDefaults;

var
    TempVersion: string;
    TempObjLicense: TLicenseCode;
    TempLicenseStatus: TLicense_Status;
    TempPromptWhen: TLicense_PromptWhen;
    TempUniqueString: array[0..254] of char;

begin
    { default company name }
    if (FCompany = '') or (FCompany = '(RES)') then
        FCompany := FExeFileInfo.QueryVersionString('CompanyName');

    { default product title }
    if FTitle = '' then
    begin
        FTitle := FExeFileInfo.QueryVersionString('ProductName'); // Version resource of executable
        if FTitle = '' then
            FTitle := Application.Title;            // Project Options from Delphi IDE
        if FTitle = '' then
            FTitle := FExeFileInfo.Name;            // Name of the executable file
    end;

    { default application title }
    if UpperCase(Application.Title) = UpperCase(FExeFileInfo.Name)
        then Application.Title := FTitle;

    { default copyright notice }
    if (FCopyright = '') or (FCopyright = '(RES)') then
        FCopyright := FExeFileInfo.QueryVersionString('LegalCopyright');

    if FReleaseStatus = '(RES)' then FReleaseStatus := '';
    { append FileFlags from VersionInfo resource - if any }
    if FExeFileInfo.FixedInfo <> nil then   // Do we have fixed VersionInfo resources?
        begin
        FReleaseStatus := FReleaseStatus + ' ';
        with FExeFileInfo.FixedInfo^ do
            begin
            if (dwFileFlags and dwFileFlagsMask and VS_FF_Patched) > 0 then
               FReleaseStatus := FReleaseStatus + 'Patched ';
            if (dwFileFlags and dwFileFlagsMask and VS_FF_Debug) > 0 then
               FReleaseStatus := FReleaseStatus + 'Debug ';
            if (dwFileFlags and dwFileFlagsMask and VS_FF_PreRelease) > 0 then
               FReleaseStatus := FReleaseStatus + 'Pre-Release ';
            if (dwFileFlags and dwFileFlagsMask and VS_FF_PrivateBuild) > 0 then
               FReleaseStatus := FReleaseStatus + 'Private Build ';
            if (dwFileFlags and dwFileFlagsMask and VS_FF_SpecialBuild) > 0 then
               FReleaseStatus := FReleaseStatus + 'Special Build ';
            end; {with}
        FReleaseStatus := trim(FReleaseStatus);
        end;

    with FVersion do
        if ((vv+rr+mm+pp) = 0) and (FExeFileInfo.ProductVersion <> nil) then
            { extract file version from executable }
            FVersion.Assign(FExeFileInfo.ProductVersion);

    { To detect multiple instances, create unique project identifiers }
    StrLCopy(TempUniqueString, PChar(UniquePrefix + FTitle), SizeOf(TempUniqueString) - 1);
    CM_Restore := RegisterWindowMessage(TempUniqueString);

    { See if identifier already exists }
    Mutex := Windows.OpenMutex(MUTEX_ALL_ACCESS, false, TempUniqueString);
    if Mutex <> 0 then
        { There may already be an instance }
        begin
        FFirstInstance := False;                        // Flag as secondary instance
        if FoundInstance or Application.Terminated then // Handle secondary instance
            exit;                                       // no sense continuing
        end
    else
        begin
        { This appears to be the first instance }
        FFirstInstance := True;            // Flag as first instance
        Mutex := Windows.CreateMutex(nil, false, TempUniqueString);
        { Register message handler }
        PWndMethod := MakeObjectInstance(WndMethod);
        POldWndMethod := Pointer(SetWindowLong(Application.Handle, GWL_WNDPROC, LongInt(PWndMethod)));
        end;


    { load users license registration from disk }
    with FVersion do
        if (vv+rr+mm+pp) > 0 then
            { Use the major and minor version for registry key }
            TempVersion := IntToStr(vv) + '.' + IntToStr(rr)
        else TempVersion := '';
    TempObjLicense := TLicenseCode.CreateFromName( FCompany, FTitle, TempVersion );

    with TempObjLicense do
        { extract properties }
        begin
        FUserName := UserName;
        FOptions := Options;
        FRegistryKey := RegistryKey;

        { display registration screen if no valid license found }
        if NagScreen then TempPromptWhen := DuringTrial
        else TempPromptWhen := NoLicense;
        TempLicenseStatus := Prompt(TempPromptWhen, FEnforceSetup);
        FLicenseStatus := LicenseStatus;
        Free;
        end;

    { Abort application if no valid License Registration }
    if (TempLicenseStatus <> License_OK) and (TempLicenseStatus <> License_Trial) then
        Application.Terminate;

end;


destructor TProjectInfo.Destroy;
begin
    FVersion.Free;                        // Free the version object
    FExeFileInfo.Free;                    // Free the file information object
    if (Mutex <> 0) and (FFirstInstance or (FInstanceCheck = false)) then
		 Windows.CloseHandle(Mutex);
    if POldWndMethod <> nil then          // Restore original message handler
        SetWindowLong(Application.Handle, GWL_WNDPROC, LongInt(POldWndMethod));
    if PWndMethod <> nil then
        FreeObjectInstance(PWndMethod);
    inherited destroy;                    // Call the inherited destructor last
end;


{ Property Editors }
function TProjVersionProperty.GetValue: string;
begin
  //  Result := BuildVersionString( TProjVersion(GetComponent(0)) );
    Result := BuildVersionString( TProjVersion(GetOrdValue) );
end;



{ Custom Message Handler }
procedure TProjectInfo.WndMethod(var WinMsg: TMessage);
begin
    with WinMsg do
        begin
        if Msg = CM_Restore then
            { We are told to make our window active }
            begin
            Application.Restore;
            if Application.MainForm.WindowState = wsMinimized then
                Application.MainForm.WindowState := wsNormal;
            SetForegroundWindow(Application.Mainform.Handle);
            end
        else
            { Pass all other messages to old message handler }
            Result := CallWindowProc(POldWndMethod, Application.Handle, Msg, wParam, lParam);
        end;
end;


{ An existing instance was found
  Returns 'true' if this instance should be shut down}
function TProjectInfo.FoundInstance: boolean;

var
    BSMReceipients: DWORD;                      // Set to actual receipients

begin
    result := false;
    if Assigned(FOnFoundInstance) then          // Call Event Handler
        FOnFoundInstance(Self);
    if FInstanceCheck then                      // Does user want single instance?
        begin
        Application.ShowMainform := false;      // suppress display
        { Broadcast a message to other instance }
        BSMReceipients := BSM_APPLICATIONS;     // send to all applications
        BroadCastSystemMessage(BSF_IGNORECURRENTTASK or BSF_POSTMESSAGE,
                               @BSMReceipients, CM_Restore, 0, Application.Handle);
        Application.Terminate;                  // shut down this instance
        result := true;
        end;
end;


{ Get the Version ID String }
function TProjectInfo.GetVersionString: string;
begin
    Result := BuildVersionString(FVersion);
end;


{ Construct a Version ID String in the format vv.mm.rr.pp
  Skip any .rr.pp that are .0.0 }
function BuildVersionString(PProjVersion: TProjVersion): string;
begin
    with PProjVersion do
        if (mm = 0) and (pp = 0) then
            Result := IntToStr(vv) + '.' + IntToStr(rr)
        else
            Result := IntToStr(vv) + '.' + IntToStr(rr) + '.' +
                      IntToStr(mm) + '.' + IntToStr(pp);
end;

end.
 