unit DirectSound ;

interface

uses Forms, SysUtils, DSound, MMSystem, OLE2, Windows, DirectDraw ;

type
  TDSMono16Array = Array [0..0] of SmallInt ;
  PDSMono16Array = ^TDSMono16Array ;

  TDSMono8Array = Array [0..0] of ShortInt ;
  PDSMono8Array = ^TDSMono8Array ;

  TDSStereo16Array = Array [0..0] of record Left, Right: SmallInt end ;
  PDSStereo16Array = ^TDSStereo16Array ;

  TDSStereo8Array = Array [0..0] of record Left, Right: ShortInt end ;
  PDSStereo8Array = ^TDSStereo8Array ;

  TDSSound = class ( TObject )
  private
    SoundBuffer: IDirectSoundBuffer ;
  public
    constructor Create ( Frequency: Extended ) ;
    destructor Destroy ; override ;
    procedure Play ; virtual ;
    procedure Stop ; virtual ;
  end ;

  TDSLoopedSound = class ( TDSSound )
  public
    constructor Create ( Frequency: Extended ) ;
    procedure Play ; override ;
  end ;

  TDSPolySound = class ( TDSSound )
  private
    ExtraSoundBuffers: Array [ 0..3 ] of IDirectSoundBuffer ;
    NextBuffer: Integer ;
  public
    constructor Create ( Frequency: Extended ) ;
    destructor Destroy ; override ;
    procedure Play ; override ;
    procedure Stop ; override ;
  end ;

implementation

var
  DSound: IDirectSound ;

{-----------------------------------------------------------
 |
 | TDSSOUND.CREATE
 |
 | Set up a DirectSoundBuffer object
 |
 -----------------------------------------------------------}

constructor TDSSound.Create ( Frequency: Extended ) ;
var
  BufferDesc: DSBUFFERDESC ;
  WaveFormat: TWaveFormatEx ;
  Buffer1, Buffer2: PDSStereo16Array ;
  BufSize1, BufSize2: DWORD ;
  Index: Integer ;
begin
  inherited Create ;

  { Make sure DirectSound is active }

  if DSound = nil then
    raise Exception.Create ( 'DirectSound not present' ) ;

  { Set up a sound description }

  ZeroMemory ( @WaveFormat, sizeof ( WaveFormat ) ) ;
  WaveFormat.wFormatTag := WAVE_FORMAT_PCM ;
  WaveFormat.nChannels := 2 ;
  WaveFormat.nSamplesPerSec := 22050 ;
  WaveFormat.nAvgBytesPerSec := 22050 * 4 ;
  WaveFormat.nBlockAlign := 4 ;
  WaveFormat.wBitsPerSample := 16 ;

  ZeroMemory ( @BufferDesc, sizeof ( BufferDesc ) ) ;
  BufferDesc.dwSize := sizeof ( DSBUFFERDESC ) ;
  BufferDesc.dwFlags := DSBCAPS_STATIC + DSBCAPS_CTRLDEFAULT ;
  BufferDesc.dwBufferBytes := 44100 ;
  BufferDesc.lpwfxFormat := @WaveFormat ;

  { Create the sound and lock it }

  DXCheck ( DSound.CreateSoundBuffer ( BufferDesc, SoundBuffer, nil ) ) ;
  DXCheck ( SoundBuffer.Lock ( 0, 44100, Pointer ( Buffer1 ),
      BufSize1, Pointer ( Buffer2 ), BufSize2, 0 ) ) ;

  { Fill it with a sample }

  for Index := 0 to 11024 do
  with Buffer1 [ Index ] do
  begin
    Left := Round (
      ( Sin ( Index / Frequency ) + Sin ( Index / ( Frequency + 1 ) ) ) / 2
      * ( ( 11024 - Index ) / 11024 )
      * 32000 ) ;
    Right := Left ;
  end ;

  { Unlock it }

  DXCheck ( SoundBuffer.Unlock ( Pointer ( Buffer1 ), BufSize1,
      Pointer ( Buffer2 ), BufSize2 ) ) ;
end ;

{-----------------------------------------------------------
 |
 | TDSSOUND.DESTROY
 |
 | Release a DirectSoundBuffer object
 |
 -----------------------------------------------------------}

destructor TDSSound.Destroy ;
begin
  if SoundBuffer <> nil then
  begin
    Stop ;
    SoundBuffer.Release ;
    SoundBuffer := nil ;
  end ;
  inherited Destroy ;
end ;

{-----------------------------------------------------------
 |
 | TDSSOUND.PLAY
 |
 | Play a DirectSoundBuffer
 |
 -----------------------------------------------------------}

procedure TDSSound.Play ;
begin
  SoundBuffer.Play ( 0, 0, 0 ) ;
end ;

{-----------------------------------------------------------
 |
 | TDSSOUND.STOP
 |
 | Stop a DirectSoundBuffer
 |
 -----------------------------------------------------------}

procedure TDSSound.Stop ;
begin
  SoundBuffer.Stop ;
end ;

{-----------------------------------------------------------
 |
 | TDSLOOPEDSOUND.CREATE
 |
 | Set up a DirectSoundBuffer object
 |
 -----------------------------------------------------------}

constructor TDSLoopedSound.Create ( Frequency: Extended ) ;
var
  Buffer1, Buffer2: PDSStereo16Array ;
  BufSize1, BufSize2: DWORD ;
  Index: Integer ;
begin
  inherited Create ( Frequency ) ;

  { Lock the buffer }

  DXCheck ( SoundBuffer.Lock ( 0, 44100, Pointer ( Buffer1 ),
      BufSize1, Pointer ( Buffer2 ), BufSize2, 0 ) ) ;

  { Fill it with a sample }

  for Index := 0 to 11024 do
  with Buffer1 [ Index ] do
  begin
    Left := Round (
      Sin ( Index / Frequency )
      * 32000 ) ;
    Right := Left ;
  end ;

  { Unlock it }

  DXCheck ( SoundBuffer.Unlock ( Pointer ( Buffer1 ), BufSize1,
      Pointer ( Buffer2 ), BufSize2 ) ) ;
end ;

{-----------------------------------------------------------
 |
 | TDSLOOPEDSOUND.PLAY
 |
 | Play a DirectSoundBuffer
 |
 -----------------------------------------------------------}

procedure TDSLoopedSound.Play ;
begin
  SoundBuffer.Play ( 0, 0, DSBPLAY_LOOPING ) ;
end ;

{-----------------------------------------------------------
 |
 | TDSPOLYSOUND.CREATE
 |
 | Set up a DirectSoundBuffer object
 |
 -----------------------------------------------------------}

constructor TDSPolySound.Create ( Frequency: Extended ) ;
var
  Index: Integer ;
begin
  inherited Create ( Frequency ) ;
  for Index := LOW ( ExtraSoundBuffers ) to HIGH ( ExtraSoundBuffers ) do
    DSound.DuplicateSoundBuffer ( SoundBuffer, ExtraSoundBuffers [ Index ] ) ;
  NextBuffer := LOW ( ExtraSoundBuffers ) ;
end ;

{-----------------------------------------------------------
 |
 | TDSPOLYSOUND.DESTROY
 |
 | Clean up a DirectSoundBuffer object
 |
 -----------------------------------------------------------}

destructor TDSPolySound.Destroy ;
var
  Index: Integer ;
begin
  Stop ;
  for Index := LOW ( ExtraSoundBuffers ) to HIGH ( ExtraSoundBuffers ) do
  begin
    ExtraSoundBuffers [ Index ].Release ;
    ExtraSoundBuffers [ Index ] := nil ;
  end ;
  inherited Destroy ;
end ;

{-----------------------------------------------------------
 |
 | TDSPOLYSOUND.PLAY
 |
 | Play a DirectSoundBuffer
 |
 -----------------------------------------------------------}

procedure TDSPolySound.Play ;
begin
  ExtraSoundBuffers [ NextBuffer ].Play ( 0, 0, 0 ) ;
  Inc ( NextBuffer ) ;
  if NextBuffer > HIGH ( ExtraSoundBuffers ) then
    NextBuffer := LOW ( ExtraSoundBuffers ) ;
end ;

{-----------------------------------------------------------
 |
 | TDSPOLYSOUND.STOP
 |
 | Stop playing all DirectSoundBuffers
 |
 -----------------------------------------------------------}

procedure TDSPolySound.Stop ;
var
  Index: Integer ;
begin
  for Index := LOW ( ExtraSoundBuffers ) to HIGH ( ExtraSoundBuffers ) do
  begin
    if ExtraSoundBuffers [ Index ] <> nil then
      ExtraSoundBuffers [ Index ].Stop ;
  end ;
end ;

{-----------------------------------------------------------
 |
 | INITIALIZATION
 |
 | Set up the DirectSound object
 |
 -----------------------------------------------------------}

initialization
begin

  { Initialize the DirectSound system and
    set sound cooperation with other applications }

  DXCheck ( DirectSoundCreate ( nil, DSound, nil ) ) ;
  DXCheck ( DSound.SetCooperativeLevel ( Application.Handle,
      DSSCL_EXCLUSIVE ) ) ;
end ;

{-----------------------------------------------------------
 |
 | FINALIZATION
 |
 | Clean up the DirectSound object
 |
 -----------------------------------------------------------}

finalization
begin
  if DSound <> nil then DSound.Release ;
  DSound := nil ;
end ;

end.

