unit DIBUnit24;

{Author      Ian Hodger   21st June 1997  ianhodger@compuserve.com}

{The TDIBSection class as defined creates what is in effect a memory
bitmap with a 24bit colour depth, and provides methods with which to 
change the colour of individual pixels, at speed, and display the 
resulting image on any TCanvas}

{Please read the accompanying Readme file re. terms of use etc.}

interface

uses Windows,Classes,Graphics;

type
TDIBSection = class
            private
            Bitmap:HBITMAP;
            lpvbits:pointer;
            Bitmapinfo :PBitMapinfo;
            FWidth:integer;
            FHeight:integer;
            FSize:integer;
            FLineLen:integer;
            FLineAdj:integer;
            public
            constructor Create(ACanvas:TCanvas;Width,Height:integer);
            destructor Destroy;override;
            function Draw(ACanvas:TCanvas;X,Y:integer):integer;
            procedure FillRGB(Color:integer);
            procedure PlotRGB(X,Y,Color:integer);
            function GetColor(Red,Green,Blue:integer):integer;
            function GetPixel(X,Y:integer):integer;
            end;

{Standard class definition really, the integer fields are pretty straightforward
and are better explained in the constructor definition. Bitmap will hold the 
windows handle to the DIB we are going to create on start up, lpvbits holds 
the address of the pixel array, and BitmapInfo is the pointer to structure 
required by windows to create the DIB. Please note, there ain't much in the
way of error checking}

implementation

constructor TDIBSection.Create(ACanvas:TCanvas;Width,Height:integer);
begin
     FWidth := Width;
     FHeight := Height;
     FLineLen := 3*Width;
     FLineAdj := FLineLen mod 4;
     FLineAdj := 4 - FLineAdj;
     FLineAdj := FLineAdj mod 4;
     FLineLen := FLineLen + FLineAdj;
     FSize := FLineLen*Height;
{All the above is easy enough, however one should note that the length of
each scan line should be a multiple of four, ie. dword aligned. The field
FLineAdj was introduced because I thought I would need it. Might use it 
in the future}
     New(BitmapInfo);
     with BitmapInfo^.bmiHeader do
     begin
          bisize:=40;
          biWidth := FWidth;
          biHeight := FHeight;
          biPlanes := 1;
          biBitCount := 24;
          biCompression := BI_RGB;
          biSizeImage := FSize;
          biXPelsPerMeter := 0;
          biYPelsPerMeter := 0;
          biClrUsed := 0;
          biClrImportant := 0;
     end;
{Create windows compatible structure to define the properties of the 
resulting DIB. 24bit True Colour was chosen as this does not need
an accompanying palette structure. For speed, using the system
palette would be better but I needed true colour at the time}
     Bitmap := CreateDIBSection(ACanvas.Handle,BitmapInfo^,
               DIB_RGB_COLORS,lpvbits,nil,0);
{if sucessful, and it always has been so far, Bitmap holds the handle of the
DIB and lpvbits points to the address of the pixel array. The Canvas handle
is not used, and could be any valid handle, since DIB_RGB_COLORS was
used as the palette definition, ie. each pixel is an explicit RGB value}
end;

destructor TDIBSection.Destroy;
begin
     DeleteObject(Bitmap);
     Dispose(BitmapInfo);
{Tidy up the mess we've created}
end;

function TDIBSection.Draw(ACanvas:TCanvas;X,Y:integer):integer;
begin
     StretchDiBits(ACanvas.Handle,X,Y,FWidth,FHeight,0,0,FWidth,FHeight,lpvbits,
        BitmapInfo^,DIB_RGB_COLORS,SRCCOPY);
     Result := GetLastError;
{SetDIBits would be quicker, but if you have a look at the definition of TCanvas you''ll
see why I chosen StretchDIBits. See the Delphi/WinAPi helpfiles for more info}
end;

procedure TDIBSection.PlotRGB(X,Y,Color:integer);assembler;
asm
{start of the serious stuff}
   push ebx
   push ecx
{chuck ebx on the stack, save ecx or rather the value of Y, since ecx will be
changed soon, and make sure both X & Y are within the range of our memory 
bitmap before we cause an 0E memory exception error}
   cmp ecx,0
   jl @PlotRGBdone
   cmp ecx,[eax].FHeight
   jge @PlotRGBdone
   cmp edx,0
   jl @PlotRGBdone
   cmp edx,[eax].FWidth
   jge @PlotRGBdone
   mov ecx,[eax].Bitmap
   jecxz @PlotRGBdone
{if the handle Bitmap isn't zero then the DIB should have been created and it is
safe to continue}
   mov ecx,Color
   and ecx,00ffffffh
{Remember we are using 24bit Colours}
   pop ebx
   push ebx
   push eax
{Recap on what is what. ebx = Y, edx = X, ecx = Color, and eax is on the stack}
   mov eax,edx
   shl edx,1
   add edx,eax
{edx = 3*X}
   pop eax
   push eax
{Retrive Object base address}
   push edx
{save edx before it's abused by the mul operation}
   mov eax,[eax].FLineLen
   mul ebx
   pop edx
{Recap, eax = Y*FLineLen, edx = 3*X, ecx = Color}
   add edx,eax
   pop eax
   mov ebx,[eax].lpvbits
   add ebx,edx
{ebx = address of pixel we are after}
   mov eax,[ebx]
   and eax,0ff000000h
{We are applying three bytes of data, ie the Color, we must keep the highest byte
value as it is part of the RGB value of the next pixel}
   xor eax,ecx
   mov [ebx],eax
{Adjust lowest three bytes as defined by value Color, and set pixel value}
@PlotRGBdone:
   pop ecx
   pop ebx
{Clean up stack and restore ebx before exiting}
end;

function TDIBSection.GetColor(Red,Green,Blue:integer):integer;assembler;
asm
{Straightforward function, neither eax, nor ebx are affected, so need not be saved.
Parameters Red, Green and Blue must be in range 0-255, sorted out using binary
operations. Integer RGB Value returned}
   push ecx
   push edx
   mov ecx,Red
   and ecx,0ffh
   shl ecx,16
   xor edx,edx
   xor edx,ecx
   pop ecx
   push ecx
   and ecx,0ffh
   shl ecx,8
   xor edx,ecx
   mov ecx,Blue
   and ecx,0ffh
   xor edx,ecx
   mov @Result,edx
   pop edx
   pop ecx
end;

procedure TDIBSection.FillRGB(Color:integer);assembler;
asm
{This is the Plot routine nested in two loops. Range checking is not needed, but the
existance of a valid Bitmap handle is. The stack is used to preserve the 'variables'
controlling the loops. Since the data is three bytes in length it is not possible to use
rep stos unless the colour to be applied is a shade of grey. I have tried to use a 
32bit color definition to overcome this, but all my attempts thus far have resulted in
the need for a palette definition}
   push ebx
   mov ecx,[eax].Bitmap
   jecxz @FillRGBdone
   mov ecx,Color
   and ecx,00ffffffh
   xor ebx,ebx
   xor edx,edx
@FillRGB00:
   push eax
   push edx
   push ebx
   push eax
   mov eax,edx
   shl edx,1
   add edx,eax
   pop eax
   push eax
   push edx
   mov eax,[eax].FLineLen
   mul ebx
   pop edx
   add edx,eax
   pop eax
   mov ebx,[eax].lpvbits
   add ebx,edx
   mov eax,[ebx]
   and eax,0ff000000h
   xor eax,ecx
   mov [ebx],eax
   pop ebx
   pop edx
   pop eax
   inc edx
   cmp edx,[eax].FWidth
   jl @FillRGB00
   xor edx,edx
   inc ebx
   cmp ebx,[eax].FHeight
   jl @FillRGB00
   @FillRGBdone:
   pop ebx
end;

function TDIBSection.GetPixel(X,Y:integer):integer;assembler;
asm
{Same as Plot Routine except at the last moment}
   push ebx
   push ecx
   cmp ecx,0
   jl @GetPixeldone
   cmp ecx,[eax].FHeight
   jge @GetPixeldone
   cmp edx,0
   jl @GetPixeldone
   cmp edx,[eax].FWidth
   jge @GetPixeldone
   mov ecx,[eax].Bitmap
   jecxz @GetPixeldone
   pop ebx
   push ebx
   push eax
   mov eax,edx
   shl edx,1
   add edx,eax
   pop eax
   push eax
   push edx
   mov eax,[eax].FLineLen
   mul ebx
   pop edx
   add edx,eax
   pop eax
   mov ebx,[eax].lpvbits
   add ebx,edx
   mov eax,[ebx]
{This is where we differ from the plot routine, since we have only to retreive the 
pixel data remebering it is in 24bit format}
   and eax,00ffffffh
   mov @Result,eax
@GetPixeldone:
   pop ecx
   pop ebx
end;

end.
