{
  ISAPI and CGI interface classes

  Written by Blake Stone
  (c) 1996 by DKW Systems Corporation
}

unit DKWHTTP ;

interface

uses Windows, SysUtils, HTTPExt ;

{ Declare a standard set of base classes for HTTP extensions }

type

  { Generic HTTP Extension interface }

  THTTPExtension = class
  protected
    procedure Write ( var Buffer: Char ; Length: Integer ) ; virtual ; abstract ;
    function ServerVariable ( VariableName: String ): String ; virtual ; abstract ;
  public
    class function Description: String ; virtual ; abstract ;
    procedure HandleRequest ; virtual ;
    procedure Get ; virtual ; abstract ;
  end ;

  { ISAPI Extension interface }

  TISAPIExtension = class ( THTTPExtension )
  private
    ECB: ^TEXTENSION_CONTROL_BLOCK ;
  protected
    procedure Write ( var Buffer: Char ; Length: Integer ) ; override ;
    function ServerVariable ( VariableName: String ): String ; override ;
  public
    constructor Create ( var AECB: TEXTENSION_CONTROL_BLOCK ) ;
  end ;
  TISAPIExtensionClass = class of TISAPIExtension ;

  { CGI Extension interface }

  TCGIExtension = class ( THTTPExtension )
  private
    StdOut: THANDLE ;
  protected
    procedure Write ( var Buffer: Char ; Length: Integer ) ; override ;
    function ServerVariable ( VariableName: String ): String ; override ;
  public
    constructor Create ;
    destructor Destroy ; override ;
  end ;
  TCGIExtensionClass = class of TCGIExtension ;

implementation

{
  THTTPExtension

  Provide a generic interface for both CGI and ISAPI
}

{ Handle an HTTP request }

procedure THTTPExtension.HandleRequest ;
begin
  if ServerVariable ( 'REQUEST_METHOD' ) = 'GET' then
  begin
    Get ;
    Exit ;
  end ;
  raise Exception.Create ( 'Only GET is currently supported' ) ;
end ;

{
  TISAPIExtension

  Provide an interface with Microsoft's ISAPI specification for
  extensions written as native Delphi classes
}

{ ISAPI Extensions need an Extensinon_Control_Block that they use to
  look up values they need and perform callbacks }

constructor TISAPIExtension.Create ( var AECB: TEXTENSION_CONTROL_BLOCK ) ;
begin
  inherited Create ;
  ECB := @AECB ;
end ;

{ ISAPI Extensions write via a callback }

procedure TISAPIExtension.Write ( var Buffer: Char ; Length: Integer ) ;
begin
  ECB.WriteClient ( ECB.ConnID, @Buffer, Length, 0 ) ;
end ;

{ Standard CGI variable emulation }

function TISAPIExtension.ServerVariable ( VariableName: String ): String ;
var
  szVariable: Array [ 0..40 ] of Char ;
  szResult: Array [ 0..255 ] of Char ;
  Size: Integer ;
begin

  { Deal with the ones we already have prepared }

  if VariableName = 'QUERY_STRING' then
  begin
    Result := StrPas ( ECB.lpszQueryString ) ;
    Exit ;
  end ;

  if VariableName = 'PATH_INFO' then
  begin
    Result := StrPas ( ECB.lpszPathInfo ) ;
    Exit ;
  end ;

  if VariableName = 'PATH_TRANSLATED' then
  begin
    Result := StrPas ( ECB.lpszPathTranslated ) ;
    Exit ;
  end ;

  { Otherwise, fetch one through the callback }

  Size := sizeof ( szResult ) ;
  StrPCopy ( szVariable, VariableName ) ;
  Result := '' ;

  if ECB.GetServerVariable ( ECB.ConnID, szVariable, @szResult [ 0 ], Size ) then
    Result := StrPas ( szResult ) ;
end ;

{
  TCGIExtension

  Provide an interface with the CGI specification for
  extensions written as native Delphi classes
}

{ Keep a handle for STDOUT around while active }

constructor TCGIExtension.Create ;
begin
  inherited ;
  StdOut := GetStdHandle ( STD_OUTPUT_HANDLE ) ;
end ;

destructor TCGIExtension.Destroy ;
begin
  CloseHandle ( StdOut ) ;
  inherited ;
end ;

{ CGI Extensions write to STDOUT }

procedure TCGIExtension.Write ( var Buffer: Char ; Length: Integer ) ;
var
  Written: Integer ;
begin
  WriteFile ( StdOut, Buffer, Length, Written, nil ) ;
end ;

{ Standard CGI variable emulation }

function TCGIExtension.ServerVariable ( VariableName: String ): String ;
var
  szVariable: Array [ 0..40 ] of Char ;
  szResult: Array [ 0..255 ] of Char ;
  Size: Integer ;
begin
  Size := sizeof ( szResult ) ;
  StrPCopy ( szVariable, VariableName ) ;
  Result := '' ;
  if GetEnvironmentVariable ( szVariable, szResult, Size ) > 0 then
    Result := StrPas ( szResult ) ;
end ;

end.

