{
  NOTE:

  This is NOT a fully featured encoder.  It assumes a
  fixed 4-color palette and doesn't do anything other
  than 2-bit depth.  Most of the code IS generic, but
  any kind of palette resolution has been avoided for
  performance reasons.
}

unit GIFUtils;

interface

uses Windows, SysUtils ;

const
  MAX_LZW = 4095 ;

type
  TWriteProc = procedure ( var Buffer: Char ; Length: Integer ) of Object ;
  TGIFBytes = Array [ 0..0 ] of Byte ;
  TGIFCode = record
    Prefix: Integer ;
    Suffix: Integer ;
  end ;

  TGIFMap = class
  private
    Writer: TWriteProc ;
    Block: Array [ 0..255 ] of Byte ;
    BlockSize: Integer ;
    Blocking: Boolean ;
    LZWCodes: Array [ 0..MAX_LZW ] of TGIFCode ;
    LZWClear, LZWEnd, LZWNext, LZWPrior: Integer ;
    LZWBits, LZWByte, LZWBitPos: Byte ;
    LZWPriorWrite: Boolean ;

    procedure WriteHeader ;
    procedure WriteLogicalScreen ( Width, Height: Integer ) ;
    procedure WriteColorTable ;
    procedure WriteImageDescriptor ( Width, Height: Integer ) ;
    procedure WriteImage ;
    procedure WriteTerminator ;

    procedure StartLZW ;
    procedure StopLZW ;
    procedure ResetLZW ;
    procedure ClearLZW ;
    procedure WriteLZW ( Code: Integer ) ;
    procedure EncodeLZW ( Code: Integer ) ;

    procedure WriteBytes ( Bytes: String ) ;
    procedure BeginBlocking ;
    procedure EndBlocking ;
    procedure WriteBlock ;
  private
    Cache: ^TGIFBytes ;
    CacheWidth, CacheHeight: Integer ;
    procedure CheckPixelCache ;
    procedure SetPixel ( X, Y: Integer ; Value: Byte ) ;
    function GetPixel ( X, Y: Integer ): Byte ;
  public
    Width, Height: Integer ;
    destructor Destroy ; override ;
    procedure Encode ( AWriter: TWriteProc ) ;
    property Pixels [ X, Y: Integer ]: Byte
      read GetPixel write SetPixel ;
  end ;

implementation

const
  BITS_PER_PIXEL = 2 ;

  GIF_GLOBAL_COLOR_TABLE      = $80 ;
  GIF_256_COLORS              = $70 ;
  GIF_128_COLORS              = $60 ;
  GIF_64_COLORS               = $50 ;
  GIF_32_COLORS               = $40 ;
  GIF_16_COLORS               = $30 ;
  GIF_8_COLORS                = $20 ;
  GIF_4_COLORS                = $10 ;
  GIF_2_COLORS                = $00 ;
  GIF_GLOBAL_COLORS_SORTED    = $08 ;
  GIF_256_GLOBAL_COLORS       = $07 ;
  GIF_128_GLOBAL_COLORS       = $06 ;
  GIF_64_GLOBAL_COLORS        = $05 ;
  GIF_32_GLOBAL_COLORS        = $04 ;
  GIF_16_GLOBAL_COLORS        = $03 ;
  GIF_8_GLOBAL_COLORS         = $02 ;
  GIF_4_GLOBAL_COLORS         = $01 ;
  GIF_2_GLOBAL_COLORS         = $00 ;

  GIF_LOCAL_COLOR_TABLE       = $80 ;
  GIF_INTERLACED              = $40 ;
  GIF_LOCAL_COLORS_SORTED     = $20 ;
  GIF_256_LOCAL_COLORS        = $07 ;
  GIF_128_LOCAL_COLORS        = $06 ;
  GIF_64_LOCAL_COLORS         = $05 ;
  GIF_32_LOCAL_COLORS         = $04 ;
  GIF_16_LOCAL_COLORS         = $03 ;
  GIF_8_LOCAL_COLORS          = $02 ;
  GIF_4_LOCAL_COLORS          = $01 ;
  GIF_2_LOCAL_COLORS          = $00 ;

destructor TGIFMap.Destroy ;
begin
  if Cache <> nil then FreeMem ( Cache ) ;
  inherited ;
end ;

procedure TGIFMap.CheckPixelCache ;
begin
  if ( Cache = nil ) or ( CacheWidth <> Width ) or
      ( CacheHeight <> Height ) then
  begin
    if Cache <> nil then FreeMem ( Cache ) ;
    Cache := AllocMem ( Width * Height ) ;
    CacheWidth := Width ;
    CacheHeight := Height ;
  end ;
end ;

procedure TGIFMap.SetPixel ( X, Y: Integer ; Value: Byte ) ;
begin
  CheckPixelCache ;
  Cache [ Y * Width + X ] := Value ;
end ;

function TGIFMap.GetPixel ( X, Y: Integer ): Byte ;
begin
  CheckPixelCache ;
  result := Cache [ Y * Width + X ] ;
end ;

procedure TGIFMap.Encode ( AWriter: TWriteProc ) ;
begin
  Writer := AWriter ;

  WriteHeader ;
  WriteLogicalScreen ( Width, Height ) ;
  WriteColorTable ;

  WriteImageDescriptor ( Width, Height ) ;
  WriteImage ;
  WriteTerminator ;
end ;

procedure TGIFMap.WriteHeader ;
begin
  WriteBytes ( 'GIF' ) ;                          { Signature }
  WriteBytes ( '87a' ) ;                          { Version }
end ;

procedure TGIFMap.WriteLogicalScreen ( Width, Height: Integer ) ;
begin
  WriteBytes ( Char ( Width mod 256 ) +           { Width }
      Char ( Width div 256 ) ) ;
  WriteBytes ( Char ( Height mod 256 ) +          { Height }
      Char ( Height div 256 ) ) ;
  WriteBytes ( Char ( GIF_GLOBAL_COLOR_TABLE or   { Flags }
      GIF_4_COLORS or GIF_4_GLOBAL_COLORS ) ) ;
  WriteBytes ( Char ( 0 ) ) ;                     { Background Color }
  WriteBytes ( Char ( 0 ) ) ;                     { Pixel aspect ratio }
end ;

procedure TGIFMap.WriteColorTable ;
begin
  WriteBytes ( Char ( 0 ) + Char ( 0 ) +          { Dark blue }
      Char ( 99 ) ) ;
  WriteBytes ( Char ( 85 ) + Char ( 85 ) +        { 33% mix }
      Char ( 66 ) ) ;
  WriteBytes ( Char ( 170 ) + Char ( 170 ) +      { 66% mix }
      Char ( 33 ) ) ;
  WriteBytes ( Char ( 255 ) + Char ( 255 ) +      { Yellow }
      Char ( 0 ) ) ;
end ;

procedure TGIFMap.WriteImageDescriptor ( Width, Height: Integer ) ;
begin
  WriteBytes ( Char ( $2C ) ) ;                   { Image desc. ID }
  WriteBytes ( Char ( 0 ) + Char ( 0 ) ) ;        { Image left }
  WriteBytes ( Char ( 0 ) + Char ( 0 ) ) ;        { Image top }
  WriteBytes ( Char ( Width mod 256 ) +           { Width }
      Char ( Width div 256 ) ) ;
  WriteBytes ( Char ( Height mod 256 ) +          { Height }
      Char ( Height div 256 ) ) ;
  WriteBytes ( Char ( 0 ) ) ;                     { Flags }
end ;

procedure TGIFMap.WriteImage ;
var
  Index: Integer ;
begin
  WriteBytes ( Char ( BITS_PER_PIXEL ) ) ;         { LZW code size - 1 }
  BeginBlocking ;
  StartLZW ;
  for Index := 0 to ( Height * Width ) - 1 do
  begin
    WriteLZW ( Pixels [ Index mod Width, Index div Width ] ) ;
  end ;
  StopLZW ;
  EndBlocking ;
end ;

procedure TGIFMap.StartLZW ;
begin

  { Set up LZW contants }

  LZWClear := 1 shl BITS_PER_PIXEL ;
  LZWEnd := LZWClear + 1 ;

  { Initialize the bitpattern buffer }

  LZWByte := 0 ;
  LZWBitPos := 0 ;

  { Reset the LZW table and encode and initial clear }

  ResetLZW ;
  EncodeLZW ( LZWClear ) ;
end ;

procedure TGIFMap.StopLZW ;
begin

  { Clean out the pattern pipeline }

  if LZWPrior <> -1 then EncodeLZW ( LZWPrior ) ;

  { Encode the final bitpattern }

  EncodeLZW ( LZWEnd ) ;

  { Flush the bitpattern buffer }

  if LZWBitPos > 0 then WriteBytes ( Char ( LZWByte ) ) ;
end ;

procedure TGIFMap.ResetLZW ;
begin

  { Reset the pattern pipeline }

  LZWPrior := -1 ;

  { Reduce LZW table to bare minimum }

  LZWNext := LZWEnd + 1 ;
  LZWBits := 1 ;
  while ( ( LZWNext - 1 ) shr LZWBits ) > 0 do Inc ( LZWBits ) ;
end ;

procedure TGIFMap.ClearLZW ;
begin

  { Encode a clear at the current bit size }

  EncodeLZW ( LZWClear ) ;

  { Reset the LZW table }

  ResetLZW ;
end ;

procedure TGIFMap.WriteLZW ( Code: Integer ) ;
var
  Found: Integer ;
begin

  { Always match a simple prefix }

  if LZWPrior = -1 then
  begin
    LZWPrior := Code ;
    Exit ;
  end ;

  { Look for a matching pattern (and use a shortcut for the search) }

  Found := LZWPrior ;
  if Found < LZWEnd + 1 then Found := LZWEnd + 1 ;
  while ( Found < LZWNext ) do
  begin

    { If found, keep it and continue }

    if ( LZWCodes [ Found ].Prefix = LZWPrior ) and
        ( LZWCodes [ Found ].Suffix = Code ) then
    begin
      LZWPrior := Found ;
      Exit ;
    end ;

    { Otherwise, keep looking }

    Inc ( Found ) ;
  end ;

  { Generate the previous code }

  EncodeLZW ( LZWPrior ) ;

  { Create the new pattern entry }

  LZWCodes [ LZWNext ].Prefix := LZWPrior ;
  LZWCodes [ LZWNext ].Suffix := Code ;
  Inc ( LZWNext ) ;
  while ( ( LZWNext - 1 ) shr LZWBits ) > 0 do Inc ( LZWBits ) ;

  { Test for LZW table overrun and clear the table if necessary }

  if LZWNext = MAX_LZW then ClearLZW ;

  { Start the search again }

  LZWPrior := Code ;
end ;

procedure TGIFMap.EncodeLZW ( Code: Integer ) ;
var
  BitsToWrite: Byte ;
  BitsWritten: Byte ;
begin
  BitsToWrite := LZWBits ;
  while ( BitsToWrite > 0 ) do
  begin

    { Write into the byte }

    LZWByte := LZWByte or ( Code shl LZWBitPos ) ;

    { Determine how many bits were written }

    BitsWritten := 8 - LZWBitPos ;
    if BitsWritten > BitsToWrite then BitsWritten := BitsToWrite ;

    { Find out if we need to write the results }

    LZWBitPos := LZWBitPos + BitsWritten ;
    if LZWBitPos > 8 then raise Exception.Create ( 'LZW Encode Logic Error.' ) ;
    if LZWBitPos = 8 then
    begin
      WriteBytes ( Char ( LZWByte ) ) ;
      LZWByte := 0 ;
      LZWBitPos := 0 ;
    end ;

    { Prepare to keep writing, if necessary }

    BitsToWrite := BitsToWrite - BitsWritten ;
    Code := Code shr BitsWritten ;
  end ;
end ;

procedure TGIFMap.WriteTerminator ;
begin
  WriteBytes ( Char ( $3B ) ) ;                    { Terminator }
end ;

procedure TGIFMap.WriteBytes ( Bytes: String ) ;
begin
  if Blocking then
  begin
    if length ( Bytes ) > 255 then raise Exception.Create (
        'Block writes must be less 256 bytes' ) ;
    if length ( Bytes ) + BlockSize > 255 then WriteBlock ;
    StrPCopy ( @Block [ BlockSize ], Bytes ) ;
    BlockSize := BlockSize + Length ( Bytes ) ;
    Exit ;
  end ;

  Writer ( Bytes [ 1 ], length ( Bytes ) ) ;
end ;

procedure TGIFMap.BeginBlocking ;
begin
  if Blocking then raise Exception.Create ( 'Already writing a block!' ) ;
  Blocking := True ;
  BlockSize := 0 ;
end ;

procedure TGIFMap.EndBlocking ;
begin
  if not Blocking then raise Exception.Create ( 'Not writing a block!' ) ;

  WriteBlock ;
  Blocking := False ;
  WriteBytes ( Char ( 0 ) ) ;
end ;

procedure TGIFMap.WriteBlock ;
begin
  if not Blocking then raise Exception.Create ( 'Not writing a block!' ) ;
  if BlockSize = 0 then Exit ;

  Writer ( Char ( BlockSize ), 1 ) ;
  Writer ( Char ( Block [ 0 ] ), BlockSize ) ;
  BlockSize := 0 ;
end ;

end.

