unit DynArray;

interface

const
   MAX_COORDS = 3000;

type
   tCoord = record
      y, x, z : Double;
   end;
   tCoordArray = array[1..MAX_COORDS] of tCoord;

   tCoordsArray = class( TObject )
   private
      fLowerBound : integer;
      fUpperBound : integer;
      fFileName : string;
      fCoords : ^tCoordArray;
      function getCoord(  i : Integer ) : tCoord;
      procedure setCoord(  i : Integer;  coord : tCoord );
      procedure writeCoord( const i : Integer );
   protected

   public
      // Create a new file:
      constructor createNew( const iSize : Integer; const sFName : string);
      // Create from a file:
      constructor createFromFile( const sFName : string );
      destructor destroy; override;
      function isValid( const i : Integer ) : boolean;
      function isValidElev ( const i: Integer ) : boolean;
      function print( const i : Integer ) : string;
      property coord[i: Integer]: TCoord
         read GetCoord write SetCoord; default;
      property upperBound : Integer read fUpperBound;

   published
      // As this is not an interface object, it doesn't has
      //   published properties.
   end;

implementation

uses
   SysUtils;

{$I Dynamic.Inc}
{$R 'Dynamic.Res'}

const
	COOR_YX = 'Coor Y %12.4n  Coor X %12.4n';
	COOR_YXZ = COOR_YX + '  Elev %9.3n';

   (*** Constants for minimum and maximun ***)
	COOR_MIN = 0.001;
   COOR_MAX = 150000.0;
	ELEV_MIN = -100.0;
   ELEV_MAX = 1500.0;

type
   eCoordArray = class(Exception);

function tCoordsArray.getCoord( i: Integer ) : tCoord;
begin
   if i < fLowerBound then
      raise eCoordArray.CreateResFmt( E_LOWER_BOUND, [fLowerBound] )
   else if fUpperBound < i then
      raise eCoordArray.CreateResFmt( E_UPPER_BOUND, [fUpperBound] );
   // Return the coords at i:
   Result := fCoords^[ i ];
end;

procedure tCoordsArray.setCoord( i: Integer; coord : tCoord );
begin
   if i < fLowerBound then
      raise eCoordArray.CreateResFmt( E_LOWER_BOUND, [fLowerBound] );
   if fUpperBound < i then
      raise eCoordArray.CreateResFmt( E_UPPER_BOUND, [fUpperBound] );
   with coord do
   begin
      if ( x = 0.0 ) and ( y = 0.0 ) then
      begin
         if ( z <> 0.0 ) then
            eCoordArray.CreateRes( E_INVALID_COORDS );
      end
      else
      begin
         if ( x = 0.0 ) or ( y = 0.0 ) then
            eCoordArray.CreateRes( E_INVALID_COORDS );
         if ( x < COOR_MIN ) or ( COOR_MAX < x ) then
            eCoordArray.CreateRes( E_INVALID_X );
         if ( y < COOR_MIN ) or ( COOR_MAX < y ) then
            eCoordArray.CreateRes( E_INVALID_Y );
         if ( z < ELEV_MIN ) or ( ELEV_MAX < z ) then
            eCoordArray.CreateRes( E_INVALID_ELEV );
      end;
   end;
   // Ok, the coords are good, save it
   fCoords^[i] := coord;
   writeCoord( i );
end;

procedure tCoordsArray.writeCoord( const i : Integer );
var
   fHandle : file of tCoord;
begin
   AssignFile( fHandle, fFileName );
   Reset( fHandle );
   try
      // Seek is zero based:
      Seek( fHandle, i - fLowerBound );
      write( fHandle, fCoords^[i] );
   finally
      close( fHandle );
   end;
end;

constructor tCoordsArray.createNew( const iSize : Integer; const sFName : string);
var
   fHandle : file;
   test: integer;
begin
   inherited Create;

   if fCoords <> nil then
      raise eCoordArray.CreateRes( E_EXISTING );

   if MAX_COORDS < iSize then
      raise eCoordArray.CreateResFmt( E_MAX_COORDS, [MAX_COORDS] );

   AssignFile( fHandle, sFName );
   Rewrite( fHandle, SizeOf( tCoord ) );
   try
      fCoords := AllocMem( iSize * SizeOf( tCoord ));
      try
         // Record size is tCoord:
         BlockWrite( fHandle, fCoords^, iSize );
         fLowerBound := 1;
         fUpperBound := iSize;
         fFileName := sFName;
      except
         else
         begin
            FreeMem( fCoords );
            fCoords := nil;
            raise;
         end;
      end;
   finally
      CloseFile( fHandle );
   end;
end;

// Create from a file:
constructor tCoordsArray.createFromFile( const sFName : string );
var
   fHandle : file;
   iSize : Integer;
begin
   if fCoords <> nil then
      raise eCoordArray.CreateRes( E_EXISTING );

   inherited Create;

   AssignFile( fHandle, sFName );
   // Record size must be 1 to read all the file:
   Reset( fHandle, 1 );
   try
      iSize := FileSize( fHandle );
      GetMem( fCoords, iSize );
      try
         BlockRead( fHandle, fCoords^, iSize );
         fLowerBound := 1;
         fUpperBound := iSize div SizeOf( tCoord );
         fFileName := sFName;
      except
         else
         begin
            FreeMem( fCoords );
            fCoords := nil;
            raise;
         end;
      end;
   finally
      CloseFile( fHandle );
   end;
end;

destructor tCoordsArray.destroy;
begin
   if fCoords <> nil then
      FreeMem( fCoords );
   inherited;
end;

function tCoordsArray.isValid( const i : Integer ) : boolean;
begin
   with fCoords^[i] do
      Result := ( x <> 0.0 ) and ( y <> 0.0 );
end;

function tCoordsArray.isValidElev ( const i: Integer ) : boolean;
begin
   with fCoords^[i] do
      Result := z <> 0.0;
end;

function tCoordsArray.print( const i : Integer ) : string;
begin
   with fCoords^[i] do
   begin
      if isValid( i ) then
         begin
            if isValidElev( i ) then
               Result := Format( COOR_YXZ, [ y, x, z ] )
            else
               Result := Format( COOR_YX, [ y, x ] );
         end
      else
         Result := '';
   end;
end;

end.
