{===EZDSLLST==========================================================

Part of the Delphi Structures Library--the single linked list.

EZDSLLST is Copyright (c) 1993, 1996 by  Julian M. Bucknall

VERSION HISTORY
13Mar96 JMB 2.00 release for Delphi 2.0
12Nov95 JMB 1.01 fixed Iterate bug
18Jun95 JMB 1.00 conversion of EZStrucs to Delphi
======================================================================}
{ Copyright (c) 1993, 1996, Julian M. Bucknall. All Rights Reserved   }

unit EZDSLLst;

{$I EZDSLDEF.INC}
{---Place any compiler options you require here-----------------------}


{---------------------------------------------------------------------}
{$I EZDSLOPT.INC}

interface

uses
  SysUtils,
  WinTypes,
  WinProcs,
  Classes,
  EZDSLCts,
  EZDSLSup,
  EZDSLBse;

type
  TLinkList = class(TAbstractContainer)
    {-Single linked list object}
    private
      FIsSorted : boolean;
      Cursor, BF, AL  : PNode;

    public
      constructor Create(DataOwner : boolean); override;
      constructor Clone(Source : TAbstractContainer;
                        DataOwner : boolean; NewCompare : TCompareFunc); override;

      procedure Delete;
      procedure Empty; override;
      procedure Erase;
      function  Examine : pointer;
      procedure InsertAfter(aData : pointer);
      procedure InsertBefore(aData : pointer);
      procedure InsertSorted(aData : pointer);
      function  IsAfterLast : boolean;
      function  IsBeforeFirst : boolean;
      function  Iterate(Action : TIterator; Backwards : boolean;
                        ExtraData : pointer) : pointer;
      procedure Join(List : TLinkList);
      procedure Next;
      procedure Prev;
      function  Replace(aData : pointer) : pointer;
      function  Search(aData : pointer) : boolean;
      procedure SetBeforeFirst;
      procedure SetAfterLast;
      function  Split : TLinkList;

      {properties}
      property IsSorted: boolean
         read FIsSorted;
  end;

implementation

{-An iterator for cloning a single linked list}
function SListClone(SL : TAbstractContainer;
                    aData : pointer;
                    NSL : pointer) : boolean; far;
  var
    NewList : TLinkList absolute NSL;
    NewData : pointer;
  begin
    {Note: assumes that NewList.IsAfterLast is true}
    Result := true;
    with NewList do
      begin
        if IsDataOwner then
             NewData := DupData(aData)
        else NewData := aData;
        try
          InsertBefore(NewData);
        except
          DisposeData(NewData);
          raise;
        end;
      end;
  end;

{-An iterator for cloning a SORTED single linked list}
function SListSortedClone(SL : TAbstractContainer;
                          aData : pointer;
                          NSL : pointer) : boolean; far;
  var
    NewList : TLinkList absolute NSL;
    NewData : pointer;
  begin
    Result := true;
    with NewList do
      begin
        if IsDataOwner then
             NewData := DupData(aData)
        else NewData := aData;
        try
          InsertSorted(NewData);
        except
          DisposeData(NewData);
          raise;
        end;
      end;
  end;

{=TLinkList===========================================================}
constructor TLinkList.Create(DataOwner : boolean);
  begin
    NodeSize := 8;
    inherited Create(DataOwner);
    BF := acNewNode(nil);
    FCount := 0;
    AL := acNewNode(nil);
    FCount := 0;
    BF^.Link := AL;
    AL^.Link := nil;
    Cursor := BF;
    FIsSorted := true;
  end;
{--------}
constructor TLinkList.Clone(Source : TAbstractContainer;
                            DataOwner : boolean;
                            NewCompare : TCompareFunc);
  var
    OldList : TLinkList absolute Source;
  begin
    Create(DataOwner);
    Compare := NewCompare;
    DupData := OldList.DupData;
    DisposeData := OldList.DisposeData;

    if not (Source is TLinkList) then
      RaiseError(escBadSource);

    if OldList.IsEmpty then Exit;

    SetAfterLast;
    if OldList.IsSorted then
         OldList.Iterate(SListSortedClone, false, Self)
    else OldList.Iterate(SListClone, false, Self);
  end;
{--------}
procedure TLinkList.Delete;
  var
    Temp : PNode;
  begin
    {$IFDEF DEBUG}
    Assert((not IsBeforeFirst) and (not IsAfterLast), ascDeleteEdges);
    {$ENDIF}
    Temp := Cursor^.Link;
    acDisposeNode(Cursor);
    Cursor := BF^.Link;
    BF^.Link := Cursor^.Link;
    Cursor^.Link := Temp;
    if IsEmpty then
      FIsSorted := true;
  end;
{--------}
procedure TLinkList.Empty;
  begin
    if not IsEmpty then
      begin
        if IsBeforeFirst then
          Next;
        while not IsAfterLast do
          Erase;
        while not IsEmpty do
          begin
            Prev;
            Erase;
          end;
      end;
    if InDone then
      begin
        if Assigned(BF) then
          acDisposeNode(BF);
        if Assigned(AL) then
          acDisposeNode(AL);
      end
    else
      begin
        BF^.Link := AL;
        AL^.Link := nil;
        Cursor := BF;
      end;
  end;
{--------}
procedure TLinkList.Erase;
  begin
    if IsDataOwner then
      DisposeData(Examine);
    Delete;
  end;
{--------}
function  TLinkList.Examine : pointer;
  begin
    {$IFDEF DEBUG}
    Assert((not IsBeforeFirst) and (not IsAfterLast), ascExamineEdges);
    {$ENDIF}
    Examine := Cursor^.Data;
  end;
{--------}
procedure TLinkList.InsertAfter(aData : pointer);
  var
    Node : PNode;
  begin
    {$IFDEF DEBUG}
    Assert(not IsAfterLast, ascInsertEdges);
    {$ENDIF}
    Node := acNewNode(aData);
    Node^.Link := BF^.Link;
    BF^.Link := Node;
    FIsSorted := false;
  end;
{--------}
procedure TLinkList.InsertBefore(aData : pointer);
  var
    Node : PNode;
  begin
    {$IFDEF DEBUG}
    Assert(not IsBeforeFirst, ascInsertEdges);
    {$ENDIF}
    Node := acNewNode(aData);
    Node^.Link := Cursor^.Link;
    Cursor^.Link := Node;
    FIsSorted := false;
  end;
{--------}
procedure TLinkList.InsertSorted(aData : pointer);
  begin
    if not IsSorted then
      if IsBeforeFirst then
           InsertAfter(aData)
      else InsertBefore(aData)
    else {the list is sorted}
      begin
        if Search(aData) then
          RaiseError(escInsertDup)
        else
          begin
            InsertBefore(aData);
            FIsSorted := true;
          end;
      end;
  end;
{--------}
function  TLinkList.IsAfterLast : boolean;
  begin
    IsAfterLast := (Cursor = AL);
  end;
{--------}
function  TLinkList.IsBeforeFirst : boolean;
  begin
    IsBeforeFirst := (Cursor = BF);
  end;
{--------}
function  TLinkList.Iterate(Action : TIterator; Backwards : boolean;
                            ExtraData : pointer) : pointer;
  begin
    if Backwards then
      begin
        SetAfterLast;
        Prev;
        while not IsBeforeFirst do
          if Action(Self, Examine, ExtraData) then              {!!.01}
            Prev
          else
            begin
              Result := Examine;
              Exit;
            end;
      end
    else
      begin
        SetBeforeFirst;
        Next;
        while not IsAfterLast do
          if Action(Self, Examine, ExtraData) then              {!!.01}
            Next
          else
            begin
              Result := Examine;
              Exit;
            end;
      end;
    Result := nil;                                              {!!.01}
  end;
{--------}
procedure TLinkList.Join(List : TLinkList);
  var
    JoinNode : PNode;
    Data     : pointer;
  begin
    {$IFDEF DEBUG}
    Assert(not IsAfterLast, ascCannotJoinHere);
    Assert(List.IsDataOwner = IsDataOwner, ascCannotJoinData);
    {$ENDIF}
    if not Assigned(List) then Exit;

    if not List.IsEmpty then
      begin
        {prepare}
         with List do
           begin
             SetBeforeFirst;
             Next;
           end;
        {if we are sorted, add new nodes in sorted order}
        if {Self.}IsSorted then
          while not List.IsAfterLast do
            begin
              Data := List.Examine;
              List.Delete;
              InsertSorted(Data);
            end
        {if we are not sorted, add new nodes directly}
        else {Self is unsorted}
          begin
            JoinNode := List.Cursor;
            with List do
              begin
                SetAfterLast;
                Prev;
              end;
            JoinNode^.Link := Cursor;
            Cursor := List.Cursor;
            inc(FCount, List.Count);
            {patch up List to be empty}
            with List do
              begin
                Cursor := BF;
                FCount := 0;
              end;
          end;
      end;
    List.Free;
  end;
{--------}
procedure TLinkList.Next;
  var
    Temp : PNode;
  begin
    {$IFDEF DEBUG}
    Assert(not IsAfterLast, ascAlreadyAtEnd);
    {$ENDIF}
    Temp := Cursor;
    Cursor := BF^.Link;
    BF^.Link := Cursor^.Link;
    Cursor^.Link := Temp;
  end;
{--------}
procedure TLinkList.Prev;
  var
    Temp : PNode;
  begin
    {$IFDEF DEBUG}
    Assert(not IsBeforeFirst, ascAlreadyAtStart);
    {$ENDIF}
    Temp := Cursor^.Link;
    Cursor^.Link := BF^.Link;
    BF^.Link := Cursor;
    Cursor := Temp;
  end;
{--------}
function  TLinkList.Replace(aData : pointer) : pointer;
  begin
    {$IFDEF DEBUG}
    Assert((not IsBeforeFirst) and (not IsAfterLast), ascReplaceEdges);
    {$ENDIF}
    if IsSorted then
      begin
        Replace := Examine;
        Delete;
        InsertSorted(aData);
      end
    else
      with Cursor^ do
        begin
          Replace := Data;
          Data := aData;
        end;
  end;
{--------}
function TLinkList.Search(aData : pointer) : boolean;
  var
    CompResult   : integer;
    MoveForwards : boolean;
    StillLooking : boolean;
    Found        : boolean;
  begin
    if IsSorted then
      begin
        if IsBeforeFirst then
          MoveForwards := true
        else if IsAfterLast then
          MoveForwards := false
        else
          begin
            CompResult := Compare(aData, Examine);
            if (CompResult = 0) then
              begin
                Search := true;
                Exit;
              end;
            MoveForwards := (CompResult > 0);
          end;
        if MoveForwards then
          begin
            CompResult := 1;
            while (CompResult > 0) do
              begin
                Next;
                if IsAfterLast then
                     CompResult := -1
                else CompResult := Compare(aData, Examine);
              end;
            Search := CompResult = 0;
          end
        else {move backwards}
          begin
            CompResult := -1;
            while (CompResult < 0) do
              begin
                Prev;
                if IsBeforeFirst then
                     CompResult := +1
                else CompResult := Compare(aData, Examine);
              end;
            if (CompResult = 0) then
              Search := true
            else
              begin
                Next;
                Search := false;
              end;
          end;
      end
    else {the list is not currently sorted, search from the start}
      begin
        SetBeforeFirst;
        StillLooking := true;
        Found := false;
        while StillLooking and (not Found) do
          begin
            Next;
            if IsAfterLast then
                 StillLooking := false
            else Found := (Compare(aData, Examine) = 0);
          end;
        Search := Found;
      end;
  end;
{--------}
procedure TLinkList.SetAfterLast;
  var
    NextLink,
    Temp : PNode;
  begin
    {for speed reasons, code from first principles,
     this is equivalent to:
       while not IsAfterLast do Next;}
    NextLink := BF^.Link;
    while (Cursor <> AL) do
      begin
        Temp := Cursor;
        Cursor := NextLink;
        NextLink := Cursor^.Link;
        Cursor^.Link := Temp;
      end;
    BF^.Link := NextLink;
  end;
{--------}
procedure TLinkList.SetBeforeFirst;
  var
    NextLink,
    Temp : PNode;
  begin
    {for speed reasons, code from first principles,
     this is equivalent to:
       while not IsBeforeFirst do Prev;}
    NextLink := BF^.Link;
    while (Cursor <> BF) do
      begin
        Temp := Cursor^.Link;
        Cursor^.Link := NextLink;
        NextLink := Cursor;
        Cursor := Temp;
      end;
    BF^.Link := NextLink;
  end;
{--------}
function  TLinkList.Split : TLinkList;
  var
    TempCount : longint;
    NewList   : TLinkList;
    LastNodeLeftBehind,
    JoinNode  : PNode;
  begin
    {$IFDEF DEBUG}
    Assert((not IsBeforeFirst) and (not IsAfterLast), ascSplitEdges);
    {$ENDIF}

    NewList := TLinkList(TAbstractContainerClass(ClassType).Create(IsDataOwner));
    NewList.Compare := Compare;
    NewList.DupData := DupData;
    NewList.DisposeData := DisposeData;
    Result := NewList;

    LastNodeLeftBehind := Cursor^.Link;

    TempCount := 0;
    JoinNode := Cursor;
    while not IsAfterLast do
      begin
        inc(TempCount);
        Next;
      end;

    JoinNode^.Link := NewList.BF;
    NewList.Cursor := AL^.Link;
    NewList.Next;
    NewList.FCount := TempCount;
    NewList.FIsSorted := IsSorted;

    dec(FCount, TempCount);
    AL^.Link := LastNodeLeftBehind;
    if IsEmpty then
      FIsSorted := true;
  end;
{---------------------------------------------------------------------}

end.
