unit FilterDemo;

{FilterDemo by
 Atul Vaidya
 Compuserve: 100025.3466
 Internet: clgs20@strath.ac.uk
 This application demonstrates how a single Filter object can be
 used to deal with the process of validating keypresses in a text
 box, a grid, or any other component that takes keyboard input.

 FILTER.DCU is freeware and you may use it in any way you wish.
 If you can think of any improvements, please let me know.

 To purchase the source code for FILTER.DCU go SWREG 15394.

 The FILTER unit contains the definitions of the various Filter
 objects that are used for this purpose. All the Filters are
 derived from TFilter which has the following key methods & fields.

  TextLen - records the length of the current text entry.
  Key - used to store the current keypress.
  KeyText - used to store the current text string
  *These fields are private to TFilter*

  MaxLen: This is a public field. When TFilter.Create is called
  this is set to 10. It is used to determine the maximum length
  of the entry that is permitted by the Filter.

  Create - this method creates the filter object and sets its
  MaxLen field.

  Initialize(FilterType,MLen:Byte;OutKeys:Set of Char;KeyText:String)

  Call this method immediately after creating the Filter object.

  FilterType - this parameter is used to assign procedures for
  handling various types of keypresses. It must have a value in
  the range [0..5] - the various filter constants listed below.

  MLen - This parameter is used to set the MaxLen field of the
  filter object.

  OutKeys - this parameter is used to initialize the text filter.
  Use it to define the subset of keys, if any, which you wish to
  disallow in the text filter.  For instance, ['0'..'9'] will
  prevent any numbers from being typed in.

  KeyText - when you move to a new keyboard input component,
  such as a text box or a grid, you may wish the next keypress
  to append existing text in the component.  To ensure that this
  happens correctly pass the relevant string, the Text property
  of an edit box or the Cells[Col,Row] property of a string grid,
  as this parameter value. If you want the existing text to be
  overwritten pass a null string.

  ReSet - this method resets all the fields of the filter object.
  It also returns a null string which can be used to update the
  relevant property of the keyboard input component. You could,
  for instance, call this method to deal with the pressing of the
  Delete key in order to annul the entire text string in the
  keyboard input component.

  Validate(Key) - call this method from the keypress event of
  your keyboard input component. Once the filter has been created
  and initilized the Validate method will correctly deal with the
  keypress and return an updated string for display in the
  component. After calling this method you should set the Key
  parameter in the calling KeyPress event to #0 in order to
  prevent the keypress from being echoed.}

interface

uses
  Windows,Messages,SysUtils,Classes,Graphics,Controls,Forms,Dialogs,
  Filter,StdCtrls, Grids;

{These constants identify the various types of filter objects
defined in FILTER.DCU. The four text boxes in this form have
their tag property set to 0,1,2 and 4. The grid on this form
has 6 rows. Moving between rows changes the filter assignment to
Row - 1 = [0..5].}
const ftFloat=0;ftInteger=1;ftCardinal = 2;ftNone=3;ftText=4;
ftToggle=5;

type
  TMaster = class(TForm)
    CloseBtn: TButton;
    NumberBox: TEdit;{Tag = 0. Allows floating point numbers}
    TextBox: TEdit; {Tag = 4. Allows text characters}
    IntegerBox: TEdit;{Allows any integral value}
    CardinalBox: TEdit;{Allows positive integers only}
    FilterGrid: TStringGrid;{Each row has a different filter type}
    procedure FormCloseQuery(Sender: TObject; var CanClose: Boolean);
    procedure CloseBtnClick(Sender: TObject);
    procedure BoxesEnter(Sender: TObject);
    procedure BoxesKeyPress(Sender: TObject; var Key: Char);
    procedure FormCreate(Sender: TObject);
    procedure FilterGridEnter(Sender: TObject);
    procedure FilterGridKeyPress(Sender: TObject; var Key: Char);
    procedure FilterGridKeyUp(Sender: TObject; var Key: Word;
      Shift: TShiftState);
    procedure FilterGridKeyDown(Sender: TObject; var Key: Word;
      Shift: TShiftState);
    procedure FilterGridClick(Sender: TObject);
  private
    {TFilter is the base object from which the other filters are
     derived. When we move from one text box to another we check
     its tag and create the appropriate type of filter and assign
     it to the Filter field. Likewise, when we move between rows
     in the grid we use the Row property to create the correct
     filter and assign it to the Filter field.}
    Filter:TFilter;
    procedure UpDateFilter;
  public
  end;

var
  Master:TMaster;

implementation

{$R *.DFM}

procedure TMaster.FormCloseQuery(Sender:TObject;var CanClose:Boolean);
begin
  {Free the Filter object if it exists.}
  if Assigned(Filter) then Filter.Free;
end;

procedure TMaster.CloseBtnClick(Sender: TObject);
begin
  Close;
end;

procedure TMaster.BoxesEnter(Sender:TObject);
var i:Integer;
    KeyText:String;
begin
{This routine is shared by all the text boxes. Upon entering a
 text box we retrieve the text in it and check its Tag property.
 We also annul the current assignment of the Filter field. Then
 we use the Tag property to reassign the appropriate filter.
 Finally, we initialize the Filter object so that it knows about
 the existing text in the current text box.}
  if not(Filter = nil) then Filter.Free;
  Filter:=nil;
  i:=TEdit(Sender).Tag;
  KeyText:=TEdit(Sender).Text;
  case i of
    ftFloat..ftCardinal:Filter:=TNumFilter.Create;
    ftText:Filter:=TTextFilter.Create;
  end;
  if not(Filter = nil) then
  Filter.Initialize(i,10,['$','&'],KeyText);
end;

procedure TMaster.BoxesKeyPress(Sender:TObject;var Key:Char);
begin
  {The Validate method of the Filter object deals with the key
  press.  Once this is done the keystroke is annuled to prevent
  it from being echoed.}
  TEdit(Sender).Text:=Filter.Validate(Key);
  Key:=#0;
end;

procedure TMaster.FormCreate(Sender: TObject);
begin
  FilterGrid.Cells[0,0]:='              Entry';
end;

procedure TMaster.FilterGridEnter(Sender:TObject);
begin
  {On entering the grid, i.e., when the user clicks on it, we
  must recrete the filter assignment.}
  UpDateFilter;
end;

procedure TMaster.FilterGridKeyPress(Sender: TObject; var Key:Char);
begin
  {As with the text boxes we deal with the keypress by calling the
  Validate method of the Filter object and then annul the keypress}
  with FilterGrid do Cells[Col,Row]:=Filter.Validate(Key);
  Key:=#0;
end;

procedure TMaster.FilterGridKeyUp(Sender: TObject; var Key: Word;
  Shift: TShiftState);
begin
{If the user navigates in the grid by using the cursor keys the
 Filter must be reassigned. This MUST be done on the KeyUp event
 since the Row property of the grid is not updated until this
 event occurs.}
  case Key of
    VK_UP,VK_DOWN:UpDateFilter;
  end;
end;

procedure TMaster.FilterGridClick(Sender: TObject);
begin
  {The user may have clicked on a different row so we must update
   the filter.}
  UpDateFilter;
end;

procedure TMaster.FilterGridKeyDown(Sender: TObject; var Key: Word;
  Shift: TShiftState);
begin
  {When the delete key is pressed we call the ReSet method of the
  Filter. This re-initializes the filter and returns a null string
  which is assigned to the current cell in the grid.}
  if not(Key = VK_DELETE) then exit;
  with FilterGrid do Cells[Col,Row]:=Filter.ReSet;
end;

procedure TMaster.UpDateFilter;
var HintText,KeyText:String;
    i:Integer;
begin
  i:=FilterGrid.Row - 1;
  if Assigned(Filter) then Filter.Free;
  Filter:=nil;
  with FilterGrid do KeyText:=Cells[0,Row];
  case i of
    ftFloat..ftNone:Filter:=TNumFilter.Create;
    ftText:Filter:=TTextFilter.Create;
    ftToggle:Filter:=TToggleFilter.Create;
  end;
  Filter.Initialize(i,10,['0'..'9'],KeyText);
  case i of
    ftFloat:HintText:='Row 1 has a floating point filter.';
    ftInteger:HintText:='Row 2 only allows integer entries.';
    ftCardinal:HintText:='Row 3 restricts you to positive integers.';
    ftNone:HintText:='No keyboard input is allowed in Row 4.';
    ftText:HintText:='Any non-numeric character is acceptable in Row 5.';
    ftToggle:HintText:='Pressing the spacebar in this row toggles between a blank row & the '''' character.';
  end;
  FilterGrid.Hint:=HintText;
end;

end.
