unit OPSystm;
{
   Project "Opnek Standard Library (Delphi)"
   Opnek Research
   Copyright  1995. All Rights Reserved.

   SUBSYSTEM:    System Information Object
   TEST STUB:    OPTst.dpr
   FILE:         OPSystm.pas
   AUTHOR:       Jay Cole
   WRITTEN:      01/21/1995
   LAST ASSERT:  38

   OVERVIEW
   ========
   System level information object that contains information on available
   drives, space available, disk drive info, memory info, system level/win
   information.  All low-level system access/information should be added to
   this base system information class.

   UPDATE HISTORY
   ==============
   01/27/95 (JLC) - Added manual page to the file bottom
   02/08/95 (JLC) - Added GetFreeMemory(), GetFreeResources(),
                    HasMathCoProcessor() to the system object
   03/08/95 (JLC) - Converted to Delphi/Object Pascal.
   04/20/95 (JLC) - Created virtual constructor's/destructor's.
}

{ Conditionals available }
{ DEFINE _NDEBUG}

{-- Externally exported objects,types,procs and variables --}
interface
uses classes;

{ Exported constants }
const
   cDriveAvailable : char = 'Y';
   cDriveNotAvailable : char = 'N';
   cDriveArrSize = 27;
   cDriveNotReady : longint = -1;
   cNotInitialized = -1;

{ System object type }
type
   { Method types for the TSystemInfo class() }
   TScopeType = (evPassedDrives, evAllDrives);
   TDriveType = (evNotPresent, evRemovable, evFixed, evRemote);

   {- Contains encapsulated procedures/vars for system wide information -}
   TSystemInfo = class(TObject)
   private
      { Local variables and methods }
      driveCnt : integer;
      availHandles : word;
      driveStr : array[0..cDriveArrSize] of char;
      procedure InitDrives;                                                                   { D Done }

   public
      { Constructors/destructors }
      constructor Init;                                                                       { D Done }

      { General system information functions }
      function NumDrives : integer;                                                           { D Done }
      procedure DrivesAvail(var driveList : string);                                          { D Done }
      procedure DrivesAvailSortedbyFreeSpace(var driveList : string);                         { D Done }
      function GetAvailDriveSpace(driveLetter : char) : longint;                              { D Done }
      function GetTotalDriveSpace(driveLetter : char) : longint;                              { D Done }
      function GetLargestFreeDrive(var driveLetter : char) : longint;                         { D Done }
      function TotalAvailSpace : double;                                                      { D Done }
      function GetSysDriveType(driveLetter : char) : TDriveType;                              { D Done }
      function TranslateDriveType(driveType : tDriveType; var driveMsg : string) : boolean;   { D Done }
      procedure GetCurrentDirectory(var dirString : string);                                  { D Done }
      function FindInDrives(var dirMask : string; scope : TScopeType; var volumes : string;   { D Done }
                            var dirCollection : TStringList) : boolean;

      { Windows system information }
      procedure GetScreenSize(var x_dim : integer; var y_dim : integer);                      { D Done }
      function SetNumFileHandles(numHandles : integer) : boolean;                             { D Done }
      function GetNumFileHandles : integer;                                                   { D Done }
      function GetWinDirectory : string;                                                      { D Done }
      function GetFreeMemory : longint;                                                       { D Done }
      function GetFreeResources : integer;                                                    { D Done }
      function HasMathCoProcessor : boolean;                                                  { D Done }
   end;


const cDefaultProcessHandles : integer = 20;


{-- Begin the internal code here --}
implementation
uses {System, } WinProcs, SysUtils, Asserts, WinTypes;

const cModuleName : string = 'OPSYSTM.PAS';

{- TSystemInfo -}

{ (Final) CONSTRUCTOR: Set up the system initially. }
constructor TSystemInfo.Init;
begin
   inherited Create;

   driveCnt := cNotInitialized;
   availHandles := cDefaultProcessHandles;

   { N - means not available, Y - Means avaible (see cDRIVENOTAVAILABLE) }
   fillchar(driveStr, sizeof(driveStr), cDriveNotAvailable);
   InitDrives;
end;

{ (Final) PRIVATE: Initalize the available drives. }
procedure TSystemInfo.InitDrives;
var
   driveLtr : char;
   driveType : TDriveType;
   driveNum : word;
begin
   { Haven't been initialized, setup the array of available. }
   driveCnt := 0;
   {$IFNDEF _NDEBUG} Assert(((ord('Z')-ord('A')+1) < cDriveArrSize), cModuleName, 1); {$ENDIF}

   { Start at drive C because A and B are usually floppies, and it's not }
   { important to scan them.  Another routine can check the size, types  }
   { and validity of those A, B floppy drives.                           }
   for driveLtr := 'C' to 'Z' do begin
      driveType := GetSysDriveType(driveLtr);
      driveNum := ord(driveLtr) - ord('A');

      if (driveType <> evNotPresent) then begin
         {$IFNDEF _NDEBUG} Assert(driveNum < cDriveArrSize, cModuleName, 2); {$ENDIF}
         driveStr[driveNum] := cDriveAvailable;
         inc(driveCnt);
         if (GetAvailDriveSpace(driveLtr) = cDriveNotReady) then begin
            { We threw an exception, remove it. }
            dec(driveCnt);
            driveStr[driveNum] := cDriveNotAvailable;
         end
      end;
   end;
end;

{ (Final) PUBLIC: Return the number of drives on the system. }
function TSystemInfo.NumDrives : integer;
begin
   NumDrives := driveCnt;
end;

{ (Final) PUBLIC: Return the drives available in a sequencial list ie. ABCDGH }
procedure TSystemInfo.DrivesAvail(var driveList : string);
var outDriveArr : string;
    outNdx, drvNdx : word;
    driveChar : char;
begin
   outNdx := 1;

   { Construct the short string from the Y, N string. }
   for drvNdx := 0 to sizeof(driveStr)-1 do begin
      if (driveStr[drvNdx] = cDriveAvailable) then begin
         driveChar := chr(ord('A') + drvNdx);
         {$IFNDEF _NDEBUG} Assert(outNdx < cDriveArrSize, cModuleName, 3); {$ENDIF}
         outDriveArr[outNdx] := driveChar;
         inc(outNdx);
      end;
   end;

   { Append the size. }
   outDriveArr[0] := chr(outNdx);
   driveList := outDriveArr;
end;


{ (Final) PUBLIC: Return the string just like GetDrivesAvail() but sort it by size. }
procedure TSystemInfo.DrivesAvailSortedbyFreeSpace(var driveList : string);
var 
   outNdx, drvNdx : integer;
   { +1 to allow for shifting down on sort }
   outDriveArr : array[0..cDriveArrSize] of char;
   outDriveSize : array[0..cDriveArrSize] of longint;
   driveChar : char;
   driveSize : longint;
   insertPt, ndx : integer;

begin
   outNdx := 0;

   { Zero out for consistant debugging. }
   fillchar(outDriveArr, sizeof(outDriveArr), 0);
   fillchar(outDriveSize, sizeof(outDriveSize), 0);

   { Cycle list of 'Y'/'N' drive list. }
   for drvNdx := 0 to sizeof(driveStr)-1 do begin
      if (driveStr[drvNdx] = cDriveAvailable) then begin
         { Find the size of the current drive. }
         driveChar := chr(ord('A') + drvNdx);
         driveSize := GetAvailDriveSpace(driveChar);
         {$IFNDEF _NDEBUG} Assert(driveSize <> cDriveNotReady, cModuleName, 4); {$ENDIF}

         { Find the insertion point based on size. }
         insertPt := outNdx;
         for ndx := 0 to outNdx - 1 do begin
            {$IFNDEF _NDEBUG} Assert(insertPt < cDriveArrSize, cModuleName, 6); {$ENDIF}
            if (outDriveSize[ndx] < driveSize) then begin
               insertPt := outNdx;
               break;
            end;
         end;

         { Shift other entries down if we are not on the first element. }
         if (outNdx > 0) then begin
            for ndx := outNdx downto insertPt do begin
               {$IFNDEF _NDEBUG} Assert(ndx+1 < cDriveArrSize + 1, cModuleName, 7); {$ENDIF}
               outDriveSize[ndx+1] := outDriveSize[ndx];
               outDriveArr[ndx+1] := outDriveArr[ndx];
            end;
         end;

         { Insert our drive at the correct location. }
         {$IFNDEF _NDEBUG} Assert(insertPt < cDriveArrSize + 1, cModuleName, 8); {$ENDIF}
         outDriveArr[insertPt] := driveChar;
         outDriveSize[insertPt] := driveSize;
         inc(outNdx);
         {$IFNDEF _NDEBUG} Assert(outNdx < cDriveArrSize + 1, cModuleName, 9); {$ENDIF}
      end;
   end;

   { Add a /0 to terminate the string. }
   outDriveArr[outNdx] := chr(0);
   driveList := strpas(outDriveArr); 
end;


{ (Final) PUBLIC: Calculate free disk space on passed drive }
function TSystemInfo.GetAvailDriveSpace(driveLetter : char) : longint;
var 
   freeSpace : longint;
   diskTable : TSearchRec;
   driveNumber : integer;
   prevErrMode : integer;
   x : integer;
begin
   freeSpace := 0;

   { Remember in _dos_getdiskfree() 0 means default drive }
   {$IFNDEF _NDEBUG} Assert(((upcase(driveLetter) >= 'A') and (upcase(driveLetter) <= 'Z')), cModuleName, 10); {$ENDIF}
   driveNumber := ord(upcase(driveLetter)) - ord('A');
   {$IFNDEF _NDEBUG} Assert(driveNumber < cDriveArrSize, cModuleName, 11); {$ENDIF}
   {$IFNDEF _NDEBUG} Assert(driveStr[driveNumber] = cDriveAvailable, cModuleName, 12); {$ENDIF}

   { Function call for space. }
   try 
      { We add one because this routine assumes driveNumber=0 is current drive not 'A' }
      prevErrMode := SetErrorMode(SEM_FAILCRITICALERRORS);
      if (DiskFree(driveNumber + 1) <> -1) then begin
         { Calculate the amount of available disk space }
         freeSpace := DiskFree(driveNumber+1);
      end else begin
         freeSpace := cDriveNotReady;
      end;
   except
      { Problem w/ drive, flag. }
      freeSpace := cDriveNotReady;
   end;

   { Set back to original error mode. }
   SetErrorMode(prevErrMode);    
   GetAvailDriveSpace := freeSpace;
end;


{ (Final) PUBLIC: Calculate free disk space on passed drive }
function tSystemInfo.GetTotalDriveSpace(driveLetter : char) : longint;
var 
   totalSpace : longint;
   diskTable : TSearchRec;
   driveNumber : integer;
   prevErrMode : integer;

begin            
   totalSpace := 0;

   { Remember in _dos_getdiskfree() 0 means default drive. }
   {$IFNDEF _NDEBUG} Assert((upcase(driveLetter) >= 'A') and (upcase(driveLetter) <= 'Z'), cModuleName, 13); {$ENDIF}

   driveNumber := ord(upcase(driveLetter)) - ord('A');
   {$IFNDEF _NDEBUG} Assert(driveNumber < cDriveArrSize, cModuleName, 14); {$ENDIF}
   {$IFNDEF _NDEBUG} Assert(driveStr[driveNumber] = cDriveAvailable, cModuleName, 15); {$ENDIF}

   { Function call for space. }
   try 
      { We add one because this routine assumes driveNumber=0 is current drive not 'A'. }
      { Remember to disable the critical error handler.                                 }
      prevErrMode := SetErrorMode(SEM_FAILCRITICALERRORS);
      if (DiskSize(driveNumber + 1) >= 0) then begin
         { Calculate the amount of total disk space }
         totalSpace := DiskSize(driveNumber + 1);
      end else begin
         totalSpace := cDriveNotReady;
      end;
   except
      { No special exceptions }
      totalSpace := cDriveNotReady;
   end;

   SetErrorMode(prevErrMode);    
   GetTotalDriveSpace := totalSpace;
end;

{ (Final) PUBLIC: Get the largest free space drive on the system }
function TSystemInfo.GetLargestFreeDrive(var driveLetter : char) : longint;
var
   largestSpace, thisDrivesSpace : longint;
   largestVol, driveChar : char;
   upBound, drvNdx : integer;

begin
   largestSpace := 0;
   thisDrivesSpace := 0;
   largestVol := ' ';

   upBound := sizeof(driveStr) div sizeof(driveStr[0]);
   for drvNdx := 0 to upBound - 1 do begin
      if (driveStr[drvNdx] = cDriveAvailable) then begin
         driveChar := chr(ord('A') + drvNdx);
         thisDrivesSpace := GetAvailDriveSpace(driveChar);

         { Greater than largest drive to date?  If so, new candidate }
         if (thisDrivesSpace > largestSpace) then begin
            {$IFNDEF _NDEBUG} Assert(thisDrivesSpace <> cDriveNotReady, cModuleName, 16); {$ENDIF}
            if (thisDrivesSpace <> cDriveNotReady) then begin
               largestVol := driveChar;
               largestSpace := thisDrivesSpace;
            end;
         end;
      end;
   end;

   {$IFNDEF _NDEBUG} Assert(largestVol <> ' ', cModuleName, 17); {$ENDIF}
   {$IFNDEF _NDEBUG} Assert(largestSpace <> 0, cModuleName, 18); {$ENDIF}
   driveLetter := largestVol;
   GetLargestFreeDrive := largestSpace;
end;


{ (Final) PUBLIC: Get the total space on all available volumes }
function TSystemInfo.TotalAvailSpace : double;
var 
   allDiskSpace : double;
   thisDrivesSpace : longint;
   driveChar : char;
   upBound, drvNdx : integer;

begin
   allDiskSpace := 0.0;
   thisDrivesSpace := 0;

   upBound := sizeof(driveStr) div sizeof(driveStr[0]);
   { Loop through all drives }
   for drvNdx := 0 to upBound - 1 do begin
      if (driveStr[drvNdx] = cDRIVEAVAILABLE) then begin
         { Valid drive?  Sum the free amount in }
         driveChar := chr(ord('A') + drvNdx);
         thisDrivesSpace := GetAvailDriveSpace(driveChar);
         {$IFNDEF _NDEBUG} Assert(thisDrivesSpace <> cDriveNotReady, cModuleName, 19); {$ENDIF}
         if (thisDrivesSpace <> cDriveNotReady) then allDiskSpace := allDiskSpace + thisDrivesSpace;
      end;
   end;

   TotalAvailSpace := allDiskSpace;
end;


{ (Final) PUBLIC: Return the drive type of the letter attached drive }
function TSystemInfo.GetSysDriveType(driveLetter : char) : TDriveType;
var driveNum, driveType : integer;
    retWhat : TDriveType;

begin
   {$IFNDEF _NDEBUG} Assert((upcase(driveLetter) >= 'A') and (upcase(driveLetter) <= 'Z'), cModuleName, 20); {$ENDIF}
   driveNum := ord(upcase(driveLetter)) - ord('A');
   driveType := GetDriveType(driveNum); 
   retWhat := evNotPresent;                       

   { Note, specifically that this routine avoids the (drivesAvail[ndx] == cDRIVEAVAILABLE) }
   { Assert because it's used before the actual drives are assigned avail or not avail     }
   { leave that Assert out of the system.                                                  }
   case driveType of
      DRIVE_REMOVABLE : retWhat := evRemovable;
      DRIVE_FIXED     : retWhat := evFixed;
      DRIVE_REMOTE    : retWhat := evRemote;
      0               : retWhat := evNotPresent;
   else 
      {$IFNDEF _NDEBUG} Assert(1 = 1, cModuleName, 21);  {$ENDIF} { should never be here. }
   end;

   GetSysDriveType := retWhat;  
end;


{ (Final) PUBLIC:Translate the enumerated drive type value to a string }
function TSystemInfo.TranslateDriveType(driveType : TDriveType; var driveMsg : string) : boolean;
var retWhat : boolean;
begin
   retWhat := true;

   case (driveType) of
      evRemote     : driveMsg := 'CDRom, Remote or Network drive';
      evFixed      : driveMsg := 'Fixed hard drive';
      evNotPresent : driveMsg := 'Not present';
      evRemovable  : driveMsg := 'Removable storage media';
   else
      { Again, should never get here. }
      {$IFNDEF _NDEBUG} Assert(1=1, cModuleName, 22); {$ENDIF}
      driveMsg := 'Error: Unknown drive type';
      retWhat := false;
   end;

   TranslateDriveType := retWhat;
end;


{ (Final) PUBLIC: Return the current directory }
procedure TSystemInfo.GetCurrentDirectory(var dirString : string);
begin
   { Call the dir.h function to get current directory with drive info. }
   GetDir(0, dirString);
   if (dirString[length(dirString)] <> '\') then dirString := dirString + '\';
end;


{ (Final) PUBLIC: Return an array class full of directory entries meeting the passed mask and volume requirements. }
function TSystemInfo.FindInDrives(var dirMask : string; scope : TScopeType; var volumes : string; 
                                  var dirCollection : TStringList) : boolean;
type  
   PString = ^string;
const 
   cStackMax = 500;
var   
   fullFilename, driveList : String;
   chrDirMask, directStr : String;
   dirStack : array[0..cStackMax] of PString;
   stackTop, ndx, driveNdx : integer;
   fileFound : boolean;
   currDir, srchFile, lastDir, fileString : string;
   fileBlock : TSearchRec;

begin
   stackTop := -1;
   fileFound := false;

   { Allocate memory for the stack. }
   try 
       for ndx := 0 to cStackMax do begin
           dirStack[ndx] := NIL;
           dirStack[ndx] := new(PString);
           {$IFNDEF _NDEBUG} Assert(dirStack[Ndx] <> NIL, cModuleName, 37); {$ENDIF}
       end;
   except { Catch in else clause }
       MessageBox(0, 'Not enough memory to search for files', 'ERROR', MB_OK);
       FindInDrives := false;
       exit;
   end;

   { Copy directory mask into more useable form. }
   chrDirMask := dirMask;

   { Set up the volumes needed to cycle. }
   if (scope = evPassedDrives) then driveList := volumes else DrivesAvail(driveList); 

   try 
      { loop for each volume on the list }
      for driveNdx := 1 to length(driveList) do begin
         { push the root of the current volume for root directory. }
         directStr := driveList[driveNdx]; 
         {$IFNDEF _NDEBUG} Assert((directStr[1] >= 'A') and (directStr[1] <= 'Z'), cModuleName, 23); {$ENDIF}
         directStr := directStr + ':\';
         inc(stackTop);
         {$IFNDEF _NDEBUG} Assert(stackTop < cStackMax, cModuleName, 24); {$ENDIF}
         dirStack[stackTop]^ := directStr;

         { Cycle the current directory }
         repeat 
            { Set current search directory stack top, pop stack top. }
            {$IFNDEF _NDEBUG} Assert(length(dirStack[stackTop]^)+length(currDir)+length(chrDirMask) < sizeof(srchFile), 
                                     cModuleName, 
                                     26); {$ENDIF}
            {$IFNDEF _NDEBUG} Assert(length(dirStack[stackTop]^) < sizeof(currDir), cModuleName, 27); {$ENDIF}
            currDir := dirStack[stackTop]^;
            srchFile := dirStack[stackTop]^ + chrDirMask;
            {$IFNDEF _NDEBUG} Assert(length(currDir) < sizeof(lastDir), cModuleName, 28); {$ENDIF}
            lastDir := currDir;

            if (findfirst(srchFile, 0, fileBlock) = 0) then begin
               repeat
                  { We have some, now add them to the users collection. }
                  fullFilename := currDir;
                  fileString := fileBlock.name;
                  fullFilename := fullFileName + fileString;
                  dirCollection.Add(fullFilename);
                  fileFound := true;
               until (findnext(fileBlock) <> 0);
            end;

            { Pop off current directory before adding any other directories. }
            dec(stackTop);

            { scan directory for sub-directories and push onto the stack }
            {$IFNDEF _NDEBUG} Assert(length(currDir) + length('*.*') < sizeof(srchFile), cModuleName, 29); {$ENDIF}
            srchFile := currDir + '*.*';
            if (FindFirst(srchFile, faDirectory, fileBlock) = 0) then begin
               repeat
                  { We have some, now append them. }
                  if ((fileBlock.Attr and faDirectory) <> 0) then begin
                     { Remember ignore . and .. }
                     if (fileBlock.Name[1] <> '.') then begin
                        {$IFNDEF _NDEBUG} Assert(length(currDir) + sizeof(fileBlock.Name) < sizeof(directStr),
                                                 cModuleName, 30); {$ENDIF}
                        directStr := currDir + fileBlock.Name + '\';
                        {$IFNDEF _NDEBUG} Assert(stackTop < (sizeof(dirStack) / sizeof(dirStack[0])),
                                                 cModuleName, 31); {$ENDIF}
                        if (stackTop < (sizeof(dirStack) / sizeof(dirStack[0]))) then begin
                           inc(stackTop);
                           dirStack[stackTop]^ := directStr;
                        end;
                     end;
                  end;
               until (FindNext(fileBlock) <> 0);
            end;

         until (stackTop = -1);
   
      end;
   except {Handle in the general else }
      { Not enough memory to perform function. }
      MessageBox(0, 'Not enough memory to search for files', 'ERROR', MB_OK);
   end;

   { Free our stack up }
   for ndx := 0 to cStackMax do begin
      {$IFNDEF _NDEBUG} Assert(dirStack[ndx] <> NIL, cModuleName, 32); {$ENDIF}
      Dispose(dirStack[ndx]);
   end;

   FindInDrives := fileFound;
end;


{ (Final) PUBLIC: Set the number of possible file handles that can be opened. }
function TSystemInfo.SetNumFileHandles(numHandles : integer) : boolean;
var actualHandlesSet : integer;
begin
   actualHandlesSet := 0;
   {$IFNDEF _NDEBUG} Assert(numHandles < 255, cModuleName, 33); {$ENDIF}
   if (numHandles < 255) then begin
      actualHandlesSet := SetHandleCount(numHandles);
      availHandles := actualHandlesSet;
   end;

   SetNumFileHandles := (numHandles = actualHandlesSet);
end;

{ (Final) PUBLIC: Set the number of file handles }
function TSystemInfo.GetNumFileHandles : integer;
begin
   GetNumFileHandles := availHandles;
end;

{ (Final) PUBLIC: Return the windows directory on the current system. }
function TSystemInfo.GetWinDirectory : string;
var 
   wndDir : array[0..200] of char;
   windowsDir : string;
   retRslt : word;
begin
   retRslt := GetWindowsDirectory(wndDir, sizeof(wndDir));
   {$IFNDEF _NDEBUG} Assert(retRslt <> 0, cModuleName, 34); {$ENDIF}
   if (retRslt > 0) then windowsDir := StrPas(wndDir);
   {$IFNDEF _NDEBUG} Assert(length(windowsDir) > 1, cModuleName, 35); {$ENDIF}
   if (windowsDir[length(windowsDir)-1] <> '\') then windowsDir := windowsDir + '\';
   GetWinDirectory := windowsDir;
end;

{ (Final) PUBLIC: Get the size of the screen via windows metric functions. }
procedure TSystemInfo.GetScreenSize(var x_dim, y_dim : integer);
begin
   x_dim := GetSystemMetrics(SM_CXSCREEN);
   y_dim := GetSystemMetrics(SM_CYSCREEN);
end;

{ (Final) PUBLIC: Get the free amount of resources in the system }
function TSystemInfo.GetFreeResources : integer;
begin
   {$IFNDEF _NDEBUG} Assert(GetFreeSystemResources(GFSR_SYSTEMRESOURCES) <= 100, cModuleName, 36); {$ENDIF}
   GetFreeResources := GetFreeSystemResources(GFSR_SYSTEMRESOURCES);
end;

{ (Final) PUBLIC: Get the amount of free memory on the system }
function TSystemInfo.GetFreeMemory : longint;
begin
   GetFreeMemory := GetFreeSpace(0);
end;

{ (Final) PUBLIC: Determine if we have a math co-processor present on the runtime system }
function TSystemInfo.HasMathCoProcessor : boolean;
begin
   HasMathCoProcessor := ((GetWinFlags and WF_80x87) <> 0);
end;


{-- Begin Documentation --}

{===============================================================================

Common abbreviation list
========================
drv   - Drive
tot   - Total
Avail - Available
ltr   - Same as letter.

Common Terms
============
Free,
Avail - The amount of free space available on the drive or in the case of
        of drives it means are the online to the system.
Total - Is the original total of space free+used.

Enum Values/Meanings
====================
tDriveType (s)
    evNotPresent    - No drive online with that letter
    evRemovable     - Floppy drive, etc...
    evFixed         - Hard drive on the system including subst drives.
    evRemote        - Network drive and CD-ROMs, etc...

tScopeType
    evPassedDrive   - Scan for file on the passed volume letter
    evAllDrives     - Scan on all available volumes.

Suggested enhancements
======================
- Add another routine to include the floppy drives when scanning drives
  available.  Use that routine specifically for floppy drives.
- Also add a routine to check floppy drive sizes and availability.
- Possibly add volume ID checking to assure your not dealing w/ subst drives
- Currently only support system resources, possibly routines for other
  resources.
- Add a find() function that also masks based on file attributes.

CONSTRUCTORS/DESTRUCTOR documentation
=====================================
Init() - Default constructor not much done in this constructor it simply
         initializes the local variables and then cycles through every
         drive available on the system and marks it avail/unavail.
Done() - Default destructor, again, not much done.

METHODS/PROCEDURES function document
====================================
TSystemInfo.DrivesAvail(var string driveList)
    driveList   Contains an array of characters where each entry is a drive
                letter available in the current system.  It is returned in
                alphabetical order and terminated with a \0.
Looks in the entire system and returns every drive on the system that is
currently available.

TSystemInfo.DrivesAvailSortedbyFreeSpace(var string driveList)
    driveList   :   Contains an array of characters where each entry is a
                    drive letter available in the current system.  It is
                    returned in largest free space to smallest free space
                    order and terminated with a \0.
Takes the drives same as DrivesAvail() and sorts them according to the one
that has the most free space left.

function TSystemInfo.GetAvailDriveSpace(tUChar driveLetter) : longint;
    driveLetter :   Gets the space available on the passed drive letter.
    Return      :   the amount of free disk space on the passed drive
                    'driveLetter' or cDRIVENOTREADY if critical error.
Return the available free space on the passed drive.

function TSystemInfo.GetTotalDriveSpace(tUChar driveLetter) : longint;
    driveLetter :   Drive to return the total disk space of.
    Return      :   the size of the drive that is freespace+usedspace
                    or cDRIVENOTREADY if critical error.
Return the original size of the attached fixed drive.  Used in calcing % of
free value.

function TSystemInfo.GetLargestFreeDrive(tUChar &driveLetter) : longint;
    driveLetter :   Set to the drive that has the largest disk space free
    Return      :   The amount of free disk space on 'driveLetter' drive
                    or cDRIVENOTREADY if critical error.
Searches all available fixed drives on the system and returns the drive
letter of the one with the most free space and the free space on that drive.

function TSystemInfo.TotalAvailSpace : double;
    return      :   Total drive space on all available volumes summed up.
Traverses every drive in the system that is fixed and sums up the free disk
space, then returns it as the total drive free space available.

function TSystemInfo.GetSysDriveType(driveLetter : char) TDriveType;
    driveLetter :   Drive letter of the drive type we are checking
    return      :   The type of drive, check out the tDriveType enum() vals
Returns the enumerated value of the drive type that is attached to the
passed drive letter.

function TSystemInfo.NumDrives : integer;
    noparams
    return      :   Returns the number of physically fixed or network drives
                    on the system.  Note, floppy drives are not included on
                    this list.
Send back to the user the total number of fixed and/or remote drives on the
system.

function TSystemInfo.TranslateDriveType(driveType : TDriveType,
                                        var driveMsg : string) : boolean;
    driveType   :   Contians the enumerated drivetype value to be translated
    driveMsg    :   contains the string with the english message.
    return      :   True if a valid drivetype value.
Pass in the enumerated drive type and the system will return the english
readable enumerated tag.  useful for list box user displays.

procedure TSystemInfo.GetCurrentDirectory(var dirString : string)
    dirString   :   Is set to the current working directory
Looks up the current directory and places it in a null term'ed string in
dirString.

procedure TSystemInfo.GetScreenSize(var x_dim, y_dim : integer);
    x_dim       :   X (width) of the screen totally.
    y_dim       :   Y (height) of the screen totally.
Return the x-y size of the screen.  Used for min resolutions in programs.
So, if we require 600x800 for our program or above, this is the check.

function TSystemInfo.FindInDrives(var dirMask : string, scope : TScopeType,
                                  var volumes : string,
                                  var TStringList dirCollection) : boolean;
    dirMask     :   The directory file name to look for including possibly
                    wild cards such as config??.* or foo*.exe
    scope       :   Can take on two values.  evPassedDrives means search
                    only the drives in the parameter volumes, or evAllDrives
                    which means search all drives present on the system.
    volumes     :   if scope = evPassedDrive then this contains a list of
                    drives to search for the passed dirMask.
    dirCollection:  Is a TIArrayAsVector class the user creates with the
                    type tDirList.  It is dynamically filled with all
                    entries on the searched volumes that meet the dirMask.
    returns     :   true if there is at least one match otherwise false.
Searches all volumes or the passed volumes for every occurence on the system
of dirMask.  It adds each match to the passed dirCollection structure.  The
calling procedure is responsible for freeing up all resources associated
with it.

function TSystemInfo.SetNumFileHandles(numHandles : integer) : boolean;
    numHandles  :   Number of total file handles to set that can be opened
                    at one time.  Max(255).
    returns     :   true if it could allocate that many handles, false if
                    less were allocated.
Allows the user to increase the max number of file handles that the system
can open simultaneously.

function TSystemInfo.GetNumFileHandles : integer;
    returns     :   The maximum number of file handles that can be open
                    simulataneously.  Remember, this is the total file
                    handles avail() for this process, not the number of ones
                    left to be open.
Returns the total file handles available to the current process.

function TSystemInfo.GetWinDirectory(var string : windowsDir) : boolean
    windowsDir  :   string that will contain the full path that points to
                    the users windows directory.
    returns     :   true if it could find the windows directory.
                    false if it can't find it, or the path is too big to
                    fit in a 400 byte temporary.
Get the windows operating directory to store .ini and other files.

function TSystemInfo.GetFreeMemory : longint
    returns     :   The total number of bytes available on the system.
Returns the virtual space available under windows for memory allocation.

function TSystemInfo.GetFreeResources : integer
    returns     :   The percentage of free resource available in the
                    system.
Used to determine when you are low on system level resources.  Remember
Windows silly 64k resource table limit.

function TSystemInfo.HasMathCoProcessor : boolean
    returns     : true if any 80x87 processor is available, otherwise
                  false
Used to determine the presence of a math co-processor.

===============================================================================}


end.
