unit RegInt;
{******************************************************************************
* Unit RegInt                                                                 *
* --------------------------------------------------------------------------- *
* Provides an interface between the RegDll.dll and your application. Exports  *
* an object, which behaves similar to a TINIFile Object. The registry (at     *
* least in Win31) only accepts strings. The .dll handles all the conversions  *
* necessary for reading or writing. Furthermore a default value can be        *
* spezified, which will be returned, in case the key could not be found or    *
* an error occured during read                                                *
******************************************************************************}

interface

uses
  SysUtils, WinTypes, WinProcs, Messages, Classes,Registry;

const
REG_CREATEERROR=-1;
  REG_OPENERROR=-2;
  REG_QUERYINVALIDDATA=-3;
  REG_VALUENOTFOUND=-4;
  REG_INVALIDKEYHANDLE=-5;
  REG_WRITEERROR=-6;
  REG_DELETEERROR=-7;
  REG_MERGEERROR=-8;
  REG_SUCCESS=0;

type
  TRegistry=class(TObject)
  private
    RegBuf:PChar;
    KeyBuf:PChar;
  public
    function ReadInteger(const Key: String; const defValue: Integer; var Value: Integer): Longint;
    function ReadString(const Key: String; const defValue: String; var Value: String): Longint;
    function ReadBool(const Key: String; const defValue: Boolean; var Value: Boolean): Longint;
    function CreateKey(const Key: String): Longint;
    procedure CloseKey;
    function WriteInteger(const Key: String; const Value: Integer): Longint;
    function WriteString(const Key: String; const Value: String): Longint;
    function WriteBool(const Key: String; const Value: Boolean): Longint;
    function OpenKey(const Key: String): Longint;
    function Merge(const Filename: String): Longint;
    function DeleteKey(const Key: String): Longint;
    constructor Create(const BufSize: Integer);
    destructor Destroy;
  end;

implementation

function RQueryIntegerValue(const Key: String; const defValue: Integer; var Value: Integer;
                            const RegBuf,KeyBuf: PChar): Longint; far; external 'Regdll';
function RQueryStringValue(const Key: String; const defValue: String; var Value: String;
                           const RegBuf,KeyBuf: PChar): Longint; far; external 'Regdll';
function RQueryBoolValue(const Key: String; const defValue: Boolean; var Value: Boolean;
                         const RegBuf,KeyBuf: PChar): Longint; far; external 'Regdll';
function RCreateKey(const Key: String; const KeyBuf: PChar): Longint; far; external 'RegDll';
procedure RCloseKey; far; external 'Regdll';
function RSetIntegerValue(const Key: String; const Value: Integer;const RegBuf,KeyBuf: PChar): Longint; far; external 'Regdll';
function RSetStringValue(const Key: String; const Value: String; const RegBuf,KeyBuf: PChar): Longint; far; external 'Regdll';
function RSetBoolValue(const Key: String; const Value: Boolean; const RegBuf,KeyBuf: PChar): Longint; far; external 'Regdll';
function ROpenKey(const Key: String; const KeyBuf: PChar):Longint; far; external 'Regdll';
function RDeleteKey(const Key: String; const KeyBuf: PChar):Longint; far; external 'Regdll';
function RMerge(const RegFile: String):Longint; far; external 'RegDll';

constructor TRegistry.Create(const BufSize: Integer);
{------------------------------------------------------------------------------
 Creates the object and allocates two buffers, which will be needed for
 registry I/O.
 -----------------------------------------------------------------------------}
begin
  inherited Create;
  RegBuf:=StrAlloc(BufSize);
  KeyBuf:=StrAlloc(BufSize);
end;

destructor TRegistry.Destroy;
{------------------------------------------------------------------------------
 Frees object memory and memory allocated for two buffers.
 -----------------------------------------------------------------------------}
begin
  inherited Destroy;
  StrDispose(RegBuf);
  StrDispose(KeyBuf);
end;

function TRegistry.DeleteKey(const Key: String): Longint;
{-----------------------------------------------------------------------------
 PURPOSE:
 Delete a key from the registry

 PARAMS:
 Key: The key to be deleted

 RETURNS:
 REG_SUCCESS: if key could be deleted
 REG_DELETEERROR: if key could not be deleted
 -----------------------------------------------------------------------------}
begin
  DeleteKey:=RDeleteKey(Key,KeyBuf);
end;

function TRegistry.Merge(const Filename: String): Longint;
{------------------------------------------------------------------------------
 PURPOSE:
 Merges a reg file into the registry database.
 WARNING: Has only be tested with the 347 build of win95 !

 PARAMS:
 Filename: The path and filename (including extension) to merged into the registry

 RETURNS:
 REG_SUCCESS if merge was successful
 REG_MERGEERROR otherwise
 -----------------------------------------------------------------------------}
begin
  Merge:=RMerge(Filename);
end;

function TRegistry.ReadInteger(const Key: String; const defValue: Integer; var Value: Integer): Longint;
{------------------------------------------------------------------------------
 PURPOSE:
 Reads an integer value from the registry. Calls the QueryIntegerValue
 function from the RegDll.dll. The Key must be opened using CreateKey
 or OpenKey.

 PARAMS:
 Key:      String:  The key from which to read the value
 defValue: Integer: The value returned in case of error or if key could not
                    be found.
 Value:    Integer: Contains the value read from the registry

 RETURNS:
 Longint: REG_SUCCESS in case of success
          REG_VALUENOTFOUND if value could not be read  (either due to an
                            error or value does not exist)
          REG_QUERYINVALIDDATA if the read value is not an integer value
 -----------------------------------------------------------------------------}
begin
  ReadInteger:=RQueryIntegerValue(Key,defValue,Value,RegBuf,KeyBuf);
end;

function TRegistry.ReadString(const Key: String; const defValue: String; var Value: String): Longint;
{------------------------------------------------------------------------------
 PURPOSE:
 Reads a string-value from the registry database. Calls RQueryStringValue from
 the Regdll.dll. A Key must be opened using OpenKey or CreateKey first.

 PARAMS:
 Key:      String:  The key from which to read the value
 defValue: String:  The value returned in case of error or if key could not
                    be found.
 Value:    String:  Contains the value read from the registry

 RETURNS:
 Longint: REG_SUCCESS in case of success
          REG_VALUENOTFOUND if value could not be read  (either due to an
                            error or value does not exist)
          REG_QUERYINVALIDDATA if the read value is not an integer value
 -----------------------------------------------------------------------------}
begin
  ReadString:=RQueryStringValue(Key,defValue,Value,RegBuf,KeyBuf);
end;

function TRegistry.ReadBool(const Key: String; const defValue: Boolean; var Value: Boolean): Longint;
{------------------------------------------------------------------------------
 PURPOSE:
 Reads a boolean value from the registry database. Calls RQueryBoolValue from
 Regdll.dll. A Key must be opened using OpenKey or CreateKey first.

 PARAMS:
 Key:      String:  The key from which to read the value
 defValue: Boolean: The value returned in case of error or if key could not
                    be found.
 Value:    Boolean: Contains the value read from the registry

 RETURNS:
 Longint: REG_SUCCESS in case of success
          REG_VALUENOTFOUND if value could not be read  (either due to an
                            error or value does not exist)
          REG_QUERYINVALIDDATA if the read value is not a boolean value
 -----------------------------------------------------------------------------}
begin
  ReadBool:=RQueryBoolValue(Key,defValue,Value,RegBuf,KeyBuf);
end;

function TRegistry.CreateKey(const Key: String): Longint;
{------------------------------------------------------------------------------
 PURPOSE:
 Creates a key in the registry database. If the key already exists, it opens
 the key. Calls RCreateKey from regdll.dll.

 PARAMS:
 Key:      String:  The key to open or to create.

 RETURNS:
 Longint: REG_SUCCESS in case of success
          REG_CREATERROR if open or creation failed
 -----------------------------------------------------------------------------}
begin
  CreateKey:=RCreateKey(Key,KeyBuf);
end;

function TRegistry.OpenKey(const Key: String): Longint;
{------------------------------------------------------------------------------
 PURPOSE:
 Opens an existing key in the registry database. Unlike CreateKey OpenKey does
 not create a new key, if the specified key does not exist. Calls ROpenKey from
 Regdll.dll,

 PARAMS:
 Key:      String:  The key to open or to create.

 RETURNS:
 Longint: REG_SUCCESS in case of success
          REG_OPENERROR if open or creation failed
 -----------------------------------------------------------------------------}
begin
  OpenKey:=ROpenKey(Key,KeyBuf);
end;

procedure TRegistry.CloseKey;
{------------------------------------------------------------------------------
 PURPOSE:
 Closes an open key opened using CreateKey or OpenKey. Calls RCloseKey from
 Regdll.dll.

 PARAMS:
 -

 RETURNS:
 -
 -----------------------------------------------------------------------------}
begin
  RCloseKey;
end;

function TRegistry.WriteString(const Key: String; const Value: String): Longint;
{------------------------------------------------------------------------------
 PURPOSE:
 Writes a string value to the registry. Calls RSetStringValue from Regdll.dll.
 A Key must be opened using OpenKey or CreateKey first, before a value can be
 written.

 PARAMS:
 Key:      String:  The key for which to write the value.
 Value:    String:  The value to be written.

 RETURNS:
 Longint: REG_SUCCESS in case of success
          REG_WRITEERROR if value could not be written
          REG_INVALIDHANDLE if the key was not yet opened
 -----------------------------------------------------------------------------}
begin
  WriteString:=RSetStringValue(Key,Value,RegBuf,KeyBuf);
end;

function TRegistry.WriteInteger(const Key: String; const Value: Integer): Longint;
{------------------------------------------------------------------------------
 PURPOSE:
 Writes a string value to the registry. Calls RSetIntegerValue from Regdll.dll.
 A Key must be opened using OpenKey or CreateKey first, before a value can be
 written.

 PARAMS:
 Key:      String:  The key for which to write the value.
 Value:    String:  The value to be written.

 RETURNS:
 Longint: REG_SUCCESS in case of success
          REG_WRITEERROR if value could not be written
          REG_INVALIDHANDLE if the key was not yet opened
 -----------------------------------------------------------------------------}
begin
  WriteInteger:=RSetIntegerValue(Key,Value,RegBuf,KeyBuf);
end;

function TRegistry.WriteBool(const Key: String; const Value: Boolean): Longint;
{------------------------------------------------------------------------------
 PURPOSE:
 Writes a boolean value to the registry. Calls RSetBoolValue from Regdll.dll.
 A Key must be opened using OpenKey or CreateKey first, before a value can be
 written.

 PARAMS:
 Key:      String:   The key for which to write the value.
 Value:    Boolean:  The value to be written.

 RETURNS:
 Longint: REG_SUCCESS in case of success
          REG_WRITEERROR if value could not be written
          REG_INVALIDHANDLE if the key was not yet opened
 -----------------------------------------------------------------------------}
begin
  WriteBool:=RSetBoolValue(Key,Value,RegBuf,KeyBuf);
end;

end.
