unit Registry;
{******************************************************************************
* Unit Registry                                                               *
* Handles registry database. Implements functions, which allow to read and    *
* write not only string values, but Integer and boolean as well. Furthermore  *
* a default value can be specified, which will be returned in case of error   *
******************************************************************************}

interface

uses
  SysUtils, WinTypes, WinProcs, Messages, Classes,ShellAPI;

function RQueryIntegerValue(const Key: String; const defValue: Integer; var Value: Integer;
                            const RegBuf,KeyBuf: PChar): Longint; export;
function RQueryStringValue(const Key: String; const defValue: String; var Value: String;
                           const RegBuf,KeyBuf: PChar): Longint; export;
function RQueryBoolValue(const Key: String; const defValue: Boolean; var Value: Boolean;
                         const RegBuf,KeyBuf: PChar): Longint; export;
function RCreateKey(const Key: String; const KeyBuf: PChar): Longint; export;
procedure RCloseKey; export;
function RSetIntegerValue(const Key: String; const Value: Integer;const RegBuf,KeyBuf: PChar): Longint; export;
function RSetStringValue(const Key: String; const Value: String; const RegBuf,KeyBuf: PChar): Longint; export;
function RSetBoolValue(const Key: String; const Value: Boolean; const RegBuf,KeyBuf: PChar): Longint; export;
function ROpenKey(const Key: String; const KeyBuf: PChar):Longint; export;
function RDeleteKey(const Key: String; const KeyBuf: PChar):Longint; export;
function RMerge(const RegFile: String):Longint; export;

implementation

const
  REG_CREATEERROR=-1;
  REG_OPENERROR=-2;
  REG_QUERYINVALIDDATA=-3;
  REG_VALUENOTFOUND=-4;
  REG_INVALIDKEYHANDLE=-5;
  REG_WRITEERROR=-6;
  REG_DELETEERROR=-7;
  REG_MERGEERROR=-8;
  REG_SUCCESS=0;

var
  KeyHandle:      HKey;
  ValidKeyHandle: Boolean;
  CurOpenKey:     String;

function RMerge(const RegFile: String): Longint;
{------------------------------------------------------------------------------
 Merges a given registry file (extension .REG) into the registry using the
 /i parameter.
 -----------------------------------------------------------------------------}
var
  Buf: PChar;
begin
  Buf:=StrAlloc(Length(RegFile)+1);
  StrPCopy(Buf,'/i '+RegFile);
  if ShellExecute(hInstance,nil,'regedit.exe',Buf,nil,SW_HIDE) < 32 then
    RMerge:=REG_MERGEERROR
  else
    RMerge:=REG_SUCCESS;
  StrDispose(Buf);
end;

function RCreateKey(const Key: String; const KeyBuf: PChar): Longint;
{------------------------------------------------------------------------------
 Opens a given key. If the key does not exist, then it will be created. The
 handle returned is stored internally.
 -----------------------------------------------------------------------------}
var
  hndKey: HKey;
begin
  if ValidKeyHandle then
      RegCloseKey(KeyHandle);
  if RegCreateKey(HKEY_CLASSES_ROOT,StrPCopy(KeyBuf,Key),hndKey) = ERROR_SUCCESS then begin
    KeyHandle:=hndKey;
    RCreateKey:=REG_SUCCESS;
    ValidKeyHandle:=true;
    CurOpenKey:=Key;
  end
  else
    RCreateKey:=REG_CREATEERROR;
end;

function ROpenKey(const Key: String; const KeyBuf: PChar):Longint;
{------------------------------------------------------------------------------
 Opens a key.
 -----------------------------------------------------------------------------}
var
  hndKey: HKey;
begin
  if ValidKeyHandle then
      RegCloseKey(KeyHandle);

  if RegOpenKey(HKEY_CLASSES_ROOT,StrPCopy(KeyBuf,Key),hndKey) = ERROR_SUCCESS then begin
    KeyHandle:=hndKey;
    ROpenKey:=REG_SUCCESS;
    ValidKeyHandle:=true;
    CurOpenKey:=Key;
  end
  else
    ROpenKey:=REG_OPENERROR;
end;

function RDeleteKey(const Key: String; const KeyBuf: PChar):Longint;
{------------------------------------------------------------------------------
 Deletes a key from the registry
 -----------------------------------------------------------------------------}
begin
  if ValidKeyHandle then
    if RegDeleteKey(KeyHandle,KeyBuf)=ERROR_SUCCESS then
      RDeleteKey:=REG_SUCCESS
    else
      RDeleteKey:=REG_DELETEERROR;
end;

procedure RCloseKey;
{------------------------------------------------------------------------------
 Closes Key and invalidates the key handle
 -----------------------------------------------------------------------------}
begin
  if ValidKeyHandle then
    RegCloseKey(KeyHandle);

  CurOpenKey:='';
  ValidKeyHandle:=false;
end;

function RSetIntegerValue(const Key: String; const Value: Integer;const RegBuf,KeyBuf: PChar): Longint;
{------------------------------------------------------------------------------
 Writes an integer value to the registry.
 -----------------------------------------------------------------------------}
begin
  if ValidKeyHandle then begin
    if RegSetValue(HKEY_CLASSES_ROOT,StrPCopy(KeyBuf,CurOpenKey+'\'+Key),REG_SZ,StrPCopy(RegBuf,IntToStr(Longint(Value))),255)
                   =ERROR_SUCCESS then
      RSetIntegerValue:=REG_SUCCESS
    else
      RSetIntegerValue:=REG_WRITEERROR;
  end
  else
    RSetIntegerValue:=REG_INVALIDKEYHANDLE;
end;

function RSetStringValue(const Key: String; const Value: String; const RegBuf,KeyBuf: PChar): Longint;
{------------------------------------------------------------------------------
 Writes a string value to the registry
 -----------------------------------------------------------------------------}
begin
  if ValidKeyHandle then begin
    if RegSetValue(HKEY_CLASSES_ROOT,StrPCopy(KeyBuf,CurOpenKey+'\'+Key),REG_SZ,StrPCopy(RegBuf,Value),255)=ERROR_SUCCESS then
      RSetStringValue:=REG_SUCCESS
    else
      RSetStringValue:=REG_WRITEERROR;
  end
  else
    RSetStringValue:=REG_INVALIDKEYHANDLE;
end;

function RSetBoolValue(const Key: String; const Value: Boolean; const RegBuf,KeyBuf: PChar): Longint;
{------------------------------------------------------------------------------
 Writes a boolean value to the registry.
 -----------------------------------------------------------------------------}
begin
  if ValidKeyHandle then begin
    if RegSetValue(HKEY_CLASSES_ROOT,StrPCopy(KeyBuf,CurOpenKey+'\'+Key),REG_SZ,StrPCopy(RegBuf,IntToStr(Longint(Value))),255)
                   =ERROR_SUCCESS then
      RSetBoolValue:=REG_SUCCESS
    else
      RSetBoolValue:=REG_WRITEERROR;
  end
  else
    RSetBoolValue:=REG_INVALIDKEYHANDLE;
end;

function RQueryStringValue(const Key: String; const defValue: String; var Value: String; const RegBuf,KeyBuf: PChar): Longint;
{------------------------------------------------------------------------------
 Retrieves a string value from the registry
 -----------------------------------------------------------------------------}
var
  cb:     Longint;
  regVal: String;
  qrvRet: Longint;
begin
  if ValidKeyHandle then begin
    cb:=StrBufSize(RegBuf);
    if RegQueryValue(HKEY_CLASSES_ROOT,StrPCopy(KeyBuf,CurOpenKey+'\'+Key),RegBuf,cb)=ERROR_SUCCESS then
      RQueryStringValue:=REG_SUCCESS
    else begin
      RQueryStringValue:=REG_VALUENOTFOUND;
      Value:=defValue;
    end;
    try
      regVal:=StrPas(RegBuf);
      Value:=regVal;
      RQueryStringValue:=REG_SUCCESS;
    except
      RQueryStringValue:=REG_QUERYINVALIDDATA;
      Value:=defValue;
    end;
  end;
end;

function RQueryIntegerValue(const Key: String; const defValue: Integer; var Value: Integer;
                            const RegBuf,KeyBuf: PChar): Longint;
{------------------------------------------------------------------------------
 Retrieves an Integer Value from the registry
 -----------------------------------------------------------------------------}
var
  cb:     Longint;
  regVal: Integer;
  qrvRet: Longint;
begin
  if ValidKeyHandle then begin
    cb:=StrBufSize(RegBuf);
    if RegQueryValue(HKEY_CLASSES_ROOT,StrPCopy(KeyBuf,CurOpenKey+'\'+Key),RegBuf,cb)=ERROR_SUCCESS then
      RQueryIntegerValue:=REG_SUCCESS
    else begin
      RQueryIntegerValue:=REG_VALUENOTFOUND;
      Value:=defValue;
    end;
    try
      regVal:=Integer(StrToInt(StrPas(RegBuf)));
      Value:=regVal;
    except
      RQueryIntegerValue:=REG_QUERYINVALIDDATA;
      Value:=defValue;
    end;
  end;
end;

function RQueryBoolValue(const Key: String; const defValue: Boolean; var Value: Boolean; const RegBuf,KeyBuf: PChar): Longint;
{------------------------------------------------------------------------------
 Retrieves a boolean value from the registry
 ----------------------------------------------------------------------------}
var
  cb:     Longint;
  regVal: Boolean;
begin
  if ValidKeyHandle then begin
     cb:=StrBufSize(RegBuf);
     if RegQueryValue(HKEY_CLASSES_ROOT,StrPCopy(KeyBuf,CurOpenKey+'\'+Key),RegBuf,cb)=ERROR_SUCCESS then
       RQueryBoolValue:=REG_SUCCESS
     else begin
       RQueryBoolValue:=REG_VALUENOTFOUND;
       Value:=defValue;
     end;
     try
       regVal:=Boolean(StrToInt(StrPas(RegBuf)));
       Value:=regVal;
     except
       RQueryBoolValue:=REG_QUERYINVALIDDATA;
       Value:=defValue;
     end;
  end;
end;

initialization

begin
  CurOpenKey:='';
  ValidKeyHandle:=false;
end;

end.
