unit Iconhckr;

interface

uses WinTypes,
     WinProcs,
     SysUtils,
     Headers;

type
  PANSICHAR = PCHAR;

const
  MAX_PATH = 144;

const
  RT_GROUPICON = MAKEINTRESOURCE(14);

type
  EResourceError = class(Exception);

(*
 * TICONENUMERATOR is a callback function that
 * recieves the names of the RT_GROUP_ICON resources
 * in a given image.
 *
 * Return :
 *   TRUE - continue enumeration
 *  FALSE - stop now.
 *)
type
  TICONENUMERATOR = function ( szIconName: PANSICHAR ): BOOLEAN of Object;

type
  TResourceFile = class
    constructor Create ( szFileName: PANSICHAR );

    (*
     * FindIcon
     *
     * Locates the given Icon ID in the resource file.
     *
     * szResourceID - the name of the resouce or the ID in MAKEINTRESOURCE format.
     *
     * Returns:
     *   0 : The requested icon was not found in the file.
     *  !0 : The handle to the icon (actually file position)
     *)
    function FindIcon ( szResourceID: PANSICHAR ): LONGINT;

    (*
     * UpdateIcon
     *
     * Locates the specified resource and moves the given icons into
     * the exe file.
     *
     * szResourceID - the name of the resource or the ID in MAKEINTRESOURCE format.
     * szIcon - the name of the .ICO file.
     *
     * Returns:
     *   TRUE - successful
     *  FALSE - failed
     *)
    function UpdateIcon ( szResourceID: PANSICHAR; szIcon: PANSICHAR ): BOOLEAN;

    (*
     * EnumIcons
     *
     * Enumerates through all icons in the current image file.
     * Calls the method given by lpfnCallback for each icon.
     *
     * lpfnCallback - the method address to call for each icon.
     *
     * Returns:
     *   TRUE - the function was successful, does not guarentee that
     *          the callback function was called.
     *
     *  FALSE - the function failed.
     *
     *)
    function EnumIcons ( lpfnCallback: TICONENUMERATOR ): BOOLEAN;

    (*
     * UpdateIconFromImage
     *
     * Updates an icon from another TResourceFile.
     *
     * szResourceID - the name of the resource to be updated (destination)
     * ResFile      - the TResourceFile class from which the icon is to be retrieved
     * szSourceID   - the name of the source icon.
     *
     * Returns:
     *   TRUE - succesful
     *  FALSE - failed
     *)
    function UpdateIconFromImage ( szResourceID: PANSICHAR; ResFile: TResourceFile; szSourceID: PANSICHAR ): BOOLEAN;

    (*
     * GetIconDIB
     *
     * Returns the physical offset in the TResourceFile where
     * the DIB for the specified icon can be found.
     *
     * szResourceID - the name of the icon to find (comes from the RT_GROUP_ICON resource)
     *
     * Returns:
     *   0 : failed
     *   > 0 : the offset of the dib in the image.
     *)
    function GetIconDIB ( szResourceID: PANSICHAR ): LONGINT;

    destructor Destroy; override;
  private
    function FindNEIcon ( szResourceID: PANSICHAR ): LONGINT;
    function FindPEIcon ( szResourceID: PANSICHAR ): LONGINT;

    function GetPEIconDIB ( szResourceID: PANSICHAR ): LONGINT;
    function GetNEIconDIB ( szResourceID: PANSICHAR ): LONGINT;

    (* PE helper routines *)
    function ImageDirectoryOffset ( dwIMAGE_DIRECTORY : LONGINT; var VirtualAddress: LONGINT ): LONGINT;
    function SectionHeaderOffset: LONGINT;

    function UpdatePEIcon ( szResourceID: PANSICHAR; szIcon: PANSICHAR ): BOOLEAN;
    function UpdateNEIcon ( szResourceID: PANSICHAR; szIcon: PANSICHAR ): BOOLEAN;

    function EnumPEIcons ( lpfnCallback: TICONENUMERATOR ): BOOLEAN;
    function EnumNEIcons ( lpfnCallback: TICONENUMERATOR ): BOOLEAN;

    function UpdatePEIconFromImage ( szResourceID: PANSICHAR; ResFile: TResourceFile; szSourceID: PANSICHAR ): BOOLEAN;
    function UpdateNEIconFromImage ( szResourceID: PANSICHAR; ResFile: TResourceFile; szSourceID: PANSICHAR ): BOOLEAN;

  private
    m_szFileName : PANSICHAR;         (* Name of the file *)
    m_file       : FILE;              (* File variable *)
    m_bPortable  : BOOLEAN;           (* True, image is a PE; FALSE, image is an NE *)
    m_MZHeader   : TMZHeader;         (* DOS exe header - common to all images *)
    m_NEHeader   : TNEHeader;         (* Win16 header *)
    m_PEHeader   : TIMAGE_NT_HEADERS; (* Win32 header *)
  end;

implementation

constructor TResourceFile.Create ( szFileName: PANSICHAR );
var nReadCount : CARDINAL;
begin
  (* must have a valid szFileName .. *)
  if (not(Assigned(szFileName))) or
     (IsBadStringPtr(szFileName, MAX_PATH)) then
    raise EResourceError.Create('File name is NIL');

  (* might raise a memory exception here *)
  m_szFileName := StrNew(szFileName);

  (* make sure IOResult doesn't cause a problem to keep us from opening the file *)
  if (IOResult = 0) then;
  AssignFile(m_file, m_szFileName);
  Reset(m_file, 1);
  if (IOResult <> 0) then
    raise EResourceError.Create('Unable to open file');

  BlockRead(m_file, m_MZHeader, SizeOf(m_MZHeader), nReadCount);
  if (nReadCount <> SizeOf(m_MZHeader)) or
     (m_MZHeader.e_magic <> IMAGE_DOS_SIGNATURE) or
     (m_MZHeader.e_lfanew = 0) then
    raise EResourceError.Create('Invalid image');

  (*
   * The two blocks below assume that all PEs and NEs are large
   * enough to contain both the PE header and an NE header..
   *
   * Hopefully, this assumption will never be proven incorrect.
   *)

  Seek(m_file, m_MZHeader.e_lfanew);
  BlockRead(m_file, m_NEHeader, SizeOf(m_NEHeader), nReadCount);
  if (nReadCount <> SizeOf(m_NEHeader)) then
    raise EResourceError.Create('Invalid image');

  Seek(m_file, m_MZHeader.e_lfanew);
  BlockRead(m_file, m_PEHeader, SizeOf(m_PEHeader), nReadCount);
  if (nReadCount <> SizeOf(m_PEHeader)) then
    raise EResourceError.Create('Invalid image');

  if (m_NEHeader.ne_magic = IMAGE_OS2_SIGNATURE) then
  begin
    m_bPortable := FALSE;
  end else
  if (m_PEHeader.Signature = IMAGE_NT_SIGNATURE) then
  begin
    m_bPortable := TRUE;
  end else
    raise EResourceError.Create('Invalid image, must be a PE or NE.');
end;

function TResourceFile.FindIcon ( szResourceID: PANSICHAR ): LONGINT;
begin
  Result := 0;
  if (Assigned(szResourceID)) then
  begin
    (* Punt to the appropriate routine for the given file *)
    if (m_bPortable) then
      Result := FindPEIcon(szResourceID)
    else
      Result := FindNEIcon(szResourceID);
  end;
end;

function TResourceFile.GetNEIconDIB ( szResourceID: PANSICHAR ): LONGINT;
var ResTable    : LONGINT;
    ResAlign    : WORD;

    ResType     : WORD;
    ResCount    : WORD;
    ResReserved : LONGINT;

    nCount      : INTEGER;

    ResOffset   : WORD;
    ResLength   : WORD;
    ResFlags    : WORD;
    ResID       : WORD;
    ResResd     : LONGINT;

  function CompareName ( Name: WORD ): BOOLEAN;
  var SavedPos  : LONGINT;
      nLen      : BYTE;
      NameChars : Array [ 0 .. 255 ] of CHAR;
  begin
    Result := FALSE;
    if (HIWORD(LONGINT(szResourceID)) = 0) then
    begin
      if (Name and $8000 = $8000) then
      begin
        Result := (Name and not($8000) = LONGINT(szResourceID));
      end;
    end else
    begin
      if (Name and $8000 = 0) then
      begin
        SavedPos := FilePos(m_File);

        Seek(m_File, ResTable + Name);
        BlockRead(m_File, nLen, SizeOf(nLen));
        BlockRead(m_File, NameChars, nLen);

        NameChars[nLen] := #0;
        Result := (StrComp(szResourceID, NameChars) = 0);

        Seek(m_file, SavedPos);
      end;
    end;
  end;

begin
  ResTable := LONGINT(m_MZHeader.e_lfanew) + m_NEHeader.ne_rsrctab;

  Seek(m_File, ResTable);
  BlockRead(m_File, ResAlign, SizeOf(ResAlign));

  BlockRead(m_File, ResType, SizeOf(ResType));
  BlockRead(m_File, ResCount, SizeOf(ResCount));
  BlockRead(m_File, ResReserved, SizeOf(ResReserved));

  while (ResType <> 0) do
  begin

    if (ResType and $8000 = $8000) and
       ((ResType and not ($8000)) = LONGINT(RT_ICON)) then
    begin
      for nCount := 0 to (INTEGER(ResCount) - 1) do
      begin
        BlockRead(m_File, ResOffset, SizeOf(ResOffset));
        BlockRead(m_File, ResLength, SizeOf(ResLength));
        BlockRead(m_File, ResFlags, SizeOf(ResFlags));
        BlockRead(m_File, ResID, SizeOf(ResID));
        BlockRead(m_File, ResResd, SizeOf(ResResd));

        if (CompareName(ResID)) then
        begin
          Result := ResOffset * (1 shl ResAlign);
          break;
        end;
      end;
      break;
    end else
      Seek(m_File, FilePos(m_File) + ResCount * 12);

    BlockRead(m_File, ResType, SizeOf(ResType));
    BlockRead(m_File, ResCount, SizeOf(ResCount));
    BlockRead(m_File, ResReserved, SizeOf(ResReserved));
  end;
end;

function TResourceFile.FindNEIcon ( szResourceID: PANSICHAR ): LONGINT;
var ResTable    : LONGINT;
    ResAlign    : WORD;

    ResType     : WORD;
    ResCount    : WORD;
    ResReserved : LONGINT;

    nCount      : INTEGER;

    ResOffset   : WORD;
    ResLength   : WORD;
    ResFlags    : WORD;
    ResID       : WORD;
    ResResd     : LONGINT;

  function CompareName ( Name: WORD ): BOOLEAN;
  var SavedPos  : LONGINT;
      nLen      : BYTE;
      NameChars : Array [ 0 .. 255 ] of CHAR;
  begin
    Result := FALSE;
    if (HIWORD(LONGINT(szResourceID)) = 0) then
    begin
      if (Name and $8000 = $8000) then
      begin
        Result := (Name and not($8000) = LONGINT(szResourceID));
      end;
    end else
    begin
      if (Name and $8000 = 0) then
      begin
        SavedPos := FilePos(m_File);

        Seek(m_File, ResTable + Name);
        BlockRead(m_File, nLen, SizeOf(nLen));
        BlockRead(m_File, NameChars, nLen);

        NameChars[nLen] := #0;
        Result := (StrComp(szResourceID, NameChars) = 0);

        Seek(m_file, SavedPos);
      end;
    end;
  end;

begin
  ResTable := LONGINT(m_MZHeader.e_lfanew) + m_NEHeader.ne_rsrctab;

  Seek(m_File, ResTable);
  BlockRead(m_File, ResAlign, SizeOf(ResAlign));

  BlockRead(m_File, ResType, SizeOf(ResType));
  BlockRead(m_File, ResCount, SizeOf(ResCount));
  BlockRead(m_File, ResReserved, SizeOf(ResReserved));

  while (ResType <> 0) do
  begin

    if (ResType and $8000 = $8000) and
       ((ResType and not ($8000)) = LONGINT(RT_GROUP_ICON)) then
    begin
      for nCount := 0 to (INTEGER(ResCount) - 1) do
      begin
        BlockRead(m_File, ResOffset, SizeOf(ResOffset));
        BlockRead(m_File, ResLength, SizeOf(ResLength));
        BlockRead(m_File, ResFlags, SizeOf(ResFlags));
        BlockRead(m_File, ResID, SizeOf(ResID));
        BlockRead(m_File, ResResd, SizeOf(ResResd));

        if (CompareName(ResID)) then
        begin
          Result := ResOffset * (1 shl ResAlign);
          break;
        end;
      end;
      break;
    end else
      Seek(m_File, FilePos(m_File) + ResCount * 12);

    BlockRead(m_File, ResType, SizeOf(ResType));
    BlockRead(m_File, ResCount, SizeOf(ResCount));
    BlockRead(m_File, ResReserved, SizeOf(ResReserved));
  end;

end;

function TResourceFile.SectionHeaderOffset: LONGINT;
begin
  Result := m_MZHeader.e_lfanew + SizeOf(TIMAGE_FILE_HEADER) + SizeOf(TIMAGE_OPTIONAL_HEADER) + SizeOf(LONGINT);
end;

function TResourceFile.ImageDirectoryOffset ( dwIMAGE_DIRECTORY : LONGINT; var VirtualAddress: LONGINT ): LONGINT;
var nSections  : WORD;
    VAImageDir : LONGINT;
    shOffset   : LONGINT;
    sh         : TIMAGE_SECTION_HEADER;
    nReadCount : WORD;
    i          : INTEGER;
begin
  nSections := m_PEHeader.FileHeader.NumberOfSections;
  VirtualAddress := 0;

  (* must be 0 thru (NumberOfRVAandSizes - 1) *)
  if (dwIMAGE_DIRECTORY >= (m_PEHeader.OptionalHeader.NumberOfRvaAndSizes - 1)) then
  begin
    Result := 0;
    exit;
  end;

  (* Locate image directory's RVA *)
  VAImageDir := m_PEHeader.OptionalHeader.dataDirectory[dwIMAGE_DIRECTORY].VirtualAddress;

  (* Find the section containing the image directory *)
  i := 0;
  shOffset := SectionHeaderOffset;
  Seek(m_file, shOffset);
  while (i < nSections) do
  begin
    BlockRead(m_file, sh, SizeOf(sh), nReadCount);
    if (nReadCount = SizeOf(sh)) then
    begin
      if (sh.VirtualAddress <= VAImageDir) and
         (sh.VirtualAddress + sh.SizeOfRawData > VAImageDir) then
      begin
        VirtualAddress := sh.VirtualAddress;
        break;
      end;
    end;
    Inc(i);
  end;

  if (i >= nSections) then
  begin
    Result := 0;
    exit;
  end;

  (* Return the image directory offset *)
  Result := sh.PointerToRawData;
end;

function TResourceFile.GetPEIconDIB ( szResourceID: PANSICHAR ): LONGINT;
var ResourceOffset : LONGINT;
    VirtualAddress : LONGINT;
    Directory      : TIMAGE_RESOURCE_DIRECTORY;
    Entry          : TIMAGE_RESOURCE_DIRECTORY_ENTRY;
    nRead          : CARDINAL;
    nCount         : INTEGER;
    DataOfs        : LONGINT;

  procedure SeekData ( Offset: LONGINT );
  begin
    (* have to mask out the msb *)
    Seek(m_file, ResourceOffset + (Offset and not($80000000)));
  end;

  function CompareName ( Name: LONGINT ): BOOLEAN;
  var SavedPos   : LONGINT;
      NameLength : WORD;
      NameChar   : WORD;
      nRead      : WORD;
      szName     : PCHAR;
  begin
    (*
     * This function does a cheap comparison of ANSI and UNICODE
     * characters.. it should be exact as long as the image is
     * English..
     *)
    Result := FALSE;
    if (HIWORD(LONGINT(szResourceID)) = 0) then
    begin
      if (Name and $80000000 = 0) then
      begin
        if (Name and not($80000000) = LONGINT(szResourceID)) then
        begin
          Result := TRUE;
          exit;
        end;
      end;
    end else
    begin
      if (Name and $80000000 <> 0) then
      begin
        SavedPos := FilePos(m_file);

        SeekData(Name);

        BlockRead(m_file, NameLength, SizeOf(NameLength), nRead);
        if (nRead <> SizeOf(NameLength)) then
        begin
          Seek(m_File, SavedPos);
          exit;
        end;

        (* validate that the name is the proper length *)
        if (NameLength <> StrLen(szResourceID)) then
        begin
          Seek(m_File, SavedPos);
          exit;
        end;

        szName := szResourceID;
        BlockRead(m_file, NameChar, Sizeof(NameChar), nRead);
        if (nRead <> SizeOf(NameChar)) then
        begin
          Seek(m_File, SavedPos);
          exit;
        end;
        while (BYTE(szName^) = NameChar) and
              (szName^ <> #0) do
        begin
          Inc(szName);

          BlockRead(m_file, NameChar, Sizeof(NameChar), nRead);
          if (nRead <> SizeOf(NameChar)) then
          begin
            Seek(m_File, SavedPos);
            exit;
          end;
        end;

        Result := (szName^ = #0);
        Seek(m_File, SavedPos);
      end;
    end;
  end;

  function GetIconData ( Offset: LONGINT ): LONGINT;
  var Directory      : TIMAGE_RESOURCE_DIRECTORY;
      Entry          : TIMAGE_RESOURCE_DIRECTORY_ENTRY;
      Resource       : TIMAGE_RESOURCE_DATA_ENTRY;
      nRead          : CARDINAL;
  begin
    Result := 0;

    SeekData(Offset);
    BlockRead(m_File, Directory, SizeOf(Directory), nRead);
    if (nRead <> SizeOf(Directory)) then
      exit;

    (* this always assumes the first entry is the correct icon *)
    (* this may not be the case if the icon has multiple versions or languages.. *)

    BlockRead(m_File, Entry, SizeOf(Entry), nRead);
    if (nRead <> SizeOf(Entry)) then
      exit;

    SeekData(Entry.Offset);

    BlockRead(m_File, Resource, SizeOf(Resource), nRead);
    if (nRead <> SizeOf(Resource)) then
      exit;

    Result := (Resource.OffsetToData - VirtualAddress) + ResourceOffset;
  end;

begin
  Result := 0;
  (* Find the offset to the .rsrc section *)
  ResourceOffset := ImageDirectoryOffset(IMAGE_DIRECTORY_ENTRY_RESOURCE, VirtualAddress);

  if (ResourceOffset <> 0) then
  begin
    Seek(m_File, ResourceOffset);
    BlockRead(m_File, Directory, SizeOf(Directory), nRead);
    if (nRead <> SizeOf(Directory)) then
      exit;

    (* find the icon directory .. *)
    for nCount := 0 to (Directory.NumberOfNamedEntries + Directory.NumberOfIDEntries - 1) do
    begin
      BlockRead(m_File, Entry, SizeOf(Entry), nRead);
      if (nRead <> SizeOf(Entry)) then
        exit;

      (* this doesn't bother checking if it's a directory pointer.. just assumes it is *)
      if (Entry.Name = LONGINT(RT_ICON)) then
      begin
        SeekData(Entry.Offset);
        BlockRead(m_File, Directory, SizeOf(Directory), nRead);
        if (nRead <> SizeOf(Directory)) then
          exit;

        (* this could be a bit smarter to just to check the named or id'd entries *)
        for nCount := 0 to (Directory.NumberOfNamedEntries + Directory.NumberOfIDEntries - 1) do
        begin
          BlockRead(m_File, Entry, SizeOf(Entry), nRead);
          if (nRead <> SizeOf(Entry)) then
            exit;

          if (CompareName(Entry.Name)) then
          begin
            Result := GetIconData(Entry.Offset);
            exit;
          end;
        end;
      end;
    end;
  end;
end;

function TResourceFile.FindPEIcon ( szResourceID: PANSICHAR ): LONGINT;
var ResourceOffset : LONGINT;
    VirtualAddress : LONGINT;
    Directory      : TIMAGE_RESOURCE_DIRECTORY;
    Entry          : TIMAGE_RESOURCE_DIRECTORY_ENTRY;
    nRead          : CARDINAL;
    nCount         : INTEGER;
    DataOfs        : LONGINT;

  procedure SeekData ( Offset: LONGINT );
  begin
    (* have to mask out the msb *)
    Seek(m_file, ResourceOffset + (Offset and not($80000000)));
  end;

  function CompareName ( Name: LONGINT ): BOOLEAN;
  var SavedPos   : LONGINT;
      NameLength : WORD;
      NameChar   : WORD;
      nRead      : WORD;
      szName     : PCHAR;
  begin
    (*
     * This function does a cheap comparison of ANSI and UNICODE
     * characters.. it should be exact as long as the image is
     * English..
     *)
    Result := FALSE;
    if (HIWORD(LONGINT(szResourceID)) = 0) then
    begin
      if (Name and $80000000 = 0) then
      begin
        if (Name and not($80000000) = LONGINT(szResourceID)) then
        begin
          Result := TRUE;
          exit;
        end;
      end;
    end else
    begin
      if (Name and $80000000 <> 0) then
      begin
        SavedPos := FilePos(m_file);

        SeekData(Name);

        BlockRead(m_file, NameLength, SizeOf(NameLength), nRead);
        if (nRead <> SizeOf(NameLength)) then
        begin
          Seek(m_File, SavedPos);
          exit;
        end;

        (* validate that the name is the proper length *)
        if (NameLength <> StrLen(szResourceID)) then
        begin
          Seek(m_File, SavedPos);
          exit;
        end;

        szName := szResourceID;
        BlockRead(m_file, NameChar, Sizeof(NameChar), nRead);
        if (nRead <> SizeOf(NameChar)) then
        begin
          Seek(m_File, SavedPos);
          exit;
        end;
        while (BYTE(szName^) = NameChar) and
              (szName^ <> #0) do
        begin
          Inc(szName);

          BlockRead(m_file, NameChar, Sizeof(NameChar), nRead);
          if (nRead <> SizeOf(NameChar)) then
          begin
            Seek(m_File, SavedPos);
            exit;
          end;
        end;

        Result := (szName^ = #0);
        Seek(m_File, SavedPos);
      end;
    end;
  end;

  function GetIconData ( Offset: LONGINT ): LONGINT;
  var Directory      : TIMAGE_RESOURCE_DIRECTORY;
      Entry          : TIMAGE_RESOURCE_DIRECTORY_ENTRY;
      Resource       : TIMAGE_RESOURCE_DATA_ENTRY;
      nRead          : CARDINAL;
  begin
    Result := 0;

    SeekData(Offset);
    BlockRead(m_File, Directory, SizeOf(Directory), nRead);
    if (nRead <> SizeOf(Directory)) then
      exit;

    (* this always assumes the first entry is the correct icon *)
    (* this may not be the case if the icon has multiple versions or languages.. *)

    BlockRead(m_File, Entry, SizeOf(Entry), nRead);
    if (nRead <> SizeOf(Entry)) then
      exit;

    SeekData(Entry.Offset);

    BlockRead(m_File, Resource, SizeOf(Resource), nRead);
    if (nRead <> SizeOf(Resource)) then
      exit;

    Result := (Resource.OffsetToData - VirtualAddress) + ResourceOffset;
  end;

begin
  Result := 0;
  (* Find the offset to the .rsrc section *)
  ResourceOffset := ImageDirectoryOffset(IMAGE_DIRECTORY_ENTRY_RESOURCE, VirtualAddress);

  if (ResourceOffset <> 0) then
  begin
    Seek(m_File, ResourceOffset);
    BlockRead(m_File, Directory, SizeOf(Directory), nRead);
    if (nRead <> SizeOf(Directory)) then
      exit;

    (* find the icon directory .. *)
    for nCount := 0 to (Directory.NumberOfNamedEntries + Directory.NumberOfIDEntries - 1) do
    begin
      BlockRead(m_File, Entry, SizeOf(Entry), nRead);
      if (nRead <> SizeOf(Entry)) then
        exit;

      (* this doesn't bother checking if it's a directory pointer.. just assumes it is *)
      if (Entry.Name = LONGINT(RT_GROUPICON)) then
      begin
        SeekData(Entry.Offset);
        BlockRead(m_File, Directory, SizeOf(Directory), nRead);
        if (nRead <> SizeOf(Directory)) then
          exit;

        (* this could be a bit smarter to just to check the named or id'd entries *)
        for nCount := 0 to (Directory.NumberOfNamedEntries + Directory.NumberOfIDEntries - 1) do
        begin
          BlockRead(m_File, Entry, SizeOf(Entry), nRead);
          if (nRead <> SizeOf(Entry)) then
            exit;

          if (CompareName(Entry.Name)) then
          begin
            Result := GetIconData(Entry.Offset);
            exit;
          end;
        end;
      end;
    end;
  end;
end;

function TResourceFile.UpdateIcon ( szResourceID: PANSICHAR; szIcon: PANSICHAR ): BOOLEAN;
begin
  Result := FALSE;
  if (Assigned(szResourceID)) then
  begin
    (* Punt to the appropriate routine for the given file *)
    if (m_bPortable) then
      Result := UpdatePEIcon(szResourceID, szIcon)
    else
      Result := UpdateNEIcon(szResourceID, szIcon);
  end;
end;

function TResourceFile.UpdatePEIcon ( szResourceID: PANSICHAR; szIcon: PANSICHAR ): BOOLEAN;
var fileIcon : FILE;
    Header   : TFILE_ICON_HEADER;
    Icon     : TFILE_ICON_RESOURCE;
    nRead    : WORD;
    nCount   : INTEGER;
    IconData : LONGINT;
    SavedPos : LONGINT;
    Buffer   : PCHAR;    (* This is used as a standard buffer - strAlloc'd so I don't have to size it for free *)

  function FindResource: LONGINT;
  var GroupOffset : LONGINT;
      ResHeader   : TICON_HEADER;
      ResResource : TICON_RESOURCE;
      nRead       : WORD;
      nCount      : INTEGER;
  begin
    Result := 0;

    (* uses whatever is in the current Icon variable for a match *)
    GroupOffset := FindPEIcon(szResourceID);
    if (GroupOffset > 0) then
    begin
      Seek(m_File, GroupOffset);
      BlockRead(m_File, ResHeader, SizeOf(ResHeader), nRead);
      if (nRead <> SizeOf(ResHeader)) then
        exit;

      (* the INT typecast here is just in some really wierd case there are 0 icons.. *)
      for nCount := 0 to (INTEGER(ResHeader.wCount) - 1) do
      begin
        BlockRead(m_File, ResResource, SizeOf(ResResource), nRead);
        if (nRead <> SizeOf(ResResource)) then
          exit;

        if (ResResource.bWidth = Icon.bWidth) and
           (ResResource.bHeight = Icon.bHeight) and
           (ResResource.bColorCount = Icon.bColorCount) then
        begin
          Result := GetPEIconDIB(MAKEINTRESOURCE(ResResource.wNameOrdinal));
          exit;
        end;
      end;
    end;
  end;

begin
  Result := FALSE;
  if (IOResult = 0) then;
  Assign(fileIcon, szIcon);
  Reset(fileIcon, 1);
  if (IOResult = 0) then
  begin
    BlockRead(fileIcon, Header, SizeOf(Header), nRead);
    if (nRead <> SizeOf(Header)) then
    begin
      Close(fileIcon);
      exit;
    end;

    for nCount := 0 to (Header.wCount - 1) do
    begin
      BlockRead(fileIcon, Icon, SizeOf(Icon), nRead);
      if (nRead <> SizeOf(Icon)) then
      begin
        Close(fileIcon);
        exit;
      end;

      IconData := FindResource;
      if (IconData <> 0) then
      begin
        SavedPos := FilePos(fileIcon);
        Seek(fileIcon, Icon.lImageOffset);

        Buffer := StrAlloc(Icon.lBytesInRes);
        if (Assigned(Buffer)) then
        begin
          (* assumes this read works *)
          BlockRead(fileIcon, Buffer^, Icon.lBytesInRes, nRead);

          Seek(m_file, IconData);
          BlockWrite(m_file, Buffer^, Icon.lBytesInRes);

          StrDispose(Buffer);
        end;

        Seek(fileIcon, SavedPos);
      end;
    end;

    Close(fileIcon);
  end;
end;

function TResourceFile.UpdateNEIcon ( szResourceID: PANSICHAR; szIcon: PANSICHAR ): BOOLEAN;
var fileIcon : FILE;
    Header   : TFILE_ICON_HEADER;
    Icon     : TFILE_ICON_RESOURCE;
    nRead    : WORD;
    nCount   : INTEGER;
    IconData : LONGINT;
    SavedPos : LONGINT;
    Buffer   : PCHAR;    (* This is used as a standard buffer - strAlloc'd so I don't have to size it for free *)

  function FindResource: LONGINT;
  var GroupOffset : LONGINT;
      ResHeader   : TICON_HEADER;
      ResResource : TICON_RESOURCE;
      nRead       : WORD;
      nCount      : INTEGER;
  begin
    Result := 0;

    (* uses whatever is in the current Icon variable for a match *)
    GroupOffset := FindNEIcon(szResourceID);
    if (GroupOffset > 0) then
    begin
      Seek(m_File, GroupOffset);
      BlockRead(m_File, ResHeader, SizeOf(ResHeader), nRead);
      if (nRead <> SizeOf(ResHeader)) then
        exit;

      (* the INT typecast here is just in some really wierd case there are 0 icons.. *)
      for nCount := 0 to (INTEGER(ResHeader.wCount) - 1) do
      begin
        BlockRead(m_File, ResResource, SizeOf(ResResource), nRead);
        if (nRead <> SizeOf(ResResource)) then
          exit;

        if (ResResource.bWidth = Icon.bWidth) and
           (ResResource.bHeight = Icon.bHeight) and
           (ResResource.bColorCount = Icon.bColorCount) then
        begin
          Result := GetNEIconDIB(MAKEINTRESOURCE(ResResource.wNameOrdinal));
          exit;
        end;
      end;
    end;
  end;

begin
  Result := FALSE;
  if (IOResult = 0) then;
  Assign(fileIcon, szIcon);
  Reset(fileIcon, 1);
  if (IOResult = 0) then
  begin
    BlockRead(fileIcon, Header, SizeOf(Header), nRead);
    if (nRead <> SizeOf(Header)) then
    begin
      Close(fileIcon);
      exit;
    end;

    for nCount := 0 to (Header.wCount - 1) do
    begin
      BlockRead(fileIcon, Icon, SizeOf(Icon), nRead);
      if (nRead <> SizeOf(Icon)) then
      begin
        Close(fileIcon);
        exit;
      end;

      IconData := FindResource;
      if (IconData <> 0) then
      begin
        SavedPos := FilePos(fileIcon);
        Seek(fileIcon, Icon.lImageOffset);

        Buffer := StrAlloc(Icon.lBytesInRes);
        if (Assigned(Buffer)) then
        begin
          (* assumes this read works *)
          BlockRead(fileIcon, Buffer^, Icon.lBytesInRes, nRead);

          Seek(m_file, IconData);
          BlockWrite(m_file, Buffer^, Icon.lBytesInRes);

          StrDispose(Buffer);
        end;

        Seek(fileIcon, SavedPos);
      end;
    end;

    Close(fileIcon);
  end;
end;

function TResourceFile.EnumPEIcons ( lpfnCallback: TICONENUMERATOR ): BOOLEAN;
var ResourceOffset : LONGINT;
    VirtualAddress : LONGINT;
    Directory      : TIMAGE_RESOURCE_DIRECTORY;
    Entry          : TIMAGE_RESOURCE_DIRECTORY_ENTRY;
    nRead          : CARDINAL;
    nCount         : INTEGER;
    DataOfs        : LONGINT;

  procedure SeekData ( Offset: LONGINT );
  begin
    (* have to mask out the msb *)
    Seek(m_file, ResourceOffset + (Offset and not($80000000)));
  end;

  function CompareName ( Name: LONGINT ): BOOLEAN;
  var SavedPos   : LONGINT;
      NameLength : WORD;
      NameChar   : WORD;
      nRead      : WORD;
      nCount     : INTEGER;
      szName     : Array [ 0 .. 255 ] of CHAR;
  begin
    if (Name and $80000000 = 0) then
    begin
      (* Name is an ID *)

      (*
       * Construct name as a MAKEINTRESOURCE for the callback..
       *)
      Result := not(lpfnCallback(MAKEINTRESOURCE(Name and not($80000000))));
    end else
    begin
      (* Name is a lengthed UNICODE string *)
      SavedPos := FilePos(m_file);

      SeekData(Name);

      BlockRead(m_file, NameLength, SizeOf(NameLength), nRead);
      if (nRead <> SizeOf(NameLength)) then
      begin
        Seek(m_File, SavedPos);
        exit;
      end;

      for nCount := 0 to (INTEGER(NameLength) - 1) do
      begin
        BlockRead(m_File, NameChar, SizeOf(NameChar), nRead);
        if (nRead <> SizeOf(NameChar)) then
        begin
          Seek(m_File, SavedPos);
          exit;
        end;

        szName[nCount] := CHR(NameChar);
      end;

      szName[NameLength] := #0;

      Result := not(lpfnCallback(szName));

      Seek(m_File, SavedPos);
    end;
  end;

  function GetIconData ( Offset: LONGINT ): LONGINT;
  var Directory      : TIMAGE_RESOURCE_DIRECTORY;
      Entry          : TIMAGE_RESOURCE_DIRECTORY_ENTRY;
      Resource       : TIMAGE_RESOURCE_DATA_ENTRY;
      nRead          : CARDINAL;
  begin
    Result := 0;

    SeekData(Offset);
    BlockRead(m_File, Directory, SizeOf(Directory), nRead);
    if (nRead <> SizeOf(Directory)) then
      exit;

    (* this always assumes the first entry is the correct icon *)
    (* this may not be the case if the icon has multiple versions or languages.. *)

    BlockRead(m_File, Entry, SizeOf(Entry), nRead);
    if (nRead <> SizeOf(Entry)) then
      exit;

    SeekData(Entry.Offset);

    BlockRead(m_File, Resource, SizeOf(Resource), nRead);
    if (nRead <> SizeOf(Resource)) then
      exit;

    Result := (Resource.OffsetToData - VirtualAddress) + ResourceOffset;
  end;

begin
  Result := FALSE;
  (* Find the offset to the .rsrc section *)
  ResourceOffset := ImageDirectoryOffset(IMAGE_DIRECTORY_ENTRY_RESOURCE, VirtualAddress);

  if (ResourceOffset <> 0) then
  begin
    Seek(m_File, ResourceOffset);
    BlockRead(m_File, Directory, SizeOf(Directory), nRead);
    if (nRead <> SizeOf(Directory)) then
      exit;

    (* find the icon directory .. *)
    for nCount := 0 to (Directory.NumberOfNamedEntries + Directory.NumberOfIDEntries - 1) do
    begin
      BlockRead(m_File, Entry, SizeOf(Entry), nRead);
      if (nRead <> SizeOf(Entry)) then
        exit;

      (* assume everything worked at this point *)
      Result := TRUE;

      (* this doesn't bother checking if it's a directory pointer.. just assumes it is *)
      if (Entry.Name = LONGINT(RT_GROUPICON)) then
      begin
        SeekData(Entry.Offset);
        BlockRead(m_File, Directory, SizeOf(Directory), nRead);
        if (nRead <> SizeOf(Directory)) then
          exit;

        Result := TRUE;

        (* this could be a bit smarter to just to check the named or id'd entries *)
        for nCount := 0 to (Directory.NumberOfNamedEntries + Directory.NumberOfIDEntries - 1) do
        begin
          BlockRead(m_File, Entry, SizeOf(Entry), nRead);
          if (nRead <> SizeOf(Entry)) then
            exit;

          (* CompareName returns the result of lpfnCallback *)
          if (CompareName(Entry.Name)) then
          begin
            exit;
          end;
        end;
      end;
    end;
  end;
end;

function TResourceFile.EnumNEIcons ( lpfnCallback: TICONENUMERATOR ): BOOLEAN;
var ResTable    : LONGINT;
    ResAlign    : WORD;

    ResType     : WORD;
    ResCount    : WORD;
    ResReserved : LONGINT;

    nCount      : INTEGER;

    ResOffset   : WORD;
    ResLength   : WORD;
    ResFlags    : WORD;
    ResID       : WORD;
    ResResd     : LONGINT;

  function CompareName ( Name: WORD ): BOOLEAN;
  var SavedPos  : LONGINT;
      nLen      : BYTE;
      NameChars : Array [ 0 .. 255 ] of CHAR;
  begin
    Result := FALSE;
    if (Name and $8000 = $8000) then
    begin
      Result := not(lpfnCallback(MAKEINTRESOURCE(Name and not($8000))));
    end else
    begin
      SavedPos := FilePos(m_File);

      Seek(m_File, ResTable + Name);
      BlockRead(m_File, nLen, SizeOf(nLen));
      BlockRead(m_File, NameChars, nLen);

      NameChars[nLen] := #0;
      Result := not(lpfnCallback(NameChars));

      Seek(m_file, SavedPos);
    end;
  end;

begin
  ResTable := LONGINT(m_MZHeader.e_lfanew) + m_NEHeader.ne_rsrctab;

  Seek(m_File, ResTable);
  BlockRead(m_File, ResAlign, SizeOf(ResAlign));

  BlockRead(m_File, ResType, SizeOf(ResType));
  BlockRead(m_File, ResCount, SizeOf(ResCount));
  BlockRead(m_File, ResReserved, SizeOf(ResReserved));

  while (ResType <> 0) do
  begin
    if (ResType and $8000 = $8000) and
       ((ResType and not ($8000)) = LONGINT(RT_GROUP_ICON)) then
    begin
      for nCount := 0 to (INTEGER(ResCount) - 1) do
      begin
        BlockRead(m_File, ResOffset, SizeOf(ResOffset));
        BlockRead(m_File, ResLength, SizeOf(ResLength));
        BlockRead(m_File, ResFlags, SizeOf(ResFlags));
        BlockRead(m_File, ResID, SizeOf(ResID));
        BlockRead(m_File, ResResd, SizeOf(ResResd));

        if (CompareName(ResID)) then
        begin
          break;
        end;
      end;
    end else
      Seek(m_File, FilePos(m_File) + ResCount * 12);

    BlockRead(m_File, ResType, SizeOf(ResType));
    BlockRead(m_File, ResCount, SizeOf(ResCount));
    BlockRead(m_File, ResReserved, SizeOf(ResReserved));
  end;
end;

function TResourceFile.EnumIcons ( lpfnCallback: TICONENUMERATOR ): BOOLEAN;
begin
  Result := FALSE;
  if (Assigned(lpfnCallback)) then
  begin
    (* Punt to the appropriate routine for the given file *)
    if (m_bPortable) then
      Result := EnumPEIcons(lpfnCallback)
    else
      Result := EnumNEIcons(lpfnCallback);
  end;
end;

function TResourceFile.UpdateIconFromImage ( szResourceID: PANSICHAR; ResFile: TResourceFile; szSourceID: PANSICHAR ): BOOLEAN;
begin
  Result := FALSE;
  if (Assigned(szResourceID)) and
     (Assigned(szSourceID)) then
  begin
    (* Punt to the appropriate routine for the given file *)
    if (m_bPortable) then
      Result := UpdatePEIconFromImage(szResourceID, ResFile, szSourceID)
    else
      Result := UpdateNEIconFromImage(szResourceID, ResFile, szSourceID);
  end;
end;

function TResourceFile.UpdatePEIconFromImage ( szResourceID: PANSICHAR; ResFile: TResourceFile;
                                               szSourceID: PANSICHAR ): BOOLEAN;
var GroupOffset : LONGINT;
    Header      : TICON_HEADER;
    Icon        : TICON_RESOURCE;
    nRead       : WORD;
    nCount      : INTEGER;
    SavedPos    : LONGINT;
    SourceDIB   : LONGINT;
    DestDIB     : LONGINT;
    srcBuffer   : PANSICHAR;

  function FindResource: LONGINT;
  var GroupOffset : LONGINT;
      ResHeader   : TICON_HEADER;
      ResResource : TICON_RESOURCE;
      nRead       : WORD;
      nCount      : INTEGER;
  begin
    Result := 0;

    (* uses whatever is in the current Icon variable for a match *)
    GroupOffset := FindPEIcon(szResourceID);
    if (GroupOffset > 0) then
    begin
      Seek(m_File, GroupOffset);
      BlockRead(m_File, ResHeader, SizeOf(ResHeader), nRead);
      if (nRead <> SizeOf(ResHeader)) then
        exit;

      (* the INT typecast here is just in some really wierd case there are 0 icons.. *)
      for nCount := 0 to (INTEGER(ResHeader.wCount) - 1) do
      begin
        BlockRead(m_File, ResResource, SizeOf(ResResource), nRead);
        if (nRead <> SizeOf(ResResource)) then
          exit;

        if (ResResource.bWidth = Icon.bWidth) and
           (ResResource.bHeight = Icon.bHeight) and
           (ResResource.bColorCount = Icon.bColorCount) then
        begin
          Result := GetPEIconDIB(MAKEINTRESOURCE(ResResource.wNameOrdinal));
          exit;
        end;
      end;
    end;
  end;

begin
  Result := FALSE;
  GroupOffset := ResFile.FindIcon(szSourceID);
  if (GroupOffset <> 0) then
  begin
    Seek(ResFile.m_File, GroupOffset);
    BlockRead(ResFile.m_File, Header, SizeOf(Header), nRead);
    if (nRead <> SizeOf(Header)) then
      exit;

    for nCount := 0 to (INTEGER(Header.wCount) - 1) do
    begin
      BlockRead(ResFile.m_File, Icon, SizeOf(Icon), nRead);
      if (nRead <> SizeOf(Icon)) then
        exit;

      SavedPos := FilePos(ResFile.m_File);

      DestDIB := FindResource;
      if (DestDIB <> 0) then
      begin
        Seek(m_File, DestDIB);
        SourceDIB := ResFile.GetIconDIB(MAKEINTRESOURCE(Icon.wNameOrdinal));
        if (SourceDIB <> 0) then
        begin
          Seek(ResFile.m_File, SourceDIB);
          srcBuffer := StrAlloc(Icon.lBytesInRes);
          BlockRead(ResFile.m_File, srcBuffer^, Icon.lBytesInRes);
          BlockWrite(m_File, srcBuffer^, Icon.lBytesInRes);
          StrDispose(srcBuffer);
        end;
      end;

      Seek(ResFile.m_File, SavedPos);
    end;

    Result := TRUE;
  end;
end;

function TResourceFile.UpdateNEIconFromImage ( szResourceID: PANSICHAR; ResFile: TResourceFile;
                                               szSourceID: PANSICHAR ): BOOLEAN;
var GroupOffset : LONGINT;
    Header      : TICON_HEADER;
    Icon        : TICON_RESOURCE;
    nRead       : WORD;
    nCount      : INTEGER;
    SavedPos    : LONGINT;
    SourceDIB   : LONGINT;
    DestDIB     : LONGINT;
    srcBuffer   : PANSICHAR;

  function FindResource: LONGINT;
  var GroupOffset : LONGINT;
      ResHeader   : TICON_HEADER;
      ResResource : TICON_RESOURCE;
      nRead       : WORD;
      nCount      : INTEGER;
  begin
    Result := 0;

    (* uses whatever is in the current Icon variable for a match *)
    GroupOffset := FindNEIcon(szResourceID);
    if (GroupOffset > 0) then
    begin
      Seek(m_File, GroupOffset);
      BlockRead(m_File, ResHeader, SizeOf(ResHeader), nRead);
      if (nRead <> SizeOf(ResHeader)) then
        exit;

      (* the INT typecast here is just in some really wierd case there are 0 icons.. *)
      for nCount := 0 to (INTEGER(ResHeader.wCount) - 1) do
      begin
        BlockRead(m_File, ResResource, SizeOf(ResResource), nRead);
        if (nRead <> SizeOf(ResResource)) then
          exit;

        if (ResResource.bWidth = Icon.bWidth) and
           (ResResource.bHeight = Icon.bHeight) and
           (ResResource.bColorCount = Icon.bColorCount) then
        begin
          Result := GetNEIconDIB(MAKEINTRESOURCE(ResResource.wNameOrdinal));
          exit;
        end;
      end;
    end;
  end;

begin
  Result := FALSE;
  GroupOffset := ResFile.FindIcon(szSourceID);
  if (GroupOffset <> 0) then
  begin
    Seek(ResFile.m_File, GroupOffset);
    BlockRead(ResFile.m_File, Header, SizeOf(Header), nRead);
    if (nRead <> SizeOf(Header)) then
      exit;

    for nCount := 0 to (INTEGER(Header.wCount) - 1) do
    begin
      BlockRead(ResFile.m_File, Icon, SizeOf(Icon), nRead);
      if (nRead <> SizeOf(Icon)) then
        exit;

      SavedPos := FilePos(ResFile.m_File);

      DestDIB := FindResource;
      if (DestDIB <> 0) then
      begin
        Seek(m_File, DestDIB);
        SourceDIB := ResFile.GetIconDIB(MAKEINTRESOURCE(Icon.wNameOrdinal));
        if (SourceDIB <> 0) then
        begin
          Seek(ResFile.m_File, SourceDIB);
          srcBuffer := StrAlloc(Icon.lBytesInRes);
          BlockRead(ResFile.m_File, srcBuffer^, Icon.lBytesInRes);
          BlockWrite(m_File, srcBuffer^, Icon.lBytesInRes);
          StrDispose(srcBuffer);
        end;
      end;

      Seek(ResFile.m_File, SavedPos);
    end;

    Result := TRUE;
  end;
end;

function TResourceFile.GetIconDIB ( szResourceID: PANSICHAR ): LONGINT;
begin
  (* Punt to the appropriate routine for the given file *)
  if (m_bPortable) then
    Result := GetPEIconDIB(szResourceID)
  else
    Result := GetNEIconDIB(szResourceID);
end;

destructor TResourceFile.Destroy;
begin
  (* make sure the file is open before we close it *)
  if (TFILEREC(m_file).mode <> fmClosed) then
    Close(m_file);
end;


end.
