{******************************************************}
{                rsRFM Classes V2.6                    }
{        Copyright 1997 RealSoft Development           }
{           support:  www.realsoftdev.com              }
{                   -------------                      }
{ Although this unit is owned by RealSoft, It may be   }
{ freely distributed in its entirety only. You may not }
{ copy any portion of this code without permission.    }
{******************************************************}

{$DEFINE _FLY}
{$DEFINE BDE}

unit rsRfm;

interface

uses
  SysUtils, WinTypes, WinProcs, Messages, Classes, Graphics, Controls, Buttons, Forms,
  Dialogs, StdCtrls, ExtCtrls, Printers, DsgnIntf, Menus, FileCtrl, rsDynSlm, rsRfmrec
  {$IFDEF FLY},FlyCtrl{$ENDIF} {$IFDEF BDE},DbCtrls{$ENDIF};

const
  INCH = 96;
  VERSIONMAJOR = 2;
  VERSIONMINOR = 6;

type
  {$IFDEF WIN32}
  {$WARNINGS OFF}
  {$ENDIF}
  TRFM = class(TPersistent)
    private
      FOnChange: TNotifyEvent;
      FDesigning: boolean;
      function GetCanvas: TCanvas;
      function GetEmpty: boolean;
    protected
      procedure DefineProperties(Filer : TFiler); override;
    public
      Bitmap      : TBitmap;
      DynArray    : TrsDynArray;
      Header      : RFormHdr;
      StrTable    : TStrings;
      DefTextFont : String[60];
      DefTextSize : smallint;
      DefFldFont  : String[60];
      DefFldSize  : smallint;
      Warnings    : boolean;
      constructor Create;
      destructor Destroy; override;
      procedure Clear;
      procedure ClearBMP;
      procedure Assign(Source: TRFM);
      procedure Draw;
      procedure ScaleDraw(MyRect: TRect);
      procedure DrawTo(MyRect: TRect; MyCanvas: TCanvas; Clear: boolean);
      {file access}
      procedure MergeRFM(filename: String);
      procedure LoadFromfile(filename: String);
      procedure SaveToFile(filename: String);
      procedure LoadFromStream(Stream: TStream);
      procedure SaveToStream(Stream: TStream);
      function  LoadRFL(filename: String) : boolean;
      function  SaveRFL(filename: String) : boolean;
      procedure SaveAsBitmap(filename: String);
      procedure SaveAsMetafile(filename: String);
      {Objects}
      procedure DrawObj(MyCanvas : TCanvas; MyRect : TRect; Num : smallint);
      function  AddObject( DType : TDrawType; X: smallint; Y: smallint ): smallint;
      procedure DeleteObj(Index : smallint);
      {StrTable}
      function  AddString( S: String ) : smallint;
      procedure DelString( Index : smallint );
      procedure UpdString( S: String; Index : smallint );
      procedure FixString( Index: smallint; RFMRec : smallint );
      function  GetString( Index: smallint ) : String;
      {Misc}
      function  ColorToByte (AColor : TColor) : Byte;
      function  ByteToColor (AByte: byte) : TColor;
      function  VerifyName(N: String): String;
      function  NameCheck( Name : String; Index: smallint) : boolean;
      function  NextName: String;
      function  MakeRect: TRect;
      procedure SizeText( Index : smallint);
      procedure SizeField( Index : smallint);

      property  Width: word               read Header.Width;
      property  Height: word              read Header.Height;
      property  Canvas: TCanvas           read GetCanvas;
      property  Empty: boolean            read GetEmpty;
      property  Designing: boolean        read FDesigning    write FDesigning;
      property  OnChange: TNotifyEvent    read FOnChange     write FOnChange;
    end;
  {$IFDEF WIN32}
  {$WARNINGS ON}
  {$ENDIF}

  TOnPrintField = procedure ( Sender: TObject; FieldName: String; Format: TFieldFormat; var Text : String ) of object;

  TDuplexType = (dtNone, dtHorizontal, dtVertical);

  TrsRFMControl = class(TGraphicControl);

  TrsRFMImage = class(TrsRFMControl)
  private
    FRFM : TRFM;
    FAutoSize: Boolean;
    FStretch: Boolean;
    FCenter: Boolean;
    FAspect: Boolean;
    FPaperSize: TPaperSize;
    FOrientation: TOrientation;
    FNumCopies: smallint;
    FAutoFill: Boolean;
    FAutoLogo: Boolean;
    FAutoFields: Boolean;
    FFillList: TStrings;
    FLogoList: TStrings;
    FDuplexRFM: TrsRFMControl;
    FDuplexType: TDuplexType;
    FOnChange: TNotifyEvent;
    FOnPrintField: TOnPrintField;
    procedure UpdateRFM;
    function  CalcRect : TRect;
    procedure RFMChanged(Sender : TObject);
    procedure SetAspect(AValue: Boolean);
    procedure SetAutoSize(AValue: Boolean);
    procedure SetCenter(AValue: Boolean);
    procedure SetStretch(AValue: Boolean);
    procedure SetRFM(AValue: TRFM);
    procedure SetFillList(AValue: TStrings);
    procedure SetLogoList(AValue: TStrings);
    function  GetPageWidth: word;
    function  GetPageHeight: word;
    procedure SetPaperSize(AValue: TPaperSize);
    procedure SetOrientation(AValue: TOrientation);
    procedure SetNumCopies(AValue: smallint);
  protected
    procedure Paint; override;
    procedure WMSize(var Message : TWMSize); message WM_SIZE;
    procedure Notification( AComponent : TComponent; Operation : TOperation); override;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    function  RFMRect : TRect;
    procedure Clear;
    procedure Print;
    procedure PrintDirect;
    procedure ModPrintOptions;
    procedure PrintTo(ACanvas : TCanvas; MyRect : TRect);
    procedure FillTo(ACanvas : TCanvas; MyRect : TRect);
    procedure LogoTo(ACanvas : TCanvas; MyRect : TRect);
    procedure FieldsTo(ACanvas : TCanvas; MyRect : TRect);
  published
    property AutoSize: Boolean         read FAutoSize      write SetAutoSize default False;
    property AutoFill: Boolean         read FAutoFill      write FAutoFill   default False;
    property AutoLogo: Boolean         read FAutoLogo      write FAutoLogo   default False;
    property AutoFields: Boolean       read FAutoFields    write FAutoFields default False;
    property Center: Boolean           read FCenter        write SetCenter   default False;
    property Stretch: Boolean          read FStretch       write SetStretch  default False;
    property Aspect: Boolean           read FAspect        write SetAspect   default True;
    property RFM: TRFM                 read FRFM           write SetRFM;
    property FillList: TStrings        read FFillList      write SetFillList;
    property LogoList: TStrings        read FLogoList      write SetLogoList;
    property PaperSize: TPaperSize     read FPaperSize     write SetPaperSize;
    property Orientation: TOrientation read FOrientation   write SetOrientation;
    property NumCopies: smallint       read FNumCopies     write SetNumCopies;
    property DuplexRFM: TrsRFMControl  read FDuplexRFM     write FDuplexRFM;
    property DuplexType: TDuplexType   read FDuplexType    write FDuplexType;
    property PageWidth: word           read GetPageWidth;
    property PageHeight: word          read GetPageHeight;
    property Align;
    property DragCursor;
    property DragMode;
    property Enabled;
    property ParentShowHint;
    property PopupMenu;
    property ShowHint;
    property Visible;
    property OnClick;
    property OnDblClick;
    property OnDragDrop;
    property OnDragOver;
    property OnEndDrag;
    property OnMouseDown;
    property OnMouseMove;
    property OnMouseUp;
    property OnChange: TNotifyEvent       read FOnChange      write FOnChange;
    property OnPrintField : TOnPrintField read FOnPrintField  write FOnPrintField;
  end;

  TRFMProperty = class( TPropertyEditor )
  private
    Dialog : TForm;
    Label1 : TLabel;
    DirectoryListBox1 : TDirectoryListBox;
    FileListBox1 : TFileListBox;
    DriveComboBox1 : TDriveComboBox;
    Edit1 : TEdit;
    btnCancel : TBitBtn;
    btnOK : TBitBtn;
    btnClear : TBitBtn;
    ScrollBox1 : TScrollBox;
    RFMImage1 : TrsRFMImage;
    ckScale : TCheckBox;
    btnSaveBMP : TBitBtn;
    btnSaveWMF : TBitBtn;
    btnSaveRFM : TBitBtn;
    SaveDialog1 : TSaveDialog;
    SaveDialog2 : TSaveDialog;
    SaveDialog3 : TSaveDialog;
    procedure ckScaleMouseUp(Sender: TObject; Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
    procedure ButtonClick(Sender : TObject);
    procedure ButtonDblClick(Sender : TObject);
    procedure CreateForm;
    procedure Edit; override;
    function GetAttributes : TPropertyAttributes; override;
    function GetValue : string; override;
  end;

  TRFMEditor = class(TDefaultEditor)
  public
    procedure EditProperty(PropertyEditor: TPropertyEditor; var Continue, FreeEditor: Boolean); override;
  end;

procedure register;

implementation
{$IFDEF WIN32}
{$R *.R32}
{$ELSE}
{$R *.R16}
{$ENDIF}

procedure register;
begin
  RegisterComponents('RSD', [TrsRFMImage]);
  RegisterPropertyEditor(TypeInfo(TRFM), nil, '', TRFMProperty);
  RegisterComponentEditor(TrsRFMImage, TRFMEditor);
end;

{*****************************}
{     TRFM Public Methods     }
{*****************************}
constructor TRFM.Create;
begin
  inherited Create;
  Dynarray    := TrsDynArray.Create(sizeof(RFormRec));
  StrTable    := TStringList.Create;
  Bitmap      := TBitmap.Create;
  Designing   := false;
  FillChar(Header, sizeof(RFormHdr), 0);
  DefTextFont := 'Arial';
  DefTextSize := 10;
  DefFldFont  := 'Courier New';
  DefFldSize  := 10;
  Warnings    := true;
end;

destructor TRFM.Destroy;
begin
  Bitmap.Free;
  Dynarray.Free;
  StrTable.Free;
  inherited Destroy;
end;

procedure TRFM.DefineProperties(Filer : TFiler);
begin
  inherited DefineProperties(Filer);
  Filer.DefineBinaryProperty('RFMData', LoadFromStream, SaveToStream, True);
end;

function TRFM.GetEmpty : boolean;
begin
  Result:= Dynarray.Count = 0;
end;

function TRFM.GetCanvas: TCanvas;
begin
  Result:= Bitmap.Canvas;
end;

procedure TRFM.Clear;
begin
  StrTable.Clear;
  Dynarray.Clear;
  FillChar(Header, sizeof(RFormHdr), 0);
  ClearBMP;
  if Assigned(FOnChange) then FOnChange(Self); {notify OnChange}
end;

procedure TRFM.Assign(Source: TRFM);
begin
  Clear;
  if not Source.Empty then begin
    Dynarray.Assign(Source.Dynarray);
    StrTable.Assign(Source.StrTable);
    Header := Source.Header;
    end;
  if Assigned(FOnChange) then FOnChange(Self); {notify OnChange}
end;

procedure TRFM.Draw;
var R : TRect;
begin
  R:= MakeRect;
  ClearBMP;
  Bitmap.Free;
  Bitmap   := TBitmap.Create;
  Bitmap.Width:= R.Right - R.Left;
  Bitmap.Height:= R.Bottom - R.Top;
  DrawTo(R, Bitmap.Canvas, true);
end;

procedure TRFM.ScaleDraw(MyRect : TRect);
var R : TRect;
begin
  ClearBMP;
  Bitmap.Free;
  Bitmap   := TBitmap.Create;
  Bitmap.Width:= MyRect.Right - MyRect.Left;
  Bitmap.Height:= MyRect.Bottom - MyRect.Top;
  R:= Rect(0,0,Bitmap.Width, Bitmap.Height);
  DrawTo(R, Bitmap.Canvas, true);
end;

procedure TRFM.DrawTo(MyRect: TRect; MyCanvas: TCanvas; Clear: boolean);
var
  x: smallint;
  HScale{ ,VScale}: real;
begin
  if Empty then Exit;

  HScale:= (myRect.Right - myRect.Left) / Header.Width;
  {VScale:= (myRect.Bottom - myRect.Top) / Header.Height;}

  with MyCanvas do begin
    Brush.Style := bsSolid;
    Brush.Color := clWhite;
    Pen.Color:= clBlack;
    Pen.Style:= psClear;
    Pen.Width:= trunc(Header.Thick * HScale);
    if Header.PStyle = psSolid then Pen.style:=psSolid;
    if Header.PStyle = psDash  then Pen.style:=psDash;
    if Header.PStyle = psDot   then Pen.style:=psDot;
    if Header.PStyle = psClear then Pen.style:=psClear;
    {draw border}
    if Clear then FillRect( MyRect )
    else FrameRect( MyRect );
    end;
  for x:= 0 to Dynarray.Count -1 do DrawObj(MyCanvas, MyRect, x);
end;

{****************}
{  File Access   }
{****************}
procedure TRFM.MergeRFM(filename: String);
var
  MS : TMemoryStream;
  tmprec : RFormRec;
  x : smallint;
  MHeader   : RFormHdr;
  MStrTable : TStrings;
  MDynarray : TrsDynarray;
begin
  MS:= NIL;
  MStrTable:= NIL;
  MDynarray:= NIL;

  try {load RFM to be merged into temp vars}
    MS:= TMemoryStream.Create;
    MStrTable:= TStringList.Create;
    MDynarray:= TrsDynarray.Create(sizeof(RFormRec));
  if fileexists(filename) then begin
    MS.loadfromfile(filename);
    MS.Seek(0,0);
    if MS.Read(MHeader, sizeof(RFormHdr)) = sizeof(RFormHdr) then begin
      if (MHeader.VersionMajor <= 2) and (MHeader.VersionMinor < 1) then begin
        Messagedlg('The RFM has been saved with an obsolete version of Realforms.'+#13+
                   'Please load with the latest version and re-save it before merging.',mtWarning, [mbOK],0);
        Exit;
        end;
      for x:= 0 to MHeader.NumRecs - 1 do
        if MS.Read(tmprec, sizeof (RFormRec)) = sizeof (RFormRec) then
           MDynarray.Add( tmprec );
      MStrTable.LoadFromStream(MS);
      end;
    {Do Merge}
    for x:= 0 to MHeader.NumRecs - 1 do begin
      tmprec:= RFormRec(MDynarray[x]^);
      Dynarray.Add(tmprec);
      if tmprec.DrawType = rfText then begin
        StrTable.Add( MStrTable[RFormText(tmprec).StrIdx] );
        RFormText(Dynarray[Dynarray.Count-1]^).StrIdx:= StrTable.Count-1;
        StrTable.Add( MStrTable[RFormText(tmprec).FontIdx] );
        RFormText(Dynarray[Dynarray.Count-1]^).FontIdx:= StrTable.Count-1;
        end;
      if tmprec.DrawType = rfField then begin
        StrTable.Add( VerifyName(MStrTable[RFormField(tmprec).StrIdx]) );
        RFormField(Dynarray[Dynarray.Count-1]^).StrIdx:= StrTable.Count-1;
        StrTable.Add( MStrTable[RFormField(tmprec).FontIdx] );
        RFormField(Dynarray[Dynarray.Count-1]^).FontIdx:= StrTable.Count-1;
        end;
      end;

    {notify OnChange}
    if Assigned(FOnChange) then FOnChange(Self);
    end;
  finally
    MS.Free;
    MStrTable.Free;
    MDynarray.Free;
  end;
end;

procedure TRFM.Loadfromfile(filename : String);
var
  MS : TMemoryStream;
  tmprec : RFormRec;
  x : smallint;
begin
  MS:= nil;
  try
    MS:= TMemoryStream.Create;
  if fileexists(filename) then begin
    MS.loadfromfile(filename);
    MS.Seek(0,0);
    if MS.Read(Header, sizeof(RFormHdr)) = sizeof(RFormHdr) then begin
      if Header.VersionMajor <= 2 then {check version}
        if Header.VersionMinor < 1 then begin
        case Header.PaperSize of
          psLetter: begin Header.Width:= 800;  Header.Height:= 1035; end;
          psLegal:  begin Header.Width:= 800;  Header.Height:= 1320; end;
          psA3:     begin Header.Width:= 1100; Header.Height:= 1556; end;
          psA4:     begin Header.Width:= 778;  Header.Height:= 1095; end;
          psA5:     begin Header.Width:= 551;  Header.Height:= 778;  end;
          else      begin Header.Width:= 800;  Header.Height:= 1035; end;
          end;
        Header.VersionMajor := VERSIONMAJOR;
        Header.VersionMinor := VERSIONMINOR;
        Header.Orientation:= orPortrait;
        end;
      DynArray.Clear;
      for x:= 0 to Header.NumRecs - 1 do
        if MS.Read(tmprec, sizeof (RFormRec)) = sizeof (RFormRec) then
           Dynarray.Add( tmprec );
      StrTable.LoadFromStream(MS);
      end;
    if Header.VersionMajor <= 2 then {check version for scaling bug}
      if Header.VersionMinor < 6 then
        if Warnings then
          Messagedlg('This RFM has been saved with an obsolete version of Realforms.'+#13+
                     'Please load with the latest version and re-save it before continuing.',
                     mtWarning, [mbOK],0);

    {notify OnChange}
    if Assigned(FOnChange) then FOnChange(Self);
    end;
  finally
    MS.Free;
  end;
end;

procedure TRFM.SavetoFile(filename : String);
var
  x : smallint;
  MS : TMemoryStream;
begin
  MS:= nil;
  try
    MS:= TMemoryStream.Create;
  if DynArray.Count > 0 then begin
    Header.NumRecs:= DynArray.Count;
    MS.Write(Header, sizeof(RFormHdr));
    for x:= 0 to DynArray.Count - 1 do
      MS.Write( DynArray[x]^, sizeof(RFormRec) );
    StrTable.SavetoStream(MS);
    MS.SavetoFile(filename);
    end;
  finally
    MS.Free;
  end;
end;

procedure TRFM.LoadfromStream(Stream : TStream);
var
  tmprec : RFormRec;
  x : smallint;
begin
  if Stream.Read(Header, sizeof(RFormHdr)) = sizeof(RFormHdr) then begin
    if Header.VersionMajor <= 2 then {check version}
      if Header.VersionMinor < 1 then begin
        case Header.PaperSize of
          psLetter: begin Header.Width:= 800;  Header.Height:= 1035; end;
          psLegal:  begin Header.Width:= 800;  Header.Height:= 1320; end;
          psA3:     begin Header.Width:= 1100; Header.Height:= 1556; end;
          psA4:     begin Header.Width:= 778;  Header.Height:= 1095; end;
          psA5:     begin Header.Width:= 551;  Header.Height:= 778;  end;
          else      begin Header.Width:= 800;  Header.Height:= 1035; end;
          end;
        Header.VersionMajor := VERSIONMAJOR;
        Header.VersionMinor := VERSIONMINOR;
        Header.Orientation:= orPortrait;
        end;
    DynArray.Clear;
    for x:= 0 to Header.NumRecs - 1 do
      if Stream.Read(tmprec, sizeof (RFormRec)) = sizeof (RFormRec) then
         Dynarray.Add( tmprec );
    StrTable.LoadFromStream(Stream);
    if Header.VersionMajor <= 2 then {check version for scaling bug}
      if Header.VersionMinor < 6 then
        if Warnings then
          Messagedlg('This RFM has been saved with an obsolete version of Realforms.'+#13+
                     'Please load with the latest version and re-save it before continuing.',
                     mtWarning, [mbOK],0);
    {notify OnChange}
    if Assigned(FOnChange) then FOnChange(Self);
    end;
end;

procedure TRFM.SavetoStream(Stream : TStream);
var x : smallint;
begin
  if DynArray.Count > 0 then begin
    Header.NumRecs:= DynArray.Count;
    Stream.Write(Header, sizeof(RFormHdr));
    for x:= 0 to DynArray.Count - 1 do
      Stream.Write( DynArray[x]^, sizeof(RFormRec) );
    StrTable.SavetoStream(Stream);
    end;
end;

procedure TRFM.SaveAsBitmap(filename : String);
var E : boolean;
begin
  E:= Bitmap.Empty;
  if E then Draw;
  Bitmap.Savetofile(filename);
  if E then ClearBMP;
end;

procedure TRFM.SaveAsMetafile(filename : String);
var
  FMeta   : TMetaFile;
  {$IFDEF WIN32}
  FCanvas : TMetaFileCanvas;
  {$ELSE}
  FCanvas : TCanvas;
  {$ENDIF}
begin
  FMeta:= TMetafile.Create;
{$IFDEF WIN32}
  FCanvas:= TMetafileCanvas.Create(FMeta,0);
{$ELSE}
  FCanvas:= TCanvas.Create;
  FCanvas.Handle:= CreateMetaFile(NIL);
  SetMapMode(FCanvas.Handle, MM_ANISOTROPIC);
  SetWindowOrg(FCanvas.Handle, 0, 0);
  SetWindowExt(FCanvas.Handle,longint(width) * INCH div Screen.PixelsPerINCH,
               longint(height) * INCH div Screen.PixelsPerINCH);
  FCanvas.Font.PixelsPerINCH:= INCH;
{$ENDIF}

  DrawTo(Rect(0,0,Header.Width, Header.Height), FCanvas, True);

{$IFDEF WIN32}
  FCanvas.Free;
{$ELSE}
  FMeta.Handle := CloseMetafile(FCanvas.Handle);
{$ENDIF}
  FMeta.Height:= Header.Height;
  FMeta.Width:= Header.Width;
  FMeta.Savetofile(filename);
end;

function TRFM.LoadRFL(filename: String): boolean;
var
  i:       smallint;
  Dyn:     TrsDynarray;
  tmprfm:  RFormField;
  tmprfl:  RRFLField;
  BMP:     TBitmap;
begin
  result:= false;
  Dyn:= nil;
  Bmp:= nil;
  try
  Dyn:= TrsDynarray.Create( Sizeof(RRFLField) );
  BMP:= TBitmap.Create;
  if FileExists(filename) then Dyn.LoadFromfile(Filename);
  if Dyn.count = 0 then Exit;
  result:= true;
  for i:= 0 to Dyn.Count - 1 do begin
    tmprfl:= RRFLField(Dyn[i]^);
    if (tmprfl.ver < 26) or (tmprfl.ver > 29) then begin
      Messagedlg('Obsolete RFL file cannot be imported.',mtError,[mbOk],0);
      result:= false;
      Exit;
      end;
    Tmprfm.DrawType := rfField;
    tmprfl.Name:= VerifyName(tmprfl.Name);
    StrTable.Add( tmprfl.Name );
    Tmprfm.StrIdx   := StrTable.Count-1;
    StrTable.Add( tmprfl.FontName );
    Tmprfm.FontIdx  := StrTable.Count-1;
    Tmprfm.Left     := tmprfl.Left;
    Tmprfm.Top      := tmprfl.Top;
    Tmprfm.FontSize := tmprfl.FontSize;
    Tmprfm.FontStyle:= tmprfl.FontStyle;
    Tmprfm.FontHt   := tmprfl.FontHt;
    Tmprfm.Color    := tmprfl.FontColor;
    Tmprfm.Format   := tmprfl.Format;
    Tmprfm.Datlen   := tmprfl.Datlen;
    {text widths}
    BMP.Width:= 1; BMP.Height:= 1;
    BMP.Canvas.Font.Name:= tmprfl.FontName;
    BMP.Canvas.Font.Style:= tmprfl.FontStyle;
    BMP.Canvas.Font.Color:= BytetoColor( tmprfl.FontColor );
    BMP.Canvas.Font.Size:= tmprfl.FontSize;
    if tmprfl.FontHt > 0 then
      BMP.Canvas.Font.Height:= - tmprfl.fontht;

    Tmprfm.Width    := BMP.Canvas.TextWidth(tmprfl.Name);
    Tmprfm.Height   := BMP.Canvas.TextHeight(tmprfl.Name);
    Dynarray.Add( tmprfm ); {add to main Dyn}
    end;
  finally
  Dyn.Free;
  BMP.Free;
  end;
  {notify OnChange}
  if Assigned(FOnChange) then FOnChange(Self);
end;

function TRFM.SaveRFL(filename: String): boolean;
var
  i:       smallint;
  Dyn:     TrsDynarray;
  tmprec:  RFormField;
  tmprfl:  RRFLField;
  tmpstr:  String;
begin
  result:= false;
  Dyn:= nil;
  try
  Dyn:= TrsDynarray.Create( Sizeof(RRFLField) );
  for i:= 0 to Dynarray.Count - 1 do begin
    tmprec:= RFormField(DynArray[i]^);
    if tmprec.DrawType = rfField then begin
      result:= true;
      FillChar(TmpRfl, sizeof(RRFLField), 0);
      tmpstr:= GetString(tmprec.FontIdx);
      {$IFDEF WIN32}
      if Length(tmpstr) > 20 then SetLength(tmpstr,20);
      {$ELSE}
      if Length(tmpstr) > 20 then tmpstr[0]:= #20;
      {$ENDIF}
      TmpRfl.FontName  := tmpstr;
      tmpstr:= GetString(tmprec.StrIdx);
      {$IFDEF WIN32}
      if Length(tmpstr) > 32 then SetLength(tmpstr,32);
      {$ELSE}
      if Length(tmpstr) > 32 then tmpstr[0]:= #32;
      {$ENDIF}
      TmpRfl.Name      := tmpstr;
      TmpRfl.Left      := tmprec.Left;
      TmpRfl.Top       := tmprec.Top;
      TmpRfl.Format    := tmprec.Format;
      TmpRfl.FontStyle := tmprec.FontStyle;
      TmpRfl.FontSize  := tmprec.FontSize;
      TmpRfl.FontColor := tmprec.Color;
      TmpRfl.FontHt    := tmprec.FontHt;
      TmpRfl.Datlen    := tmprec.Datlen;
      TmpRfl.Ver       := (VERSIONMAJOR*10) + VERSIONMINOR;
      Dyn.Add( TmpRfl );
      end;
    end;
  if Dyn.Count > 0 then Dyn.Savetofile(filename);
  finally
  Dyn.Free;
  end;
end;

{****************}
{ Object Theory  }
{****************}
function TRFM.AddObject( DType : TDrawType; X: smallint; Y: smallint ): smallint;
var
  TmpRec : RFormRec;
  SelRec : smallint;
  tmpstr : String;
begin
  if (Dynarray.Count < 2500) then Dynarray.Add( TmpRec )
  else messagedlg('Maximum 2500 objects reached! Combine and/or delete objects.', mtWarning,[mbok],0);

  {Select the new Object}
  SelRec:= Dynarray.Count - 1;
  Result:= SelRec;

  {put in common data}
  RFormRec( DynArray[SelRec]^ ).DrawType:= DType;
  RFormRec( DynArray[SelRec]^ ).Left    := X;
  RFormRec( DynArray[SelRec]^ ).Top     := Y;
  RFormRec( DynArray[SelRec]^ ).Width   := 50;
  RFormRec( DynArray[SelRec]^ ).Height  := 50;

  {put in special data}
  case DType of
    rfText  : begin
              RFormText( DynArray[SelRec]^ ).StrIdx := AddString('Text');
              RFormText( DynArray[SelRec]^ ).FontIdx := AddString(DefTextFont);
              RFormText( DynArray[SelRec]^ ).FontSize:= DefTextSize;
              RFormText( DynArray[SelRec]^ ).FontStyle:= [];
              RFormText( DynArray[SelRec]^ ).Color  := 0;
              RFormText( DynArray[SelRec]^ ).Vert   := 0;
              SizeText(SelRec);
              end;
    rfField : begin
              TmpStr:= NextName;
              RFormField( DynArray[SelRec]^ ).StrIdx := AddString(tmpstr);
              RFormField( DynArray[SelRec]^ ).FontIdx := AddString(DefFldFont);
              RFormField( DynArray[SelRec]^ ).FontSize:= DefFldSize;
              RFormField( DynArray[SelRec]^ ).FontStyle:= [];
              RFormField( DynArray[SelRec]^ ).Color  := 0;
              RFormField( DynArray[SelRec]^ ).Format := ffText;
              RFormField( DynArray[SelRec]^ ).DatLen := 40;
              RFormField( DynArray[SelRec]^ ).Vert   := 0;
              SizeField(SelRec);
              end;
    rfFrame : begin
              RFormFrame( DynArray[SelRec]^ ).Height := 100;
              RFormFrame( DynArray[SelRec]^ ).Width  := 150;
              RFormFrame( DynArray[SelRec]^ ).PStyle := psSolid;
              RFormFrame( DynArray[SelRec]^ ).FStyle := fsClear;
              RFormFrame( DynArray[SelRec]^ ).Color   := 15;
              RFormFrame( DynArray[SelRec]^ ).LColor  := 0;
              RFormFrame( DynArray[SelRec]^ ).Thick  := 1;
              end;
    rfLine  : begin
              RFormLine( DynArray[SelRec]^ ).Height := 20;
              RFormLine( DynArray[SelRec]^ ).Width  := 50;
              RFormLine( DynArray[SelRec]^ ).LType  := ltHorz;
              RFormLine( DynArray[SelRec]^ ).PStyle := psSolid;
              RFormLine( DynArray[SelRec]^ ).LType  := ltHorz;
              RFormLine( DynArray[SelRec]^ ).Color  := 0;
              RFormLine( DynArray[SelRec]^ ).Thick  := 1;
              end;
    rfCkBox : begin
              RFormCkBox( DynArray[SelRec]^ ).Height := 14;
              RFormCkBox( DynArray[SelRec]^ ).Width  := 14;
              RFormCkBox( DynArray[SelRec]^ ).Color  := 0;
              RFormCkBox( DynArray[SelRec]^ ).Thick  := 1;
              end;
    rfRoundRect : begin
              RFormRoundRect( DynArray[SelRec]^ ).Height := 100;
              RFormRoundRect( DynArray[SelRec]^ ).Width  := 150;
              RFormRoundRect( DynArray[SelRec]^ ).PStyle := psSolid;
              RFormRoundRect( DynArray[SelRec]^ ).FStyle := fsClear;
              RFormRoundRect( DynArray[SelRec]^ ).Color  := 15;
              RFormRoundRect( DynArray[SelRec]^ ).LColor := 0;
              RFormRoundRect( DynArray[SelRec]^ ).Thick  := 1;
              RFormRoundRect( DynArray[SelRec]^ ).Corner := 40;
              end;
    rfEllipse : begin
              RFormEllipse( DynArray[SelRec]^ ).Height := 40;
              RFormEllipse( DynArray[SelRec]^ ).Width  := 100;
              RFormEllipse( DynArray[SelRec]^ ).PStyle := psSolid;
              RFormEllipse( DynArray[SelRec]^ ).FStyle := fsClear;
              RFormEllipse( DynArray[SelRec]^ ).Color  := 15;
              RFormEllipse( DynArray[SelRec]^ ).LColor := 0;
              RFormEllipse( DynArray[SelRec]^ ).Thick  := 1;
              end;

     end; {case}
end;

procedure TRFM.DeleteObj(Index : smallint);
begin
  if (Index > Dynarray.Count-1) or (Index < 0) then Exit;

  if (RFormRec(DynArray[Index]^).DrawType = rfText) or
     (RFormRec(DynArray[Index]^).DrawType = rfField) then begin
    {Delete Strings}
    DelString( RFormRec(DynArray[Index]^).StrIdx1 );
    FixString( RFormRec(DynArray[Index]^).StrIdx1, Index );
    DelString( RFormRec(DynArray[Index]^).StrIdx2 );
    FixString( RFormRec(DynArray[Index]^).StrIdx2, Index );
    end;
  Dynarray.Delete(Index);
end;

procedure TRFM.DrawObj(MyCanvas : TCanvas; MyRect : TRect; Num : smallint);
var
  FormRec          : RFormRec;
  VScale, HScale   : real;
  FLeft, FTop,
  FRight, FBottom,
  FHCent, FVCent   : smallint;
  tmpb             : byte;
  cornerH,cornerV  : smallint;
  offx,offy        : smallint;
  LogFont          : TLogFont;
  NewHFont         : HFont;
  OldHFont         : HFont;
begin
  FormRec:= RFormRec(DynArray[Num]^);
  NewHFont:= 0;
  OldHFont:= 0;

  HScale:= (myRect.Right - myRect.Left) / Header.Width;
  VScale:= (myRect.Bottom - myRect.Top) / Header.Height;


  FLeft:=   trunc(FormRec.Left * HScale) + myRect.Left;
  FTop :=   trunc(FormRec.Top * VScale) + myRect.Top;
  FRight:=  trunc((FormRec.Width + FormRec.Left) * HScale) + myRect.Left;
  FBottom:= trunc((FormRec.Height + FormRec.Top) * VScale) + myRect.Top;
  FVCent:=  trunc((FRight - FLeft) / 2)+ FLeft;
  FHCent:=  trunc((FBottom - FTop) / 2)+ FTop;

  with MyCanvas do begin
    Brush.Color := clWhite;
    Brush.Style := bsClear;
    Pen.Color   := clBlack;
    Pen.Width   := 1;
    case FormRec.DrawType of
      rfText :  {Text}
        begin
        Font.Name:= GetString(RFormText( FormRec ).FontIdx);
        Font.Style:= RFormText( FormRec ).FontStyle;
        Font.Color:= BytetoColor( RFormText( FormRec ).Color );
        if RFormText( FormRec ).FontHt > 0 then
          Font.Height:= - round(RFormText( FormRec ).fontht * HScale)
        else Font.Height:= - round(RFormText( FormRec ).Height * HScale);
        offx:= 0; offy:= 0;

        {check orientation}
        NewHFont:= 0;
        if RFormText( FormRec ).Vert = 1 then begin
          GetObject (Font.Handle, SizeOf (TLogFont), @LogFont);
          LogFont.lfEscapement := 2700;
          NewHFont:= CreateFontIndirect (LogFont);
          OldHFont:= SelectObject (Handle, NewHFont);
          offx:= trunc(RFormText( FormRec ).Width * HScale);
          offy:= 0;
          end;
        if RFormText( FormRec ).Vert = 2 then begin
          GetObject (Font.Handle, SizeOf (TLogFont), @LogFont);
          LogFont.lfEscapement := -2700;
          NewHFont:= CreateFontIndirect (LogFont);
          OldHFont:= SelectObject (Handle, NewHFont);
          offx:= 0;
          offy:= trunc(RFormText( FormRec ).Height * VScale);
          end;

        TextOut( FLeft+offx, FTop+offy, GetString(RFormText( FormRec ).StrIdx) );

        if NewHFont <> 0 then begin
          NewHFont:= SelectObject(Handle, OldHFont);
          DeleteObject (NewHFont);
          end;
        end;

      rfField :  {Field Label}
        begin
        if Designing then begin
          Font.Name:= GetString(RFormField( FormRec ).FontIdx);
          Font.Style:= RFormField( FormRec ).FontStyle;
          Font.Color:= BytetoColor( RFormText( FormRec ).Color );
          if RFormField( FormRec ).FontHt > 0 then
          Font.Height:= - round(RFormField(FormRec ).fontht * HScale)
          else Font.Height:= - round(RFormField( FormRec ).Height * HScale);
          offx:= 0; offy:= 0;

          {check orientation}
          if RFormField( FormRec ).Vert = 1 then begin
            GetObject (Font.Handle, SizeOf (TLogFont), @LogFont);
            LogFont.lfEscapement := 2700;
            NewHFont:= CreateFontIndirect (LogFont);
            OldHFont:= SelectObject (Handle, NewHFont);
            offx:= trunc(RFormField( FormRec ).Width * HScale);
            offy:= 0;
            end;
          if RFormField( FormRec ).Vert = 2 then begin
            GetObject (Font.Handle, SizeOf (TLogFont), @LogFont);
            LogFont.lfEscapement := -2700;
            NewHFont:= CreateFontIndirect (LogFont);
            OldHFont:= SelectObject (Handle, NewHFont);
            offx:= 0;
            offy:= trunc(RFormField( FormRec ).Height * VScale);
            end;

          TextOut( FLeft+offx, FTop+offy, GetString(RFormField( FormRec ).StrIdx) );

          if NewHFont <> 0 then begin
            NewHFont:= SelectObject(Handle, OldHFont);
            DeleteObject (NewHFont);
            end;
          end;
        end;

      rfFrame:  {Frame}
        begin
        Brush.color      := BytetoColor( RFormFrame(FormRec).Color  );
        Pen.color        := BytetoColor( RFormFrame(FormRec).LColor );
        tmpb:= trunc(RFormFrame(FormRec).Thick * HScale);
        Pen.width := tmpb;
        case RFormFrame(FormRec).PStyle of
          psSolid:    Pen.style:=psSolid;
          psDash:     Pen.style:=psDash;
          psDot:      Pen.style:=psDot;
          psClear:    Pen.style:=psClear;
          end;
        case RFormFrame(FormRec).FStyle of
          fsClear:    Brush.style:= bsClear;
          fsSolid:    Brush.style:= bsSolid;
          fsDiagonal: Brush.style:= bsFDiagonal;
          fsHash:     Brush.style:= bsDiagCross;
          end;
        Rectangle(FLeft, FTop, FRight, FBottom);
        end;

      rfLine  :  {Line}
        begin
        case RFormLine(FormRec).PStyle of
          psSolid:    Pen.style:=psSolid;
          psDash:     Pen.style:=psDash;
          psDot:      Pen.style:=psDot;
          psClear:    Pen.style:=psClear;
          end;
        Pen.color := BytetoColor( RFormFrame(FormRec).Color );
        tmpb:= trunc(RFormLine(FormRec).Thick * HScale);
        Pen.width := tmpb;
        case RFormLine( FormRec ).LType of
          ltHorz: begin
            MoveTo(FLeft, FHCent);
            LineTo(FRight,FHCent);
            end;
          ltVert: begin
            MoveTo(FVCent, FTop);
            LineTo(FVCent, FBottom);
            end;
          ltDiagL: begin
            MoveTo(FLeft, FTop);
            LineTo(FRight, FBottom);
            end;
          ltDiagR: begin
            MoveTo(FRight, FTop);
            LineTo(FLeft, FBottom);
            end;
          end;
        end;

      rfCkBox :  {Checkbox}
        begin
        Pen.style   := psSolid;
        Pen.color   := BytetoColor( RFormCkBox(FormRec).Color );
        Pen.width   := trunc(RFormCkBox(FormRec).Thick * HScale);
        Rectangle(FLeft, FTop, FRight, FBottom);
        end;

      rfEllipse:  {Ellipse}
        begin
        Brush.color      := BytetoColor( RFormEllipse(FormRec).Color  );
        Pen.color        := BytetoColor( RFormEllipse(FormRec).LColor );
        tmpb:= trunc(RFormEllipse(FormRec).Thick * HScale);
        Pen.width := tmpb;
        case RFormEllipse(FormRec).PStyle of
          psSolid:    Pen.style:=psSolid;
          psDash:     Pen.style:=psDash;
          psDot:      Pen.style:=psDot;
          psClear:    Pen.style:=psClear;
          end;
        case RFormEllipse(FormRec).FStyle of
          fsClear:    Brush.style:= bsClear;
          fsSolid:    Brush.style:= bsSolid;
          fsDiagonal: Brush.style:= bsFDiagonal;
          fsHash:     Brush.style:= bsDiagCross;
          end;
        Ellipse(FLeft, FTop, FRight, FBottom);
        end;

      rfRoundRect:  {RoundRect}
        begin
        Brush.color      := BytetoColor( RFormRoundRect(FormRec).Color  );
        Pen.color        := BytetoColor( RFormRoundRect(FormRec).LColor );
        tmpb:= trunc(RFormRoundRect(FormRec).Thick * HScale);
        Pen.width := tmpb;
        cornerH:= trunc(RFormRoundRect(FormRec).Corner * HScale);
        cornerV:= trunc(RFormRoundRect(FormRec).Corner * VScale);
        case RFormRoundRect(FormRec).PStyle of
          psSolid:    Pen.style:=psSolid;
          psDash:     Pen.style:=psDash;
          psDot:      Pen.style:=psDot;
          psClear:    Pen.style:=psClear;
          end;
        case RFormRoundRect(FormRec).FStyle of
          fsClear:    Brush.style:= bsClear;
          fsSolid:    Brush.style:= bsSolid;
          fsDiagonal: Brush.style:= bsFDiagonal;
          fsHash:     Brush.style:= bsDiagCross;
          end;
        RoundRect(FLeft, FTop, FRight, FBottom, CornerH, CornerV);
        end;

      end; {case}
    end; {with canvas}
end;

{****************}
{  String Table  }
{****************}
function  TRFM.AddString( S: String ) : smallint;
begin
  StrTable.Add(S);
  result:= StrTable.Count - 1;
end;

procedure TRFM.DelString( Index : smallint );
begin
  if (Index < StrTable.Count) and (Index > -1) then StrTable.Delete(Index);
end;

procedure TRFM.UpdString( S: String; Index : smallint );
begin
  if (Index < StrTable.Count) and (Index > -1) then StrTable[Index]:= S;
end;

procedure TRFM.FixString( Index, RFMRec : smallint );
var i : smallint;
begin
  if (RFMRec > Dynarray.Count-1) or (RFMRec < 0) then Exit;
  for i:= RFMRec to Dynarray.Count - 1 do begin
    if (RFormRec(DynArray[i]^).DrawType = rfText) or
       (RFormRec(DynArray[i]^).DrawType = rfField) then begin
      if RFormRec(DynArray[i]^).StrIdx1 > Index then Dec(RFormRec(DynArray[i]^).StrIdx1);
      if RFormRec(DynArray[i]^).StrIdx2 > Index then Dec(RFormRec(DynArray[i]^).StrIdx2);
      end;
    end;
end;

function TRFM.GetString( Index : smallint ) : String;
begin
  if (Index < StrTable.Count) and (Index > -1) then Result:= StrTable[Index]
  else Result:= 'ERROR '+inttostr(Index);
end;

{****************}
{  Misc Support  }
{****************}
procedure TRFM.ClearBMP;
begin
  Bitmap.Free;
  Bitmap:= TBitmap.Create;
end;

function TRFM.MakeRect : TRect;
begin
  result:= Rect(0,0,Header.Width, Header.Height);
end;

procedure TRFM.SizeText( Index : smallint);
var B: TBitmap;
begin
  B:= Nil;
  try B:= TBitmap.Create;
  B.Canvas.Font.Name:= GetString( RFormText( DynArray[index]^ ).FontIdx );
  B.Canvas.Font.Size:= RFormText( DynArray[index]^ ).FontSize;
  B.Canvas.Font.Style:= RFormText( DynArray[index]^ ).FontStyle;

  if RFormText(DynArray[index]^).Vert > 0 then begin
    RFormText(DynArray[index]^).Width:=
      B.Canvas.TextHeight( GetString( RFormText( DynArray[index]^ ).StrIdx ) );
    RFormText(DynArray[index]^).Height:=
      B.Canvas.TextWidth( GetString( RFormText( DynArray[index]^ ).StrIdx ) )+4;
    end
  else begin
    RFormText(DynArray[index]^).Width:=
      B.Canvas.TextWidth( GetString( RFormText( DynArray[index]^ ).StrIdx ) )+4;
    RFormText(DynArray[index]^).Height:=
      B.Canvas.TextHeight( GetString( RFormText( DynArray[index]^ ).StrIdx ) );
    end;
  RFormText(DynArray[index]^).FontHt:= abs(B.Canvas.Font.Height);
  finally B.Free; end;
end;

procedure TRFM.SizeField( Index : smallint);
var B: TBitmap;
begin
  B:= nil;
  try B:= TBitmap.Create;
  B.Canvas.Font.Name:= GetString( RFormField( DynArray[index]^ ).FontIdx );
  B.Canvas.Font.Size:= RFormField( DynArray[index]^ ).FontSize;
  B.Canvas.Font.Style:= RFormField( DynArray[index]^ ).FontStyle;

  if RFormField(DynArray[index]^).Vert > 0 then begin
    RFormField(DynArray[index]^).Width:=
      B.Canvas.TextHeight( GetString( RFormField( DynArray[index]^ ).StrIdx ) );
    RFormField(DynArray[index]^).Height:=
      B.Canvas.TextWidth( GetString( RFormField( DynArray[index]^ ).StrIdx ) )+4;
    end
  else begin
    RFormField(DynArray[index]^).Width:=
      B.Canvas.TextWidth( GetString( RFormField( DynArray[index]^ ).StrIdx ) )+4;
    RFormField(DynArray[index]^).Height:=
      B.Canvas.TextHeight( GetString( RFormField( DynArray[index]^ ).StrIdx ) );
    end;
  RFormField(DynArray[index]^).FontHt:= abs(B.Canvas.Font.Height);
  finally B.Free; end;
end;

function  TRFM.ColorToByte (AColor : TColor) : Byte;
begin
  result:= 0; {default}
  if AColor =  clBlack   then result:= 0;
  if AColor =  clMaroon  then result:= 1;
  if AColor =  clGreen   then result:= 2;
  if AColor =  clOlive   then result:= 3;
  if AColor =  clNavy    then result:= 4;
  if AColor =  clPurple  then result:= 5;
  if AColor =  clTeal    then result:= 6;
  if AColor =  clGray    then result:= 7;
  if AColor =  clSilver  then result:= 8;
  if AColor =  clRed     then result:= 9;
  if AColor =  clLime    then result:= 10;
  if AColor =  clYellow  then result:= 11;
  if AColor =  clBlue    then result:= 12;
  if AColor =  clFuchsia then result:= 13;
  if AColor =  clAqua    then result:= 14;
  if AColor =  clWhite   then result:= 15;
  if AColor =  $00AAFFFF then result:= 16;
end;

function TRFM.ByteToColor (AByte : byte) : TColor;
begin
  case Abyte of
    0 : result:= clBlack;
    1 : result:= clMaroon;
    2 : result:= clGreen;
    3 : result:= clOlive;
    4 : result:= clNavy;
    5 : result:= clPurple;
    6 : result:= clTeal;
    7 : result:= clGray;
    8 : result:= clSilver;
    9 : result:= clRed;
   10 : result:= clLime;
   11 : result:= clYellow;
   12 : result:= clBlue;
   13 : result:= clFuchsia;
   14 : result:= clAqua;
   15 : result:= clWhite;
   16 : result:= $00AAFFFF; {pale yellow}
   else result:= clBlack; {default}
   end;
end;

function TRFM.VerifyName(N: String): String;
var flag: boolean;
    x,c: smallint;
begin
  c:= 1;
  repeat
    flag:= false;
    for x:= 0 to StrTable.count-1 do
      if comparetext(StrTable[x], N) = 0 then flag:= true;
    if flag then N:= 'Duplicate'+inttostr(c);
    inc(c);
  until flag = false;
  result:= N;
end;

function TRFM.NameCheck( Name : String; Index: smallint) : boolean;
var i : smallint;
begin
  result:= true;
  for i:= 0 to Strtable.Count - 1 do
    if (CompareText( Name, StrTable[i]) = 0)
      and (i <> RFormField(Dynarray[Index]^).StrIdx) then result:= false;
  if Pos(#32, Name) <> 0 then result:= false;
end;

function TRFM.NextName: String;
var
  i,x : smallint;
  flag : boolean;
begin
  x:= 0;
  repeat
   inc(x);
   flag:= false;
   result:= 'Field'+inttostr(x);
   for i:= 0 to StrTable.Count-1 do
     if CompareText( result, StrTable[i] ) = 0 then flag:= true;
   until flag = false;
end;

{*****************************}
{     TrsRFMImage Component     }
{*****************************}
constructor TrsRFMImage.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FFillList:= TStringList.Create;
  FFillList.Add('TLabel');
  FFillList.Add('TDBText');
  FLogoList:= TStringList.Create;
  FLogoList.Add('TImage');
  FLogoList.Add('TDBImage');
  FRFM:= TRFM.Create;
  Height:= 105;
  Width:= 105;
  Aspect:= true;
  AutoFill:= False;
  AutoLogo:= False;
  AutoFields:= false;
  AutoSize:= false;
  Center:= false;
  Stretch:= false;
  FPaperSize:= psLetter;
  FOrientation:= orPortrait;
  FNumCopies:= 1;
  FDuplexType:= dtNone;
  FDuplexRFM:= NIL;
  if (csDesigning in ComponentState) then RFM.Designing:= true
  else RFM.Designing:= false;
  RFM.OnChange:= RFMChanged;
end;

destructor TrsRFMImage.Destroy;
begin
  FRFM.Free;
  FFillList.Free;
  FLogoList.Free;
  inherited Destroy;
end;

function TrsRFMImage.CalcRect : TRect;
var  H,W : smallint;
begin
  {default}
  Result := ClientRect;
  {stretch}
  if Stretch then Result := ClientRect
  {center}
  else if Center then
    Result := Bounds((Width - RFM.Width) div 2, (Height - RFM.Height) div 2, RFM.Width, RFM.Height)
    else result := Rect(0, 0, RFM.Width, RFM.Height);
  {Aspect Ratio}
  if Aspect and Stretch then begin
    H:= Height;
    W:= trunc((H / RFM.Height) * RFM.Width);
    if not Center then result:= Rect(0, 0, W, H)
    else Result:= Bounds((Width - W) div 2, 0, W, H);
    end;
end;

function TrsRFMImage.RFMRect : TRect;
begin
  Result:= RFM.MakeRect;
end;

procedure TrsRFMImage.ModPrintOptions;
var
{$IFDEF Win32}
   FDevMode: PDeviceModeA;
{$ELSE}
   FDevMode: PDevMode;
{$ENDIF}
   FDevice, FDriver, FPort : Array[0..255] of Char;
   FHandle : THandle;
   Paper, Orient, Dup : smallint;
begin
  case FPaperSize of
    psLetter: Paper:= DMPAPER_LETTER;
    psLegal:  Paper:= DMPAPER_LEGAL;
    psA3:     Paper:= DMPAPER_A3;
    psA4:     Paper:= DMPAPER_A4;
    psA5:     Paper:= DMPAPER_A5;
    else      Paper:= DMPAPER_LETTER;
    end;

  if FOrientation = orLandscape then Orient:= DMORIENT_LANDSCAPE
  else Orient:= DMORIENT_PORTRAIT;

  Dup:= DMDUP_SIMPLEX;
  if (FDuplexType = dtHorizontal) and (FDuplexRFM <> nil) then Dup:= DMDUP_HORIZONTAL;
  if (FDuplexType = dtVertical) and (FDuplexRFM <> nil) then Dup:= DMDUP_VERTICAL;

  Printer.GetPrinter (FDevice, FDriver, FPort, FHandle );
  Printer.SetPrinter (FDevice, FDriver, FPort, FHandle );
  if FHandle = 0 then begin
    Printer.PrinterIndex := Printer.PrinterIndex;
    Printer.GetPrinter (FDevice, FDriver, FPort, FHandle );
    end;
  if FHandle = 0 then Exit;
{$IFDEF WIN32}
  FDevMode:= GlobalLock(FHandle);
{$ELSE}
  FDevMode:= Ptr(FHandle, 0);
{$ENDIF}
    with FDevMode^ do begin
      dmPaperSize:= Paper;
      dmOrientation:= Orient;
      dmCopies:= FNumCopies;
      dmDuplex:= Dup;
      dmFields:= dmFields or DM_PAPERSIZE or DM_ORIENTATION or DM_COPIES or DM_DUPLEX;
      end;
{$IFDEF WIN32}
   if FHandle <> 0 then GlobalUnLock(FHandle);
{$ENDIF}
end;

procedure TrsRFMImage.Print;
begin
  ModPrintOptions;
  PrintDirect;
end;

procedure TrsRFMImage.PrintDirect;
var R : TRect;
begin
  Printer.BeginDoc;
  Printer.Canvas.Font.PixelsPerINCH:= INCH;
  R:= Rect(0,0,Printer.PageWidth, Printer.PageHeight);
  RFM.DrawTo(R, Printer.Canvas, true);
  if AutoLogo then LogoTo(Printer.Canvas, R);
  if AutoFill then FillTo(Printer.Canvas, R);
  if AutoFields then FieldsTo(Printer.Canvas, R);

  if (FDuplexType <> dtNone) and (FDuplexRFM <> NIL) then begin {For Duplex Printing Only}
    Printer.NewPage;
    (FDuplexRFM as TrsRFMImage).RFM.DrawTo(R, Printer.Canvas, true);
    if (FDuplexRFM as TrsRFMImage).AutoLogo then (FDuplexRFM as TrsRFMImage).LogoTo(Printer.Canvas, R);
    if (FDuplexRFM as TrsRFMImage).AutoFill then (FDuplexRFM as TrsRFMImage).FillTo(Printer.Canvas, R);
    if (FDuplexRFM as TrsRFMImage).AutoFields then (FDuplexRFM as TrsRFMImage).FieldsTo(Printer.Canvas, R);
    end;
  Printer.EndDoc;
end;

procedure TrsRFMImage.PrintTo(ACanvas : TCanvas; MyRect : TRect);
begin
  RFM.DrawTo(MyRect, ACanvas, true);
  if AutoFill then FillTo(ACanvas, MyRect);
  if AutoLogo then LogoTo(ACanvas, MyRect);
  if AutoFields then FieldsTo(ACanvas, MyRect);
end;

procedure TrsRFMImage.LogoTo(ACanvas : TCanvas; MyRect : TRect);
var
  VScale, HScale   : real;
  PForm    : TForm;
  myimage  : TImage;
  {$IFDEF BDE}
  myDBimg  : TDBImage;
  {$ENDIF}
  i,j      : smallint;
  X,Y,W,H  : Longint;
  sx,sy    : smallint;
begin
  HScale:= (myRect.Right - myRect.Left) / RFM.Header.Width;
  VScale:= (myRect.Bottom - myRect.Top) / RFM.Header.Height;

  PForm:= (Owner as TForm);
  sx:= GetScrollPos(Parent.Handle, SB_HORZ) - Left;
  sy:= GetScrollPos(Parent.Handle, SB_VERT) - Top;
  for i:= 0 to PForm.ComponentCount - 1 do
    for j:= 0 to FLogoList.Count-1 do
      if CompareText(PForm.components[i].ClassName, FLogoList[j]) = 0 then begin
        myimage:= NIL;
        if (PForm.components[i] is TImage) then myimage:= (PForm.components[i] as TImage);
        {$IFDEF BDE}
        myDBimg:= NIL;
        if (PForm.components[i] is TDBImage) then myDBimg:= (PForm.components[i] as TDBImage);
        {$ENDIF}

        if myimage <> nil then begin
          W:= trunc((myImage.width)*HScale);
          H:= trunc((myImage.height)*VScale);
          X:= trunc((myImage.left+sx)*HScale);
          Y:= trunc((myImage.top+sy)*VScale);
          if (X > MyRect.Right) or (Y > MyRect.Bottom) then continue;
          if not myimage.visible then continue;
          if not MyImage.Picture.Bitmap.Empty then
            ACanvas.StretchDraw(Rect(X, Y, W+X, H+Y), MyImage.Picture.Bitmap)
          else if not MyImage.Picture.Metafile.Empty then
            ACanvas.StretchDraw(Rect(X, Y, W+X, H+Y), MyImage.Picture.Metafile)
          end;
        {$IFDEF BDE}
        if mydbimg <> nil then begin
          W:= trunc((mydbimg.width)*HScale);
          H:= trunc((mydbimg.height)*VScale);
          X:= trunc((mydbimg.left+sx)*HScale);
          Y:= trunc((mydbimg.top+sy)*VScale);
          if (X > MyRect.Right) or (Y > MyRect.Bottom) then continue;
          if not mydbimg.visible then continue;
          if not mydbimg.Picture.Bitmap.Empty then
            ACanvas.StretchDraw(Rect(X, Y, W+X, H+Y), mydbimg.Picture.Bitmap)
          else if not mydbimg.Picture.Metafile.Empty then
            ACanvas.StretchDraw(Rect(X, Y, W+X, H+Y), mydbimg.Picture.Metafile)
          end;
        {$ENDIF}
        end;
end;

procedure TrsRFMImage.FillTo(ACanvas : TCanvas; MyRect : TRect);
var
  VScale, HScale   : real;
  PForm    : TForm;
  mylab    : TLabel;
  {$IFDEF BDE}
  myDBlab  : TDBText;
  {$ENDIF}
  i,j      : smallint;
  X,Y      : Longint;
  sx,sy,tw : smallint;
  tmpstr   : string;
begin
  HScale:= (myRect.Right - myRect.Left) / RFM.Header.Width;
  VScale:= (myRect.Bottom - myRect.Top) / RFM.Header.Height;

  ACanvas.Brush.Style := bsClear;
  ACanvas.Pen.Color:= clBlack;
  ACanvas.Pen.Style:= psSolid;
  ACanvas.Pen.Width:= 1;

  PForm:= (Owner as TForm);
  sx:= GetScrollPos(Parent.Handle, SB_HORZ) - Left;
  sy:= GetScrollPos(Parent.Handle, SB_VERT) - Top;
  for i:= 0 to PForm.ComponentCount - 1 do
    for j:= 0 to FFillList.Count-1 do begin
      {$IFDEF FLY}
      if (PForm.components[i] is TFlyControl) then begin
        if (PForm.components[i] as TFlyControl).DisplayType = dtText then begin
          ACanvas.Font.Name:= (PForm.components[i] as TFlyControl).Font.Name;
          ACanvas.Font.Style:= (PForm.components[i] as TFlyControl).Font.Style;
          ACanvas.Font.Color:= (PForm.components[i] as TFlyControl).Font.Color;
          ACanvas.Font.Height:= trunc((PForm.components[i] as TFlyControl).Font.Height * Hscale);

          X:= trunc( (PForm.components[i] as TFlyControl).left+sx );
          Y:= trunc( (PForm.components[i] as TFlyControl).top+sy );
          ACanvas.TextOut( X, Y, (PForm.components[i] as TFlyControl).text);
          end;
        end;
      {$ENDIF}
      if CompareText(PForm.components[i].ClassName, FFillList[j]) = 0 then begin
        myLab:= NIL;
        if (PForm.components[i] is TLabel) then myLab:= (PForm.components[i] as TLabel);
        {$IFDEF BDE}
        myDBLab:= NIL;
        if (PForm.components[i] is TDBText) then myDBLab:= (PForm.components[i] as TDBText);
        {$ENDIF}
        if (myLab <> NIL) then begin
          X:= trunc((mylab.left+1+sx)*HScale);
          Y:= trunc((mylab.top+2+sy)*VScale);
          if (X > MyRect.Right) or (Y > MyRect.Bottom) or (not mylab.visible) or
            (mylab.caption = '') then continue;

          ACanvas.Font.Name:= mylab.Font.Name;
          ACanvas.Font.Style:= mylab.Font.Style;
          ACanvas.Font.Color:= mylab.Font.Color;
          ACanvas.Font.Height:= trunc(mylab.Font.Height * Hscale);

          if mylab.Alignment = taCenter then begin
            tw:= ACanvas.TextWidth(mylab.caption);
            X:= X + trunc(((MyLab.Width*Hscale) - tw) / 2);
            end;

          ACanvas.TextOut(X, Y, mylab.caption);
          end;
        {$IFDEF BDE}
        if (myDBLab <> NIL) then begin
          X:= trunc((myDBlab.left+1+sx)*HScale);
          Y:= trunc((myDBlab.top+2+sy)*VScale);
          if (X > MyRect.Right) or (Y > MyRect.Bottom) or (not myDBlab.visible) then continue;

          ACanvas.Font.Name:= myDBlab.Font.Name;
          ACanvas.Font.Style:= myDBlab.Font.Style;
          ACanvas.Font.Color:= myDBlab.Font.Color;
          ACanvas.Font.Height:= trunc(myDBlab.Font.Height * Hscale);

          if myDBLab.Field <> NIL then tmpstr:= myDBLab.Field.DisplayText
          else tmpstr:= Name;
          ACanvas.TextOut(X, Y, tmpstr);
          end;
        {$ENDIF}
        end;
      end;
end;

procedure TrsRFMImage.FieldsTo(ACanvas : TCanvas; MyRect : TRect);
var
  VScale, HScale   : real;
  i          : smallint;
  X,Y        : Longint;
  tmpstr     : string;
  tmprec     : RFormField;
  offx,offy  : smallint;
  LogFont    : TLogFont;
  NewHFont   : HFont;
  OldHFont   : HFont;

begin
  NewHFont:= 0;
  OldHFont:= 0;

  HScale:= (myRect.Right - myRect.Left) / RFM.Header.Width;
  VScale:= (myRect.Bottom - myRect.Top) / RFM.Header.Height;

  ACanvas.Brush.Style := bsClear;
  ACanvas.Pen.Color:= clBlack;
  ACanvas.Pen.Style:= psSolid;
  ACanvas.Pen.Width:= 1;

  for i:= 0 to FRFM.Dynarray.Count - 1 do begin
    if RFormRec(FRFM.Dynarray[i]^).DrawType <> rfField then continue;
    tmprec:= RFormField(FRFM.Dynarray[i]^);
    X:= trunc(tmprec.Left * HScale);
    Y:= trunc(tmprec.Top  * VScale);
    tmpstr:= '';
    if Assigned(FOnPrintField) then
      FOnPrintField( nil, FRFM.GetString(tmprec.StrIdx), TFieldFormat(tmprec.Format), tmpstr);

    ACanvas.Font.Name:= RFM.GetString(tmprec.FontIdx);
    ACanvas.Font.Style:= tmprec.FontStyle;
    ACanvas.Font.Color:= RFM.BytetoColor( tmprec.Color );
    if tmprec.FontHt > 0 then
      ACanvas.Font.Height:= - round(tmprec.fontht * HScale)
    else ACanvas.Font.Height:= - round(tmprec.Height * HScale);
    offx:= 0; offy:= 0;

    {check orientation}
    if tmprec.Vert = 1 then begin
      GetObject (ACanvas.Font.Handle, SizeOf (TLogFont), @LogFont);
      LogFont.lfEscapement := 2700;
      NewHFont:= CreateFontIndirect (LogFont);
      OldHFont:= SelectObject (ACanvas.Handle, NewHFont);
      offx:= trunc(tmprec.Width * HScale);
      offy:= 0;
      end;
    if tmprec.Vert = 2 then begin
      GetObject (ACanvas.Font.Handle, SizeOf (TLogFont), @LogFont);
      LogFont.lfEscapement := -2700;
      NewHFont:= CreateFontIndirect (LogFont);
      OldHFont:= SelectObject (ACanvas.Handle, NewHFont);
      offx:= 0;
      offy:= trunc(tmprec.Height * VScale);
      end;

    ACanvas.TextOut( X+offx, Y+offy, tmpstr );

    if NewHFont <> 0 then begin
      NewHFont:= SelectObject(ACanvas.Handle, OldHFont);
      DeleteObject (NewHFont);
      end;
    end;
end;

procedure TrsRFMImage.Paint;
begin
  {Autosize}
  if AutoSize and ((Align = alNone) or (Align = alLeft) or (Align = alTop)) then
    if (RFM.Width > 0) and (RFM.Height > 0) then begin
      Width:= RFM.Width;
      Height:= RFM.Height;
      end;
  {draw dashes at design time}
  if csDesigning in ComponentState then
    with inherited Canvas do begin
      Pen.Style := psDash;
      Brush.Style := bsClear;
      Rectangle(0, 0, Width, Height);
      end;
  {copy RFM bitmap to image canvas}
  if not RFM.Bitmap.Empty then
    inherited Canvas.Draw(CalcRect.Left, CalcRect.Top, RFM.Bitmap);
end;

procedure TrsRFMImage.Clear;
begin
  FRFM.Clear;
end;

procedure TrsRFMImage.UpdateRFM;
begin
  if FRFM.Empty then Exit;
  FRFM.ScaleDraw(CalcRect);
  Invalidate;
end;

procedure TrsRFMImage.SetAutoSize(AValue: Boolean);
begin
  if FAutoSize = AValue then Exit;
  FAutoSize := AValue;
  UpdateRFM;
end;

procedure TrsRFMImage.SetCenter(AValue: Boolean);
begin
  if FCenter = AValue then Exit;
  FCenter := AValue;
  UpdateRFM;
end;

procedure TrsRFMImage.SetStretch(AValue: Boolean);
begin
  if FStretch = AValue then Exit;
  FStretch := AValue;
  UpdateRFM;
end;

procedure TrsRFMImage.SetAspect(AValue: Boolean);
begin
  if FAspect = AValue then Exit;
  FAspect := AValue;
  UpdateRFM;
end;

procedure TrsRFMImage.SetRFM(AValue: TRFM);
begin
  FRFM.Assign(AValue);
  UpdateRFM;
  {paper size}
  FPaperSize:= RFM.Header.PaperSize;
  FOrientation:= RFM.Header.Orientation;
end;

procedure TrsRFMImage.SetFillList(AValue: TStrings);
begin
  FFillList.Assign(AValue);
end;

procedure TrsRFMImage.SetLogoList(AValue: TStrings);
begin
  FLogoList.Assign(AValue);
end;

procedure TrsRFMImage.Notification(AComponent: TComponent; Operation: TOperation);
begin
  inherited Notification(AComponent, Operation);
  if (Operation = opRemove) and (AComponent = FDuplexRFM) then FDuplexRFM := nil;
end;

procedure TrsRFMImage.RFMChanged(Sender : TObject);
begin
  if Assigned(FOnChange) then FOnChange(Self);
  UpdateRFM;
  {paper size}
  FPaperSize:= RFM.Header.PaperSize;
  FOrientation:= RFM.Header.Orientation;
end;

procedure TrsRFMImage.WMSize(var Message : TWMSize);
begin
  inherited;
  UpdateRFM;
end;

function  TrsRFMImage.GetPageWidth: word;
begin
  Result:= RFM.Header.Width;
end;

function  TrsRFMImage.GetPageHeight: word;
begin
  Result:= RFM.Header.Height;
end;

procedure TrsRFMImage.SetPaperSize(AValue: TPaperSize);
begin
  if FPapersize = AValue then Exit;
  FPapersize := AValue
end;

procedure TrsRFMImage.SetOrientation(AValue: TOrientation);
begin
  if FOrientation = AValue then Exit;
  FOrientation := AValue
end;

procedure TrsRFMImage.SetNumCopies(AValue: smallint);
begin
  if FNumCopies = AValue then Exit;
  FNumCopies := AValue
end;

{************************}
{  Double Click Editor   }
{************************}
procedure TRFMEditor.EditProperty(PropertyEditor: TPropertyEditor; var Continue, FreeEditor: Boolean);
var PropName: string;
begin
  PropName := PropertyEditor.GetName;
  if CompareText(PropName, 'RFM') <> 0 then Exit;
  PropertyEditor.Edit;
  Continue := False;
end;

{**************************}
{   TRFM Property Editor   }
{**************************}
function TRFMProperty.GetAttributes : TPropertyAttributes;
begin
  Result := [ paDialog ];
end;

function TRFMProperty.GetValue : string;
begin
  Result := '(RealForm Image)';
end;

procedure TRFMProperty.Edit;
var myRFM : TRFM;
begin
  myRFM := TRFM(GetOrdValue);
  try
    CreateForm;
    RFMImage1.RFM.Assign(myRFM);
    Dialog.ShowModal;
    if Dialog.Modalresult = mrOk then begin
      myRFM.Assign(RFMImage1.RFM);
      Designer.Modified;
      end;
  finally
    Dialog.Free;
  end;
end;

procedure TRFMProperty.ButtonClick(Sender: TObject);
begin
  if Sender = btnClear then RFMImage1.RFM.Clear;
  if Sender = btnSaveBMP then
    if SaveDialog1.Execute then begin
      RFMImage1.RFM.ClearBMP;
      RFMImage1.RFM.SaveAsBitmap(SaveDialog1.Filename);
      RFMImage1.UpdateRFM;
      end;
  if Sender = btnSaveWMF then
    if SaveDialog2.Execute then RFMImage1.RFM.SaveAsMetafile(SaveDialog2.Filename);
  if Sender = btnSaveRFM then
    if SaveDialog3.Execute then RFMImage1.RFM.SaveToFile(SaveDialog3.Filename);
  if Sender = FileListBox1 then
    if FileExists(FileListBox1.FileName) then
      RFMImage1.RFM.Loadfromfile(FileListBox1.FileName);
end;

procedure TRFMProperty.ButtonDblClick(Sender: TObject);
begin
  if Sender = FileListBox1 then
    if FileExists(FileListBox1.FileName) then btnOk.Click;
end;

procedure TRFMProperty.ckScaleMouseUp(Sender: TObject; Button: TMouseButton;
  Shift: TShiftState; X, Y: Integer);
begin
  if ckScale.checked then begin
    RFMImage1.Stretch:= true;
    RFMImage1.AutoSize:= false;
    RFMImage1.Width:= 175;
    RFMImage1.Height:= 147;
    RFMImage1.Center := True;
    end
  else begin
    RFMImage1.Stretch:= false;
    RFMImage1.AutoSize:= true;
    end;
  RFMImage1.UpdateRFM;
end;

procedure TRFMProperty.CreateForm;
begin
  Dialog := TForm.CreateNew(Application);
  Label1 := TLabel.create(Dialog);
  DirectoryListBox1 := TDirectoryListBox.create(Dialog);
  FileListBox1 := TFileListBox.create(Dialog);
  DriveComboBox1 := TDriveComboBox.create(Dialog);
  Edit1 := TEdit.create(Dialog);
  btnOK := TBitBtn.create(Dialog);
  btnCancel := TBitBtn.create(Dialog);
  btnClear := TBitBtn.create(Dialog);
  ScrollBox1 := TScrollBox.create(Dialog);
  RFMImage1 := TrsRFMImage.create(Dialog);
  ckScale := TCheckBox.create(Dialog);
  btnSaveBMP := TBitBtn.create(Dialog);
  btnSaveWMF := TBitBtn.create(Dialog);
  btnSaveRFM := TBitBtn.create(Dialog);
  SaveDialog1 := TSaveDialog.create(Dialog);
  SaveDialog2 := TSaveDialog.create(Dialog);
  SaveDialog3 := TSaveDialog.create(Dialog);
  with Dialog do begin
    Left := 314;
    Top := 133;
    BorderIcons := [biSystemMenu];
    BorderStyle := bsDialog;
    Caption := 'Open RFM';
    ClientHeight := 291;
    ClientWidth := 307;
    Font.Color := clBlack;
    Font.Height := -11;
    Font.Name := 'MS Sans Serif';
    Font.Style := [];
    PixelsPerInch := 96;
    Position := poScreenCenter;
    end;
  with Label1 do begin
    Parent:= Dialog;
    Name:= 'Label1';
    Left := 94;
    Top := 274;
    Width := 181;
    Height := 13;
    Caption := 'Copyright 1997 RealSoft Development';
    Font.Color := clNavy;
    Font.Height := -11;
    Font.Name := 'MS Sans Serif';
    Font.Style := [];
    ParentFont := False;
    end;
  with Edit1 do begin
    Parent:= Dialog;
    Name:= 'Edit1';
    Left := 160;
    Top := 8;
    Width := 145;
    Height := 20;
    TabOrder := 3;
    Text := '*.RFM';
    end;
  with FileListBox1 do begin
    Parent:= Dialog;
    Name:= 'FileListBox1';
    Left := 160;
    Top := 32;
    Width := 145;
    Height := 81;
    FileEdit := Edit1;
    ItemHeight := 13;
    Mask := '*.RFM';
    TabOrder := 1;
    OnClick := ButtonClick;
    OnDblClick := ButtonDblClick;
    end;
  with DirectoryListBox1 do begin
    Parent:= Dialog;
    Name:= 'DirectoryListBox1';
    Left := 8;
    Top := 32;
    Width := 145;
    Height := 81;
    FileList := FileListBox1;
    ItemHeight := 16;
    TabOrder := 0;
    end;
  with DriveComboBox1 do begin
    Parent:= Dialog;
    Name:= 'DriveComboBox1';
    Left := 8;
    Top := 8;
    Width := 145;
    Height := 19;
    DirList := DirectoryListBox1;
    TabOrder := 2;
    end;
  with btnOK do begin
    Parent:= Dialog;
    Name:= 'btnOK';
    Caption:= 'OK';
    Left := 201;
    Top := 220;
    Width := 95;
    Height := 25;
    TabOrder := 5;
    Kind := bkCustom;
    ModalResult:= mrOK;
    end;
  with btnCancel do begin
    Parent:= Dialog;
    Name:= 'btnCancel';
    Caption:= 'Cancel';
    Left := 201;
    Top := 244;
    Width := 95;
    Height := 25;
    TabOrder := 4;
    Kind := bkCustom;
    ModalResult:= mrCancel;
    end;
  with btnClear do begin
    Parent:= Dialog;
    Name:= 'btnClear';
    Left := 201;
    Top := 119;
    Width := 95;
    Height := 25;
    Caption := 'Clear Image';
    TabOrder := 6;
    OnClick := ButtonClick;
    end;
  with ScrollBox1 do begin
    Parent:= Dialog;
    Name:= 'ScrollBox1';
    Left := 8;
    Top := 120;
    Width := 177;
    Height := 149;
    TabOrder := 7;
    BorderStyle:= bsNone;
    Ctl3D:= false;
    end;
  with RFMImage1 do begin
    Parent:= ScrollBox1;
    Name:= 'RFMImage1';
    Left := 0;
    Top := 0;
    Width := 175;
    Height := 147;
    Center := True;
    Stretch := True;
    end;
  with ckScale do begin
    Parent:= Dialog;
    Name:= 'ckScale';
    Left := 8;
    Top := 273;
    Width := 76;
    Height := 17;
    Caption := 'Auto Scale';
    State := cbChecked;
    TabOrder := 8;
    OnMouseUp := ckScaleMouseUp;
    end;
  with btnSaveBMP do begin
    Parent:= Dialog;
    Name:= 'btnSaveBMP';
    Left := 201;
    Top := 143;
    Width := 95;
    Height := 25;
    Caption := 'Save as Bitmap';
    TabOrder := 9;
    OnClick := ButtonClick;
    end;
  with btnSaveWMF do begin
    Parent:= Dialog;
    Name:= 'btnSaveWMF';
    Left := 201;
    Top := 167;
    Width := 95;
    Height := 25;
    Caption := 'Save as Metafile';
    TabOrder := 10;
    OnClick := ButtonClick;
    end;
  with btnSaveRFM do begin
    Parent:= Dialog;
    Name:= 'btnSaveRFM';
    Left := 201;
    Top := 191;
    Width := 95;
    Height := 25;
    Caption := 'Save as RFM';
    TabOrder := 11;
    OnClick := ButtonClick;
    end;
  with SaveDialog1 do begin
    Name:= 'SaveDialog1';
    DefaultExt := 'BMP';
    Filter := 'Bitmap Files|*.BMP';
    end;
  with SaveDialog2 do begin
    Name:= 'SaveDialog2';
    Filter := 'Metafiles|*.WMF';
    end;
  with SaveDialog3 do begin
    Name:= 'SaveDialog3';
    Filter := 'Realform Images|*.RFM';
    end;
end;


end.
