;****************************************************************************
; MODEX32W.ASM                                      Copyright 1996 by MUTTLEY
;----------------------------------------------------------------------------
; Mode X Graphics Routines for 32-Bit Protected mode. 
; Assembled using TASM v4.0 and designed for use with the DOS4G/W 
; DOS Extender and Watcom C++ v10.5
;****************************************************************************

.386p                                   ; Support Pentium and Protected mode            
.model flat                             ; Flat memory model

ASSUME CS:_TEXT                         ; Associate segment regs with the       
ASSUME DS:_DATA                         ; code and data segment names

; Macro to out a 16 bit value to an I/O port
OUT_16  MACRO register, value
	IFDIFI <register>, <dx>         ; If dx not setup
	mov dx, register                ; then select register
	ENDIF
	IFDIFI <value>, <ax>            ; If ax not setup
	mov ax, value                   ; then get data value
	ENDIF
	out dx, ax                      ; Output the value to the register
	ENDM

; Macro to out an 8 bit value to an I/O port
OUT_8   MACRO register, value
	IFDIFI <register>, <dx>         ; If dx not setup
	mov dx, register                ; then select register
	ENDIF
	IFDIFI <value>, <al>            ; If al not setup
	mov al, value                   ; then get data value
	ENDIF
	out dx, al                      ; Output the value to the register
	ENDM

SC_INDEX        EQU     03c4h           ; Sequencer Controller Index
SC_DATA         EQU     03c5h           ; VGA Sequencer data port
CRTC_INDEX      EQU     03d4h           ; CRT Controller Index
MISC_OUTPUT     EQU     03c2h           ; Miscellaneous Output register
MAP_MASK        EQU       02h           ; Index in SC of Map Mask register
GC_INDEX        EQU     03ceh           ; Graphics Controller Index
READ_MAP        EQU       04h           ; Index in GC of Read Map register

ATTRIB_CTRL     EQU 	03c0h           ; Vga attribute controller
PIXEL_PAN_REG   EQU      033h           ; Attrib Index: Pixel Pan Register

DAC_WRITE_ADDR  EQU     03c8h           ; Vga DAC write address register
DAC_READ_ADDR   EQU     03c7h           ; Vga DAC read address register
PEL_DATA_REG    EQU     03c9h           ; Vga DAC/PEL data register r/w

INP_STAT_1      EQU     03dah           ; Input Status 1 register
START_ADDR_HIGH EQU       0ch           ; Bitmap start address high byte
START_ADDR_LOW  EQU       0dh           ; Bitmap start address low byte


_DATA SEGMENT PUBLIC DWORD 'DATA' USE32 ; Define our data segment


DISPLAY_MEM     DD      0a0000h         ; Start of display memory (Flat mem)

SCREEN_WIDTH    DD      50h             ; Normal width of the screen in bytes
SCREEN_HEIGHT	DW	240		; Normal height of screen in pixels
MAX_WIDTH	DW      640		; Max width of virtual screen 
MAX_HEIGHT	DW	240		; Max height of a virtual screen
VPAGE_SIZE	DW	  0 		; Will hold virtual page size in bytes

CURR_X_OFFSET   DW 	  0             ; Current display X offset
CURR_Y_OFFSET   DW 	  0             ; Current display Y offset
MAX_X_OFFSET	DW	  0		; Max X display offset
MAX_Y_OFFSET	DW	  0		; Max Y display offset
CURR_START_OFF  DD 	  0             ; Current start offset in memory

LeftClipMask    DB      0fh, 0eh, 0ch, 08h ; Plane mask values for clipping
RightClipMask   DB      01h, 03h, 07h, 0fh ; the left & right edges of a line

; Index/data pairs for CRT Controller registers 
; that differ between mode 13h and mode X (320 x 240)
CRTParms LABEL  WORD
		DW      00d06h          ; Vertical total
		DW      03e07h          ; Overflow (bit 8 of vertical counts)
		DW      04109h          ; Cell height (2 to double-scan)
		DW      0ea10h          ; V sync start
		DW      0ac11h          ; V sync end and protect cr0-cr7
		DW      0df12h          ; Vertical displayed
		DW      00014h          ; Turn off dword mode
		DW      0e715h          ; V blank start
		DW      00616h          ; V blank end
		DW      0e317h          ; Turn on byte mode
CRT_PARM_LENGTH EQU     (($-CRTParms)/2)


_DATA ENDS                              ; End of data segment


_TEXT SEGMENT PUBLIC DWORD 'CODE' USE32 ; Define our code segment

PUBLIC  _SetModeX, \			; Declare functions to the linker
	_ResetModeX, _WrPixelX,\
	_RdPixelX, _LineX, _RectX,\
	_SetDACReg, _GetDACReg,\
	_ShowPage, _SyncDisplay
	

;----------------------------------------------------------------------------
; SetModex() 
; Routine to set display to 320x240 Mode X
;----------------------------------------------------------------------------
_SetModeX       PROC NEAR
	push    ebp                     ; Preserve caller's stack frame
	mov     ebp, esp
	push    esi edi ebx             ; Preserve C register variables
		
	mov     ax,13h                  ; Let BIOS set standard 256-Color
	int     10h                     ; mode 13h (320x200 pixels)

	mov     dx, SC_INDEX            ; Get Sequencer controller index
	mov     ax, 0604h               ; and...
	out     dx, ax                  ; Disable chain-4 mode
	mov     ax, 0100h
	out     dx, ax                  ; Synchronous reset while switching
					; clocks using Misc. Output register
	mov     dx, MISC_OUTPUT           
	mov     al, 0e3h                ; Reset dot clock to 28 MHz     
	out     dx, al                  ; and 60 Hz scanning rate

	mov     dx, SC_INDEX            ; Get Sequencer controller index
	mov     ax, 0300h               ; and undo the synchronous reset
	out     dx, ax                  ; ie restart the Sequencer

	mov     dx, CRTC_INDEX          ; Reprogram the CRT Controller...
	mov     al, 11h                 ; VSync End reg contains register 
	out     dx, al                  ; write protect bit
	inc     dx                      ; Get CRT Controller Data register...
	in      al, dx                  ; Get current VSync End setting
	and     al, 7fh                 ; Remove the write protect on 
	out     dx, al                  ; various CRTC registers
	dec     dx                      ; Get CRT Controller Index again...
	cld                             ; Clear direction flag ready for...
	mov     esi, offset CRTParms    ; Point to CRT parameter table
	mov     ecx, CRT_PARM_LENGTH    ; Get # of parameter table entries
SetCRTParmsLoop:
	lodsw                           ; Get the next CRT Index/Data pair
	out     dx, ax                  ; Set the next CRT Index/Data pair
	loop    SetCRTParmsLoop

	mov     dx, SC_INDEX            ; Get Sequencer controller index
	mov     ax, 0f02h               ; and...
	out     dx, ax                  ; enable writes to all four planes
	mov     edi, DISPLAY_MEM        ; Point es:edi to display memory
	sub     ax, ax                  ; Set ax to 0...
	mov     ecx, 8000h              ; Number of words in display memory
	rep     stosw                   ; Clear all display memory to 0

	pop     ebx edi esi             ; Restore C register variables
	pop     ebp                     ; Restore caller's stack frame
	ret
_SetModeX       ENDP

;----------------------------------------------------------------------------
; ResetModeX() 
; Routine to set display back to text mode
;----------------------------------------------------------------------------
_ResetModeX     PROC
	push    ebp

	mov     ax, 3                   ; Use BIOS to reset screen
	int     10h                     ; to text mode

	pop     ebp
	ret
_ResetModeX     ENDP

;----------------------------------------------------------------------------
; WrPixelX(int X, int Y, int Color, int PageBase) 
; Draws a pixel on the display
;----------------------------------------------------------------------------
_WrPixelX   PROC NEAR

	ARG     X:DWORD, Y:DWORD, Color:DWORD, PageBase:DWORD 
	
	push    ebp                     ; Preserve caller's stack frame
	mov     ebp, esp                ; Point to local stack frame

	mov     eax, SCREEN_WIDTH
	mul     [Y]                     ; Offset of pixel's scan line in page
	mov     ebx, [X]                ; and...
	shr     ebx, 2                  ; X/4 = byte offset of pixel in line
	add     ebx, eax                ; = offset of pixel in page 
	add     ebx, [PageBase]         ; = offset of pixel in display memory
	add     ebx, DISPLAY_MEM        ; = offset of pixel in flat memory

	mov     cl, byte ptr [X]        ; Pixel's X position MOD 3...
	and     cl, 011b                ; = the pixel's plane number (0-3)
	mov     ax, 0100h + MAP_MASK    ; al = Index in SC of Map Mask reg
	shl     ah, cl                  ; Set pixel's required plane # to 1
	mov     dx, SC_INDEX            ; Set the Map Mask to only enable the
	out     dx, ax                  ; required plane for this pixel

	mov     al, byte ptr [Color]    ; Get the pixel color
	mov     [ebx], al               ; Draw the pixel!

	pop     ebp                     ; Restore caller's stack frame
	ret        
_WrPixelX       ENDP

;----------------------------------------------------------------------------
; unsigned int RdPixelX(int X, int Y, int PageBase);
; Read a pixel from the display
;---------------------------------------------------------------------------- 
_RdPixelX   PROC NEAR

	ARG     X:DWORD, Y:DWORD, PageBase:DWORD

	push    ebp                     ; Preserve caller's stack frame
	mov     ebp, esp                ; Point to local stack frame

	mov     eax, SCREEN_WIDTH
	mul     [Y]                     ; Offset of pixel's scan line in page
	mov     ebx, [X]                ; and...
	shr     ebx, 2                  ; X/4 = offset of pixel in scan line
	add     ebx, eax                ; = Offset of pixel in page
	add     ebx, [PageBase]         ; = Offset of pixel in display memory
	add     ebx, DISPLAY_MEM        ; = Offset of pixel in flat memory

	mov     ah, byte ptr [X]        ; Pixel's X position MOD 3...
	and     ah, 011b                ; = Pixel's plane number (0-3)
	mov     al, READ_MAP            ; al = Index in GC of Read Map reg
	mov     dx, GC_INDEX            ; Set Read Map to only read the plane
	out     dx, ax                  ; for this pixel

	mov     al, [ebx]               ; Read the pixel's color and
	sub     ah, ah                  ; convert it to an unsigned int

	pop     ebp                     ; Restore caller's stack frame
	ret
_RdPixelX       ENDP

;----------------------------------------------------------------------------
; LineX(int X1, int Y1, int X2, int Y2, int LineColor, int PageBase)
; Draws a line on the display 
;----------------------------------------------------------------------------
_LineX  PROC NEAR

	ARG     X1:WORD, Y1:WORD, X2:WORD, Y2:WORD,\
		LineColor:BYTE, PageBase:DWORD         

	push    ebp                     ; Preserve caller's stack frame
	mov     ebp, esp                ; Set up local stack frame
	push    esi edi                 ; Save important registers

	cld                             ; Clear direction flag to forward
	OUT_8   SC_INDEX, MAP_MASK      ; Point to reg in SC for plane select
	mov     ch, byte ptr [LineColor] ; Save line color in ch

; First, check what type of line this is...

	movzx   esi, [X1]               ; si = X1   
	movzx   edi, [X2]               ; di = X2
	cmp     si, di                  ; Is X1 < X2 ?
	je      @VertLine               ; If X1 = X2 then draw vertical line
	jl      SHORT @NoSwap1          ; If X1 < X2 then don't swap
	xchg    si, di                  ; If X1 > X2 then we must swap them

@NoSwap1:                               ; si = X1, di = X2 and X1 < X2
	mov     ax, [Y1]                ; Get Y1 into ax
	cmp     ax, [Y2]                ; Is Y1 = Y2 ?
	je      SHORT @HorizLine        ; Yes, so draw a horizontal line...
	jmp     @Breznham               ; No, so draw a slanted line... 

; This section draws a horizontal line where si = X1, di = X2, ax = Y1 & Y2..

@HorizLine:
	mul     SCREEN_WIDTH            ; Find offset of line within page
	mov     dx, ax                  ; Save it in dx for later use

	mov     ax, si                  ; Get left edge (X1) and save in ax
	and     si, 03h                 ; Mask out bits 0-1 of X1 to find 
	mov     bl, LeftClipMask[esi]   ; required index for left edge mask
	mov     cx, di                  ; Get right edge (X2) and save in cx
	and     di, 03h                 ; Mask out bits 0-1 of X2 to find
	mov     bh, RightClipMask[edi]  ; required index for right edge mask byte

	shr     ax, 2                   ; Get byte pos of X1 by finding X1/4
	shr     cx, 2                   ; Get byte pos of X2 by finding X2/4

	movzx   eax, ax                 ; Zero out high words before adding
	movzx   edx, dx
	movzx   ecx, cx

	mov     edi, edx                ; Get offset to start of line in page
	add     edi, eax                ; = Offset to pixel X1 in page
	add     edi, [PageBase]         ; = Offset of pixel in display memory
	add     edi, DISPLAY_MEM        ; = Offset of pixel in flat memory

	sub     cx, ax                  ; cx = # of 4-pixel bands(-1) to draw
	jnz     SHORT @LongLine         ; Jump if longer than one band
	and     bl, bh                  ; Otherwise, merge clip masks together

@LongLine:
	OUT_8   SC_DATA, bl             ; Select planes for left clip mask
	mov     al, [LineColor]         ; Get line color
	mov     bl, al                  ; and copy into bl
	stosb                           ; Draw the left (1-4) pixels of line
	jcxz    SHORT @LineExit1        ; Finished if only one line segment
	dec     cx                      ; Reduce number of middle segments
	jz      SHORT @RightSeg         ; If no middle segments....

; Draw the middle segments of the line...

	OUT_8   dx, 0fh                 ; Enable write to all planes
	mov     al, bl                  ; Get line color from bl
	rep     stosb                   ; Draw the middle (4 pixel) segments

@RightSeg:                              
	OUT_8   dx, bh                  ; Select planes for right clip mask
	mov     al, bl                  ; Get line color
	stosb                           ; Draw the right (1-4) pixels of line

@LineExit1:
	jmp     @LineFinished           ; We are finished...


; This section draws a vertical line where ch = line color, si & di = X1

@VertLine:
	mov     ax, [Y1]                ; ax = Y1
	mov     si, [Y2]                ; si = Y2
	cmp     ax, si                  ; Is Y1 < Y2 ?
	jle     SHORT @NoSwap2          ; If so then don't swap them
	xchg    ax, si                  ; Otherwise swap so that Y1 < Y2

@NoSwap2:
	sub     si, ax                  ; si = line height 
	inc     si                      ; si = line height + 1

; At this stage, ax = Y1, di = X1...

	mul     SCREEN_WIDTH            ; ax = offset of Y1 in display page
	mov     dx, di                  ; Copy X1 into dx
	shr     di, 2                   ; = Offset in scan line of X1 in bytes 
	add     ax, di                  ; = Offset of pixel in display page
	movzx   eax, ax                 ; Zero out high word before adding
	mov     edi, [PageBase]         ; Point to active page in display mem
	add     edi, eax                ; = Offset of pixel in display mem
	add     edi, DISPLAY_MEM        ; = Offset of pixel in flat memory

	mov     cl, dl                  ; cl = low byte of X1
	and     cl, 03h                 ; X1 mod 4 = pixel's plane number 
	mov     ax, 0100h + MAP_MASK    ; al = Index in SC of Map Mask reg 
	shl     ah, cl                  ; Set required plane # to 1
	OUT_16  SC_INDEX, ax            ; Enable the plane for this pixel

	mov     al, ch                  ; Get the saved color
	mov     ebx, SCREEN_WIDTH       ; Offset to add to draw next pixel


@VertLoop:
	mov     [edi], al               ; Draw single pixel
	add     edi, ebx                ; Add offset to point to next pixel
	dec     si                      ; Decrement height count
	jz      SHORT @LineExit2        ; Exit if height count is now 0 
	jmp     SHORT @VertLoop         ; Else loop back for next pixel

@LineExit2:
	jmp     @LineFinished           ; We are finished...


; This rest of this procedure draws a slanted line...

@Breznham:
	mov     edi, [PageBase]         ; Point to active page in display mem

	mov     ax, [Y1]                ; Get Y1 
	mov     bx, [Y2]                ; Get Y2 
	mov     cx, [X1]                ; Get X1

	cmp     bx, ax                  ; Compare Y1 with Y2
	jnc     SHORT @DeltaYOk         ; If Y2 >= Y1 then goto...
	xchg    bx, ax                  ; If Y2 < Y1 then swap them...
	mov     cx, [X2]                ; and also get X2 instead

@DeltaYOk:
	mul     SCREEN_WIDTH            ; Get offset of Y1 within page
	movzx   eax, ax                 ; and store it

	add     edi, eax                ; = Offset of y1 within display mem
	mov     ax, cx                  ; Copy X1 into ax and divide
	shr     ax, 2                   ; by 4 to find X1 pos in bytes
	add     edi, eax                ; = Offset of 1st pixel in display mem
	add     edi, DISPLAY_MEM        ; = Offset of 1st pixel in flat mem

	mov     al, 11h                 ; Starting mask
	and     cl, 03h                 ; Get plane number
	shl     al, cl                  ; and shift it into place
	mov     ah, [LineColor]         ; Get line color in high byte
	push    ax                      ; Save the mask and line color...

	mov     ah, al                  ; Copy plane number into ah
	mov     al, MAP_MASK            ; Select plane register
	OUT_16  SC_INDEX, ax            ; Select initial plane

	mov     ax, [X1]                ; Get X1 
	mov     bx, [Y1]                ; Get Y1 
	mov     cx, [X2]                ; Get X2 
	mov     dx, [Y2]                ; Get Y2 

	mov     ebp, SCREEN_WIDTH       ; Use bp for line width to
					; to avoid extra memory access
	
	sub     dx, bx                  ; Calculate delta_y
	jnc     SHORT @DeltaYOk2        ; and jump if Y2 >= Y1
	add     bx, dx                  ; Else copy Y2 into Y1
	neg     dx                      ; and find abs(delta_y)
	xchg    ax, cx                  ; and exchange X1 and X2 too

@DeltaYOk2:
	mov     bx, 08000h              ; Seed for fraction accumulator

	sub     cx, ax                  ; Calculate delta_x
	jc      SHORT @DrawLeft         ; If negative, go left...
	jmp     @DrawRight              ; Else line slopes right...

@DrawLeft:
	neg     cx                      ; Find abs(delta_x)
	cmp     cx, dx                  ; Is delta_x < delta_y?
	jb      SHORT @SteepLeft        ; If yes go draw steep line to left


; If we reach here then draw a shallow line to the left ...

@ShallowLeft:
	sub     ax,ax                   ; Zero low word of delta_y * 10000h
	sub     ax, dx                  ; dx:ax <- dx * 0ffffh
	sbb     dx, 0                   ; Include carry
	div     cx                      ; Divide by delta_x

	mov     si, bx                  ; si = accumulator
	mov     bx, ax                  ; bx = add fraction
	pop     ax                      ; Get color and bit plane mask
	mov     dx, SC_DATA             ; Get Sequence Controller data reg
	inc     cx                      ; Inc delta_x so we can unroll loop

; Use delta_x as loop to draw pixels, move left each iteration & maybe down..

@SLLLoop:
	mov     [edi], ah               ; Set first pixel, plane data set up
	dec     cx                      ; Decrement delta_x 
	jz      SHORT @SLLExit          ; and exit if done

	add     si, bx                  ; Add numerator to accumulator
	jnc     SHORT @SLLL2nc          ; Move down on carry
	add     edi, ebp                ; Move down one line...

@SLLL2nc:
	dec     edi                     ; Move left one pixel (address)
	ror     al, 1                   ; Move left one plane, back on 0 1 2
	cmp     al, 87h                 ; Wrap?, if al < 88 then carry set
	adc     edi, 0                  ; Adjust address: di = di + carry
	out     dx, al                  ; Set up new bit plane mask

	mov     [edi], ah               ; Draw the pixel
	dec     cx                      ; Decrement delta_x
	jz      SHORT @SLLExit          ; and exit if done

	add     si, bx                  ; Add numerator to accumulator,
	jnc     SHORT @SLLL3nc          ; Move down on carry
	add     edi, ebp                ; Move down one line...

@SLLL3nc:                               
	dec     edi                     ; Now move left one pixel (address)
	ror     al, 1                   ; Move left one plane, back on 0 1 2
	cmp     al, 87h                 ; Wrap?, if al < 88 then carry set
	adc     edi, 0                  ; Adjust address: di = di + carry
	out     dx, al                  ; Set up new bit plane mask
	jmp     SHORT @SLLLoop          ; Loop until done

@SLLExit:
	jmp     @LineFinished           ; Finito...


; If we reach here then draw a steep line to the left 

@SteepLeft:
	sub     ax, ax                  ; Zero low word of delta_y * 10000h
	xchg    dx, cx                  ; Switch delta_y with delta_x
	div     cx                      ; Divide by delta_y

	mov     si, bx                  ; si = accumulator
	mov     bx, ax                  ; bx = add fraction
	pop     ax                      ; Get color and bit mask
	mov     dx, SC_DATA             ; Get Sequence Controller data reg
	inc     cx                      ; Inc delta_y so we can unroll loop

; Use delta_x as loop to draw pixels, move down each iteration & maybe left

@STLLLoop:
	mov     [edi], ah               ; Set first pixel
	dec     cx                      ; Decrement delta_y
	jz      SHORT @STLLExit         ; and exit if done

	add     si, bx                  ; Add numerator to accumulator
	jnc     SHORT @STLLLnc2         ; No carry so just move down!

	dec     edi                     ; Move left one pixel (address)
	ror     al, 1                   ; Move left one plane, back on 0 1 2
	cmp     al, 87h                 ; Wrap?, if al < 88 then carry set
	adc     edi, 0                  ; Adjust address: di = di + carry
	out     dx, al                  ; Set up new bit plane mask

@STLLLnc2:
	add     edi, ebp                ; Advance to next line.
	mov     [edi], ah               ; Draw the pixel
	dec     cx                      ; Decrement delta_y
	jz      SHORT @STLLExit         ; and exit if done

	add     si, bx                  ; Add numerator to accumulator
	jnc     SHORT @STLLLnc3         ; No carry so just move down!

	dec     edi                     ; Move left one pixel (address)
	ror     al, 1                   ; Move left one plane, back on 0 1 2
	cmp     al, 87h                 ; Wrap?, if al < 88 then carry set
	adc     edi, 0                  ; Adjust address: di = di + carry
	out     dx, al                  ; Set up new bit plane mask

@STLLLnc3:
	add     edi, ebp                ; Advance to next line.
	jmp     SHORT @STLLLoop         ; Loop until done

@STLLExit:
	jmp     @LineFinished           ; Finito...


; If we reach here then we must draw a line that goes to the right...

@DrawRight:
	cmp     cx, dx                  ; Is delta_x < delta_y?
	jb      SHORT @SteepRight       ; Yes, so go do steep line


; If we reach here then we must draw a shallow line to the right 

@ShallowRight:
	sub     ax, ax                  ; Zero low word of delta_y * 10000h
	sub     ax, dx                  ; dx:ax <- dx * 0ffffh
	sbb     dx, 0                   ; Include carry
	div     cx                      ; Divide by delta_x

	mov     si, bx                  ; si = accumulator
	mov     bx, ax                  ; bx = add fraction
	pop     ax                      ; Get color and bit mask
	mov     dx, SC_DATA             ; Get Sequence Controller data reg
	inc     cx                      ; Inc delta_x so we can unroll loop

; Use delta_x as loop to draw pixels, move right each iteration & maybe down..

@SLRLoop:
	mov     [edi], ah               ; Set first pixel, mask is set up
	dec     cx                      ; Decrement delta_x 
	jz      SHORT @SLRExit          ; and exit if done

	add     si, bx                  ; Add numerator to accumulator
	jnc     SHORT @SLRL2nc          ; Don't move down if carry not set
	add     edi, ebp                ; Else move down one line...

@SLRL2nc: 
	rol     al, 1                   ; Move right one addr if plane = 0
	cmp     al, 12h                 ; Wrap? if al >12 then carry not set
	adc     edi, 0                  ; Adjust address: di = di + carry
	out     dx, al                  ; Set up new bit plane mask

	mov     [edi], ah               ; Draw the pixel
	dec     cx                      ; Decrement delta_x
	jz      SHORT @SLRExit          ; and exit if done

	add     si, bx                  ; Add numerator to accumulator
	jnc     SHORT @SLRL3nc          ; Don't move down if carry not set
	add     edi, ebp                ; Else move down one line...

@SLRL3nc:
	rol     al, 1                   ; Move right one addr if plane = 0
	cmp     al, 12h                 ; Wrap? if al >12 then carry not set
	adc     edi, 0                  ; Adjust address: di = di + carry
	out     dx, al                  ; Set up new bit plane mask
	jmp     SHORT @SLRLoop          ; Loop till done

@SLRExit:
	jmp     SHORT @LineFinished     ; Finito


; If we reach here then draw a steep line to the right...

@SteepRight:
	sub     ax, ax                  ; Zero low word of delta_y * 10000h
	xchg    dx, cx                  ; Switch delta_y with delta_x
	div     cx                      ; Divide by delta_y

	mov     si, bx                  ; si = accumulator
	mov     bx, ax                  ; bx = add fraction
	pop     ax                      ; Get color and bit mask
	mov     dx, SC_DATA             ; Get Sequence Controller data reg
	inc     cx                      ; Inc delta_y so we can unroll loop

; Use delta_x as loop to draw pixels, move down each iteration & maybe right..

@STLRLoop:
	mov     [edi], ah               ; Set first pixel, mask is set up
	dec     cx                      ; Decrement delta_y
	jz      SHORT @LineFinished     ; and exit if done

	add     si, bx                  ; Add numerator to accumulator
	jnc     SHORT @STLRLnc2         ; If no carry then just go down...

	rol     al, 1                   ; Move right one addr if plane = 0
	cmp     al, 12h                 ; Wrap? if al >12 then carry not set
	adc     edi, 0                  ; Adjust address: di = di + carry
	out     dx, al                  ; Set up new bit plane mask

@STLRLnc2:
	add     edi, ebp                ; Advance to next line
	mov     [edi], ah               ; Draw the pixel
	dec     cx                      ; Decrement delta_y
	jz      SHORT @LineFinished     ; and exit if done

	add     si, bx                  ; Add numerator to accumulator
	jnc     SHORT @STLRLnc3         ; If no carry then just go down...

	rol     al, 1                   ; Move right one addr if plane = 0
	cmp     al, 12h                 ; Wrap? if al >12 then carry not set
	adc     edi, 0                  ; Adjust address: di = di + carry
	out     dx, al                  ; Set up new bit plane mask

@STLRLnc3:
	add     edi, ebp                ; Advance to next line.
	jmp     SHORT @STLRLoop         ; Loop till done

@LineFinished:
	 pop edi esi ebp                ; Restore saved registers
	 ret                            ; Exit 

_LineX   ENDP


;----------------------------------------------------------------------------
; FillRectX(int X1, int Y1, int X2, int Y2, int RectColor, int PageBase)
; Fills a rectangle using the specified color on the display 
;----------------------------------------------------------------------------
_RectX  PROC NEAR

	ARG     X1:DWORD, Y1:DWORD, X2:DWORD, Y2:DWORD,\
		RectColor:DWORD, PageBase:DWORD         

	push    ebp                     ; Preserve caller's stack frame
	mov     ebp, esp                ; Set up local stack frame
	push    esi edi ebx             ; Save important registers
	
	cld                             ; Set direction flag to forward 
	mov     eax, SCREEN_WIDTH       ; Get screen width
	imul    eax, [Y1]               ; = offset in page of top scan line
	mov     edi, [X1]               ; Get x pos of top-left of rect and
	shr     edi, 2                  ; = offset of 1st rect pixel in line
	add     edi, eax                ; = offset of 1st rect pixel in page
	add     edi, [PageBase]         ; = offset of 1st pixel in display memory
	add     edi, DISPLAY_MEM        ; = offset of 1st pixel in flat mem

	mov     dx, SC_INDEX            ; Get Sequence Controller Index to
	mov     al, MAP_MASK            ; point to the Map Mask register
	out     dx,al
	inc     dx                      ; Point dx to the SC Data register
	mov     esi, [X1]               ; Get left edge position of rect and
	and     esi, 0003h              ; AND it with left edge plane mask
	mov     bh, LeftClipMask[esi]   ; and put in bh
	mov     esi, [X2]               ; Get right edge position of rect and
	and     esi, 0003h              ; AND it with right edge plane mask
	mov     bl, RightClipMask[esi]  ; and put in bl
	
	mov     ecx, [X2]               ; Get X1 and
	mov     esi, [X1]               ; Get X2
	cmp     ecx, esi                ; Compare them...
	jle     SHORT @FillDone         ; skip if rect has 0 or -ive width
	dec     ecx                     ; Else reduce X2 by 1 
	and     esi, not 011b           ; Clear 2 LSB's of X1
	sub     ecx, esi                ; Find width of rectangle in pixels
	shr     ecx, 2                  ; Get width of rect in bytes  (-1)
	jnz     SHORT @MasksSet         ; There's more than one byte to draw
	and     bh, bl                  ; Else only 1 byte, so merge masks

@MasksSet:
	mov     esi, [Y2]               ; Get Y2
	sub     esi, [Y1]               ; bx = height of rectangle
	jle     SHORT @FillDone         ; Skip if rect has 0 or -ive height
	mov     ah, byte ptr [RectColor] ; Color with which to fill rect
	mov     ebp, SCREEN_WIDTH       ; Stack frame isn't needed any more
	sub     ebp, ecx                ; Get dist from end of one scan 
	dec     ebp                     ; line to start of next

@FillRowsLoop:
	push    ecx                     ; Remember width of rect in bytes (-1)
	mov     al, bh                  ; Put left-edge clip mask in AL
	out     dx, al                  ; Set left-edge plane clip mask
	mov     al, ah                  ; Get color in al
	stosb                           ; Draw the left edge
	dec     ecx                     ; Count off left edge byte
	js      SHORT @FillLoopBottom   ; That's the only byte...
	jz      SHORT @DoRightEdge      ; Else there are only two bytes...
	mov     al, 00fh                ; Else do middle bands 4 pixels a pop
	out     dx, al                  ; Set middle pixels mask to no clip
	mov     al, ah                  ; Get color in al
	rep     stosb                   ; Draw middle bands four pixels a pop

@DoRightEdge:
	mov     al, bl                  ; Gett right-edge clip mask in AL
	out     dx, al                  ; Set the right-edge plane clip mask
	mov     al, ah                  ; Gut color in al
	stosb                           ; Draw the right edge

@FillLoopBottom:
	add     edi, ebp                ; Point to start of next scan line 
	pop     ecx                     ; Get width in addresses (-1)
	dec     esi                     ; Count down scan lines
	jnz     SHORT @FillRowsLoop     ; Keep going until rectangle drawn

@FillDone:
	pop     ebx edi esi             ; Restore caller's register variables
	pop     ebp                     ; Restore caller's stack frame
	ret

_RectX ENDP


;----------------------------------------------------------------------------
; SetDACReg(int Reg, int Red, int Green, int Blue)
; Routine to set a single RGB vga palette register
;----------------------------------------------------------------------------
_SetDACReg PROC NEAR

	ARG     Reg:BYTE, Red:BYTE, Green:BYTE, Blue:BYTE

	push    ebp                     ; Preserve caller's stack frame
	mov     ebp, esp                ; Set up local stack frame

	OUT_8   DAC_WRITE_ADDR, [Reg]   ; Select which vga register to modify

	mov     dx, PEL_DATA_REG        ; Get DAC data register
	OUT_8   dx, [Red]               ; Set red intensity
	OUT_8   dx, [Green]             ; Set green intensity
	OUT_8   dx, [Blue]              ; Set blue intensity

	pop     ebp                     ; Restore caller's stack frame
	ret                             

_SetDACReg ENDP


;----------------------------------------------------------------------------
; GetDACReg(int Reg, int Red, int Green, int Blue)
; Routine to get the RGB values of a single vga palette register
;----------------------------------------------------------------------------
_GetDACReg PROC NEAR

	ARG     Reg:BYTE, Red:DWORD, Green:DWORD, Blue:DWORD

	push    ebp                     ; Preserve caller's stack frame
	mov     ebp, esp                ; Set up local stack frame

	OUT_8   DAC_READ_ADDR, [Reg]    ; Select which vga register to read 

	mov     dx, PEL_DATA_REG        ; Point to DAC data register
	sub     ax, ax                  ; Clear ax

	in      al, dx                  ; Read red value
	mov     ebx, [Red]              ; Get address of red parameter
	mov     [ebx], ax               ; Store value of red component

	in      al, dx                  ; Read green value
	mov     ebx, [Green]            ; Get address of green parameter
	mov     [ebx], ax               ; Store value of green component

	in      al, dx                  ; Read blue value
	mov     ebx, [Blue]             ; Get address of blue parameter
	mov     [ebx], ax               ; Store value of blue component

	pop     ebp                     ; Restore caller's stack frame
	ret                             

_GetDACReg ENDP


;----------------------------------------------------------------------------
; ShowPage(unsigned int PageBase)
; Sets the currently visible display page
;----------------------------------------------------------------------------
_ShowPage PROC NEAR
	ARG     PageBase:DWORD

	push    ebp                     ; Preserve caller's stack frame
	mov     ebp, esp                ; Set up local stack frame

	mov     ecx, PageBase           ; Get offset of required page

	; Wait if we are currently in a vertical retrace
	 mov    dx, INP_STAT_1          ; Get input status #1 register

@SPWait0:
	 in     al, dx                  ; Get vga status
	 and    al, 08h                 ; Are we in display mode yet?
	 jnz    SHORT @SPWait0          ; If not, wait for it

	 ; Set the start display address to the new page
	 cli
	 mov    dx, CRTC_INDEX          ; Get CRT Controller index

	 mov    al, START_ADDR_LOW      ; Display start address low register
	 mov    ah, cl                  ; Get low 8 bits of start address
	 out    dx, ax                  ; Set display start address low reg

	 mov    al, START_ADDR_HIGH     ; Display start address high register
	 mov    ah, ch                  ; Get high 8 bits of start address
	 out    dx, ax                  ; Set display start address high reg
	 sti

	; Wait for a vertical retrace to smooth out things
	 mov    dx, INP_STAT_1          ; Get input status #1 register

@SPWait1:
	 in     al, dx                  ; Get vga status
	 and    al, 08h                 ; Has vertical retrace started ?
	 jz     SHORT @SPWait1          ; If not, wait for it

@SPExit:
	pop     ebp                     ; Restore caller's stack frame
	ret

_ShowPage ENDP

;----------------------------------------------------------------------------
; SyncDisplay()
; Pauses the computer until the next vertical retrace starts
;----------------------------------------------------------------------------
_SyncDisplay PROC NEAR

	mov     dx, INP_STAT_1          ; Get Input Status #1 register

; Wait for any current retrace to end

@Wait0:
	in      al, dx                  ; Get vga status
	and     al, 08h                 ; Are we in display mode yet?
	jnz     SHORT @Wait0            ; If not then wait until we are

; Wait for the start of the next vertical retrace

@Wait1:
	in      al, dx                  ; Get vga status
	and     al, 08h                 ; Vertical retrace started?
	jz      SHORT @Wait1            ; If not then wait until we are

	ret

_SyncDisplay ENDP


;----------------------------------------------------------------------------


_TEXT ENDS                              ; End of code segment

	END
