/*
GAME PROGRAMMING TUTORIAL
Copyright (C) 1996 Emmanuel Lagare

PIXEL PLOTTING WITH FARPTR.H

Thanks to Brennan Underwood (brennan@rt66.com) for helping me with this.

When we "draw" to the screen, we are actually drawing to the video buffer.
The video buffer is the virtual image of the screen that you see. It's
nothing but RAM that is used by the VGA hardware to update the screen. Its
starting location (base) is at segment 0xa000 offset 0x0000. In the case of
Mode 13h, it is a contiguous block of memory from offset 0 to 64000
(320 x 200). To be able to access it, you simply add the following
declaration in your real mode code:

	char far *video_buffer = 0xa0000000l;
		or
	char far *video_buffer = MK_FP(0xa000,0x0000);

Right? Not quite, with DJGPP we are no longer using segments and offsets, so
we use the actual address of the VGA buffer which is 0xa0000 (0xa000 * 0x10).

To plot a pixel at location x,y on the screen, all you need to do is multiply
y by 320 and add x and you have the offset relative to the video buffer.
Simply put:

	x,y on the screen = y * 320 + x on the video buffer

Then simply assign a color value (a byte for Mode 13h) at that location and
then VGA will display your pixel in the specified color.

    (vga buffer base + y * 320 + x) = color

We can make pixel plotting a little bit faster by using bit-shifting instead
of multiplication. Shifting a number to the right is the same as multiplying
that number by 2. Shifting it right twice is the same as multiplying by 4.
In general, shifting a number n-times is the same as multiplying that number
by 2 to the nth power. But 320 is not a power of 2, you might think. And
you are right. But what it is is the sum of powers of 2, 320 is equal to
256 + 64:

    y * 320 = (y << 8) + (y << 6) + x

Thus:

    (vga buffer base + (y << 8) + (y << 6) + x) = color

There are several methods of writing to the video buffer. One of them is
using far pointers. Below is a sample code to demonstrate pixel plotting
in Mode 13h using far pointers. Note the use of long for the video_buffer.
*/

#include <sys/farptr.h>
#include <go32.h> /* for _dos_ds */
#include <dos.h>

#define GRAPHICS        0x013
#define TEXT            0x03

long video_buffer = 0xa0000;

void set_video_mode(int mode)
{
	union REGS regs;

	regs.x.ax = mode;
	int86(0x10, &regs, &regs);
}

void put_pixel(short int x, short int y, char color)
{
	_farpokeb(_dos_ds,video_buffer + (y << 8) + (y << 6) + x, color);
}

void main(void)
{
	unsigned count;

	/* go to graphics mode */
	set_video_mode(GRAPHICS);

	/* fill up the screen with random pixels */
	for(count = 0; count < 32000; count++) {
	put_pixel(rand() % 320, rand() % 200, rand() % 256);
	}

	/* go back to text mode */
	set_video_mode(TEXT);
}
