/*
GAME PROGRAMMING TUTORIAL
Copyright (C) 1996 Emmanuel Lagare

BIT BLTS

bit_blt stands for Bit Block Transfer. Essentially, you are transferring
blocks of data (well call the block a bitmap) into another place, in our
case, into the VGA video buffer.

As a convention, the bitmap will be an array of bytes formatted as
follows:

        width, height   (2 bytes)
        strip 1         (l bytes)
        strip 2         (l bytes)
                :
                :
        strip h-1       (l bytes)
        strip h         (l bytes)

To do bitblts we first calculate the offset, the place where we start
copying the data:

        offset = (y << 8) + (y << 6) + x

Then we copy a strip of the bitmap into the video buffer at the calculated
position. We then add 320 (the width of the screen) to the offset. This
will take us to the next line in the screen. Then we copy the next strip of
the bitmap into the video buffer. We keep doing this until we have copied
all strips of the bitmap.

To copy a strip to the video buffer, we have two options:

        dosmemput(), the method of accessing the VGA buffer that we haven't
        talked about;
                or
        near pointers and memcpy().

And since near pointers are a lot faster we will use it. We will use the
other method only when we are debugging.

Below is a sample code to bit blitting. The code using dosmemput() can be
activated by adding a #define DEBUG. You might want to see how slow it is.
*/

#include <sys/nearptr.h>
#include <go32.h> /* for _dos_ds */
#include <dos.h>

#define GRAPHICS        0x013
#define TEXT            0x03

#ifndef DEBUG
char *video_buffer = (char *)0xa0000;
#else
long video_buffer = 0xa0000;
#endif

void set_video_mode(int mode)
{
        union REGS regs;

        regs.x.ax = mode;
        int86(0x10, &regs, &regs);
}

void bit_blt(int x, int y, char *bitmap)
{
        int yindex, offset = (y << 8) + (y << 6) + x;
        char width= bitmap[0], height = bitmap[1];

        bitmap += 2; /* skip the first two bytes */
        for (yindex = 0; yindex < height; yindex++) {
        #ifndef DEBUG
        memcpy((char *)video_buffer+offset,bitmap,width);
        #else
        dosmemput(bitmap,width,video_buffer+offset);
        #endif
        offset += 320; /* next line of video buffer */
        bitmap += width; /* next line of bitmap */
        }
}

char bitmap[102] = {
        10,10,                          /* width and height of block */
        15,15,15,15,15,15,15,15,15,15,  /* strip 1 */
        15,00,00,00,00,00,00,00,00,15,
        15,00,00,00,00,00,00,00,00,15,
        15,00,00,00,00,00,00,00,00,15,
        15,00,00,00,00,00,00,00,00,15,
        15,00,00,00,00,00,00,00,00,15,
        15,00,00,00,00,00,00,00,00,15,
        15,00,00,00,00,00,00,00,00,15,
        15,00,00,00,00,00,00,00,00,15,
        15,15,15,15,15,15,15,15,15,15   /* strip 10 */
};

void main(void)
{
        unsigned count;

        #ifndef DEBUG
        /* disable all memory protection */
        __djgpp_nearptr_enable();
        video_buffer += __djgpp_conventional_base;
        #endif

        /* go to graphics mode */
        set_video_mode(GRAPHICS);

        /* fill up the screen with our bitmap */
        for(count = 0; count < 32000; count++) {
                bit_blt(rand() % 310, rand() % 190, bitmap);
        }

        /* go back to text mode */
        set_video_mode(TEXT);

        #ifndef DEBUG
        /* reenable memory protection */
        __djgpp_nearptr_disable();
        #endif
}