/*
GAME PROGRAMMING TUTORIAL
Copyright (C) 1996 Emmanuel Lagare

COLORS

So far, I've been saying that the color parameter of the pixels we put into
the screen are their colors. For simplicity's sake, I lied. The truth is
that the values we are putting into the screen are not colors. They are
indices into a palette with 256 entries. Much like a painter's palette, each
entries are mixes of red, green, and blue (RGB) values which you can define.

To access the palette, you must first write 0xff into port 0x3c6 to
initialize it. If you want to read a palette entry, you need to write the
index of the entry you want to read int port 0x3c7. If you want to modify a
palette entry, you need to write the index of the entry you want to read into
port 0x3c8. You can then read or write the red, green, and blue values (in
that order) from port 0x3c9. Take note that all the things we read and write
to the ports are bytes which can have up to 256 values.

If we take all the combinations of the RGB values, we, theoretically, would
have 16.7 million colors (256x256x256), right? Nope. The RGB values use only
6 bits (64 values) so at the most we have 262,144 colors (64x64x64). A lot
smaller but enough for most purposes. The 2 bits are simply dropped off.

So what can you do with these? The most important thing you can do is to
customize your palette to your needs. If you don't want bright colors,
replace them with dark foreboding colors (look at Quake to see what I mean).
Also you can create special effects like fade in, fade out, palette
rotation... the list goes on. Use your imagination.

Below is the implementation and neat example.
*/

#include <sys/nearptr.h>
#include <go32.h> /* for _dos_ds */
#include <dos.h>

#define GRAPHICS        0x013
#define TEXT            0x03

char *video_buffer = (char *)0xa0000;

/* palette */
#define PALETTE_MASK        0x3c6
#define PALETTE_REGISTER_RD 0x3c7
#define PALETTE_REGISTER_WR 0x3c8
#define PALETTE_DATA        0x3c9

typedef struct rgb_color_type {
	unsigned char red;
	unsigned char green;
	unsigned char blue;
} rgb_color;

void set_video_mode(int mode)
{
        union REGS regs;

        regs.x.ax = mode;
        int86(0x10, &regs, &regs);
}

void put_pixel(short int x, short int y, char color)
{
        video_buffer[(y << 8) + (y << 6) + x] = color;
}

/* new */
unsigned char get_pixel(short int x, short int y)
{
        return video_buffer[(y << 8) + (y << 6) + x];
}

/* new */
void get_palette_register(unsigned char index, rgb_color *color)
{
        outportb(PALETTE_MASK,0xff);
        outportb(PALETTE_REGISTER_RD,index);
        color->red = inportb(PALETTE_DATA);
        color->green = inportb(PALETTE_DATA);
        color->blue = inportb(PALETTE_DATA);
}

/* new */
void set_palette_register(unsigned char index, rgb_color *color)
{
        outportb(PALETTE_MASK,0xff);
        outportb(PALETTE_REGISTER_WR,index);
        outportb(PALETTE_DATA,color->red);
        outportb(PALETTE_DATA,color->green);
        outportb(PALETTE_DATA,color->blue);
}

/* new */
void rotate_palette(unsigned char index1, unsigned char index2)
{
        rgb_color color1, color2;
        int index;

        get_palette_register(index1,&color2);
	for (index = index1; index < index2; index++) {
                get_palette_register(index+1,&color1);
                set_palette_register(index,&color1);
	}
        set_palette_register(index2,&color2);
}

void new_color(int xa, int ya, int x, int y, int xb, int yb)
{
        unsigned char color;

        if (!get_pixel(x,y)) {
                color = abs(yb-ya)+abs(xb-xa) + 1;
                color = rand() % color - color/2;
                color += (get_pixel(xa,ya) + get_pixel(xb,yb) + 1) / 2; 
                if (color < 1) color = 1;
                if (color > 255) color = 255;
                put_pixel(x,y,color);
        }
}

void quad_partition(int x1, int y1, int x2, int y2)
{
        int x, y, z;
        unsigned char color;

        x = x1+(x2-x1)/2;
        y = y1+(y2-y1)/2;

        if (!((x2-x1 < 2)&&(y2-y1 < 2))) {
                new_color(x1,y1,x,y1,x2,y1);
                new_color(x2,y1,x2,y,x2,y2);
                new_color(x1,y2,x,y2,x2,y2);
                new_color(x1,y1,x1,y,x1,y2);

                color = (get_pixel(x1,y1)+ get_pixel(x2,y1)+
                        get_pixel(x2,y2)+ get_pixel(x1,y2)) / 4;
                /*color += rand()%((y2-y1+x2-x1)/2);*/
                put_pixel(x,y,color);

                quad_partition(x1,y1,x,y);
                quad_partition(x1,y,x,y2);
                quad_partition(x,y1,x2,y);
                quad_partition(x,y,x2,y2);
        }
}

void main(void)
{
        unsigned count;
        rgb_color color;

        /* disable all memory protection */
        __djgpp_nearptr_enable();
        video_buffer += __djgpp_conventional_base;

        /* go to graphics mode */
        set_video_mode(GRAPHICS);

        /* create a grayscale palette */
        for(count = 0; count < 128; count++) {
                color.red = color.green = color.blue = count / 4 + 1;
                set_palette_register(count,&color);
        }
        for(count = 128; count < 256; count++) {
                color.red = color.green = color.blue = (256-count) / 4 + 1;
                set_palette_register(count,&color);
        }

        /* fill up the screen recursively */
        put_pixel(0,0,rand()%255+1);
        put_pixel(319,0,rand()%255+1);
        put_pixel(319,199,rand()%255+1);
        put_pixel(0,199,rand()%255+1);
        quad_partition(0,0,319,199);

        /* rotate the palette to get neat display */
        while (!kbhit()) {
                rotate_palette(1,255);
                delay(10);
        }
        
        /* go back to text mode */
        set_video_mode(TEXT);

        /* reenable memory protection */
        __djgpp_nearptr_disable();
}
